

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <setjmp.h>
#include "cs.h"
#include "cs_com.h"
#include "cs_pval.h"

namespace tis {

  /* keyword table */
  static struct {
    const char *kt_keyword;
    int         kt_token;
  } ktab[] = {//{"type", T_TYPE},
              {"function", T_FUNCTION},
              {"var", T_VAR},
              {"if", T_IF},
              {"else", T_ELSE},
              {"while", T_WHILE},
              {"return", T_RETURN},
              {"for", T_FOR},
              {"break", T_BREAK},
              {"continue", T_CONTINUE},
              {"do", T_DO},
              {"switch", T_SWITCH},
              {"case", T_CASE},
              {"default", T_DEFAULT},
              {"null", T_NULL},
              {"super", T_SUPER},
              {"new", T_NEW},
              {"try", T_TRY},
              {"catch", T_CATCH},
              {"finally", T_FINALLY},
              {"throw", T_THROW},
              {"typeof", T_TYPEOF},
              {"instanceof", T_INSTANCEOF},
              {"in", T_IN},
              {"property", T_PROPERTY},
              {"const", T_CONST},
              //{"get", T_GET},
              //{"set", T_SET},
              {"include", T_INCLUDE},
              {"like", T_LIKE},
              {"class", T_CLASS},
              {"namespace", T_NAMESPACE},
              {"this", T_THIS},
              {"assert", T_ASSERT},
              {"delete", T_DELETE},
              {"otherwise", T_OTHERWISE},
              {"with", T_WITH},

              {"__FILE__", T___FILE__},
              {"__FOLDER__", T___FOLDER__},
              {"__LINE__", T___LINE__},
              {"__TRACE__", T___TRACE__},
              {"debug", T_DEBUG},
              {"yield", T_YIELD},
              {"await", T_AWAIT},
              {"event", T_EVENT},
              {"async", T_ASYNC},
              {"let", T_LET },
              {"...", T_DOTDOTDOT },
              {"import", T_IMPORT },
              {"export", T_EXPORT },
              {"void", T_VOID },
              {"static", T_STATIC },


              {NULL, 0}};

  /* CsToken name table */
  static const char *t_names[] = {
      "<string>",   "<identifier>", "<integer>", "<float>",    "<symbol>",
      "function",   "var",          "if",        "else",       "while",
      "return",     "for",          "break",     "continue",   "do",
      "switch",     "case",         "default",   "null",       "<=",
      "==",         "!=",           ">=",        "<<",         ">>",
      "&&",         "||",           "++",        "--",         "+=",
      "-=",         "*=",           "/=",        "%=",         "&=",
      "|=",         "^=",           "<<=",       ">>=",        "???", //"type",
      "super",      "new",          "..",        "try",        "catch",
      "finally",    "throw",        "typeof",    "instanceof", "in",
      "%>...<%",    "===",          "!==",       "property",   "const",
      "get",        "set",          "include",   "like",       "<<<",
      ">>>",        "<<<=",         ">>>=",      "~/",         "~%",
      "/~",         "%~",           "class",     "namespace",  "assert",
      "this",       "delete",       "otherwise", "with",       "__FILE__",
      "__FOLDER__", "__LINE__",     "__TRACE__", "<length>",   "<angle>",
      "<duration>", "debug",        "yield",     "await",      "event",
      "async","=>","let", "...",    "import",    "export"

  };

  /* prototypes */
  static int  rtoken(CsCompiler *c, bool exp_symbol = false);
  static int  getstring(CsCompiler *c, int delim = '"');
  static int  getcharacter(CsCompiler *c);
  static int  literalch(CsCompiler *c, int ch);
  static int  getid(CsCompiler *c, int ch);
  static int  getsymbol(CsCompiler *c, int ch = 0);
  static int  getnumber(CsCompiler *c, int ch, int sign = 1);
  static int  getradixnumber(CsCompiler *c, int radix);
  static bool isradixdigit(int ch, int radix);
  static int  getdigit(int ch);
  static int  skipspaces(CsCompiler *c);
  bool        isidchar(int ch);
  static int  getch(CsCompiler *c);
  int         getoutputstring(CsCompiler *c);

  /* CsInitScanner - initialize the scanner */
  void CsInitScanner(CsCompiler *c, stream *s) {
    /* remember the input stream */
    c->input = s;

    /* setup the line buffer */
    c->line.clear();
    c->line.push(0);
    c->linePtr            = c->line.head();
    c->lineNumberChangedP = false;
    c->lineNumber         = 1;

    /* no lookahead yet */
    c->savedToken = T_NOTOKEN;
    c->savedChar  = '\0';

    int fc = s->get(); // UTF BOM stuff:
    if (fc == 0xFEFF) fc = s->get();

    // shebang?
    if (c->line().starts_with(WCHARS("#!"))) // yes it is, skip the first line
    {
      c->lineNumber = 1;
      for (int nc = '#'; (nc = c->input->get()) != stream::EOS && nc != '\n';)
        ;
    } else
      c->savedChar = fc;

    /* not eof yet */
    c->atEOF = false;
  }

  /* CsToken - get the next CsToken */
  int CsToken(CsCompiler *c, bool expect_literal) {
    int tkn;

    if ((tkn = c->savedToken) != T_NOTOKEN)
      c->savedToken = T_NOTOKEN;
    else
      tkn = rtoken(c, expect_literal);
    return tkn;
  }

  /* CsSaveToken - save a CsToken */
  void CsSaveToken(CsCompiler *c, int tkn) { c->savedToken = tkn; }

  /* CsTokenName - get the name of a CsToken */
  const char *CsTokenName(int tkn) {
    static char tname[2];
    if (tkn == T_EOF)
      return "<eof>";
    else if (tkn >= _TMIN && tkn < _TMAX)
      return t_names[tkn - _TMIN];
    tname[0] = char(tkn);
    tname[1] = '\0';
    return tname;
  }

  /* rtoken - read the next CsToken */
  static int rtoken(CsCompiler *c, bool expect_literal) {
    int ch, ch2;

    
    /* check the next character */
    for (;;)
      switch (ch = skipspaces(c)) {
      case EOF: return T_EOF;
      case '\"': return getstring(c);
      case '`': return getstring(c, '`');
      case '\'': 
#ifdef TISCRIPT_CHAR_LITERALS
        return getcharacter(c);
#else
        return getstring(c, '\'');
#endif //

      case '<':
        switch (ch = getch(c)) {
        case '=': return T_LE;
        case '/': return T_TAIL_TAG_START;
        case '<':
          if ((ch = getch(c)) == '=')
            return T_SHLEQ;
          else if (ch == '<') {
            if ((ch = getch(c)) == '=') return T_USHLEQ;
            c->savedChar = ch;
            return T_USHL;
          }
          c->savedChar = ch;
          return T_SHL;
        default: c->savedChar = ch; return '<';
        }
      case '=':
        if ((ch = getch(c)) == '=') {
          if ((ch = getch(c)) == '=') { return T_EQ_STRONG; }
          c->savedChar = ch;
          return T_EQ;
        } else if(ch == '>')
          return T_FARROW;
        c->savedChar = ch;
        return '=';
      case '!':
        if ((ch = getch(c)) == '=') {
          if ((ch = getch(c)) == '=') { return T_NE_STRONG; }
          c->savedChar = ch;
          return T_NE;
        }
        c->savedChar = ch;
        return '!';
      case '>':
        switch (ch = getch(c)) {
        case '=': return T_GE;
        case '>':
          if ((ch = getch(c)) == '=')
            return T_SHREQ;
          else if (ch == '>') {
            if ((ch = getch(c)) == '=') return T_USHREQ;
            c->savedChar = ch;
            return T_USHR;
          }
          c->savedChar = ch;
          return T_SHR;
        default: c->savedChar = ch; return '>';
        }
      case '&':
        switch (ch = getch(c)) {
        case '&': return T_AND;
        case '=': return T_ANDEQ;
        default: c->savedChar = ch; return '&';
        }
      case '%':
        switch (ch = getch(c)) {
        case '>': return getoutputstring(c);
        case '=': return T_REMEQ;
        case '~': return T_RCDR;
        default: c->savedChar = ch; return '%';
        }

      case '|':
        switch (ch = getch(c)) {
        case '|': return T_OR;
        case '=': return T_OREQ;
        default: c->savedChar = ch; return '|';
        }
      case '^':
        if ((ch = getch(c)) == '=') return T_XOREQ;
        c->savedChar = ch;
        return '^';
      case '+':
        switch (ch = getch(c)) {
        case '+': return T_INC;
        case '=': return T_ADDEQ;
        default: 
          c->savedChar = ch; return '+';
        }
      case '-':
        switch (ch = getch(c)) {
        case '-': return T_DEC;
        case '=': return T_SUBEQ;
        case '0': case '1': case '2': case '3': case '4':
        case '5': case '6': case '7': case '8': case '9':
          if(expect_literal)
            return getnumber(c, ch, -1);
          //else
          // fall through
        default: 
          c->savedChar = ch; return '-';
        }
      case '~':
        switch (ch = getch(c)) {
        case '/': return T_CAR;
        case '%': return T_CDR;
        default: c->savedChar = ch; return '~';
        }
      case '*':
        if ((ch = getch(c)) == '=') return T_MULEQ;
        c->savedChar = ch;
        return '*';
      case '/':
        switch (ch = getch(c)) {
        case '=': return T_DIVEQ;
        case '/':
          while ((ch = getch(c)) != EOF)
            if (ch == '\n') break;
          break;
        case '*':
          ch = ch2 = EOF;
          for (; (ch2 = getch(c)) != EOF; ch = ch2)
            if (ch == '*' && ch2 == '/') break;
          break;
        case '~': return T_RCAR;
        default: c->savedChar = ch; return '/';
        }
        break;
      case '.':
        if ((ch = getch(c)) != EOF && is_digit(ch)) {
          c->savedChar = ch;
          return getnumber(c, '.');
        } else if (ch == '.') {
          if ((ch = getch(c)) == '.')
          {
            c->t_token[0] = '.';
            c->t_token[1] = '.';
            c->t_token[2] = '.';
            c->t_token[3] = '\0';
            return T_DOTDOTDOT;
          }
          else {
            c->savedChar = ch;
            c->t_token[0] = '.';
            c->t_token[1] = '.';
            c->t_token[2] = '\0';
            return T_DOTDOT;
          }
        } else {
          c->savedChar  = ch;
          c->t_token[0] = '.';
          c->t_token[1] = '\0';
          return '.';
        }
        break;

      case '?':
        switch (ch = getch(c)) {
          case '.': return T_QUESTION_DOT;
          case '?': return T_QUESTION_QUESTION;
          default:
            c->savedChar = ch; return '?';
        }
        break;

      case 1: 
        ch = getch(c);
        if (ch == '.') 
          return T_DOTDOTDOT;
        else
          c->savedChar = ch;
        return T_DOTDOT;

      case '0':
        switch (ch = getch(c)) {
        case 'x':
        case 'X': return getradixnumber(c, 16);
        case 'b':
        case 'B': return getradixnumber(c, 2);
        case 'o':
        case 'O': return getradixnumber(c, 8);
        default:
          c->savedChar = ch;
          //if (ch >= '0' && ch <= '7')
          //  return getradixnumber(c, 8);
          //else
          return getnumber(c, '0');
        }
        break;
      case '#': 
        return getsymbol(c);

      default:
        if (is_digit(ch))
          return getnumber(c, ch);
        else if (isidchar(ch)) {
          if (expect_literal)
            return getsymbol(c, ch);
          else
            return getid(c, ch);
        } else {
          c->t_token[0] = char(ch);
          c->t_token[1] = '\0';
          return ch;
        }
      }
  }

  /* getstring - get a string */
  static int getstring(CsCompiler *c, int delim) {
    int ch, len = 0;
    // wchar *p;
    /* get the string */
    // p = c->t_wtoken;
    c->t_wtoken.clear();
    while ((ch = getch(c)) != EOF && ch != delim) {

      /* get the first byte of the character */
      if ((ch = literalch(c, ch)) == EOF)
        CsParseError(c, "end of file in literal string");

      c->t_wtoken.push(wchar(ch));
    }
    if (ch == EOF) c->savedChar = EOF;
    c->t_wtoken.push(0);
    c->t_wtoken.pop();
    len = c->t_wtoken.size();
    //*p = '\0';
    return T_STRING;
  }

  /* getstring - get a string between %> and <% */
  int getoutputstring(CsCompiler *c) {
    int ch;
    // wchar *p;
    /* get the string */
    // p = c->t_wtoken;
    c->t_wtoken.clear();

    while ((ch = getch(c)) != EOF) {
      if (ch != '\r' && ch != '\n') break;
    }

    while (ch != EOF) {
      if (ch == '<') {
        if ((ch = getch(c)) == '%')
          break;
        else {
          c->t_wtoken.push('<');
          c->t_wtoken.push(wchar(ch));
        }
      } else {
        /* get character */
        if ((ch = literalch(c, ch)) == EOF)
          CsParseError(c, "end of file in literal string");

        c->t_wtoken.push(wchar(ch));
      }
      ch = getch(c);
    }
    if (ch == EOF) c->savedChar = EOF;
    c->t_wtoken.push(0);
    c->t_wtoken.pop();
    //*p = '\0';
    return T_OUTPUT_STRING;
  }

  // $( something { prm } something )
  int scan_stringizer_string(CsCompiler *c, int &level) {
    int ch;
    c->t_wtoken.clear();
    while (true) {
      ch = getch(c);
      // Note: things like \r go as they are.
      // if( ch == '\\' )
      //  ch = literalch(c,ch);
      // else
      if (ch == EOF)
        break;
      else if (ch == ')') {
        if (--level == 0) break;
      } else if (ch == '(') {
        ++level;
      } else if (ch == '{')
        break;
      c->t_wtoken.push(wchar(ch));
    }
    c->t_wtoken.push(0);
    c->t_wtoken.pop();

    if (ch == EOF) {
      c->savedChar = EOF;
      CsParseError(c, "end of file in stringizer expression");
    }
    return ch;
  }

  // $( something )
  bool scan_stringizer_literal(CsCompiler *c) {
    // caller consumed '$'
    c->t_wtoken.clear();

    int ch = getch(c);
    if (ch != '(') return false;

    int level = 1;

    while (true) {
      ch = getch(c);
      // Note: things like \r go as they are.
      // if( ch == '\\' )
      //  ch = literalch(c,ch);
      // else
      if (ch == EOF)
        break;
      else if (ch == ')') {
        if (--level == 0) break;
      } else if (ch == '(') {
        ++level;
      }
      c->t_wtoken.push(wchar(ch));
    }
    c->t_wtoken.push(0);
    c->t_wtoken.pop();

    if (ch == EOF) {
      c->savedChar = EOF;
      CsParseError(c, "end of file in stringizer expression");
    }
    return true;
  }

  // scans SSX text content, returns delimeter
  char scan_ssx_text(CsCompiler *c) {

    int ch = 0;
    array<wchar> buf;
    
    while (true) {
      ch = getch(c);
      if (ch == EOF)
        break;
      else if (ch == '{') {
        c->savedChar = ch;
        break;
      }
      else if (ch == '<') {
        c->savedChar = ch;
        break;
      }
      buf.push(wchar(ch));
    }

    if (ch == EOF) {
      c->savedChar = EOF;
      CsParseError(c, "end of file in SSX expression");
    }
    c->t_wtoken.clear();
    html_unescape(buf(),c->t_wtoken);

    return char(ch);
  }


  /* getregexp - get a regexp literal
     at exit:
     wtoken should contain - re itself
     token - flags */

  void getregexp(CsCompiler *c) {
    int ch;

    /* get the string */
    c->t_wtoken.clear();
    while (true) {
      ch = getch(c);
      if (ch == '\\') {
        c->t_wtoken.push('\\');
        ch = getch(c);
      } else if (ch == EOF || ch == '/')
        break;
      c->t_wtoken.push(wchar(ch));
    }

    c->t_wtoken.push(0);
    c->t_wtoken.pop();

    if (ch == EOF) {
      c->savedChar = EOF;
      CsParseError(c, "end of file in literal regexp");
    }

    char *pc = c->t_token;
    while (true) {
      ch = getch(c);
      if (ch == 'i' || ch == 'g' || ch == 'm')
        *pc++ = char(ch);
      else {
        c->savedChar = ch;
        break;
      }
    }
    *pc = '\0';
  }

  /* getcharacter - get a character constant */
  static int getcharacter(CsCompiler *c) {
    c->t_value    = literalch(c, getch(c));
    c->t_token[0] = (char)c->t_value.to_int();
    c->t_token[1] = '\0';
    if (getch(c) != '\'') CsParseError(c, "Expecting a closing single quote");
    return T_INTEGER;
  }

  /* CollectHexChar - collect a hex character code */
  static int CollectHexChar(CsCompiler *c) {
    int value, ch;
    if ((ch = getch(c)) == EOF || !isxdigit(ch)) {
      c->savedChar = ch;
      return 0;
    }
    value = is_digit(ch) ? ch - '0' : tolower(ch) - 'a' + 10;
    if ((ch = getch(c)) == EOF || !isxdigit(ch)) {
      c->savedChar = ch;
      return value;
    }
    return (value << 4) | (is_digit(ch) ? ch - '0' : tolower(ch) - 'a' + 10);
  }

  /* CollectOctalChar - collect an octal character code */
  static int CollectOctalChar(CsCompiler *c, int ch) {
    int value = ch - '0';
    if ((ch = getch(c)) == EOF || ch < '0' || ch > '7') {
      c->savedChar = ch;
      return value;
    }
    value = (value << 3) | (ch - '0');
    if ((ch = getch(c)) == EOF || ch < '0' || ch > '7') {
      c->savedChar = ch;
      return value;
    }
    return (value << 3) | (ch - '0');
  }

  /* CollectUnicodeChar - collect a unicode character code */
  static int CollectUnicodeChar(CsCompiler *c) {
    int value, ch;
    if ((ch = getch(c)) == EOF || !isxdigit(ch)) {
      c->savedChar = ch;
      return 0;
    }
    value = is_digit(ch) ? ch - '0' : tolower(ch) - 'a' + 10;
    if ((ch = getch(c)) == EOF || !isxdigit(ch)) {
      c->savedChar = ch;
      return value;
    }
    value = (value << 4) | (is_digit(ch) ? ch - '0' : tolower(ch) - 'a' + 10);
    if ((ch = getch(c)) == EOF || !isxdigit(ch)) {
      c->savedChar = ch;
      return value;
    }
    value = (value << 4) | (is_digit(ch) ? ch - '0' : tolower(ch) - 'a' + 10);
    if ((ch = getch(c)) == EOF || !isxdigit(ch)) {
      c->savedChar = ch;
      return value;
    }
    return (value << 4) | (is_digit(ch) ? ch - '0' : tolower(ch) - 'a' + 10);
  }

  /* literalch - get a character from a literal string */
  static int literalch(CsCompiler *c, int ch) {
    if (ch == '\\') switch (ch = getch(c)) {
      case 'b': ch = '\b'; break;
      case 'f': ch = '\f'; break;
      case 'n': ch = '\n'; break;
      case 'r': ch = '\r'; break;
      case 't': ch = '\t'; break;
      case 'x': ch = CollectHexChar(c); break;
      case 'u': ch = CollectUnicodeChar(c); break;
      case '"': ch = '"'; break;
      case '`': ch = '`'; break;
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7': ch = CollectOctalChar(c, ch); break;
      case EOF:
        ch           = '\\';
        c->savedChar = EOF;
        break;
      }
    return ch;
  }

  /* getid - get an identifier */
  static int getid(CsCompiler *c, int ch) {
    int   len = 1, i;
    char *p;

    /* get the identifier */
    p    = c->t_token;
    *p++ = char(ch);
    while ((ch = getch(c)) != EOF && isidchar(ch)) {
      if (++len > TKNSIZE) CsParseError(c, "identifier too long");
      *p++ = char(ch);
    }
    c->savedChar = ch;
    *p           = '\0';

    /* check to see if it is a keyword */
    for (i = 0; ktab[i].kt_keyword != NULL; ++i)
      if (strcmp(ktab[i].kt_keyword, c->t_token) == 0) {
        if (ktab[i].kt_token == T_EVENT) {
          // only 'event click' or 'event "click"' constructs can be considered
          // as events.
          ch           = skipspaces(c);
          c->savedChar = ch;
          if (ch == '"' || isidchar(ch) || ch == '~')
            return T_EVENT;
          else
            return T_IDENTIFIER;
        }
        return ktab[i].kt_token;
      }
    return T_IDENTIFIER;
  }

  /* getsymbol - get a symbol #something, caller consumed # */
  static int getsymbol(CsCompiler *c, int ch) {
    int   len = 0;
    char *p   = c->t_token;
    if (ch) *p++ = char(ch);
    /* get the identifier */
    while ((ch = getch(c)) != EOF && (isidchar(ch) || ch == '-')) {
      if (++len > TKNSIZE) CsParseError(c, "identifier is too long");
      *p++ = char(ch);
    }
    c->savedChar = ch;
    *p           = '\0';
    return T_SYMBOL;
  }

  static int crackUnits(CsCompiler *c, int token, const char *t_unit) {
    if (t_unit[0] == 0) return token;

    chars units = tool::chars_of(t_unit);

    double d = c->t_value.to_float();
    tool::value v = tool::value::parse_units(d, units, false);

    if (v.is_length())
    {
      c->t_value = v;
      return T_LENGTH;
    }

    if (v.is_duration())
    {
      c->t_value = v;
      return T_DURATION;
    }

    if (v.is_angle())
    {
      c->t_value = v;
      return T_ANGLE;
    }

    CsParseError(c, string::format("unrecognized unit <%s>", t_unit));
    return token;

  }

  /* getnumber - get a number */
  static int getnumber(CsCompiler *c, int ch, int sign) {
    char  t_unit[UNITSIZE + 1];
    char *p   = c->t_token;
    int   tkn = T_INTEGER;

    /* get the part before the decimal point */
    if (ch != '.') {
      *p++ = char(ch);
      while ((ch = getch(c)) != EOF && is_digit(ch))
        *p++ = char(ch);
    }

    /* check for a fractional part */
    if (ch == '.') {
      if ((ch = getch(c)) == '.') {
        c->savedChar = 1;
        goto GOT_NUMBER;
      }
      *p++ = '.';
      if (is_digit(ch)) {
        *p++ = char(ch);
        while ((ch = getch(c)) != EOF && is_digit(ch))
          *p++ = char(ch);
      }
      tkn = T_FLOAT;
    }

    /* check for an exponent */
    if (ch == 'e' || ch == 'E') {
      *p++ = char(ch);
      if ((ch = getch(c)) == '+' || ch == '-') {
        *p++ = char(ch);
        ch   = getch(c);
      }
      while (ch != EOF && is_digit(ch)) {
        *p++ = char(ch);
        ch   = getch(c);
      }
      tkn = T_FLOAT;
    }
    /* terminate the token string */
    // c->savedChar = ch;
  GOT_NUMBER:
    *p        = '\0';
    t_unit[0] = 0;
    if (c->savedChar != 1) {
      int nu = 0;
      while (is_alpha(ch) && nu < UNITSIZE) {
        t_unit[nu++] = char(ch);
        ch           = getch(c);
      }
      t_unit[nu]   = 0;
      c->savedChar = ch;
    }

    /* convert the string to a number */
    if (tkn == T_FLOAT) {
      // c->t_fvalue = (float_t)atof(c->t_token);
      c->t_value = sign * (float_t)tool::atof(c->t_token);
    } else {
      chars t(c->t_token, p - c->t_token);
      int i;
      if (!parse_int(t, i)) {
        // c->t_fvalue = (float_t)atof(c->t_token);
        c->t_value = sign * (float_t)tool::atof(c->t_token);
        tkn = T_FLOAT;
      }
      else
        c->t_value = sign * i;
    }
    /* return the CsToken type */
    return crackUnits(c, tkn, t_unit);
  }

  /* getradixnumber - read a number in a specified radix */
  static int getradixnumber(CsCompiler *c, int radix) {
    char  t_unit[UNITSIZE + 1];
    char *p   = c->t_token;
    int_t val = 0;
    int   ch;

    /* get number */
    while ((ch = getch(c)) != stream::EOS) {
      if (islower(ch)) ch = toupper(ch);
      if (!isradixdigit(ch, radix)) break;
      val  = val * radix + getdigit(ch);
      *p++ = char(ch);
    }
    // c->savedChar = ch;
    *p        = '\0';
    t_unit[0] = 0;
    if (c->savedChar != 1) {
      int nu = 0;
      while (is_alpha(ch) && nu < UNITSIZE) {
        t_unit[nu++] = char(ch);
        ch           = getch(c);
      }
      t_unit[nu]   = 0;
      c->savedChar = ch;
    }

    /* convert the string to a number */
    c->t_value = val;
    return crackUnits(c, T_INTEGER, t_unit);
  }

  /* isradixdigit - check to see if a character is a digit in a radix */
  static bool isradixdigit(int ch, int radix) {
    switch (radix) {
    case 2: return ch >= '0' && ch <= '1';
    case 8: return ch >= '0' && ch <= '7';
    case 10: return ch >= '0' && ch <= '9';
    case 16: return (ch >= '0' && ch <= '9') || (ch >= 'A' && ch <= 'F');
    }
    return false; /* never reached */
  }

  /* getdigit - convert an ascii code to a digit */
  static int getdigit(int ch) { return ch <= '9' ? ch - '0' : ch - 'A' + 10; }

  /* skipspaces - skip leading spaces */
  static int skipspaces(CsCompiler *c) {
    int ch;
    while ((ch = getch(c)) != '\0' && is_space(ch));
    c->firstTokenChar = ch;
    return ch;
  }

  /* isidchar - is this an identifier character */
  bool isidchar(int ch) {
    return is_alpha(ch) || is_digit(ch) || ch == '_' || ch == '$' || ch == '@';
  }

  /* getch - get the next character */
  static int getch(CsCompiler *c) {
    int ch;

    /* check for a lookahead character */
    if ((ch = c->savedChar) != '\0') {
      c->savedChar = '\0';
      return ch;
    }

    /* check for a buffered character */
    while (*c->linePtr == '\0') {

      /* check for end of file */
      if (c->atEOF) return stream::EOS;

      /* read another line */
      else {
        c->line.clear();
        /* read the line */
        c->lineStartInputPos = c->input->get_pos();
        for (; (ch = c->input->get()) != stream::EOS && ch != '\n';)
          c->line.push(wchar(ch));
        c->line.push('\n');
        c->line.push('\0');
        c->lineNumberChangedP = true;
        c->linePtr            = c->line.head();
        ++c->lineNumber;

        /* check for end of file */
        if (ch < 0) c->atEOF = true;
      }
    }

    /* return the current character */
    return *c->linePtr++;
  }

  /* CsParseError - report an error in the current line */
  void CsParseError(CsCompiler *c, const char *msg) {
    int pos = max(0, c->linePtr - c->line.head()) - 1;
    assert(pos >= 0);

    tool::array<char> buf; //___^
    buf.size(pos + 2);
    if (pos >= 0) {
      memset(buf.head(), '_', pos);
      buf[pos] = '^';
    }
    buf[pos + 1] = 0;
    CsThrowKnownError(c->ic, CsErrSyntaxError, msg, c->line.head(), buf.head());
    // c->linePtr
  }

  void  CsParseError(CsCompiler *c, const expr::node* pn, const char *msg) {
    int pos = pn->pos_in_ln;
    assert(pos >= 0);

    tool::array<char> buf; //___^
    buf.size(pos + 2);
    if (pos >= 0) {
      memset(buf.head(), '_', pos);
      buf[pos] = '^';
    }
    buf[pos + 1] = 0;

    c->input->set_pos(pn->line_start);
    tool::array<wchar> line;
    for (int ch; (ch = c->input->get()) != stream::EOS && ch != '\n';)
      line.push(wchar(ch));

    CsThrowKnownError(c->ic, CsErrSyntaxError, msg, line.head(), buf.head());
  }

  void CsParseWarning(CsCompiler *c, const char *msg) {
    c->ic->standardError->printf(W("%s(%d) : warning :%S\n"),
                                 c->input->stream_name(), c->lineNumber, msg);
  }

} // namespace tis
