#include "cs.h"
#include "cs_com.h"
#include "cs_int.h"

namespace tis {

  static int  addliteral(CsCompiler *c, value lit, bool unique = false);


  namespace expr {

    struct node : public resource
    {
      int ln; int pos_in_ln; int_ptr line_start;
      node(CsCompiler *c) { ln = c->lineNumber; pos_in_ln = int(c->linePtr - c->line.cbegin()); line_start = c->lineStartInputPos; }
      node(node *n) { ln = n->ln; pos_in_ln = n->pos_in_ln; line_start = n->line_start; }
      virtual void do_delete(CsCompiler *c) { CsParseError(c, this, "Expecting lvalue"); }
      virtual void do_fetch_root(CsCompiler *c) { do_fetch(c); }
      virtual void do_fetch_initial(CsCompiler *c) { do_fetch(c); }
      virtual void do_fetch(CsCompiler *c) { 
        CsParseError(c, this, "Expecting rvalue"); 
      }
      virtual void do_load(CsCompiler *c) { do_fetch(c); } // fetch only () 

      virtual uint length() const { return 0; }
      virtual bool is_lvalue() const { 
        return false; 
      }
      virtual bool is_var_ref(string* pn = nullptr) const { return false; }
      virtual bool is_literal(tool::value* pv) const { return false; }
      virtual bool is_rest() const { return false; }
      virtual bool is_assignment() const { return false; }
      virtual bool is_list() const { return false; }
      
      virtual void do_store(CsCompiler *c) { 
        CsParseError(c, this, "Expecting lvalue"); 
      }
      virtual void do_assign(CsCompiler *c, node* r) { 
        CsParseError(c, this, "Expecting lvalue");
      }

      virtual void do_store_only(CsCompiler *c) {
        do_store(c);
      }

      virtual void do_increment(CsCompiler *c, byte code, node* r) {
        do_fetch(c);
        if (r) {
          putcbyte(c, BC_PUSH);
          r->do_fetch(c);
        }
        putcbyte(c, code);
        do_store(c);
      }

      virtual string source_name(CsCompiler *c) const {
        CsParseError(c, "Expecting name");
        return string();
      }

      virtual bool fold(tool::value& r) const { return false; }

      virtual handle<node> stringify(CsCompiler *c); // generates CsToString

      virtual void visit(const function<void(node*)>& visitor) {
        visitor(this);
      }

    };

    struct noop : public node // void
    {
      noop(CsCompiler *c) : node(c) {}
      virtual void do_store(CsCompiler *c) override { }
      virtual void do_delete(CsCompiler *c) override { }
      virtual void do_fetch(CsCompiler *c) override { }
    };

    struct novalue : public node // void
    {
      novalue(CsCompiler *c) : node(c) {}
      virtual void do_fetch(CsCompiler *c) {
        putcbyte(c,BC_UNDEFINED);
      }
    };


    struct binary : node {
      byte code; // BC_ADD, BC_SUB, ...
      handle<node> left, right;
      tristate_v   folded;
      binary(CsCompiler *comp, byte c, node* l, node* r) : node(comp), code(c), left(l), right(r) {}
      virtual void do_fetch(CsCompiler *c) override;

      virtual bool fold(tool::value& r) const override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(left);
        visitor(right);
      }


    };

    struct logical_or : node {
      handle<node> left, right;
      logical_or(CsCompiler *comp,node* l, node* r) : node(comp),left(l), right(r) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual bool fold(tool::value& r) const override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(left);
        visitor(right);
      }
    };

    struct logical_and : node {
      handle<node> left, right;
      logical_and(CsCompiler *comp, node* l, node* r) : node(comp), left(l), right(r) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual bool fold(tool::value& r) const override;
      virtual void visit(const function<void(node*)>& visitor)  override {
        visitor(this);
        visitor(left);
        visitor(right);
      }
    };

    struct ternary : node {
      handle<node> cond, truthy, falsy;
      ternary(CsCompiler *comp, node* c, node* t, node* f) : node(comp), cond(c), truthy(t), falsy(f) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(cond);
        visitor(truthy);
        visitor(falsy);
      }
    };

    struct unary : node {
      byte code; // BC_NEG, BC_XOR, ... 
      handle<node> right;
      tristate_v   folded;
      unary(CsCompiler *comp, byte c, node* r) : node(comp), code(c), right(r) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual bool fold(tool::value& r) const override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(right);
      }
    };

    struct index : node { // coll[index]
      handle<node> coll, idx;
      index(CsCompiler *comp,node* c, node* i) : node(comp), coll(c), idx(i) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void do_store(CsCompiler *c) override;
      virtual void do_assign(CsCompiler *c, node* r) override;
      virtual void do_increment(CsCompiler *c, byte code, node* r) override;
      virtual void do_delete(CsCompiler *c) override;
      virtual bool is_lvalue() const override { return true; }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(coll);
        visitor(idx);
      }
    };

    struct index_range : node { // name:value
      handle<node> coll, start, end;
      index_range(CsCompiler *comp, node *c, node *s, node* e): node(comp),coll(c),start(s),end(e) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(coll);
        visitor(start);
        visitor(end);
      }
    };


    struct property : node { // coll.prop, coll.(prop) 
      handle<node> coll, prop;
      property(CsCompiler *comp, node* c, node* p) : node(comp), coll(c), prop(p) {}
      virtual void do_fetch(CsCompiler *c) override;

      virtual void do_store(CsCompiler *c) override;
      virtual void do_assign(CsCompiler *c, node* r) override;
      virtual void do_increment(CsCompiler *c, byte code, node* r) override;
      virtual void do_delete(CsCompiler *c) override;
      virtual bool is_lvalue() const override { return true; }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(coll);
        visitor(prop);
      }
    };

    struct optional_property : property { // coll?.prop 
      DEFINE_TYPE_ID(optional_property)
      optional_property(CsCompiler *comp, node* c, node* p) : property(comp,c,p) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void do_store(CsCompiler *c) override { return node::do_store(c); }
      virtual void do_assign(CsCompiler *c, node* r) override { return node::do_assign(c,r); }
      virtual void do_increment(CsCompiler *c, byte code, node* r) override { return node::do_increment(c, code, r); }
      virtual void do_delete(CsCompiler *c) override { return node::do_delete(c); }
      virtual bool is_lvalue() const override { return false; }
    };

    struct optional_fallback : node { // coll?.prop 
      DEFINE_TYPE_ID(optional_fallback)
      handle<node> optional, fallback;
      optional_fallback(CsCompiler *comp, node* opt, node* fback) : node(comp), optional(opt), fallback(fback) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(optional);
        visitor(fallback);
      }
    };
    
    struct pair : public node { // name:value
      DEFINE_TYPE_ID(pair)
      handle<node> name, val;
      pair(CsCompiler *comp,node* pn,node* pv): node(comp),name(pn),val(pv) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void do_store(CsCompiler *c) override;
      virtual void do_fetch_initial(CsCompiler *c) override;
      virtual void do_assign(CsCompiler *c, node* r) override {
        val->do_assign(c, r);
      }

      virtual string source_name(CsCompiler *c) const override;
      string target_name(CsCompiler *c) const;
      
      virtual handle<node> stringify(CsCompiler *c)
      {
        val = val->stringify(c);
        return this;
      }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(name);
        visitor(val);
      }
    };


    struct assignment : node {
      handle<node> target, source;
      assignment(CsCompiler *comp, node* t, node* s) : node(comp), target(t), source(s) {}

      virtual bool is_assignment() const { return true; }

      virtual void do_fetch(CsCompiler *c) override { 
        target->do_assign(c,source); 
      }
      virtual void do_store(CsCompiler *c) override { 
        target->do_assign(c,source); 
      }
      virtual void do_assign(CsCompiler *c, node* r) override { 
        target->do_assign(c, r); 
      }

      virtual void do_store_only(CsCompiler *c) override {
        target->do_store_only(c);
      }

      virtual void do_delete(CsCompiler *c) override { target->do_delete(c); }
      virtual bool is_lvalue() const override { 
        return target->is_lvalue(); 
      }
      virtual string source_name(CsCompiler *c) const {
        return target->source_name(c);
      }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(target);
        visitor(source);
      }

    };

    struct mutating_assignment : node {
      byte code;
      handle<node> target, source;
      mutating_assignment(CsCompiler *comp, byte bc, node* t, node* s) : node(comp), code(bc), target(t), source(s) {}
      virtual void do_fetch(CsCompiler *c) override { 
        target->do_increment(c,code,source); 
      }
      //virtual void do_delete(CsCompiler *c) override { target->do_delete(c); }
      virtual bool is_lvalue() const override { return target->is_lvalue(); }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(target);
        visitor(source);
      }
    };

    struct literal : node {
      DEFINE_TYPE_ID(literal)
      tool::value lval;
      int_v lindex;
      //literal(int literal_index) : lindex(literal_index) {}
      literal(CsCompiler *comp,const tool::value& val) : node(comp), lval(val) {}
      literal(node *from, const tool::value& val) : node(from), lval(val) {}
      virtual void do_fetch(CsCompiler *c) override {
        putcbyte(c, BC_LIT);
        putcword(c, literal_index(c));
      }
      virtual bool is_literal(tool::value* pv) const override { 
        if (pv) *pv = lval; 
        return true; 
      }
      int literal_index(CsCompiler *c);
      virtual bool fold(tool::value& r) const override { r = lval; return true; }

      virtual string source_name(CsCompiler *c) const {
        if (lval.is_string_symbol())
          return lval.get<ustring>();
        CsParseError(c, this, "Expecting name");
        return string();
      }

      virtual handle<node> stringify(CsCompiler *c) {
        if (lval.is_string())
          return this;
        else
          return node::stringify(c);
      }

    };

    struct literal_code : node {
      byte code;
      literal_code(CsCompiler *comp,byte bc) : node(comp), code(bc) {}
      virtual void do_fetch(CsCompiler *c) override { putcbyte(c, code); }
    };

    struct literal_regexp : node {
      ustring re, flags;
      literal_regexp(CsCompiler *comp): node(comp) {}
      virtual void do_fetch(CsCompiler *c) override;
    };

    struct evar_immutable : node {
      DEFINE_TYPE_ID(evar_immutable)
      int    offset;
      int    level;
      string name;
      evar_immutable(CsCompiler *comp,const string& nm, int lev, int off) : node(comp), name(nm), offset(off),level(lev) {}
      virtual void do_fetch(CsCompiler *c) override { 
        putcbyte(c, BC_EREF);
        putcbyte(c, level);
        putcbyte(c, offset);
      }
      virtual bool is_var_ref(string* pn = nullptr) const override { 
        if (pn) *pn = name;
        return true; 
      }
      virtual string source_name(CsCompiler *c) const {
        return name;
      }
    };

    struct evar : evar_immutable {
      
      DEFINE_TYPE_ID_DERIVED(evar,evar_immutable)

      evar(CsCompiler *comp, const string& nm,int lev, int off) : evar_immutable(comp,nm,lev,off) {}
      virtual void do_fetch(CsCompiler *c) override {
        putcbyte(c, BC_EREF);
        putcbyte(c, level);
        putcbyte(c, offset);
      }

      virtual void do_store(CsCompiler *c) override {
        putcbyte(c, BC_ESET);
        putcbyte(c, level);
        putcbyte(c, offset);
      }
      virtual void do_assign(CsCompiler *c, node* r) override {
        r->do_fetch(c);
        do_store(c);
      }

      virtual void do_delete(CsCompiler *c) override { 
        putcbyte(c, BC_UNDEFINED);
        do_store(c);
      }
      virtual bool is_lvalue() const override { return true; }

    };

    struct rest : node {
      DEFINE_TYPE_ID(rest)
      handle<node> principal;
      rest(CsCompiler *comp,node* p): node(comp), principal(p) {}
      virtual bool is_rest() const override { return true; }
      virtual void do_fetch(CsCompiler *c) override {
        principal->do_fetch(c);
      }
      virtual void do_store(CsCompiler *c) override {
        principal->do_store(c);
      }
      virtual void do_assign(CsCompiler *c, node* r) override {
        r->do_fetch(c);
        do_store(c);
      }
      virtual string source_name(CsCompiler *c) const {
        return principal->source_name(c);
      }
      virtual handle<node> stringify(CsCompiler *c) {
        principal = principal->stringify(c);
        return this;
      }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(principal);
      }

    };

    struct gvar : node {

      DEFINE_TYPE_ID(gvar)

      enum TYPE {
        GLOBAL,
        NEW_NAMESPACE_VAR,
        NEW_NAMESPACE_CONST,
        NEW_MEMBER_VAR,
      };

      int_v  s_index;
      TYPE   type;
      string name;
      gvar(CsCompiler *comp, const string& nm, TYPE t) : node(comp), name(nm), type(t) {}

      virtual void do_fetch_initial(CsCompiler *c) override {
        switch (type) {
        case GLOBAL:         putcbyte(c, BC_GREF); break;
        case NEW_NAMESPACE_VAR:   putcbyte(c, BC_UNDEFINED); return;
        case NEW_NAMESPACE_CONST: putcbyte(c, BC_UNDEFINED); return;
        case NEW_MEMBER_VAR: putcbyte(c, BC_UNDEFINED); return;
        }
        putcword(c, symbol_index(c));
      }

      virtual void do_fetch(CsCompiler *c) override {
        putcbyte(c, BC_GREF);
        putcword(c, symbol_index(c));
      }

      virtual void do_store(CsCompiler *c) override {
        switch (type) {
        case GLOBAL:         
          putcbyte(c, BC_GSET); break;
        case NEW_NAMESPACE_VAR:   
          putcbyte(c, BC_GSETNS_NEW); break;
        case NEW_NAMESPACE_CONST: 
          putcbyte(c, BC_GSETNS_NEW_CONST); break;
        case NEW_MEMBER_VAR: 
          putcbyte(c, BC_MTSET); break;
        }
        putcword(c, symbol_index(c));
      }

      virtual void do_store_only(CsCompiler *c) override {
        switch (type) {
        case GLOBAL:
          putcbyte(c, BC_GSET); break;
        case NEW_NAMESPACE_VAR:
          putcbyte(c, BC_GSETNS); break;
        case NEW_NAMESPACE_CONST:
          putcbyte(c, BC_GSETNS); break;
        case NEW_MEMBER_VAR:
          putcbyte(c, BC_MTSET); break;
        }
        putcword(c, symbol_index(c));
      }


      virtual void do_assign(CsCompiler *c,node* r) override {
        r->do_fetch(c);
        do_store(c);
      }

      virtual void do_delete(CsCompiler *c) override {
        putcbyte(c, BC_GDEL);
        putcword(c, symbol_index(c));
      }
      virtual bool is_lvalue() const override { return true; }
      virtual bool is_var_ref(string* pn = nullptr) const override { 
        if (pn) *pn = name;
        return true; 
      }
      virtual string source_name(CsCompiler *c) const {
        return name;
      }

      int symbol_index(CsCompiler *c);

    };

    struct list : node {
      enum LIST_TYPE { VARS, VECTOR, MAP, EXPRESSIONS };
      enum LIST_ROLE { UNKNOWN, LOCAL_VARS, LOCAL_CONSTS, GLOBAL_VARS, GLOBAL_CONSTS};
      LIST_TYPE type;
      LIST_ROLE role = UNKNOWN; 
      bool      is_mutable;
      array<handle<node>> elements;
      handle<node>        tag; // tuple

      DEFINE_TYPE_ID(list)

      list(CsCompiler *comp, LIST_TYPE lt, bool mut):node(comp), type(lt), is_mutable(mut) {}

      virtual bool is_list() const { return true; }
            
      virtual void do_fetch(CsCompiler *c) override;
              void do_fetch_vars(CsCompiler *c);
              void do_fetch_map(CsCompiler *c);
              void do_fetch_vector(CsCompiler *c);
              void do_fetch_expressions(CsCompiler *c);

      virtual void do_store(CsCompiler *c) override;
              void do_store_vars(CsCompiler *c);
              void do_store_map(CsCompiler *c);
              void do_store_vector(CsCompiler *c);

      virtual void do_assign(CsCompiler *c, node* r) override;

      virtual uint length() const override { return elements.size(); }
      /*virtual bool is_lvalue() const override {
        for (auto el : elements)
          if (!el->is_lvalue())
            return false;
        return true; 
      }*/

      virtual bool is_lvalue() const { 
        return is_mutable; 
      }

      virtual handle<node> stringify(CsCompiler *c) {
        switch (type) {
          case VECTOR: stringify_vector(c); break;
          case MAP: stringify_map(c); break;
          default: assert(false);
        }
        return this;
      }
      void stringify_vector(CsCompiler *c);
      void stringify_map(CsCompiler *c);

      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        for(auto el: elements)
          visitor(el);
      }
    };

    struct postop : node {
      byte code; // BC_INC, BC_DEC, ... 
      handle<node> right;
      postop(CsCompiler *comp, byte c, node* r) : node(comp), code(c), right(r) {}
      virtual void do_fetch(CsCompiler *c) override {
        right->do_increment(c, code, nullptr);
        byte invcode;
        switch (code) {
          case BC_INC: invcode = BC_DEC; break;
          case BC_DEC: invcode = BC_INC; break;
          default: assert(false); return;
        }
        putcbyte(c, invcode);
      }
      virtual void do_fetch_root(CsCompiler *c) override { 
        // when it is a root expr node, like ;a++; it is an equivalent of ;++a;
        right->do_increment(c, code, nullptr);
      }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(right);
      }

    };

    struct preop : node {
      byte code; // BC_INC, BC_DEC, ... 
      handle<node> right;
      preop(CsCompiler *comp, byte c, node* r) : node(comp), code(c), right(r) {}
      preop(postop* po) : node(po), code(po->code), right(po->right) {}
      virtual void do_fetch(CsCompiler *c) override {
        right->do_increment(c, code, nullptr);
      }
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(right);
      }
    };

    struct bytecodes : node { // name:value
      DEFINE_TYPE_ID(bytecodes)
      array<byte> data;
      bytecodes(CsCompiler *comp, bytes d): node(comp), data(d) {}
      virtual void do_fetch(CsCompiler *c) override;
    };

    struct call : node {
      handle<node> funcref;
      array<handle<node>> parameters;
      call(CsCompiler *comp, node* pfr) : node(comp), funcref(pfr) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(funcref);
        for (auto el : parameters)
          visitor(el);
      }
    };

    struct call_method : node {
      handle<node> coll,prop;
      array<handle<node>> parameters;
      call_method(CsCompiler *comp, node* c, node* p) : node(comp), coll(c),prop(p) {}
      virtual void do_fetch(CsCompiler *c) override;
              void do_fetch_nopreambula(CsCompiler *c);
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(coll);
        visitor(prop);
        for (auto el : parameters)
          visitor(el);
      }
    };

    struct optional_call_method : call_method {
      optional_call_method(CsCompiler *comp, node* c, node* p) : call_method(comp,c,p) {}
      virtual void do_fetch(CsCompiler *c) override;
    };

    struct call_method_super : call_method {
      handle<node> _this, _next;
      call_method_super(CsCompiler *comp, node* t, node* n, node* c, node* p) : call_method(comp, c, p), _this(t), _next(n)  {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        call_method::visit(visitor);
        visitor(_this);
        visitor(_next);
      }
    };

    struct new_object : node {
      handle<node> clas;
      handle<call_method> call;
      new_object(CsCompiler *comp, node* cl, call_method* ca) : node(comp),clas(cl), call(ca) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(clas);
        visitor(call);
      }
    };

    // SSX stuff:

    /*struct tsx_call : node {
      handle<node> what; // either function or class with render(atts,kids) method
      array<handle<node>> parameters;
      tsx_call(CsCompiler *comp, node* pfr) : node(comp), what(pfr) {}
      virtual void do_fetch(CsCompiler *c) override;
    };*/

    // vnode factory
    /*struct ssx_vnode_factory : node {
      handle<node> fact; // either Function or Class
      handle<node> atts; 
      handle<node> kids;
      handle<node> states;
      ssx_vnode_factory(CsCompiler *comp, node* pf, node* pa, node* pk, node* ps) : node(comp), fact(pf), atts(pa), kids(pk), states(ps) {}
      virtual void do_fetch(CsCompiler *c) override;
      virtual void visit(const function<void(node*)>& visitor) override {
        visitor(this);
        visitor(fact);
        visitor(atts);
        visitor(kids);
        visitor(states);
      }
    };*/


  }

  typedef handle<expr::node> PVAL;

}