#include "html.h"
#include "html-behaviors.h"

#if defined(USE_DS_VIDEO) 

#pragma warning(disable : 4995)

#include <tchar.h>
#include <dshow.h>
#include <atlbase.h>
#include <dshowasf.h>
#include <dvdmedia.h>
#include <evr.h>

#include "external/directshow/streams.h"
#include "external/directshow/renbase.h"

#pragma comment(lib, "strmiids.lib")
#pragma comment(lib, "wmvcore.lib")
#pragma comment(lib, "mfuuid.lib")
#pragma comment(lib, "strmiids.lib")

#ifndef MONOLITHIC
#pragma comment(lib, "direct-show.lib")
#endif

//#pragma optimize("", off)
//#pragma pack( 16 )

// {269BA141-1FDE-494B-9024-453A17838B9F}
const GUID CLSID_SciterVideoRenderer = {
    0x269ba141,
    0x1fde,
    0x494b,
    {0x90, 0x24, 0x45, 0x3a, 0x17, 0x83, 0x8b, 0x9f}};

// {34E5B77C-CCBA-4EC0-88B5-BABF6CF3A1D2}
// const GUID IID_IVideoRenderer =
//{ 0x34e5b77c, 0xccba, 0x4ec0, { 0x88, 0xb5, 0xba, 0xbf, 0x6c, 0xf3, 0xa1, 0xd2
//} };

#define FILTER_NAME L"Sciter Video Renderer"

namespace html {
  namespace behavior {

    class ds_video_ctl;

    class VideoRenderer : public CBaseVideoRenderer //, public IVideoRenderer
    {
      typedef CBaseVideoRenderer super;

    public:
      DECLARE_IUNKNOWN;

      VideoRenderer(ds_video_ctl *ctl, HRESULT &hr);
      virtual ~VideoRenderer(void);

      virtual HRESULT DoRenderSample(IMediaSample *pMediaSample);
      virtual HRESULT CheckMediaType(const CMediaType *pmt);
      virtual HRESULT SetMediaType(const CMediaType *pmt);

      virtual HRESULT OnStartStreaming() override;
      virtual HRESULT OnStopStreaming() override;

      void DetachCtl() { m_ctl = 0; }

    private:
      HWND             m_hWnd;
      ds_video_ctl *   m_ctl;
      BITMAPINFOHEADER m_bmpInfo;
      CMediaType       m_mediaType;
      // handle<color_space_converter> m_converter;
    };

    class ds_video_ctl : public video_ctl {
      typedef video_ctl super;

      handle<gool::color_space_converter> converter;

      CComPtr<IFilterGraph>  filter_graph;
      CComPtr<IMediaControl> media_ctrl;
      CComPtr<IBasicAudio>   audio_ctrl;
      CComPtr<IQualProp>     qual_prop;
      CComPtr<IBaseFilter>   renderer;
      CComPtr<IMediaSeeking> positioner;

      tristate_v _is_open;
      tristate_v _is_playing;

      uint32           _pitch;
      BITMAPINFOHEADER _bitmap_info;
      bool             _flip_horizontally;
      long             _generation;
      gool::COLORSPACE _cspace;
      // mutex            _guard;

    public:
      ds_video_ctl() : _generation(0), _cspace(COLORSPACE_UNKNOWN) {
        memzero(_bitmap_info);
      }
      virtual ~ds_video_ctl() { 
        _is_playing = _is_playing; 
      }

      bool init();

      virtual bool load_movie(const tool::url &url) override;
      virtual void close_movie() override;

      virtual bool play() override; // starts the video playing.
      virtual void stop() override; // stops the video playing.
      virtual bool is_movie_open() const override;
      virtual bool is_playing() const override;
      virtual bool is_movie_ended() const override;

      virtual double get_movie_duration() const;
      virtual void   goto_start() override;
      virtual bool   set_position(double seconds); // Sets the video's position to a given time.
      virtual double get_position() const;

      virtual size   get_movie_normal_size() const override;

      virtual bool   set_volume(double new_volume) override;
      virtual double get_volume() const override;
      virtual bool   set_balance(double new_balance) override;
      virtual double get_balance() const override;

      virtual void on_start_streaming() {
        _is_playing = TRUE;
        this->notify_video_started(false);
      }
      virtual void on_stop_streaming(bool ended) {
        this->notify_video_stopped(ended);
        _is_playing = FALSE;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override;

      // animator
      virtual uint step(view &, element *, uint /*current_clock*/) override;

      bool check_sample_buffer();
      bool prepare_sample_buffer(gool::COLORSPACE        cspace,
                                 const BITMAPINFOHEADER &bih,
                                 bool                    bFlipHorizontally);
      bool draw_sample(bytes data);
    };

    video_ctl *produce_ds_video_ctl(element *el) { return new ds_video_ctl(); }

    bool ds_video_ctl::init() {
      HRESULT hr = S_OK; // CoInitialize(NULL);

      media_ctrl   = 0;
      audio_ctrl   = 0;
      qual_prop    = 0;
      renderer     = 0;
      positioner   = 0;
      filter_graph = 0;

      chars error_msg;
      
      CComPtr<IFilterGraph2> filterGraph2;

      hr = filter_graph.CoCreateInstance(CLSID_FilterGraph);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to create filter manager");
        return false;
      }
      hr = filter_graph->QueryInterface(IID_IFilterGraph2,(void **)&filterGraph2);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to query for IFilterGraph2 interface");
        return false;
      }
      hr = filter_graph->QueryInterface(IID_IMediaControl, (void **)&media_ctrl);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to query for IMediaControl interface");
        return false;
      }

      hr = filter_graph->QueryInterface(IID_IBasicAudio, (void **)&audio_ctrl);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to query for IBasicAudio interface");
        return false;
      }

      hr = filter_graph->QueryInterface(IID_IMediaSeeking, (void **)&positioner);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to query for IMediaSeeking interface");
        return false;
      }

      //CComPtr<IBaseFilter> trenderer = new VideoRenderer(this, hr);
      renderer = new VideoRenderer(this, hr);

      if (FAILED(hr)) {
        error_msg = CHARS("Failed to create VideoRenderer");
        return false;
      }
      hr = renderer->QueryInterface(IID_IQualProp, (void **)&qual_prop);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to query for IQualProp interface");
        return false;
      }

      hr = filterGraph2->AddFilter(renderer, FILTER_NAME);
      if (FAILED(hr)) {
        error_msg = CHARS("Failed to add video renderer to graph");
        return false;
      }

      return true;
    }

    bool ds_video_ctl::load_movie(const tool::url &url) {

      init();

      if (!media_ctrl) return false;

      ustring path;

      if (url.is_local()) {
        path = url::unescape(url.filename);
        path.replace_all('/', '\\');
      } else {
        path = url.src; //url::unescape(url.src);
      }

      BSTR    bstrUrl = SysAllocStringLen(path, UINT(path.length()));
      HRESULT hr      = media_ctrl->RenderFile(bstrUrl);
      SysFreeString(bstrUrl);

      _is_open = SUCCEEDED(hr);

      return _is_open.val(0) != 0;
    }
    void ds_video_ctl::close_movie() {
      if (!_is_open) return;
      if (media_ctrl) {

        media_ctrl->Stop();
        if (renderer) static_cast<VideoRenderer *>(renderer.p)->DetachCtl();

        frame    = nullptr;
        _is_open = 0;

        media_ctrl   = 0;
        audio_ctrl   = 0;
        qual_prop    = 0;
        renderer     = 0;
        positioner   = 0;
        filter_graph = 0;

        if (_view && _self) _view->refresh(_self);
      }
    }

    bool ds_video_ctl::play() // starts the video playing.
    {
      if (!media_ctrl) return false;
      if (!_is_open) return false;
      HRESULT hr = media_ctrl->Run();
      return SUCCEEDED(hr);
    }

    void ds_video_ctl::stop() {
      if (!media_ctrl) return;
      if (!_is_open) return;
      HRESULT hr = media_ctrl->Stop();
      assert(SUCCEEDED(hr));
      hr = hr;
    }

    void ds_video_ctl::goto_start() {
      if (!positioner) return;
      if (!_is_open) return;
      LONGLONG pos = 0;
      HRESULT  hr  = positioner->SetPositions(
          &pos, AM_SEEKING_AbsolutePositioning, NULL, AM_SEEKING_NoPositioning);
      assert(SUCCEEDED(hr));
      hr = hr;
    }

    bool ds_video_ctl::set_position(
        double seconds) // Sets the video's position to a given time.
    {
      if (!positioner) return false;
      if (!_is_open) return false;
      LONGLONG tim = LONGLONG(seconds * 10L * 1000L * 1000L);
      LONGLONG pos = 0;

      HRESULT hr = positioner->ConvertTimeFormat(&pos, NULL, tim,
                                                 &TIME_FORMAT_MEDIA_TIME);

      hr = positioner->SetPositions(&pos, AM_SEEKING_AbsolutePositioning, NULL,
                                    AM_SEEKING_NoPositioning);
      assert(SUCCEEDED(hr));
      return SUCCEEDED(hr);
    }

    double ds_video_ctl::get_position() const {
      if (!positioner || !_is_open) return -1.0;

      LONGLONG current = 0;
      LONGLONG tim     = 0;

      HRESULT hr = positioner->GetCurrentPosition(&current);
      assert(SUCCEEDED(hr));
      hr = hr;
      hr = positioner->ConvertTimeFormat(&tim, &TIME_FORMAT_MEDIA_TIME, current,
                                         NULL);
      assert(SUCCEEDED(hr));
      hr = hr;

      double out = double(tim);
      out /= 10;
      out /= 1000;
      out /= 1000;
      return out;
    }

    double ds_video_ctl::get_movie_duration() const {
      if (!positioner || !_is_open) return 0.0;

      LONGLONG dur = 0;
      LONGLONG tim = 0;

      HRESULT hr = positioner->GetDuration(&dur);

      assert(SUCCEEDED(hr));
      hr = hr;
      hr = positioner->ConvertTimeFormat(&tim, &TIME_FORMAT_MEDIA_TIME, dur,
                                         NULL);
      assert(SUCCEEDED(hr));
      hr = hr;

      double out = double(tim);
      out /= 10;
      out /= 1000;
      out /= 1000;
      return out;
    }

    bool ds_video_ctl::set_volume(double new_volume) {
      long vol = limit<long>(long(new_volume * 10000) - 10000, -10000, 0);
      if (audio_ctrl) { audio_ctrl->put_Volume(vol); return true; }
      return false;
    }
    double ds_video_ctl::get_volume() const {
      if (audio_ctrl) {
        long vol = 0;
        audio_ctrl->get_Volume(&vol);
        return double(10000 + vol) / 10000.0;
      }
      return 0.0f;
    }
    bool ds_video_ctl::set_balance(double new_balance) {
      long vol = limit<long>(long(new_balance * 10000), -10000, 10000);
      if (audio_ctrl) { audio_ctrl->put_Balance(vol); return true; }
      return false;
    }
    double ds_video_ctl::get_balance() const {
      if (audio_ctrl) {
        long vol = 0;
        audio_ctrl->get_Balance(&vol);
        return double(vol * 10000) / 10000.0;
      }
      return 0.0f;
    }

    size ds_video_ctl::get_movie_normal_size() const {
      if (_is_open)
        return size(abs(_bitmap_info.biWidth), abs(_bitmap_info.biHeight));
      return size();
    }

    bool ds_video_ctl::is_movie_open() const {
      return _is_open.val(0) != FALSE;
      // if(renderer)
      //  static_cast<VideoRenderer*>(renderer.p)->
    }
    bool ds_video_ctl::is_playing() const {
      return _is_playing.val(0) != FALSE;
    }
    bool ds_video_ctl::is_movie_ended() const {
      if (renderer) {
        BOOL b = static_cast<VideoRenderer *>(renderer.p)->IsEndOfStream();
        return b != FALSE;
      }
      return true;
    }

    uint ds_video_ctl::step(view &v, element *self, uint current_clock) {
      // int frame_rate = 1;
      // HRESULT hr = qual_prop->get_AvgFrameRate(&frame_rate);
      if (frame && (frame->_generation != _generation)) {
        _generation = frame->_generation;
        v.refresh(self);
      }

      if (renderer) {
        BOOL b = static_cast<VideoRenderer *>(renderer.p)->IsEndOfStream();
        if (b) static_cast<VideoRenderer *>(renderer.p)->StopStreaming();
      }

      // hr = hr;
      // dbg_printf( "frame_rate = %d\n", frame_rate);
      // if(SUCCEEDED(hr))
      //  return limit(100000 / frame_rate -2,16,66);
      return ANIMATION_TIMER_SPAN;
    }

    bool ds_video_ctl::prepare_sample_buffer(gool::COLORSPACE        cspace,
                                             const BITMAPINFOHEADER &bih,
                                             bool bFlipHorizontally) {
      _bitmap_info       = bih;
      _flip_horizontally = bFlipHorizontally;
      _pitch             = _bitmap_info.biWidth * 4;
      _cspace            = cspace;

      frame     = nullptr;
      converter = nullptr;

      notify_video_initialized(true);

      // return check_sample_buffer();
      return true;
    }
    bool ds_video_ctl::check_sample_buffer() {
      if (_bitmap_info.biWidth == 0 || _bitmap_info.biHeight == 0) return false;
      size dim(abs(_bitmap_info.biWidth), abs(_bitmap_info.biHeight));
      frame     = new bitmap(dim, false, false);
      converter = gool::color_space_converter::factory(_cspace, dim);
      return true;
    }

    bool ds_video_ctl::draw_sample(bytes data) {
      if (!_self || !_view) return false;

      // critical_section _(_guard);

      if (!frame && !check_sample_buffer()) return false;

      frame->set_bits(data, converter);

      return true;
    }

    bool ds_video_ctl::on(view &v, element *self, event_behavior &evt) {
      bool r = super::on(v, self, evt);
      switch (evt.cmd) {
      case VIDEO_STARTED | EVENT_SINKING:
      case VIDEO_STARTED | EVENT_SINKING | EVENT_HANDLED:
        start_animation(&v, self);
        break;
      case VIDEO_STOPPED | EVENT_SINKING:
      case VIDEO_STOPPED | EVENT_SINKING | EVENT_HANDLED:
        stop_animation(&v, self);
        break;
      }
      return r;
    }

#define HR(x)                                                                  \
  if (FAILED(x)) { return x; }

    VideoRenderer::VideoRenderer(ds_video_ctl *ctl, HRESULT &hr)
        : CBaseVideoRenderer(CLSID_VideoRenderer, FILTER_NAME, NULL, &hr),
          m_hWnd(0), m_ctl(ctl) {
    }

    VideoRenderer::~VideoRenderer(void) {
    }

    HRESULT VideoRenderer::DoRenderSample(IMediaSample *pMediaSample) {
      bytes input;
      HR(pMediaSample->GetPointer((BYTE **)&input.start));
      input.length = pMediaSample->GetSize();
      if (m_ctl) return m_ctl->draw_sample(input);
      return E_FAIL;
    }

    HRESULT VideoRenderer::CheckMediaType(const CMediaType *pmt) {
      CheckPointer(pmt, E_POINTER);

      if (pmt->majortype != MEDIATYPE_Video &&
          pmt->majortype != MEDIATYPE_Stream) {
        return E_FAIL;
      }

      const GUID *SubType = pmt->Subtype();
      if (SubType == nullptr) return E_FAIL;

      if (*SubType != MEDIASUBTYPE_YV12 && *SubType != WMMEDIASUBTYPE_I420 &&
          *SubType != MEDIASUBTYPE_IYUV && *SubType != MEDIASUBTYPE_NV12 &&
          *SubType != MEDIASUBTYPE_YUY2 && *SubType != MEDIASUBTYPE_RGB555 &&
          *SubType != MEDIASUBTYPE_RGB565 && *SubType != MEDIASUBTYPE_RGB24 &&
          *SubType != MEDIASUBTYPE_RGB32) {
        return E_FAIL;
      }

      if (pmt->formattype != FORMAT_VideoInfo &&
          pmt->formattype != FORMAT_VideoInfo2) {
        return E_FAIL;
      }

      return S_OK;
    }

    HRESULT VideoRenderer::SetMediaType(const CMediaType *pmt) {
      CheckPointer(pmt, E_POINTER);
      CAutoLock m_lock(this->m_pLock);

      m_mediaType              = *pmt;
      VIDEOINFOHEADER * vInfo  = NULL;
      VIDEOINFOHEADER2 *vInfo2 = NULL;

      if (pmt->formattype == FORMAT_VideoInfo) {
        vInfo = (VIDEOINFOHEADER *)pmt->pbFormat;
        if (vInfo == nullptr ||
            vInfo->bmiHeader.biSize != sizeof(BITMAPINFOHEADER)) {
          return E_INVALIDARG;
        }
        m_bmpInfo = vInfo->bmiHeader;
      } else if (pmt->formattype == FORMAT_VideoInfo2) {
        vInfo2 = (VIDEOINFOHEADER2 *)pmt->pbFormat;
        if (vInfo2 == nullptr ||
            vInfo2->bmiHeader.biSize != sizeof(BITMAPINFOHEADER)) {
          return E_INVALIDARG;
        }
        m_bmpInfo = vInfo2->bmiHeader;
      } else {
        return E_INVALIDARG;
      }

      bool hFlip = (pmt->subtype == MEDIASUBTYPE_RGB24 ||
                    pmt->subtype == MEDIASUBTYPE_RGB32 ||
                    pmt->subtype == MEDIASUBTYPE_RGB555 ||
                    pmt->subtype == MEDIASUBTYPE_RGB565);

      gool::COLORSPACE cs = gool::COLORSPACE_UNKNOWN;

      if (pmt->subtype == MEDIASUBTYPE_YV12)
        cs = gool::COLORSPACE_YV12;
      else if (pmt->subtype == MEDIASUBTYPE_IYUV)
        cs = gool::COLORSPACE_IYUV; // swap the u and v planes
      else if (pmt->subtype == WMMEDIASUBTYPE_I420)
        cs = gool::COLORSPACE_IYUV; // the same
      else if (pmt->subtype == MEDIASUBTYPE_NV12)
        cs = gool::COLORSPACE_NV12;
      else if (pmt->subtype == MEDIASUBTYPE_YUY2)
        cs = gool::COLORSPACE_YUY2;
      else if (pmt->subtype == MEDIASUBTYPE_RGB24)
        cs = gool::COLORSPACE_RGB24;
      else if (pmt->subtype == MEDIASUBTYPE_RGB555)
        cs = gool::COLORSPACE_RGB555;
      else if (pmt->subtype == MEDIASUBTYPE_RGB565)
        cs = gool::COLORSPACE_RGB565;
      else
        dbg_printf("Unknowm media subtype!\n");

      return m_ctl->prepare_sample_buffer(cs, m_bmpInfo, hFlip);
    }

    HRESULT VideoRenderer::OnStartStreaming() {
      m_ctl->on_start_streaming();
      return super::OnStartStreaming();
    }

    HRESULT VideoRenderer::OnStopStreaming() {
      m_ctl->on_stop_streaming(IsEndOfStream() != FALSE);
      return super::OnStopStreaming();
    }

  } // namespace behavior
} // namespace html

#endif