#include "html.h"

#include "osx/osx-ref.h"

#include <CoreFoundation/CoreFoundation.h>
#include <CFNetwork/CFNetwork.h>
#include <CFNetwork/CFHTTPStream.h>
#include <SystemConfiguration/SystemConfiguration.h>

#define BUFLEN 4096 // Size of HTTP Buffer...
#define MAX_REDIRECTS 6

namespace html {
  class osx_inet_client : public inet_client {
  public:
    osx_inet_client() : active_connections(0) {}
    virtual ~osx_inet_client() { stop(); }

    virtual void exec(pump *pm, request *prq) override;
    void         stop();

    uint active_connections;
  };

  void osx_inet_client::stop() {
    while (active_connections)
      sched_yield();
  }

  void pump::open_internet() {
    if (!inet) inet = new osx_inet_client();
  }

  void ReadStreamClientCallBack(CFReadStreamRef   readStream,
                                CFStreamEventType eventType,
                                void *            clientCallBackInfo) {
    handle<request> hrq = static_cast<request *>(clientCallBackInfo);

    // we shall start reading before requesting headers
    byte    buff[BUFLEN];
    CFIndex numBytesRead;

    // uint start = tool::get_ticks();

    if ((eventType & kCFStreamEventHasBytesAvailable) != 0) {
      if (hrq->status == 0) {
        CF::ref<CFHTTPMessageRef> response =
            (CFHTTPMessageRef)CFReadStreamCopyProperty(
                readStream, kCFStreamPropertyHTTPResponseHeader);

        if (!response) {
          hrq->done_with_failure(1);
          hrq->release();
          CFReadStreamUnscheduleFromRunLoop(readStream, CFRunLoopGetCurrent(),
                                              kCFRunLoopCommonModes);
          CFReadStreamClose(readStream);
          return;
        }

        CF::ref<CFURLRef> finalUrl = (CFURLRef)CFReadStreamCopyProperty(
            readStream, kCFStreamPropertyHTTPFinalURL);

        hrq->status = (uint)CFHTTPMessageGetResponseStatusCode(response);

        if (finalUrl) hrq->content_url = cvt(CFURLGetString(finalUrl));

        CF::ref<CFStringRef> cfContentType =
            CFHTTPMessageCopyHeaderFieldValue(response, CFSTR("Content-Type"));
        //                if(!cfContentType) {
        //                    hrq->done_with_failure(1); // to define a proper
        //                    failure code return;
        //                }

        tool::string contentType = cvt(cfContentType);

        if (!verify_content_type(hrq, contentType)) {
          hrq->done_with_failure(2);
          hrq->release();
          CFReadStreamUnscheduleFromRunLoop(readStream, CFRunLoopGetCurrent(),
                                              kCFRunLoopCommonModes);
          CFReadStreamClose(readStream);
          return;
        }

        CF::ref<CFStringRef> cfContentLength =
            CFHTTPMessageCopyHeaderFieldValue(response,
                                              CFSTR("Content-Length"));
        if (cfContentLength) {
          tool::string contentLength = cvt(cfContentLength);
          hrq->content_length        = atoi(contentLength);
        }


        CF::ref<CFDictionaryRef> responseHeaders = CFHTTPMessageCopyAllHeaderFields(response);

        auto gatherFields = [](const void* key, const void* value, void* context) {
          request* prq = (request*)context;
          CFStringRef cfKey = (CFStringRef)key;
          CFStringRef cfVal = (CFStringRef)value;
          prq->rs_headers[cvt(cfKey)] = cvt(cfVal);
        };
        CFDictionaryApplyFunction(responseHeaders, gatherFields, hrq.ptr());

        hrq->data.clear();
      }

      numBytesRead = CFReadStreamRead(readStream, buff, BUFLEN);
      if (numBytesRead > 0) {
        hrq->data_chunk_arrived(bytes(buff, numBytesRead));
      } else if (numBytesRead < 0) {
        eventType = kCFStreamEventErrorOccurred;
      } else // if (numBytesRead == 0)
      {
        eventType = kCFStreamEventEndEncountered;
      }
    }
    if ((eventType & kCFStreamEventErrorOccurred) != 0) {
      CFStreamError error = CFReadStreamGetError(readStream);
      CFReadStreamUnscheduleFromRunLoop(readStream, CFRunLoopGetCurrent(),
                                        kCFRunLoopCommonModes);
      CFReadStreamClose(readStream);
      hrq->done_with_failure(error.error);
      hrq->release(); // extra release
    } else if ((eventType & kCFStreamEventEndEncountered) != 0) {
      hrq->data_chunk_arrived(bytes());
      hrq->done_with_success(hrq->status);
      CFReadStreamUnscheduleFromRunLoop(readStream, CFRunLoopGetCurrent(),
                                        kCFRunLoopCommonModes);
      CFReadStreamClose(readStream);
      hrq->release(); // extra release
    }
  }

  /*    void RequestReleaseCallBack(void *info)
      {
          request* prq = (request*)info;
          if (prq)
              prq->release();
      }
      

      void * RequestRetainCallBack(void *info)
      {
          request* prq = (request*)info;
          if (prq)
              prq->add_ref();
          return info;
      } */

  void osx_inet_client::exec(pump *pm, request *prq) {
    handle<request> hrq = prq;
    // int redirects = 0;

    hrq->seal_request();

    tool::url uri = hrq->used_url();

    if (uri.hostname.length() == 0) {
      assert(false);
      hrq->done_with_failure(0);
      return;
    }

    // https://developer.apple.com/library/ios/documentation/Networking/Conceptual/CFNetwork/CFStreamTasks/CFStreamTasks.html

    CF::ref<CFStringRef> used_url = cfstr(uri.compose_without_anchor());
    CF::ref<CFStringRef> http_verb;
    switch (hrq->rq_type) {
    default:
    case RQ_GET: http_verb = cfstr(CHARS("GET")); break;
    case RQ_POST: http_verb = cfstr(CHARS("POST")); break;
    case RQ_PUT: http_verb = cfstr(CHARS("PUT")); break;
    case RQ_DELETE: http_verb = cfstr(CHARS("DELETE")); break;
    }

    CF::ref<CFURLRef> cfUrl =
        CFURLCreateWithString(kCFAllocatorDefault, used_url, NULL);
    CF::ref<CFHTTPMessageRef> cfHttpReq = CFHTTPMessageCreateRequest(
        kCFAllocatorDefault, http_verb, cfUrl, kCFHTTPVersion1_1);

    for (int n = 0; n < hrq->rq_headers.size(); ++n) {
      CF::ref<CFStringRef> name = cfstr(hrq->rq_headers.key(n));
      CF::ref<CFStringRef> val  = cfstr(hrq->rq_headers.value(n));
      CFHTTPMessageSetHeaderFieldValue(cfHttpReq, name, val);
    }

    if (hrq->data.size()) {
      CF::ref<CFDataRef> cfBody = CFDataCreate(
          kCFAllocatorDefault, hrq->data.cbegin(), hrq->data.size());
      CFHTTPMessageSetBody(cfHttpReq, cfBody);
    }

    CF::ref<CFReadStreamRef> readStream =
        CFReadStreamCreateForHTTPRequest(kCFAllocatorDefault, cfHttpReq);

    CFReadStreamSetProperty(readStream, kCFStreamPropertyHTTPShouldAutoredirect,
                            kCFBooleanTrue);

    CF::ref<CFDictionaryRef> proxy;

    if (hrq->proxy_host.is_defined() && hrq->proxy_port.is_defined()) {
      CF::ref<CFStringRef> host = cfstr(hrq->proxy_host);
      CFIndex              pn   = hrq->proxy_port.val(0);
      CF::ref<CFNumberRef> port =
          CFNumberCreate(kCFAllocatorDefault, kCFNumberCFIndexType, &pn);

      // Loook for the reference here
      // http://stackoverflow.com/questions/24485192/cant-set-proxy-to-cfstream-on-kcfstreampropertyhttpproxy-parameter
      bool        is_https = uri.protocol == CHARS("https");
      const void *keys[]   = {is_https ? kCFStreamPropertyHTTPSProxyHost
                                     : kCFStreamPropertyHTTPProxyHost,
                            is_https ? kCFStreamPropertyHTTPSProxyPort
                                     : kCFStreamPropertyHTTPProxyPort};
      const void *values[] = {host, port};

      proxy =
          CFDictionaryCreate(0, keys, values, 2, &kCFTypeDictionaryKeyCallBacks,
                             &kCFTypeDictionaryValueCallBacks);

    } else {
      proxy = SCDynamicStoreCopyProxies(NULL);
    }

    if (proxy)
      CFReadStreamSetProperty(readStream, kCFStreamPropertyHTTPProxy, proxy);

    CFStreamClientContext dataStreamContext = {
        0, prq, NULL /*&RequestRetainCallBack*/,
        NULL /*&RequestReleaseCallBack*/, NULL};


    // will be released in ReadStreamClientCallBack

    hrq->add_ref();
      
    if (CFReadStreamSetClient(readStream,
                              kCFStreamEventHasBytesAvailable |
                                  kCFStreamEventErrorOccurred |
                                  kCFStreamEventEndEncountered, // subscription
                              &ReadStreamClientCallBack, &dataStreamContext)) {
      CFReadStreamScheduleWithRunLoop(readStream, CFRunLoopGetCurrent(),
                                      kCFRunLoopCommonModes);
    }

    // send it

    if (!CFReadStreamOpen(readStream)) {
      hrq->done_with_failure(1);
      hrq->release();
      assert(false);
    }
  }

} // namespace html
