#include "html.h"

#include <curl/curl.h>
#include <dlfcn.h>

typedef CURLcode (*curl_global_init_t)(long flags);
typedef struct curl_slist *(*curl_slist_append_t)(struct curl_slist *list,
                                                  const char *       string);
typedef void (*curl_slist_free_all_t)(struct curl_slist *list);

typedef CURL *(*curl_easy_init_t)(void);
typedef CURLcode (*curl_easy_setopt_t)(CURL *curl, CURLoption option, ...);
typedef CURLcode (*curl_easy_getinfo_t)(CURL *curl, CURLINFO info, ...);
typedef CURLcode (*curl_easy_perform_t)(CURL *curl);
typedef void (*curl_easy_cleanup_t)(CURL *curl);

static curl_global_init_t    _curl_global_init    = nullptr;
static curl_slist_append_t   _curl_slist_append   = nullptr;
static curl_slist_free_all_t _curl_slist_free_all = nullptr;

static curl_easy_init_t    _curl_easy_init    = nullptr;
static curl_easy_setopt_t  _curl_easy_setopt  = nullptr;
static curl_easy_getinfo_t _curl_easy_getinfo = nullptr;
static curl_easy_perform_t _curl_easy_perform = nullptr;
static curl_easy_cleanup_t _curl_easy_cleanup = nullptr;

namespace html {

  static string client_name;

  class _curl_inst {
    char  errbuf[CURL_ERROR_SIZE];
    CURL *h;

  public:
    _curl_inst() : h(nullptr) {}
    ~_curl_inst() { close(); }

    void open() {
      close();
      errbuf[0] = 0;
      h         = _curl_easy_init();
      _curl_easy_setopt(h, CURLOPT_ERRORBUFFER, errbuf);
    }
    void close() {
      if (h) {
        _curl_easy_cleanup(h);
        h = 0;
      }
    }
    operator CURL *() { return h; }

    NONCOPYABLE(_curl_inst)
  };

  class task : public tool::async::task {
    handle<request> hrq;
    _curl_inst      curl;

    task(request *prq) { hrq = prq; }

public:

    static void create(request *prq) {
      if(task *pt = new task(prq))
        pt->start();
    }

    virtual ~task() {
    }

    virtual void exec() {
      do_exec();
      stop();
    }
    virtual void stop() { curl.close(); }
    void         do_exec();

    void data_ready(uint status) {
      hrq->done_with_success(status);
      curl.close();
    }
    void failed(uint status) {
      hrq->done_with_failure(status);
      curl.close();
    }
    bool is_active() {
      if (!hrq->the_pump) return false;
      return hrq->the_pump->is_active();
    }
  };

  class curl_inet_client : public inet_client {
    bool         active;

  public:
    curl_inet_client() : active(0) {
      // Must initialize libcurl before any threads are started
      _curl_global_init(CURL_GLOBAL_ALL);
    }

    virtual ~curl_inet_client() {
    }

    virtual void exec(pump *pm, request *rq) override;
  };

  void pump::open_internet() {
    if (!client_name.length()) {
      const char *osname = environment::get_os_version_name();
      client_name        = string::format("sciter %s; %s; www.sciter.com )",
                                   SCITER_VERSION_FULL_STR, osname);
    }
    if (inet) return;

    static bool attempt_was_made = false;

    if (attempt_was_made) return;

    attempt_was_made = true;
#if 1
    void *handle = dlopen("libcurl.so.4", RTLD_NOW);
    if (!handle) handle = dlopen("libcurl.so", RTLD_NOW);
    if (!handle) {
      alert("HTTP: libcurl not found on this machine, http request ignored.");
      return;
    }

    _curl_global_init = (curl_global_init_t)dlsym(handle, "curl_global_init");
    _curl_slist_append =
        (curl_slist_append_t)dlsym(handle, "curl_slist_append");
    _curl_slist_free_all =
        (curl_slist_free_all_t)dlsym(handle, "curl_slist_free_all");

    _curl_easy_init   = (curl_easy_init_t)dlsym(handle, "curl_easy_init");
    _curl_easy_setopt = (curl_easy_setopt_t)dlsym(handle, "curl_easy_setopt");
    _curl_easy_getinfo =
        (curl_easy_getinfo_t)dlsym(handle, "curl_easy_getinfo");
    _curl_easy_perform =
        (curl_easy_perform_t)dlsym(handle, "curl_easy_perform");
    _curl_easy_cleanup =
        (curl_easy_cleanup_t)dlsym(handle, "curl_easy_cleanup");

    if (_curl_global_init && _curl_slist_append && _curl_slist_free_all &&
        _curl_easy_init && _curl_easy_setopt && _curl_easy_getinfo &&
        _curl_easy_perform && _curl_easy_cleanup)
      inet = new curl_inet_client();
#endif	  
  }

  void curl_inet_client::exec(pump *pm, request *prq) {
    active = true;
    task::create(prq);
  }

  size_t request_write_func(void *ptr, size_t size, size_t nmemb, request *rq) {
    rq->data_chunk_arrived(bytes((byte *)ptr, size * nmemb));
    const char *msg = (const char *)ptr;
    return nmemb;
  }

  size_t request_read_func(void *ptr, size_t size, size_t nmemb, request *rq) {
    size_t len = min(size * nmemb, rq->data.length());
    memcpy(ptr, rq->data.cbegin(), len);
    const char *msg = (const char *)ptr;
    rq->data.remove(0, len);
    return len / size;
  }

  void task::do_exec() {
    curl.open();

    handle<request> keeper = hrq;

    hrq->seal_request();

    tool::url uri = hrq->used_url();

    if (uri.hostname.length() == 0) {
      assert(false);
      failed(0);
      return;
    }

    // bool is_https = uri.protocol == "https";

    _curl_easy_setopt(curl, CURLOPT_USERAGENT, client_name.c_str());
    _curl_easy_setopt(curl, CURLOPT_WRITEDATA, hrq.ptr());
    _curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, request_write_func);

    _curl_easy_setopt(curl, CURLOPT_READDATA, hrq.ptr());
    _curl_easy_setopt(curl, CURLOPT_READFUNCTION, request_read_func);
    //_curl_easy_setopt(curl, CURLOPT_NOPROGRESS, 0L);
    //_curl_easy_setopt(curl, CURLOPT_PROGRESSFUNCTION, my_progress_func);
    //_curl_easy_setopt(curl, CURLOPT_PROGRESSDATA, Bar);

    _curl_easy_setopt(curl, CURLOPT_TIMEOUT, 15);
    _curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION,
                      1); // enable location redirects
    _curl_easy_setopt(curl, CURLOPT_MAXREDIRS,
                      1); // set maximum allowed redirects
    _curl_easy_setopt(
        curl, CURLOPT_NOSIGNAL,
        1); // to prevent 'longjmp causes uninitialized stack frame'

    /*
        UINT connection_flags = INTERNET_FLAG_NO_AUTO_REDIRECT;
        if( hrq->rq_type == RQ_POST )
           connection_flags |= INTERNET_FLAG_KEEP_CONNECTION |
       INTERNET_FLAG_NO_CACHE_WRITE | INTERNET_FLAG_FORMS_SUBMIT; else
           connection_flags |= INTERNET_FLAG_KEEP_CONNECTION;
        if( is_https )
           connection_flags |= HTTPS_FLAGS;

        hConnection  = ::InternetConnectA(  h_root_handle,  // internet opened
       handle uri.hostname, // server name INTERNET_PORT(uri.port),     // ports
                                  uri.username, // user name
                                  uri.password, // password
                                  INTERNET_SERVICE_HTTP, // service type
                                  connection_flags, // service option
                                  0); // context call-back option

        if( !hConnection )
        {
          failed(::GetLastError());
          return;
        }

        string param_data = hrq->url_encoded_params();

        if( hrq->rq_type == RQ_GET && param_data.length() )
        {
          if( uri.params.length())
            uri.params += '&';
          uri.params += param_data;
        }
    */

    _curl_easy_setopt(curl, CURLOPT_URL, uri.compose_without_anchor().c_str());

    if (!uri.username.is_empty())
      _curl_easy_setopt(curl, CURLOPT_USERNAME, uri.username.c_str());
    if (!uri.password.is_empty())
      _curl_easy_setopt(curl, CURLOPT_PASSWORD, uri.password.c_str());

    // send it

    const char *pc = hrq->url;

    struct curl_slist *header_list =
        nullptr; // http headers to send with request
    header_list = _curl_slist_append(header_list, "Accept: */*");

    // chars headers = hrq->headers;
    array<char> out;
    for (int n = 0; n < hrq->rq_headers.size(); ++n) {
      out.clear();
      out.push(hrq->rq_headers.key(n));
      out.push(CHARS(": "));
      out.push(hrq->rq_headers.value(n));
      out.push(0);
      header_list = _curl_slist_append(header_list, out.cbegin());
    }

    if (hrq->rq_type == RQ_GET) {
      _curl_easy_setopt(curl, CURLOPT_HTTPGET, 1L);
    } else {
      _curl_easy_setopt(curl, CURLOPT_POST, 1L);
      _curl_easy_setopt(curl, CURLOPT_CUSTOMREQUEST, get_http_verb(hrq));

      if (hrq->data.length()) {
        _curl_easy_setopt(curl, CURLOPT_POSTFIELDSIZE,
                          (long)hrq->data.length());
        _curl_easy_setopt(curl, CURLOPT_COPYPOSTFIELDS, hrq->data.cbegin());
      }
    }

    if (header_list) _curl_easy_setopt(curl, CURLOPT_HTTPHEADER, header_list);

    keeper->data.clear();

    auto rcode = _curl_easy_perform(curl);

    ///* return */
    // return rcode;

    char *real_url = 0;

    if (CURLE_OK ==
            _curl_easy_getinfo(curl, CURLINFO_EFFECTIVE_URL, &real_url) &&
        real_url)
      hrq->content_url = real_url;

    long status = -1;
    if (CURLE_OK == _curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &status))
      hrq->status = uint(status);

    char *content_type = 0;
    if (CURLE_OK ==
            _curl_easy_getinfo(curl, CURLINFO_CONTENT_TYPE, &content_type) &&
        content_type) {
      verify_content_type(hrq, content_type);
    }

    keeper->data_chunk_arrived(bytes());

    data_ready(status);

    curl.close();

    if (header_list) _curl_slist_free_all(header_list);
  }

} // namespace html
