#include "html.h"

namespace html {

  event::event(element *trg, int op) : cmd(op), target(trg), alt_state(view::last_alts_keys_state) {}

  uint   view::last_alts_keys_state = 0;

  void view::dispatch_posted_event(handle<posted_event> pe) {
    send_behavior_event(*pe);
  }

  void view::handle_posted_event(handle<posted_event> pe) {
    if (pe->is_internal()) {
      if (pe->target) pe->target->on_internal(*this, *pe);
    } else
      dispatch_posted_event(pe);
  }

  void view::process_posted_things(bool repost) {
    const int REPOST_DEPTH = 32;

    for (int run = 0; run < REPOST_DEPTH; ++run) // posted things handling can
                                                 // lead to generation of other
                                                 // posted things so we do this
                                                 // REPOST_DEPTH times
    {
      updater of(this, 0, false, false);

      array<handle<posted_event>> _posted_events;
      array<handle<functor>>  _posted_functors;

      {
        critical_section _pg(posted_guard);
        _posted_events.swap(posted_events);
        posted_events.clear();
        _posted_functors.swap(posted_functors);
        posted_functors.clear();

        if (_posted_events.size() == 0 && _posted_functors.size() == 0) return;
      }

      while (_posted_events.size()) {

        if (this->inside_animation_frame && _posted_events[0]->cmd == ANIMATION && _posted_events[0]->reason == 0) {
          critical_section _pg(posted_guard);
          posted_events.push(_posted_events.pop_front());
        }
        else {
          handle<posted_event> evt = _posted_events.pop_front();
//#ifdef _DEBUG
//          if (evt.cmd == ANIMATION && evt.reason == 0)
//            dbg_printf("animationend\n");
//#endif
          handle_posted_event(evt);
        }
      }
      while (_posted_functors.size()) {
        handle<functor> ft = _posted_functors.pop_front();
        if (!ft->operator()()) {
          // it is not complete
          if (repost) {
            critical_section _pg(posted_guard);
            if (posted_functors.size() == 0) request_idle();
            posted_functors.push(ft); // repost it again.
          }
        }
      }
    }
    // alert("process_posted_things was not complete in 32 runs");
    //if (_posted_events.size() > 32) assert(false);
  }

  bool view::idle_pileup_detected() const {
    auto cnt = items_in_idle_queue();
    if (cnt >= 100) {
      dbg_printf("WARNING: pileup of idle tasks detected, n = %u\n", cnt);
      return true;
    }
    return false;
  }

  void view::request_idle() {
    //critical_section _pg(posted_guard); // request_idle can be called from worker threads
    //if (items_in_idle_queue() == 0)
      do_request_idle();
  }

  void view::on_visibility_changed(bool visible) {
    if (visible) {
      //this->is_awaiting_update_layered = false;
      this->on_animation_tick();
    } else {
      this->remove_finite_animations();
    }
  }

  size_t view::items_in_idle_queue() const {
    critical_section _pg(posted_guard);
    size_t           cnt = 0;
    cnt += posted_events.length();
    cnt += posted_functors.length();
    if (wants_idle()) cnt += 1;
    return cnt;
  }

  bool view::on_idle() {
    // assert(!dismissing);
    if (dismissing) return true;

    current_view_state _(this);
    return handle_on_idle();
  }

  // returns true if there are no pending idle tasks
  bool view::handle_on_idle()
  {
    if (invalid_style_root) {
      helement t = invalid_style_root; invalid_style_root = nullptr;
      t->resolve_styles(*this);
    }

    bool empty_queues = items_in_idle_queue() == 0;

    if (!empty_queues) 
    {
      updater of(this, 0, false, true);
      process_posted_things(true);
      if (exec_idle())
        request_idle();
    }
    return !items_in_idle_queue();
  }


  uint_ptr get_timer_uid() {
    static uint_ptr timer_cnt = 1000;
    if (++timer_cnt < 1000) timer_cnt = 1001;
    return timer_cnt;
  }

  bool view::handle_view_timer(timer_id& id, TIMER_KIND kind)
  {
    if (kind == INTERNAL_TIMER && id == MOUSE_LEAVE_TIMER_ID)
    {
      if (is_mouse_inside) {
        point cp = cursor_pos();
        point screen_cursor_pos = cp + client_screen_pos();
        if (!is_at_position(screen_cursor_pos)) {
          handle_mouse(MOUSE_LEAVE, mouse_buttons, 0, cp);
          is_mouse_inside = false;
          stop_timer(nullptr, MOUSE_LEAVE_TIMER_ID, INTERNAL_TIMER);
          stop_timer(nullptr, MOUSE_IDLE_TIMER_ID, INTERNAL_TIMER);
        }
      }
      return true;
    }
    else if (kind == INTERNAL_TIMER && id == MOUSE_IDLE_TIMER_ID)
    {
      if (is_mouse_inside) {
        stop_timer(nullptr, MOUSE_IDLE_TIMER_ID, INTERNAL_TIMER);
        handle_mouse(MOUSE_IDLE, 0, 0, mouse_pos);
      }
      return true;
    }
    else if (kind == INTERNAL_TIMER && id == MOUSE_TICK_TIMER_START_ID)
    {
      stop_timer(nullptr, MOUSE_TICK_TIMER_START_ID, INTERNAL_TIMER);
      if(handle_mouse(MOUSE_TICK, mouse_buttons, 0, mouse_pos))
        start_timer(nullptr, MOUSE_TICK_TIMER_TICK_MS, MOUSE_TICK_TIMER_TICK_ID, INTERNAL_TIMER);
      return true;
    }
    else if (kind == INTERNAL_TIMER && id == MOUSE_TICK_TIMER_TICK_ID)
    {
      if(handle_mouse(MOUSE_TICK, mouse_buttons, 0, mouse_pos))
        start_timer(nullptr, MOUSE_TICK_TIMER_TICK_MS, MOUSE_TICK_TIMER_TICK_ID, INTERNAL_TIMER);
      return true;
    }
    return false;
  }

  bool view::on_timer(uint_ptr uid)
  {
    if (uid < 1000)
      return false;

    timer_def td;
    int       idx = -1;

    FOREACH(n, element_timers) {
      timer_def &t = element_timers[n];
      if (t.uid == uid) {
        td  = t;
        idx = n;
        goto FOUND;
      }
    }
    return false;

  FOUND:

    helement el = td.blk;

    if (!el) 
    {
        if (td.is_script_timer()) {
          set_timer(td.uid, 0, td.sys_id);
          element_timers.remove(idx);
          return false;
        }
        else
          return handle_view_timer(td.id, td.kind);
    }

    if (pdoc == 0) return false;

    updater of(this, 0, false, true);

    // suspend the timer
    element_timers.remove(idx);
    set_timer(td.uid, 0, td.sys_id);

    if (el->pview() != this)
      return false;

    if (!el->on_timer(*this, td))
      return false;

    // restart/resume the timer
    set_timer(td.uid, td.ms, td.sys_id);
    td.until = get_ticks() + td.ms;
    element_timers.push(td);

    return true;
  }

  timer_def& view::start_timer(element *b, uint ms, timer_id id, TIMER_KIND kind) {
    stop_timer(b, id, kind);
    timer_def td;
    assert(b || (kind == INTERNAL_TIMER));
    td.blk  = b;
    td.id   = id;
    td.uid  = get_timer_uid();
    td.ms   = ms;
    td.until = ms + get_ticks();
    td.kind = kind;
    assert(td.uid);
    set_timer(td.uid, ms, td.sys_id);
    element_timers.push(td);
#ifdef _DEBUG
    assert(element_timers.size() < 100); // too many timers?  
#endif //  _DEBUG
    return element_timers.last();
  }

  void view::stop_timer(element *b, timer_id id, TIMER_KIND kind) {
    // timer_def td;
    // int idx = -1;
    //if (id == MOUSE_TICK_TIMER_TICK_ID && kind == INTERNAL_TIMER)
    //  id = id;

    FOREACH(n, element_timers) {
      timer_def &t = element_timers[n];
      if (t.blk == b && t.kind == kind && t.id == id) {
        // idx = n;
        // td = t;
        set_timer(t.uid, 0, t.sys_id);
        element_timers.remove(n);
        return;
      }
    }
  }

  void view::stop_timer_if(element * b, function<bool(timer_id id, TIMER_KIND kind)> pred) {
    FOREACH(n, element_timers) {
      timer_def &t = element_timers[n];
      if (t.blk == b && pred(t.id, t.kind)) {
        set_timer(t.uid, 0, t.sys_id);
        element_timers.remove(n);
      }
    }
  }


  void view::check_timers_overdue_in_all_views() {
    THREAD_LOCAL static uint check_no = 0;
    if(++check_no % 16 == 0) // we do this check each 16th paint event,
                             // this function gets called from paint handler
                             // to prevent timers queue stall on frequent paint calls (like animation)
      for (int i = 0; i < view::all.size(); ++i) {
        auto vptr = view::all.elements()[i];
        if (vptr)
#if defined(WINDOWS) && !defined(WINDOWLESS)
          if(GetWindowThreadProcessId(vptr->get_hwnd(),NULL) == GetCurrentThreadId())
#endif
            vptr->check_timers_overdue();
      }
  }

  void view::check_timers_overdue()
  {
    updater of(this, 0, false, true);
    uint cutoff = get_ticks();
    for( uint n = 0; n < element_timers.length(); ++n ) {
      timer_def &t = element_timers[n];
      if (t.until <= cutoff)
        on_timer(t.uid);
    }
    on_idle();
  }

  void view::stop_all_timers() {
    while (element_timers.size()) {
      timer_def td = element_timers.pop();
      set_timer(td.uid, 0, td.sys_id);
    }
  }

  void view::notify_media_change() {
    if (notifying_media_change)
      return;
    notifying_media_change = true;
    on_media_changed();
    notifying_media_change = false;
    //this->post(delegate(this, &view::on_media_changed), true);
  }


  element *get_enabled_mouse_target(view &v, element *b) {
    if (!v.is_enabled()) return nullptr;
    element *topmost = 0;
    element *t       = b;
    while (t) {
      if (t->animator && t->animator->is_of_type<scroll_animator>()) {
        b = t;
        break;
      }
      t = t->ui_parent(v);
    }
    while (b) {
      if (!topmost) topmost = b;
      if (b->state.disabled()) topmost = 0;
      b = b->ui_parent(v);
    }
    return topmost;
  }

  void view::check_mouse(bool always) {
#ifdef USE_TOUCH
    if (is_touched.is_defined() && !is_touched)
      return;
#endif
    if (!is_mouse_inside || in_ui_replacement)
      return;
    
    if (always || (mouse_msg != MOUSE_MOVE && mouse_msg != MOUSE_CHECK)) {
      handle_mouse(MOUSE_CHECK, mouse_buttons, last_alts_keys_state, mouse_pos);
    }
  }


  bool view::on_mouse(int msg, uint buttons, uint alts, gool::point pt)
  {
#ifdef USE_TOUCH
    if (msg == MOUSE_TOUCH_START) {
        //is_touch_active = TRUE;
        //dbg_printf("TOUCH ON");
    }
    else if (msg == MOUSE_TOUCH_END) {
        //is_touch_active = FALSE;
        //dbg_printf("TOUCH OFF");
    }
    else 
#endif
    if (msg == MOUSE_MOVE)
    {
#if defined(WINDOWS)
      start_timer(nullptr, GetDoubleClickTime() + 2, MOUSE_IDLE_TIMER_ID, INTERNAL_TIMER);
#else
      start_timer(nullptr, MOUSE_IDLE_TIMER_MS, MOUSE_IDLE_TIMER_ID, INTERNAL_TIMER);
#endif

#if defined(OSX)

      bool inside = is_at_position( pt + client_screen_pos() )
                    || mouse_capture_element.ptr()
                    || (buttons == MAIN_BUTTON); // OSX "auto captures" pressed mouse
      if (  !is_mouse_inside && inside)
      {
        is_mouse_inside = true;
        start_timer(nullptr, MOUSE_LEAVE_TIMER_MS, MOUSE_LEAVE_TIMER_ID, INTERNAL_TIMER);
        handle_mouse(MOUSE_ENTER, buttons, alts, pt);
      }
      else if ( is_mouse_inside && !inside)
      {
        is_mouse_inside = false;
        stop_timer(nullptr, MOUSE_LEAVE_TIMER_ID, INTERNAL_TIMER);
        return handle_mouse(MOUSE_LEAVE, buttons, alts, pt);
      }
      else if (!inside)
      {
        return false;
      }
#elif defined(WINDOWS)

      if (  !is_mouse_inside )
      {
        is_mouse_inside = true;
        start_timer(nullptr, MOUSE_LEAVE_TIMER_MS, MOUSE_LEAVE_TIMER_ID, INTERNAL_TIMER);
        handle_mouse(MOUSE_ENTER, buttons, alts, pt);
      }

#endif
    }

#if defined(LINUX)
    // gtk generates MOUSE_ENTER / MOUSE_LEAVE out of the box
    if (msg == MOUSE_ENTER)
    {
      is_mouse_inside = true;
      return handle_mouse(MOUSE_ENTER, buttons, alts, pt);
    }
    else if (msg == MOUSE_LEAVE)
    {
      is_mouse_inside = false;
      return handle_mouse(MOUSE_LEAVE, buttons, alts, pt);
    }
#endif

    bool r = handle_mouse(msg, buttons, alts, pt);

    if (msg == MOUSE_UP)
    {
      stop_timer(nullptr, MOUSE_TICK_TIMER_START_ID, INTERNAL_TIMER);
      stop_timer(nullptr, MOUSE_TICK_TIMER_TICK_ID, INTERNAL_TIMER);
#ifdef USE_TOUCH
      if ((alts & ALT_TOUCH) != 0) {
        is_touched = FALSE;
        handle_mouse(html::MOUSE_LEAVE, 0, alts, pt);
        is_mouse_inside = false;
      }
#endif
    }
    else if (msg == MOUSE_DOWN)
    {
      stop_timer(nullptr, MOUSE_TICK_TIMER_TICK_ID, INTERNAL_TIMER);
      start_timer(nullptr, MOUSE_TICK_TIMER_START_MS, MOUSE_TICK_TIMER_START_ID, INTERNAL_TIMER);
    }

    return r;
  }


  bool view::handle_mouse(int msg, uint buttons, uint alts, gool::point pt) {

    if(msg != MOUSE_CHECK)
      last_alts_keys_state = alts;
    //else
    //  printf("MOUSE_CHECK\n");

    if (pdoc == 0 || pdoc->pview() != this) return false;

    // handle<cursor> new_cursor = cursor::system(cursor_auto);

    bool is_real_mouse_msg = false;
    switch (msg) {
      case MOUSE_DCLICK: {
        this->mouse_dblclick_time = get_ticks();
        is_real_mouse_msg = true;
      } break;
      case MOUSE_DOWN:
        is_real_mouse_msg = is_real_mouse_msg;
      case MOUSE_ENTER:
      case MOUSE_LEAVE:
      case MOUSE_MOVE:
      case MOUSE_UP:
        is_real_mouse_msg = true;
      default:
        break;
    }

    helement b = 0;

    if (is_real_mouse_msg) {
      mouse_pos = pt;
      mouse_buttons = buttons;
    }
    mouse_msg     = msg;

    event_mouse evt(nullptr, 0, pt, buttons, alts);
    evt.cursor = cursor::system(0);

    helement over_element    = mouse_over_element;
    helement capture_element = mouse_capture_element;
    if (!capture_element) capture_element = event_capture_element;

    if (capture_element)
    {
      if (mouse_capture_strict ||
          capture_element->get_style(*this)->display == display_inline)
        b = over_element = capture_element;
      else {
        b = find_element(pt);
        if (b && !b->belongs_to(*this, capture_element, true)) b = 0;
        // b = capture_element->find_element(*this, pt -
        // capture_element->view_pos() + capture_element->pos);
      }
      if (!b) b = capture_element;
      if (b) evt.cursor = b->get_style(*this)->cursor;
    }
    else
    {
      if (msg == MOUSE_CHECK || msg == MOUSE_TICK || msg == MOUSE_IDLE) {
        // b = over_element;
        b = element_under_cursor(mouse_pos);
      } else if (msg == MOUSE_LEAVE)
        b = 0;
      else { // if( msg != MOUSE_LEAVE ) {
        b = find_element(pt);
#ifdef DEBUG
        //b->dbg_report("mouse");
#endif
      }
#if defined(WINDOWS)
      {
        event_mouse mhtevt;
        mhtevt.cmd          = MOUSE_HIT_TEST;
        mhtevt.button_state = 0;
        mhtevt.pos_view = mhtevt.pos = pt;
        mhtevt.target                = doc();
        if (traverse_mouse(doc(), mhtevt) && mhtevt.data.to_int() != HTCLIENT) {
          evt.cursor = nullptr;
        }
      }
#endif
    }

    if (msg == MOUSE_UP) {
      set_capture(0); // this assumes that capture is used while handling
                      // MOUSE_DOWN only
    }

    b = get_enabled_mouse_target(*this, b);

#ifdef DEBUG
    //if(b)
    //   b->dbg_report("enabled mouse target");
#endif

    evt.target = b;

    updater of(this, b, !is_real_mouse_msg /*it is passive in this case*/, is_real_mouse_msg);

    if (b != over_element || (b && !b->state.hover()) /*&& is_real_mouse_msg*/) {

      mouse_over_element = b;

      helement p = element::find_common_ui_parent( *this, b, over_element);

      // 1) notify full tree
      // evt.cmd = MOUSE_LEAVE | EVENT_SINKING;
      // traverse_mouse_child_parent(over_element,0,evt);
      // 2) notify only those which really loosing mouse

      if (!evt.target) evt.target = doc();

      evt.cmd = MOUSE_LEAVE | EVENT_SINKING;
      traverse_mouse_parent_child(over_element, p, evt);
      evt.cmd = MOUSE_LEAVE;
      traverse_mouse_child_parent(over_element, p, evt);

      if (msg != MOUSE_LEAVE) {
        // 1) ...
        evt.cmd = MOUSE_ENTER | EVENT_SINKING;
        traverse_mouse_parent_child(b, p, evt);
        // 2) ...
        evt.cmd = MOUSE_ENTER;
        traverse_mouse_child_parent(b, p, evt);
        over_element = b;
      } else // msg == MOUSE_LEAVE
      {
        over_element = 0;
      }
      // dbg_printf("MOUSE_ENTER / LEAVE\n");
    }
    /*else if(b) {
      if(b->state.hover())
        b->dbg_report("hover:hover did not change");
      else
        b->dbg_report("NO hover:hover did not change");
    }*/

    evt.cmd = msg;

    if (msg == MOUSE_CHECK) {
      // evt.cursor = cursor::system(0);
      if (over_element) {
        evt.cursor       = nullptr;
        evt.button_state = 0;
        traverse_mouse(over_element, evt);
        if(evt.cursor)
          set_cursor(evt.cursor);
      }
      return true;
    }

    bool r = false;

    switch (msg) {
    case MOUSE_MOVE:
#if defined(OSX)
      if (evt.is_control() && evt.is_point_button()) {
        r = true;
        break;
      }
#endif

#if NOT_YET
      if (!drag_drv && evt.is_point_button() &&
          (gool::distance(evt.pos_view, mouse_down_pos) > DD_THRESHOLD) &&
          evt.target && evt.target->state.pressed()) {
        mouse_down_pos       = evt.pos_view;
        element *t_draggable = get_draggable(*this, evt.target);
        if (t_draggable) {
          event_mouse d_evt = evt;
          exec_drag(t_draggable, d_evt);
          return true;
        }
      }
#endif
      if (evt.is_point_button() && !mouse_drag_request_issued &&
          (gool::distance(evt.pos_view, mouse_down_pos) >
           pixels_per_dip(size(MOUSE_DRAG_THRESHOLD)).x)) {
        event_mouse dr_evt(evt);
        dr_evt.cmd = MOUSE_DRAG_REQUEST;
        r = traverse_mouse(over_element, dr_evt);
        mouse_drag_request_issued = true;
      }
      if (!r)
        r = traverse_mouse(over_element, evt);
#if NOT_YET
      if (!drag_drv && r && evt.dragging && evt.dragging_mode) {
        exec_drag(evt.dragging, evt);
        return true;
      }
#endif

      if (evt.is_point_button())
        on_idle(); // seems like MOUSE_MOVE has higher priority than WM_TIMER

      break;

    case MOUSE_MOVE | EVENT_HANDLED: // special case of the updater
      // if(evt.cursor ) new_cursor = evt.cursor;
      if (evt.cursor) set_cursor(evt.cursor);
      return true;

    case MOUSE_UP:
#if defined(OSX)
      if (evt.is_control() && evt.is_point_button()) {
        r = true;
        break;
      }
#endif

      r = traverse_mouse(over_element, evt);
      if (r) {
        mouse_down_element = 0;
        if (evt.cursor) set_cursor(evt.cursor);
        return true;
      }
      if (mouse_down_element && evt.is_point_button()) {
        evt.target = element::find_base(mouse_down_element, mouse_over_element);
        mouse_down_element = 0;
        evt.cmd            = MOUSE_CLICK;
        if (evt.target) traverse_mouse(evt.target, evt);
      }
#if defined(OSX) || defined(LINUX)
      if (evt.is_prop_button()) r = on_context_menu(evt.pos_view);
#endif
      mouse_down_element = 0;
      break;
      // if(r)
      //  return true;
      // break;

    case MOUSE_DCLICK: {
      helement prev_focus = focus_element;
      r = traverse_mouse(over_element, evt);
      if (prev_focus == focus_element && !r) // focus_element was not updated
        set_focus(over_element, BY_MOUSE);
      break; // return r;
    }
    case MOUSE_TCLICK: {
      r = traverse_mouse(over_element, evt);
      break; // return r;
    }

    case MOUSE_DOWN: {

#if defined(OSX)
      if (evt.is_control() && evt.is_point_button() &&
          on_context_menu(evt.pos_view)) {
        r = true;
        break;
      }
#endif
      mouse_down_on_caption     = false;
      mouse_down_element        = evt.is_point_button() ? over_element : 0;
       
      auto prev_mouse_down_pos = mouse_down_pos;
        
      mouse_down_pos            = evt.pos_view;
      mouse_drag_request_issued = false;
      helement prev_focus       = focus_element;

      dont_change_focus = false;

      //#if defined(OSX) || defined(LINUX)
      if (windows.size() > 0) {
        element *target =
            find_element(pt); // over_element? over_element : doc();
        if (!target) target = doc();
        if (target) {
          target = target->get_windowed_container(*this, true);
          if (target) {
            //target->dbg_report("windowed container of popup");
            event_behavior t(target, target, CLOSE_POPUP_TREE, 0);
            send_behavior_event(t); // NOTE: not post_behavior_event()!
          }
        }
      }
      //#endif
        
#if !defined(OSX)
      
      if (
        ((get_ticks() - mouse_dblclick_time) < dblclick_timeout()) 
        && gool::distance(evt.pos_view, prev_mouse_down_pos) <= pixels_per_dip(size(MOUSE_DRAG_THRESHOLD)).x )
      {
        mouse_dblclick_time = 0;
        evt.cmd = MOUSE_TCLICK;
        r = traverse_mouse(over_element, evt);
        if (!r) {
          evt.cmd = MOUSE_DOWN;
          r = traverse_mouse(over_element, evt);
        }
      }
      else
#endif
      {
        r = traverse_mouse(over_element, evt);
      }

      if (dont_change_focus)
        ;
      else if (prev_focus != focus_element) // focus_element was not updated
      {
        set_focus(over_element, BY_MOUSE);
      }

#if NOT_YET
      if (r && evt.dragging && evt.dragging_mode) {
        exec_drag(evt.dragging, evt);
        return true;
      }
#endif
    } break; // return r;

    case MOUSE_WHEEL: {
      mouse_buttons   = 0;
      /*uint wheel_tick = get_ticks();
      if (last_wheel_direction != evt.button_state)
          last_wheel_ticks.clear();
      last_wheel_direction = evt.button_state;
      last_wheel_ticks.push(wheel_tick);

      // be aware - black russian magic!

      uint d = 1500;
      for (uint i = 1; i < last_wheel_ticks.capacity(); ++i) {
        if (i >= last_wheel_ticks.length())
          d += 1500u;
        else
          d += last_wheel_ticks[i] - last_wheel_ticks[i - 1];
      }
      d /= (uint)last_wheel_ticks.capacity();

      evt.wheel_frequency = limit(d / 400 + 1u, 1u, 16u);*/

      r = traverse_mouse(over_element, evt);
      return r; // do not change cursor on mouse_wheel
      //break;
    }
    case MOUSE_LEAVE: {
      traverse_mouse(over_element, evt);
      behavior_h pc = ground_behavior;
      while (pc) {
        if (pc->will_handle(HANDLE_MOUSE) && pc->on(*this, b, evt))
          evt.cmd |= EVENT_HANDLED;
        behavior_h next = pc->next;
        pc              = next;
      }
    }
      return (evt.cmd & EVENT_HANDLED) != 0;
    case MOUSE_IDLE: r = traverse_mouse(over_element, evt); break;
    case MOUSE_TICK: r = traverse_mouse(over_element, evt); break;
#ifdef USE_TOUCH
    case MOUSE_TOUCH_START: r = traverse_mouse(over_element, evt); break;
    case MOUSE_TOUCH_END: r = traverse_mouse(over_element, evt); break;
#endif
    }

    // if (evt.cursor) {
    //  new_cursor = evt.cursor;
    //  set_cursor(new_cursor);
    //}
    if (evt.cursor && !is_dragging)
      set_cursor(evt.cursor);

    assert(evt.get_ref_count() == 0);

    return r;
  }

  bool view::on_key(int cmd, int code, int alts) {

    last_alts_keys_state = alts;

    if ((alts & ALT_ALT) && (cmd == KEY_DOWN))
      alt_key_down_handled = false;

    if (!pdoc || !is_enabled()) return false;

    if (cmd == KEY_CHAR && code < ' ')
      //return on_key_nothandled(cmd, code, alts);
      return false;

    updater of(this, 0, false, true);

    element *pfocus = focus_element ? focus_element : doc();
    if (pfocus && pfocus->pview() != this) focus_element = pfocus = 0;

    bool handled = false;

    if (pfocus) {
      // ATTN: logic changed, keyboard events are being delivered to
      // :focus/:current element now
      element *current = get_current(pfocus);
      if (current) current = get_enabled(*this, current);
      if (current) pfocus = current;
      event_key            evt(pfocus, cmd, code, alts);
      traverser<event_key> trv(*this);
      if (trv.traverse(pfocus, evt)) {
        handled = true;
        goto DONE;
      }
    }
    handled = on_key_nothandled(cmd, code, alts);
  DONE:
    if ((alts & ALT_ALT) && (cmd == KEY_DOWN))
      alt_key_down_handled = handled;
    if ((alts & ALT_ALT) && (cmd == KEY_CHAR)) //-- Portugese ALT+2 => @
      return alt_key_down_handled || handled;
    return handled;
  }

  bool view::on_key_nothandled(int cmd, int code, int alts) {
    if (!doc()) return false;
    if (cmd != KEY_DOWN) return false;

    switch (code) {

    case KB_TAB:
      if (alts & (ALT_ALT | ALT_CONTROL)) return false;
      return set_focus_on((alts & ALT_SHIFT) ? FOCUS_PREV : FOCUS_NEXT);
    case KB_ESCAPE: {
      element *b =
          find_first(*this, doc(), WCHARS("[role='cancel-button']"), true);
      if (b) {
        method_params params;
        params.method_id = DO_CLICK;
        return call_behavior_method(b, &params);
      }
    } break;
    case KB_RETURN: {
      element *b =
          find_first(*this, doc(), WCHARS("[role='default-button']"), true);
      if (b) {
        method_params params;
        params.method_id = DO_CLICK;
        return call_behavior_method(b, &params);
      }
    } break;
    }

    // accesskey handling.
    ustring keyname = get_key_name(code, alts);
    if (keyname.is_undefined()) return false;

    ustring selector =
        ustring::format(W("[accesskey~='%s']:not(:disabled)"), keyname.c_str());

    helement hot_key_element =
        find_first(*this, doc(), selector, false /*only_visible*/, true);

    if (hot_key_element.is_null()) return false;

    // ask child activation (e.g. behavior menu)
    helement hot_element_parent = hot_key_element->get_owner();
    if (hot_element_parent) {
      event_behavior evt(hot_key_element, hot_key_element, ACTIVATE_CHILD, 0);
      if (send_behavior_event(evt)) return true;
    }

    if (hot_key_element->is_visible(*this)) {
      method_params prm;
      prm.method_id = DO_CLICK;
      if (call_behavior_method(hot_key_element, &prm)) return true;

      // no luck, just navigate to it
      this->set_focus(hot_key_element, html::BY_KEY_SHORTCUT);
      if (this->focus_element != hot_key_element)
        this->ensure_visible(hot_key_element, false);

      return true;
    }

    return false;
  }

  struct key_code_name {
    uint  code;
    chars name;
  };

  static key_code_name key_code_names[] = {
      {KB_1, CHARS("1")},
      {KB_2, CHARS("2")},
      {KB_3, CHARS("3")},
      {KB_4, CHARS("4")},
      {KB_5, CHARS("5")},
      {KB_6, CHARS("6")},
      {KB_7, CHARS("7")},
      {KB_8, CHARS("8")},
      {KB_9, CHARS("9")},
      {KB_0, CHARS("0")},
      {KB_A, CHARS("A")},
      {KB_S, CHARS("S")},
      {KB_D, CHARS("D")},
      {KB_F, CHARS("F")},
      {KB_H, CHARS("H")},
      {KB_G, CHARS("G")},
      {KB_Z, CHARS("Z")},
      {KB_X, CHARS("X")},
      {KB_C, CHARS("C")},
      {KB_V, CHARS("V")},
      {KB_B, CHARS("B")},
      {KB_Q, CHARS("Q")},
      {KB_W, CHARS("W")},
      {KB_E, CHARS("E")},
      {KB_R, CHARS("R")},
      {KB_Y, CHARS("Y")},
      {KB_T, CHARS("T")},
      {KB_O, CHARS("O")},
      {KB_U, CHARS("U")},
      {KB_I, CHARS("I")},
      {KB_P, CHARS("P")},
      {KB_L, CHARS("L")},
      {KB_J, CHARS("J")},
      {KB_K, CHARS("K")},
      {KB_N, CHARS("N")},
      {KB_M, CHARS("M")},

      {KB_BACK, CHARS("BACK")},
      {KB_BACK, CHARS("BACKSPACE")},
      {KB_RETURN, CHARS("RETURN")},
      {KB_RETURN, CHARS("ENTER")},

      {KB_ESCAPE, CHARS("ESCAPE")},
      {KB_CAPITAL, CHARS("CAPSLOCK")},
      {KB_NUMLOCK, CHARS("NUMLOCK")},
      {KB_SCROLL, CHARS("SCROLLLOCK")},
      {KB_CONTROL, CHARS("CONTROL")},
      {KB_SHIFT, CHARS("SHIFT")},

      {KB_SPACE, CHARS("SPACE")},
      {KB_PRIOR, CHARS("PRIOR")},
      {KB_NEXT, CHARS("NEXT")},
      {KB_END, CHARS("END")},
      {KB_HOME, CHARS("HOME")},
      {KB_LEFT, CHARS("LEFT")},
      {KB_UP, CHARS("UP")},
      {KB_RIGHT, CHARS("RIGHT")},
      {KB_DOWN, CHARS("DOWN")},
      {KB_INSERT, CHARS("INSERT")},
      {KB_DELETE, CHARS("DELETE")},

      {KB_NUMPAD0, CHARS("NUMPAD0")},
      {KB_NUMPAD1, CHARS("NUMPAD1")},
      {KB_NUMPAD2, CHARS("NUMPAD2")},
      {KB_NUMPAD3, CHARS("NUMPAD3")},
      {KB_NUMPAD4, CHARS("NUMPAD4")},
      {KB_NUMPAD5, CHARS("NUMPAD5")},
      {KB_NUMPAD6, CHARS("NUMPAD6")},
      {KB_NUMPAD7, CHARS("NUMPAD7")},
      {KB_NUMPAD8, CHARS("NUMPAD8")},
      {KB_NUMPAD9, CHARS("NUMPAD9")},
      {KB_MULTIPLY, CHARS("MULTIPLY")},
      {KB_ADD, CHARS("ADD")},

      {KB_SUBTRACT, CHARS("SUBTRACT")},
      {KB_DECIMAL, CHARS("DECIMAL")},
      {KB_DIVIDE, CHARS("DIVIDE")},
      {KB_F1, CHARS("F1")},
      {KB_F2, CHARS("F2")},
      {KB_F3, CHARS("F3")},
      {KB_F4, CHARS("F4")},
      {KB_F5, CHARS("F5")},
      {KB_F6, CHARS("F6")},
      {KB_F7, CHARS("F7")},
      {KB_F8, CHARS("F8")},
      {KB_F9, CHARS("F9")},
      {KB_F10, CHARS("F10")},
      {KB_F11, CHARS("F11")},
      {KB_F12, CHARS("F12")},
  };

  // get symbolic name of the key combination
  ustring get_key_name(uint vk_code, uint alts) {
    if (alts == ALT_SHIFT) return ustring(); // shift+s alone is not a hotkey

    if (vk_code == KB_MENU) return ustring();

    ustring buffer;

    if ((alts & ALT_SHORTCUT) != 0)
      buffer = W("^"); // ctrl
    else if ((alts & ALT_ALT) == 0)
      buffer = W("!"); // key without ALT modificator (as it is).
    if ((alts & ALT_SHIFT) != 0)
      buffer += W("_"); // shift

    for (auto p : key_code_names) {
      if (p.code == vk_code) {
        buffer += ustring(p.name);
        return buffer;
      }
    }

    return ustring();
  }

  uint get_key_code_by_name(chars vk_name) {
    for (auto p : key_code_names) {
      if ( lexical::ci::eq(p.name,vk_name)) {
        return p.code;
      }
    }
    return 0;
  }

  // system shell drag-n-drop support

  bool view::handle_exchange_event(exchange_cmd     cmd,
                                   unsigned long &  clipboard_dd_modes,
                                   clipboard::data *pd, element *b, point p,
                                   element *src) {
    // if( !b)
    //  return false;

    current_view_state _(this);

    // if(mouse_over_element && !mouse_over_element->doc())
    //  mouse_over_element = 0;

    auto find_x_drop_target = [&](element* b) -> element* {
      element* accepted_by = nullptr;
      event_exchange evt(b, X_WILL_ACCEPT_DROP, (DD_MODE)clipboard_dd_modes, pd, p, src);
      // evt.dd_source = this->system_dragging_source_element;
      evt.cmd = X_WILL_ACCEPT_DROP | EVENT_SINKING;
      traverse_mouse_parent_child(b, 0, evt,&accepted_by);
      evt.cmd = X_WILL_ACCEPT_DROP;
      traverse_mouse_child_parent(b, 0, evt,&accepted_by);
      return evt.is_canceled() ? nullptr : accepted_by;
    };

    if (cmd == X_DRAG_ENTER) {

      helement target = find_x_drop_target(b);
      //if (!target)
      //  return false;
      event_exchange evt(target, X_DRAG_ENTER, (DD_MODE)clipboard_dd_modes, pd, p, src);
      // evt.dd_source = this->system_dragging_source_element;
      // 1) ...
      evt.cmd = X_DRAG_ENTER | EVENT_SINKING;
      traverse_mouse_parent_child(target, 0, evt);
      // 2) ...
      evt.cmd = X_DRAG_ENTER;
      traverse_mouse_child_parent(target, 0, evt);
      x_drop_target_element = target;
    } else if (cmd == X_DRAG_LEAVE) {
      //if (!x_drop_target_element)
      //  return false;
      event_exchange evt(x_drop_target_element, X_DRAG_LEAVE,(DD_MODE)clipboard_dd_modes, pd, p, src);
      // evt.dd_source = this->system_dragging_source_element;
      evt.cmd = X_DRAG_LEAVE | EVENT_SINKING;
      traverse_mouse_parent_child(x_drop_target_element, 0, evt);
      evt.cmd = X_DRAG_LEAVE;
      traverse_mouse_child_parent(x_drop_target_element, 0, evt);
      x_drop_target_element = 0;
    } else {
      helement target = find_x_drop_target(b);
      if (target != x_drop_target_element) {
        // evt.dd_source = this->system_dragging_source_element;
        element *parent = element::find_common_ui_parent(*this, b, mouse_over_element);

        event_exchange evt(target, X_DRAG, (DD_MODE)clipboard_dd_modes, pd, p, src);
        // 1) notify full tree
        // evt.cmd = MOUSE_LEAVE | EVENT_SINKING;
        // traverse_mouse_child_parent(mouse_over_block,0,evt);
        // 2) notify only those which really loosing mouse
        evt.cmd = X_DRAG_LEAVE | EVENT_SINKING;
        traverse_mouse_parent_child(x_drop_target_element, parent, evt);
        evt.cmd = X_DRAG_LEAVE;
        traverse_mouse_child_parent(x_drop_target_element, parent, evt);

        // 1) ...
        evt.cmd = X_DRAG_ENTER | EVENT_SINKING;
        traverse_mouse_parent_child(target, parent, evt);
        // 2) ...
        evt.cmd = X_DRAG_ENTER;
//#ifdef _DEBUG
//        b->dbg_report("X_DRAG_ENTER");
//#endif
        traverse_mouse_child_parent(target, parent, evt);
        x_drop_target_element = target;
      }
      if (x_drop_target_element) {
        event_exchange evt(x_drop_target_element, cmd, (DD_MODE)clipboard_dd_modes, pd, p, src);
        if (_traverse_mouse(x_drop_target_element, evt)) {
          // clipboard_dd_modes = evt.dd_modes;
          return true;
        }
      }
    }
    return false;
  }

  bool view::on_drop(unsigned long &clipboard_dd_modes, clipboard::data *pd,
                     point pt, element *src) {
    helement b = find_element(pt);
    if (b &&
        handle_exchange_event(X_DROP, clipboard_dd_modes, pd, b, pt, src)) {
      // WRONG: set_focus(b,BY_MOUSE);
      return true;
    }
    return false;
  }

  bool view::on_drag_enter(unsigned long &  clipboard_dd_modes,
                           clipboard::data *pd, point pt, element *src) {
    // element* b = doc();
    element *b = find_element(pt);
    return handle_exchange_event(X_DRAG_ENTER, clipboard_dd_modes, pd, b, pt, src);
  }

  bool view::on_drag_over(unsigned long &  clipboard_dd_modes,
                          clipboard::data *pd, point pt, element *src) {
    element *b = find_element(pt);
    return handle_exchange_event(X_DRAG, clipboard_dd_modes, pd, b, pt, src);
  }
  void view::on_drag_leave(element *src) {
    unsigned long clipboard_dd_modes = 0;
    handle_exchange_event(X_DRAG_LEAVE, clipboard_dd_modes, nullptr, nullptr,
                          point(), src);
  }
  void view::on_drag_cancel(element *src) {
    unsigned long clipboard_dd_modes = 0;
    handle_exchange_event(X_DRAG_CANCEL, clipboard_dd_modes, nullptr, nullptr,
                          point(), src);
  }

  namespace behavior {

    static bool belongs_to_caption(element *el) {
      while (el) {
        ustring role = el->attr_role();
        if (role == WCHARS("window-caption"))
          return true;
        if (el->handles<event_mouse>()) return false;
        el = el->parent;
      }
      return false;
    }

#ifndef  HTCLIENT
  #define HTERROR             (-2)
  #define HTTRANSPARENT       (-1)
  #define HTNOWHERE           0
  #define HTCLIENT            1
  #define HTCAPTION           2
  #define HTSYSMENU           3
  #define HTGROWBOX           4
  #define HTSIZE              HTGROWBOX
  #define HTMENU              5
  #define HTHSCROLL           6
  #define HTVSCROLL           7
  #define HTMINBUTTON         8
  #define HTMAXBUTTON         9
  #define HTLEFT              10
  #define HTRIGHT             11
  #define HTTOP               12
  #define HTTOPLEFT           13
  #define HTTOPRIGHT          14
  #define HTBOTTOM            15
  #define HTBOTTOMLEFT        16
  #define HTBOTTOMRIGHT       17
  #define HTBORDER            18
  #define HTREDUCE            HTMINBUTTON
  #define HTZOOM              HTMAXBUTTON
  #define HTSIZEFIRST         HTLEFT
  #define HTSIZELAST          HTBOTTOMRIGHT
  #define HTOBJECT            19
  #define HTCLOSE             20
  #define HTHELP              21
#endif

#define HTBORDER_FIRST_CODE HTLEFT
#define HTBORDER_LAST_CODE HTBOTTOMRIGHT

    static bool belongs_to_window_ctl(element *el, wchars ctl_role) {
      while (el) {
        ustring role = el->attr_role();
        if (role == ctl_role) return true;
        if (el->handles<event_mouse>()) return false;
        el = el->parent;
      }
      return false;
    }

    /*static element *get_body(view &v) {
      return find_first(v, v.doc(), WCHARS("body"), true);
    }*/

    size window_frame_ctl::border_band(view &v, element *self) {
      if (self) {
        size_v border = self->atts.get_size("window-resizable");
        if (border.is_defined()) {
          int px = pixels(v, self, border).width();
          return size(px, px);
        }
      }
#ifdef SM_CXSIZEFRAME
      return size(v.get_system_metrics(SM_CXSIZEFRAME), v.get_system_metrics(SM_CYSIZEFRAME));
#else
      return v.pixels_per_dip(size(4, 4));
#endif
    }

    uint window_frame_ctl::hittest(view &v, element *self, point pos_view) {

      if (v.get_resizeable() && v.get_window_state() != WINDOW_FULL_SCREEN) {
        if (!v.doc()) return HTCLIENT;

        v.doc()->commit_measure(v);
        rect body_rc = v.doc()->content_box(v, element::TO_VIEW);

        rect rc = v.client_dim();
        rc <<= border_band(v,self);
        body_rc &= rc;

        if (!body_rc.contains(pos_view)) {
          if (pos_view.y < body_rc.top() + 10) {
            if (pos_view.x < body_rc.left() + 10) return HTTOPLEFT;
            if (pos_view.x > body_rc.right() - 10) return HTTOPRIGHT;
          }
          else if (pos_view.y > body_rc.bottom() - 10) {
            if (pos_view.x < body_rc.left() + 10) return HTBOTTOMLEFT;
            if (pos_view.x > body_rc.right() - 10) return HTBOTTOMRIGHT;
          }

          if (pos_view.y < body_rc.top()) return HTTOP;
          if (pos_view.y > body_rc.bottom()) return HTBOTTOM;
          if (pos_view.x < body_rc.left()) return HTLEFT;
          if (pos_view.x > body_rc.right()) return HTRIGHT;
        }
      }

      element *foundel = v.find_element(pos_view);

      if (belongs_to_window_ctl(foundel, WCHARS("window-icon")))
        return HTSYSMENU;

      if (belongs_to_window_ctl(foundel, WCHARS("window-corner")))
        return HTBOTTOMRIGHT;
      if (belongs_to_window_ctl(foundel, WCHARS("window-min")) ||
        belongs_to_window_ctl(foundel, WCHARS("window-max")) ||
        belongs_to_window_ctl(foundel, WCHARS("window-close")))
        return HTCLIENT;

      if (belongs_to_caption(foundel))
          return HTCAPTION;

      return HTCLIENT;
    }

    bool window_frame_ctl::on(view &v, element *self, event_mouse &evt) {

      if (evt.cmd == MOUSE_HIT_TEST) {
        evt.data = tool::value(hittest(v, self, evt.pos_view));
        return true;
      }
      else if (evt.cmd == MOUSE_CHECK) 
      {
        if (HTCLIENT != hittest(v, self, evt.pos_view))
        {
          evt.cursor = nullptr;
          return true;
        }
        return false;
      }

      if (!dragging) 
      {
        ht_code = hittest(v, self, evt.pos_view);
        if (evt.cmd == MOUSE_DCLICK && ht_code == HTCAPTION && v.get_maximizable()) {
          if( v.get_window_state() == gool::WINDOW_MAXIMIZED )
              v.set_window_state(gool::WINDOW_SHOWN);
          else
              v.set_window_state(gool::WINDOW_MAXIMIZED);
          return true;
        }
#if !defined(WINDOWS) /*&& !defined(OSX)*/
        else if (evt.cmd == (MOUSE_DRAG_REQUEST | EVENT_SINKING) && ht_code == HTCAPTION && evt.is_point_button()) {
          v.mouse_down_on_caption = true;
          if (!v.perform_window_move()) {
            dragging_offset = evt.pos;
            drag_loop(v, evt);
          }
          return true;
        }
#endif
      }
      else // dragging
      {
        if (evt.cmd == MOUSE_MOVE && ht_code == HTCAPTION) {
          point pos = evt.pos_view + v.screen_pos() - dragging_offset;
          rect rc = rect(pos, v.window_dim());
          v.on_move_request(rc);
          v.move_window(rc);
          //dk_flush();
          return true;
        }
      }
      return false;
    }

    bool window_frame_ctl::on(view &v, element *self, event_behavior &evt) {
      if (!evt.target) return false;

      ustring role = evt.target->attr_role();
      if (role.is_undefined()) return false;

      if (evt.cmd == BUTTON_CLICK && role.like(W("window-min*"))) {
        v.set_window_state(gool::WINDOW_MINIMIZED);
        return true;
      } else if (evt.cmd == BUTTON_CLICK && role.like(W("window-max*"))) {
        if (v.get_window_state() == gool::WINDOW_MAXIMIZED)
          v.set_window_state(gool::WINDOW_SHOWN);
        else
          v.set_window_state(gool::WINDOW_MAXIMIZED);
        return true;
      } else if (evt.cmd == BUTTON_CLICK && role == WCHARS("window-close")) {
        v.ask_close_window(true);
        return true;
      }

      return false;
    }

    bool window_frame_ctl::drag_loop(view &v, event_mouse &evt) {
      helement caption = evt.target;
      dragging         = true;

      v.on_start_ui_replacement();

      v.set_capture_strict(caption);
      bool result;
      if (v.do_event(html::DO_EVENT_UNTIL_MOUSE_UP, result)) {
        dragging = false;
        v.set_capture(nullptr);
      }

      v.on_end_ui_replacement();
      return result;
    }

    bool window_frame_ctl::size_loop(view &v, event_mouse &evt) {
      dragging = true;

      v.on_start_ui_replacement();

      v.set_capture_strict(v.doc());
      bool result;
        v.set_cursor(evt.cursor);
      if (v.do_event(html::DO_EVENT_UNTIL_MOUSE_UP, result)) {
        dragging = false;
        v.set_capture(nullptr);
      }
      v.on_end_ui_replacement();
      return result;
    }


  } // namespace behavior


  FRAME_TYPE view::get_frame_type() const { return iwindow::get_frame_type(); }

  bool view::set_frame_type(FRAME_TYPE on) {
    iwindow::set_frame_type(on);
    this->detach_behavior(CHARS("window-frame"));
//#if defined(OSX)
//    if (on != STANDARD && on != STANDARD_EXTENDED)
//      this->attach_behavior(new behavior::window_frame_ctl());
//#else
    if (on != STANDARD)
      this->attach_behavior(new behavior::window_frame_ctl());
//#endif
    return true;
  }

  void view::set_enabled(bool on_off) {
    if (on_off) {
      // pd->state_off(*this, S_DISABLED);
      this->is_disabled = false;
      if (old_focus_element) set_focus(old_focus_element, BY_CODE, true);
    } else
      this->is_disabled = true;
    // pd->state_on(*this, S_DISABLED);
    iwindow::set_enabled(on_off);
  }
  bool view::is_enabled() const {
    // if (document *pd = doc())
    //  return !pd->state.disabled();
    // return false;
    return !this->is_disabled;
  }

  bool view::drag_element(html::element *el, html::clipboard::data *with_data, uint &ddm) {
    html::helement self = el;
    rect           vbox, vsrc;

    self->state_on(*this, html::S_DRAG_SOURCE);
    self->reset_styles(*this);
    //self->drop_layout();
    //self->measure_inplace(*this);

    if (self->is_box_element(*this)) {
      vsrc = vbox = self->border_box(*this, html::element::TO_VIEW);
    }
    else {
      vsrc = html::rbox(*this, self->start_caret_pos(*this),
        self->end_caret_pos(*this));
      self = self->nearest_box();
      vbox = self->border_box(*this, html::element::TO_VIEW);
    }
    assert(!vsrc.empty());
    if (vsrc.empty()) return false;
    handle<bitmap> drag_image = new bitmap(vsrc.size(), true, false);
    if (drag_image) {
      handle<graphics> sfi = app->create_bitmap_bits_graphics(drag_image, argb(0, 0, 0, 0));
      if (!sfi) return false;
      sfi->offset(-vsrc.s);
      // sfi->set_clip_rc(vsrc);
      auto_state<graphics *> _1(drawing_surface, sfi);
      point                  vpt = self->view_pos(*this);
      self->draw(*this, sfi, vpt, false);
      commit_drawing_buffers();
    }
    self->state_off(*this, html::S_DRAG_SOURCE);
    self->reset_styles(*this);
    //self->check_layout(*this);
    //self->measure_inplace(*this);

    return exec_drag(with_data, ddm, self, drag_image, mouse_down_pos - vsrc.s);
  }

  bool view::drag_data(gool::bitmap *pb, gool::point off, html::clipboard::data *with_data, uint &ddm)
  {
    handle<bitmap> drag_image = pb;
    return exec_drag(with_data, ddm, doc(), drag_image, off);
  }


} // namespace html
