#ifndef __html_pump_h
#define __html_pump_h

#include "tool/tool.h"

#if defined(SCITERJS)
#include "xdomjs/xjs.h"
#endif

#include "html-dom.h"
using namespace tool;

namespace html {
  using namespace tool;

  enum REQUEST_TYPE {
    RQ_GET    = 1,
    RQ_POST   = 2,
    RQ_PUT    = 3,
    RQ_DELETE = 4,
    RQ_PUSH   = 5, // synthetic, simulated request
  };

  struct request_param {
    ustring name;
    ustring value;
  };

  struct request;

  struct request_callback : public resource {
    function<bool(request *rq)> cb; // 'true' if handled
    handle<request_callback>    next;
  };

  struct request_progress_callback : public resource {
    function<bool(request *rq, uint data_size, uint total_size)> cb; // 'true' if handled
    handle<request_progress_callback>                            next;
  };

  class pump;

  struct response {
    virtual html::request *as_request() = 0;
  };

  struct request : public resource, public weakable, public response {
    uint   signature;
    pump * the_pump = nullptr;
    string url;         // requested URL
    string content_url; // actual resource URL. url by default but can be
                        // changed by redirections
    int_v                content_length;
    RESOURCE_DATA_TYPE   data_type;
    REQUEST_TYPE         rq_type;
    array<request_param> rq_params;

    uint started; // sys ticks, milliseconds
    uint ended;   // sys ticks, milliseconds

    dictionary<string, string> rq_headers; // request headers
    dictionary<string, string> rs_headers; // response headers

    array<byte>     data;
    string          data_content_type;     // mime type
    string          data_content_encoding; // content-encoding
    locked::counter ready_flag;
    bool            success_flag;
    uint            status;
    ustring         mediaq;

    string proxy_host;
    int_v  proxy_port;

    string username;
    string password;

    string rq_id;    // auxiliary  id assosiated with the request. used by CSS
                     // async parsing
    uint rq_purpose; // auxiliary field used for defining purpose of the
                     // data/request used in e.g. richtext.

    weak_handle<view>                 dst_view;
    weak_handle<element>              dst;
    weak_handle<element>              initiator;
    handle<request_callback>          chain;
    handle<request_progress_callback> progress_chain;
    locked::counter                   progress_handle_rq;

    tristate_v      consumed;
    bool            is_sync; // sync request
    array<uint_ptr> request_notify_receivers;
    array<uint_ptr> arrived_notify_receivers;

    tristate_v no_cache;

    tristate_v external_load; // true if SC_LOAD_DATA returns LOAD_MYSELF

#if defined(SCITER)
    tis::value obj = 0; // Request object
#elif defined(SCITERJS)
    // qjs::hvalue obj;
#endif
    uint data_represantaion_hint; // used by view.request

    request(const string &src, RESOURCE_DATA_TYPE type)
        : signature(0xAFEDAFED), the_pump(nullptr), url(src), data_type(type), rq_type(RQ_GET), status(0), rq_purpose(0), success_flag(false),
          ready_flag(false), is_sync(false), data_represantaion_hint(0) {
      started = tool::get_ticks();
      ended   = 0;
    }

    request(const request *src)
        : signature(0xAFEDAFED), the_pump(src->the_pump), url(src->url), data_type(src->data_type), rq_type(src->rq_type), status(0), rq_purpose(0),
          success_flag(false), ready_flag(false), is_sync(false), data_represantaion_hint(0), rq_headers(src->rq_headers), rq_params(src->rq_params),
          proxy_host(src->proxy_host), proxy_port(src->proxy_port), username(src->username), password(src->password) {
      started = tool::get_ticks();
      ended   = 0;
    }

    html::request *as_request() override { return this; }

    virtual ~request();
    void start(pump *by);
    void end(); // mark waiting phase

    bool is_valid() const { return uint_ptr(this) > 1000 && signature == 0xAFEDAFED; }

    string real_url() const {
      if (content_url.length()) {
        if (content_url != url) {

          tool::url ou(url);
          tool::url cu(content_url);
          cu.anchor = ou.anchor;
          return cu.compose();
        }
        return content_url;
      }
      return url;
    }

    tool::url used_url() const {
      tool::url uri;
      uri.parse(real_url());
      if (rq_type == RQ_GET && rq_params.length()) {
        string param_data = url_encoded_params();
        if (uri.params.length()) uri.params += '&';
        uri.params += param_data;
      }
      return uri;
    }

    string response_mime_type() {
      if (data_content_type.is_undefined()) data_content_type = guess_mime_type(real_url(), data);
      const char *pc = data_content_type.c_str();
      return data_content_type;
    }

    string url_encoded_params() const {
      string param_data;
      for (int n = 0; n < rq_params.size(); ++n) {
        const request_param &p = rq_params[n];
        string               t = url::escape_param(p.name());
        t += '=';
        t += url::escape_param(p.value());
        param_data += t;
        if (n != rq_params.last_index()) param_data += '&';
      }
      return param_data();
    }

    request *add(function<bool(request *rq)> cb) {
      request_callback *nc = new request_callback();
      nc->next             = chain;
      nc->cb               = cb;
      chain                = nc;
      return this;
    }
    bool process_callbacks() {
      handle<request_callback> callbacks;
      swap(callbacks, chain);
      for (handle<request_callback> cb = callbacks; cb; cb = cb->next)
        if (cb->cb(this)) {
          consumed = true;
          return true;
        }
      return false;
    }
    request *add_progress(function<bool(request *rq, uint data_size, uint total_size)> cb) {
      request_progress_callback *nc = new request_progress_callback();
      nc->next                      = progress_chain;
      nc->cb                        = cb;
      progress_chain                = nc;
      return this;
    }

    bool process_progress_callbacks(uint data_size, uint total_size) {
      for (handle<request_progress_callback> cb = progress_chain; cb; cb = cb->next)
        cb->cb(this, data_size, total_size);
      progress_handle_rq = false; // mark that we have done with it
      return true;
    }

    void done_with_success(uint status);
    void done_with_failure(uint status);
    void data_chunk_arrived(bytes chunk);

    void cancel();

    void seal_request() {
#if defined(WINDOWS)
      rq_headers["Accept-Encoding"] = "gzip, deflate";
#endif
      if (rq_type == RQ_GET) return;
      if(!rq_headers.exists(CHARS("Content-Type")))
        rq_headers[CHARS("Content-Type")] = data_content_type.length() ? data_content_type : string("application/x-www-form-urlencoded;charset=utf-8");
      data_content_type.clear();
      if (data.size())
        rq_headers[CHARS("Content-Length")] = string::format("%d", data.size());
      else {
        data = url_encoded_params().chars_as_bytes();
        data.push(0);
        data.pop();
      }
    }

    string headers_list() const {
      array<char> out;
      for (int n = 0; n < rq_headers.size(); ++n) {
        out.push(rq_headers.key(n));
        out.push(CHARS(": "));
        out.push(rq_headers.value(n));
        out.push(CHARS("\r\n"));
      }
      return out();
    }

    /*value headers_map() const {
      value map = value::make_map();
      for (int n = 0; n < rq_headers.size(); ++n) {
        string nam = rq_headers.key(n);
        ustring val = rq_headers.value(n);
        map.set_item(nam, value(val));
      }
      return map;
    }

    void headers_map(const value& map) {
      map.visit([&](const value &k, const value &v) -> bool {
        rq_headers[k.get<string>()] = v.get<string>();
        return true;
      });
    }*/

    chars requested_mime_type() const {
      switch (data_type) {
        case DATA_STYLE: return CHARS("text/css");
        case DATA_HTML: return CHARS("text/html");
        case DATA_IMAGE: return CHARS("image/*");
        case DATA_CURSOR: return CHARS("image/*");
        case DATA_SCRIPT: return CHARS("text/*script");
        case DATA_RAW_DATA: return CHARS("application/octet-stream");
        default: return CHARS("");
      }
    }
  };

  extern const char *get_http_verb(const request *prq);
  extern const char *HTTP_POST_HDR;
  extern const char *HTTP_ACCEPT_DATA[2];
  extern bool        verify_content_type(request *rq, const char *content_type);

  class pump;

  class inet_client : public resource {
  public:
    virtual void exec(pump *pm, request *rq) = 0;
  };

  class pump {
    friend class task;

  public:
    typedef html::request request;
    typedef request_param param;

    bool   active;
    mutex  guard;
    string headers;

    static string _user_agent;
    static uint   connection_timeout;
    static uint   https_error_action; // 0 - reject connection, 1 - use dialog, 2
                                      // - accept connection (retry)
    static uint max_http_data_length; // in megabytes

    // multipart/form-data composer, stack only object
    struct multipart_composer {
      request *rq;
      string   boundary;
      multipart_composer(request *prq);
      ~multipart_composer();
      void add(chars name, wchars data);
      void add(chars name, bytes data, chars filename, chars mimetype = chars());
    };

    pump() : active(true) {}
    virtual ~pump() { stop(); }

    static string get_user_agent();
    static void   set_user_agent(string s);

    void restart() {
      critical_section cs(guard);
      stop();
      active = true;
    }
    bool is_active() const { return active; }

    void stop() {
      critical_section cs(guard);
      each_running_request([](request *pr) { pr->cancel(); });
      active = false;
      inet   = nullptr;
    }

    void set_headers(const char *additional_headers, size_t additional_headers_length) {
      critical_section cs(guard);
      headers = string(additional_headers, int(additional_headers_length));
    }

    virtual void async(function<bool()> f, bool wait = false) = 0;

    virtual bool dispatch_request(handle<pump::request> rq) = 0; // gets called in UI thread

    void open_internet();

    void send(request *rq) {
      open_internet();
      if (inet) {
        rq->start(this);
        inet->exec(this, rq);
      } else
        rq->done_with_failure(uint(-1));
    }
    // bool send_now(request* rq);

    void on_start(request *rq) {
      for (auto &rr : running_requests)
        if (!rr.ptr()) {
          rr = rq;
          return;
        }
      running_requests.push(rq);
    }

    void on_end(request *rq) { running_requests.remove_by_value(rq); }

    void each_running_request(function<void(request *)> cb) {
      for (auto rr : running_requests)
        if (rr.ptr()) cb(rr);
    }

  protected:
    array<weak_handle<request>> running_requests;
    handle<inet_client>         inet;
  };

  inline request::~request() {
    // dbg_printf("destroying request: %s, data length = %d\n", url.c_str(),
    // data.size());  data.destroy(); dst = 0; initiator = 0;  ended   =
    // 0;assert(ended);
  }

  inline void request::start(pump *by) {
    if (!the_pump) {
      the_pump = by;
      // the_pump->running_requests.push(this);
      the_pump->on_start(this);
    }
  }

  inline void request::end() {
    if (the_pump) the_pump->on_end(this);
  }

  inline void request::done_with_success(uint status) {
    this->status       = status;
    this->success_flag = status >= 200 && status < 300;
    this->ready_flag   = true;
    if (the_pump) {
      handle<request> that = this;
      the_pump->async([that]() -> bool {
        that->the_pump->dispatch_request(that);
        return true;
      });
    }
  }

  inline void request::done_with_failure(uint status) {
    this->status       = status;
    this->success_flag = false;
    this->ready_flag   = true;
    if (the_pump) {
      handle<request> that = this;
      the_pump->async([that]() -> bool {
        that->the_pump->dispatch_request(that);
        return true;
      });
    }
  }

  inline void request::cancel() {
    if (this->consumed) return;
    this->status       = 0;
    this->success_flag = false;
    this->ready_flag   = true;
    if (the_pump) the_pump->dispatch_request(this);
  }

  inline void request::data_chunk_arrived(bytes chunk) {
    data.push(chunk);
    if (the_pump && !progress_handle_rq) {
      progress_handle_rq         = true;
      handle<request> that       = this;
      uint            data_size  = uint(data.length());
      uint            total_size = (uint)this->content_length.val(0);
      if (that->progress_chain) {
        the_pump->async([that, data_size, total_size]() -> bool {
          that->process_progress_callbacks(data_size, total_size);
          return true;
        });
        // yield();
      }
    }
  }

  /*
  class queue
  {
  protected:
    mutex                           guard;

  public:
    array< handle<request> >  items;

    queue() {}
    ~queue() { clear(); }

    void clear()
    {
      critical_section cs(guard);
      while( items.size() )
      {
        handle<request> t = items.pop();
        t = 0;
      }
    }

    void push( handle<request> rq )
    {
      handle<request> hrq = rq;
      critical_section cs(guard);
      items.insert(0,hrq);
    }

    bool pop( handle<request>& rq )
    {
        critical_section cs(guard);

        if(items.size() == 0)
          return false;

        rq = items.last();
        items.pop();
        return true;
    }

    bool is_pending(const string& url)
    {
        critical_section cs(guard);
        foreach(i, items)
        {
          if(items[i]->url == url) return true;
        }
        return false;
    }

  }; */

} // namespace html

#endif
