#ifndef __dom_primitives_h__
#define __dom_primitives_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-symbols.h"
#include "html-style-types.h"
#include "html-style-enums.h"
#if defined(SCITER)
  #include "tiscript/include/cs.h"
  typedef tis::value script_expando;
#elif defined(SCITERJS)
  #include "xdomjs/xjs.h"
  typedef qjs::hvalue script_expando;
#endif
#include "html-events.h"
#include "html-streams.h"
#include <array>

#define NOT_YET 0
#define POPUP_NOT_YET 0

#if (DEVICE != EOT) || defined(SKIA_BACKEND) 
#define SMOOTH_SCROLL_DEFAULT true
#else
#define SMOOTH_SCROLL_DEFAULT false
#endif

namespace html {
  using namespace tool;
  using namespace gool;

  struct element;
  struct document;
  class  view;

  template <typename C> class irange_t {
  public:
    C l;
    C h;

    irange_t() : l(0), h(-1) {}
    irange_t(C ll, C hh) : l(ll), h(hh) {}

    bool empty() const { return (l > h); }
    C    length() const { return (l <= h) ? h - l + 1 : 0; }
    C    middle() const { return (l <= h) ? ((l + h) / 2) : l; }
    void clear() {
      l = 0;
      h = -1;
    }
    C pointOf(C m, C d) const {
      return (l <= h) ? (l + ((h - l) * m) / d) : l;
    }

    static irange_t make(C start, C length) { return irange_t(start, start + length - 1); }

    inline irange_t &operator-=(C offset) {
      l -= offset;
      h -= offset;
      return *this;
    }
    inline irange_t &operator+=(C offset) {
      l += offset;
      h += offset;
      return *this;
    }
    inline irange_t operator+(C offset) const {
      return irange_t(l + offset, h + offset);
    }
    inline irange_t operator-(C offset) const {
      return irange_t(l - offset, h - offset);
    }
    inline irange_t &operator&=(irange_t<C> a) {
      l = max(l, a.l);
      h = min(h, a.h);
      return *this;
    }
    inline irange_t &operator|=(irange_t<C> b) {
      if (empty())
        *this = b;
      else if (!b.empty()) {
        l = min(l, b.l);
        h = max(h, b.h);
      }
      return *this;
    }

    bool covers(irange_t other) const {
      assert(!other.empty());
      return (other & *this) == other;
    }
    bool contains(C v) const { return (v >= l) && (v <= h); }
    bool overlaps_with(irange_t other) const {
      return (max(l, other.l)) <= (min(h, other.h));
    }
  };

  /*
    //range is not empty - true
    operator bool() const         { return (l <= h); }
    //range is empty - true
    bool operator!() const        { return (l < h); }
  */

  template <typename C>
  inline bool operator==(const irange_t<C> &a, const irange_t<C> &b) {
    return (a.l == b.l) && (a.h == b.h);
  }
  template <typename C>
  inline bool operator!=(const irange_t<C> &a, const irange_t<C> &b) {
    return (a.l != b.l) || (a.h != b.h);
  }

  /*template <typename C> inline bool operator&&(C a,irange_t<C> b)
  { return (a >= b.l) && (a <= b.h); } template <typename C> inline bool
  operator&&(irange_t<C> a,C b)               { return (b >= a.l) && (b <=
  a.h); } template <typename C> inline bool operator&&(irange_t<C> a,irange_t<C>
  b)
  {
    return
      (max(a.l,b.l)) <=
      (min(a.h,b.h));
  }*/

  template <typename C>
  inline irange_t<C> operator&(irange_t<C> a, irange_t<C> b) {
    return irange_t<C>(max(a.l, b.l), min(a.h, b.h));
  }
  template <typename C>
  inline irange_t<C> operator|(irange_t<C> a, irange_t<C> b) {
    return irange_t<C>(min(a.l, b.l), max(a.h, b.h));
  }

  typedef irange_t<int> irange;

  struct context {
    virtual document* pdoc() const = 0;
    virtual view* pview() const = 0;
    operator view* () const { return pview(); }
  };

  struct document_context : public context
  {
    document* _pdoc;
    view* _pview;

    document_context(const element* pe);
    document_context(view& v, document* pd) : _pview(&v), _pdoc(pd) {}
    document_context(context& ctx) : _pview(ctx.pview()), _pdoc(ctx.pdoc()) {}

    virtual document* pdoc() const override { return _pdoc; }
    virtual view* pview() const override { return _pview; }
  };

  struct element_context : public document_context
  {
    handle<element> _pel;
    style* _ps = nullptr;

    element_context(const element* pe) : document_context(pe), _pel(const_cast<element*>(pe)) {}
    element_context(view& v, document* pd, const element* pe, const style* ps = nullptr) : document_context(v, pd), _pel(const_cast<element*>(pe)), _ps(const_cast<style*>(ps)) {}
    element_context(context& dctx, element* pe) : document_context(dctx), _pel(pe) {}
    element_context(view& v, const element* pe, const style* ps = nullptr);

    virtual element* pel() const { return _pel; }
    virtual style* pstyle() const { return _ps; }

    gool::argb color_value(tool::value val) const;
    size_v length_value(tool::value val) const;


  };

  value resolve_color_function(const element_context& ctx, value val);
  value resolve_length_function(const element_context& ctx, value val);

#define MARKER_START_FRAGMENT "<!--StartFragment-->"
#define MARKER_END_FRAGMENT "<!--EndFragment-->"

#define MARKER_START_SELECTION "<!--StartSelection-->"
#define MARKER_END_SELECTION "<!--EndSelection-->"

#define INLINE_IMAGE_DATA_MIME_TYPE "text/base64"

  // external resource data types
  enum RESOURCE_DATA_TYPE {
    DATA_ALL  = -1,
    DATA_HTML = 0,
    DATA_IMAGE,
    DATA_STYLE,
    DATA_CURSOR,
    DATA_SCRIPT,
    // DATA_SCRIPT_DATA, // e.g. JSON request
    DATA_RAW_DATA, // raw data request
    DATA_FONT,
    DATA_SOUND, // wav bytes
  };

  enum ANIMATION_PRESETS {
    ANIMATION_TIMER_SPAN = 16u, // ms - 60 FPS
  };

  enum TIMER_KIND {
    STOPPED_TIMER         = 0, // removed
    INTERNAL_TIMER        = 1,
    EXTERNAL_TIMER        = 2,
    CSSS_TIMER            = 3, // obsolete
    SCRIPT_TIMER          = 4,
    SCRIPT_INTERVAL       = 5,
    SCRIPT_ELEMENT_TIMER  = 6,
  };

  enum KNOWN_TIMERS {
    CARET_TIMER_ID,           // editors
    PRESENT_SUBMENU_TIMER_ID, // popup menu
    SHOOT_SUBMENU_TIMER_ID,   // popup menu
    SCROLLBAR_TIMER_ID,       // scrollbar-indicator timer
    CSSS_TIMER_ID,            // csss!
    DELAYED_MEASURE_TIMER_ID,
    SPELL_CHECK_TIMER_ID,     // editors
    DELAYED_INIT_TIMER_ID,
    SCROLLBAR_INACTIVE_TIMER_ID,
    SCROLLBAR_EXPANSION_TIMER_ID,
    TIMED_ANIMATION_TIMER_ID,
    MOUSE_LEAVE_TIMER_ID,
    MOUSE_IDLE_TIMER_ID,
    MOUSE_TICK_TIMER_START_ID,
    MOUSE_TICK_TIMER_TICK_ID,
  };
  
  enum KNOWN_TIMER_TIMES {
    CARET_TIMER_MS           = 500,
    SHOOT_SUBMENU_TIMER_MS   = 400,
    PRESENT_SUBMENU_TIMER_MS = 50,
    SCROLLBAR_TIMER_MS       = ANIMATION_TIMER_SPAN,
    DELAYED_MEASURE_TIMER_MS = 20,
    SPELL_CHECK_TIMER_MS     = 400,
    DELAYED_INIT_TIMER_MS = 20,
    SCROLLBAR_INACTIVE_TIMER_MS = 2000,
    SCROLLBAR_EXPANSION_TIMER_MS = 100,
    TIMED_ANIMATION_TIMER_MS = 41,
    MOUSE_LEAVE_TIMER_MS = 40,
    MOUSE_IDLE_TIMER_MS = 402,
    MOUSE_TICK_TIMER_START_MS = 408,
    MOUSE_TICK_TIMER_TICK_MS = MOUSE_TICK_TIMER_START_MS / 8,
  };

  enum ENSURE_VISIBLE_MANNER {
    AS_PER_CSS,
    SET_POS,
    SCROLL,
    SCROLL_SMOOTH,
    SCROLL_PAN, // gesture
  };

  // insert_adjacent_html 
  enum SE_TYPE {
    SE_REPLACE,
    SE_INSERT,
    SE_APPEND,
    SE_OUTER_REPLACE,
    SE_OUTER_INSERT_BEFORE,
    SE_OUTER_INSERT_AFTER,
  };

  typedef uint64 timer_id; // tis::value 

 // string combine_url(string base, const string& rel);

  typedef geom::size_t<int_v>  dim_v_t;
  typedef geom::point_t<int_v> point_v_t;

  // typedef markup::instream<wchar> istream;
  // typedef markup::mem_istream<wchar> mem_istream;

  typedef dictionary<ustring, value> media_variables;

  struct name_or_symbol {
    attr::symbol_t att_sym;
    name_or_symbol(attr::symbol_t sym) : att_sym(sym) {}
    name_or_symbol(int sym) : att_sym(uint(sym)) {}
    name_or_symbol(const char *name) : att_sym(attr::symbol(name)) {}
    name_or_symbol(chars name) : att_sym(attr::symbol(name)) {}
    name_or_symbol(const tool::string &name) : att_sym(attr::symbol(name)) {}
    name_or_symbol(const tool::ustring &name) : att_sym(attr::symbol(name)) {}

    bool operator==(const name_or_symbol &rs) const {
      return att_sym == rs.att_sym;
    }
    bool operator!=(const name_or_symbol &rs) const {
      return att_sym != rs.att_sym;
    }
  };

  // name/ustring map

  struct attribute_bag {
    struct item {
      attr::symbol_t att_sym;
      ustring        att_value;

      item() {}
      item(const item &i) : att_sym(i.att_sym), att_value(i.att_value) {}
      item &operator=(const item &i) {
        if (this != &i) {
          att_sym   = i.att_sym;
          att_value = i.att_value;
        }
        return *this;
      }
      ~item() {}

      bool operator==(const item &rs) const {
        return att_sym == rs.att_sym && att_value == rs.att_value;
      }
      bool operator!=(const item &rs) const {
        return att_sym != rs.att_sym || att_value != rs.att_value;
      }
    };

    array<item>   items;
    mutable uint  hashv;
    array<wchars> classes;
    uint          has_bound; // count of attributes starting from @

    attribute_bag() : hashv(0), has_bound(0) {}
    attribute_bag(NO_INIT ni) : items(ni), classes(ni) {}
    attribute_bag(const attribute_bag &ab)
        : items(ab.items), hashv(0), classes(ab.classes),
          has_bound(ab.has_bound) {}
    attribute_bag &operator=(const attribute_bag &ab) {
      items     = ab.items;
      classes   = ab.classes;
      hashv     = ab.hashv;
      has_bound = ab.has_bound;
      return *this;
    }

    void emit(ostream &os);

    tool::ustring operator()(name_or_symbol ns, const wchar *dv = 0) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) return p->att_value;
      }
      return dv ? tool::ustring(dv) : tool::ustring();
    }

    bool get(name_or_symbol ns, tool::ustring &v) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            v = p->att_value;
            return true;
          }
      }
      return false;
    }

    bool get_index(name_or_symbol ns, int &idx) const {
      if (items.size()) {
        const item *start = &items[0];
        const item *p     = start;
        const item *end   = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            idx = int(p - start);
            return true;
          }
      }
      return false;
    }

    const tool::ustring operator[](name_or_symbol ns) const {
      return operator()(ns);
    }

    bool set(name_or_symbol ns, wchars wc) {
      return set(ns, tool::ustring(wc));
    }
    bool set(name_or_symbol ns, const tool::ustring &v) {
      if (items.size()) {
        item *p   = &items[0];
        item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            if (p->att_value == v) return false;
            p->att_value = v;
            if (ns.att_sym == attr::a_class) update_classes(v);
            hashv = 0;
            return true;
          }
      }
      item ni;
      ni.att_sym   = ns.att_sym;
      ni.att_value = v;
      items.push(ni);
      if (ns.att_sym == attr::a_class) update_classes(v);
      if (attr::symbol_name(ns.att_sym)().starts_with('@')) ++has_bound;
      hashv = 0;
      return true;
    }

    bool set(name_or_symbol ns, const wchar *s) { return set(ns, ustring(s)); }
    bool set(name_or_symbol ns) { return set(ns, tool::ustring()); }

    bool remove(name_or_symbol ns) {
      for (int i = 0; i < items.size(); ++i)
        if (items[i].att_sym == ns.att_sym) {
          hashv = 0;
          items.remove(i);
          if (ns.att_sym == attr::a_class) classes.clear();
          if (attr::symbol_name(ns.att_sym)().starts_with('@')) --has_bound;
          return true;
        }
      return false;
    }

    void remove_at(int idx) {
      hashv = 0;
      if (items[idx].att_sym == attr::a_class) classes.clear();
      if (attr::symbol_name(items[idx].att_sym)().starts_with('@')) --has_bound;
      items.remove(idx);
    }

    inline ustring value(int idx) const { return items[idx].att_value; }

    inline void value(int idx, const tool::ustring &v) {
      hashv                = 0;
      items[idx].att_value = v;
      if (items[idx].att_sym == attr::a_class) update_classes(v);
    }

    inline tool::string name(int idx) const {
      return attr::symbol_name(items[idx].att_sym);
    }
    inline uint symbol(int idx) const { return items[idx].att_sym; }

    inline void item_at(int i, uint &sym, ustring &val) const {
      const item &it = items[i];
      sym            = it.att_sym;
      val            = it.att_value;
    }

    inline void names(array<tool::string> &na) const {
      for (int i = 0; i < items.size(); ++i)
        na.push(attr::symbol_name(items[i].att_sym));
    }

    inline void symbols(array<attr::symbol_t> &na) const {
      for (int i = 0; i < items.size(); ++i)
        na.push(items[i].att_sym);
    }

    inline bool exist(name_or_symbol ns) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) return true;
      }
      return false;
    }
    inline bool exist(name_or_symbol ns, ustring &v) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            v = p->att_value;
            return true;
          }
      }
      return false;
    }
    inline bool contains(const attribute_bag &atts) const {
      for (int i = 0; i < atts.size(); ++i) {
        ustring val;
        if (!exist(atts.items[i].att_sym, val)) return false;
        if (atts.items[i].att_value.is_defined() &&
            atts.items[i].att_value != val)
          return false;
      }
      return true;
    }

    inline void inherit(const attribute_bag &rs) {
      for (index_t n = rs.items.last_index(); n >= 0; --n) {
        const item &         rsi     = rs.items[n];
        const attr::symbol_t rsi_sym = rsi.att_sym;
        item *               p       = items.head();
        item *               end     = items.tail();
        for (; p < end; ++p)
          if (p->att_sym == rsi_sym) {
            p->att_value = rsi.att_value;
            if (p->att_sym == attr::a_class) update_classes(p->att_value);
            goto NEXT;
          }
        items.push(rsi);
        if (rsi.att_sym == attr::a_class) update_classes(rsi.att_value);
        if (attr::symbol_name(rsi.att_sym)().starts_with('@')) ++has_bound;
      NEXT:
        continue;
      }
      hashv = 0;
    }

    inline bool is_empty() const { return items.size() == 0; }

    inline void swap(attribute_bag &rs) {
      items.swap(rs.items);
      classes.swap(rs.classes);
      tool::swap(hashv, rs.hashv);
      tool::swap(has_bound, rs.has_bound);
    }

    bool operator==(const attribute_bag &rs) const {
      if (hashv && rs.hashv && hashv != rs.hashv) return false;
      return items == rs.items;
    }
    bool operator!=(const attribute_bag &rs) const {
      if (hashv && rs.hashv && hashv != rs.hashv) return true;
      return items != rs.items;
    }

    void clear() {
      items.clear();
      classes.clear();
      has_bound = false;
    }

    inline index_t size() const { return items.size(); }
    inline size_t  length() const { return items.length(); }

    tool::ustring get_ustring(name_or_symbol ns, const wchar *dv = 0) const {
      return operator()(ns, dv);
    }

    tool::string get_string(name_or_symbol ns) const {
      return tool::string(get_ustring(ns));
    }

    tool::string get_class() const { return get_string(attr::a_class); }

    bool add_class(wchars cls, ustring &nv) {
      if (has_class(cls)) return false;
      if (get(attr::a_class, nv)) {
        nv += L' ';
        nv += cls;
      } else
        nv = cls;
      return true;
    }
    bool remove_class(wchars cls, ustring &nv) {
      if (!has_class(cls)) return false;
      for (int n = 0; n < classes.size(); ++n) {
        if (classes[n] == cls) continue;
        if (nv.length()) nv += L' ';
        nv += classes[n];
      }
      return true;
    }
    bool has_class(wchars cls) const { return classes.get_index(cls) >= 0; }

    /*dimension_v get_dimension(name_or_symbol ns) const
    {
      dimension_v v;
      from_string(v,operator()(ns));
      return v;
    }*/

    halign_ev get_align() const {
      halign_ev v; v.set(operator()(attr::a_align));
      return v;
    }
    halign_ev get_align(halign_e dv) const {
      halign_ev v;
      if (v.set(operator()(attr::a_align))) return v;
      return dv;
    }

    valign_ev get_valign() const {
      valign_ev v; v.set(operator()(attr::a_valign));
      return v;
    }

    valign_ev get_valign(valign_e dv) const {
      valign_ev v;
      if (v.set(operator()(attr::a_valign))) return v;
      return dv;
    }

    enum_v get_enum(name_or_symbol ns, slice<enum_item_def> items) {
      enum_v v;
      parse_enum(v, operator()(ns), items);
      return v;
    }

    color_v get_color(name_or_symbol ns) const {
      color_v v;
      from_string(v, operator()(ns));
      return v;
    }

    dimension_v get_width() const {
      dimension_v d;
      from_string(d, operator()(attr::a_width));
      return d;
    }
    dimension_v get_height() const {
      dimension_v d;
      from_string(d, operator()(attr::a_height));
      return d;
    }

    size_v get_width_size() const {
      size_v d;
      from_string(d, operator()(attr::a_width), NUMBER_DIP);
      return d;
    }
    size_v get_height_size() const {
      size_v d;
      from_string(d, operator()(attr::a_height), NUMBER_DIP);
      return d;
    }

    space_v get_cell_padding() const {
      space_v v;
      from_string(v, operator()(attr::a_cellpadding));
      return v;
    }
    space_v get_cell_spacing() const {
      space_v v;
      from_string(v, operator()(attr::a_cellspacing));
      return v;
    }

    space_v get_border(int defv = 1) const {
      space_v v;
      ustring us;
      if (exist(attr::a_border, us)) {
        if (us.length() == 0)
          v = defv;
        else
          from_string(v, us);
      }
      return v;
      // from_string(v,operator()(attr::a_border));
    }
    color_v get_border_color() const { return get_color(attr::a_bordercolor); }
    color_v get_back_color() const { return get_color(attr::a_bgcolor); }

    // image_ref     back_image_id;

    // halign_v      alignment; // sticked left / right

    // dimension_v   get_width() const { return get_dimension(attr::a_width); }
    // dimension_v   get_height() const { return get_dimension(attr::a_height);
    // }

    space_v get_fixed_rows() const {
      space_v t;
      from_string(t, get_ustring(attr::a_fixedrows));
      return t;
    }
    space_v get_fixed_cols() const {
      space_v t;
      from_string(t, get_ustring(attr::a_fixedcols));
      return t;
    }

    int get_colspan() const {
      return limit(get_int(attr::a_colspan, 1), 1, 20000);
    }
    int get_rowspan() const {
      return limit(get_int(attr::a_rowspan, 1), 1, 20000);
    }

    /*void          set_rowspan(int n) {
                    assert( n > 0 );
                    if( n <= 1 ) this->remove(attr::a_rowspan);
                    else this->set(attr::a_rowspan,tool::itoa(int(n)));
                  }
    void          set_colspan(int n) {
                    assert( n > 0 );
                    if( n <= 1 ) this->remove(attr::a_colspan);
                    else this->set(attr::a_colspan,tool::itoa(int(n)));
                  }*/

    size_v get_size(name_or_symbol ns, size_v dv = size_v(), SIZE_FROM_STRING nf = NUMBER_NUMBER) const {
      size_v v;
      from_string(v, operator()(ns),nf);
      return v.is_undefined() ? dv : v;
    }

    tool::string get_url(const tool::string &base_url, name_or_symbol ns) const;

    bool has_href() const { return get_href().length() != 0; }

    tool::string get_href() const { return get_string(attr::a_href); }

    int get_int(name_or_symbol ns, int dv) const {
      tool::ustring s = get_ustring(ns);
      if (s.length() == 0) return dv;
      int v;
      return stoi(s, v) ? v : dv;
    }

    bool get_bool(name_or_symbol ns, bool dv = false) const {
      ustring v;
      if (!exist(ns, v)) return dv;
      return v != WCHARS("false") && v != WCHARS("no") && v != WCHARS("off");
    }

    tristate_v get_bool_v(name_or_symbol ns) const {
      ustring v;
      if (!exist(ns, v)) return tristate_v();
      return v != WCHARS("false") && v != WCHARS("no") && v != WCHARS("off");
    }

    float get_float(name_or_symbol ns, float dv) const {
      tool::ustring s = get_ustring(ns);
      if (s.length() == 0) return dv;
      float v;
      return stof(s, v) ? v : dv;
    }

    double get_float(name_or_symbol ns, double dv) const {
      tool::ustring s = get_ustring(ns);
      if (s.length() == 0) return dv;
      double v;
      return stof(s, v) ? v : dv;
    }

    // returns 0.0 ... 1.0
    float_v get_percent(name_or_symbol ns) const {
      tool::ustring s   = get_ustring(ns);
      tool::wchars  c   = s;
      float         val = 0;
      if (!tool::parse_real(c, val)) return float_v();
      if (*c == '%') val /= 100;
      return limit(val, 0.0f, 1.0f);
    }

    unsigned int hash() const {
      if (!hashv) {
        hashv     = (uint)items.length();
        item *p   = items.head();
        item *end = items.tail();
        for (; p < end; ++p) {
          hash_combine(hashv, p->att_sym);
          hash_combine(hashv, p->att_value.hash());
        }
      }
      return hashv;
    }

  private:
    void update_classes(const tool::ustring &v) {
      classes.clear();
      tool::wtokens ts(v(), WCHARS(" "));
      for (tool::wchars item; ts.next(item);)
        classes.push(item);
    }
  };

  // name/value map

  struct attribute_bag_v {
    struct item {
      uint  att_sym;
      value att_value;

      item() : att_sym(0) {}
      item(const item &i) : att_sym(i.att_sym), att_value(i.att_value) {}
      item &operator=(const item &i) {
        if (this != &i) {
          att_sym   = i.att_sym;
          att_value = i.att_value;
        }
        return *this;
      }
      ~item() {}

      bool operator==(const item &rs) const {
        return att_sym == rs.att_sym && att_value == rs.att_value;
      }
      bool operator!=(const item &rs) const {
        return att_sym != rs.att_sym || att_value != rs.att_value;
      }
    };

    array<item>          items;
    mutable unsigned int hashv;

    attribute_bag_v() : hashv(0) {}
    attribute_bag_v(NO_INIT ni) : items(ni) {}
    attribute_bag_v(const attribute_bag_v &ab) : items(ab.items), hashv(0) {}
    attribute_bag_v &operator=(const attribute_bag_v &ab) {
      if (this == &ab) return *this;
      items = ab.items;
      hashv = ab.hashv;
      return *this;
    }

    tool::value operator()(name_or_symbol ns) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) return p->att_value;
      }
      return tool::value();
    }

    bool get(name_or_symbol ns, tool::value &v) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            v = p->att_value;
            return true;
          }
      }
      return false;
    }

    /*bool get_color(uint sym, gool::argb &c) const {
      tool::value v;
      if (get(attr::symbol_t(sym), v) && v.is_color()) {
        color_v cv = v;
        c          = cv.to_argb();
        return true;
      }
      return false;
    }*/

    const tool::value operator[](name_or_symbol ns) const {
      return operator()(ns);
    }

    bool set(name_or_symbol ns, const tool::value &v) {
      if (items.size()) {
        item *p   = &items[0];
        item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            if (p->att_value == v) return false;
            p->att_value = v;
            hashv        = 0;
            return true;
          }
      }
      item ni;
      ni.att_sym   = ns.att_sym;
      ni.att_value = v;
      items.push(ni);
      hashv = 0;
      return true;
    }

    bool remove(name_or_symbol ns) {
      for (int i = 0; i < items.size(); ++i)
        if (items[i].att_sym == ns.att_sym) {
          hashv = 0;
          items.remove(i);
          return true;
        }
      return false;
    }

    void remove_at(int idx) {
      hashv = 0;
      items.remove(idx);
    }

    inline tool::value value(int idx) const { return items[idx].att_value; }

    inline void value(int idx, const tool::value &v) {
      hashv                = 0;
      items[idx].att_value = v;
    }

    inline tool::string name(int idx) const {
      return attr::symbol_name(items[idx].att_sym);
    }
    inline uint symbol(int idx) const { return items[idx].att_sym; }

    inline void item_at(int i, uint &sym, tool::value &val) const {
      const item &it = items[i];
      sym            = it.att_sym;
      val            = it.att_value;
    }

    inline void names(array<tool::string> &na) const {
      for (int i = 0; i < items.size(); ++i)
        na.push(attr::symbol_name(items[i].att_sym));
    }

    inline void symbols(array<dword> &na) const {
      for (int i = 0; i < items.size(); ++i)
        na.push(items[i].att_sym);
    }

    inline bool exist(name_or_symbol ns) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) return true;
      }
      return false;
    }
    inline bool exist(name_or_symbol ns, tool::value &v) const {
      if (items.size()) {
        const item *p   = &items[0];
        const item *end = p + items.size();
        for (; p < end; ++p)
          if (p->att_sym == ns.att_sym) {
            v = p->att_value;
            return true;
          }
      }
      return false;
    }

    inline void inherit(const attribute_bag_v &rs) {
      for (index_t n = rs.items.last_index(); n >= 0; --n) {
        const item &rsi     = rs.items[n];
        const dword rsi_sym = rsi.att_sym;
        item *      p       = items.head();
        item *      end     = items.tail();
        for (; p < end; ++p)
          if (p->att_sym == rsi_sym) {
            p->att_value = rsi.att_value;
            goto NEXT;
          }
        items.push(rsi);
      NEXT:
        continue;
      }
      hashv = 0;
    }

    inline bool is_empty() const { return items.size() == 0; }

    inline void swap(attribute_bag_v &rs) {
      items.swap(rs.items);
      swop(hashv, rs.hashv);
    }

    bool operator==(const attribute_bag_v &rs) const {
      if (hashv && rs.hashv && hashv != rs.hashv) return false;
      return items == rs.items;
    }
    bool operator!=(const attribute_bag_v &rs) const {
      if (hashv && rs.hashv && hashv != rs.hashv) return true;
      return items != rs.items;
    }

    void clear() { items.clear(); }

    inline index_t size() const { return items.size(); }
    inline size_t  length() const { return items.length(); }

    unsigned int hash() const {
      if (!hashv) {
        hashv     = (uint)items.length();
        item *p   = items.head();
        item *end = items.tail();
        for (; p < end; ++p) {
          hash_combine(hashv, p->att_sym);
          hash_combine(hashv, p->att_value.hash());
        }
      }
      return hashv;
    }
  };

  // name/ value reference map
  struct attribute_bag_vref {
    struct item : public resource {
      uint  att_sym;
      value att_value;

      item() : att_sym(0) {}
      item(const item &i) : att_sym(i.att_sym), att_value(i.att_value) {}
      item &operator=(const item &i) {
        if (this != &i) {
          att_sym   = i.att_sym;
          att_value = i.att_value;
        }
        return *this;
      }
      ~item() {}

      bool operator==(const item &rs) const {
        return att_sym == rs.att_sym && att_value == rs.att_value;
      }
      bool operator!=(const item &rs) const {
        return att_sym != rs.att_sym || att_value != rs.att_value;
      }
    };

    typedef handle<item> hitem;

    array<hitem> items;
    bool         items_owner;
    mutable unsigned int  hashv = 0;

    attribute_bag_vref() : items_owner(true) {}
    attribute_bag_vref(const attribute_bag_vref &ab) {
      items.attach_data(ab.items);
      items_owner = false;
    }
    attribute_bag_vref &operator=(const attribute_bag_vref &ab) {
      if (this == &ab) return *this;
      if (this->items.cbegin() == ab.items.cbegin()) return *this;
      items.attach_data(ab.items);
      items_owner = false;
      hashv = 0;
      return *this;
    }

    /*tool::value operator()(name_or_symbol ns) const
    {
      if(items.size())
      {
        const item* p = &items[0];
        const item* end = p + items.size();
        for(; p < end; ++p)
          if(p->att_sym == ns.att_sym)
            return p->att_value;
      }
      return tool::value();
    }*/

    bool get(name_or_symbol ns, tool::value &v) const {
      if (items.size()) {
        const hitem *p   = &items[0];
        const hitem *end = p + items.size();
        for (; p < end; ++p)
          if ((*p)->att_sym == ns.att_sym) {
            v = (*p)->att_value;
            return true;
          }
      }
      return false;
    }

    bool get_color(uint sym, color_v &c) const {
      tool::value v;
      if (get(attr::symbol_t(sym), v) && v.is_color()) {
        c = v;
        return true;
      }
      return false;
    }

    bool get_length(uint sym, html::size_v &sz) const {
      tool::value v;
      if (get(attr::symbol_t(sym), v) && v.is_length()) {
        if (size_v::from_value(v, sz)) return true;
      }
      return false;
    }

    /*const tool::value operator[](name_or_symbol ns) const
    {
      return operator()(ns);
    }*/

    void check_writeable() {
      if (!items_owner) {
        items       = array<hitem>(items);
        items_owner = true;
      }
    }

    bool set(name_or_symbol ns, const tool::value &v) {
      check_writeable();

      for (int i = items.last_index(); i >= 0; --i) {
        if (items[i]->att_sym == ns.att_sym) {
          if (v.is_undefined()) {
            hashv = 0;
            items.remove(i);
            return true;
          }
          if (items[i]->att_value == v) 
            return false;
          items[i]->att_value = v;
          hashv = 0;
          return true;
        }
      }

      if (v.is_defined()) {
        hitem ni      = new item();
        ni->att_sym   = ns.att_sym;
        ni->att_value = v;
        items.push(ni);
        hashv = 0;
        return true;
      }
      return false;
    }

    bool remove(name_or_symbol ns) {
      for (int i = 0; i < items.size(); ++i)
        if (items[i]->att_sym == ns.att_sym) {
          check_writeable();
          items.remove(i);
          hashv = 0;
          return true;
        }
      return false;
    }

    void remove_at(int idx) {
      check_writeable();
      items.remove(idx);
      hashv = 0;
    }

    inline tool::value value(int idx) const { return items[idx]->att_value; }

    inline void value(int idx, const tool::value &v) {
      check_writeable();
      items[idx]->att_value = v;
      hashv = 0;
    }

    inline tool::string name(int idx) const {
      return attr::symbol_name(items[idx]->att_sym);
    }
    inline uint symbol(int idx) const { return items[idx]->att_sym; }

    inline void item_at(int i, uint &sym, tool::value &val) const {
      hitem it = items[i];
      sym      = it->att_sym;
      val      = it->att_value;
    }

    inline void names(array<tool::string> &na) const {
      for (int i = 0; i < items.size(); ++i)
        na.push(attr::symbol_name(items[i]->att_sym));
    }

    inline void symbols(array<dword> &na) const {
      for (int i = 0; i < items.size(); ++i)
        na.push(items[i]->att_sym);
    }

    inline bool exist(name_or_symbol ns) const {
      if (items.size()) {
        const hitem *p   = &items[0];
        const hitem *end = p + items.size();
        for (; p < end; ++p)
          if ((*p)->att_sym == ns.att_sym) return true;
      }
      return false;
    }
    inline bool exist(name_or_symbol ns, tool::value &v) const {
      if (items.size()) {
        const hitem *p   = &items[0];
        const hitem *end = p + items.size();
        for (; p < end; ++p)
          if ((*p)->att_sym == ns.att_sym) {
            v = (*p)->att_value;
            return true;
          }
      }
      return false;
    }

    void each(function<bool(tool::string name, tool::value val)> cb) const {
      if (items.size()) {
        const hitem *p   = &items[0];
        const hitem *end = p + items.size();
        for (; p < end; ++p)
          if (!cb(attr::symbol_name((*p)->att_sym), (*p)->att_value)) break;
      }
    }

    inline bool inherit(const attribute_bag_vref &rs) {
      if (rs.is_empty()) return false;

      if (items.length() == 0 && rs.items.length() != 0) {
        items.attach_data(rs.items);
        items_owner = false;
        return true;
      }
      check_writeable();
      int changes = 0;
      for (index_t n = rs.items.last_index(); n >= 0; --n) {
        const hitem &rsi     = rs.items[n];
        const dword  rsi_sym = rsi->att_sym;
        hitem *      p       = items.head();
        hitem *      end     = items.tail();
        for (; p < end; ++p)
          if ((*p)->att_sym == rsi_sym) {
            if (*p != rsi) {
              ++changes;
              *p = rsi;
            }
            goto NEXT;
          }
        items.push(rsi);
        ++changes;
      NEXT:
        continue;
      }
      if(changes)
        hashv = 0;
      return changes > 0;
    }

    inline bool is_empty() const { return items.size() == 0; }

    inline void swap(attribute_bag_vref &rs) {
      items.swap(rs.items);
      std::swap(items_owner, rs.items_owner);
    }

    bool operator==(const attribute_bag_vref &rs) const {
      return items == rs.items;
    }
    bool operator!=(const attribute_bag_vref &rs) const {
      return items != rs.items;
    }

    void clear() {
      items.clear();
      items_owner = true;
    }

    inline index_t size() const { return items.size(); }
    inline size_t  length() const { return items.length(); }

    //unsigned int hash() const {
      //return (unsigned int)(uint_ptr)(items.cbegin());
    //}
    unsigned int hash() const {
      if (!hashv) {
        hashv = (uint)items.length();
        auto p = items.cbegin();
        auto end = items.cend();
        for (; p < end; ++p) {
          hash_combine(hashv, (*p)->att_sym);
          hash_combine(hashv, (*p)->att_value.hash());
        }
      }
      return hashv;
    }
  };

  struct symbol_list {
    struct data : resource {
      array<uint> symbols;
    };
    handle<data> d;

    symbol_list() {}
    symbol_list(const symbol_list &src) : d(src.d) {}
    symbol_list &operator=(const symbol_list &src) {
      d = src.d;
      return *this;
    }

    inline bool operator==(const symbol_list &src) const {
      if (d.is_null() && src.d.is_null()) return true;
      if (!d.is_null() && !src.d.is_null()) return d->symbols == src.d->symbols;
      return false;
    }

    inline index_t size() const { return d ? d->symbols.size() : 0; }
    inline size_t  length() const { return d ? d->symbols.length() : 0; }

    inline bool match(name_or_symbol a) const {
      if (d) return d->symbols.get_index(a.att_sym) >= 0;
      return false;
    }
    void add(name_or_symbol a) {
      if (!d) {
        d = new data;
        d->symbols.push(a.att_sym);
      } else if (!match(a)) {
        d->symbols.push(a.att_sym);
      }
    }

    inline bool is_defined() const { return !d.is_null(); }
    inline bool is_undefined() const { return d.is_null(); }
    inline void inherit(const symbol_list &src) {
      if (src.is_defined()) *this = src;
    }
  };

  enum SIDE {
    NO_SIDE = 0,
    TOP = 0x1,
    BOTTOM = 0x2,
    LEFT = 0x4,
    RIGHT = 0x8
  };

  ENABLE_BIT_ENUM_OPERATORS(SIDE,uint)


  const uint64 S_LINK   = 0x00000001ull;
  const uint64 S_HOVER  = 0x00000002ull;
  const uint64 S_ACTIVE = 0x00000004ull;
  const uint64 S_FOCUS  = 0x00000008ull;

  const uint64 S_VISITED  = 0x00000010ull;
  const uint64 S_CURRENT  = 0x00000020ull; // current (hot) item
  const uint64 S_CHECKED  = 0x00000040ull; // element is checked (or selected)
  const uint64 S_DISABLED = 0x00000080ull; // element is disabled

  const uint64 S_READONLY     = 0x00000100ull; // readonly input element
  const uint64 S_EXPANDED     = 0x00000200ull; // expanded state - nodes in tree view
  const uint64 S_COLLAPSED    = 0x00000400ull; // collapsed state - nodes in tree
                                            // view - mutually exclusive with
  const uint64 S_INCOMPLETE   = 0x00000800ull; // one of fore/back images requested but not delivered

  const uint64 S_ANIMATING    = 0x00001000ull; // is animating currently
  const uint64 S_FOCUSABLE    = 0x00002000ull; // will accept focus
  const uint64 S_ANCHOR       = 0x00004000ull; // anchor in selection (used with current in selects)
  const uint64 S_SYNTHETIC    = 0x00008000ull; // this is a synthetic element - don't emit it's head/tail

  const uint64 S_OWNS_POPUP   = 0x00010000ull; // this is a synthetic element - don't emit it's head/tail
  const uint64 S_TABFOCUS     = 0x00020000ull; // focus gained by tab traversal
  const uint64 S_EMPTY        = 0x00040000ull; // empty - element is empty (text.size()
                                        // == 0 && subs.size() == 0)
  const uint64 S_BUSY         = 0x00080000ull;  // busy; loading

  const uint64 S_DRAG_OVER    = 0x00100000ull; // drag over the block that can accept it (so is current
                     // drop target). Flag is set for the drop target block
  const uint64 S_DROP_TARGET  = 0x00200000ull; // active drop target.
  const uint64 S_MOVING       = 0x00400000ull; // dragging/moving - the flag is set for the moving block.
  const uint64 S_COPYING      = 0x00800000ull; // dragging/copying - the flag is set
                                          // for the copying block.

  const uint64 S_DRAG_SOURCE  = 0x01000000ull; // element that is a drag source.
  const uint64 S_DROP_MARKER  = 0x02000000ull; // element is drop marker
  const uint64 S_PRESSED      = 0x04000000ull; // pressed - close to active but has wider life span - e.g.
                     // in MOUSE_UP it is still on; so behavior can check it in
                     // MOUSE_UP to discover CLICK condition.
  const uint64 S_POPUP        = 0x08000000ull; // this element is out of flow - popup

  const uint64 S_IS_LTR       = 0x10000000ull; // the element or one of its containers
                                         // has dir=ltr declared
  const uint64 S_IS_RTL       = 0x20000000ull; // the element or one of its containers
                                         // has dir=rtl declared

  const uint64 S_READY        = 0x40000000ull; // element is ready (behavior has finished initialization)
  const uint64 S_UNCHECKED    = 0x80000000ull; // element is unchecked (or unselected)

  // behvior shall be notified if one of these is changing
  const uint64 NOTIFICATION_STATES_BITMASK = S_CURRENT | S_CHECKED;
  //........FFFFFFFF
  const uint64 S_ROOT             = 0x8000000000000000ull; // element is a root
  const uint64 S_HAS_CHILD        = 0x4000000000000000ull; // has single child.
  const uint64 S_HAS_CHILDREN     = 0x2000000000000000ull; // has more than one child.
  const uint64 S_NON_TABABLE      = 0x1000000000000000ull; // non tab traversible.

  const uint64 S_NODE             = 0x0800000000000000ull; // true if either :expanded or :collapsed is set.
  const uint64 S_CONTENT_EDITABLE = 0x0400000000000000ull; // true if the element is inside the editor
                             // (richtext) and editing is allowed
  const uint64 S_CONTENT_NON_EDITABLE 
                                  = 0x0200000000000000ull; // true if the element is inside the editor
                             // (richtext) and editing is disabled
  const uint64 S_OWNS_FOCUS       = 0x0100000000000000ull; // the element itself or
                                                     // one of its children has
                                                     // focus

  const uint64 S_INVALID          = 0x0000000100000000ull; // invalid (input).
  const uint64 S_WANTS_FOCUS      = 0x0000000200000000ull; // the element wants focus
  const uint64 S_WANTS_NO_FOCUS   = 0x0000000400000000ull; // the element does not want focus

  const uint64 S_SELECTED         = 0x0000000800000000ull; // the element is selected (richtext)

  const uint64 S_HOVER_TOP        = 0x0000001000000000ull; // mouse is at top of content area
  const uint64 S_HOVER_BOTTOM     = 0x0000002000000000ull; // mouse is at bottom of content area
  const uint64 S_HOVER_LEFT       = 0x0000004000000000ull; // mouse is at left of content area
  const uint64 S_HOVER_RIGHT      = 0x0000008000000000ull; // mouse is at right of content area

  const uint64 S_HOVER_ALL_SIDES      = S_HOVER_TOP | S_HOVER_BOTTOM | S_HOVER_LEFT | S_HOVER_RIGHT;


  inline uint64 side_to_state(SIDE s) { return uint64(s) << 36; }

  const uint64 PUBLIC_STATE_BITMASK =
      0xFFFFFFFFull | S_INVALID | S_OWNS_FOCUS | S_CONTENT_EDITABLE |
      S_CONTENT_NON_EDITABLE | S_SELECTED | S_WANTS_FOCUS | S_WANTS_NO_FOCUS;
  const uint64 STATE_ONOFF_BITMASK =
      0xFFFFFFFFull | S_INVALID | S_OWNS_FOCUS | S_CONTENT_EDITABLE |
      S_CONTENT_NON_EDITABLE | S_SELECTED | S_WANTS_FOCUS | S_WANTS_NO_FOCUS | S_HOVER_ALL_SIDES;

  const uint64 SM_UI_STATES = 0xFFFFFFFFull | S_INVALID | S_OWNS_FOCUS |
                              S_CONTENT_EDITABLE | S_CONTENT_NON_EDITABLE |
                              S_SELECTED | S_HOVER_ALL_SIDES;
  const uint64 SM_UI_STATES_NA = SM_UI_STATES & ~S_ANIMATING;

  const uint64 STATE_CLONEABLE_BITMASK =
      S_LINK | S_CONTENT_EDITABLE | S_CONTENT_NON_EDITABLE | S_WANTS_FOCUS |
      S_WANTS_NO_FOCUS | S_SYNTHETIC | S_DISABLED | S_READONLY;

  const int MAX_PUBLIC_BITS = 33;

  struct ui_state {
    uint64 data;

    // external bits - accessible through API
    bool link() const { return get_bit(S_LINK, data); }

    bool active() const { return get_bit(S_ACTIVE, data); }
    bool focus() const { return get_bit(S_FOCUS, data); }
    bool tabfocus() const { return get_bit(S_TABFOCUS, data); }
    bool visited() const { return get_bit(S_VISITED, data); }
    bool current() const {  return get_bit(S_CURRENT, data); } // current (hot) item
    bool checked() const { return get_bit(S_CHECKED, data); } // element is checked (or selected)
    bool unchecked() const { return get_bit(S_UNCHECKED, data); } // element is unchecked (or unselected)
    bool disabled() const { return get_bit(S_DISABLED, data); } // element is disabled
    bool readonly() const { return get_bit(S_READONLY, data); } // readonly input element
    bool expanded() const { return get_bit(S_EXPANDED, data); } // expanded state - nodes in tree view
    bool collapsed() const { return get_bit(S_COLLAPSED, data); } // expanded state - nodes in tree view
    bool incomplete() const { return get_bit(S_INCOMPLETE, data); } // one of fore/back images requested but not delivered
    bool animating() const { return get_bit(S_ANIMATING, data); } // animating
    bool focusable() const { return get_bit(S_FOCUSABLE, data); } // focusable

    bool pressed() const { return get_bit(S_PRESSED, data) != 0; } // pressed
    void pressed(bool v) { set_bit(S_PRESSED, data, v); }          // pressed

    bool busy() const { return (data & S_BUSY) != 0; } // busy, loading
    void busy(bool v) { set_bit(S_BUSY, data, v); }    // busy, loading

    bool hover() const { return get_bit(S_HOVER, data); }
    bool hover(SIDE s) const { return get_bit(side_to_state(s), data); }

    bool node() const { return get_bit(S_NODE, data); }
    void node(bool v) { set_bit(S_NODE, data, v); }

    bool content_editable() const { return get_bit(S_CONTENT_EDITABLE, data); }
    void content_editable(bool v) {
      set_bit(S_CONTENT_EDITABLE, data, v);
      if (v && get_bit(S_CONTENT_NON_EDITABLE, data))
        set_bit(S_CONTENT_NON_EDITABLE, data, false);
    }
    bool content_non_editable() const {
      return get_bit(S_CONTENT_NON_EDITABLE, data);
    }
    void content_non_editable(bool v) {
      set_bit(S_CONTENT_NON_EDITABLE, data, v);
      if (v && get_bit(S_CONTENT_EDITABLE, data))
        set_bit(S_CONTENT_EDITABLE, data, false);
    }
    bool content_editable_defined() const {
      return get_bit(S_CONTENT_EDITABLE, data) ||
             get_bit(S_CONTENT_NON_EDITABLE, data);
    }

    void link(bool v) { set_bit(S_LINK, data, v); }
    void hover(bool v) { set_bit(S_HOVER, data, v); 
                         if(!v) set_bit(S_HOVER_TOP | S_HOVER_BOTTOM | S_HOVER_LEFT | S_HOVER_RIGHT,  data, false); }
    void active(bool v) { set_bit(S_ACTIVE, data, v); }
    void focus(bool v) { set_bit(S_FOCUS, data, v); }
    void tabfocus(bool v) { set_bit(S_TABFOCUS, data, v); }
    void visited(bool v) { set_bit(S_VISITED, data, v); }
    void current(bool v) { set_bit(S_CURRENT, data, v); } // current (hot) item

    void hover(SIDE s, bool v) { set_bit(side_to_state(s), data, v); 
                                 if(v) set_bit(S_HOVER, data, true); }

    void checked(bool v) {
      set_bit(S_CHECKED, data, v);
      set_bit(S_UNCHECKED, data, !v);
    } // element is checked (or selected)
    void unchecked(bool v) {
      set_bit(S_UNCHECKED, data, v);
      set_bit(S_CHECKED, data, !v);
    } // element is unchecked (or unselected)

    void disabled(bool v) {
      set_bit(S_DISABLED, data, v);
    } // element is disabled
    void readonly(bool v) {
      set_bit(S_READONLY, data, v);
    } // readonly input element

    void expanded(bool v) {
      set_bit(S_EXPANDED, data, v);
      set_bit(S_COLLAPSED, data, !v);
      // WRONG node(expanded() || collapsed());
    } // expanded state - nodes in tree view
    void collapsed(bool v) {
      set_bit(S_COLLAPSED, data, v);
      set_bit(S_EXPANDED, data, !v);
      //WRONG node(expanded() || collapsed());
    } // expanded state - nodes in tree view

    void incomplete(bool v) {
      set_bit(S_INCOMPLETE, data, v);
    } // one of fore/back images requested but not delivered
    void animating(bool v) { set_bit(S_ANIMATING, data, v); } // animating
    void focusable(bool v) { set_bit(S_FOCUSABLE, data, v); } // focusable

    bool anchor() const {
      return get_bit(S_ANCHOR, data);
    } // anchor in selection (used with current in selects)
    void anchor(bool v) { set_bit(S_ANCHOR, data, v); }

    bool ready() const {
      return get_bit(S_READY, data);
    } // ready, has finished initialization of behaviors.
    void ready(bool v) { 
      set_bit(S_READY, data, v); 
    }

    bool synthetic() const {
      return get_bit(S_SYNTHETIC, data);
    } // synthetic element
    void synthetic(bool v) { set_bit(S_SYNTHETIC, data, v); }

    bool owns_popup() const {
      return get_bit(S_OWNS_POPUP, data);
    } // parent of the popup element when popup is shown
    void owns_popup(bool v) { set_bit(S_OWNS_POPUP, data, v); }

    // internal bits - not accessible through API
    bool popup() const {
      return get_bit(S_POPUP, data);
    } // this element is out of flow - popup
    void popup(bool v) { set_bit(S_POPUP, data, v); }

    bool root() const {
      return get_bit(S_ROOT, data);
    } // element is a layout root
    void root(bool v) { set_bit(S_ROOT, data, v); }

    bool empty() const { return get_bit(S_EMPTY, data); } // state empty - no content or no current value
    void empty(bool v) { set_bit(S_EMPTY, data, v); } // state empty

    bool has_children() const { return get_bit(S_HAS_CHILDREN, data); }
    void has_children(bool v) { set_bit(S_HAS_CHILDREN, data, v); }

    bool non_tabable() const { return get_bit(S_NON_TABABLE, data); }
    void non_tabable(bool v) { set_bit(S_NON_TABABLE, data, v); }

    bool has_child() const { return get_bit(S_HAS_CHILD, data); }
    void has_child(bool v) { set_bit(S_HAS_CHILD, data, v); }

    bool drag_over() const { return get_bit(S_DRAG_OVER, data); }
    void drag_over(bool v) { set_bit(S_DRAG_OVER, data, v); }

    bool drop_target() const { return get_bit(S_DROP_TARGET, data); }
    void drop_target(bool v) { set_bit(S_DROP_TARGET, data, v); }

    bool moving() const { return get_bit(S_MOVING, data); }
    void moving(bool v) {
      set_bit(S_MOVING, data, v);
      if (v) set_bit(S_COPYING, data, false);
    }

    bool copying() const { return get_bit(S_COPYING, data); }
    void copying(bool v) {
      set_bit(S_COPYING, data, v);
      if (v) set_bit(S_MOVING, data, false);
    }

    bool drag_source() const { return get_bit(S_DRAG_SOURCE, data); }
    void drag_source(bool v) { set_bit(S_DRAG_SOURCE, data, v); }

    bool drop_marker() const { return get_bit(S_DROP_MARKER, data); }
    void drop_marker(bool v) { set_bit(S_DROP_MARKER, data, v); }

    bool is_dragging() const { return moving() || copying(); }

    /*    bool  leaf() const          { return get_bit(S_LEAF,data); }
        void  leaf(bool v)          { set_bit(S_LEAF,data,v);
       set_bit(S_NODE,data,!v); }  */

    void rtl(bool v) {
      set_bit(S_IS_RTL, data, v);
      if (v && get_bit(S_IS_LTR, data)) set_bit(S_IS_LTR, data, false);
    } // dir=rtl is defined up in the tree
    void ltr(bool v) {
      set_bit(S_IS_LTR, data, v);
      if (v && get_bit(S_IS_RTL, data)) set_bit(S_IS_RTL, data, false);
    } // dir=ltr is defined up in the tree
    bool rtl() const {
      return get_bit(S_IS_RTL, data);
    } // dir=rtl is defined up in the tree
    bool ltr() const {
      return get_bit(S_IS_LTR, data);
    } // dir=ltr is defined up in the tree

    void wants_focus(bool v) {
      set_bit(S_WANTS_FOCUS, data, v);
      set_bit(S_WANTS_NO_FOCUS, data, !v);
    }
    void wants_no_focus(bool v) {
      set_bit(S_WANTS_NO_FOCUS, data, v);
      set_bit(S_WANTS_FOCUS, data, !v);
    }
    bool wants_focus() const { return get_bit(S_WANTS_FOCUS, data); }
    bool wants_no_focus() const { return get_bit(S_WANTS_NO_FOCUS, data); }

    bool owns_focus() const {
      return get_bit(S_OWNS_FOCUS, data);
    } // element itself or one of its children has focus
    void owns_focus(bool v) { set_bit(S_OWNS_FOCUS, data, v); } //

    bool invalid() const { return get_bit(S_INVALID, data); }
    void invalid(bool v) { set_bit(S_INVALID, data, v); }

    bool selected() const { return get_bit(S_SELECTED, data) != 0; } // selected
    void selected(bool v) { set_bit(S_SELECTED, data, v); }          // selected

    ui_state() : data(0) {}
    ui_state(uint64 bits) : data(bits) {}
    ui_state(NO_INIT) {}

    int specificity() const {
      int cnt = 0;
      if (is_defined()) {
        uint32 m = 1;
        for (int i = 0; i < MAX_PUBLIC_BITS; ++i, m <<= 1)
          if ((m & data) != 0) ++cnt;
      }
      return cnt;
    }

    inline bool is_defined() const { return data != 0; }

    inline bool match(const ui_state &ss) const {
      return (data & ss.data) == ss.data;
    }

    inline ui_state &operator+=(uint64 bits) {
      data |= (STATE_ONOFF_BITMASK & bits);
      if (bits & S_EXPANDED) {
        data &= ~S_COLLAPSED;
        //data |= S_NODE;
      } else if (bits & S_COLLAPSED) {
        data &= ~S_EXPANDED;
        //data |= S_NODE;
      }
      if (bits & S_WANTS_FOCUS)
        data &= ~S_WANTS_NO_FOCUS;
      else if (bits & S_WANTS_NO_FOCUS)
        data &= ~S_WANTS_FOCUS;

      if (bits & S_CONTENT_EDITABLE)
        data &= ~S_CONTENT_NON_EDITABLE;
      else if (bits & S_CONTENT_NON_EDITABLE)
        data &= ~S_CONTENT_EDITABLE;

      if (bits & S_CHECKED)
        data &= ~S_UNCHECKED;
      else if (bits & S_UNCHECKED)
        data &= ~S_CHECKED;

      return *this;
    }
    inline ui_state &operator-=(uint64 bits) {
      uint64 prev_data = data;
      data &= ~(STATE_ONOFF_BITMASK & bits);

      if ((bits & (S_EXPANDED | S_COLLAPSED)) == (S_EXPANDED | S_COLLAPSED))
        ;
      else if ((bits & S_EXPANDED) && (prev_data & S_EXPANDED))
        data |= S_COLLAPSED;
      else if ((bits & S_COLLAPSED) && (prev_data & S_COLLAPSED))
        data |= S_EXPANDED;

      /*if (data & (S_EXPANDED | S_COLLAPSED))
        data |= S_NODE;
      else
        data &= ~S_NODE;*/
      /*if( (bits & S_WANTS_FOCUS) && (prev_data & S_WANTS_FOCUS) )
        data |= S_WANTS_NO_FOCUS;
      else if( (bits & S_WANTS_NO_FOCUS) && (prev_data & S_WANTS_NO_FOCUS) )
        data |= S_WANTS_NO_FOCUS;*/
      return *this;
    }

    inline ui_state operator|(ui_state other) {
      return (data | (STATE_ONOFF_BITMASK & other.data));
    }

    inline bool operator==(const ui_state &ss) const {
      return (data == ss.data);
    }
    inline bool operator!=(const ui_state &ss) const {
      return (data != ss.data);
    }

    inline bool similar(const ui_state &ss) const { return (data == ss.data); }

    inline ui_state diff(const ui_state &ss, const ui_state &mask) const {
      ui_state r = (mask.data & ss.data) ^ (mask.data & data);
      return r;
    }
    inline static bool intersect(const ui_state &s1, const ui_state &s2) {
      return (s1.data & s2.data & STATE_ONOFF_BITMASK) != 0;
    }

    inline ui_state mask(const ui_state &mask) const {
      ui_state r = data & mask.data;
      return r;
    }
    inline ui_state mask(uint64 m) const {
      ui_state r = data & m;
      return r;
    }

    //inline bool is_any_set(const ui_state &s1) {
    //  return (data & s1.data & STATE_ONOFF_BITMASK) != 0;
    //}
    inline bool is_set(const ui_state &s1)
    {
      return (data & s1.data & STATE_ONOFF_BITMASK) != 0;
    }

    inline bool is_clear(const ui_state &s1) {
      return (data & s1.data & STATE_ONOFF_BITMASK) == 0;
    }

    ui_state clone() const { return ui_state(data & STATE_CLONEABLE_BITMASK); }
  };

  /*enum SYNTHETIC_ELEMENT_TYPE
  {
    NON_SYNTHETIC,
    SYNTHETIC_BLOCK,
    SYNTHETIC_BEFORE,
    SYNTHETIC_AFTER,
  };*/

  struct element_flags {
    element_flags() { memzero(*this); }
    element_flags(NO_INIT) {}
    void clear() { memzero(*this); }

    uint disable_fast_css_match : 1;
    uint state_initialized : 1;
    // uint airborn                :1; // the element is not rendered at its normal place.
    uint assigned_running : 1;
    uint intrinsic_width_processing : 1;
    uint indexes_are_valid : 1; // computed indexes of child elements are valid.
    uint has_window : 1;        // has windowed ctl
    uint force_refresh : 1;     // needes deep refresh ( behavior:reflection )
    // uint can_scroll_fast_defined:1; // cached result of can_scroll_this_fast
    // uint can_scroll_fast:1;         // cached result of can_scroll_this_fast
    uint disable_text_selection : 1; // text selection shall treat this as a
                                     // solid block
    uint non_focusable : 1;          // that is non focusable element by any
                                     // circumstances.
    uint has_transformed : 1;        // has transformed children;
    uint layout_ctl_valid : 1; // 1 if element has valid layout controller.
    uint delayed_layout   : 1;       // this element is queued for layout calculations 
    uint need_delayed_measurement : 1; // delayed measurement was requested for
                                       // the element
    uint delayed_measurement : 1; // this element is running delayed measurement
    uint is_synthetic : 1; // this is synthetic element, anonymous paragraph or
                           // <TBODY>
    uint has_on_size_script : 1;               // has onSize/onsizechange handler in script.
    uint has_on_pre_size_script : 1;           // has onsizechanging handler in script.  
    uint has_on_visibility_changed_script : 1; // onVisibilityChanged handler in script
    //uint visibility_status : 1;                // current visibility status
    uint awaiting_on_size_handling : 1;        // true if it has pending
                                               // size_change_notifiers.
    //uint style_assignment : 1;                 // inside on_style_changed.
    //uint need_style_update : 1;                // style needs to be updated after changes in on_style_changed handler.    
    uint model_valid : 1; // false - model needs to be reset.
    uint in_setup_layout : 1;
    uint in_determine_style : 1;
    // uint inside_editor          :1; // true if the element is inside
    // behavior:richtext  uint synthetic_type         :2; // see
    // SYNTHETIC_ELEMENT_TYPE
    uint strayed : 1; // true if element is in process of DOM detachment
    uint has_letter_spacing : 1; // it of its children have letter-spacing
                                 // defined
    uint was_drawn : 1; // element was drawn, false if invalidate_rect was
                        // issued on it.
    uint discard_animation : 1; // discard animation creation.

    uint was_a_style_change : 1; // true if it had element @style change in runtime

#if defined(SCITER) || defined(SCITERJS)
    uint script_draws_background : 1; // element has drawing methods in script
    uint script_draws_content : 1;    //
    uint script_draws_foreground : 1; //
    uint script_draws_outline : 1;    //
    uint script_has_is_point_inside : 1; //
    uint script_need_attached_call : 1; //
    uint script_reactors_prototype : 1; // prototype controlled by reactor
    uint ssx_reconciliation        : 2; // 0 - default, 1 - disable, 2 - enable 
    uint has_global_subscriptions : 1;
#endif

    uint layered : 1;  // has behavior layer attached
    uint has_sibling_selector : 1; // CSS has selectors this + that, or this ~ that

    uint marker_layout_valid : 1;   // ::marker
    uint shadow_layout_valid : 1;   // ::shadow

#ifdef _DEBUG
    uint debug_trigger : 1;
#endif
    
    uint has_percent_widths  :1;    // there are children that use %
    uint has_percent_heights :1;    // there are children that use %

    //uint svg_context : 1;           // 1 if the element is in SVG context

    uint animating_scroll : 1;

    uint virtual_v_scrollbar : 1;
    uint virtual_h_scrollbar : 1;

    uint suppress_change_event : 1;   //

    //uint popup_attachment : 2; // 0-top [to anchor bottom], 1-right, 2-bottom, 3-left[to anchor right]

    int  index;    // DOM index
    int  ui_index; // index in render tree;
    //uint scan_generation; // each_any_child
  };

  inline int absneg(int v) {
    if (v < 0) return -v;
    return 0;
  }
  inline int abspos(int v) {
    if (v > 0) return v;
    return 0;
  }
  inline int overlap(int ppix, int npix) {
    return max(abspos(ppix), abspos(npix)) - max(absneg(ppix), absneg(npix));
  }

  template <typename E> inline int first_comparable(int n, E &env) {
    int n_elements = env.total();
    for (; n < n_elements; ++n)
      if (env.is_comparable(n)) return n;
    return n;
  }
  template <typename E> inline int last_comparable(int n, E &env) {
    // int n_elements = env.total();
    for (; n >= 0; --n)
      if (env.is_comparable(n)) return n;
    return n;
  }

  template <typename E> int bsearch(E &env) {
    int low  = first_comparable(0, env);
    int high = last_comparable(env.total() - 1, env);
    while (low < high) {
      int mid = low + ((high - low) / 2);

      int tmid = last_comparable(mid, env);
      if (tmid < low) {
        tmid = first_comparable(mid, env);
        if (tmid == high) return high;
        if (tmid > high) return -1;
      }
      mid = tmid;

      if (env.is_less(mid))
        low = mid + 1;
      else
        // can't be high = mid-1: here A[mid] >= value,
        // so high can't be < mid if A[mid] == value
        high = mid;
    }

    if ((low < env.total()) && (env.is_equal(low)))
      return low; // found
    else
      return -1; // not found
  }

  struct element_or_text {
    ustring         text;
    handle<element> elem;
  };

  bool produce_filter_graph(view &v, element *el, slice<value> filters,
                            gool::filter_graph_builder *dest, bool as_ppx = true);

  namespace ease // animation, ease functions
  {
    struct ease_params {
      float x1 = 0; 
      float y1 = 0; 
      float x2 = 0; 
      float y2 = 0;
      bool operator == (const ease_params& other) const { return x1 == other.x1 && y1 == other.y1 && x2 == other.x2 && y2 == other.y2; }
      bool operator != (const ease_params& other) const { return x1 != other.x1 || y1 != other.y1 || x2 != other.x2 || y2 != other.y2; }
    };

    typedef float function_t(ease_params& p,float t, float b, float c, float d);

    struct function 
    {
      function_t* pf;
      uint_ptr    inversed; // must be uint but not bool, participates in styles
      ease_params params;

      function(function_t* p = nullptr, bool inv = false):pf(p),inversed(inv) {}
      function(const ease_params& pars);
      function(const function& other) :pf(other.pf), inversed(other.inversed), params(other.params) {}
      function& operator=(const function& other) { pf = other.pf; inversed = other.inversed; params = other.params; return *this; }
      float operator()(float t, float b, float c, float d) { return inversed ? (c - pf(params,d - t, 0, c, d) + b) : pf(params, t, b, c, d); }
      uint hash() const { return (uint)(uint_ptr)pf + uint(inversed) + hash_value(params.x1) + hash_value(params.y1) + hash_value(params.x2) + hash_value(params.y2); }

      explicit operator bool() const { return pf != nullptr; }
      bool operator !() const { return pf == nullptr; }

      bool operator == (const function& other) const { return pf == other.pf && inversed == other.inversed && params == other.params; }
      bool operator != (const function& other) const { return pf != other.pf || inversed != other.inversed || params != other.params; }

      void clear() { pf = nullptr; }

      bool is_undefined() const { return pf == nullptr; }
      bool is_defined() const { return pf != nullptr; }
      bool is_inherit() const;

      function val() const;
            
      void inherit(const function &v) {
        if (v.is_defined())
          *this = v;
      }

    };

    function get_ease_func(const value &name);
    function get_default_ease_func();

    float inherit(ease_params& p,float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float none(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float linear(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_quad(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_quad(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_quad(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_cubic(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_cubic(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_cubic(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_quart(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_quart(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_quart(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_quint(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_quint(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_quint(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_sine(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_sine(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_sine(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_expo(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_expo(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_expo(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_circ(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_circ(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_circ(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_elastic(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_elastic(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_elastic(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_back(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_back(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_back(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_back_x(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_back_x(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_back_x(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_back_xx(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_back_xx(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_back_xx(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float out_bounce(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_bounce(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);
    float in_out_bounce(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);

    float cubic_bezier(ease_params& p, float t, float b = 0.f, float c = 1.f, float d = 1.f);

  } // namespace ease

  struct box_shadow_params {
    bool inner;
    size offset;
    int  radius;
    int  distance;
    argb color;
    // shape
    std::array<size, 4> bradius; // topleft topright, bottomright, bottomleft
    size                dim;

    box_shadow_params()
        : inner(false), radius(0), distance(0), color(0, 0, 0, 0) {}

    bool operator==(const box_shadow_params &rs) const {
      return inner == rs.inner && offset == rs.offset && radius == rs.radius &&
             distance == rs.distance && color == rs.color &&
             bradius == rs.bradius && dim == rs.dim;
    }
    bool operator!=(const box_shadow_params &rs) const {
      return !operator==(rs);
    }
    unsigned int hash() const {
      uint v = 13;
      hash_combine(v, hash_value(uint(inner)));
      hash_combine(v, hash_value(offset.x));
      hash_combine(v, hash_value(offset.y));
      hash_combine(v, hash_value(radius));
      hash_combine(v, hash_value(distance));
      hash_combine(v, color.hash());
      for (uint n = 0; n < bradius.size(); ++n) {
        hash_combine(v, hash_value(bradius[n].x));
        hash_combine(v, hash_value(bradius[n].y));
      }
      hash_combine(v, hash_value(dim.x));
      hash_combine(v, hash_value(dim.y));
      return v;
    }
  };

  class image_map;

  class image_map_fragment : public gool::image {
  public:
    DEFINE_TYPE_ID_DERIVED(image_map_fragment,image);

    image_map_fragment(image_map *im, const ustring &name,
                       const image_filter *pf      = 0,
                       MAPPING_TYPE        mapping = mapping_default);

    virtual bool is_valid() const;
    virtual bool is_transparent() const;
    virtual bool is_animated(uint_ptr /*site_id*/) const { return false; }

    virtual gool::handle<bitmap> get_bitmap(gool::graphics *pg, size sz) override;

    virtual void draw(gool::graphics *gfx, rect dst, rect src,
                      byte opacity) override;
    virtual void draw(gool::graphics *gfx, rectf dst, rect src,
                      byte opacity) override;
    virtual void expand(gool::graphics *gfx, const rect &dst,
                        const SECTION_DEFS &sds, rect area) override;
    virtual bool is_solid_color(rect src, argb &clr) override;

    virtual size dim() const override;
    virtual void drop_cache() override;

    // string  get_url() const { return url; }
    // void    set_url(const string& s) { url = s; }

    virtual image *mapped_left_to_right() override {
      image_map_fragment *n =
          new image_map_fragment(map, name, filters, mapping_left_to_right);
      return n ? n : this;
    }
    virtual image *mapped_top_to_right() override {
      image_map_fragment *n =
          new image_map_fragment(map, name, filters, mapping_top_to_right);
      return n ? n : this;
    }

    virtual image *transform(const image_filter *ptran) override {
      image_map_fragment *n = new image_map_fragment(map, name, ptran, mapping);
      return n ? n : this;
    }

    bool fetch(view &v, document *pd);

    virtual bool is_bitmap() const override;
    virtual ~image_map_fragment() {}

    handle<image_map>    map;
    tool::ustring        name;
    handle<image_filter> filters;
    MAPPING_TYPE         mapping;
  };

  class view;
  struct document;

  class image_map : public gool::image {
    friend struct image_ref;
    friend struct document;

  public:
    struct image_ref_dpi {
      image_ref iref; // the image ref;
      int       dpi;  // max dpi supported by the image;
      enum_v    dir;  // direction_ltr, direction_rtl, etc.
    };

    DEFINE_TYPE_ID_DERIVED(image_map,image);

    image_map() : cols(0), rows(0) {}
    // static image* create(bytes data, const string& url);
    // static image_map* create(array<byte>& data, const string& url); //
    // creates an image and stores the data in it

    virtual bool is_valid() const override {
      return base(0) && rows && cols && base(0)->is_valid();
    }
    virtual bool is_transparent() const override {
      return base(0) && base(0)->is_transparent();
    } // has alpha bits
    virtual bool is_animated(uint_ptr /*site_id*/) const override {
      return false;
    }

    virtual gool::handle<bitmap> get_bitmap(gool::graphics *pg, size sz) override {
      return base(0) ? base(0)->get_bitmap(pg, sz) : 0;
    }

    virtual void draw(gool::graphics *gfx, rect dst, rect src,
                      byte opacity) override {
      if (base(0)) base(0)->draw(gfx, dst, src, opacity);
    }
    virtual void draw(gool::graphics *gfx, rectf dst, rect src,
                      byte opacity) override {
      if (base(0)) base(0)->draw(gfx, dst, src, opacity);
    }
    virtual void expand(gool::graphics *gfx, const rect &dst,
                        const SECTION_DEFS &sds, rect area) override {
      if (base(0)) expand(gfx, dst, sds, area);
    }
    virtual bool is_solid_color(rect src, argb &clr) override {
      return base(0) ? base(0)->is_solid_color(src, clr) : false;
    }

    virtual size dim() const override {
      return base(0) ? base(0)->dim() : size();
    }
    // virtual size dim(const rect& for_dst) const { return base()?
    // base()->dim(for_dst): size();  }
    virtual void drop_cache() override {
      if (base(0)) base(0)->drop_cache();
    }

    // string  get_url() const { return url; }
    // void    set_url(const string& s) { url = s; }

    virtual bool is_bitmap() const override {
      return base(0) ? base(0)->is_bitmap() : false;
    }
    virtual ~image_map() {}

    // virtual image* transform(const image_filter* ptran) { return this; }
    // virtual image* mapped_left_to_right() { return this; }
    // virtual image* mapped_top_to_right() { return this; }

    bool fragment_is_valid(const image_map_fragment *fr) const {
      return !part_area(fr->name, fr->mapping).empty();
    }
    void fragment_draw(const image_map_fragment *fr, gool::graphics *gfx, rect dst, rect src, byte opacity);
    void fragment_draw(const image_map_fragment *fr, gool::graphics *gfx, rectf dst, rect src, byte opacity);

    void fragment_expand(const image_map_fragment *fr, gool::graphics *gfx,
                         const rect &dst, const SECTION_DEFS &sds) {
      rect cr = part_area(fr->name, fr->mapping);
      if (cr.empty()) return;
      base(fr->filters, fr->mapping)
          ->expand(gfx, dst, sds, cr); // src + cr.s,opacity
    }
    bool fragment_is_solid_color(const image_map_fragment *fr, rect src,
                                 argb &clr) {
      rect cr = part_area(fr->name, fr->mapping);
      if (cr.empty()) return false;
      return base(fr->filters, fr->mapping)
          ->is_solid_color(src + cr.s, clr);
    }
    size fragment_dim(const image_map_fragment *fr) const {
      return part_area(fr->name, fr->mapping).size();
    }
    void fragment_drop_cache(const image_map_fragment * /*fr*/) {
      if (base(0)) base(0)->drop_cache();
    }
    bool fragment_is_bitmap() const {
      return base(0) ? base(0)->is_bitmap() : false;
    }

    bool fetch(view &v, document *pd);

  protected:
    gool::image *base(const image_filter *ptran,
                      MAPPING_TYPE        mapping = mapping_default) const {
      gool::image *pimg = iref().img();
      if (!pimg) return 0;
      switch (mapping) {
      case mapping_left_to_right: pimg = pimg->mapped_left_to_right(); break;
      case mapping_top_to_right: pimg = pimg->mapped_top_to_right(); break;
      default: break;
      }
      if (ptran) pimg = pimg->transform(ptran);
      return pimg;
    }
    image_ref &iref() const;

    array<image_ref_dpi> irefs; // multiple images for different resolutions
    hash_table<ustring, rect> parts;
    int                       cols;
    int                       rows;
    size cell_size(MAPPING_TYPE mapping) const; // pixel size of single cell
    rect part_area(const ustring &name, MAPPING_TYPE mapping) const;
  };

  inline image_map_fragment::image_map_fragment(image_map *         im,
                                                const ustring &     nm,
                                                const image_filter *pf,
                                                MAPPING_TYPE        mp)
      : map(im), name(nm), filters(pf), mapping(mp) {}

  inline handle<gool::bitmap> image_map_fragment::get_bitmap(gool::graphics *pg, size sz){
    assert(false); // image_map_fragment as repeatable image, this requires bitmap_fragment implementation, sigh. 
    return map ? map->get_bitmap(pg, sz) : 0;
  }

  inline bool image_map_fragment::is_transparent() const {
    return map && map->is_transparent();
  } // has alpha bits

  inline bool image_map_fragment::is_valid() const {
    return map && map->is_valid() && map->fragment_is_valid(this);
  }

  inline void image_map_fragment::draw(gool::graphics *gfx, rect dst, rect src,
                                       byte opacity) {
    if (map) map->fragment_draw(this, gfx, dst, src, opacity);
  }

  inline void image_map_fragment::draw(gool::graphics *gfx, rectf dst, rect src,
                                       byte opacity) {
    if (map) map->fragment_draw(this, gfx, dst, src, opacity);
  }

  inline void image_map_fragment::expand(gool::graphics *gfx, const rect &dst,
                                         const SECTION_DEFS &sds, rect area) {
    if (map) map->fragment_expand(this, gfx, dst, sds);
  }

  inline bool image_map_fragment::is_solid_color(rect src, argb &clr) {
    if (map) return map->fragment_is_solid_color(this, src, clr);
    return false;
  }

  inline size image_map_fragment::dim() const {
    return map ? map->fragment_dim(this) : size();
  }

  inline void image_map_fragment::drop_cache() {
    if (map) map->drop_cache();
  }

  inline bool image_map_fragment::is_bitmap() const {
    return map ? map->is_bitmap() : false;
  }

  inline size
  image_map::cell_size(MAPPING_TYPE mt) const // pixel size of single cell
  {
    image *pim = base(0);
    if (pim) {
      size t = pim->dimension() / size(cols, rows);
      if (mt == mapping_top_to_right) swap(t.x, t.y);
      return t;
    }
    return size();
  }
  inline rect image_map::part_area(const ustring &name, MAPPING_TYPE mt) const {
    size cs = cell_size(mt);
    rect r;
    if (!cs.empty() && parts.find(name, r)) {
      switch (mt) {
      case mapping_left_to_right:
        r.s.x = cols - 1 - r.s.x;
        r.e.x = cols - 1 - r.e.x;
        break;
      case mapping_top_to_right:
        swap(r.s.x, r.s.y);
        swap(r.e.x, r.e.y);
        break;
      default: break;
      }
      r.s.x *= cs.x;
      r.s.y *= cs.y;
      r.e.x *= cs.x;
      r.e.y *= cs.y;
    }
    return r;
  }

  template <class RT>
  inline bool each_resource(const array<handle<resource>> &ar, function<bool(const RT *r)> cb) {
    for (uint n = 0; n < ar.length(); ++n) {
      const resource *pr = ar[n];
      if (!pr->is_of_type<RT>()) continue;
      if (cb(static_cast<const RT *>(pr))) return true;
    }
    return false;
  }
  template <class RT> inline void remove_resource(array<handle<resource>> &ar) {
    for (int n = ar.size() - 1; n >= 0; --n) {
      const resource *pr = ar[n];
      if (!pr->is_of_type<RT>()) continue;
      ar.remove(n);
    }
  }


} // namespace html

#endif
