#ifndef __html_layout_types_h__
#define __html_layout_types_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-dom.h"
#include "html-primitives.h"

namespace html {
  using namespace tool;
  using namespace gool;

  // controller of float elements
  class floats_ctx : public resource {
    friend struct block;

    array<handle<element>> lefts;
    array<handle<element>> rights;

    range xr;
    // gool::range yr;

    element *sandbox;

  public:
    floats_ctx(view &v, element *parent);
    ~floats_ctx() {}

    void  push_left(view &v, int y, element *blk);
    void  push_right(view &v, int y, element *blk);
    range get_space_at(view &v, range y,
                       element *for_blk); // returns left and right margins at y

    int find_free_space(view &v, range y, int width,
                        element *for_blk); // returns first y where we can place
                                           // rect.size().x == width

    int get_next_y(view &v, int y, int clears); // return first clear position

    void fix_width(int w);

    bool is_empty() const { return lefts.size() == 0 && rights.size() == 0; }

    void reset(view &v);

    void remove(element *blk);
    bool has(element *blk) const;

    void     draw(view &v, graphics *pg, point pos);
    bookmark find(view &v, point pos);
    element *find_element(view &v, point zpos);

    int get_max_y(view &v);
    int get_min_width(view &v);
    int get_max_width(view &v);
  };

  // Z stack ctx
  class z_ctx {
  public:
    struct element_pos {
      helement el;
      point    pos; // postion of the element relative to the container
      bool     operator==(const element *rs) const { return el == rs; }
      bool     operator<(const element_pos& rs) const { return el->c_style->z_index < rs.el->c_style->z_index; }
      bool     operator>(const element_pos& rs) const { return el->c_style->z_index > rs.el->c_style->z_index; }
    };

    struct data // :public l2elem<data>
    {
      const element *    container;
      array<element_pos> arr;
      tristate_v         layout_valid;
      data(const element *cont = 0) : container(cont) {}
    };

    data *_data;

    z_ctx() : _data(0) {}
    ~z_ctx() {
      clear();
      delete _data;
    }

    void replace(view &v, element *container);
    void draw(view &v, graphics *sf, point p, element *container, bool above);
    void draw_owned_popups(view &v, graphics *sf, element *container);

    element *find_element(view &v, point pos, point nt_pos, element *container,
                          bool above = true);
    //                                  translated,not translated

    void push(view &v, element *container, element *b);
    void remove(element *b, element *container);

    void clear();

    rect content_outline(view &v, element *b);

    inline bool has_positioned() const {
      return _data && (_data->arr.size() > 0);
    }
    inline bool is_empty() const { return !_data || (_data->arr.size() == 0); }
    bool        has_fixed(view &v) const;
    void        refresh(view &v);
    void        refresh(view &v, element *pb);

    bool has_it(const element *b) const {
      if (_data) { return _data->arr.get_index(b) >= 0; }
      return false;
    }
    point get_position(const element *b) const {
      if (_data) {
        index_t n = _data->arr.get_index(b);
        if (n >= 0) return _data->arr[n].pos;
      }
      assert(0);
      return point();
    }
    void request_replace();
  };

  struct trans_ctx : resource // transformation ctx
  {
    affine_mtx_f mtx;
    void         clear() { mtx.reset(); }
  };

  struct list_marker : resource {
    list_style_type_e  type;
    int                item_no;
    handle<text_block> tl;
  };

  #define UNKNOWN_MAX_WIDTH 32000 // arbitrary number that designates max allowed width of element

  // layout controller data
  struct layout_data : resource {
    DEFINE_TYPE_ID(layout_data);

    layout_data()
        : border_width(0, 0, 0, 0), inner_border_width(0, 0, 0, 0), padding_width(0, 0, 0, 0), inner_padding_width(0,0,0,0),
          margin_width(0, 0, 0, 0), is_initialized(false),
          is_initializing(false), page_no_start(0), page_no_end(0) {}
    layout_data(element *)
        : border_width(0, 0, 0, 0), inner_border_width(0, 0, 0, 0), padding_width(0, 0, 0, 0), inner_padding_width(0, 0, 0, 0),
          margin_width(0, 0, 0, 0), is_initialized(false),
          is_initializing(false), page_no_start(0), page_no_end(0) {}
    virtual ~layout_data() {}

    // virtual flow_e type() const { return flow_null; }
    virtual void drop() {
      is_initialized = false;
      drop_minmax_dim();
      lmarker = nullptr;
      //n_synthetics = 0;
    }
    virtual void drop_minmax_dim() {
      dim_min.x.clear();
      dim_min.y.clear();
      dim_max.x.clear();
      dim_max.y.clear();
    }
    // data
    size      dim;
    point     pos;         // pos in static layouts
    point_v_t scroll_pos;  // scroll position - can be undefined
    size      offset;      // kind of scroll - used when above is undeinfed.
    dim_v_t   dim_min;     // min-content without percents 
    dim_v_t   dim_max;     // max-content without percents 
    dim_v_t   content_dim; // used content dim 
    int_v     baseline;
    size      inner_dim; // measured inner dim - dim of content
    size      used_dim;  // used inner dim - used in on_size_changed detector.
    rect16    border_width;        
    rect16    inner_border_width;  // box-sizing: border-box
    rect      padding_width;
    rect      inner_padding_width; // box-sizing: border-box | padding-box
    rect      margin_width;
    rect      foreground_box; // foreground image box
    //uint      n_synthetics = 0;
    handle<floats_ctx>  fctx;
    z_ctx               zctx;
    handle<trans_ctx>   xctx;
    scrollbars          sb;
    handle<list_marker> lmarker;
    bool                is_initialized;
    bool                is_initializing;
    uint16 page_no_start; // pagination: page number where element starts
    uint16 page_no_end;   // pagination: page number where element ends

    virtual bool is_valid() {
      return dim_min.x.is_defined() && dim_min.y.is_defined() &&
             dim_max.x.is_defined() && dim_max.y.is_defined() &&
             inner_dim == used_dim;
    }

    // offset of margin box from content box
    int outer_left() const {
      return (margin_width.s.x) + border_width.s.x +
             padding_width.s.x;
    }
    int outer_right() const {
      return (margin_width.e.x) + border_width.e.x +
             padding_width.e.x;
    }
    int outer_top() const {
      return (margin_width.s.y) + border_width.s.y +
             padding_width.s.y;
    }
    int outer_bottom() const {
      return (margin_width.e.y) + border_width.e.y +
             padding_width.e.y;
    }

    // offset of border box from content box
    int borpad_left() const {  return border_width.s.x + padding_width.s.x; }
    int borpad_right() const { return border_width.e.x + padding_width.e.x; }
    int borpad_top() const { return border_width.s.y + padding_width.s.y; }
    int borpad_bottom() const { return border_width.e.y + padding_width.e.y; }

    int inner_borpad_left() const { return inner_border_width.s.x + inner_padding_width.s.x; }
    int inner_borpad_right() const { return inner_border_width.e.x + inner_padding_width.e.x; }
    int inner_borpad_top() const { return inner_border_width.s.y + inner_padding_width.s.y; }
    int inner_borpad_bottom() const { return inner_border_width.e.y + inner_padding_width.e.y; }

    // additional space added to min/max content calculations
    int inner_borpad_x() const { return inner_borpad_left() + inner_borpad_right(); }
    int inner_borpad_y() const { return inner_borpad_top() + inner_borpad_bottom(); }

    INLINE rect used_border_width() const { 
      return rect(border_width.s + inner_border_width.s,
                  border_width.e + inner_border_width.e);
    }

    INLINE rect content_box() const { return rect(dim); }
    INLINE rect padding_box() const { return content_box() >> padding_width; }
    INLINE rect border_box() const { return padding_box() >> border_width; }
    INLINE rect margin_box() const { return border_box() >> margin_width; }

    virtual void push(view &v, element *self, element *el) { 
#ifdef _DEBUG
      self->dbg_report("unknown layout_data on");
#endif 
      assert(false); 
    }
    virtual element *next_ui_element(element *el) { return el->next_element(); }
    virtual element *prev_ui_element(element *el) { return el->prev_element(); }

    NONCOPYABLE(layout_data)
  };

  template <typename TO_T> inline TO_T *turn_element_to(element *p) {
    static_assert(sizeof(element) == sizeof(TO_T), "incompatible types");
    if (int(p->layout_type()) == int(TO_T::LAYOUT_TYPE) &&
        p->ldata->is_of_type<typename TO_T::layout_data>())
      return static_cast<TO_T *>(p);

    flow_e ft = p->layout_type();

    p->flags.layout_ctl_valid = 1;
    TO_T *pt = int(ft) != int(TO_T::LAYOUT_TYPE) ? ::new (p) TO_T(NO_INIT())
                                                 : static_cast<TO_T *>(p);
    pt->ldata = new typename TO_T::layout_data(pt);

    return pt;
  }

} // namespace html

#endif
