#ifndef __html_layout_text_flow_h__
#define __html_layout_text_flow_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-dom.h"
#include "html-primitives.h"

#ifdef WINDOWS

#include <usp10.h>

#else
struct SCRIPT_STATE {
  uint16 uBidiLevel : 5; // Unicode Bidi algorithm embedding level (0-16)
  uint16 fOverrideDirection : 1; // Set when in LRO/RLO embedding
  uint16 fInhibitSymSwap : 1;    // Set by U+206A (ISS), cleared by U+206B (ASS)
  uint16 fCharShape : 1;       // Set by U+206D (AAFS), cleared by U+206C (IAFS)
  uint16 fDigitSubstitute : 1; // Set by U+206E (NADS), cleared by U+206F (NODS)
  uint16 fInhibitLigate : 1; // Equiv !GCP_Ligate, no Unicode control chars yet
  uint16 fDisplayZWG : 1; // Equiv GCP_DisplayZWG, no Unicode control characters
                          // yet
  uint16 fArabicNumContext : 1; // For EN->AN Unicode rule
  uint16 fGcpClusters : 1; // For Generating Backward Compatible GCP Clusters
                           // (legacy Apps)
  uint16 fReserved : 1;
  uint16 fEngineReserved : 2; // For use by shaping engine
};

/////   SCRIPT_ANALYSIS
//
//      Each analysed item is described by a SCRIPT_ANALYSIS structure.
//      It also includes a copy of the Unicode algorithm state (SCRIPT_STATE).
//
//
struct SCRIPT_ANALYSIS {
  uint16 eScript : 10;      // Shaping engine
  uint16 fRTL : 1;          // Rendering direction
  uint16 fLayoutRTL : 1;    // Set for GCP classes ARABIC/HEBREW and LOCALNUMBER
  uint16 fLinkBefore : 1;   // Implies there was a ZWJ before this item
  uint16 fLinkAfter : 1;    // Implies there is a ZWJ following this item.
  uint16 fLogicalOrder : 1; // Set by client as input to ScriptShape/Place
  uint16 fNoGlyphIndex : 1; // Generated by ScriptShape/Place - this item does
                            // not use glyph indices
  SCRIPT_STATE s;
};
//
#endif

namespace html {
  namespace tflow {
    using namespace tool;
    using namespace gool;

    enum BREAK_CONDITION {
      /// <summary>
      /// Whether a break is allowed is determined by the condition of the
      /// neighboring text span or inline object.
      /// </summary>
      BREAK_CONDITION_NEUTRAL,

      /// <summary>
      /// A break is allowed, unless overruled by the condition of the
      /// neighboring text span or inline object, either prohibited by a
      /// May Not or forced by a Must.
      /// </summary>
      BREAK_CONDITION_CAN_BREAK,

      /// <summary>
      /// There should be no break, unless overruled by a Must condition from
      /// the neighboring text span or inline object.
      /// </summary>
      BREAK_CONDITION_MAY_NOT_BREAK,

      /// <summary>
      /// The break must happen, regardless of the condition of the adjacent
      /// text span or inline object.
      /// </summary>
      BREAK_CONDITION_MUST_BREAK
    };

    //#pragma pack(push,8)
    struct LINE_BREAKPOINT {
      /// <summary>
      /// Breaking condition before the character.
      /// </summary>
      uint8 break_condition_before : 2; // BREAK_CONDITION

      /// <summary>
      /// Breaking condition after the character.
      /// </summary>
      uint8 break_condition_after : 2; // BREAK_CONDITION

      /// <summary>
      /// The character is some form of whitespace, which may be meaningful
      /// for justification.
      /// </summary>
      uint8 is_whitespace : 1;

      /// <summary>
      /// The character is a soft hyphen, often used to indicate hyphenation
      /// points inside words.
      /// </summary>
      uint8 is_soft_hyphen : 1;

      uint8 padding : 2;
    };
    //#pragma pack(pop)

    /// <summary>
    /// Direction for how reading progresses.
    /// </summary>
    enum READING_DIRECTION {
      /// <summary>
      /// Reading progresses from left to right.
      /// </summary>
      READING_DIRECTION_LEFT_TO_RIGHT,

      /// <summary>
      /// Reading progresses from right to left.
      /// </summary>
      READING_DIRECTION_RIGHT_TO_LEFT
    };

#if defined(WINDOWS) && defined(DWRITE_LINE_BREAKPOINT)
    static_assert(sizeof(LINE_BREAKPOINT) == sizeof(DWRITE_LINE_BREAKPOINT),
                  "LINE_BREAKPOINT != DWRITE_LINE_BREAKPOINT");
#endif

    /// <summary>
    /// Optional adjustment to a glyph's position. An glyph offset changes the
    /// position of a glyph without affecting the pen position. Offsets are in
    /// logical, pre-transform units.
    /// </summary>
    struct GLYPH_OFFSET {
      /// <summary>
      /// Offset in the advance direction of the run. A positive advance offset
      /// moves the glyph to the right (in pre-transform coordinates) if the run
      /// is left-to-right or to the left if the run is right-to-left.
      /// </summary>
      float advance_offset;

      /// <summary>
      /// Offset in the ascent direction, i.e., the direction ascenders point. A
      /// positive ascender offset moves the glyph up (in pre-transform
      /// coordinates).
      /// </summary>
      float ascender_offset;
    };

    struct text_flow;

    struct text_run {
      text_run() throw()
          : text_node_start(), text_start(), text_length(), glyph_start(),
            glyph_count(), script_analysis()
      // no_visual(),
      // is_number_substituted(),
      // is_sideways()
      // bidi_level(),
      //,do_glyphs_fallback()
      {}

      uint text_node_start; // starting text position of this run in node chars
                            // array
      uint text_start;      // starting text position of this run
      uint text_length;     // number of contiguous code units covered
      uint glyph_start;     // starting glyph in the glyphs array
      uint glyph_count;     // number of glyphs associated with this run of text
      // DWRITE_SCRIPT_ANALYSIS script;
      // byte bidi_level;
      SCRIPT_ANALYSIS script_analysis;
      // uint no_visual:1; // true if the run has no visuals
      // uint is_number_substituted :1;
      // uint is_sideways :1;
      handle<style> pstyle;
      handle<font>  used_font;
      char_mark_t     marks;

      tool::handle<html::node> node;

      inline bool contains_text_position(uint desired_text_position) const
          throw() {
        return desired_text_position >= text_start &&
               desired_text_position < text_start + text_length;
      }

      inline bool operator==(uint desired_text_position) const throw() {
        // Search by text position using std::find
        return contains_text_position(desired_text_position);
      }

      wchars chars(const wchar *text) const {
        return wchars(text + text_start, text_length);
      }
      bool is_inline_block_pos(wchars text) const {
        return text_length == 1 && text[text_start] == 0;
      }

      bool is_rtl() const throw() { return bool(bidi_level() & 1); }
      byte bidi_level() const { return script_analysis.s.uBidiLevel; }
      void bidi_level(byte b) { script_analysis.s.uBidiLevel = b; }

      bool no_visual() const {
        return script_analysis.fNoGlyphIndex;
      } // true if the run has no visuals
      void no_visual(bool b) { script_analysis.fNoGlyphIndex = b; }

      bool is_number_substituted() const {
        return script_analysis.s.fDigitSubstitute;
      }
      void is_number_substituted(bool b) {
        script_analysis.s.fDigitSubstitute = b;
      }

      bool is_sideways() const { return script_analysis.s.fReserved; }
      void is_sideways(bool b) { script_analysis.s.fReserved = b; }

      style *  get_style(view &v) const;
      element *get_element() const;
      font *   get_used_font(view &v) const;
    };

    struct glyph_run {
      glyph_run()
          : glyph_start(), glyph_count(), bidi_level(), is_sideways(), x(),
            valign(valign_baseline) {}

      inline bool contains_glyph_index(uint glyph_index) const throw() {
        return glyph_index >= glyph_start &&
               glyph_index < glyph_start + glyph_count;
      }

      inline bool operator==(uint glyph_index) const throw() {
        // Search by text position using std::find
        return contains_glyph_index(glyph_index);
      }

      uint32 glyph_end() const throw() { return glyph_start + glyph_count; }

      bool is_rtl() const throw() { return bool(bidi_level & 1); }

      slice<float> glyph_advances(const text_flow &tf) const;

      float reduce_to_fit(const text_flow &tf, float width, bool first_run);

      ~glyph_run() {}

      tool::handle<html::style> pstyle;
      tool::handle<html::node>  node;
      tool::handle<gool::font>  font;
      float                     x;
      uint32                    glyph_start; // in _glyph_indices
      uint32                    glyph_count;
      byte                      bidi_level;
      bool                      is_sideways;
      byte                      valign;
      uint                      line_no;
      char_mark_t                 marks;

      element *get_element() const;
      element *get_inline_block_element(view &v) const;

      style *get_style(view &v) const {
        assert(node);
        if (pstyle.is_defined()) return pstyle;
        element *pel = get_element();
        return pel ? pel->get_style(v) : nullptr;
      }
    };

    struct layout_line {
      int    y;
      int    baseline;  // offset
      int    height;    // height
      uint   first_run; // first glyph run
      uint   last_run;  // last glyph run (inclusive)
      uint   start_text_index;
      uint   end_text_index;
      range  yr() const { return range(y, y + height); }
      uint16 page_no;
      uint16 column_no;
    };

    struct sel_context {
      argb selection_back_color;
      argb selection_fore_color;
      argb selection_text_color;
      uint sel_glyph_start;
      uint sel_glyph_end;
      uint ime_glyph_start;
      uint ime_glyph_end;
      sel_context()
          : sel_glyph_start(0), sel_glyph_end(0), ime_glyph_start(0),
            ime_glyph_end(0) {}
    };

    struct cluster_position_t {
      cluster_position_t() : text_position(), runIndex(), runEndPosition() {}

      uint32 text_position;  // Current text position
      uint32 runIndex;       // Associated analysis run covering this position
      uint32 runEndPosition; // Text position where this run ends
    };

    // flow_layout_source returns x range - space available for content at y
    // position *inside* the element
    // typedef function<bool(element* elem, int y, gool::range& x)>
    // flow_layout_source;
    // callback method used to attach floating element to the right or to the
    // left
    // typedef function<void(element* elem, int y, bool to_the_right)>
    // flow_layout_float_attach;

    struct text_flow : public element::layout_data {
      typedef element::layout_data super;
      friend struct html::text_block;

    public:
      text_flow()
          : is_locked(false)
      //_number_substitution(),
      //_reading_direction(DWRITE_READING_DIRECTION_LEFT_TO_RIGHT),
      //_max_space_width(8),
      //_is_text_analysis_complete(false)
      {}

      text_flow(element *b) : super(b), is_locked(false) {}

      virtual ~text_flow() { assert(!is_initializing && !is_locked); }

      bool is_empty_text() const {
        return (_text.size() == 1) && (_text[0] == EMPTY_STRING_PLACEHOLDER);
      }

      // STDMETHODIMP set_text_format(text_format* textFormat);
      // STDMETHODIMP set_number_substitution(IDWriteNumberSubstitution*
      // numberSubstitution);

      // perform analysis on the given text, converting text to glyphs.
      // fills _runs, _breakpoints, _glyph_clusters, etc. fields
      void setup(view &v, element *elem, slice<hnode> nodes);
      void drop() {
        super::drop();
        //assert(!is_initializing && !is_locked);
        if (is_initializing || is_locked)
          return;

        _text.clear();
        _runs.clear();
        _breakpoints.clear();
        _glyph_clusters.clear();
        _glyph_indices.clear();
        _glyph_advances.clear();
        _glyph_justified_advances.clear();
        _glyph_offsets.clear();
        _glyph_runs.clear();
        _lines.clear();
        _used_nodes.clear();
      }

      // reflow the text analysis into
      int  flow_text(view &v, element *elem); // returns computed height
      void apply_letter_spacing(view &v, element *elem);

      bool is_breakable_space(uint at) const;

      // void shape_glyph_runs(view& v, element* elem, IDWriteTextAnalyzer*
      // textAnalyzer);

      // void shape_glyph_run(
      //    view& v, element* elem,
      //    IDWriteTextAnalyzer* textAnalyzer,
      //    uint32 runIndex,
      //    IN OUT uint32& glyph_start
      //    );

      enum FIT_TEXT_MODE {
        TRY_TO_FIT,
        FORCE_WHOLE_WORD,
        // FIT_AS_MUCH_AS_POSSIBLE,
      };

      struct inline_el_pos {
        helement elem;   // element 
        uint     index;  // index in _glyph_advances 
      };

      bool fit_text(view &v, element *elem, bool plain_text, int y,
                    const cluster_position_t &cluster_start, uint32 textEnd,
                    FIT_TEXT_MODE mode, cluster_position_t &cluster_end,
                    range &x, 
                    array<helement> &left_floats, array<helement> &right_floats, array<inline_el_pos> &flexes,
                    float &max_brick_width,
                    float &line_width);

      void calc_min_max(view &v, element *elem, int &out_min_width,
                        int &out_max_width); // calc min max widths

      bool setup_line( // create line and calc glyph runs
          view &v, element *elem, bool plain_text, int y, int x1, int x2,
          float line_width, slice<inline_el_pos> flexes,
          const cluster_position_t &clusterStart, const cluster_position_t &clusterEnd, int &line_height) throw();

      node *   find_node_at(view &v, point zpos, bool exact);
      bookmark find_bookmark_at(view &v, point zpos);

      void
      produce_justified_advances(view &v, element *elem, int y, int x1, int x2,
                                 const cluster_position_t &clusterStart,
                                 const cluster_position_t &clusterEnd
                                 //, OUT tool::array<float>& justifiedAdvances
                                 ) throw();

      void flex_children(view &v, element *elem, int x1, int x2,
        const cluster_position_t &clusterStart,
        const cluster_position_t &clusterEnd,
        tool::slice<inline_el_pos> flexes,
        float& line_width
      );

      //float produce_tab_advances(view &v, element *elem,
      //                           const cluster_position_t &clusterStart,
      //                           const cluster_position_t &clusterEnd
      //                           //, OUT tool::array<float>& justifiedAdvances
      //                           ) throw(); // returns extra space added

      float get_tab_width(view &v, element *elem, const cluster_position_t &clusterLineStart, const cluster_position_t &clusterStart, const cluster_position_t &clusterEnd) throw();

      void produce_bidi_ordering(uint32 spanStart, uint32 spanCount,  uint32 *spanIndices ) const throw();

      void set_cluster_position(cluster_position_t &cluster,
                                uint32 text_position) const throw();

      void advance_cluster_position(cluster_position_t &cluster) const throw();

      uint32 get_cluster_glyph_start(const cluster_position_t &cluster) const
          throw();

      float get_cluster_range_width(const cluster_position_t &clusterStart,
                                    const cluster_position_t &clusterEnd) const
          throw();

      bool get_glyph_metrics(uint32 glyph_index, caret_metrics &m) const
          throw();

      uint land_pos(element *self, bookmark bm) const throw();
      bool get_sel_glyph_positions(element *self, bookmark caret,
                                   bookmark anchor, uint &start_glyph,
                                   uint &end_glyph) const throw();

      // DWRITE_HIT_TEST_METRICS

      // bool get_text_index_at(
      //    pointf p,
      //    OUT uint32&  text_index
      //   ) const throw();

      uint glyph_index_2_text_position(uint glyph_index, bool last) const
          throw();
      uint     text_position_2_glyph_index(uint text_position) const throw();
      uint_v   node_position_2_text_position(const element * self,
                                             const bookmark &bm, bool &rtl,
                                             bool caret_pos = true) const throw();
      bookmark text_position_2_node_position(uint text_position) const throw();

      bool next_text_position(uint& text_position) const;
      bool prev_text_position(uint& text_position) const;

      struct index_direction {
        uint index : 31;
        uint rtl : 1;
        // bool operator==(const index_direction& rs) const { return index ==
        // rs.index && rtl == rs.rtl; }
        bool operator==(const uint i) const { return index == i; }
      };

      bool get_line_no(uint text_position, uint &line_no);
      bool
      text_positions_in_visual_order(uint                    line_no,
                                     array<index_direction> &text_positions);

      void get_text(view &v, array<wchar> &vo);

      bool is_caret_pos_at(const bookmark &bm) const;

      void get_metrics(view &v, const element *self, const bookmark &bm,
                       caret_metrics &m);

      float get_cluster_range_width(uint32 glyph_start, uint32 glyph_end,
                                    const float *glyph_advances // [glyphEnd]
                                    ) const throw();

      bool advance_cluster_position_next_brick(view &              v,
                                               cluster_position_t &cluster,
                                               bool &hard_lf) const
          throw(); // false if end reached

      bool advance_cluster_position_next_brick(
          view &v, element *self, 
          const cluster_position_t &line_start_cluster,
          const cluster_position_t &cluster, 
          cluster_position_t &next_cluster, 
          element *&inline_box_element,
          float &width, float &white_space_width, bool &hard_lf,
          float max_width) throw(); // false if end reached

      inline element * get_inline_box_element_at(view &v, const cluster_position_t &cluster) const {
        if (cluster.runEndPosition - cluster.text_position > 1) // inline box element is marked by cluster having length of 1
          return 0;
        node *nd = _runs[cluster.runIndex].node;
        if (!nd) return 0;
        if (nd->is_element() && !nd->is_inline_span_element(v)) {
          element *inline_box_element = static_cast<element *>(nd);
          inline_box_element->check_layout(v);
          return inline_box_element;
        }
        return 0;
      }

      // bool setup_sel_context(element* self, bookmark caret, bookmark anchor,
      // sel_context& sctx) const;

      void reset_glyph_runs();

      uint set_glyph_run(float x, byte valign,
                         uint32 glyph_start, // start of the run in
                                             // _glyph_indices, _glyph_advances
                                             // and _glyph_offsets
                         uint32 glyph_count, node *pnode, html::style *pstyle,
                         gool::font *font,
                         // float font_size,
                         uint8 bidi_level, bool is_sideways, uint line_no,
                         char_mark_t mark); // returns glyph_run index;

      uint               lines_count() const { return uint(_lines.length()); }
      void               set_line(const layout_line &ln) { _lines.push(ln); }
      const layout_line &get_line(uint ln) const { return _lines[ln]; }
      layout_line &      get_line_ref(uint ln) const {
        return const_cast<text_flow *>(this)->_lines[ln];
      }

      bool advance_caret_pos(view &v, const element *elem, bookmark &bm,
                             ADVANCE_DIR dir, array<wchar> *out = 0);
      bool advance_caret_pos_left(view &v, const element *self, bookmark &bm);
      bool advance_caret_pos_right(view &v, const element *self, bookmark &bm);
      bool advance_caret_pos_first(view &v, const element *self, uint line_no,
                                   bookmark &bm); // in visual order, sic!
      bool advance_caret_pos_last(view &v, const element *self, uint line_no,
                                  bookmark &bm);

      uint last_caret_pos();

      tool::array<hnode> _used_nodes; // nodes holder (these nodes may contain
                                      // also syntehtic before/after elements)

      // Input information.
      tool::array<wchar> _text;

      // Output text analysis results
      tool::array<text_run>        _runs;
      tool::array<LINE_BREAKPOINT> _breakpoints;

      tool::array<uint16>       _glyph_clusters;
      tool::array<uint16>       _glyph_indices;
      tool::array<float>        _glyph_advances;
      tool::array<float>        _glyph_justified_advances; // can be empty
      tool::array<GLYPH_OFFSET> _glyph_offsets;
      tool::array<glyph_run>    _glyph_runs;

      tool::array<layout_line> _lines;
      bool                     is_locked; // locked for drop content

      color_v _default_text_color;

      // float _max_space_width;           // maximum stretch of space allowed
      // for justification  bool  _is_text_analysis_complete; // text analysis was
      // done.
    };

    inline slice<float> glyph_run::glyph_advances(const text_flow &tf) const {
      return tf._glyph_advances(index_t(glyph_start), index_t(glyph_end()));
    }
    inline float glyph_run::reduce_to_fit(const text_flow &tf,float width, bool first_run) // returns position for the next run ( e.g. ellipsis )
    {
      slice<float> advances = glyph_advances(tf);
      float        pw       = 0.0f;
      float        w        = 0.0f;

      for (uint n = 0; n < advances.length; ++n) {
        pw = w;
        w += advances[n];
        if (w > width) {
          if ((n == 0) && first_run)
            continue; // at least one glyph in case of first run
          glyph_count = n;
          w = pw;
          break;
        }
      }
      return x + (is_rtl() ? -w : w);
    }

  } // namespace tflow

  inline float width_of(const tflow::text_flow &tf,
                        const tflow::glyph_run &run) {
    slice<float> glyph_widths =
        tf._glyph_advances(run.glyph_start, run.glyph_end());
    return glyph_widths.accumulate(0.0f);
  }

  inline float width_of(const tflow::text_flow &tf,
                        slice<tflow::glyph_run> runs) {
    float r = 0;
    while (!!runs)
      r += width_of(tf, runs++);
    return r;
  }

  inline gool::rangef x_of(const tflow::text_flow &tf,
                           const tflow::glyph_run &run) {
    float        w = width_of(tf, run);
    gool::rangef r;
    if (run.is_rtl()) {
      r.e = run.x;
      r.s = run.x - w;
    } else {
      r.s = run.x;
      r.e = run.x + w;
    }
    return r;
  }

} // namespace html

#endif
