#include "html.h"
#include "html-layout-text-flow.h"
#include "html-layout-text-analysis.h"
//#include "win/win-application.h"

namespace html {
  namespace tflow {
    using namespace tool;
    using namespace gool;

    // Estimates the maximum number of glyph indices needed to hold a string of
    // a given length.  This is the formula given in the Uniscribe SDK and
    // should cover most cases. Degenerate cases will require a reallocation.

    style *text_run::get_style(view &v) const {
      assert(node);
      if (this->marks) {
        if (!pstyle)
          const_cast<text_run *>(this)->pstyle = get_element()->resolve_style_for_mark(v, this->marks);
        return pstyle;
      }
      element *pel = get_element();
      if (pel)
        return pel->get_style(v);
      else
        return element::null_style;
    }

    font *text_run::get_used_font(view &v) const {
      if (!used_font)
        const_cast<text_run*>(this)->used_font = v.get_font(get_style(v));
      return used_font.ptr();
    }

    element *text_run::get_element() const {
      assert(node);
      return node->is_element() ? node.ptr_of<element>() : node->parent.ptr();
    }

    element *glyph_run::get_inline_block_element(view &v) const {
      return (glyph_count == 1 && node->is_element() &&
              !node->is_inline_span_element(v))
                 ? node.ptr_of<element>()
                 : 0;
    }

    element *glyph_run::get_element() const {
      assert(node);
      return node->is_element() ? node.ptr_of<element>() : node->parent.ptr();
    }

    void text_flow::setup(view &v, element *elem, slice<hnode> nodes) {
      assert(!is_locked);
      assert(is_initializing);

      // if(_is_measuring) alert("PANIC: text_flow::setup while _is_measuring");
      // if(_is_initializing) alert("PANIC: text_flow::setup while
      // _is_initializing");
      _used_nodes = nodes;
      for (auto n : nodes)
        n->owner = elem;

      v.setup_text_flow(elem, *this, nodes);
      if (elem->flags.has_letter_spacing) apply_letter_spacing(v, elem);
    }

    int text_flow::flow_text(view &v, element *elem) {
      assert(!is_initializing);
      auto_state<bool> _1(is_initializing, true);

      // Reflow all the text, from source to sink.
#ifdef DEBUG
      //if (elem->is_id_test())
      //  elem = elem;
#endif 

      // Get ready for series of glyph runs.
      this->reset_glyph_runs();

      if (_text.length() == 0) 
        return 0;

      int y = this->inner_borpad_top();

      const style *ecs         = elem->get_style(v);
      bool         allow_wrap  = ecs->can_wrap();
      bool         collapse_ws = ecs->collapse_ws();

      // Set initial cluster position to beginning of text.
      cluster_position_t cluster, next_cluster;
      set_cluster_position(cluster, 0);

      uint32 text_length = static_cast<uint32>(_text.length());

      int             line_height = 0;
      array<helement> left_floats;
      array<helement> right_floats;
      array<inline_el_pos> flexes; // elements that have flex widths

      // Iteratively pull rect's from the source,
      // and push as much text will fit to the sink.
      while (cluster.text_position < text_length) {
        range x;
        left_floats.clear();
        right_floats.clear();
        flexes.clear();
        float brick_width;
        float line_width;
        // fit as many clusters between breakpoints that will go in:
        bool atLeastOneWordThatFits =
            fit_text(v, elem, allow_wrap, y, cluster, text_length, TRY_TO_FIT,
                     next_cluster, x, left_floats, right_floats, flexes,
                     brick_width, line_width);
        if (!atLeastOneWordThatFits) {
          int w = int(brick_width + 0.5f);
          y     = v.find_free_space(elem, y, w);
          left_floats.clear();
          right_floats.clear();
          flexes.clear();
          fit_text(v, elem, allow_wrap, y, cluster, text_length, FORCE_WHOLE_WORD, 
                     next_cluster, x, left_floats, right_floats, flexes,
                     brick_width, line_width);
        }

        // setup the line with glyph runs:
        if (!setup_line(v, elem, collapse_ws, y, x.s, x.e, line_width, flexes(), cluster,next_cluster, line_height))
          break;
        y += line_height;
        line_height = 0;
        // we've got left floats pending to be replaced from fit_text, let's
        // push them after the line:
        for (int n = 0; n < left_floats.size(); ++n)
          v.push_left(left_floats[n], y, elem);
        // we've got right floats from fit_text, let's push them after the line:
        for (int n = 0; n < right_floats.size(); ++n)
          v.push_right(right_floats[n], y, elem);
        cluster = next_cluster;
      }
      y += inner_padding_width.e.y + inner_border_width.e.y;
      return y;
    }

    bool text_flow::advance_cluster_position_next_brick(
        IN view &v, IN element *self,
        IN const cluster_position_t &line_start_cluster,
        IN const cluster_position_t &start_cluster,
        OUT cluster_position_t &next_cluster, OUT element *&inline_box_element,
        OUT float &width, OUT float &white_space_width, OUT bool &hard_lf,
        IN float allowed_width) throw() {
      width                          = 0;
      white_space_width              = 0;
      hard_lf                        = false;
      inline_box_element             = 0;
      uint               text_length = (uint)_text.length();
      cluster_position_t cluster(start_cluster);
      if (cluster.text_position >= text_length) return false;

      inline_box_element = get_inline_box_element_at(v, cluster);

      if (inline_box_element) {
        if (inline_box_element->oof_positioned(v))
          width = 0;
        else {
          hstyle ecs = inline_box_element->get_style(v);
          inline_box_element->owner = self;
          size elem_dim = self->dim();
          inline_box_element->measure_borders_x(v, elem_dim);
          inline_box_element->measure_borders_y(v, elem_dim);
          inline_box_element->set_width(v, inline_box_element->computed_width(v, elem_dim.x));
          inline_box_element->set_height( v, inline_box_element->computed_height(v, elem_dim.y));
          
          //premeasure(v, inline_box_element, ecs, elem_dim);
          width = float(inline_box_element->margin_box(v).width());
          if (inline_box_element->tag == tag::T_BR)
            hard_lf = true; // we have a hard return, so we've fit all we can.
        }
        next_cluster = cluster;
        advance_cluster_position(next_cluster);
        /* WHAT A HECK IS THIS?
        for(;next_cluster.text_position < text_length;)
        {
            const DWRITE_LINE_BREAKPOINT breakpoint =
        _breakpoints[next_cluster.text_position]; if( !breakpoint.isWhitespace )
              break;
            cluster = next_cluster;
            advance_cluster_position(next_cluster);
            white_space_width += get_cluster_range_width(cluster, next_cluster);
        }*/
        return true;
      }

      uint n_clusters = 0;

      for (; cluster.text_position < text_length;
           cluster = next_cluster, ++n_clusters) {
        next_cluster = cluster;
        advance_cluster_position(next_cluster);

        // bool is_breakable_ws = bp.isWhitespace &&
        // _text[cluster.text_position]

        if (is_breakable_space(cluster.text_position)) {
          if( _text[cluster.text_position] == '\t' )
            white_space_width += get_tab_width(v, self, line_start_cluster, cluster, next_cluster);
          else
            white_space_width += get_cluster_range_width(cluster, next_cluster);
        } else {
          float chunk_width = get_cluster_range_width(cluster, next_cluster);
          if (n_clusters > 0 && ((width + chunk_width) >= allowed_width)) {
            next_cluster = cluster;
            break;
          }
          width += white_space_width;
          white_space_width = 0;
          width += get_cluster_range_width(cluster, next_cluster);
        }

        if (next_cluster.text_position >= text_length) break;

        // Use breakpoint information to find where we can safely break words.
        const LINE_BREAKPOINT breakpoint =
            _breakpoints[next_cluster.text_position - 1];
        // see if we can break after this character cluster, and if so, stop
        if (breakpoint.break_condition_after != BREAK_CONDITION_MAY_NOT_BREAK) {
          if (breakpoint.break_condition_after == BREAK_CONDITION_MUST_BREAK)
            hard_lf = true; // we have a hard return, so we've fit all we can.
          break;
        }
        if (get_inline_box_element_at(
                v, next_cluster)) // blocks are brick splitters too.
          break;
      }
      return true;
    }

    bool text_flow::is_breakable_space(uint at) const {
      if (at >= _breakpoints.length()) return false;
      return _breakpoints[at].is_whitespace && _text[at] != 160 /*&nbsp;*/;
    }

    void text_flow::apply_letter_spacing(view &v, element *elem) {
      uint32 text_length = static_cast<uint32>(_text.length());

      cluster_position_t cluster;
      set_cluster_position(cluster, 0);
      cluster_position_t next_cluster(cluster);

      const html::style *last_rstyle    = nullptr;
      float              letter_spacing = 0;

      for (; cluster.text_position < text_length; cluster = next_cluster) {
        next_cluster = cluster;
        advance_cluster_position(next_cluster);
        if (next_cluster.text_position >= text_length) break;

        if (get_inline_box_element_at(v, cluster)) continue;

        const html::style *rstyle = _runs[cluster.runIndex].get_style(v);
        if (!rstyle) rstyle = elem->c_style;

        if (rstyle != last_rstyle) {
          last_rstyle    = rstyle;
          letter_spacing = pixels(v, elem, rstyle->letter_spacing).height_f();
        }

        if (letter_spacing == 0.0f) continue;

        const html::style *next_rstyle =
            _runs[next_cluster.runIndex].get_style(v);
        if ((next_rstyle != rstyle) &&
            next_rstyle->letter_spacing.is_undefined())
          continue; // don't do it on last cluster

        uint32 last = get_cluster_glyph_start(next_cluster) - 1;
        if(last < _glyph_advances.length())
          _glyph_advances[last] += letter_spacing;
      }
    }

    bool text_flow::fit_text(view &v, element *elem, bool allow_wrap, int y,
                             const cluster_position_t &cluster_start,
                             uint32 textEnd, FIT_TEXT_MODE mode,
                             OUT cluster_position_t &cluster_end, OUT range &x,
                             OUT array<helement> &left_floats,
                             OUT array<helement> &right_floats,
                             OUT array<inline_el_pos> &flexes,
                             OUT float &brick_width, OUT float &line_width) {
      if (cluster_start.text_position >= textEnd) return false;

      // Fits as much text as possible into the given width,
      // using the clusters and advances returned by DWrite.

      ////////////////////////////////////////
      // Set the starting cluster to the starting text position,
      // and continue until we exceed the maximum width or hit
      // a hard break.
      cluster_position_t cluster(cluster_start);
      cluster_position_t next_cluster(cluster_start);

      const style *elem_style = elem->get_style(v);

      bool wrap_unrestricted = elem_style->wrap_unrestricted();
      bool break_at_any_position =
          elem_style->word_break == word_break_break_all;
      // uint  max_chars_in_word = wrap_on_word_boundaries? 20000:1;

      x = v.get_x_space_at(y, elem);

      if (cluster_start.text_position == 0) // first line
      {
        style *element_style = elem->get_style(v);
        if (element_style->text_indent.is_defined()) {
          int indent = pixels(v, elem, element_style->text_indent).width();
          if (element_style->direction == direction_rtl)
            x.e -= indent;
          else
            x.s += indent;
        }
      }

      float text_width = 0;
      float max_width  = float(x.length());
        
#if _DEBUG
        if(elem->is_id_test())
            elem = elem;
#endif

      float allowed_width = 10000.0f;
      if (wrap_unrestricted)
        allowed_width = break_at_any_position
                ? 1          // 1 run == 1 cluster
                : max_width; // 1 run contains chars that fit into the width

      uint     n_bricks            = 0; // num bricks in line
      bool     hard_lf             = false;
      element *inline_box_element  = 0;
      float    brick_ws_width      = 0;
      float    last_brick_ws_width = 0;

#if 0
    uint  spaces_per_tab = 0;
    float space_width = 0;
    uint  extra_spaces = 0;
#endif

      for (; cluster.text_position < textEnd; cluster = next_cluster) {
        if (!advance_cluster_position_next_brick(v, elem, cluster_start, cluster, next_cluster, inline_box_element, brick_width, brick_ws_width, hard_lf, allowed_width))
          break;

        if (inline_box_element) {
          if (inline_box_element->positioned(v))
            inline_box_element->check_positioned_containment(v);

          if (inline_box_element->oof_positioned(v)) {
            cluster = next_cluster;
            continue;
          }

          float_e fs = inline_box_element->floats(v);
          if (fs != float_none) {
            if (fs == float_left) {
              if (cluster.text_position != cluster_start.text_position && text_width + inline_box_element->margin_box(v).width() > x.length())
                left_floats.push(inline_box_element);
              else if (left_floats.size() || right_floats.size())
                left_floats.push(inline_box_element);
              else {
                x = v.push_left(inline_box_element, y, elem);
                ++n_bricks;
              }
            } else if (fs == float_right) {
              if (cluster.text_position != cluster_start.text_position &&
                  text_width + inline_box_element->margin_box(v).width() >
                      x.length())
                right_floats.push(inline_box_element);
              else if (left_floats.size() || right_floats.size())
                right_floats.push(inline_box_element);
              else {
                x = v.push_right(inline_box_element, y, elem);
                ++n_bricks;
              }
            }
            // x = v.get_x_space_at(elem, gool::range(y, y));
            max_width = min(max_width, float(x.length()));
            cluster   = next_cluster;
            continue;
          }
        }

        if (allow_wrap && (text_width + brick_width - max_width > 0)) {
          if (n_bricks == 0) {
            if (mode == FORCE_WHOLE_WORD) {
              ++n_bricks;
              cluster = next_cluster;
            } else if (mode == TRY_TO_FIT) {
              ;
            }
          }
          break;
        }

        if (inline_box_element && inline_box_element->h_flexes(v)) {
          flexes.push({ inline_box_element, _runs[cluster.runIndex].glyph_start });
        }

        ++n_bricks;
        text_width += brick_width + brick_ws_width;
        last_brick_ws_width = brick_ws_width;

        if (hard_lf) {
          cluster = next_cluster;
          break;
        }
      }

      cluster_end = cluster;

      line_width = text_width - last_brick_ws_width;

      return n_bricks > 0;
    }

    node *text_flow::find_node_at(view &v, point zpos, bool exact) {
      layout_line line;

      for (int n = 0; n < _lines.size(); ++n) {
        line = _lines[n];
        if (zpos.y >= line.y && zpos.y < line.y + line.height) goto LINE_FOUND;
      }
      return 0;
    LINE_FOUND:

      if (!_glyph_runs.length()) return 0;

      auto_state<bool> _1(is_locked, true);

      rect rc;
      rc.s.y = line.y;
      rc.e.y = line.y + line.height;

      for (uint i = line.first_run; i <= line.last_run; ++i) {
        const glyph_run &lf_glyph_run = _glyph_runs[i];

        if (lf_glyph_run.glyph_count == 0) continue;

        html::element *el = lf_glyph_run.get_inline_block_element(v);
        if (el) {
          if (el->floats(v) || el->positioned(v) || !el->is_visible(v))
            continue;
          element *t = el->find_element(v, zpos - el->pos(), exact);
          if (t) return t;
          continue;
        }

        if (lf_glyph_run.bidi_level & 1) {
          rc.e.x = (int)ceilf(lf_glyph_run.x);
          rc.s.x = (int)floorf(
              lf_glyph_run.x -
              get_cluster_range_width(lf_glyph_run.glyph_start,
                                      lf_glyph_run.glyph_start +
                                          lf_glyph_run.glyph_count,
                                      _glyph_justified_advances.head()));
        } else {
          rc.s.x = (int)floorf(lf_glyph_run.x);
          rc.e.x = (int)ceilf(
              lf_glyph_run.x +
              get_cluster_range_width(lf_glyph_run.glyph_start,
                                      lf_glyph_run.glyph_start +
                                          lf_glyph_run.glyph_count,
                                      _glyph_justified_advances.head()));
        }
        if (rc.x().contains(zpos.x)) return lf_glyph_run.node;
      }
      return 0;
    }

    /*inline bool can_contain_caret(view& v, element* el)
    {
      if( el->tag == tag::T_INPUT )
        return false;
      if( el->tag == tag::T_SELECT )
        return false;
      if( el->tag == tag::T_BUTTON )
        return false;
      if( el->tag == tag::T_IMG )
        return false;

      return el->is_of_type<text_block>();
    }*/

    bookmark text_flow::find_bookmark_at(view &v, point zpos) {

      assert(is_valid());

      layout_line line;
      if (_lines.size()) {
        line = _lines[0];
        if (zpos.y < line.y + line.height) goto LINE_FOUND;

        line = _lines.last();
        if (zpos.y >= line.y) goto LINE_FOUND;

        for (int n = 1; n < _lines.last_index(); ++n) {
          line = _lines[n];
          if (zpos.y < line.y + line.height) goto LINE_FOUND;
        }
      }
      return bookmark();
    LINE_FOUND:

      rect rc;
      rc.s.y = line.y;
      rc.e.y = line.y + line.height;

      if (!_glyph_runs.length()) {
        assert(false);
        return bookmark();
      }

      auto_state<bool> _1(is_locked, true);

      uint glyph_index = _glyph_runs[line.first_run].glyph_start;

      glyph_run lf_glyph_run;

      uint last_visible_run = line.first_run;

      uint leftmost_visible_run    = line.first_run;
      int  leftmost_visible_run_x  = 30000;
      uint rightmost_visible_run   = line.last_run;
      int  rightmost_visible_run_x = -30000;

      bool after = false;

      for (uint i = line.first_run; i <= line.last_run; ++i) {
        lf_glyph_run = _glyph_runs[i];

        if (lf_glyph_run.glyph_count == 0) continue;

        html::element *el = lf_glyph_run.get_inline_block_element(v);
        if (el) {
          if (el->floats(v) || el->positioned(v) || !el->is_visible(v))
            continue;

          rect hb = el->hit_box(v, element::TO_PARENT);
          if (hb.x().contains(zpos.x)) {
            /*  RICHTEXT:
                el->this_pos() is wrong here:
                       bookmark bm = //bookmark(el,0, );
                              zpos.x < hb.pointOf(8).x
                              ? el->this_pos(false)
                              : el->this_pos(true);
                      return bm; */

            auto css_element = el->is_css_element();
            if (css_element == tag::T__BEFORE) {
              return el->parent->start_caret_pos(v);
            }
            else if (css_element == tag::T__AFTER) {
              return el->parent->end_caret_pos(v);
            }
            else if (el->is_atomic_box() || el->is_inline_block_element(v)) {
              bookmark bm = zpos.x < hb.pointOf(8).x ? el->start_pos() : el->end_pos();
              return bm;
            } else {
              bookmark t = el->find_text_pos(v, zpos - el->pos());
              if (t.valid()) 
                return t;
            }
          }
        }

        int width =
            int(0.49f + get_cluster_range_width(
                            lf_glyph_run.glyph_start,
                            lf_glyph_run.glyph_start + lf_glyph_run.glyph_count,
                            _glyph_justified_advances.head()));

        if (width == 0) continue;

        if (lf_glyph_run.is_rtl()) {
          rc.e.x = int(lf_glyph_run.x);
          rc.s.x = rc.e.x - width;
        } else {
          rc.s.x = int(lf_glyph_run.x);
          rc.e.x = rc.s.x + width;
        }

        if (rc.x().contains(zpos.x)) goto FOUND;

        if (rc.left() < leftmost_visible_run_x) {
          leftmost_visible_run_x = rc.left();
          leftmost_visible_run   = i;
        }
        if (rc.right() > rightmost_visible_run_x) {
          rightmost_visible_run_x = rc.left();
          rightmost_visible_run   = i;
        }
        last_visible_run = i;
      }

      if (zpos.x <= leftmost_visible_run_x) {
        lf_glyph_run = _glyph_runs[leftmost_visible_run];
        after = _glyph_runs[leftmost_visible_run].is_rtl() ? true : false;
      } else if (zpos.x >= rightmost_visible_run_x) {
        lf_glyph_run = _glyph_runs[rightmost_visible_run];
        after = _glyph_runs[rightmost_visible_run].is_rtl() ? false : true;
      } else {
        assert(false);
        lf_glyph_run = _glyph_runs[last_visible_run];
        after        = true;
      }
      
      if (html::element *el = lf_glyph_run.get_inline_block_element(v)) {
        if (el->is_css_element() == tag::T__BEFORE)
          return el->parent->start_caret_pos(v);
        else if (el->is_css_element() == tag::T__AFTER)
          return el->parent->end_caret_pos(v);
      }

    FOUND:

      float w1 = 0, w2 = 0;
      float x = zpos.x - lf_glyph_run.x;

      if (lf_glyph_run.is_rtl()) {
        x += float(rc.width());
        w2 = float(rc.width());
        for (uint n = 0; n < lf_glyph_run.glyph_count; ++n) {
          glyph_index = n;
          w1          = w2;
          w2          = w1 - /*WRONG:floorf*/(_glyph_justified_advances[lf_glyph_run.glyph_start + n]);
          if (x >= w2) {
            float mw = (w1 + w2) / 2;
            after    = x < mw;
            break;
          }
        }
      } else {
        for (uint n = 0; n < lf_glyph_run.glyph_count; ++n) {
          glyph_index = n;
          w1          = w2;
          w2          = w1 + /*WRONG:ceilf*/(_glyph_justified_advances[lf_glyph_run.glyph_start + n]); // rounding breaks calculation on long runs
          if (x < w2) {
            if (x > (w1 + w2) / 2) after = true;
            break;
          }
        }
      }

      uint text_position = glyph_index_2_text_position(
          lf_glyph_run.glyph_start + glyph_index, after);
      auto rr = _runs().find(text_position);
      uint in_node_index = (*rr).text_node_start + (text_position - (*rr).text_start);
      // @@@

      bookmark found;

      if (rr->node->is_element()) {
        if (rr->node.ptr_of<element>()->is_box_element(v))
          found = after ? rr->node->end_pos() : rr->node->start_pos();
        else
          found = after ? rr->node->end_caret_pos(v) : rr->node->start_caret_pos(v);
      } else
        found = bookmark((*rr).node, in_node_index, after);

      wchar dummy;
      while(found.valid()) {
        if (found.at_caret_pos(v))
          break;
        found.advance_backward(dummy);
      }
      return found;

    }

    void text_flow::get_metrics(view &v, const element *self,
                                const bookmark &bm, caret_metrics &m) {
      bool     rtl;
      bookmark k        = bm;
      bool     on_lf    = false;
      bool     after_it = bm.after_it;
      k.after_it        = false;

      uint text_position = node_position_2_text_position(self, k, rtl, true);

      if (bm.char_code(v) == '\n' && after_it) {
        after_it = false;
        on_lf    = true;
        ++text_position;
      }

      cluster_position_t cluster;
      set_cluster_position(cluster, text_position);

      get_glyph_metrics(get_cluster_glyph_start(cluster), m);

      // float round_delta = 0;

      m.at_end = after_it;

      if (!on_lf) {
        // m.glyph_rc = m.caret_rc;
        m.glyph_rc.s.y = int(m.y1);
        m.glyph_rc.e.y = int(m.y2);
        m.glyph_rc.s.x = int(m.x1);
        m.glyph_rc.e.x = int(m.x2);
        if (m.glyph_rc.s.x > m.glyph_rc.e.x) {
          swap(m.glyph_rc.s.x, m.glyph_rc.e.x);
        }
      }
      m.ctype = CHAR_POSITION;
    }

    uint text_flow::glyph_index_2_text_position(uint glyph_index,
                                                bool last) const throw() {
      if (_runs.size() == 0) return 0;
      uint               text_length = (uint)_text.length();
      cluster_position_t cluster;
      set_cluster_position(cluster, 0);
      while (cluster.text_position < text_length) {
        if (glyph_index-- == 0) {
          if (last) {
            advance_cluster_position(cluster);
            return cluster.text_position - 1;
          }
          return cluster.text_position;
        }
        advance_cluster_position(cluster);
      }
      return last ? text_length - 1 : 0;
    }

    bool text_flow::next_text_position(uint& text_position) const {
      if (text_position >= _text.length())
        return false;
#pragma TODO("use _glyph_clusters here")
      uint gi = text_position_2_glyph_index(text_position);
      if (gi >= uint(_glyph_indices.last_index()))
        text_position = (uint)_text.length();
      else
        text_position = glyph_index_2_text_position(gi + 1, false);

      return true;
    }

    bool text_flow::prev_text_position(uint& text_position) const {
      if (text_position <= 0)
        return false;
#pragma TODO("use _glyph_clusters here")
      uint gi = text_position_2_glyph_index(uint(text_position));
      if (gi == 0)
        text_position = 0u;
      else
        text_position = glyph_index_2_text_position(gi - 1,false);
      return true;
    }


    uint text_flow::text_position_2_glyph_index(uint text_position) const
        throw() {
      if (_runs.size() == 0) return 0;
      cluster_position_t cluster;
      set_cluster_position(cluster, text_position);
      return get_cluster_glyph_start(cluster);
    }

    bool text_flow::is_caret_pos_at(const bookmark &bm) const {
      for (uint n = 0; n < _runs.length(); ++n) {
        const text_run &run = _runs[n];
        if (bm.node == run.node 
          && bm.pos >= int(run.text_node_start) 
          && bm.pos < int(run.text_node_start + run.text_length)) {
#ifdef _DEBUG
          bool rr = run.node->get_element()->flags.is_synthetic;
          if (rr) { rr = rr; }
#endif
          return true;
        }
      }
      if (_runs.length() == 0) return bm.node->is_text() && bm.at_start();

      return false;
    }

    uint_v text_flow::node_position_2_text_position(const element * self, const bookmark &bm, bool &rtl, bool caret_pos) const throw() {
      element *t            = 0;
      uint     _runs_length = (uint)_runs.length();
      if ((t = bm.node->nearest_box()) != self) {
        while (t) {
          if (t->get_owner() == self) break;
          t = t->get_owner();
        }
        assert(t);
        if (t)
          for (uint n = 0; n < _runs_length; ++n) {
            const text_run &run = _runs[n];
            if (t == run.node) return run.text_start + int(bm.after_it);
          }
        // assert(false);
      }

      int    distance           = 0xffff;
      uint_v best_text_position;

      for (uint n = 0; n < _runs_length; ++n) {
        const text_run &run = _runs[n];
        if (bm.node == run.node && bm.pos >= int(run.text_node_start) &&
            bm.pos < int(run.text_node_start + run.text_length)) {
          uint text_index =
              run.text_start + int(bm.pos) - int(run.text_node_start);
          if (caret_pos) text_index += int(bm.after_it);
          cluster_position_t cluster;
          set_cluster_position(cluster, text_index);
          rtl = run.is_rtl();
          return cluster.text_position;
        } else if (bm.node == run.node) {
          rtl   = run.is_rtl();
          int d = abs(int(run.text_node_start) - bm.pos);
          if (d < distance) {
            distance           = d;
            best_text_position = run.text_start;
          }
          d = abs(int(run.text_node_start + run.text_length) - bm.pos);
          if (d < distance) {
            distance = d;
            best_text_position = run.text_start + run.text_length;
          }
        }
      }

      return best_text_position;
      // assert(false);
      // return (uint)_text.length();
    }

    bookmark text_flow::text_position_2_node_position(uint text_position) const throw() {
      bool after = false;
      int  li    = _text.last_index();
      if (li < 0) { return bookmark(); }
      if (int(text_position) > li) {
        text_position = li;
        after         = true;
      }
      if (int(text_position) == li && _text[text_position] == '\n') {
        if (text_position > 0) {
          --text_position;
          after = true;
        } else {
          after = false;
        }
      }

      auto rr = _runs().find(text_position);
      if (!!rr) {
        uint     delta = text_position - (*rr).text_start;
        bookmark bm;
        bm.node     = (*rr).node;
        bm.pos      = (*rr).text_node_start + delta;
        bm.after_it = after;
        return bm;
      }
      assert(false);
      return bookmark();
    }

    uint text_flow::land_pos(element *self, bookmark bm) const throw() {
      element *t = bm.node->nearest_box();
      while (t) {
        if (t->get_owner() == self) break;
        t = t->get_owner();
      }
      assert(t);
      if (t) {
        bookmark tbm = bm.after_it ? t->end_pos() : t->start_pos();
        bool     rtl;
        uint     tp = node_position_2_text_position(self, tbm, rtl);
        return tp;
      }
      assert(false);
      return 0;
    }

    bool text_flow::get_sel_glyph_positions(element *self, bookmark caret,
                                            bookmark anchor, uint &glyph_start,
                                            uint &glyph_end) const throw() {
      uint start = 0;
      uint end   = 0;
      if (!caret.valid() || !anchor.valid()) return false;

      if (anchor > caret) swap(anchor, caret);

      bool caret_inside  = caret.node->owned_by(self, true);
      bool anchor_inside = anchor.node->owned_by(self, true);

      bool caret_rtl  = false;
      bool anchor_rtl = false;

      if ((caret == anchor) && caret_inside && anchor_inside) {
        bool r;
        end = start = node_position_2_text_position(self, anchor, r);
        goto FOUND;
      }

      if (caret_inside && anchor_inside) // simple case - selection starts and ends here
      {
        start = node_position_2_text_position(self, anchor, anchor_rtl);
        end   = node_position_2_text_position(self, caret, caret_rtl);
        if (!end) end = this->_text.size();
        goto FOUND;

      }                      // more sophisticated case
      else if (caret_inside) // anchor before
      {
        start = 0;
        end   = node_position_2_text_position(self, caret, caret_rtl);
        goto FOUND;
      } else if (anchor_inside) // caret after
      {
        start = node_position_2_text_position(self, anchor, anchor_rtl);
        end   = uint(_text.length());
        goto FOUND;
      } else if (anchor <= self->this_pos(false) &&
                 caret >= self->this_pos(true)) // selection covers this in full
      {
        start = 0;
        end   = uint(_text.length());
        goto FOUND;
      }
      return false;
    FOUND:
      glyph_start = text_position_2_glyph_index(start);
      glyph_end   = text_position_2_glyph_index(end);
      return true;
    }

    inline bool is_word_char(uint c) { return ucisalnum(c) || c == '_'; }

    inline void append(array<wchar> &out, wchars s) {
      while (!!s) {
        wchar c = s++;
        if (c) out.push(c);
      }
    }

    bool text_flow::advance_caret_pos_first(view &v, const element *self,
                                            uint line_no, bookmark &bm) {
      if (line_no >= _lines.length()) return false;
      array<index_direction> text_positions;
      if (!text_positions_in_visual_order(line_no, text_positions))
        return false;
      if (self->is_empty()) {
        bm = text_position_2_node_position(0);
        bm.after_it = false;
        return true;
      }

      auto tp     = text_positions.first();
      bm          = text_position_2_node_position(tp.index);
      bm.after_it = tp.rtl ? true : false;
      return true;
    }
    bool text_flow::advance_caret_pos_last(view &v, const element *self,
                                           uint line_no, bookmark &bm) {
      if (line_no >= _lines.length()) return false;
      array<index_direction> text_positions;
      if (!text_positions_in_visual_order(line_no, text_positions))
        return false;
      auto tp = text_positions.last();
      if (self->is_empty()) {
        bm = text_position_2_node_position(0);
        bm.after_it = false;
        return true;
      } 
      else if (_text[tp.index] == '\n' || _text[tp.index] == THSP) {
        bm = text_position_2_node_position(tp.index-1);
        bm.after_it = tp.rtl ? false : true;
        return true;
      }
      bm          = text_position_2_node_position(tp.index);
      bm.after_it = tp.rtl ? false : true;
      wchar dummy = 0;
      while (bm.valid()) {
        if (bm.at_caret_pos(v)) break;
        bm.advance_backward(dummy);
      }

      return true;
    }

    bool text_flow::advance_caret_pos_left(view &v, const element *self,
                                           bookmark &bm) {
      bool rtl;
      uint_v  text_position_v = node_position_2_text_position(self, bm, rtl, true);
      if (text_position_v.is_undefined() && lines_count())
        return advance_caret_pos_last(v, self, lines_count() - 1, bm);

#if 1
      uint text_position = text_position_v;
      if (rtl) {
        if (next_text_position(text_position)) {
          bm = text_position_2_node_position(text_position);
          if (!bm.after_it) {
            bm.pos = bm.pos - 1;
            bm.after_it = true;
          }
          return true;
        }
      }
      else {
        if (prev_text_position(text_position)) {
          bm = text_position_2_node_position(text_position);
          return true;
        }
      }
      return false;
#else

      int text_position = (int)text_position_v;

      if (rtl) {
        if (self->is_disabled()) {
          bm = self->start_pos();
          return true;
        }

        if (!bm.after_it && _text[text_position] != '\n') {
          bm.after_it = true;
          return true;
        }
      } else {
        if (self->is_disabled()) {
          bm = self->end_pos();
          return true;
        }

        /*if (bm.after_it && bm.at_char_pos(v)) {
          bm.after_it = false;
          if (u16::is_suro_tail(bm.char_code(v)) && bm.pos)
            bm.pos = bm.pos - 1;
          return true;
        }*/
      }

      array<index_direction> text_positions;
      uint                   line_no = 0;
      if (!get_line_no(text_position, line_no)) return false;
      if (!text_positions_in_visual_order(line_no, text_positions))
        return false;

      int  idx = -1;
      bool rightmost = false;
      for (; text_position >= 0; --text_position) {
        idx = text_positions.get_index(text_position);
        if (idx >= 0)
          break;
        rightmost = true;
      }
      if (idx < 0) 
        return false;

      auto is_first_cr_pos = [&](int idx) {
        return _text[text_positions[idx].index] == '\n';
      };

      if (!rtl && !rightmost) --idx;

      if (idx < 0 /*|| is_first_cr_pos(idx)*/) {
        if (const_cast<element *>(self)->get_style(v)->direction ==
            direction_rtl)
          return advance_caret_pos_last(v, self, ++line_no, bm);
        else
          return advance_caret_pos_last(v, self, --line_no, bm);
      } else {
        text_position = text_positions[idx].index;
        bool after    = text_positions[idx].rtl ? true : false;
        //if (rightmost)
        //  after = !after;
#if 0
        if (text_position > 0 && u16::is_suro_tail(_text[text_position]))
          --text_position;
#else
        if(prev_text_position(text_position)) {
            if (!after && text_position)
              ++text_position;
        }
#endif
        bm          = text_position_2_node_position(text_position);
        bm.after_it = after;
      }
      return true;
#endif
    }

    bool text_flow::advance_caret_pos_right(view &v, const element *self,
                                            bookmark &bm) {
      bool     rtl;
      bookmark pbm       = bm;
      uint_v text_position_v = node_position_2_text_position(self, bm, rtl, true);

      //uint   glyph_position = text_position_2_glyph_index(text_position);

      if (text_position_v.is_undefined())
        return advance_caret_pos_first(v, self, 0, bm);

#if 1
      uint text_position = text_position_v;

      if (rtl) {
        if (prev_text_position(text_position)) {
          bm = text_position_2_node_position(text_position);
          return true;
        }
      }
      else {
        if (next_text_position(text_position)) {
          bm = text_position_2_node_position(text_position);
          if (!bm.after_it) {
            bm.pos = bm.pos - 1;
            bm.after_it = true;
          }
          return true;
        }
      }
      return false;
#else
      if (rtl) {
        if (self->is_disabled()) {
          bm = self->start_pos();
          return true;
        }

        if (bm.after_it) {
          bm.after_it = false;
          return true;
        }
      } else {
        if (self->is_disabled()) {
          bm = self->end_pos();
          return true;
        }
      }

      array<index_direction> text_positions;
      uint                   line_no = 0;
      if (!get_line_no(text_position, line_no)) return false;
      if (!text_positions_in_visual_order(line_no, text_positions))
        return false;
      int idx = text_positions.get_index(text_position);
      if (idx < 0) return false;

      auto is_last_cr_pos = [&](int idx) {
        return _text[text_positions[idx].index] == '\n';
      };

      if (rtl) { ++idx; }

      if (idx >= text_positions.size() /* || is_last_cr_pos(idx)*/) {
        if (const_cast<element *>(self)->get_style(v)->direction ==
            direction_rtl)
          return advance_caret_pos_first(v, self, --line_no, bm);
        else
          return advance_caret_pos_first(v, self, ++line_no, bm);
      } else {
        text_position = text_positions[idx].index;
#if 0        
        bool after    = text_positions[idx].rtl ? false : true;
        if (text_position == 0)
          after = !after;
        if (text_position < _text.length() &&
            u16::is_suro_head(_text[text_position]))
          text_position = text_position + 1;
#else
        bool after = text_positions[idx].rtl ? false : true;
        if (next_text_position(text_position._v)) {
          if (after && text_position._v)
            --text_position._v;
        }

#endif
        bm          = text_position_2_node_position(text_position);
        bm.after_it = after;
      }
      return bm != pbm;
#endif
    }

    bool text_flow::get_line_no(uint text_position, uint &line_no) {
      for (int i = _lines.last_index(); i >= 0; --i) {
        const layout_line &ln = _lines[i];
        if (text_position >= ln.start_text_index) {
          line_no = i;
          return true;
        }
      }
      if (text_position == 0) {
        line_no = 0;
        return true;
      }
      if (text_position == _text.length()) {
        line_no = _lines.last_index();
        return true;
      }
      assert(false);
      return false;
    };

    uint text_flow::last_caret_pos() {
      int pos = _text.last_index();
      if (pos > 0 && _text[pos - 1] == '\n') --pos;
      return pos < 0 ? 0u : uint(pos);
    }

    bool text_flow::advance_caret_pos(view &v, const element *self,
                                      bookmark &bm, ADVANCE_DIR dir,
                                      array<wchar> *out) {
      bool rtl;
      int  text_position;
      // bool after;
      bookmark initial = bm;

/*      auto start_pos = [&]() {
        if (self->flags.is_synthetic) {
          wchar c = 0;
          bm.advance_backward(c);
        }
        else
          bm = self->start_pos();
        return bm;
      };

      auto end_pos = [&]() {
        if (self->flags.is_synthetic) {
          wchar c = 0;
          bm.advance_forward(c);
        }
        else
          bm = self->end_pos();
        return bm;
      };*/

      switch (dir) {
      case ADVANCE_FIRST: {
        bm = text_position_2_node_position(0);
        return bm.valid();
      }
      case ADVANCE_LAST: {
        uint n = last_caret_pos();
        bm     = text_position_2_node_position(n);
        if (bm.valid()) {
          if (_text[n] != '\n') bm.after_it = true;
        }
        return bm.valid();
      }
      case ADVANCE_LEFT:
        if (!advance_caret_pos_left(v, self, bm)) {
            bm = self->start_pos();
        }
        return bm.valid();

      case ADVANCE_RIGHT:
        if (!advance_caret_pos_right(v, self, bm))
          bm = self->end_pos();
        return bm.valid();

      case ADVANCE_NEXT: {
        text_position          = node_position_2_text_position(self, bm, rtl);
        int text_position_next = text_position;
        if (text_position >= (int)last_caret_pos() ||
            !u16::advance(_text(), +1, text_position_next)) {
          if (bm.after_it)
            bm = self->end_pos();
          else
            bm = self->end_caret_pos(v);
          return true;
        }
        if (out) append(*out, _text(text_position, text_position_next));
        if (!bm.after_it) {
          bm          = text_position_2_node_position(text_position_next - 1);
          bm.after_it = true;
        } else
          bm = text_position_2_node_position(text_position_next);
      } break;

      case ADVANCE_PREV:
        text_position = node_position_2_text_position(self, bm, rtl);
        if (text_position >= (int)last_caret_pos() || text_position < 0 ||
            !u16::advance(_text(), -1, text_position)) {
          bm = self->start_pos();
          return true;
        }
        bm          = text_position_2_node_position(text_position);
        bm.after_it = rtl;
        if (initial == bm) {
          bm = self->start_pos();
          return true;
        }
        break;

      case ADVANCE_HOME: {
        text_position = node_position_2_text_position(self, bm, rtl, false);
        uint line_no;
        if (!get_line_no(text_position, line_no)) return false;
        if (const_cast<element *>(self)->get_style(v)->direction ==
            direction_rtl)
          return advance_caret_pos_last(v, self, line_no, bm);
        else
          return advance_caret_pos_first(v, self, line_no, bm);
      } break;
      case ADVANCE_END: {
        text_position = node_position_2_text_position(self, bm, rtl, false);
        uint line_no;
        if (!get_line_no(text_position, line_no)) return false;
        if (const_cast<element *>(self)->get_style(v)->direction ==
            direction_rtl)
          return advance_caret_pos_first(v, self, line_no, bm);
        else
          return advance_caret_pos_last(v, self, line_no, bm);
      } break;

      default: assert(false); break;
      }

      return bm.valid();
    }

    void text_flow::calc_min_max(view &v, element *elem, int &out_min_width,
                                 int &out_max_width) {
      hstyle st = elem->get_style(v);

      if (_runs.length() == 0) {
        out_min_width = 0;
        out_max_width = 0;
        return;
      }

      cluster_position_t cluster;
      set_cluster_position(cluster, 0);
      cluster_position_t next_cluster(cluster);
      cluster_position_t line_start_cluster(cluster);

      bool plain_text =
          !elem->get_style(v)->can_wrap() /* || elem->is_empty() ? */;

      // plain_text

      // bool  wrap_on_word_boundaries =
      // !elem->get_style(v)->wrap_unrestricted();  uint  max_chars_in_word =
      // wrap_on_word_boundaries? 20000:1;

      tool::inc_max<float> min_width(1.0f); // empty text block has always 1px
                                            // min width to accomodate caret
                                            // position
      tool::inc_max<float> max_width(1.0f);
      float                current_width = 0; // width until <br>
      bool                 hard_lf;

      float    brick_width = 0, brick_ws_width = 0;
      element *inline_box_element;
      uint text_end = uint(_text.length());
      //uint ntabs                               = 0;
#if 1

      if (plain_text) 
      {
        for (; cluster.text_position < text_end; cluster = next_cluster) {
          if (!advance_cluster_position_next_brick(
                  v, elem, line_start_cluster, cluster, next_cluster, inline_box_element,
                  brick_width, brick_ws_width, hard_lf,
                  10000.0f /*arbitrary large value*/))
            break;

          //if (/*_breakpoints[cluster.text_position].is_whitespace && */
          //    _text[cluster.text_position] == '\t')
          //  ++ntabs;

          current_width += brick_width + brick_ws_width;
          if (hard_lf) {
            //if (ntabs)
            //  current_width += produce_tab_advances(v, elem, line_start_cluster,
            //                                        next_cluster);
            max_width <<= current_width;
            line_start_cluster = next_cluster;
            //ntabs              = 0;
            brick_ws_width     = 0;
            current_width      = 0;
          }
          min_width <<= brick_width;
          //if (inline_box_element &&
          //    inline_box_element->get_style(v)->has_percentage_widths())
          //  elem->ldata->content_h_percent_dependent = true;
        }

        max_width <<= current_width;
        out_min_width = out_max_width = int(ceilf(max_width));

      } else { // wrappable text

        for (; cluster.text_position < text_end; cluster = next_cluster) {
          if (!advance_cluster_position_next_brick(
                  v, elem, line_start_cluster, cluster, next_cluster, inline_box_element,
                  brick_width, brick_ws_width, hard_lf,
                  10000.0f /*arbitrary large value*/))
            break;

          current_width += brick_width + brick_ws_width;
          if (hard_lf) {
            max_width <<= current_width - brick_ws_width;
            brick_ws_width = 0;
            current_width  = 0;
          }
          min_width <<= brick_width;
          //if (inline_box_element &&
          //    inline_box_element->get_style(v)->has_percentage_widths())
          //  elem->ldata->content_h_percent_dependent = true;
        }

        max_width <<= current_width - brick_ws_width;

        out_min_width = elem->get_style(v)->wrap_unrestricted()
                            ? 10
                            : int(ceilf(min_width));
        out_max_width = int(ceilf(max_width));
      }

#else
      for (; cluster.text_position < text_end; cluster = next_cluster) {
        if (!advance_cluster_position_next_brick(
                v, elem, cluster, next_cluster, inline_box_element, brick_width,
                brick_ws_width, hard_lf, 10000.0f /*arbitrary large value*/))
          break;

        if (plain_text && _breakpoints[cluster.text_position].is_whitespace &&
            _text[cluster.text_position] == '\t')
          ++ntabs;

        current_width += brick_width + brick_ws_width;
        if (hard_lf) {
          if (plain_text) {
            if (ntabs)
              current_width += this->produce_tab_advances(
                  v, elem, line_start_cluster, next_cluster);
            max_width <<= current_width;
            line_start_cluster = next_cluster;
            ntabs              = 0;
          } else
            max_width <<= current_width - brick_ws_width;
          brick_ws_width = 0;
          current_width  = 0;
        }
        min_width <<= brick_width;
        if (inline_box_element &&
            inline_box_element->get_style(v)->has_percentage_widths())
          _has_percentage_widths = true;
      }

      if (plain_text)
        max_width <<= current_width;
      else
        max_width <<= current_width - brick_ws_width;

      if (!plain_text) {
        out_min_width = elem->get_style(v)->wrap_unrestricted()
                            ? 10
                            : int(ceilf(min_width));
        out_max_width = int(ceilf(max_width));
      } else {
        out_min_width = out_max_width = int(ceilf(max_width));
      }
#endif
    }

    /*struct  index_direction {
          uint index:31;
          uint rtl:1;
        };*/

    bool text_flow::text_positions_in_visual_order(
        uint line_no, array<index_direction> &text_positions) {
      if (line_no >= _lines.length()) return false;

      auto line = _lines[line_no];

      cluster_position_t cluster_start;
      cluster_position_t cluster_end;
      set_cluster_position(cluster_start, line.start_text_index);
      set_cluster_position(cluster_end, line.end_text_index);

      uint runIndexEnd = cluster_end.runIndex;
      if (cluster_end.text_position > _runs[runIndexEnd].text_start)
        ++runIndexEnd; // Only partially cover the run, so round up.

      // Fill in mapping from visual run to logical sequential run.
      // uint32 bidiOrdering[100];
      // uint32 total_runs = min(runIndexEnd - cluster_start.runIndex,
      // static_cast<uint32>(items_in(bidiOrdering)));
      uint32              total_runs = runIndexEnd - cluster_start.runIndex;
      buffer<uint32, 100> bidiOrdering(total_runs);

      produce_bidi_ordering(cluster_start.runIndex, total_runs,
                            bidiOrdering.begin());

      for (int i = 0; i < int(total_runs); ++i) {
        const text_run &run = _runs[bidiOrdering[index_t(i)]];
        if (run.is_rtl()) {
          uint t = run.text_start + run.text_length - 1;
          for (;;) {
            index_direction id;
            id.index = t;
            id.rtl   = 1;
            text_positions.push(id);
            if (t-- == run.text_start) break;
          }
        } else {
          uint m =
              min(run.text_start + run.text_length, cluster_end.text_position);
          for (uint t = max(run.text_start, cluster_start.text_position); t < m;
               ++t) {
            index_direction id;
            id.index = t;
            id.rtl   = 0;
            text_positions.push(id);
          }
        }
      }
      return true;
    }

    void text_flow::get_text(view &v, array<wchar> &to) {
      for (index_t i = 0; i < _runs.size(); ++i) {
        const text_run &run = _runs[i];
        if (run.is_inline_block_pos(_text())) {
          element *el = run.get_element();
          el->get_ui_text(v, to);
        } else
          to.push(run.chars(_text.head()));
      }
    }

    const glyph_run *find_glyph_run(slice<glyph_run> gruns, uint32  glyph_index);

    bool text_flow::setup_line(view &v, element *elem, bool collapse_ws, int y,
                               int x1, int x2, float line_width, slice<inline_el_pos> flex_elements,
                               const cluster_position_t &cluster_start,
                               const cluster_position_t &cluster_end,
                               int &                     line_height) throw() {

#ifdef _DEBUG
      if (elem->is_id_test())
        elem = elem;
      //if (this->_text[0] == ZWSP)
      //  elem = elem;
#endif 

      ////////////////////////////////////////
      // Figure out how many runs we cross, because this is the number
      // of distinct glyph runs we'll need to reorder visually.

      uint runIndexEnd = cluster_end.runIndex;
      if (cluster_end.text_position > _runs[runIndexEnd].text_start)
        ++runIndexEnd; // Only partially cover the run, so round up.

      // Fill in mapping from visual run to logical sequential run.

      uint32              total_runs = runIndexEnd - cluster_start.runIndex;
      buffer<uint32, 100> bidiOrdering(total_runs);

      produce_bidi_ordering(cluster_start.runIndex, total_runs,
                            bidiOrdering.begin());

      // Look backward from end until we find non-space.
      uint32 trailing_ws_position = cluster_end.text_position;
      if (collapse_ws/* && !elem->state.content_editable()*/ && _text.last() != EMPTY_STRING_PLACEHOLDER)
        for (; trailing_ws_position > cluster_start.text_position;
             --trailing_ws_position) {
          if (!is_breakable_space(trailing_ws_position - 1))
            break; // Encountered last significant character.
        }
      // Set the glyph run's ending cluster to the last whitespace.
      cluster_position_t cluster_ws_end(cluster_start);

      set_cluster_position(cluster_ws_end, trailing_ws_position);

      ////////////////////////////////////////
      // Determine starting point for alignment.

      float x = float(x1);
      // float y = rect.bottom;

      html::style *elem_style = elem->get_style(v);
      html::style *style      = nullptr;
      gool::font * font       = nullptr;
      gool::font * used_font  = nullptr;

      if(flex_elements.length)
        flex_children(v, elem, x1, x2, cluster_start,cluster_ws_end, flex_elements,line_width);

      switch (elem_style->get_text_align()) {
      default:
        //if (!collapse_ws)
        //  produce_tab_advances(v, elem, cluster_start, cluster_ws_end);
        break;
      case align_right:
      DO_RIGHT_ALIGN:
        // For RTL, we neeed the run width to adjust the s
        // so it starts on the right side.
        x = x2 - line_width;
        //if (!collapse_ws)
        //  produce_tab_advances(v, elem, cluster_start, cluster_ws_end);
        break;
      case align_center:
        x = x1 + (x2 - x1 + 1 - line_width) / 2;
        //if (!collapse_ws)
        //  produce_tab_advances(v, elem, cluster_start, cluster_ws_end);
        break;
      case align_justify:
        if ((cluster_start.text_position == 0 &&
             cluster_end.text_position == _text.length()) ||
            (cluster_end.text_position == _text.length())) {
          // that is either first and the only line,
          // or it is last line among others
          // don't do justification in this case.
          if (elem_style->direction == direction_rtl)
            goto DO_RIGHT_ALIGN;
          else
            break;
        }
        produce_justified_advances(v, elem, y, x1, x2, cluster_start,
                                   cluster_ws_end);
        break;
      }

      ////////////////////////////////////////
      // Send each glyph run to the sink.

      tool::inc_max<int>  ascent;
      tool::inc_max<int>  descent;
      tool::inc_max<int>  height;
      tool::inc_max<bool> middles;
      tool::inc_max<bool> tops;
      tool::inc_max<bool> bottoms;

      int    n_inline_blocks = 0;
      int    text_ascent     = 0;
      int    text_descent    = 0;
      uint_v first_run;
      uint_v last_run;

      byte valign = valign_baseline;

      for (size_t i = 0; i < total_runs; ++i) {
        const text_run &run = _runs[bidiOrdering[index_t(i)]];

        uint32 glyph_start = run.glyph_start;
        uint32 glyph_end   = glyph_start + run.glyph_count;

        float run_width;

        bool         before    = false;
        html::style *run_style = nullptr;
        bool         after     = false;

        element *el = run.get_element();

        if (!el) break;

        if (run.is_inline_block_pos(_text())) {
          if (el->oof_positioned(v)) {
            el->set_pos(point(int(x), y));
            continue;
          }
#if DEBUG
          if (el->is_id_test())
            el = el;
#endif
          style          = el->get_style(v);
          font           = v.get_font(style);
          used_font      = run.get_used_font(v);
          float_e fs     = el->floats(v);
          if (fs) continue;

          int a = 0;
          int d = 0;
          int h = 0;
          el->get_inline_block_metrics(v, a, d, h);

          valign_e va = el->get_style(v)->get_inline_vertical_align();

          switch (va) {
            case valign_text_top:
            case valign_top:
              a = 0; d = 0;
              height <<= h;
              tops <<= true;
              break;

            case valign_middle:
              a = 0; d = 0;
              height <<= h;
              middles <<= true;
              break;

            case valign_bottom:
            case valign_text_bottom:
              a = 0; d = 0;
              height <<= h;
              bottoms <<= true;
              break;

            case valign_auto:
            default: 
            {
              a = 0; d = 0;
              height <<= h;
            } break;

            case valign_baseline:
            {
              ascent <<= a;
              descent <<= d;
            } break;

            case valign_sub:
            case valign_super: {
              int shift = el->get_baseline_shift(v, elem);
              ascent <<= a - shift;
              descent <<= d + shift;
            } break;
          }

          ++n_inline_blocks;
          el->set_margin_pos(point(int(x), 0 /*will be set downstream*/));
          // measure_borders_x(v, el, style, elem->dim() );
          // rect ot = el->outer_distance(v);
          // run_width = el->computed_width(v,elem->dim().x) + ot.left() +
          // ot.right();
          run_width = float(el->margin_box(v).width());
          _glyph_justified_advances[glyph_start] = run_width;
          goto ACCOUNT_RUN;
        }

        before    = run.node != 0;
        run_style = run.get_style(v);
        after     = run.node != 0;
        assert(before == after);
        before = before;
        after  = after;

        valign = valign_baseline;

        if (style != run_style || used_font != run.get_used_font(v) || !font) {
          style     = run_style;
          used_font = run.get_used_font(v);
          // do_glyphs_fallback = run.do_glyphs_fallback;
          font         = v.get_font(style);
          text_ascent  = font->ascent;
          text_descent = font->descent;
          // CSS: line-height handling
          if (style->line_height.is_defined()) {
            int lineheight = pixels(v, run.node->get_element(), style->line_height, text_ascent + text_descent).height();
            text_ascent += (lineheight - (text_ascent + text_descent)) / 2;
            text_descent = lineheight - text_ascent;

            // text_descent += (lineheight - (text_ascent + text_descent)) / 2;
            // text_ascent = lineheight - text_descent;

            //float ratio = text_ascent / float(text_descent + text_ascent);
            //text_ascent = int(lineheight * ratio + 0.5f);
            //text_descent = lineheight - text_ascent;
          }
          valign = run.get_element() == elem ? byte(valign_baseline) : byte(style->get_text_vertical_align());
        }

        // If the run is only partially covered, we'll need to find
        // the subsection of glyphs that were fit.
        if (cluster_start.text_position > run.text_start) {
          glyph_start = get_cluster_glyph_start(cluster_start);
        }
        if (cluster_ws_end.text_position < run.text_start + run.text_length) {
          glyph_end = max(glyph_start, get_cluster_glyph_start(cluster_ws_end));
        }

        if (glyph_start >= glyph_end &&
            run.chars(_text.head()) !=
                WCHARS("\r")) //- check test-cases/text-flow/br.htm
        {
          // The itemizer told us not to draw this character,
          // either because it was a formatting, control, or other hidden
          // character.
          continue;
        }

        switch (valign) {
        default:
        case valign_sub:
        case valign_super: {

          if (el != elem) {
            int shift = el->get_baseline_shift(v, elem);
            ascent <<= text_ascent - shift;
            descent <<= text_descent + shift;
          } else {
            ascent <<= text_ascent;
            descent <<= text_descent;
          }
        } break;
        case valign_text_top:
        case valign_top:
          height <<= text_ascent + text_descent;
          tops <<= true;
          break;

        case valign_middle:
          height <<= text_ascent + text_descent;
          middles <<= true;
          break;

        case valign_bottom:
        case valign_text_bottom:
          height <<= text_ascent + text_descent;
          bottoms <<= true;
          break;
        }

        /*if ((glyph_start >= glyph_end) - check test-cases/text-flow/br.htm
         || (run.script.shapes & DWRITE_SCRIPT_SHAPES_NO_VISUAL))
        {
            // The itemizer told us not to draw this character,
            // either because it was a formatting, control, or other hidden
        character. continue;
        }*/

        // The run width is needed to know how far to move forward,
        // and to flip the s for right-to-left text.

        // if(run.no_visual())
        //  run_width = 0;
        // else
        run_width =
            get_cluster_range_width(glyph_start /* - justificationglyph_start*/,
                                    glyph_end /*   - justificationglyph_start*/,
                                    _glyph_justified_advances.head());
      ACCOUNT_RUN:

        // Flush this glyph run.
        uint gr_idx = set_glyph_run(
            run.is_rtl() ? (x + run_width)
                         : (x), // s starts from right if RTL
            valign, glyph_start, glyph_end - glyph_start, run.node,
            // used_font_face(font,run),
            run.pstyle, run.get_used_font(v),
            // font->size,
            run.bidi_level(), run.is_sideways(), lines_count(), run.marks);

        x += run_width;

        if (first_run.is_undefined()) first_run = gr_idx;
        last_run = gr_idx;
      }

      if (height > ascent + descent) {
        if (middles || (tops && bottoms)) {
          ascent.acc += (height - (ascent + descent)) / 2;
          descent.acc = height - ascent;
        } else if (tops)
          descent.acc = height - ascent;
        else if (bottoms)
          ascent.acc = height - descent.acc;
        else {
//          assert(false);
          ascent.acc += (height - (ascent + descent)) / 2;
          descent.acc = height - ascent;
        }
      }

      layout_line ll;
      ll.first_run        = first_run;
      ll.last_run         = last_run;
      ll.start_text_index = cluster_start.text_position;
      ll.end_text_index   = cluster_end.text_position;
      ll.y                = y;
      ll.baseline         = ascent;
      ll.page_no          = 0;
      ll.column_no        = 0;
      // ll.cluster_start = cluster_start;
      // ll.cluster_end = cluster_end;
      line_height = ll.height = ascent + descent;
      set_line(ll);

      // adjust positions of inline block elements
      // if( n_inline_blocks  || tops || bottoms || middles )
      for (size_t i = 0; i < total_runs && n_inline_blocks > 0; ++i) {
        const text_run &run = _runs[bidiOrdering[index_t(i)]];
        if (!run.is_inline_block_pos(_text())) 
          continue;

        element *el = run.get_element();
        if (!el) break;

        if (el->floats(v)) continue;

        el->set_height(v, el->computed_height(v, elem->dim().y));

        --n_inline_blocks;
        auto valign = el->get_style(v)->get_text_vertical_align();
        switch (valign) {
        case valign_auto:
        case valign_baseline:
        case valign_sub:
        case valign_super: {
          int a = 0;
          int d = 0;
          int h = 0;
          el->get_inline_block_metrics(v, a, d, h);
          int shift = el->get_baseline_shift(v, elem);
          if (h > a + d)
            el->set_y_pos(ll.y + ll.baseline - h + el->outer_distance(v).top() + shift);
          else
            el->set_y_pos(ll.y + ll.baseline - a + el->outer_distance(v).top() + shift);
        } break;
        case valign_text_top:
        case valign_top:
          el->set_y_pos(ll.y + el->outer_distance(v).top());
          break;
        case valign_middle: {
          int margin_box_height = el->margin_box(v).height();
          int  ty = (ll.height - margin_box_height) / 2;
          el->set_y_pos(ll.y + ty + el->outer_distance(v).top());
        } break;

        case valign_text_middle:
        {
          int margin_box_height = el->margin_box(v).height();
          if (margin_box_height < ll.height) {
            // Align the vertical midpoint of the box with the
            // baseline of the parent box plus half the
            // x-height of the parent.
            gool::font *ppf = v.get_font(el->parent->get_style(v));
            int         xh = ppf->x_height;
            int         c = ll.baseline - xh / 2;
            int         eh = margin_box_height;
            int         ty = c - eh / 2;
            if (ty + eh > ll.height) ty = ll.height - eh;
            el->set_y_pos(ll.y + ty + el->outer_distance(v).top());
          }
          else
          {
            int  ty = (ll.height - margin_box_height) / 2;
            el->set_y_pos(ll.y + ty + el->outer_distance(v).top());
          }
        } break;

        case valign_text_bottom:
#pragma TODO("valign:text-bottom")
        case valign_bottom:
          el->set_y_pos(ll.y + ll.height - el->dim().y - el->outer_distance(v).bottom());
          break;
        }
      }

      return true;
    }

    void text_flow::produce_bidi_ordering(uint32 spanStart, uint32 spanCount,
                                          OUT uint32 *spanIndices // [spanCount]
                                          ) const throw() {
      // Produces an index mapping from sequential order to visual bidi order.
      // The function progresses forward, checking the bidi level of each
      // pair of spans, reversing when needed.
      //
      // See the Unicode technical report 9 for an explanation.
      // http://www.unicode.org/reports/tr9/tr9-17.html

      // Fill all entries with initial indices
      for (uint32 i = 0; i < spanCount; ++i) {
        spanIndices[i] = spanStart + i;
      }

      if (spanCount <= 1) return;

      size_t runStart     = 0;
      uint32 currentLevel = _runs[spanStart].bidi_level();

      // Rearrange each run to produced reordered spans.
      for (size_t i = 0; i < spanCount; ++i) {
        size_t runEnd    = i + 1;
        uint32 nextLevel = (runEnd < spanCount)
                               ? _runs[spanIndices[runEnd]].bidi_level()
                               : 0; // past last element

        // We only care about transitions, particularly high to low,
        // because that means we have a run behind us where we can
        // do something.

        if (currentLevel <=
            nextLevel) // This is now the beginning of the next run.
        {
          if (currentLevel < nextLevel) {
            currentLevel = nextLevel;
            runStart     = i + 1;
          }
          continue; // Skip past equal levels, or increasing stairsteps.
        }

        do // currentLevel > nextLevel
        {
          // Recede to find start of the run and previous level.
          uint32 previousLevel;
          for (;;) {
            if (runStart <= 0) // reached front of index list
            {
              previousLevel = 0; // position before string has bidi level of 0
              break;
            }
            if (_runs[spanIndices[--runStart]].bidi_level() < currentLevel) {
              previousLevel = _runs[spanIndices[runStart]].bidi_level();
              ++runStart; // compensate for going one element past
              break;
            }
          }

          // Reverse the indices, if the difference between the current and
          // next/previous levels is odd. Otherwise, it would be a no-op, so
          // don't bother.
          if ((min(currentLevel - nextLevel, currentLevel - previousLevel) & 1) != 0) {
            std::reverse(spanIndices + runStart, spanIndices + runEnd);
          }

          // Descend to the next lower level, the greater of previous and next
          currentLevel = max(previousLevel, nextLevel);
        } while (currentLevel >
                 nextLevel); // Continue until completely flattened.
      }
    }

    void text_flow::produce_justified_advances(
        view &v, element *elem, int y, int x1, int x2,
        const cluster_position_t &cluster_start,
        const cluster_position_t &cluster_end
        /*,OUT tool::array<float>& justifiedAdvances*/
        ) throw() {
      // Performs simple inter-word justification
      // using the breakpoint analysis whitespace property.

      // Copy out default, unjustified advances.
      uint32 glyph_start = get_cluster_glyph_start(cluster_start);
      uint32 glyph_end   = get_cluster_glyph_start(cluster_end);

      /*try
      {
        //justifiedAdvances.assign(_glyph_advances.begin() + glyph_start,
      _glyph_advances.begin() + glyph_end); justifiedAdvances =
      _glyph_advances(glyph_start,glyph_end);
      }
      catch (...)
      {
          return ExceptionToHResult();
      }*/

      if (glyph_end - glyph_start == 0) return; // No glyphs to modify.

      float maxWidth = float(x2 - x1 + 1);
      if (maxWidth <= 0) return; // Text can't fit anyway.

      style *elem_style          = elem->get_style(v);
      float  justified_space_cap = v.get_font(elem_style)->size;
      // justified_space_cap *= 2; // otherwise it looks ugly.

      ////////////////////////////////////////
      // First, count how many spaces there are in the text range.

      cluster_position_t cluster(cluster_start);
      uint32             whitespaceCount = 0;

      while (cluster.text_position < cluster_end.text_position) {
        if (_breakpoints[cluster.text_position].is_whitespace) {
          ++whitespaceCount;
        }
        advance_cluster_position(cluster);
      }
      if (whitespaceCount <= 0)
        return; // Can't justify using spaces, since none exist.

      ////////////////////////////////////////
      // Second, determine the needed contribution to each space.

      float lineWidth = get_cluster_range_width(glyph_start, glyph_end,
                                                &_glyph_advances.first());
      float justificationPerSpace = (maxWidth - lineWidth) / whitespaceCount;

      if (justificationPerSpace <= 0)
        return; // Either already justified or would be negative justification.

      if (justificationPerSpace > justified_space_cap /*(maxWidth / 8)*/)
        return; // Avoid justification if it would space the line out awkwardly
                // far.

      ////////////////////////////////////////
      // Lastly, adjust the advance widths, adding the difference to each space
      // character.

      cluster = cluster_start;
      while (cluster.text_position < cluster_end.text_position) {
        if (_breakpoints[cluster.text_position].is_whitespace)
          _glyph_justified_advances[get_cluster_glyph_start(cluster)] +=
              justificationPerSpace;

        advance_cluster_position(cluster);
      }
    }

    void text_flow::flex_children(view &v, element *elem, int x1, int x2,
      const cluster_position_t &clusterStart,
      const cluster_position_t &clusterEnd,
      tool::slice<inline_el_pos> flex_elements,
      float& line_width
    ) {

      flex::engine sc(flex_elements.size() * 7);

      int      width = elem->dim().x; // container width

      int      pix = 0, spr = 0;

      int width_to_distribute = x2 - x1 - int(line_width);

      for (inline_el_pos ep : flex_elements) {
        element* b = ep.elem;
        hstyle bcs = b->get_style(v);

        h_layout_data bld = b->ldata;

        bcs->margin[0].pixels_n_spring_h(v, b, width, pix, spr);
        sc.add(pix, int_v(), spr);

        bcs->used_border_width(0).pixels_n_spring_w(v, b, width, pix, spr);
        sc.add(pix, int_v(), spr);

        bcs->used_padding(0).pixels_n_spring_h(v, b, width, pix, spr);
        sc.add(pix, int_v(), spr);

        int   spring_width = 0;
        //int   percent_width = 0;
        //int   pref_width = 0;
        int   min_width = 0; //;
        int_v max_width;

        if (bcs->width.is_defined()) {
          spring_width = bcs->width.flex1000();
          //percent_width = bcs->width.percent();
          //pref_width = bcs->width.preferred_pixels_width(v, b, width);
          min_width = b->min_width(v, width);
          max_width = b->max_width(v, width);
        }
        else if (bcs->min_width.is_defined()) {
          min_width = b->min_width(v, width);
          max_width = b->max_width(v, width);
        }
        else {
          min_width = b->min_width(v, width);
          int cwidth = b->max_content_width(v);
          max_width = b->max_width(v, width).val(cwidth); // shrink to fit
          spring_width = 1000;
        }

        sc.add(min_width, max_width, spring_width);

        bcs->used_padding(2).pixels_n_spring_h(v, b, width, pix, spr);
        sc.add(pix, int_v(), spr);

        bcs->used_border_width(2).pixels_n_spring_w(v, b, width, pix, spr);
        sc.add(pix, int_v(), spr);

        bcs->margin[2].pixels_n_spring_h(v, b, width, pix, spr);
        sc.add(pix, int_v(), spr);

        width_to_distribute += b->computed_width(v, width) + b->outer_int_x_extra(v,width);
       
      }

      sc.calc(width_to_distribute);

      int n = 0;

      for (inline_el_pos ep : flex_elements) {
        element* b = ep.elem;
        hstyle bcs = b->get_style(v);
        handle<layout_data> bld = b->ldata;

        int w = 0;
        w += bld->margin_width.s.x = sc.val(n++);
        w += bld->border_width.s.x = sc.val(n++);
        w += bld->padding_width.s.x = sc.val(n++);
        int tw = sc.val(n++); b->set_width(v, tw); w += tw;
        w += bld->padding_width.e.x = sc.val(n++);
        w += bld->border_width.e.x = sc.val(n++);
        w += bld->margin_width.e.x = sc.val(n++);
        _glyph_justified_advances[ep.index] = float(w);
      }

    }

    uint spaces_for_tab(wchars text, uint tab_pos, uint spaces_per_tab) {
      uint ac = 0;
      for (uint n = 0; n < text.length; ++n) {
        wchar c = text[n];
        if (c == '\t') {
          uint r = spaces_per_tab - ac % spaces_per_tab;
          uint spaces = r ? r : spaces_per_tab;
          if (n == tab_pos)
            return spaces;
          ac += spaces;
        }
        else
          ++ac;
      }
      return 1;
    }

    float text_flow::get_tab_width(view &v, element *elem, const cluster_position_t &line_cluster_start, const cluster_position_t &cluster_start, const cluster_position_t &cluster_end) throw() {
      uint32 glyph_start = get_cluster_glyph_start(cluster_start);
      uint32 glyph_end = get_cluster_glyph_start(cluster_end);

      if (glyph_end - glyph_start == 0) return 0; // no glyphs.

      uint  spaces_per_tab = elem->get_tab_size();
      float space_width = v.get_font(elem->get_style(v))->ch();

      cluster_position_t cluster(cluster_start);

      float tabs_width = 0;

      while (cluster.text_position < cluster_end.text_position) {
        if ( _text[cluster.text_position] == '\t') {
          uint gs = get_cluster_glyph_start(cluster);
          uint spaces = spaces_for_tab(_text(line_cluster_start.text_position), cluster_start.text_position, spaces_per_tab);
          float width = spaces * space_width;
          tabs_width += width;
          //_glyph_advances[gs] - do not change original width
          _glyph_justified_advances[gs] = width;
        }
        advance_cluster_position(cluster);
      }
      return tabs_width;
    }


    ////////////////////////////////////////////////////////////////////////////////
    // Text/cluster navigation.
    //
    // Since layout should never split text clusters, we want to move ahead
    // whole clusters at a time.

    void text_flow::set_cluster_position(IN OUT cluster_position_t &cluster,
                                         uint32 text_position) const throw() {
      // Updates the current position and seeks its matching text analysis run.

      auto runs = _runs();
      if (!runs.length)
        return;

      cluster.text_position = text_position;

      // If the new text position is outside the previous analysis run,
      // find the right one.

      if (text_position >= cluster.runEndPosition ||
          !_runs[cluster.runIndex].contains_text_position(text_position)) {
        // If we can resume the search from the previous run index,
        // (meaning the new text position comes after the previously
        // seeked one), we can save some time. Otherwise restart from
        // the beginning.

        uint32 newRunIndex = 0;
        if (text_position >= _runs[cluster.runIndex].text_start) {
          newRunIndex = cluster.runIndex;
        }

        // Find new run that contains the text position.

        newRunIndex =
            uint(runs(newRunIndex).find(text_position).start - runs.start);

        // newRunIndex = static_cast<uint32>(
        //                    std::find(_runs.head() + newRunIndex,
        //                    _runs.tail(), text_position)
        //                    - _runs.head()
        //                    );

        // Keep run index within the list, rather than pointing off the end.
        if (newRunIndex >= _runs.length()) {
          newRunIndex = uint(_runs.length() - 1);
        }

        // Cache the position of the next analysis run to efficiently
        // move forward in the clustermap.
        const text_run &matchingRun = _runs[newRunIndex];
        cluster.runIndex            = newRunIndex;
        cluster.runEndPosition =
            matchingRun.text_start + matchingRun.text_length;
      }
    }

    void text_flow::advance_cluster_position(
        IN OUT cluster_position_t &cluster) const throw() {
      // Looks forward in the cluster map until finding a new cluster,
      // or until we reach the end of a cluster run returned by shaping.
      //
      // Glyph shaping can produce a clustermap where a:
      //  - A single codepoint maps to a single glyph (simple Latin and
      //  precomposed CJK)
      //  - A single codepoint to several glyphs (diacritics decomposed into
      //  distinct glyphs)
      //  - Multiple codepoints are coalesced into a single glyph.
      //
      uint32 text_position = cluster.text_position;
      uint32 cluster_id    = _glyph_clusters[text_position];

      for (++text_position; text_position < cluster.runEndPosition;
           ++text_position) {
        if (_glyph_clusters[text_position] != cluster_id) {
          // Now pointing to the next cluster.
          cluster.text_position = text_position;
          return;
        }
      }
      if (text_position == cluster.runEndPosition) {
        // We crossed a text analysis run.
        set_cluster_position(cluster, text_position);
      }
    }

    bool text_flow::advance_cluster_position_next_brick(
        IN view &v, IN OUT cluster_position_t &cluster, OUT bool &hard_lf) const
        throw() {
      uint text_length = (uint)_text.length();
      if (cluster.text_position >= text_length) return false;
      // cluster_position_t next_cluster = cluster;
      for (;;) {
        element *inline_box_element = get_inline_box_element_at(v, cluster);
        // Use breakpoint information to find where we can safely break words.
        advance_cluster_position(cluster);
        if (inline_box_element) break;
        if (cluster.text_position >= text_length) break;
        const LINE_BREAKPOINT breakpoint =
            _breakpoints[cluster.text_position - 1];
        // see if we can break after this character cluster, and if so, stop
        if (breakpoint.break_condition_after != BREAK_CONDITION_MAY_NOT_BREAK) {
          if (breakpoint.break_condition_after == BREAK_CONDITION_MUST_BREAK)
            hard_lf = true; // we have a hard return, so we've fit all we can.
          else
            hard_lf = false;
          break;
        }
      }
      return true;
    }

    uint32
    text_flow::get_cluster_glyph_start(const cluster_position_t &cluster) const
        throw() {
      // Maps from text position to corresponding starting index in the glyph
      // array. This is needed because there isn't a 1:1 correspondence between
      // text and glyphs produced.

      uint32 glyph_start = _runs[cluster.runIndex].glyph_start;

      return (cluster.text_position < _glyph_clusters.length())
                 ? glyph_start + _glyph_clusters[cluster.text_position]
                 : glyph_start + _runs[cluster.runIndex].glyph_count;
    }

    float text_flow::get_cluster_range_width(
        const cluster_position_t &cluster_start,
        const cluster_position_t &cluster_end) const throw() {
      // Sums the glyph advances between two cluster positions,
      // useful for determining how long a line or word is.

      if (_glyph_advances.length() == 0)
        return 0.0f;
      return get_cluster_range_width(get_cluster_glyph_start(cluster_start),
                                     get_cluster_glyph_start(cluster_end),
                                     &_glyph_advances.first());
    }

    float
    text_flow::get_cluster_range_width(uint32 glyph_start, uint32 glyph_end,
                                       const float *glyphAdvances // [glyph_end]
                                       ) const throw() {
      // Sums the glyph advances between two glyph offsets, given an explicit
      // advances array - useful for determining how long a line or word is.
      float sum = 0.0f;
      for (uint n = glyph_start; n < glyph_end; ++n)
        sum += glyphAdvances[n];
      return sum;
    }

    const glyph_run *find_glyph_run(slice<glyph_run> gruns,
                                    uint32           glyph_index) {
      for (uint n = 0; n < gruns.length; ++n) {
        const glyph_run &gr = gruns[n];
        if (glyph_index >= gr.glyph_start &&
            glyph_index < (gr.glyph_start + gr.glyph_count))
          return &gr;
      }
      // no luck, try boundaries
      for (uint n = 0; n < gruns.length; ++n) {
        const glyph_run &gr = gruns[n];
        if (glyph_index >= gr.glyph_start &&
            glyph_index <= (gr.glyph_start + gr.glyph_count))
          return &gr;
      }
      return 0;
    }

    bool text_flow::get_glyph_metrics(uint32         glyph_index,
                                      caret_metrics &m) const throw() {
      bool after = false;
      uint mx    = this->_glyph_indices.last_index();
      if (glyph_index > mx) {
        glyph_index = mx;
        after       = true;
      }
      // auto gr = _glyph_runs().find(glyph_index);
      const glyph_run *gr = find_glyph_run(_glyph_runs(), glyph_index);
      if (!gr) {
        if (_glyph_runs.size() == 0) {
          // assert(!!gr);
          return false;
        }
        gr = &_glyph_runs[0];
      }

      if (!gr->is_rtl()) { // LTR
        m.x1 = gr->x + get_cluster_range_width(gr->glyph_start, glyph_index, _glyph_justified_advances.head());
        m.x2 = m.x1 + _glyph_justified_advances[glyph_index] - 1;
        if (glyph_index == 0 && is_empty_text())
          m.x2 = m.x1;
      } else {
        // RTL
        m.x1 = gr->x - get_cluster_range_width(gr->glyph_start, glyph_index, _glyph_justified_advances.head());
        m.x2 = max(0.0f, m.x1 - _glyph_justified_advances[glyph_index]) - 1;
        if (glyph_index == 0 && is_empty_text())
          m.x1 = m.x2;
      }

      if (after) m.x1 = m.x2;

      layout_line line = _lines[gr->line_no];
      int         fa, fd, fx;
      gr->font->metrics(fa, fd, fx, gr->font->size);

      m.line_y1 = line.y;
      m.line_y2 = line.y + line.height - 1;
      m.y1      = max(m.line_y1, line.y + line.baseline - fa);
      m.y2      = min(m.line_y2, line.y + line.baseline + fd);
      m.line_no = gr->line_no;
      return true;
    }

    void text_flow::reset_glyph_runs() {
      _glyph_runs.clear();
      _lines.clear();
      _glyph_justified_advances = _glyph_advances;
    }

    uint text_flow::set_glyph_run(float x, byte valign, uint32 glyph_start,
                                  uint32 glyph_count, node *pnode,
                                  html::style *pstyle, gool::font *font,
                                  // float font_size,
                                  byte bidi_level, bool is_sideways,
                                  uint line_no, char_mark_t marks) {
      assert(glyph_count < 10000);
      // append this glyph run to the list.
      glyph_run &glyph_run = _glyph_runs.push();
      // assert(font_face);
      glyph_run.x           = x;
      glyph_run.valign      = valign;
      glyph_run.bidi_level  = bidi_level;
      glyph_run.glyph_start = glyph_start;
      glyph_run.glyph_count = glyph_count;
      glyph_run.is_sideways = is_sideways;
      glyph_run.node        = pnode;
      glyph_run.line_no     = line_no;
      glyph_run.marks       = marks;
      glyph_run.font        = font;
      // glyph_run.font_size     = font_size;
      glyph_run.pstyle = pstyle;
      return (uint)_glyph_runs.last_index();
    }

  } // namespace tflow
} // namespace html
