#include "html.h"

namespace html {
  static helement placeholder = new element(0);

  int cell_spacing_x(view &v, element *el, int width);
  int cell_spacing_y(view &v, element *el, int height);

  block_table_body *block_table_body::setup_on(view &v, element *el) {
    block_table_body *tb = turn_element_to<block_table_body>(el);
    return tb;
  }

  bool out_of_flow(view &v, element *pel) {
    if (!pel)
      return true;
    hstyle cs = pel->get_style(v);
    return cs->collapsed() || pel->oof_positioned(v);
  }

  array<col_def> &get_col_def(block_table_body *tb) {
    element *p = tb->get_owner(); // parent_or_owner(tb);
    while (p) {
      if (!p->flags.layout_ctl_valid || !p->ldata->is_initialized) {
        view *pv = p->pview();
        if (pv) p->setup_layout(*pv);
      }
      if (p->is_table())
        return p->ldata.ptr_of<block_table::layout_data>()->cols;
      p = p->get_owner(); // parent_or_owner(p);
    }
    assert(false);
    static array<col_def> dummy;
    return dummy;
  }

  block_table_body::layout_data::layout_data(block_table_body *self)
      : super(self), cols(get_col_def(self)) {}

  void block_table_body::layout_data::append_cell(int row_n, element *el) {
    int rs = el->atts.get_rowspan();
    int cs = el->atts.get_colspan();

    assert(row_n < rows.size());

    row_def &rd    = rows[row_n];
    int      col_n = 0;
    for (; col_n < rd.cells.size(); ++col_n)
      if (rd.cells[col_n].b == 0) break;
    set_cell_at(row_n, col_n, el, rs, cs);
  }

  void block_table_body::layout_data::set_cell_at(int row_n, int col_n,
                                                  element *el, int rs, int cs) {
    cell_def cd;
    cd.b                 = el;
    cd.cols.l            = col_n;
    cd.cols.h            = col_n + cs - 1;
    cd.rows.l            = row_n;
    cd.rows.h            = row_n + rs - 1;
    cd.b->flags.ui_index = col_n;

    int ncolumns = col_n + cs;
    int nrows    = row_n + rs;
    if (ncolumns > cols.size()) // we need more columns
    {
      cols.size(ncolumns);
      for (int r = 0; r < rows.size(); ++r)
        rows[r].cells.size(ncolumns);
    } else
      ncolumns = cols.size();

    while (nrows > rows.size()) // we need more rows
      rows.push().cells.size(ncolumns);

    if (rs > 1 || cs > 1) {
      for (int r = cd.rows.l; r <= cd.rows.h; ++r)
        for (int c = cd.cols.l; c <= cd.cols.h; ++c)
          rows[r].cells[c].b = placeholder;
    }
    rows[row_n].cells[col_n] = cd;
  }

  void block_table_body::fixup_layout(view &v) {
    layout_data *pld = ldata.ptr_of<layout_data>();
    uint         rr  = 0;
    for (int r = pld->rows.last_index(); r >= 0; --r) {
      if (!pld->rows[r].row) {
        pld->rows.pop();
        ++rr;
      }
    }
    if (rr == 0) return; // nothing to fix;

    irange yr(0, pld->rows.last_index());

    for (int r = pld->rows.last_index(); r >= 0; --r) {
      row_def &rd = pld->rows[r];
      int      nc = rd.cells.size();
      for (int c = 0; c < nc; ++c) {
        cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;
        cd.rows &= yr;
      }
    }
  }

  bool is_row(view &v, element *el);

  void block_table_body::layout_data::push(view &v, element *self,
                                           element *el) {
    el->get_style(v);
    //el->check_layout(v);
#ifdef _DEBUG
    //if (el->layout_type() != flow_table_row && !el->flags.in_setup_layout) {
    if(!is_row(v,el)) {
      //lt = el->check_layout(v);
      el->dbg_report("wrong element type in TBODY");
      assert(false);
    }
#endif
    int row_n = rows.size();
    for (int r = row_n - 1; r >= 0; --r) {
      if (rows[r].row) break;
      row_n = r;
    }
    row_def &rd = row_n >= rows.size() ? rows.push() : rows[row_n];
    rd.cells.size(cols.size());
    rd.row = el;
    assert(el->parent);
    if (!el->parent) el->parent = self;
    el->owner          = self;
    el->flags.ui_index = row_n;

    el->each_child([this, row_n](element *cell) -> bool {
      this->append_cell(row_n, cell);
      return false;
    });

    drop_minmax_dim();
  }

  /*element* block_table_body::first_ui_element() const// ui child
  {
    layout_data* pld = ldata.ptr_of<layout_data>();
    if(pld->rows.size() )
      return pld->rows.first().row;
    return 0;
  }
  element* block_table_body::last_ui_element() const // ui child
  {
    layout_data* pld = ldata.ptr_of<layout_data>();
    if(pld->rows.size() )
      return pld->rows.last().row;
    return 0;
  }

  element* block_table_body::next_ui_child(const element* el) const
  {
    layout_data* pld = ldata.ptr_of<layout_data>();
    assert(el->flags.ui_index >= 0);
    int idx = el->flags.ui_index + 1;
    if( idx < pld->rows.last_index() )
      return pld->rows[idx].row;
    return 0;
  }

  element* block_table_body::prev_ui_child(const element* el) const
  {
    layout_data* pld = ldata.ptr_of<layout_data>();
    assert(el->flags.ui_index >= 0);
    int idx = el->flags.ui_index - 1;
    if( idx >= 0 )
      return pld->rows[idx].row;
    return 0;
  }*/

  int block_table_body::n_ui_children() const {
    layout_data *pld = ldata.ptr_of<layout_data>();
    return pld->rows.size();
  }
  element *block_table_body::ui_child(int idx) const {
    layout_data *pld = ldata.ptr_of<layout_data>();
    return pld->rows[idx].row;
  }

  int vflex(const size_v &w) {
    if (w.is_spring()) return w.flex1000();
    if (w.is_percent()) return w.percent() * 10;
    return 0;
  }

  void block_table_body::_commit_measure(view &v) {
    get_style(v);
    if (!ldata->is_valid()) {
      // the best we can do here is to measure it in place
      size dim = ldata->dim;
      int  h   = set_width(v, dim.x);
      set_height(v, max(h, dim.y));
      v.refresh(this);
      assert(ldata->is_valid());
    } else
      for (element *c = first_ui_element(); c; c = c->next_ui_element())
        c->_commit_measure(v);
  }

  element *block_table_body::drop_layout(view *pv)
  {
    return super::drop_layout(pv);
  }

  void block_table_body::drop_content_layout(view *pv) {
    super::drop_content_layout(pv);
    block_table *tb = this->parent_table();
    if (tb) tb->drop_content_layout(pv);
  }

  void block_table_body::calc_intrinsic_widths(view &v) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    assert(ld->is_of_type<block_table_body::layout_data>());
    // if(!ld->is_of_type<block_table_body::layout_data>())
    //  alert("!!!");

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_rows = ld->rows.size();
    int n_cols = ld->cols.size();

    int inners = ld->inner_borpad_x();

    if (n_rows == 0 || n_cols == 0) {
      ld->dim_min.x = inners;
      ld->dim_max.x = inners;
      return;
    }

    this->flags.in_setup_layout = 1;

    // styles of all cells need to known at this point as they may drop calculation of array<col_def> &cols 
    resolve_styles(v);

    bool collapsed_model = cs->border_collapse == border_collapse;
    int  inter_cell_x    = cell_spacing_x(v, this, ld->dim.x);

    int min_w = inter_cell_x * (n_cols - 1) + inners;
    int max_w = inter_cell_x * (n_cols - 1) + inners;

    int first_static_row = 0;
    for (; first_static_row < n_rows; ++first_static_row) {
      if (!out_of_flow(v, rows[first_static_row].row)) break;
    }

    if (is_fixed() && tag == tag::T_TBODY)
      n_rows = min(n_rows,first_static_row + 1); // only first row participates in width calculations
    
    int container_width = ld->dim.x;

    uint n_x_spanned = 0;
    for (int r = first_static_row; r < n_rows; ++r) {
      row_def &rd = rows[r];
      int      nc = rd.cells.size();
      if (nc < n_cols) rd.cells.size(n_cols);
      // assert(nc == n_cols); nc = nc;
      for (int c = 0; c < n_cols; ++c) {
        const cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;
        assert(!cd.rows.empty());
        assert(!cd.cols.empty());
        style *bcs = cd.b->get_style(v);
        if (bcs->is_display_none())
          continue;

        premeasure(v, cd.b, bcs, ld->dim);
        flex_value fv;
       
        int borpad =
            cd.b->cell_int_x_extra(v, container_width, collapsed_model);
        fv.px     = cd.b->min_width(v) + borpad;
        fv.px_max = cd.b->max_width(v).val(cd.b->max_content_width(v)) + borpad;
        fv.flex   = vflex(bcs->width);
        if (cd.cols.length() == 1)
          cols[cd.cols.l].minmax.accum(fv);
        else
          ++n_x_spanned;
        // this thing spans more than one column so we need to distribute its
        // min/max/weight
      }
    }
    for (int r = 0; n_x_spanned > 0 && r < n_rows; ++r) {
      row_def &rd = rows[r];
      for (int c = 0; n_x_spanned > 0 && c < n_cols; ++c) {
        cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;
        if (cd.cols.length() == 1) continue;
        // spanned cell
        --n_x_spanned;
        style *bcs                  = cd.b->get_style(v);
        if (bcs->is_display_none())
          continue;

        int    spanned_inter_cell_x = (cd.cols.length() - 1) * inter_cell_x;

        flex_value fv;
        int        borpad = cd.b->cell_int_x_extra(v, container_width, collapsed_model);
        fv.px = cd.b->min_width(v) + borpad - spanned_inter_cell_x;
        fv.px_max = cd.b->max_width(v).val(cd.b->max_content_width(v)) + borpad - spanned_inter_cell_x;
        fv.flex = vflex(bcs->width);
        flex_value spanned;
        for (int i = cd.cols.l; i <= cd.cols.h; ++i) {
          col_def &t = cols[i];
          spanned.flex += t.minmax.flex;
          spanned.px += t.minmax.px;
          spanned.px_max += t.minmax.px_max;
        }
        if (spanned.px < fv.px) {
          int extra_width  = fv.px - spanned.px;
          int total_weight = max(cd.cols.length(), spanned.px + spanned.px_max);
          for (int i = cd.cols.l; i <= cd.cols.h; ++i) {
            if (extra_width <= 0) break;
            if (total_weight <= 0) break;
            col_def &t      = cols[i];
            int      weight = max(1, t.minmax.px + t.minmax.px_max);
            int      extra  = (weight * extra_width) / total_weight;
            total_weight -= weight;
            t.minmax.px += extra;
            extra_width -= extra;
          }
        }
        if (spanned.px_max < fv.px_max) {
          int extra_width  = fv.px_max - spanned.px_max;
          int total_weight = max(cd.cols.length(), spanned.px_max);
          for (int i = cd.cols.l; i <= cd.cols.h; ++i) {
            if (extra_width <= 0) break;
            if (total_weight <= 0) break;
            col_def &t      = cols[i];
            int      weight = max(1, t.minmax.px_max);
            int      extra  = (weight * extra_width) / total_weight;
            total_weight -= weight;
            t.minmax.px_max += extra;
            extra_width -= extra;
          }
        }
        if (spanned.flex < fv.flex) {
          int extra_flex   = fv.flex - spanned.flex;
          int total_weight = max(cd.cols.length(), spanned.px_max);
          for (int i = cd.cols.l; i <= cd.cols.h; ++i) {
            if (extra_flex <= 0) break;
            if (total_weight <= 0) break;
            col_def &t      = cols[i];
            int      weight = max(1, t.minmax.px_max);
            int      extra  = (weight * extra_flex) / total_weight;
            total_weight -= weight;
            t.minmax.flex += extra;
            extra_flex -= extra;
          }
        }

      } // int c;
    }   // int r;
    for (int n = 0; n < cols.size(); ++n) {
      col_def &t = cols[n];
      min_w += t.minmax.px;
      max_w += t.minmax.px_max;
    }

    ld->dim_min.x = min_w;
    ld->dim_max.x = max(ld->dim_min.x.val(0), max_w);

    this->flags.in_setup_layout = 0;

  }

  int block_table_body::layout_width(view &v, int width) {
    if (is_fixed() && tag == tag::T_TBODY) return layout_width_fixed(v, width);

    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->dim.x = width;

    size inner_dim = client_rect(v).size();

    if (ld->dim_min.x.is_defined() && ld->dim_min.y.is_defined() &&
        ld->inner_dim.x == inner_dim.x)
      return ld->dim_min.y; // already measured

    ld->inner_dim.x = inner_dim.x;
    ld->inner_dim.y = 0;
    ld->dim.y       = 0; // ?
    ld->dim_min.y.clear();
    ld->dim_max.y.clear();

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_cols = cols.size();
    int n_rows = rows.size();

    if (ld->dim_min.x.is_undefined() || flags.has_percent_widths)
      calc_intrinsic_widths(v);

    if (n_cols == 0 || n_rows == 0) {
      return ld->dim_min.y = ld->dim_max.y = 0;
    }

    bool collapsed_model = cs->border_collapse == border_collapse;
    int  inter_cell_x    = cell_spacing_x(v, this, ld->dim.x);
    int  inter_cell_y    = cell_spacing_y(v, this, ld->dim.y);

    // adjust rows

    int n_y_spanned = 0;

    for (int r = 0; r < n_rows; ++r) {
      row_def &rd = rows[r];
      rd.minmax   = flex_value();
      if (out_of_flow(v, rd.row)) continue;
      for (int c = 0; c < n_cols; ++c) {
        cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;

        int tc, w = 0;
        for (tc = cd.cols.l; tc <= cd.cols.h; ++tc) {
          col_def &cd = cols[tc];
          w += cd.value;
          w += inter_cell_x;
        }
        w -= inter_cell_x;

        if (cd.b->positioned(v)) cd.b->check_positioned_containment(v);

        cd.b->set_cell_width(v, w, collapsed_model);

        style *bcs = cd.b->get_style(v);

        int        borpad = cd.b->borpad_int_y_extra(v);
        flex_value fv;
        fv.px     = cd.b->min_height(v) + borpad;
        fv.px_max = cd.b->max_height(v) + borpad;

        fv.flex = bcs->height.flex1000();

        if (cd.rows.length() == 1)
          rows[cd.rows.l].minmax.accum(fv);
        else
          ++n_y_spanned; // spans more than one row

      } // int c
    }   // int r

    for (int r = 0; n_y_spanned > 0 && r < n_rows; ++r) {
      row_def &rd = rows[r];
      for (int c = 0; n_y_spanned > 0 && c < n_cols; ++c) {
        const cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;

        if (cd.rows.length() == 1) continue;
        // this thing spans more than one column so we need to distribute its
        // min/max/weight
        --n_y_spanned;

        style *    bcs                  = cd.b->get_style(v);
        int        spanned_inter_cell_y = (cd.rows.length() - 1) * inter_cell_y;
        flex_value fv;
        int        borpad = cd.b->cell_int_y_extra(v, 0, collapsed_model);
        fv.px             = cd.b->min_height(v) + borpad - spanned_inter_cell_y;
        fv.px_max = cd.b->max_height(v) + borpad - spanned_inter_cell_y;
        fv.flex   = vflex(bcs->height);
        flex_value spanned;
        for (int i = cd.rows.l; i <= cd.rows.h; ++i) {
          row_def &t = rows[i];
          spanned.flex += t.minmax.flex;
          spanned.px += t.minmax.px;
          spanned.px_max += t.minmax.px_max;
        }
        if (spanned.px < fv.px) {
          int extra_height = fv.px - spanned.px;
          int total_weight = max(cd.rows.length(), spanned.px + spanned.px_max);
          for (int i = cd.rows.l; i <= cd.rows.h; ++i) {
            if (extra_height <= 0) break;
            if (total_weight <= 0) break;
            row_def &t      = rows[i];
            int      weight = max(1, t.minmax.px + t.minmax.px_max);
            int      extra  = (weight * extra_height) / total_weight;
            total_weight -= weight;
            t.minmax.px += extra;
            // min_h += extra;
            extra_height -= extra;
          }
        }
        if (spanned.px_max < fv.px_max) {
          int extra_height = fv.px_max - spanned.px_max;
          int total_weight = max(cd.rows.length(), spanned.px_max);
          for (int i = cd.rows.l; i <= cd.rows.h; ++i) {
            if (extra_height <= 0) break;
            if (total_weight <= 0) break;
            row_def &t      = rows[i];
            int      weight = max(1, t.minmax.px_max);
            int      extra  = (weight * extra_height) / total_weight;
            total_weight -= weight;
            t.minmax.px_max += extra;
            // max_h += extra;
            extra_height -= extra;
          }
        }
        if (spanned.flex < fv.flex) {
          int extra_flex   = fv.flex - spanned.flex;
          int total_weight = max(cd.rows.length(), spanned.px_max);
          for (int i = cd.rows.l; i <= cd.rows.h; ++i) {
            if (extra_flex <= 0) break;
            if (total_weight <= 0) break;
            row_def &t      = rows[i];
            int      weight = max(1, t.minmax.px_max);
            int      extra  = (weight * extra_flex) / total_weight;
            total_weight -= weight;
            t.minmax.flex += extra;
            extra_flex -= extra;
          }
        }
      } // int c
    }   // int r

    int inners = ld->inner_borpad_y();
    int min_h = inter_cell_y * (n_rows - 1) + inners;
    int max_h = inter_cell_y * (n_rows - 1) + inners;

    for (uint n = 0; n < rows.length(); ++n) {
      row_def &rd = rows[n];

      if (out_of_flow(v, rd.row)) {
        min_h -= inter_cell_y;
        max_h -= inter_cell_y;
      } else {
        min_h += rd.minmax.px;
        max_h += rd.minmax.px_max;
      }
    }

    ld->dim_min.y = min_h;
    ld->dim_max.y = max(ld->dim_min.y, max_h);
    layout_height(v, ld->dim_min.y);
    return ld->dim_min.y;
  }

  INLINE void copy_metrics(layout_data *dst, const layout_data *src) {
    dst->border_width  = src->border_width;
    dst->margin_width  = src->margin_width;
    dst->padding_width = src->padding_width;
  }

  static int overlapping_y(view &v, element *prev_blk, element *blk,
                           int inter_cell_y) {
    int ppix  = 0;
    int npix  = 0;
    int pspr  = 0;
    int nspr  = 0;
    int ppref = 0;
    int npref = 0;

    if (prev_blk) calc_margin_bottom(v, prev_blk, 0, ppix, pspr, ppref);
    if (blk) calc_margin_top(v, blk, 0, npix, nspr, npref);

    int r = overlap(ppix, npix);
    if (prev_blk && blk)
      return max(r, inter_cell_y);
    else
      return r;
  }

  int block_table_body::layout_width_fixed(view &v, int width) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->dim.x = width;

    size inner_dim = client_rect(v).size();

    if (ld->dim_min.x.is_defined() && ld->dim_min.y.is_defined() &&
        ld->inner_dim.x == inner_dim.x)
      return ld->dim_min.y; // already measured

    ld->inner_dim.x = inner_dim.x;
    ld->inner_dim.y = 0;
    ld->dim.y       = 0; // ?
    ld->dim_min.y.clear();
    ld->dim_max.y.clear();

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_cols = cols.size();
    int n_rows = rows.size();

    if (ld->dim_min.x.is_undefined()) calc_intrinsic_widths(v);

    if (n_cols == 0 || n_rows == 0) {
      return ld->dim_min.y = ld->dim_max.y = 0;
    }

    // int inter_cell_x = cs->border_spacing_x.pixels_width(v,this,ld->dim.x);
    // int inter_cell_y = cs->border_spacing_y.pixels_height(v,this,ld->dim.y);
    bool collapsed_model = cs->border_collapse == border_collapse;
    int  inter_cell_x    = cell_spacing_x(v, this, ld->dim.x);
    int  inter_cell_y    = cell_spacing_y(v, this, ld->dim.y);

    // adjust rows

    flex_value row_fv;

    int first_static_row = 0;
    for (; first_static_row < n_rows; ++first_static_row) {
      if (!out_of_flow(v, rows[first_static_row].row)) break;
    }

    if (first_static_row >= n_rows) {
      return ld->dim_min.y = ld->dim_max.y = 0;
    }

    // if(first_static_row)
    // first_static_row = first_static_row;

    row_def &rd0 = rows[first_static_row];

    int inners = ld->inner_borpad_y();

    int min_h = inners;
    int max_h = inners;
    
    element *prev = nullptr;

    {
      rd0.minmax = flex_value();
      for (int c = 0; c < n_cols; ++c) {
        cell_def &cd = rd0.cells[c];
        if (!cd.b || cd.b == placeholder) continue;

        int tc, w = 0;
        for (tc = cd.cols.l; tc <= cd.cols.h; ++tc) {
          col_def &cd = cols[tc];
          w += cd.value;
          w += inter_cell_x;
        }
        w -= inter_cell_x;

        cd.b->set_cell_width(v, w, collapsed_model);

        style *bcs = cd.b->get_style(v);

        int        borpad = cd.b->cell_int_y_extra(v, 0, collapsed_model);
        flex_value fv;
        row_fv.px     = cd.b->min_height(v) + borpad;
        row_fv.px_max = cd.b->max_height(v) + borpad;
        row_fv.flex   = bcs->height.flex1000();

        // if( cd.rows.length() == 1 ) - fixed layout does not support row spans
        rows[cd.rows.l].minmax.accum(row_fv);
        // else
        //  ++n_y_spanned; // spans more than one row

      } // int c

      int overlapping = overlapping_y(v, prev, rd0.row, inter_cell_y);
      prev            = rd0.row;
      min_h += overlapping;
      max_h += overlapping;
      min_h += rd0.minmax.px;
      max_h += rd0.minmax.px_max;

    } // int r

    row_fv = rd0.minmax;

    for (int r = first_static_row + 1; r < n_rows; ++r) {
      row_def &rd = rows[r];

      if (out_of_flow(v, rd.row)) continue;

      size rd_dim = rd.row->dim();

      rd.minmax = row_fv; // flex_value();
      for (int c = 0; c < n_cols; ++c) {
        cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;

        int tc, w = 0;
        for (tc = cd.cols.l; tc <= cd.cols.h; ++tc) {
          col_def &cd = cols[tc];
          w += cd.value;
          w += inter_cell_x;
        }
        w -= inter_cell_x;

        // cell_def& cd0 = rd0.cells[c];
        // style* cs = cd.b->get_style(v);

        cd.b->check_layout(v);
        cd.b->measure_borders_x(v, rd_dim);
        cd.b->measure_borders_y(v, rd_dim);
        // WRONG: copy_metrics(cd.b->ldata,cd0.b->ldata); cells may have
        // different paddings for example  WRONG, horizontal alignment:
        // cd.b->set_border_width_nm(v,w);
        cd.b->set_cell_width(v, w, collapsed_model);

        rows[cd.rows.l].minmax.accum(row_fv);

      } // int c

      int overlapping = overlapping_y(v, prev, rd.row, inter_cell_y);
      prev            = rd.row;
      min_h += overlapping;
      max_h += overlapping;
      min_h += rd.minmax.px;
      max_h += rd.minmax.px_max;

    } // int r

    if (prev && !prev->get_style(v)->is_display_none()) {
      int last_overlapping = overlapping_y(v, prev, nullptr, inter_cell_y);
      min_h += last_overlapping;
      max_h += last_overlapping;
    }

    ld->dim_min.y = min_h;
    ld->dim_max.y = max(ld->dim_min.y, max_h);
    return ld->dim_min.y;
  }

  int block_table_body::layout_height(view &v, int height) {
    if (is_fixed() && tag == tag::T_TBODY)
      return layout_height_fixed(v, height);

    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->dim.y = height;

    rect client_place = client_rect(v);
    size inner_dim    = client_place.size();

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_cols = cols.size();
    int n_rows = rows.size();

    if (n_cols == 0 || n_rows == 0) return ld->dim.x;

    bool collapsed_model = cs->border_collapse == border_collapse;
    // int inter_cell_x = cell_spacing_x(v, this, ld->dim.x);
    int inter_cell_y = cell_spacing_y(v, this, ld->dim.y);

    if (ld->dim_min.y.is_defined() && ld->dim_max.y.is_defined() &&
        ld->inner_dim.y == inner_dim.y)
      return ld->dim.x;

    ld->inner_dim = inner_dim;

    flex::engine se(rows.size());

    int height_to_distribute =
        max(ld->dim_min.y, inner_dim.y) - (rows.size() - 1) * inter_cell_y;

    for (int r = 0; r < n_rows; ++r) {
      row_def &t = rows[r];
      // if(t.row->get_style(v)->visibility != visibility_collapse)
      if (!out_of_flow(v, t.row))
        se.add(t.minmax.px, int_v(), t.minmax.flex);
      else {
        se.add(0);
        height_to_distribute -= inter_cell_y;
      }
    }

    se.calc(height_to_distribute);

    int y = ld->inner_borpad_top();
    int x = ld->inner_borpad_left();
    int i = 0;
    for (int r = 0; r < n_rows; ++r) // setup heights
    {
      row_def &rd = rows[r];
      rd.pos      = y;
      if (out_of_flow(v, rd.row)) continue;
      y += (rd.value = se.val(i++));
      rd.spanned_value = rd.value;
      y += inter_cell_y;
    }

    bool rtl = cs->direction == direction_rtl;

    for (int r = 0; r < n_rows; ++r) // setup heights
    {
      row_def &rd = rows[r];
      for (int c = 0; c < n_cols; ++c) // setup heights
      {
        const cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;
        int h = 0;
        for (int tr = cd.rows.l; tr <= cd.rows.h; ++tr) {
          row_def &rd = rows[tr];
          if (rd.row->c_style->visibility != visibility_collapse) {
            h += rd.value;
            h += inter_cell_y;
          }
        }
        h -= inter_cell_y;
        if (rd.spanned_value < h) rd.spanned_value = h;
        cd.b->set_cell_height(v, h, collapsed_model);
        point pt;
        if (rtl)
          pt.x = cols[cd.cols.h].pos - client_place.s.x + x;
        else
          pt.x = cols[cd.cols.l].pos - client_place.s.x + x;
        pt.y = rows[cd.rows.l].pos;
        cd.b->set_cell_pos(pt, collapsed_model);
      }
      rd.row->set_pos(x, rd.pos);
      rd.row->ldata->dim.y = rd.spanned_value;
      rd.row->ldata->dim.x = inner_dim.x;
      rd.layout_valid      = true;
    }
    return ld->dim.x;
  }

  int block_table_body::layout_height_fixed(view &v, int height) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    ld->dim.y = height;

    rect client_place = client_rect(v);
    size inner_dim    = client_place.size();

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_cols = cols.size();
    int n_rows = rows.size();

    if (n_cols == 0 || n_rows == 0) return ld->dim.x;

    bool collapsed_model = cs->border_collapse == border_collapse;
    int  inter_cell_y    = cell_spacing_y(v, this, ld->dim.y);

    if (ld->dim_min.y.is_defined() && ld->dim_max.y.is_defined() &&
        ld->inner_dim.y == inner_dim.y)
      return ld->dim.x;

    ld->inner_dim = inner_dim;

    int y = ld->inner_borpad_top();
    int x = ld->inner_borpad_left();

    element *prev = nullptr;

    int row_width = 0;
    for (int c = cols.last_index(); c >= 0; --c) {
      row_width += cols[c].value;
    }

    bool rtl = cs->direction == direction_rtl;

    for (int r = 0; r < n_rows; ++r) // setup heights
    {
      row_def &rd = rows[r];

      y += overlapping_y(v, prev, rd.row, inter_cell_y);
      prev = rd.row;

      rd.pos           = y;
      rd.spanned_value = rd.value = rd.minmax.px;
      y += rd.value;

      for (int c = 0; c < n_cols; ++c) // setup heights
      {
        const cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;
        int h = 0;
        for (int tr = cd.rows.l; tr <= cd.rows.h; ++tr) {
          row_def &rd = rows[tr];
          if (rd.row->c_style->visibility != visibility_collapse) {
            h += rd.value;
            h += inter_cell_y;
          }
        }
        h -= inter_cell_y;
        if (rd.spanned_value < h) rd.spanned_value = h;
        cd.b->set_cell_height_nm(v, h, collapsed_model);
        point pt;
        if (rtl)
          pt.x = cols[cd.cols.h].pos - client_place.s.x + x;
        else
          pt.x = cols[cd.cols.l].pos - client_place.s.x + x;
        pt.y = rows[cd.rows.l].pos;
        cd.b->set_cell_pos(pt, collapsed_model);
      }
      rd.row->set_pos(x, rd.pos);
      rd.row->ldata->dim.y = rd.spanned_value;
      rd.row->ldata->dim.x = row_width;
      rd.layout_valid      = true;
    }
    return ld->dim.x;
  }

  struct find_row_pos_env_v // find_element traits for binary search
  {
    view &         v;
    point          pt;
    slice<row_def> elements;

    find_row_pos_env_v(view &v_, slice<row_def> elems, point pt_)
        : v(v_), pt(pt_), elements(elems) {}

    int total() const { return int(elements.length); }

    bool is_less(int n) {
      const row_def &rd = elements[n];
      return rd.pos + rd.spanned_value < pt.y;
    }
    bool is_equal(int n) {
      const row_def &rd = elements[n];
      return pt.y >= rd.pos && pt.y < rd.pos + rd.spanned_value;
    }
    bool is_comparable(int n) {
      /*element* b = elements[n];
      if( b->state.popup() ) // it is handled already for the popups
         return false;
      if( b->state.moving() || b->state.copying())
         return false; // this is dragging element, skip it
      return b->c_style->floats == 0 && b->c_style->position == 0 &&
      !b->c_style->collapsed();*/
      return true;
    }
    NONCOPYABLE(find_row_pos_env_v)
  };

  bookmark block_table_body::find_text_pos(view &v, point zpos) {
    return super::find_text_pos(v, zpos);
  }

  element *block_table_body::find_child_element(
      view &v, point at /*at is this relative*/, bool exact) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_cols = cols.size();
    int n_rows = rows.size();

    if (n_cols == 0 || n_rows == 0) return 0;

    find_row_pos_env_v env(v, rows(), at);
    int                first_r = max(0, bsearch(env));
    int                last_y  = at.y + min(ld->dim.y, v.dimension().y);

    if (exact) {
      for (int r = first_r; r < n_rows; ++r) {
        row_def &rd = rows[r];
        if (rd.pos > last_y) 
          return nullptr;
        for (int c = 0; c < n_cols; ++c) {
          const cell_def &cd = rd.cells[c];
          if (!cd.b || cd.b == placeholder)
            goto HAS_CELL_SPANS;
          element *t = cd.b->find_element(v, at - cd.b->pos(), true);
          if (t) return t;
          // cd.b->draw(v,pg, cd.b->pos() + pos);
        }
        if (at.y >= rd.pos && at.y < rd.pos + rd.value) return rd.row;
      }
      return nullptr;
    HAS_CELL_SPANS:
      for (int r = 0; r < n_rows; ++r) {
        row_def &rd = rows[r];
        if (rd.pos > last_y) 
          return nullptr;
        for (int c = 0; c < n_cols; ++c) {
          const cell_def &cd = rd.cells[c];
          if (!cd.b || cd.b == placeholder)
            continue;
          element *t = cd.b->find_element(v, at - cd.b->pos(), true);
          if (t) 
            return t;
          // cd.b->draw(v,pg, cd.b->pos() + pos);
        }
        if (at.y >= rd.pos && at.y < rd.pos + rd.value) 
          return rd.row;
      }

    }
    else // closest
    {
      const style *cs = get_style(v);
      size inter_cell(pixels(v, this, cs->border_spacing_x, ld->dim).width(),
                      pixels(v, this, cs->border_spacing_y, ld->dim).width());
      for (int r = first_r; r < n_rows; ++r) {
        row_def &rd = rows[r];
        if (rd.pos > last_y) 
          return nullptr;
        for (int c = 0; c < n_cols; ++c) {
          const cell_def &cd = rd.cells[c];
          if (!cd.b || cd.b == placeholder) {
            goto HAS_CELL_SPANS_CLOSEST;
          }
          point ati  = at;
          rect  crc  = cd.b->border_box(v) + cd.b->pos();
          rect  crco = crc;
          crco >>= inter_cell;
          if (crco.contains(at)) {
            ati.x      = limit(ati.x, crc.s.x, crc.e.x);
            ati.y      = limit(ati.y, crc.s.y, crc.e.y);
            element *t = cd.b->find_element(v, ati - cd.b->pos(), true);
            if (t) 
              return t;
          }
        }
        // if( at.y >= rd.pos &&
        //    at.y < rd.pos + rd.value )
        //    return rd.row;
      }
      return nullptr;
    HAS_CELL_SPANS_CLOSEST:
      for (int r = 0; r < n_rows; ++r) {
        row_def &rd = rows[r];
        if (rd.pos > last_y) 
          return nullptr;
        for (int c = 0; c < n_cols; ++c) {
          const cell_def &cd = rd.cells[c];
          if (!cd.b || cd.b == placeholder) continue;
          point ati = at;
          rect  crc = cd.b->border_box(v) + cd.b->pos();
          rect  crco = crc;
          crco >>= inter_cell;
          if (crco.contains(at)) {
            ati.x = limit(ati.x, crc.s.x, crc.e.x);
            ati.y = limit(ati.y, crc.s.y, crc.e.y);
            element *t = cd.b->find_element(v, ati - cd.b->pos(), true);
            if (t)
              return t;
          }
        }
      }
    }

    return nullptr;
  }

  extern rect clip_rect(view &v, element *b);

  int find_first_row_to_draw(view &v, block_table_body *tbody, point pt) {
    handle<block_table_body::layout_data> ld =
        tbody->ldata.ptr_of<block_table_body::layout_data>();

    find_row_pos_env_v env(v, ld->rows(), pt - tbody->scroll_pos());
    int                first_r = max(0, bsearch(env));
    return first_r;
  }

  bool drawing_under_transforms(view &v, element *el);

  /*void block_table_row::draw(view& v, graphics* pg, point pos, bool
  check_animator = true)
  {
      block_table_body* tbody = parent_table_body();
      if(!tbody)
        return;

      handle<block_table_row::layout_data> rd_ldata =
  ldata.ptr_of<block_table_row::layout_data>();
      handle<block_table_body::layout_data> tbody_ldata =
  tbody->ldata.ptr_of<block_table_body::layout_data>();

      array<row_def>&   rows = tbody_ldata->rows;
      row_def& rd = rows[this->index()];

  }*/

  void block_table_row::draw_content(view &v, graphics *pg, point pos,
                                     bool clip) {
    // this method is not called in normal draw sequence
    // (block_table_body::draw_content draws cells) the only known case when it
    // gets called is when <tr> is airborn (Element.move() is called on it)

    block_table_body *tbody = parent_table_body();
    if (!tbody) return;
    if (!c_style->visible()) return;

    handle<block_table_row::layout_data> rd_ldata =
        ldata.ptr_of<block_table_row::layout_data>();

    handle<block_table_body::layout_data> tbody_ldata =
        tbody->ldata.ptr_of<block_table_body::layout_data>();

    array<col_def> &cols = tbody_ldata->cols;
    array<row_def> &rows = tbody_ldata->rows;

    row_def &rd = rows[this->index()];

    for (int c = 0; c < cols.size(); ++c) {
      const cell_def &cd = rd.cells[c];
      if (!cd.b || cd.b == placeholder) continue;
      hstyle ccs = cd.b->get_style(v);
      if (!ccs->display || ccs->visibility != visibility_visible) continue;
      if (ccs->position > position_static) continue;
      if (cd.b->state.popup()) continue;

      point cell_pos = cd.b->pos();

      cd.b->set_border_height(v, rd.row->dim().y);
      cell_pos.y = cd.b->border_distance(v).top();
      cd.b->draw(v, pg, cell_pos + pos);
    }
  }

  void block_table_body::draw_content(view &v, graphics *pg, point pos,
                                      bool clip) {
    hstyle              cs = get_style(v);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    rect client_place = client_rect(v);

    // size inner_dim = client_place.size();

    array<col_def> &cols = ld->cols;
    array<row_def> &rows = ld->rows;

    int n_cols = cols.size();
    int n_rows = rows.size();

    // if(n_cols == 0 || n_rows == 0)
    //  return;

    if (ld->zctx.is_empty() && (n_cols == 0 || n_rows == 0)) return;

    rect clip_area = clip_rect(v, this);
    if (clip_area.empty()) return;

    // rect vr(v.dimension());
    // rect vr(v.dimension());

    bool  clip_overflow = c_style->overflow_y > overflow_visible;
    point client_offset = client_place.s;
    clip_area += pos;

    rect rendering_rc = // pg->get_clip_rc();
        rect(v.client_dim()) | pg->get_clip_rc();
    // rendering_rc.origin = pg->screen_to_world(rendering_rc.origin);
    // rendering_rc.corner = pg->screen_to_world(rendering_rc.corner);
    rendering_rc &= clip_area;

    clipper _(pg, clip_area, clip_overflow, !c_style->has_solid_background());

    pos = scroll_translate(v, pos);
    pos += client_offset;

    ld->zctx.draw(v, pg, pos, this, false);

    //bool under_transforms = drawing_under_transforms(v, this);
    bool under_transforms = !pg->transform().is_identity(); //drawing_under_transforms(v, this);

    int first_r = under_transforms
                      ? 0
                      : find_first_row_to_draw(v, this, scroll_pos() - pos);

    bool has_selection = false;

    for (int r = first_r; r < n_rows; ++r) {
      row_def rd  = rows[r];
      if (!rd.row) // why is that? in some cases it is null, to investigate.
        continue; 
      hstyle   rcs = rd.row->get_style(v);
      if (out_of_flow(v, rd.row)) continue;

      if (!under_transforms && (rd.pos + pos.y > rendering_rc.bottom())) break;

      if (!rcs->visible()) continue;

      handle<block_table_row::layout_data> rd_ldata =
          rd.row->ldata.ptr_of<block_table_row::layout_data>();

      //if (rcs->has_background() || rd.row->flags.script_draws_background)
      rd.row->p_drawn_style = rd.row->d_style;
      rd.row->do_draw_background(v, pg, rd.row->pos() + pos,false);

      for (int c = 0; c < n_cols; ++c) {
        const cell_def &cd = rd.cells[c];
        if (!cd.b || cd.b == placeholder) continue;
        hstyle ccs = cd.b->get_style(v);
        if (!ccs->display || ccs->visibility != visibility_visible) continue;
        if (ccs->position > position_static) continue;
        if (cd.b->state.popup()) continue;
        cd.b->draw(v, pg, cd.b->pos() + pos);
      }

      //if (rcs->has_background() || rd.row->flags.script_draws_foreground || rd.row->behavior)
      rd.row->do_draw_foreground(v, pg, rd.row->pos() + pos, false);
      
      /*if (rcs->has_foreground() || rd.row->behavior) {
        rect r;
        switch (rcs->fore_image.clip) {
        case clip_content_box: r = rd_ldata->content_box(); break;
        case clip_padding_box: r = rd_ldata->padding_box(); break;
        case clip_margin_box: r = rd_ldata->margin_box(); break;
        default: r = rd_ldata->border_box(); break;
        }
        r += rd.row->pos();
        r += pos;
        rcs->draw_foreground(v, pg, r, rd.row);
      }*/

      if (rcs->has_outline()) {
        // rcs->draw_outline(v, pg, rd_ldata->border_box() + rd_ldata->pos +
        // pos, rd.row);
        pg->request_overlay();
      }

      if (v._selection_ctx && v._selection_ctx->is_valid() &&
          v._selection_ctx->caret.node == rd.row &&
          v._selection_ctx->get_selection_type(v) == ONLY_CARET) {
        has_selection = true;
      }
    }

    if (pg->has_overlays(_) || has_selection)
      draw_outlines(v, pg, pos, true, false);

    ld->zctx.draw(v, pg, pos, this, true);
  }

  void block_table_body::draw_outlines(view &v, graphics *sf, point pos,
                                       bool draw_children, bool draw_self,
                                       bool apply_transforms) {
    hstyle cs = get_style(v);

    if (draw_self && cs->has_outline()) draw_outline(v, sf, pos);

    if (draw_children) {

      handle<layout_data> ld = ldata.ptr_of<layout_data>();

      size inner_dim = client_rect(v).size();

      array<col_def> &cols = ld->cols;
      array<row_def> &rows = ld->rows;

      int n_cols = cols.size();
      int n_rows = rows.size();
      int y_max  = scroll_pos().y + inner_dim.y;

      int r = find_first_row_to_draw(v, this, scroll_pos());

      for (; r < n_rows; ++r) {
        row_def &rd  = rows[r];
        hstyle   rcs = rd.row->get_style(v);
        if (!rcs->display || rcs->visibility != visibility_visible) continue;
        if (rcs->position > position_static) continue;
        if (rd.row->state.popup()) continue;
        // rect rc = rd.row->rendering_box(v,TO_PARENT) + pos;
        if (rd.pos > y_max) break;

        point tp = rd.row->pos() + pos;
        if (rcs->has_outline()) rd.row->draw_outline(v, sf, tp);

        handle<block_table_row::layout_data> rd_ldata =
            rd.row->ldata.ptr_of<block_table_row::layout_data>();

        for (int c = 0; c < n_cols; ++c) {
          const cell_def &cd = rd.cells[c];
          if (!cd.b || cd.b == placeholder) continue;
          hstyle ccs = cd.b->get_style(v);
          if (!ccs->display || ccs->visibility != visibility_visible) continue;
          if (ccs->position > position_static) continue;
          if (cd.b->state.popup()) continue;
          point tp = cd.b->pos() + pos;
          if (ccs->overflow() < overflow_hidden)
            cd.b->draw_outlines(v, sf, tp, true, true);
          else
            cd.b->draw_outlines(v, sf, tp, false, true);
        }

        if (v._selection_ctx && v._selection_ctx->is_valid() &&
            v._selection_ctx->caret.node == rd.row &&
            v._selection_ctx->get_selection_type(v) == ONLY_CARET) {
          caret_metrics cm;
          cm.at_end = v._selection_ctx->caret.at_element_end();
          cm.elem   = rd.row;
          cm.ctype  = ROW_POSITION;

          int sz          = pixels(v, rd.row, rcs->font_size).height();
          cm.glyph_rc     = size(sz, sz);
          rect rowbox     = rd_ldata->border_box();
          rowbox.e.y = rowbox.s.y + rd.value;

          rowbox += rd.row->pos();
          rowbox += pos;

          if (cm.at_end)
            cm.glyph_rc.pointOf(3, rowbox.pointOf(3));
          else
            cm.glyph_rc.pointOf(7, rowbox.pointOf(7));

          v._selection_ctx->draw_caret(v, sf, cm);
        }
      }
    }
  }

  bool block_table_body::get_rows_cols(uint &rows, uint &cols) const {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    rows                   = (uint)ld->rows.length();
    cols                   = (uint)ld->cols.length();
    return true;
  }

  element *block_table_body::get_cell_at(uint r, uint c) const {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    if (r >= ld->rows.length()) return 0;
    if (c >= ld->cols.length()) return 0;
    element *cl = ld->rows[r].cells[c].b;
    return cl == placeholder ? 0 : cl;
  }

  element *block_table_body::get_actual_cell_at(uint r, uint c, irange &ar,irange &ac) const {
    element *cell = get_cell_at(r, c);
    uint     r1, r2, c1, c2;
    if (cell) {
      get_cell_rows_cols(cell, r1, r2, c1, c2);
      ar.l = int(r1);
      ar.h = int(r2);
      ac.l = int(c1);
      ac.h = int(c2);
      return cell;
    }
    for (int tr = int(r); tr >= 0; --tr) {
      for (int tc = int(c); tc >= 0; --tc) {
        element *cell = get_cell_at(uint(tr), uint(tc));
        if (!cell) continue;
        get_cell_rows_cols(cell, r1, r2, c1, c2);
        if (r >= r1 && r <= r2 && c >= c1 && c <= c2) {
          ar.l = int(r1);
          ar.h = int(r2);
          ac.l = int(c1);
          ac.h = int(c2);
          return cell;
        }
      }
    }
    ASSERT(false);
    return nullptr;
  }

  block_table_body *block_table_body::of(const element *cell) {
    if (!cell->is_table_cell()) return 0;
    element *owner = cell->get_owner();
    block_table_row *tr = owner->cast<block_table_row>();
    return tr->get_owner()->cast<block_table_body>();
  }

  element *get_cell_of(const block_table_body *tb, const node *n) {
    element *el   = const_cast<node *>(n)->get_element();
    element *cell = nullptr;
    while (el && el != tb) {
      if (el->is_table_cell()) cell = el;
      el = el->parent;
    }
    return cell;
  }

  bool block_table_body::get_cell_row_col(const node *incell, uint &r,
                                          uint &c) const {
    uint r2, c2;
    return get_cell_rows_cols(incell, r, r2, c, c2);
  }

  bool block_table_body::get_cell_rows_cols(const node *incell, uint &r1,
                                            uint &r2, uint &c1,
                                            uint &c2) const {
    element *cell = get_cell_of(this, incell);
    if (!cell) return false;
    if (!cell->is_table_cell()) return false;
    block_table_row *   tr = cell->get_owner()->cast<block_table_row>();
    if (tr->parent.ptr() != this) return false;
    uint r  = uint(tr->flags.ui_index);
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    assert(ld->rows[r].row == tr);
    if (cell->flags.ui_index < 0 || cell->flags.ui_index >= ld->cols.size())
      return false;
    if (ld->rows[r].row == tr) {
      uint c = uint(cell->flags.ui_index);
      if (ld->rows[r].cells[c].b == cell) {
        r1 = r; r2 = r + cell->atts.get_rowspan() - 1;
        c1 = c; c2 = c + cell->atts.get_colspan() - 1;
        return true;
      }
    }
    return false;
  }

  void block_table_body::on_vscrollbar_show(
      view &v) { // after scrollbar shown/hidden
#if 0
    v.post(delegate(this,&block_table_body::_on_vscrollbar_show,&v),true);
#else
    block_table *pt = parent_table();
    if (pt) {
      // pt->drop_minmax_dim();
      v.add_to_update(pt, CHANGES_DIMENSION);
      // pt->measure_inplace(*pv);
    }
#endif
  }

  bool block_table_body::_on_vscrollbar_show(
      view *pv) { // after scrollbar shown/hidden
#if 0
    block_table* pt = parent_table();
    if(pt) {
      pt->drop_minmax_dim();
      pv->add_to_update(pt,CHANGES_DIMENSION);
      //pt->measure_inplace(*pv);
    }
#endif
    return true;
  }

  int block_table_body::n_rows() {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->rows.size();
  }
  int block_table_body::n_cols() {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    return ld->cols.size();
  }

  void block_table_body::set_cell_at(int row, int col, element *cell) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    int                 rs = cell->atts.get_rowspan();
    int                 cs = cell->atts.get_colspan();
    ld->set_cell_at(row, col, cell, rs, cs);
  }

  void block_table_body::get_row(int row, array<handle<element>> &els) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    int                 nr = n_rows();
    if (row < 0 || row >= nr) return;
    const auto &rd = ld->rows[row];
    for (int c = 0; c < rd.cells.size(); ++c) {
      element *el = rd.cells[c].b;
      if (el != placeholder) els.push(el);
    }
  }

  void block_table_body::get_col(int col, array<handle<element>> &els) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    int                 nc = n_cols();
    if (col < 0 || col >= nc) return;
    for (int r = 0; r < ld->rows.size(); ++r) {
      const auto &rd = ld->rows[r];
      if (col >= rd.cells.size()) continue;
      element *el = rd.cells[col].b;
      if (el != placeholder) els.push(el);
    }
  }
  bool block_table_body::get_col_x(int col, gool::range &x) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    int                 nc = n_cols();
    if (col < 0 || col >= nc) return false;
    x = range(ld->cols[col].pos, ld->cols[col].pos + ld->cols[col].value - 1);
    return true;
  }
  bool block_table_body::get_row_y(int row, gool::range &y) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    int                 nr = n_rows();
    if (row < 0 || row >= nr) return false;
    y = range(ld->rows[row].pos, ld->rows[row].pos + ld->rows[row].value - 1);
    return true;
  }
  bool block_table_body::get_row_at(view &v, int y, int &row) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    if (!ld->rows.size()) return false;

    if (y < 0) {
      row = 0;
      return true;
    }

    point    pos(ld->dim.x / 2, y);
    element *child = find_child_element(v, pos, false);
    for (; child; child = child->get_owner()) {
      if (child->get_owner() == this) {
        row = int(child->flags.ui_index);
        return true;
      }
    }
    row = ld->rows.last_index();
    return true;
  }
  bool block_table_body::get_col_at(view &v, int x, int &col) {

    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    for (int c = 0; c < ld->cols.size(); ++c) {
      const auto &cd = ld->cols[c];
      if (x < cd.pos + cd.value) {
        col = c;
        return true;
      }
    }
    return false;
  }

  element *block_table_body::at(int row, int col) {
    return get_cell_at(uint(row), uint(col));
  }

} // namespace html
