#include "html.h"

#ifdef SVG_SUPPORT

namespace html {

  void render_children(view &v, graphics *pg, element *cont,
                       bool check_animator);

  element *svg_root(element *el, svg_root_data **svd) {
    for (; el; el = el->parent) {
      // if( el->is_svg_document() ) {
      if (el->is_of_type<block_svg>()) {
        if (svd) *svd = el->ldata.ptr_of<block_svg::layout_data>();
        return el;
      } else if (el->is_of_type<svg_document>()) {
        if (svd) *svd = el->ldata.ptr_of<svg_document::layout_data>();
        return el;
      }
      //}
    }
    return nullptr;
  }

  element *svg_root(element *el) { return svg_root(el, nullptr); }

  bool svg_positioned(const element *el) {
    if (!el->parent) return false;
    return el->parent->is_svg_element() || el->parent->is_svg_document();
  }

  block_svg *block_svg::setup_on(view &v, element *el) {
    block_svg *tb = turn_element_to<block_svg>(el);
    return tb;
  }

  void svg_root_data::calc_intrinsic_widths(view &v, element *self) {
    init(v, self);
  }

  int_v svg_root_data::auto_width(view &v, element *self) {
    init(v, self);
    //
    if(vbdim.x > 0)
      return int(v.pixels_per_dip(vbdim).x);
    return int_v();
  }

  int_v svg_root_data::auto_height(view &v, element *self) {
    init(v, self);
    if (vbdim.y > 0)
      return int(v.pixels_per_dip(vbdim).y);
    return int_v();
  }

  static bool parse_real(wchars &text, float &num) {
    while (!!text && (is_space(*text) || *text == ','))
      ++text;
    return tool::parse_real(text, num);
  }

  static bool parse_int(wchars &text, int &num) {
    while (!!text && (is_space(*text) || *text == ','))
      ++text;
    return tool::parse_int(text, num, 10U);
  }

  static bool parse_coord(wchars &text, pointf &pt) {
    if (parse_real(text, pt.x) && parse_real(text, pt.y)) return true;
    //++text;
    return false;
  }

  void svg_root_data::init(view &v, element *self) {
    self->ldata->dim_min.x = 0;
    self->ldata->dim_min.y = 0;
    self->ldata->dim_max.x = int(dimension.x);
    self->ldata->dim_max.y = int(dimension.y);

    if (self->p_drawn_style.is_identical(self->c_style) && !require_init)
      return;

    require_init = false;

    const style *cs = self->get_style(v);

    origin.x    = pixels(v, self, self->atts.get_size(attr::a_x, NUMBER_DIP)).width_f();
    origin.y    = pixels(v, self, self->atts.get_size(attr::a_y, NUMBER_DIP)).height_f();
    dimension.x = pixels(v, self, self->atts.get_size(attr::a_width, cs->width, NUMBER_DIP)).width_f();
    dimension.y = pixels(v, self, self->atts.get_size(attr::a_height, cs->height, NUMBER_DIP)).height_f();

    origin = v.ppx_to_px(origin);
    dimension = v.ppx_to_px(dimension);

    if (dimension.x <= 0) dimension.x = 100;
    if (dimension.y <= 0) dimension.y = 100;

    if (self->atts.exist(attr::a_viewbox)) {
      ustring vbstr = self->atts.get_ustring(attr::a_viewbox);
      wchars  vb    = vbstr();
      parse_real(vb, vbmin.x);
      parse_real(vb, vbmin.y);
      parse_real(vb, vbdim.x);
      parse_real(vb, vbdim.y);

      if (vbdim.x > 0 && vbdim.y > 0) {
        int placementFlags = 0;

        string aspect = self->atts.get_string("preserveaspectratio").to_lower();

        if (aspect().contains(CHARS("none"))) {
          placementFlags = rect_placement::stretchToFit;
        } else {
          if (aspect().contains(CHARS("slice")))
            placementFlags |= rect_placement::fillDestination;

          if (aspect().contains(CHARS("xmin")))
            placementFlags |= rect_placement::xLeft;
          else if (aspect().contains(CHARS("xmax")))
            placementFlags |= rect_placement::xRight;
          else
            placementFlags |= rect_placement::xMid;

          if (aspect().contains(CHARS("ymin")))
            placementFlags |= rect_placement::yTop;
          else if (aspect().contains(CHARS("ymax")))
            placementFlags |= rect_placement::yBottom;
          else
            placementFlags |= rect_placement::yMid;
        }

        drawing_placement = rect_placement(placementFlags);

        // ld->xform = rect_placement (placementFlags).get_transform_to_fit
        // (rectf(ld->vbmin,  ld->vbdim), rectf(ld->dimension));
      }
    } else if (!dimension.empty()) {
      vbdim = dimension;
    }

    if (self->is_of_type<svg_document>()) {
      html::each_child it(self);
      for (element *el; it(el);) {
        el->check_layout(v);
      }
    }
  }

  void svg_root_data::init_fragment(view &v, svg_document *self,
                                    const string &fragment) {
    if (self->p_drawn_style.is_identical(self->c_style) && !require_init)
      return;

    require_init = false;

    self->init(v);

    svg_document::layout_data *doc_layout_data =
        self->ldata.ptr_of<svg_document::layout_data>();

    *this = *doc_layout_data;

    element *viewel = self->find_by_id(fragment);
    if (!viewel) return;

    // origin.x    = self->atts.get_size(attr::a_x).pixels_width_f(v,self,0);
    // origin.y    = self->atts.get_size(attr::a_y).pixels_height_f(v,self,0);
    // dimension.x =
    // self->atts.get_size(attr::a_width).pixels_width_f(v,self,0);  dimension.y =
    // self->atts.get_size(attr::a_height).pixels_height_f(v,self,0);  if
    // (dimension.x  <= 0) dimension.x = 100;  if (dimension.y <= 0) dimension.y =
    // 100;

    if (viewel->atts.exist(attr::a_viewbox)) {
      ustring vbstr = viewel->atts.get_ustring(attr::a_viewbox);
      wchars  vb    = vbstr();
      parse_real(vb, vbmin.x);
      parse_real(vb, vbmin.y);
      parse_real(vb, vbdim.x);
      parse_real(vb, vbdim.y);

      if (vbdim.x > 0 && vbdim.y > 0) {
        int placementFlags = 0;

        string aspect = self->atts.get_string("preserveaspectratio").to_lower();

        if (aspect().contains(CHARS("none"))) {
          placementFlags = rect_placement::stretchToFit;
        } else {
          if (aspect().contains(CHARS("slice")))
            placementFlags |= rect_placement::fillDestination;

          if (aspect().contains(CHARS("xmin")))
            placementFlags |= rect_placement::xLeft;
          else if (aspect().contains(CHARS("xmax")))
            placementFlags |= rect_placement::xRight;
          else
            placementFlags |= rect_placement::xMid;

          if (aspect().contains(CHARS("ymin")))
            placementFlags |= rect_placement::yTop;
          else if (aspect().contains(CHARS("ymax")))
            placementFlags |= rect_placement::yBottom;
          else
            placementFlags |= rect_placement::yMid;
        }

        drawing_placement = rect_placement(placementFlags);

        // ld->xform = rect_placement (placementFlags).get_transform_to_fit
        // (rectf(ld->vbmin,  ld->vbdim), rectf(ld->dimension));
      }
    } else if (!dimension.empty()) {
      vbdim = dimension;
    }

    dimension.x =
        doc_layout_data->dimension.x * (vbdim.x / doc_layout_data->vbdim.x);
    dimension.y =
        doc_layout_data->dimension.y * (vbdim.y / doc_layout_data->vbdim.y);
  }

  void svg_root_data::draw_content(view &v, element *self, graphics *pg,
                                   point pos, bool clip) {
    gool::state _1(pg);

    gool::clipper _2(pg, rect(pos, self->ldata->dim), true);

    gool::affine_mtx_f mtx = drawing_placement.get_transform_to_fit(
        rectf(vbmin, vbdim), rectf(self->ldata->dim));
    mtx *= affine_translation_f(float(pos.x), float(pos.y));
    pg->transform(mtx);
    aa_mode aa(pg);
    render_children(v, pg, self, true);
  }

  point svg_root_data::translate(view &v, element *self, point pos) {
    gool::affine_mtx_f mtx = drawing_placement.get_transform_to_fit(
        rectf(vbmin, vbdim), rectf(self->ldata->dim));
    mtx.transform(pos);
    return pos;
  }
  point svg_root_data::inverse_translate(view &v, element *self, point pos) {
    gool::affine_mtx_f mtx = drawing_placement.get_transform_to_fit(
        rectf(vbmin, vbdim), rectf(self->ldata->dim));
    mtx.inverse_transform(pos);
    return pos;
  }

  // elements
  block_svg_element *block_svg_element::setup_on(view &v, element *el) {
    block_svg_element *tb = turn_element_to<block_svg_element>(el);
    return tb;
  }

  struct svg_color_stop {
    float_v offset;
    argb    color;
  };

  // returns percents_used
  bool gather_gradient_stops(view &v, array<svg_color_stop>& stops, element *gradel) {

    bool percents_used = false;

    // returns 0.0 ... 1.0
    auto get_offset = [&percents_used] (element *stopel) ->  float_v  {
      tool::ustring s = stopel->atts.get_ustring(attr::a_offset);
      tool::wchars  c = s;
      float         val = 0;
      if (!tool::parse_real(c, val)) return float_v();
      if (*c == '%') { val /= 100; percents_used = true; }
      return limit(val, 0.0f, 1.0f);
    };

    html::each_child it(gradel);
    for (element *el; it(el);) {
      if (el->tag != tag::T_STOP) continue;

      const style *cs = el->get_style(v);

      svg_color_stop cst;

      cst.offset = get_offset(el);
      cst.color = cs->stroke_color.to_argb().opacity(cs->stroke_opacity);

      stops.push(cst);
    }

    return percents_used;
  }

  gool::brush *block_svg_element::get_brush(view &v, const ustring &id,
                                            opacity_v opacity) {
    svg_root_data *prd  = nullptr;
    element *      root = svg_root(this, &prd);
    if (!root) return nullptr;

    helement gradel = root->get_element_by_id(id);
    if (!gradel) return nullptr;

    bool is_radial = false;

    if (gradel->tag == tag::T_LINEARGRADIENT)
      ;
    else if (gradel->tag == tag::T_RADIALGRADIENT)
      is_radial = true;
    else
      return nullptr;

    sizef  gradient_dim = prd->viewbox().size();
    pointf gradient_org = pointf(0.0f, 0.0f);

    const bool user_space =
        lexical::ci::eq(gradel->atts.get_ustring(attr::a_gradientunits)(),
                        WCHARS("userspaceonuse"));

    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    if (!user_space) {
      rectf bounds = get_bounds(v);
      if (!bounds.empty()) {
        gradient_org = bounds.start();
        gradient_dim = bounds.size();
      }
    }

    array<svg_color_stop> stops;

    bool percents_used = false;

    do {
      ustring buddyid = gradel->atts.get_ustring("xlink:href");
      if (!buddyid.is_empty() && buddyid[0] == '#') {
        element *buddy = root->get_element_by_id(buddyid().sub(1));
        if (buddy) {
          gather_gradient_stops(v, stops, buddy);
          break;
        }
      }
      // percents_used = ????
      gather_gradient_stops(v, stops, gradel);
    } while (0);

    const style *cs = gradel->get_style(v);

    gradient_brush *grad = nullptr;

    if (is_radial) {

      static size_v p50 = size_v(50, size_v::unit_type::pr);
      pointf        c;
      sizef         r, f;

      c.x = pixels(v, this, gradel->atts.get_size(attr::a_cx, p50, user_space ? NUMBER_NUMBER : NUMBER_PERCENT),gradient_dim).width_f();
      c.y = pixels(v, this, gradel->atts.get_size(attr::a_cy, p50, user_space ? NUMBER_NUMBER : NUMBER_PERCENT),gradient_dim).height_f();
      c += gradient_org;
      r.x = r.y = pixels(v, this, gradel->atts.get_size(attr::a_r, p50, user_space ? NUMBER_NUMBER : NUMBER_PERCENT),gradient_dim).width_f();

#pragma TODO("handle gradient focal point 'f' if needed")
      grad = new radial_brush(c, r, f);
    } else {
      static size_v p0   = size_v(0, size_v::unit_type::pr);
      static size_v p100 = size_v(100, size_v::unit_type::pr);

      pointf s, e;

      s.x = pixels(v, this, gradel->atts.get_size(attr::a_x1, p0, user_space ? NUMBER_NUMBER : NUMBER_PERCENT), gradient_dim).width_f();
      s.y = pixels(v, this, gradel->atts.get_size(attr::a_y1, p0, user_space ? NUMBER_NUMBER : NUMBER_PERCENT), gradient_dim).height_f();
      s += gradient_org;
      e.x = pixels(v, this, gradel->atts.get_size(attr::a_x2, p100, user_space ? NUMBER_NUMBER : NUMBER_PERCENT), gradient_dim).width_f();
      e.y = pixels(v, this, gradel->atts.get_size(attr::a_y2, p0, user_space ? NUMBER_NUMBER : NUMBER_PERCENT), gradient_dim).height_f();
      e += gradient_org;

      assert(s != e);

      grad = new linear_brush(s, e);
    }

    if (cs->transforms.is_defined()) {

      affine_mtx_f mtx;
      cs->transforms->apply(v, gradel, mtx);

      if (is_radial)
        grad->transform(mtx);
      else {
        linear_brush *lb = static_cast<linear_brush *>(grad);
        // transform the perpendicular vector into the new coordinate space for
        // the gradient. this vector is now the slope of the linear gradient as
        // it should appear in the new coord space
        affine_mtx_f amtx = mtx.absolute_translation(0, 0);
        pointf       perpendicular = pointf(lb->end.y - lb->start.y, lb->start.x - lb->end.x);
        amtx.transform(perpendicular);

        pointf new_start = lb->start;
        mtx.transform(new_start);
        pointf new_end = lb->end;
        mtx.transform(new_end);

        // project the transformed gradient vector onto the transformed slope of
        // the linear gradient as it should appear in the new coordinate space
        float scale = perpendicular.dot(new_end - new_start) /
                      perpendicular.dot(perpendicular);

        lb->start = new_start;
        lb->end   = new_end - perpendicular * scale;
      }
    }

    for (auto& cst : stops)
      grad->add_stop(cst.offset, cst.color);

    return grad;
  }

  byte inherit_opacity(uint p, uint t) { return byte((p * t) >> 8); }

  void block_svg_element::init(view &v) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    const style *cs = get_style(v);

    if (this->p_drawn_style.is_identical(cs) && !ld->require_init)
      return;

#ifdef DEBUG
    if (is_id_test())
      cs = cs;
#endif

    p_drawn_style = d_style;
    ld->require_init = false;

    if (cs->transforms.is_defined()) {
      affine_mtx_f mtx;
      cs->transforms->apply(v, this, mtx);
      ld->xform = mtx;
    }

    svg_root_data *prd  = nullptr;
    element *      root = svg_root(this, &prd);

    if (!root) return;

    switch (this->tag) {
    default: return;
    case tag::T_G: init_g(v, this, this->atts, prd, root, ld); break;
    case tag::T_PATH: init_path(v, this, this->atts, prd, root, ld); break;
    case tag::T_RECT: init_rect(v, this, this->atts, prd, root, ld); break;
    case tag::T_CIRCLE: init_circle(v, this, this->atts, prd, root, ld); break;
    case tag::T_ELLIPSE:
      init_ellipse(v, this, this->atts, prd, root, ld);
      break;
    case tag::T_LINE: init_line(v, this, this->atts, prd, root, ld); break;
    case tag::T_POLYLINE:
      init_polyline(v, false, this, this->atts, prd, root, ld);
      break;
    case tag::T_POLYGON:
      init_polyline(v, true, this, this->atts, prd, root, ld);
      break;
    case tag::T_SWITCH: init_switch(v, this, this->atts, prd, root, ld); break;
    case tag::T_USE:
      if (root->is_svg_document())
        init_use(v, root->cast<svg_document>(), this, this->atts, prd, root,ld); break;
      // case tag::T_LINEARGRADIENT:
      // case tag::T_RADIALGRADIENT: init_brush(v); return;
      break;
    }

    on_style_changed(v, root->doc());
  }

  bool parse_comma_separated_lengths(value list, array<float> &out) {
    if(list.is_array())
      out = list.get_array<float>();
    return true;
  }

  void block_svg_element::on_style_changed(view &v, document *pd) {
    super::on_style_changed(v, pd);
   
    if (!this->is_of_type<block_svg_element>())
      return; // element has chaged its class

    const style *cs = c_style;

#ifdef _DEBUG
    if (tag == tag::T_USE) cs = cs;
    if (tag == tag::T_LINE) cs = cs;
    //this->dbg_report("block_svg_element::on_style_changed");
#endif

    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    //WTF? ld->require_init = true; will cause recursive call - SO

    ld->bounds = rectf();

    if (ld->path) {
      ld->bounds = ld->path->bounds();
      ld->bounds >>= ld->stroke_width;
    }
    if (cs->overflow() <= overflow_visible) {
      html::each_child it(this);
      for (element *el; it(el);) {
        if (!el->is_of_type<block_svg_element>()) continue;
        block_svg_element *sel = static_cast<block_svg_element *>(el);
        rectf              cbounds = sel->get_bounds(v);
        ld->bounds |= cbounds;
      }
    }
    
    if (cs->transforms.is_defined()) {
      affine_mtx_f mtx;
      cs->transforms->apply(v, this, mtx);
      ld->xform = mtx;
    } else
      ld->xform = affine_mtx_f();

    ld->fill = nullptr;

    if (cs->fill_id == CHARS("none"))
      ;
    else if (cs->fill_id.is_defined() && (cs->fill_id != CHARS("inherit"))) {
      ld->fill = get_brush(v, cs->fill_id, cs->fill_opacity);
    }
    if (!ld->fill && cs->fill_color.is_defined()) {

      argb back = cs->fill_color.to_argb().opacity(cs->fill_opacity);
      // WRONG!: if( back.alfa )
      ld->fill = new gool::solid_brush(back);
    }

    argb fore(0, 0, 0, 0);

    ld->stroke = nullptr;

    if (cs->stroke_id == CHARS("none"))
      ;
    else if (cs->stroke_id.is_defined() && (cs->stroke_id != CHARS("inherit"))) {
      ld->stroke = get_brush(v, cs->stroke_id, cs->stroke_opacity);
    }
    if (!ld->stroke && cs->stroke_color.is_defined()) {
      fore       = cs->stroke_color.to_argb().opacity(cs->stroke_opacity.val(255));
      ld->stroke = new gool::solid_brush(fore);
    }

    if (fore.alfa || ld->stroke) {
      ld->stroke_width = cs->stroke_width.is_defined() ? pixels(v, this, cs->stroke_width).width_f() : 1.0f;
      ld->stroke_cap        = gool::CAP_STYLE(cs->stroke_linecap.val());
      ld->stroke_join       = gool::LINE_JOIN(cs->stroke_linejoin.val());
      ld->stroke_miterlimit = cs->stroke_miterlimit.val(4);

      if (cs->stroke_dasharray.is_defined()) {
        parse_comma_separated_lengths(cs->stroke_dasharray,ld->stroke_dasharray);
        if ((ld->stroke_dasharray.size() % 2) == 1)
          ld->stroke_dasharray.push(ld->stroke_dasharray.last());
      }
      ld->stroke_dashoffset =
          cs->stroke_dashoffset.is_defined()
              ? pixels(v, this, cs->stroke_dashoffset).width_f()
              : 0.0f;
    }

    if (!ld->fill && !ld->stroke) {
      // svg_document* pd = this->doc()->cast<svg_document>();
      color fc = cs->font_color.to_argb();
      // if( pd->default_style && pd->default_style->fill_color.is_defined() )
      //  fc = pd->default_style->fill_color;
      ld->fill = new gool::solid_brush(argb(fc));
    }

  }

  rectf block_svg_element::get_bounds(view &v) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    init(v);
    return ld->bounds;
  }

#if 0
  void block_svg_element::fixup_style(view &v, document *pd, style &s) {
    super::fixup_style(v, pd, s);

    if (tag != tag::T_USE) return;

    ustring buddyid = this->atts.get_ustring("xlink:href");
    if (buddyid.is_empty()) return;
    if (buddyid[0] != '#') return;

    svg_root_data *prd  = nullptr;
    element *      root = svg_root(this, &prd);

    if (!root) return;

    helement buddy = root->get_element_by_id(buddyid.substr(1));
    if (!buddy) return;

    s.resolve(v,*buddy->get_style(v));

    /*if( buddy->tag == tag::T_TEXT && !this->nodes.is_empty())
    {
      if( s.fill_color.is_defined() )
        s.font_color = s.fill_color;
    }*/
  }
#endif

  void block_svg_element::init_use(view &v, svg_document *pd, element *that,
                                   const attribute_bag &atts,
                                   svg_root_data *prd, element *root,
                                   handle<layout_data> ld) {
    ustring buddyid = this->atts.get_ustring("xlink:href");
    if (buddyid.is_empty()) return;
    if (buddyid[0] != '#') return;
    helement buddy = pd->get_element_by_id(buddyid().sub(1));
    if (!buddy) return;

    attribute_bag catts = this->atts;
    catts.inherit(buddy->atts);

    switch (buddy->tag) {
    default: return;
    case tag::T_G: init_g(v, this, catts, prd, root, ld); break;
    case tag::T_PATH: init_path(v, this, catts, prd, root, ld); break;
    case tag::T_RECT: init_rect(v, this, catts, prd, root, ld); break;
    case tag::T_CIRCLE: init_circle(v, this, catts, prd, root, ld); break;
    case tag::T_ELLIPSE: init_ellipse(v, this, catts, prd, root, ld); break;
    case tag::T_LINE: init_line(v, this, catts, prd, root, ld); break;
    case tag::T_POLYLINE:
      init_polyline(v, false, this, catts, prd, root, ld);
      break;
    case tag::T_POLYGON:
      init_polyline(v, true, this, catts, prd, root, ld);
      break;
    case tag::T_SWITCH:
      init_switch(v, this, catts, prd, root, ld);
      break;
      // NO recursive <use>: case tag::T_USE:      init_use( v, this, catts,
      // prd,root,ld ); break;

      // case tag::T_TEXT:     if( this->nodes.is_empty() )
      // this->append(buddy->clone(),&v); break;

      // case tag::T_LINEARGRADIENT:
      // case tag::T_RADIALGRADIENT: init_brush(v); return;
    }
    on_style_changed(v, pd);
  }

  /*rectf block_svg::get_bounds(view& v) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    if( ld->require_init )
      init(v);
    return rectf(ld->vbdim);
  }*/

  void block_svg_element::init_g(view &v, element *that,
                                 const attribute_bag &atts, svg_root_data *prd,
                                 element *root, handle<layout_data> ld) {
    // nothing
  }

  bool parse_d_path(gool::path* path, wchars d) {

    pointf subpath_start, last, last2, p1, p2, p3;
    wchar  last_command_char = 0;
    bool   is_relative = true;

    const wchars command_chars = WCHARS("MmLlHhVvCcSsQqTtAaZz");

    while (!!d) {
      while (!!d && ::is_space(*d))
        ++d;
      if (command_chars.index_of(*d) >= 0) {
        last_command_char = d++;
        is_relative = (last_command_char >= 'a' && last_command_char <= 'z');
      }

      switch (last_command_char) {
      case 'M':
      case 'm':
      case 'L':
      case 'l':
        if (parse_coord(d, p1)) {
          if (is_relative) p1 += last;

          if (last_command_char == 'M' || last_command_char == 'm') {
            // if( path-> )
            path->move_to(subpath_start = p1);
            last_command_char = 'l';
          }
          else
            path->line_to(p1);

          last2 = last = p1;
        }
        else
          return false;

        break;

      case 'H':
      case 'h':
        if (parse_real(d, p1.x)) {
          if (is_relative) p1.x += last.x;

          path->line_to(pointf(p1.x, last.y));

          last2.x = last.x = p1.x;
        }
        else
          return false;
        break;

      case 'V':
      case 'v':
        if (parse_real(d, p1.y)) {
          if (is_relative) p1.y += last.y;

          path->line_to(pointf(last.x, p1.y));

          last2.y = last.y = p1.y;
        }
        else
          return false;
        break;

      case 'C':
      case 'c':
        if (parse_coord(d, p1) && parse_coord(d, p2) && parse_coord(d, p3)) {
          if (is_relative) {
            p1 += last;
            p2 += last;
            p3 += last;
          }

          path->cubic_to(p3, p1, p2);

          last2 = p2;
          last = p3;
        }
        else
          return false;
        break;

      case 'S':
      case 's':
        if (parse_coord(d, p1) && parse_coord(d, p3)) {
          if (is_relative) {
            p1 += last;
            p3 += last;
          }

          p2 = last + (last - last2);
          path->cubic_to(p3, p2, p1);

          last2 = p1;
          last = p3;
        }
        else
          return false;
        break;
      case 'Q':
      case 'q':
        if (parse_coord(d, p1) && parse_coord(d, p2)) {
          if (is_relative) {
            p1 += last;
            p2 += last;
          }

          path->quadratic_to(p2, p1);

          last2 = p1;
          last = p2;
        }
        else
          return false;
        break;

      case 'T':
      case 't':
        if (parse_coord(d, p1)) {
          if (is_relative) p1 += last;

          p2 = last + (last - last2);
          path->quadratic_to(p1, p2);

          last2 = p2;
          last = p1;
        }
        else
          return false;
        break;

      case 'A':
      case 'a':
        if (parse_coord(d, p1)) {
          float num;
          int   inum;

          if (parse_real(d, num)) {
            float angle = num / 180.0f * real_traits<float>::pi();

            if (parse_int(d, inum)) {
              bool largeArc = inum != 0;

              if (parse_int(d, inum)) {
                bool sweep = inum != 0;

                if (parse_coord(d, p2)) {
                  if (is_relative) p2 += last;

                  if (last != p2)
                    path->arc_to(p2, p1, angle, largeArc, sweep);

                  last2 = last;
                  last = p2;
                }
              }
            }
          }
        }
        else
          return false;

        break;

      case 'Z':
      case 'z':
        path->close();
        last = last2 = subpath_start;

        while (!!d && ::is_space(*d))
          ++d;
        last_command_char = 'M';
        break;

      default: 
        return false;
      }
    }
    return true;
  }

  bool parse_d_path(gool::path* path, value v) {

    pointf subpath_start, last, last2, p1, p2, p3;
    wchar  last_command_char = 0;
    bool   is_relative = true;

    if (!v.is_function(WCHARS("path")))
      return false;
   
    array<value> elements;

    flatten(v, [&](const value& v) { elements.push(v); });

    slice<value> d = elements();

    auto get_number = [&](float& v) -> bool {
      if (!d->is_number()) return false;
      v = d->get_float();
      ++d;
      return true;
    };

    auto get_int = [&](int& v) -> bool {
      if (!d->is_int()) return false;
      v = d->get_int();
      ++d;
      return true;
    };

    auto get_coord = [&](pointf& v) -> bool {
      return get_number(v.x) && get_number(v.y);
    };

    const wchars command_chars = WCHARS("MmLlHhVvCcSsQqTtAaZz");

    while (!!d) {
      if (!d->is_string())
        return false;
      ustring name = d->to_string();
      if (name.length() != 1)
        return false;

      if ( command_chars.index_of(name[0]) >= 0) {
        last_command_char = name[0];
        ++d;
        is_relative = (last_command_char >= 'a' && last_command_char <= 'z');
      }

      switch (last_command_char) {
      case 'M':
      case 'm':
      case 'L':
      case 'l':
        if (get_coord(p1)) {
          if (is_relative) p1 += last;

          if (last_command_char == 'M' || last_command_char == 'm') {
            // if( path-> )
            path->move_to(subpath_start = p1);
            last_command_char = 'l';
          }
          else
            path->line_to(p1);

          last2 = last = p1;
        }
        else
          return false;

        break;

      case 'H':
      case 'h':
        if (get_number(p1.x)) {
          if (is_relative) p1.x += last.x;

          path->line_to(pointf(p1.x, last.y));

          last2.x = last.x = p1.x;
        }
        else
          return false;
        break;

      case 'V':
      case 'v':
        if (get_number(p1.y)) {
          if (is_relative) p1.y += last.y;

          path->line_to(pointf(last.x, p1.y));

          last2.y = last.y = p1.y;
        }
        else
          return false;
        break;

      case 'C':
      case 'c':
        if (get_coord(p1) && get_coord(p2) && get_coord(p3)) {
          if (is_relative) {
            p1 += last;
            p2 += last;
            p3 += last;
          }

          path->cubic_to(p3, p1, p2);

          last2 = p2;
          last = p3;
        }
        else
          return false;
        break;

      case 'S':
      case 's':
        if (get_coord(p1) && get_coord(p3)) {
          if (is_relative) {
            p1 += last;
            p3 += last;
          }

          p2 = last + (last - last2);
          path->cubic_to(p3, p2, p1);

          last2 = p1;
          last = p3;
        }
        else
          return false;
        break;
      case 'Q':
      case 'q':
        if (get_coord(p1) && get_coord(p2)) {
          if (is_relative) {
            p1 += last;
            p2 += last;
          }

          path->quadratic_to(p2, p1);

          last2 = p1;
          last = p2;
        }
        else
          return false;
        break;

      case 'T':
      case 't':
        if (get_coord(p1)) {
          if (is_relative) p1 += last;

          p2 = last + (last - last2);
          path->quadratic_to(p1, p2);

          last2 = p2;
          last = p1;
        }
        else
          return false;
        break;

      case 'A':
      case 'a':
        if (get_coord(p1)) {
          float num;
          int   inum;

          if (get_number(num)) {
            float angle = num / 180.0f * real_traits<float>::pi();

            if (get_int(inum)) {
              bool largeArc = inum != 0;

              if (get_int(inum)) {
                bool sweep = inum != 0;

                if (get_coord(p2)) {
                  if (is_relative) p2 += last;

                  if (last != p2)
                    path->arc_to(p2, p1, angle, largeArc, sweep);

                  last2 = last;
                  last = p2;
                }
              }
            }
          }
        }
        else
          return false;

        break;

      case 'Z':
      case 'z':
        path->close();
        last = last2 = subpath_start;

        //while (!!d && ::is_space(*d))
        //  ++d;
        last_command_char = 'M';
        break;

      default:
        return false;
      }
    }
    return true;
  }



  void block_svg_element::init_path(view &v, element *that,
                                    const attribute_bag &atts,
                                    svg_root_data *prd, element *root,
                                    handle<layout_data> ld) {
    const style *cs = get_style(v);

    ld->path = v.create_path();
    ld->path->reset();
    ld->path->set_even_odd(cs->fill_rule == fill_rule_evenodd);

    ustring sd = atts.get_ustring(attr::a_d);

    parse_d_path(ld->path, sd());
  }

  void block_svg_element::init_rect(view &v, element *that,
                                    const attribute_bag &atts,
                                    svg_root_data *prd, element *root,
                                    handle<layout_data> ld) {
    sizef base = prd->viewbox().size();

    ld->path = v.create_path();

    float x = pixels(v, that, atts.get_size(attr::a_x),base).width_f();
    float y = pixels(v, that, atts.get_size(attr::a_y),base).height_f();
    float w = pixels(v, that, atts.get_size(attr::a_width), base).width_f();
    float h = pixels(v, that, atts.get_size(attr::a_height),base).height_f();

    size_v rx = atts.get_size(attr::a_rx);
    size_v ry = atts.get_size(attr::a_ry);

    pointf pos = pointf(x, y);
    sizef  dim = sizef(w, h);

    if (rx.is_defined() || ry.is_defined()) {
      if (!rx.is_defined())
        rx = ry;
      else if (!ry.is_defined())
        ry = rx;

      sizef tl, tr, br, bl;
      tl = tr = br = bl = sizef(pixels(v, that, rx,base).width_f(),
                                pixels(v, that, rx,base).height_f());

      ld->path->add_round_rect(pos, dim, tl, tr, br, bl);
    } else {
      ld->path->add_rect(pos, dim);
    }
  }
  void block_svg_element::init_circle(view &v, element *that,
                                      const attribute_bag &atts,
                                      svg_root_data *prd, element *root,
                                      handle<layout_data> ld) {
    sizef base = prd->viewbox().size();

    ld->path = v.create_path();

    pointf c;

    c.x = pixels(v, that, atts.get_size(attr::a_cx, size_v(0)),base).width_f();
    c.y = pixels(v, that, atts.get_size(attr::a_cy, size_v(0)),base).height_f();

    sizef r;
    r.x = r.y = pixels(v, that, atts.get_size(attr::a_r), base).width_f();

    pointf p1 = c;
    p1.x -= r.x;
    pointf p2 = c;
    p2.x += r.x;
    ld->path->start(p1, true);
    ld->path->arc_to(p2, r, 0, false, true);
    ld->path->arc_to(p1, r, 0, false, true);
    ld->path->close();
  }
  void block_svg_element::init_ellipse(view &v, element *that,
                                       const attribute_bag &atts,
                                       svg_root_data *prd, element *root,
                                       handle<layout_data> ld) {
    sizef base = prd->viewbox().size();

    ld->path = v.create_path();

    pointf c;

    c.x = pixels(v, that, atts.get_size(attr::a_cx), base).width_f();
    c.y = pixels(v, that, atts.get_size(attr::a_cy), base).height_f();

    sizef r;
    r.x = pixels(v, that, atts.get_size(attr::a_rx),base).width_f();
    r.y = pixels(v, that, atts.get_size(attr::a_ry),base).width_f();

    pointf p1 = c;
    p1.x -= r.x;
    pointf p2 = c;
    p2.x += r.x;
    ld->path->start(p1, true);
    ld->path->arc_to(p2, r, 0, false, true);
    ld->path->arc_to(p1, r, 0, false, true);
    ld->path->close();
  }
  void block_svg_element::init_line(view &v, element *that,
                                    const attribute_bag &atts,
                                    svg_root_data *prd, element *root,
                                    handle<layout_data> ld) {
    sizef base = prd->viewbox().size();

    ld->path = v.create_path();
    ld->path->reset();

    pointf s, e;
    s.x = pixels(v, that, atts.get_size(attr::a_x1),base).width_f();
    s.y = pixels(v, that, atts.get_size(attr::a_y1),base).height_f();
    e.x = pixels(v, that, atts.get_size(attr::a_x2),base).width_f();
    e.y = pixels(v, that, atts.get_size(attr::a_y2),base).height_f();

    ld->path->start(s, false);
    ld->path->line_to(e);
  }
  void block_svg_element::init_polyline(view &v, bool closed, element *that,
                                        const attribute_bag &atts,
                                        svg_root_data *prd, element *root,
                                        handle<layout_data> ld) {
    const style *cs   = get_style(v);
    sizef        base = prd->viewbox().size();

    ustring s_points = atts.get_ustring(attr::a_points);
    wchars  text     = s_points;

    ld->path = v.create_path();
    ld->path->reset();
    ld->path->set_even_odd(cs->fill_rule == fill_rule_evenodd);

    auto parse_point = [&](pointf &pt) -> bool {
      size_v t;
      while (!!text && (::is_space(*text) || *text == ','))
        ++text;
      t = parse_size_v(text);
      if (t.is_undefined()) return false;
      pt.x = pixels(v, that, t, base).width_f();
      while (!!text && (::is_space(*text) || *text == ','))
        ++text;
      t = parse_size_v(text);
      if (t.is_undefined()) return false;
      pt.y = pixels(v, that, t, base).height_f();
      return true;
    };
    pointf first;
    if (parse_point(first)) {
      pointf p, last;

      ld->path->start(first, true);
      while (parse_point(p))
        ld->path->line_to(last = p);

      if (closed || first == last) ld->path->close();
    }
  }
  void block_svg_element::init_switch(view &v, element *that,
                                      const attribute_bag &atts,
                                      svg_root_data *prd, element *root,
                                      handle<layout_data> ld) {
    // nothing
  }

  bool block_svg_element::on_set_attr(uint att_sym, const ustring &val) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->require_init       = true;

    //svg_root_data *prd = nullptr;
    //element *      root = svg_root(this, &prd);
    //if (root) 

    return super::on_set_attr(att_sym, val);
  }

  bool block_svg_element::on_remove_attr(uint att_sym, const ustring &val) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->require_init       = true;
    return super::on_remove_attr(att_sym,val);
  }

  bool svg_document::on_set_attr(uint att_sym, const ustring &val) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->require_init       = true;
    return super::on_set_attr(att_sym, val);
  }

  bool svg_document::on_remove_attr(uint att_sym, const ustring &val) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->require_init       = true;
    return super::on_remove_attr(att_sym,val);
  }

  bool block_svg::on_set_attr(uint att_sym, const ustring &val) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->require_init       = true;
    return super::on_set_attr(att_sym, val);
  }

  bool block_svg::on_remove_attr(uint att_sym, const ustring &val) {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->require_init       = true;
    return super::on_remove_attr(att_sym,val);
  }

  void block_svg_element::render(view &v, graphics *pg, bool check_animator) {
#ifdef _DEBUG
    if (tag == tag::T_PATH) 
      pg = pg;
    if (tag == tag::T_CIRCLE) pg = pg;
#endif

    if (!is_it_visible(v)) return;

    init(v);

    p_drawn_style = d_style;

    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    const style *       cs = get_style(v);
    gool::state         _(pg);
    if (!ld->xform.is_identity()) pg->transform(ld->xform);

    bool has_layer = false;

    if (cs->opacity < 255) {
      pg->push_layer(ld->bounds, byte(cs->opacity));
      has_layer = true;
    }

    if (ld->path) {

      pg->set_fill(ld->fill);
      pg->set_stroke(ld->stroke, ld->stroke_width, ld->stroke_cap,
                     ld->stroke_join, ld->stroke_miterlimit);
      if (ld->stroke_dasharray.length())
        pg->custom_dash_style(ld->stroke_dasharray(), ld->stroke_dashoffset);
      pg->draw_path(ld->path);
    }
    render_children(v, pg, this, check_animator);

    if (has_layer) pg->pop_layer();
  }

  int vertical_offset(view &v, element *el) {
    switch (el->get_style(v)->get_text_vertical_align()) {
    case valign_sub:
    case valign_super:
    case valign_auto:
    case valign_baseline: {
      int y = 0, h = 0, a = 0;
      if (el->get_last_line_metrics(v, y, h, a)) {
        rect od = el->outer_distance(v);
        return y + a + od.top();
      }
      else {
        rect md = el->ldata->margin_width;
        return el->min_height(v) + el->borpad_int_y_extra(v) + md.top();
      }
    } break;
    case valign_text_top:
    case valign_top: {
      rect od = el->outer_distance(v);
      return od.top();
    }
    case valign_middle:
      return (el->min_height(v) + el->outer_int_y_extra(v)) / 2;
    case valign_bottom:
    case valign_text_bottom:
      return el->min_height(v) + el->outer_int_y_extra(v);
    }
    return 0;
  }
  
  inline bool is_rtl(view &v, element *el) { return el->get_style(v)->direction == direction_rtl; }

  void render_children(view &v, graphics *pg, element *cont,
                       bool check_animator) {
    if (cont->tag == tag::T_SWITCH) {
      html::each_child it(cont);
      for (element *el; it(el);) {
        if (el->tag != tag::T_G) continue;
        block_svg_element *svg_el = static_cast<block_svg_element *>(el);
        svg_el->render(v, pg);
        break;
      }
      return;
    }

    each_element it(cont);
    for (element *ch; it(ch);) {
      // if( !ch->is_visible(v) )
      //  continue;
      if (ch->tag == tag::T_DEFS) {
        it.dont_go_inside = true;
        continue;
      } else if (ch->is_of_type<block_svg_element>()) {
        it.dont_go_inside         = true;
        block_svg_element *svg_el = static_cast<block_svg_element *>(ch);
        svg_el->render(v, pg);
      } else if (ch->is_box_element(v)) {
        it.dont_go_inside = true;
        point pt;
        pt.x = pixels(v, ch, ch->atts.get_size(attr::a_x)).width();
        pt.y = pixels(v, ch, ch->atts.get_size(attr::a_y)).height();
        // float w = ch->atts.get_size(attr::a_width).pixels_width_f(v,ch, 0);
        // float h = ch->atts.get_size(attr::a_height).pixels_height_f(v,ch, 0);
        if (!ch->is_layout_valid()) {
          ch->measure_inplace(v);
          ch->set_width(v, ch->computed_width(v, 100));
          ch->set_height(v, ch->computed_height(v, 100));
        }

        handle<layout_data> ld = ch->ldata;
        const style *       cs = ch->get_style(v);
        gool::state _(pg);

        if (ch->tag == tag::T_TEXT) {
          auto ta = ch->atts.get_enum(attr::a_text_anchor, enum_pdef<text_anchor_e>()->items);
          switch (ta) {
          default:
            case text_anchor_start:   if (is_rtl(v, ch))  pt.x -= ld->dim.x; break;
            case text_anchor_middle:  pt.x -= ld->dim.x / 2; break;
            case text_anchor_end:     if (!is_rtl(v, ch))  pt.x -= ld->dim.x; break;
          }
          pt.y -= vertical_offset(v, ch);
        }

        ch->set_pos(pt);

        if (cs->transforms) {
          affine_mtx_f mtx;
          cs->transforms->apply(v, ch, mtx);
          pg->transform(mtx);
        }

        bool has_layer = false;

        if (cs->opacity < 255) {
          pg->push_layer(ld->dim, byte(cs->opacity));
          has_layer = true;
        }
        ch->do_draw(v, pg, pt);
        if (has_layer) pg->pop_layer();
      }
    }
  }

  point block_svg_element::translate(
      view &v, point pos) // translate - from "dom pixels" to "screen pixels"
  {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->xform.transform(pos);
    return pos;
  }
  point block_svg_element::inverse_translate(view &v,
                                             point pos) // inverse translate -
                                                        // to flat, non
                                                        // transformed position
                                                        // : "screen pixels" to
                                                        // "DOM pixels"
  {
    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->xform.inverse_transform(pos);
    return pos;
  }

  element *block_svg_element::find_element(view &v,
                                           point zpos /*at is this relative*/,
                                           bool  exact) {
    if (!is_visible(v)) return nullptr;

    const style *cs = get_style(v);

    zpos = inverse_translate(v, zpos);

    handle<layout_data> ld = ldata.ptr_of<layout_data>();

    tristate_v is_inside;

    if (ld->path && cs->overflow() > overflow_visible) {
      if (!(is_inside = ld->path->is_inside(zpos))) return 0;
    }

    element *ch = find_child_element(v, zpos, exact);

    if (ch) return ch;
    if (ld->path) {
      if (is_inside) return this;
      if (ld->path->is_inside(zpos)) return this;
    }
    return nullptr;
  }

  element *find_svg_child_element(view &v, element *cont,
                                  point at /*at is this relative*/,
                                  bool  exact) {
    each_element_backward it(cont);
    for (element *ch; it(ch);) {
      if (ch->is_of_type<block_svg_element>()) {
        it.dont_go_inside = true;
        element *och      = ch->find_element(v, at, exact);
        if (och) return och;
      } else if (svg_positioned(ch)) {
        point zpos        = at - ch->pos();
        it.dont_go_inside = true;
        element *och      = ch->find_element(v, zpos, exact);
        if (och) return och;
      }
    }
    return nullptr;
  }

  element *find_svg_element(view &v, element *self, gool::point zpos,
                            bool exact) {

    rect sp = self->hit_box(v);
    zpos = self->element::inverse_translate(v, zpos);
    if (!sp.contains(zpos) || !v.hit_test_element(self, zpos)) 
      return nullptr;
    
    point nt_zpos = zpos;

    zpos = self->inverse_translate(v, zpos);

    zpos += self->scroll_pos();
    zpos -= self->client_rect(v).s;

    element *t = self->ldata->zctx.find_element(v, zpos, nt_zpos, self);
    if (t) return t;

    if (exact && self->ldata->sb.hit_test(nt_zpos)/*!self->client_rect(v).contains(nt_zpos)*/) // not inside client area - on scrollbar
      return self;

    element *child = self->find_child_element(v, zpos, exact);
    return child ? child : self;
  }

  element *block_svg_element::nearest_known_box() { return svg_root(this); }

} // namespace html

#endif
