#ifndef __html_events_h__
#define __html_events_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html-primitives.h"
#include "html-clipboard.h"

#if defined(SCITERJS)
  #include "xdomjs/xjs.h"
#endif

namespace html {
  using namespace tool;
  using namespace gool;

  struct subscription;
  struct event_target;

  enum EVENT_GROUPS {
    HANDLE_MOUSE          = 0x0001,
    HANDLE_KEY            = 0x0002,
    HANDLE_FOCUS          = 0x0004,
    HANDLE_SCROLL         = 0x0008,
    HANDLE_TIMER          = 0x0010,
    HANDLE_SIZE           = 0x0020,
    HANDLE_DRAW           = 0x0040,
    HANDLE_DATA_ARRIVED   = 0x0080, // data arrived
    HANDLE_BEHAVIOR_EVENT = 0x0100, // BUTTON_CLICK, VALUE_CHANGED, etc.
    HANDLE_METHOD_CALL    = 0x0200, // behavior methods

    HANDLE_SCRIPTING_METHOD_CALL = 0x0400, /** behavior specific methods */
    //HANDLE_TISCRIPT_METHOD_CALL = 0x0800, /** behavior specific methods using direct tiscript::value's */

    HANDLE_EXCHANGE = 0x1000, // system drag-n-drop
/*OBSOLETE*/ HANDLE_GESTURE  = 0x2000, // touch gesture events,
    HANDLE_TOUCH    = 0x4000, // touch input events

    HANDLE_COMMAND                 = 0x8000,
    HANDLE_INTERNAL_BEHAVIOR_EVENT = 0x10000,
    HANDLE_ANIMATION               = 0x20000,
  };

  enum mouse_events {
    MOUSE_ENTER  = 0,
    MOUSE_LEAVE  = 1,
    MOUSE_MOVE   = 2,
    MOUSE_UP     = 3,
    MOUSE_DOWN   = 4,
    MOUSE_DCLICK = 5,
    MOUSE_WHEEL  = 6,
    MOUSE_TICK   = 7, // mouse pressed ticks
    MOUSE_IDLE   = 8, // mouse stay idle for some time

    /*OBSOLETE*/ DROP         = 9, // drag dst notification
    /*OBSOLETE*/ DRAG_ENTER   = 0x0A,
    /*OBSOLETE*/ DRAG_LEAVE   = 0x0B,
    /*OBSOLETE*/ DRAG_REQUEST = 0x0C, // drag src notification before drag
                                      // start. To cancel - return true from
                                      // handler.
    MOUSE_TCLICK = 0xF,
#ifdef USE_TOUCH
    MOUSE_TOUCH_START  = 0xFC, // touch device pressed somehow
    MOUSE_TOUCH_END    = 0xFD, // touch device depressed - clear, nothing on it
#endif // USE_TOUCH

    MOUSE_DRAG_REQUEST = 0xFE, // mouse drag start detected event
    MOUSE_CLICK        = 0xFF, // mouse click event

    /*OBSOLETE*/ DRAGGING = 0x100, // ORed with MOUSE_ENTER...MOUSE_DOWN codes above

    MOUSE_HIT_TEST = 0xFFE,
    MOUSE_CHECK    = 0xFFF,

  };

  enum mouse_buttons {
    MAIN_BUTTON   = 1, // aka left button
    PROP_BUTTON   = 2, // aka right button
    MIDDLE_BUTTON = 4,
  };

  enum sinking_flag {
    EVENT_SINKING     = 0x08000,
    EVENT_HANDLED     = 0x10000,
    EVENT_CANCELED    = 0x20000,
  };

  enum key_cmd {
    KEY_DOWN,
    KEY_UP,
    KEY_CHAR,
    KEY_COMP_STRING,
    KEY_RESULT_STRING,
  };

  enum key_alts {
    ALT_CONTROL     = 0x1,
    ALT_SHIFT       = 0x2,
    ALT_ALT         = 0x4,
    ALT_RIGHT_SHIFT = 0x8,
    ALT_COMMAND     = 0x10, // "command key" on OSX, "win" on Windows
#ifdef OSX
    ALT_SHORTCUT = ALT_COMMAND,
#else
    ALT_SHORTCUT = ALT_CONTROL,
#endif
    ALT_EXTENDED = 0x20,
#ifdef USE_TOUCH
    ALT_TOUCH    = 0x40, // touch or pen mouse event
#endif
  };

  enum FOCUS_EVENT_TYPE {
    FOCUS_OUT  = 0, // "focusout"
    FOCUS_IN   = 1, // "focusin"
    GOT_FOCUS  = 2, // "focus"
    LOST_FOCUS = 3, // "blur"
    REQUEST_FOCUS = 4, // it gets posted to parent to accept focus to be set on child
    ADVANCE_FOCUS = 5,
    //FOCUS_IN  = GOT_FOCUS,
    //FOCUS_OUT = LOST_FOCUS,
  };

  enum scroll_cmd {
    SCROLL_HOME = 0,
    SCROLL_END,
    SCROLL_STEP_PLUS,
    SCROLL_STEP_MINUS,
    SCROLL_PAGE_PLUS,
    SCROLL_PAGE_MINUS,
    SCROLL_POS,
    SCROLL_SLIDER_RELEASED,
    SCROLL_CORNER_PRESSED,
    SCROLL_CORNER_RELEASED,
    SCROLL_SLIDER_PRESSED,

    SCROLL_ANIMATION_START,
    SCROLL_ANIMATION_END,
  };

  enum SCROLL_SOURCE {
    SCROLL_SOURCE_UNKNOWN,
    SCROLL_SOURCE_KEYBOARD,
    SCROLL_SOURCE_SCROLLBAR,
    SCROLL_SOURCE_ANIMATOR,
    SCROLL_SOURCE_WHEEL,
  };

  #define MOUSE_WHEEL_UNIT (120.0f)

  enum EVENT_SOURCE {
    ES_UNAVAILABLE,
    ES_KEYBOARD,
    ES_MOUSE,
    ES_TOUCH,
    ES_PEN,
    ES_TOUCHPAD
  };

  EVENT_SOURCE get_current_event_source();

  struct event_proxy;

  struct event : public resource
  {
    DEFINE_TYPE_ID(event)

    uint            cmd;
    handle<element> target;
    tool::value     data; // auxiliary data
    uint            alt_state;
    bool            is_cancelable = true;
    handle<element> processing_target;
    ustring         ns;
    event_proxy*    proxy = nullptr;

    event(element *trg, int op);
    event(const event &src) : cmd(src.cmd)
      , target(src.target)
      , data(src.data)
      , alt_state(src.alt_state) 
      , is_cancelable(src.is_cancelable)
      , ns(src.ns)
    {
    }

    virtual ~event() { assert(!proxy); }
    virtual event* clone() const = 0;
     
    virtual bool     is_posted() const { return false; }

    virtual uint     event_group() const { return 0; }
    virtual uint     event_type() const { return cmd_no_flags(); }
    virtual uint     event_type_for_script() const { return cmd & ~EVENT_CANCELED; }
    virtual uint_v   event_reason() const { return uint_v(); }
    virtual void     event_reason(uint /*r*/) { ; }
    virtual element *event_source() const { return 0; }
    virtual void     event_source(element * /*el*/) {}
    virtual value    event_data() const { return data; }
    virtual void     event_data(const value &d) { data = d; }

    virtual ustring  event_type_name() const;
    virtual ustring  event_namespace() const { return ns; }

    // with phase and name.namespace
    virtual ustring  event_full_name() const;

    virtual value    event_result() const { return value(); }
    virtual void     event_result(const value & /*d*/) {}

    virtual bool     event_cancelable() const { return is_cancelable; }
    virtual void     event_cancelable(bool on) { is_cancelable = on; }

    virtual bool     event_bubbling() const { return true; }
    virtual void     event_bubbling(bool on) { assert(false); }
    // JS version
    virtual value    event_detail() const { return event_data(); }
    virtual void     event_detail(const value& v) { return event_data(v); }

    virtual ustring event_command_of() const { return ustring(); }

    uint cmd_no_flags() const { return cmd & 0xfff; }

    uint cmd_bubbling() const { return cmd & ~EVENT_HANDLED; }

    inline bool is_sinking() const { return (cmd & EVENT_SINKING) != 0; }
    inline bool is_handled() const { return (cmd & EVENT_HANDLED) != 0; }
    inline void is_handled(bool t) { assert(t); cmd |= EVENT_HANDLED; }
    inline bool is_canceled() const { return (cmd & EVENT_CANCELED) != 0; }

    void stop_propagation() { cmd |= EVENT_HANDLED; }
    void prevent_default() { cmd |= EVENT_CANCELED; }
    void cancel_event(bool on = true) {
      if(on) cmd |= EVENT_CANCELED;
      //else cmd &= ~EVENT_CANCELED; ???
    }
       
    static element *get_parent(element *t);
    static element *get_logical_parent(element *t);

    virtual point get_pos() const { return point(); }
    virtual point get_view_pos() const { return point(); }
    virtual int_v get_mouse_buttons() const { return 0; }
    virtual int_v get_mouse_button() const { return int_v(); }
    virtual float get_wheel_delta() const { return 0; }
    virtual bool  get_on_icon() const { return false; }
    virtual uint  get_alt_state() const { return alt_state; }
    virtual int   get_key_code() const { return 0; }
    virtual int   get_scroll_pos() const { return 0; }
    virtual bool  get_scroll_direction() const { return true; /*vertical*/ }
    virtual EVENT_SOURCE get_event_source() const { return ES_UNAVAILABLE; }

    virtual uint  get_start_time() const { return 0; }
    virtual uint  get_current_time() const { return 0; }
    virtual uint  get_end_time() const { return 0; }

    virtual bool  get_dragging_mode(uint& dd_mode) const { return false; }
    virtual bool  set_dragging_mode(uint dd_mode) { return false; }

    //element *match(view &v, wchars name, wchars selector) const;
    bool match(view &v, subscription& sub, handle<element> &principal, element* root = nullptr ) const;

    inline bool operator==(const event &rs) const {
      return cmd == rs.cmd && target == rs.target;
    }
    inline bool operator!=(const event &rs) const {
      return cmd != rs.cmd || target != rs.target;
    }

    bool is_shift() const { return (get_alt_state() & ALT_SHIFT) != 0; }
    bool is_right_shift() const { return (get_alt_state() & ALT_RIGHT_SHIFT) != 0; }
    bool is_control() const { return (get_alt_state() & ALT_CONTROL) != 0; }
    bool is_alt() const { return (get_alt_state() == ALT_ALT || get_alt_state() == (ALT_ALT | ALT_EXTENDED)); } // Alt is considered as a modifer only if it is pressed alone
    bool is_option() const { return is_alt(); } // MacOS "option" button
    bool is_command() { return (get_alt_state() & ALT_COMMAND) != 0; }
    bool is_shortcut() { return (get_alt_state() & ALT_SHORTCUT) != 0; }
    bool is_extended() { return (get_alt_state() & ALT_EXTENDED) != 0; }
    bool has_modifiers() { return (get_alt_state() & ~ALT_EXTENDED) != 0; }
    bool has_no_modifiers() { return (get_alt_state() & ~ALT_EXTENDED) == 0; }

    // onmousemove -> true, but onclick only if target == this
    virtual bool is_on_target(const event_target* pet) const;

    static bool get_key_state(uint vk_code);

    void attach_proxy(event_proxy* proxy_) {
      assert(!proxy && proxy_);
      proxy = proxy_;
    }
    void detach_proxy(event_proxy* proxy_) {
      assert(proxy_ == proxy);
      proxy = nullptr;
    }
  };

  struct event_proxy: public resource 
  {
    event*        _original = nullptr;
    handle<event> _airborn;

    event_proxy(event& evt) { _original = &evt; evt.attach_proxy(this); }
    event_proxy(event* pevt) { 
      _airborn = pevt; 
    }
    virtual ~event_proxy()  
    { 
      if (_original) _original->detach_proxy(this);
      else
        _airborn = _airborn;
    }

    void detach(event* pe) { 
      if (_original) { 
        assert(_original == pe);
        assert(pe->proxy == this);
        _airborn = _original->clone(); 
        _original = nullptr;
        pe->proxy = nullptr;
      } 
    }

    event& evt() { return _original ? *_original : *_airborn.ptr(); }
  };

  enum DRAGGING_MODE {
    NO_DRAGGING,
    DRAGGING_MOVE = 0x01,
    DRAGGING_COPY = 0x02,
  };

  struct event_mouse : public event {

    DEFINE_TYPE_ID_DERIVED(event_mouse,event)

    enum { EVENT_GROUP = HANDLE_MOUSE };
    point                pos;
    point                pos_view; // used to be pos_document, frames, sic!
    int                  button_state = 0;
    handle<gool::cursor> cursor;
    EVENT_SOURCE         event_source = get_current_event_source();

    element *is_on_icon = nullptr;  // is on icon of this element.

    handle<element>  dragging;
    DRAGGING_MODE    dragging_mode = DRAGGING_MODE();

    event_mouse(): event(nullptr, 0) {}

    event_mouse(element *trg, int op, point view_pos, int b_state, uint a_state)
        : event(trg, op), pos(view_pos), pos_view(view_pos), button_state(b_state) { 
      alt_state = a_state;
    }

    event_mouse(const event_mouse& other): event(other) {
      pos = other.pos;
      pos_view = other.pos_view;
      button_state = other.button_state;
      event_source = other.event_source;
    }
    virtual ~event_mouse() { if (proxy) proxy->detach(this); }
    virtual event* clone() const { return new event_mouse(*this); }

    virtual uint event_group() const { return EVENT_GROUP; }

    bool  is_point_button() const { return button_state == 1; }
    bool  is_prop_button() const { return button_state == 2; }
    float wheel_delta() const { return loshort((dword)button_state) / MOUSE_WHEEL_UNIT; }
    size wheel_delta_xy() const {
      sizef r = sizef(hishort((dword)button_state),
                      loshort((dword)button_state));
      if (is_shift()) swap(r.x, r.y);
      return r;
    }
    virtual bool  get_dragging_mode(uint& dd_mode) const override {
      dd_mode = dragging_mode;
      return true;
    }
    virtual bool  set_dragging_mode(uint dd_mode)  override {
      dragging_mode = (DRAGGING_MODE)dd_mode;
      return true;
    }
#ifdef USE_TOUCH
    bool is_touch() const { return (alt_state & ALT_TOUCH) != 0; }
#endif
    bool is_on_target(const event_target* pet) const override { return true; }

    EVENT_SOURCE  get_event_source() const override {
      return event_source;
    }

#ifdef USE_TOUCH
    bool is_real_wheel_tick() const {
      if (cmd_no_flags() != MOUSE_WHEEL)
        return false;
#ifdef WINDOWS
      size sz = wheel_delta_xy();
      if (abs(sz.y) == 120 && sz.x == 0)
        return true; // seems like generated by physical wheel on mouse 
      if (abs(sz.x) == 120 && sz.y == 0)
        return true;
      return false;
#else
      return !is_touch();
      //return event_source == ES_MOUSE || event_source == ES_KEYBOARD;
#endif // WINDOWS
    }

    bool is_physical_scroll() const { // true if this a MOUSE_WHEEL but not from synthetic intertia  
      if (cmd_no_flags() != MOUSE_WHEEL)
        return false;
#ifdef WINDOWS
      size sz = wheel_delta_xy();
      if (abs(sz.y) == 120 && sz.x == 0)
        return true; // seems like generated by physical wheel on mouse 
      if (abs(sz.x) == 120 && sz.y == 0)
        return true;
      return is_touch();
#else
      return true;
      //return is_touch();
#endif // WINDOWS
    }
#endif

/*
    bool is_shift() const { return (alt_state & ALT_SHIFT) != 0; }
    bool is_control() const { return (alt_state & ALT_CONTROL) != 0; }
    bool is_command() const { return (alt_state & ALT_COMMAND) != 0; }
    bool is_shortcut() const { return (alt_state & ALT_SHORTCUT) != 0; }
    bool is_alt() const {
      return (alt_state == ALT_ALT || alt_state == (ALT_ALT | ALT_EXTENDED));
    } // Alt is considered as a modifer only if it is pressed alone
    */
    // See: http://terrainformatica.com/forums/topic.php?id=1153&page&replies=1
    virtual bool get_on_icon() const override { return is_on_icon != 0; }

    virtual point get_pos() const override { return pos; }
    virtual point get_view_pos() const override { return pos_view; }
    virtual int_v get_mouse_buttons() const override { return button_state; }
    virtual int_v get_mouse_button() const override { return button_no(); }
    virtual float get_wheel_delta() const override { return wheel_delta(); }

    int_v button_no() const {
      if (!has_buttons()) return int_v();
      if (!button_state) return int_v(0);
      int n = 1;
      for (int cnt = 1; cnt <= 8; ++cnt) {
        if (n & button_state) return int_v(cnt);
        n <<= 1;
      }
      return int_v(0);
    }
    bool has_buttons() const {
      switch (cmd_no_flags()) {
        case MOUSE_ENTER:
        case MOUSE_LEAVE:
        case MOUSE_MOVE:
        case MOUSE_UP:
        case MOUSE_DOWN:
        case MOUSE_DCLICK:
            //MOUSE_WHEEL = 6,
        case MOUSE_TICK:
        case MOUSE_TCLICK:
        case MOUSE_CLICK:
          return true;
        default:
          return false;
      }
    }
    bool is_dd_event() const {
      int t = cmd & 0xf;
      return ((t >= MOUSE_ENTER && t <= MOUSE_DOWN) || t == MOUSE_TICK); // ;
    }

    bool is_dragging() const { return (cmd & DRAGGING) != 0; }
  };

  enum gesture_cmd {
    GESTURE_REQUEST = 0, // return true and fill flags if it will handle gestures.
    GESTURE_START = 1,
    GESTURE_END = 2,

    GESTURE_ZOOM,   // The zoom gesture.
    GESTURE_PAN,    // The pan gesture.
    GESTURE_ROTATE, // The rotation gesture.
    GESTURE_TAP1,   // The tap gesture.
    GESTURE_TAP2,   // The two-finger tap gesture.
    // GESTURE_SWIPE,       // The swipe gesture.
  };
  enum gesture_state {
    GESTURE_STATE_BEGIN   = 1,
    GESTURE_STATE_INERTIA = 2,
    GESTURE_STATE_END     = 4,
  };

  enum gesture_types {
    GESTURE_FLAG_ZOOM           = 0x0001,
    GESTURE_FLAG_ROTATE         = 0x0002,
    GESTURE_FLAG_PAN_VERTICAL   = 0x0004,
    GESTURE_FLAG_PAN_HORIZONTAL = 0x0008,
    GESTURE_FLAG_TAP1           = 0x0010, // single finger tap
    GESTURE_FLAG_TAP2           = 0x0020, // two fingers tap
    GESTURE_FLAG_PAN_WITH_GUTTER  = 0x4000, // PAN_VERTICAL and PAN_HORIZONTAL modifiers
    GESTURE_FLAG_PAN_WITH_INERTIA = 0x8000, //
    GESTURE_FLAGS_ALL             = 0xFFFF, //
  };



  struct event_gesture : public event {
    DEFINE_TYPE_ID_DERIVED(event_gesture, event)
    enum { EVENT_GROUP = HANDLE_GESTURE };
    point pos;
    point pos_view;
    uint  flags;     // for WILL_HANDLE_GESTURE combination of gesture_types.
                     // for others it is a combination of gesture_state's
    uint delta_time; // period of time from previous event.
    size delta_xy;   // for GESTURE_PAN it is a direction vector
    double delta_v;  // for GESTURE_ROTATE - delta angle (radians)
                     // for GESTURE_ZOOM - zoom value, is less or greater than 1.0
    event_gesture(element *target, gesture_cmd cmd, point pv)
        : event(target, cmd), flags(0), pos(pv), pos_view(pv) {}

    virtual uint event_group() const { return EVENT_GROUP; }

    virtual uint_v event_reason() const { return flags; }
    virtual void   event_reason(uint r) { flags = r; }

    virtual point get_pos() const override { return pos; }
    virtual point get_view_pos() const override { return pos_view; }

  private:
    event_gesture(const event_gesture& other) : event(other) {
      pos = other.pos;
      pos_view = other.pos_view;
      flags = other.flags;
      delta_time = other.delta_time;
      delta_xy = other.delta_xy;
      delta_v = other.delta_v;
    }
  public:
    virtual ~event_gesture() { if (proxy) { proxy->detach(this); } }
    virtual event* clone() const { return new event_gesture(*this); }

  };

  enum touch_cmd {
    TOUCH_START = 0, 
    TOUCH_MOVE = 1,
    TOUCH_END = 2,

    // logical events
    TOUCH_PAN,    // The pan gesture.
    TOUCH_ZOOM,   // The zoom gesture.
    TOUCH_ROTATE, // The rotation gesture.
    TOUCH_TAP1,   // The tap gesture, a.k.a. click
    TOUCH_TAP2,   // The two-finger tap gesture, a.k.a. right-click
    TOUCH_DOUBLE_TAP
  };

  struct event_touch : public event {
    DEFINE_TYPE_ID_DERIVED(event_touch, event)
      enum { EVENT_GROUP = HANDLE_GESTURE };
    point pos;
    point pos_view;
    uint  flags;        // for WILL_HANDLE_GESTURE combination of gesture_types.
                        // for others it is a combination of gesture_state's
    uint   delta_time;  // period of time from previous event.
    size   delta_xy;    // for GESTURE_PAN it is a direction vector
    double delta_angle; // for GESTURE_ROTATE - delta angle (radians)
    double delta_zoom;  // for GESTURE_ZOOM - zoom value, is less or greater than 1.0

    event_touch(element *target, touch_cmd cmd, point pv)
      : event(target, cmd), flags(0), pos(pv), pos_view(pv) {}

    virtual uint event_group() const { return EVENT_GROUP; }

    virtual bool is_on_target(const event_target* pet) const override { return true; }

    virtual uint_v event_reason() const { return flags; }
    virtual void   event_reason(uint r) { flags = r; }

    virtual point get_pos() const override { return pos; }
    virtual point get_view_pos() const override { return pos_view; }

  private:
    event_touch(const event_touch& other) : event(other) {
      pos = other.pos;
      pos_view = other.pos_view;
      flags = other.flags;
      delta_time = other.delta_time;
      delta_xy = other.delta_xy;
      delta_angle = other.delta_angle;
      delta_zoom = other.delta_zoom;
    }
  public:
    virtual ~event_touch() { if (proxy) { proxy->detach(this); } }
    virtual event* clone() const { return new event_touch(*this); }

  };



  ustring get_key_name(uint key_code, uint alts = ALT_ALT /*in this case it will return just key name*/);
  uint    get_key_code_by_name(chars vk_name);

  struct event_key : public event {

    DEFINE_TYPE_ID_DERIVED(event_key, event)

    enum { EVENT_GROUP = HANDLE_KEY };

    int key_code;

    struct ime_params {
      range  target;
      int_v  caret_pos;
      wchars composition;
    } ime;

    event_key() : event(0, 0), key_code(0) {}

    event_key(element *trg, int op, int k_code, uint a_state)
        : event(trg, op), key_code(k_code)  {
      alt_state = a_state;
    }

    virtual uint event_group() const { return EVENT_GROUP; }

    static element *get_parent(element *t) { return get_logical_parent(t); }

    virtual int  get_key_code() const override { return key_code; }

    event_key(const event_key& other) : event(other) {
      key_code = other.key_code;
      ime = other.ime;
    }
    virtual event* clone() const { return new event_key(*this); }
    virtual ~event_key() { if (proxy) proxy->detach(this); }

  };

  enum FOCUS_CAUSE {
    BY_CODE,
    BY_MOUSE,
    BY_KEY_NEXT,
    BY_KEY_PREV,
    BY_KEY_SHORTCUT,
  };

  struct event_focus : public event {

    DEFINE_TYPE_ID_DERIVED(event_focus, event)

    enum { EVENT_GROUP = HANDLE_FOCUS };

    bool        to_from_outside;
    FOCUS_CAUSE cause;
    bool        need_ime;

    event_focus(element *trg, int op, FOCUS_CAUSE fc, bool to_from_out)
        : event(trg, op), to_from_outside(to_from_out), cause(fc),
          need_ime(false) {}

    virtual uint   event_group() const { return EVENT_GROUP; }
    virtual uint_v event_reason() const { return uint_v(cause); }

    bool by_key() const { return cause == BY_KEY_NEXT || cause == BY_KEY_PREV; }
    bool by_code() const { return cause == BY_CODE; }
    bool by_mouse() const { return cause == BY_MOUSE; }

    static element *get_parent(element *t) { return get_logical_parent(t); }

    //bool is_on_target(const event_target* pet) const override { return true; }

  private:
    event_focus(const event_focus& other) : event(other) {
      to_from_outside = other.to_from_outside;
      cause = other.cause;
      need_ime = other.need_ime;
    }
  public:
    virtual event* clone() const { return new event_focus(*this); }
    virtual ~event_focus() { if (proxy) proxy->detach(this); }
    
  };

  struct event_scroll : public event {

    DEFINE_TYPE_ID_DERIVED(event_scroll, event)

    enum { EVENT_GROUP = HANDLE_SCROLL };

    int  pos;
    bool vertical;
    SCROLL_SOURCE source;
    uint reason;
    event_scroll(element *trg, int op, bool vscroll, int position, SCROLL_SOURCE src, uint rsn)
        : event(trg, op), vertical(vscroll), pos(position), source(src), reason(rsn) {}

    virtual uint   event_group() const override { return EVENT_GROUP; }
    virtual uint_v event_reason() const override {
      //return uint(get_scroll_direction());
      if( vertical )
        return make_dword(word(source), word(reason)) | 0x80000000;
      else
        return make_dword(word(source), word(reason));
    }

    virtual bool     event_bubbling() const { return false; }

    virtual int  get_scroll_pos() const override { return pos; }
    virtual bool get_scroll_direction() const override { return vertical; }

  private:
    event_scroll(const event_scroll& other) : event(other) {
      pos = other.pos;
      vertical = other.vertical;
      source = other.source;
      reason = other.reason;
    }
  public:
    virtual event* clone() const { return new event_scroll(*this); }
    virtual ~event_scroll() { if (proxy) proxy->detach(this); }

  };

  enum exchange_cmd {
    X_DRAG_ENTER,
    X_DRAG_LEAVE,
    X_DRAG,
    X_DROP,
    X_PASTE,
    X_DRAG_REQUEST,
    X_DRAG_CANCEL,
    X_WILL_ACCEPT_DROP,
  };

  enum DD_MODE {
    dd_none         = 0, // DROPEFFECT_NONE	( 0 )
    dd_copy         = 1, // DROPEFFECT_COPY	( 1 )
    dd_move         = 2, // DROPEFFECT_MOVE	( 2 )
    dd_copy_or_move = 3, // DROPEFFECT_COPY	( 1 ) | DROPEFFECT_MOVE	( 2 )
    dd_link         = 4, // DROPEFFECT_LINK	( 4 )
  };

  struct event_exchange : public event {

    DEFINE_TYPE_ID_DERIVED(event_exchange, event)

    enum { EVENT_GROUP = HANDLE_EXCHANGE };

    point pos;
    // point pos_screen;
    point pos_view; // used to be pos_document, frames, sic!

    handle<clipboard::data> data;
    DD_MODE                 dd_modes;
    handle<element>         dd_source; // element that started do_drag

    event_exchange() : event(0, 0), dd_modes(dd_none) {}

    event_exchange(element *trg, int op, DD_MODE modes, clipboard::data *pd,
                   point view_pos, element *src)
        : event(trg, op), data(pd), dd_modes(modes), pos(view_pos),
          pos_view(view_pos), dd_source(src) {}

    virtual uint  event_group() const { return EVENT_GROUP; }
    virtual point get_pos() const override { return pos; }
    virtual point get_view_pos() const override { return pos_view; }

    virtual bool  get_dragging_mode(uint& dd_mode) const override {
      dd_mode = dd_modes;
      return true;
    }
    virtual bool  set_dragging_mode(uint dd_mode)  override {
      dd_modes = DD_MODE(dd_mode);
      return true;
    }

    virtual value event_detail() const override {
      if (data.is_defined()) {
        if (html::clipboard::item *pd = data->get_default()) {
          value map;
          map.set_prop("dataType", value(data->get_default()->item_type_name()));
          map.set_prop("data", data_value());
          return map;
        }
      }
      return value();
    }

    value data_value() const {

      if (!data)
        return value();

      html::clipboard::item *pd = data->get_default();

      if (pd) switch (pd->data_type) {
        case html::clipboard::cf_undefined: return value();
        case html::clipboard::cf_text: {
          html::clipboard::text_item *ti = (html::clipboard::text_item *)pd;
          return value(ti->val);
        }
        case html::clipboard::cf_html: {
          html::clipboard::html_item *ti = (html::clipboard::html_item *)pd;
          return value(u8::cvt(ti->val()));
        }
        case html::clipboard::cf_picture: return value::null_val();
        case html::clipboard::cf_hyperlink: {
          html::clipboard::link_item *li = (html::clipboard::link_item *)pd;
          value map;
          map.set_prop("caption", value(li->caption));
          map.set_prop("url", value(li->url));
          return map;
        }
        case html::clipboard::cf_json: {
          html::clipboard::json_item *ji = (html::clipboard::json_item *)pd;
          ustring ws = u8::cvt(ji->val());
          wchars wcs = ws;
          return xjson::parse(wcs, false);
        }
        case html::clipboard::cf_xml: return value::null_val();
        case html::clipboard::cf_file: {
            html::clipboard::file_item *fi = (html::clipboard::file_item *)pd;
            if (fi->filenames.size() == 1)
              return value(fi->filenames[0]);
            else
            {
              value list = value::make_array(uint(fi->filenames.size()));
              for (int n = 0; n < fi->filenames.size(); ++n) {
                value t = value(fi->filenames[n]);
                list.set_element(uint(n), t);
              }
              return list;
            }
          }
        }
        return value::null_val();
      }

      virtual value event_result() const override {
        return is_canceled() ? value(false) : value(true);
      }
      virtual void  event_result(const value& d) override
      {
        if (d.is_bool() && !d.get<bool>()) cancel_event();
      }

  private:
    event_exchange(const event_exchange& other) : event(other) {
      pos = other.pos;
      pos_view = other.pos_view;
      data = other.data;
      dd_modes = other.dd_modes;
      dd_source = other.dd_source;
    }
  public:
    virtual ~event_exchange() { if (proxy) { proxy->detach(this); } }
    event* clone() const { return new event_exchange(*this); }
  };

  enum CONTENT_CHANGE_BITS {
    CONTENT_ADDED      = 0x01,
    CONTENT_REMOVED    = 0x02,
    ATTRIBUTES_CHANGED = 0x04,
    STATES_CHANGED = 0x08,
  };

  // enum GENERIC_EVENTS /* when event group is 0 */
  //{
  //    CLICK = 0,   // generic click
  //    CHANGE = 1   // generic change
  //};

  enum BEHAVIOR_EVENTS {
    BUTTON_CLICK             = 0,
    BUTTON_PRESS             = 1, // mouse down or key down in button
    BUTTON_STATE_CHANGED     = 2, // radio/checkbox changed their values
    EDIT_VALUE_CHANGING      = 3,
    EDIT_VALUE_CHANGED       = 4,
    SELECT_SELECTION_CHANGED = 5, // selection in <select> changed
    SELECT_VALUE_CHANGED     = 6, // select value changed
    POPUP_REQUEST            = 7, // request to show popup just received,
                                  // here DOM of popup element can be modifed.
    POPUP_READY              = 8, // popup element has been measured and ready to be shown on screen,
                                  // here you can use functions like ScrollToView.
    POPUP_DISMISSED          = 9, // popup element is closed,
                                  //     here DOM of popup element can be modifed again -
                                  //     e.g. some items can be removed to free memory.

    MENU_ITEM_ACTIVE = 0xA, // menu item activated by mouse hover or by keyboard
    MENU_ITEM_CLICK  = 0xB, // menu item click

    SELECT_SELECTION_CHANGING = 0xC, // selection in <select> got different
                                     // :current, but no MOUSE_UP yet -
                                     // transaction not finished.
    FORM_VALUE_CHANGED = 0xD, // form value has changed (some input element
                              // inside the form has generated
                              // EDIT_VALUE_CHANGED, etc. ).

    POPUP_SELECT_VALUE_CHANGED  = 0xE, // 

    CONTEXT_MENU_SETUP   = 0xF,  // evt.source is a menu that is about to popup
    CONTEXT_MENU_REQUEST = 0x10, //

    VISUAL_STATUS_CHANGED = 0x11, // notification being posted to the element being shown or hidden
    DISABLED_STATUS_CHANGED = 0x12, // broadcast notification being posted to
                                    // all elements of some container that
                                    // changes :disabled state

    POPUP_DISMISSING      = 0x13, // popup is about to be closed
    MASKED_EDIT_INCREMENT = 0x14, // masked edit increment

    CONTENT_CHANGED = 0x15, // content has been changed, is posted to the
                            // element that gets content changed,  reason is
                            // combination of CONTENT_CHANGE_BITS.

    CLICK  = 0x16, // generic click
    CHANGE = 0x17, // generic change

    MEDIA_CHANGED =           0x18, // media changed (screen resolution, number of displays, etc.)
    INPUT_LANGUAGE_CHANGED =  0x19, // input language has changed, data is iso lang-country string

    CONTENT_MODIFIED        = 0x1A, // posted then editable content has changed since
                             // last set_modified(false)/save point
    READONLY_STATUS_CHANGED = 0x1B, // broadcast notification being posted to
                                    // all elements of some container that
                                    // changes :read-only state

    ARIA_LIVE_AREA_CHANGED  = 0x1C, // change in aria-live="polite|assertive"
    CURRENT_ELEMENT_CHANGED = 0x1D, // got new :current

    LOAD_SUCCEEDED          = 0x1E, // on element having @src
    LOAD_FAILED             = 0x1F, // on element having @src

    INPUT                   = 0x20, // generic W3C's input 

    HYPERLINK_CLICK         = 0x80, // hyperlink click

    /*
          TABLE_HEADER_CLICK,            // click on some cell in table header,
                                         //     target = the cell,
                                         //     reason = index of the cell
       (column number, 0..n) TABLE_ROW_CLICK,               // click on data row
       in the table, target is the row
                                         //     target = the row,
                                         //     reason = index of the row
       (fixed_rows..n) TABLE_ROW_DBL_CLICK,           // mouse dbl click on data
       row in the table, target is the row
                                         //     target = the row,
                                         //     reason = index of the row
       (fixed_rows..n) */
    PASTE_IMAGE = 0x8D,
    PASTE_TEXT  = 0x8E,
    PASTE_HTML  = 0x8F,

    ELEMENT_COLLAPSED = 0x90, // element was collapsed, so far only
                              // behavior:tabs is sending these two to the
                              // panels,
    ELEMENT_EXPANDED = 0x91,  // element was expanded,

    ACTIVATE_CHILD = 0x92, // activate (select) child,
                           // used for example by accesskeys behaviors to send
                           // activation request, e.g. tab on behavior:tabs.

    DO_SWITCH_TAB = ACTIVATE_CHILD, // command to switch tab programmatically,
                                    // handled by behavior:tabs use it as
                                    // HTMLayoutPostEvent(tabsElementOrItsChild,
                                    // DO_SWITCH_TAB, tabElementToShow, 0);

    INIT_DATA_VIEW = 0x93, // request to virtual grid to initialize its view

    ROWS_DATA_REQUEST = 0x94, // request from virtual grid to data source
                              // behavior to fill data in the table parameters
                              // passed throug DATA_ROWS_PARAMS structure.

    UI_STATE_CHANGED = 0x95, // ui state changed, observers shall update their
                             // visual states. is sent for example by
                             // behavior:richtext when caret position/selection
                             // has changed.

    FORM_SUBMIT = 0x96, // behavior:form detected submission event.
                        // BEHAVIOR_EVENT_PARAMS::data field contains data to be
                        // posted. BEHAVIOR_EVENT_PARAMS::data is of type T_MAP
                        // in this case key/value pairs of data that is about to
                        // be submitted. You can modify the data or discard
                        // submission by returning TRUE from the handler.
    FORM_RESET = 0x97, // behavior:form detected reset event (from button
                       // type=reset). BEHAVIOR_EVENT_PARAMS::data field
                       // contains data to be reset. BEHAVIOR_EVENT_PARAMS::data
                       // is of type T_MAP in this case key/value pairs of data
                       // that is about to be rest. You can modify the data or
                       // discard reset by returning TRUE from the handler.

    DOCUMENT_COMPLETE = 0x98, // behavior:frame have complete document.

    HISTORY_PUSH  = 0x99, // behavior:history stuff
    HISTORY_DROP  = 0x9A,
    HISTORY_PRIOR = 0x9B,
    HISTORY_NEXT  = 0x9C,
    HISTORY_STATE_CHANGED = 0x9D, // behavior:history notification - history stack has changed

    CLOSE_POPUP = 0x9E, // close popup request,
    TOOLTIP_REQUEST = 0x9F, // request tooltip, evt.source <- is the tooltip element.

    ANIMATION        = 0xA0, // animation started (reason=1) or ended(reason=0) on the element.
    TRANSITION       = 0xA1, // transition started (reason=1) or ended(reason=0) on the element.
    SWIPE            = 0xB0, // swipe gesture detected, reason=4,8,2,6 - swipe direction

    SHOW_POPUP       = 0xB8, // posted show popup requst.
    CLOSE_POPUP_TREE = 0xB9, // posted close popup tree.

    DOCUMENT_CREATED = 0xC0, // document created, script namespace initialized.
                             // target -> the document
    DOCUMENT_CLOSE_REQUEST = 0xC1, // view::ask_unload
    DOCUMENT_CLOSE         = 0xC2, //
    DOCUMENT_READY         = 0xC3, // document got styles and behaviors
    DOCUMENT_PARSED        = 0xC4, // document parsed
    //DOCUMENT_RELOAD        = 0xC5, // request to reload the document
    DOCUMENT_CLOSING       = 0xC6, // view::notify_close
    CONTAINER_CLOSE_REQUEST= 0xC7, // document is processing DOCUMENT_CLOSE_REQUEST
    CONTAINER_CLOSING      = 0xC8, // document is processing DOCUMENT_CLOSING

    VIDEO_INITIALIZED    = 0xD1,
    VIDEO_STARTED        = 0xD2,
    VIDEO_STOPPED        = 0xD3,
    VIDEO_BIND_RQ        = 0xD4,
    VIDEO_SOURCE_CREATED = 0xD5,

    VIDEO_COORDINATE     = 0xD6, // video coordinator shall handle this to be ccordiantor
    VIDEO_FRAME          = 0xD7, // sent ot video coordinator
    VIDEO_FRAME_REQUEST  = 0xD8, // animation step, a.k.a. animation frame

    PAGINATION_STARTS = 0xE0,
    PAGINATION_PAGE   = 0xE1, // reason -> page no
    PAGINATION_ENDS   = 0xE2, // reason -> total pages

    CUSTOM            = 0xF0, // custom, script generated event
    MOUNT_COMPONENT   = 0xF1, // SSX, delayed mount_component
    COMPONENT_UPDATE  = 0xF2, // reactor, componentUpdate request
    COMPONENT_ERROR   = 0xF3, // reactor, componentDidCatch notification

    CONTENT_REQUIRED  = 0xFE,

    FIRST_APPLICATION_EVENT_CODE = 0x100
    // all custom event codes shall be greater
    // than this number. All codes below this will be used
    // solely by application - HTMLayout will not intrepret it
    // and will do just dispatching.
    // To send event notifications with  these codes use
    // HTMLayoutSend/PostEvent API.

  };

  enum INTERNAL_BEHAVIOR_EVENTS {
    INVOKE_ON_SIZE_CHANGED = 1,
    DO_SET_FOCUS           = 2,
  };

  enum EVENT_REASON {
    CLICK_BY_MOUSE         = 0,
    CLICK_BY_KEY           = 1,
    CLICK_SYNTHESIZED      = 2, //
    CLICK_BY_MOUSE_ON_ICON = 3,
  };

  enum EDIT_CHANGED_REASON {
    CHANGE_BY_INS_CHAR,  // single char insertion
    CHANGE_BY_INS_CHARS, // character range insertion, clipboard
    CHANGE_BY_DEL_CHAR,  // single char deletion
    CHANGE_BY_DEL_CHARS, // character range deletion (selection)
    CHANGE_BY_UNDO_REDO,
    CHANGE_BY_INS_CONSECUTIVE_CHAR, // single char insertion, previous character
                                    // was inserted in previous position
    CHANGE_BY_DEL_CONSECUTIVE_CHAR, // single char removal, previous character
                                    // was removed in previous position
    CHANGE_BY_CODE,
  };

  /*struct BEHAVIOR_EVENT_PARAMS
  {
      uint     cmd;        // BEHAVIOR_EVENTS

  };*/

  struct event_behavior : public event {

    DEFINE_TYPE_ID_DERIVED(event_behavior, event)

    enum _ { EVENT_GROUP = HANDLE_BEHAVIOR_EVENT };

    handle<element> source; // source element handler, e.g. on select this is clicked option
    uint_ptr reason;        // EVENT_REASON or EDIT_CHANGED_REASON - UI action causing
                            // change.
    ustring name;
    // uint           symbol;   // scripting symbol id
    bool internal; // e.g. (EVAL_...)
    bool processed;

    tristate_v is_bubbling;

    event_behavior()
        : event(0, 0), reason(0), internal(false), processed(false) {}
    event_behavior(element *src, element *target, uint op, uint_ptr r)
        : event(target, op), source(src), reason(r), internal(false),
          processed(false) {}
    event_behavior(wchars ename);

    virtual uint     event_group() const override { return EVENT_GROUP; }
    virtual uint_v   event_reason() const override { return uint(reason); }
    virtual void     event_reason(uint r) override { reason = r; }
    virtual ustring  event_type_name() const override {
      if (this->cmd_no_flags() == CUSTOM)
        return name;
      else
        return event::event_type_name();
    }
    virtual element *event_source() const override { return source; }
    virtual void     event_source(element *el) { source = el; }

    static element *get_parent(element *t) { return get_logical_parent(t); }

    inline bool operator==(const event_behavior &rs) const {
      // return memcmp(this,&rs, sizeof(event_behavior)) == 0;
      return source == rs.source && reason == rs.reason && data == rs.data &&
             name == rs.name && internal == rs.internal &&
             event::operator==(rs);
    }
    inline bool operator!=(const event_behavior &rs) const {
      // return memcmp(this,&rs, sizeof(event_behavior)) == 0;
      return source != rs.source || reason != rs.reason || data != rs.data ||
             name != rs.name || internal != rs.internal ||
             event::operator!=(rs);
    }

    // operator == but without reason comparison
    inline bool can_collapse(const event_behavior &other) const {
      return event::operator==(other) && source == other.source &&
             data == other.data && name == other.name &&
             internal == other.internal;
    }

    bool is_internal() const { return internal; }

    virtual bool     event_bubbling() const override { 
      if(is_bubbling.is_defined())
        return !!is_bubbling.val(0);
      else switch (cmd_no_flags())
      {
        case LOAD_SUCCEEDED:
        case LOAD_FAILED: // on element having @src
          return false;
        default:
          return true;
      }
    }
    virtual void     event_bubbling(bool on) { 
      is_bubbling = on;
    }

    event_behavior(const event_behavior &src)
      : event(src), source(src.source), reason(src.reason),
      internal(src.internal), name(src.name), processed(false), is_bubbling(src.is_bubbling) {}

    event* clone() const { return new event_behavior(*this); }

    virtual ~event_behavior() { if (proxy) proxy->detach(this); }


  };

  struct event_primitive: public event_behavior {
    event_primitive(wchars ename) : event_behavior(ename) {}
    bool is_on_target(const event_target* pet) const override { return true; }
  };

  struct internal_event_behavior : public event_behavior {
    enum _ { EVENT_GROUP = HANDLE_INTERNAL_BEHAVIOR_EVENT };

    internal_event_behavior(element *src, element *target, uint op, uint r)
        : event_behavior(src, target, op, r) {
      internal = true;
    }

    virtual uint event_group() const { return EVENT_GROUP; }
  };

  /*
  enum COMMANDS
  {
    INIT_CONTEXT_MENU = 0, // pmenu shall contain valid ptr
    CMD_UNDO = 1,
    CMD_REDO,
    CMD_CUT,
    CMD_COPY,
    CMD_PASTE,
    CMD_DELETE,
    CMD_SELECT_ALL
  }; */

  struct event_command : public event {

    DEFINE_TYPE_ID_DERIVED(event_command, event)

    enum _1 { EVENT_GROUP = HANDLE_COMMAND };
    enum _2 { CHECK = 0, EXEC = 1 };
    handle<element> source;
    ustring         command;
    tool::value     result; // result
    event_command(element *trg, element *src) : event(trg, 0), source(src) {
      cmd = EXEC;
    }

    event_command(const event_command& other): event(other), source(other.source), command(other.command), result(other.result) { }

    event* clone() const { return new event_command(*this); }

    ~event_command() { if (proxy) proxy->detach(this); }

    bool            doit() const { return cmd == EXEC; }
    bool            checkit() const { return cmd == CHECK; }
    static element *get_parent(element *t) { return get_logical_parent(t); }

    virtual uint event_group() const { return EVENT_GROUP; }

    virtual element *event_source() const { return source; }
    virtual void     event_source(element *el) { source = el; }

    virtual value event_result() const { return result; }
    virtual void  event_result(const value &d) { result = d; }

    virtual value event_detail() const override { return event_result(); }
    virtual void  event_detail(const value &d) override { return event_result(d); }

    virtual ustring event_command_of() const { return command; }

    virtual ustring  event_type_name() const {
      if (cmd_no_flags() == CHECK)
        return ustring::format(W("check:%s"), command.c_str());
      else if (cmd_no_flags() == EXEC)
        return ustring::format(W("exec:%s"), command.c_str());
      return ustring();
    }

    static ustring EDIT_CUT() {
      static ustring s = WCHARS("edit:cut");
      return s;
    }
    static ustring EDIT_COPY() {
      static ustring s = WCHARS("edit:copy");
      return s;
    }
    static ustring EDIT_PASTE() {
      static ustring s = WCHARS("edit:paste");
      return s;
    }
    static ustring EDIT_PASTE_TEXT() {
      static ustring s = WCHARS("edit:paste-text");
      return s;
    }
    static ustring EDIT_SELECT_ALL() {
      static ustring s = WCHARS("edit:selectall");
      return s;
    }
    static ustring EDIT_UNDO() {
      static ustring s = WCHARS("edit:undo");
      return s;
    }
    static ustring EDIT_REDO() {
      static ustring s = WCHARS("edit:redo");
      return s;
    }

    static ustring EDIT_DELETE_NEXT() {
      static ustring s = WCHARS("edit:delete-next");
      return s;
    }
    static ustring EDIT_DELETE_PREV() {
      static ustring s = WCHARS("edit:delete-prev");
      return s;
    }
    static ustring EDIT_DELETE_WORD_NEXT() {
      static ustring s = WCHARS("edit:delete-word-next");
      return s;
    }
    static ustring EDIT_DELETE_WORD_PREV() {
      static ustring s = WCHARS("edit:delete-word-prev");
      return s;
    }

    static ustring EDIT_DELETE_LINE_START() { // from caret to line start
      static ustring s = WCHARS("edit:delete-line-start");
      return s;
    }

    static ustring EDIT_DELETE_LINE_END() { // from caret to line end
      static ustring s = WCHARS("edit:delete-line-end");
      return s;
    }

    static ustring NAVIGATE_BACKWARD() {
      static ustring s = WCHARS("navigate:backward");
      return s;
    }
    static ustring NAVIGATE_WORD_START() {
      static ustring s = WCHARS("navigate:word-start");
      return s;
    }
    static ustring NAVIGATE_FORWARD() {
      static ustring s = WCHARS("navigate:forward");
      return s;
    }
    static ustring NAVIGATE_WORD_END() {
      static ustring s = WCHARS("navigate:word-end");
      return s;
    }
    static ustring NAVIGATE_UP() {
      static ustring s = WCHARS("navigate:up");
      return s;
    }
    static ustring NAVIGATE_DOWN() {
      static ustring s = WCHARS("navigate:down");
      return s;
    }
    static ustring NAVIGATE_LINE_START() {
      static ustring s = WCHARS("navigate:line-start");
      return s;
    }
    static ustring NAVIGATE_LINE_END() {
      static ustring s = WCHARS("navigate:line-end");
      return s;
    }
    static ustring NAVIGATE_START() {
      static ustring s = WCHARS("navigate:start");
      return s;
    }
    static ustring NAVIGATE_END() {
      static ustring s = WCHARS("navigate:end");
      return s;
    }
  };

  /*struct animation_event : public event {
    enum _ { EVENT_GROUP = HANDLE_ANIMATION };
    uint start_time;
    uint current_time;
    uint end_time;
    uint duration;
    animation_event(element *trg) : event(trg, 0) {}

    virtual uint event_group() const { return EVENT_GROUP; }

    static element *get_parent(element *t) { return get_logical_parent(t); }

    virtual uint get_start_time() const override { return start_time; }
    virtual uint get_current_time() const override { return current_time; }
    virtual uint get_end_time() const override { return end_time; }
  };*/
}; // namespace html

#endif
