#include "html.h"

#define BOX_SHADOW_CACHE_CAPACITY 32

namespace html {

  float setup_fill_and_stroke(gool::graphics *gfx, const element*el, html::style *ps,
                              gool::sizef sz, bool filled, byte opacity,
                              float scale) {
    argb sc;
    if (ps->stroke_color.is_defined())
      sc = ps->stroke_color.to_argb();
    else
      sc = ps->font_color.to_argb();

    argb fc(0, 0, 0, 0);
    if (filled) fc = sc;
    if (ps->fill_color.is_defined()) fc = ps->fill_color.to_argb();

    if (opacity < 255) {
      fc.alfa = argb::channel_t(fc.alfa * uint(opacity) / 255);
      sc.alfa = argb::channel_t(sc.alfa * uint(opacity) / 255);
    }

    float sw = filled ? 0.0f : 1.0f;
    if (ps->stroke_width.is_defined())
      sw = (float)ps->stroke_width.pixels(ps->font_size, sz,false,gfx->get_resolution_provider()) * scale;

    if (sw > 0) {
      handle<solid_brush> br = new gool::solid_brush(sc);
      gfx->set_stroke(br, sw, gool::CAP_STYLE(ps->stroke_linecap.val()),
                      gool::LINE_JOIN(ps->stroke_linejoin.val()),
                      ps->stroke_miterlimit.val(4));
    }

    if (fc.alfa > 0) {
      handle<solid_brush> br = new gool::solid_brush(fc);
      gfx->set_fill(br);
    }
    return sw;
  }

  void element::commit_measure(view &v) {
    // auto cid = type_id();
    if (flags.in_setup_layout) return;
    // ??? v.to_update.update(&v);
    if (ldata->dim.y == 0 || ldata->dim.x == 0) {
      if (!state.popup()) {
        if (!is_visible(v)) return;
        drop_content_layout(&v);
        helement lp = layout_parent(v);
        if (lp) {
          helement _this = this;
          lp->drop_content_layout(&v); // parent may destroy this in drop_content_layout
          lp->commit_measure(v);
          if (_this->is_connected() && !_this->is_layout_valid()) {
            size dim = { _this->declared_width(v,lp->dim().x),_this->declared_height(v,lp->dim().y) };
            _this->set_width(v, dim.x);
            _this->set_height(v, dim.y);
          }
        }
        return;
      } else {
        drop_content_layout(&v);
        _commit_measure(v);
        return;
      }
    } else if (is_layout_valid())
      return;

    helement _this = this;
    document *pd = doc();
    if (pd) pd->_commit_measure(v);
    if (!_this->is_layout_valid()) _this->_commit_measure(v); // second attempt!
#ifdef _DEBUG
      // if(!is_layout_valid())
      //  this->dbg_report("commit_measure failure on ");
#endif
  }

  void element::_commit_measure(view &v) {
    hstyle cs = get_style(v);
    if (!state.popup() && cs->collapsed()) return;

    check_layout(v);

    if (/*!ldata->is_valid() &&*/ cs->display != display_inline) {
      // the best we can do here is to measure it in place
      size dim = ldata->dim;
      set_width(v, dim.x);
      set_height(v, dim.y); // - wrong: set_height(v,max(h, dim.y));
      // assert(ldata->is_valid());
    }
  }

  image *element::get_fore_image(view &v) {
    hstyle cs  = get_style(v);
    image *img = provide_fore_image(v);
    if (!img) return 0;
    image *mapped_img = 0;

    switch (cs->mapping.u.parts.foreground_image) {
    case mapping_left_to_right: mapped_img = img->mapped_left_to_right(); break;
    case mapping_top_to_right: mapped_img = img->mapped_top_to_right(); break;
    }
    return mapped_img ? mapped_img : img;
  }

  image *element::get_back_image(view &v) {
    hstyle cs = get_style(v);
#ifdef _DEBUG
    if (attr_id() == WCHARS("help")) cs = cs;
#endif

    image *img = provide_back_image(v);
    if (!img) return 0;
    image *mapped_img = 0;
    switch (cs->mapping.u.parts.background_image) {
    case mapping_left_to_right: mapped_img = img->mapped_left_to_right(); break;
    case mapping_top_to_right: mapped_img = img->mapped_top_to_right(); break;
    }
    return mapped_img ? mapped_img : img;
  }

  image *element::provide_fore_image(view &v) const {
    if (animator) {
      image *im = animator->provide_fore_image();
      if (im) return im;
    }

    ctl *pbh = behavior;
    while (pbh) {
      image *pim = pbh->get_fore_image(v, const_cast<element *>(this));
      if (pim) return pim;
      pbh = pbh->next;
    }
    return get_style()->fore_image.id.img();
  }

  image *element::provide_back_image(view &v) const {
    if (animator) {
      image *im = animator->provide_back_image();
      if (im) return im;
    }
    return get_style()->back_image.id.img();
  }

  bool element::compute_mtx(view &v, affine_mtx_f &mtx, point pos) {
    hstyle cs = get_style(v);
    //if (!ldata || !cs->transforms) return false;
    if (!ldata->xctx) {
      ldata->xctx = new trans_ctx();
      if (!ldata->xctx) return false;
    } else
      ldata->xctx->clear();
    cs->transforms->apply(v, this, ldata->xctx->mtx);
    rect   bb = border_box(v);
    pointf s;
    if (c_style->transform_origin_x.is_undefined())
      s.x = float(bb.width()) / 2.f;
    else
      s.x = pixels(v, this, c_style->transform_origin_x,bb.size()).width_f();
    if (c_style->transform_origin_y.is_undefined())
      s.y = float(bb.height()) / 2.f;
    else
      s.y = pixels(v, this, c_style->transform_origin_y,bb.size()).height_f();
    s += pos + bb.s;
    mtx.translate(-s.x, -s.y);
    mtx *= ldata->xctx->mtx;
    mtx *= affine_translation_f(s.x, s.y);
    return true;
  }

  void element::draw(view &v, graphics *pg, point pos, bool check_animator) {
    if (v.disable_element_draw(pg, this))
      return;

    hstyle cs = get_style(v);
    p_drawn_style = d_style;

    //if (cs->collapsed() && !state.popup())
    //  return;

#ifdef _DEBUG
    if (is_id_test())
      pos = pos;
    if (tag == tag::T__MARKER)
      pos = pos;
#endif

    if (!is_it_drawable(v))
      return;

    if (pg->drawing_backdrop && cs->backdrop_filter.has_items())
      return;

    size dim;

    auto_state<element*>  _0(drawing_element, is_anonymous_text_block() ? parent.ptr() : this  );
    auto_state<style *>   _1(pg->current_style, cs);
    auto_state<element *> _2(pg->current_element, is_anonymous_text_block() ? parent.ptr() : this);

    check_layout(v);

#ifdef DEBUG_DRAW_PIPELINE
    pg->set_tag(this);
#endif

    if (!ldata->is_valid())
    {
      // the best we can do here is to measure it in place
      dim         = ldata->dim;
      /* VERY WRONG, breaks flexes: element *lp = layout_parent(v);
      if (lp) {
        measure_borders_x(v, lp->ldata->dim);
        measure_borders_y(v, lp->ldata->dim);
      }*/
      set_width(v, dim.x);
      set_height(v, dim.y);
    } else
      dim = ldata->dim;

    //ldata->used_dim = ldata->dim;

    if (!ldata->is_valid()) {
#ifdef _DEBUG
      this->dbg_report("attempt to draw incomplete element");
#endif
      // assert(false);
      // return;
    }

    if (airborn && airborn->dim.x.is_defined()) {
      assert(dim.y == airborn->dim.y);
    }

    byte opacity = byte(cs->opacity);
    if (opacity == 0) 
      return;

    if (cs->has_outline()) 
      pg->request_overlay();

    function<bool(gool::filter_graph_builder *)> filter_producer;

    if (cs->filter.has_items() && v.app->supports_filters()) {
      filter_producer = [&](gool::filter_graph_builder *target) -> bool {
        return produce_filter_graph(v, this, cs->filter.elements(), target);
      };
    }

    auto perform_draw = [&]() {
      if (filter_producer && v.app->supports_filters()) {
        gool::layer _(pg, rendering_box(v) + pos, opacity, filter_producer);
        do_draw(v, pg, pos, check_animator);
      } else if (opacity == 255) {
        do_draw(v, pg, pos, check_animator);
      } else if (opacity) {
        gool::layer _(pg, rendering_box(v) + pos, opacity);
        do_draw(v, pg, pos, check_animator);
      }
    };

    // gool::state _(pg, ldata->xctx? &ldata->xctx->mtx :0);
    if (cs->transforms) {
      affine_mtx_f mtx;
      compute_mtx(v, mtx, pos);
      if(mtx.is_valid()) {
        gool::state  _(pg);
        pg->transform(mtx);
        perform_draw();
      }
    } else
      perform_draw();

    flags.was_drawn = 1;
  }

  void element::draw_background(view &v, graphics *pg, point pos) {

    ldata->zctx.draw(v, pg, pos, this, false);

    if (c_style->has_background() /*|| behavior ??? */) {
      rect r;
      switch (c_style->back_image.clip) {
      case clip_content_box: r = ldata->content_box(); break;
      case clip_padding_box: r = ldata->padding_box(); break;
      case clip_margin_box: r = ldata->margin_box(); break;
      default: r = ldata->border_box(); break;
      }
      r += pos;
      c_style->draw_background(v, pg, r, this, pos);
    }
    if (c_style->display == display_list_item && c_style->list_style_type.val() != list_style_none)
      draw_bullet(v, pg, pos);
    if (has_shadow_dom())
      draw_shadow_dom(v, pg, pos, -1);
  }

  void element::draw_shadow_dom(view &v, graphics *pg, point pos, int level)
  {
    if (helement marker = get_marker()) {
      style* mcs = marker->get_style(v);
      int mlevel = mcs->z_index.val(-1);
      if (mlevel == level) {
        replace__marker_element(v, marker);
        marker->draw(v, pg, pos + marker->pos());
      }
    }
    if (helement shade = get_shade()) {
      style* mcs = shade->get_style(v);
      int mlevel = mcs->z_index.val(-1);
      if (mlevel == level) {
        replace__shadow_element(v, shade);
        shade->draw(v, pg, pos + shade->pos());
      }
    }
  }

  void element::draw_content(view &v, graphics *pg, point pos, bool clip) {
#ifdef _DEBUG
    dbg_report("drawing on unclassified element ");
#endif
    //assert(false);
  }
  void element::draw_foreground(view &v, graphics *pg, point pos) {
    if (c_style->has_foreground() || is_fore_image_provider(v) || behavior) {
      rect r;
      switch (c_style->fore_image.clip) {
        case clip_content_box: r = ldata->content_box(); break;
        case clip_padding_box: r = ldata->padding_box(); break;
        case clip_margin_box: r = ldata->margin_box(); break;
        default: r = ldata->border_box(); break;
      }
      r += pos;
      c_style->draw_foreground(v, pg, r, this, pos);
    }
    if (ldata->fctx) ldata->fctx->draw(v, pg, scroll_translate(v, pos));
    if (has_shadow_dom())
      draw_shadow_dom(v, pg, pos, +1);
  }

  void element::draw_border(view &v, graphics *pg, point pos) {
    if (!c_style->has_borders()) {
      if (c_style->box_shadow.is_defined()) {
        rect r = ldata->border_box() + pos;
        c_style->draw_box_shadow(v, pg, r, ldata->used_border_width(), this);
      }
      return;
    }
    rect r = ldata->border_box() + pos;
    if (c_style->box_shadow.is_defined())
      c_style->draw_box_shadow(v, pg, r, ldata->used_border_width(), this);
    c_style->draw_border(v, pg, r, ldata->used_border_width(), this, pos);
  }
  void element::draw_outline(view &v, graphics *pg, point pos) {

    hstyle cs = c_style;
    if (cs->has_outline()) {
      if (state.popup() && pg->get_drawing_root() == v.doc()) return;

      rect rcborder = border_box(v) + pos;
      cs->draw_outline(v, pg, rcborder, this, pos);
    }
    if (has_shadow_dom())
      draw_shadow_dom(v, pg, pos, +2);
  }

  void style::background_outline(view &v, graphics *gfx, rect r,
                                 gool::rect &rect, handle<gool::path> &path, element* site) {
    if (has_rounded_corners()) {
      size rtl, rtr, rbr, rbl;
      get_rounded_corners(v, site, rtl, rtr, rbr, rbl, r.size());

      path = v.app->create_path();
      if (path)
        path->add_round_rect(r.s, r.dimension(), rtl, rtr, rbr, rbl);
    } else
      rect = r;
  }

  void style::draw_background(view &v, graphics *gfx, const rect &r, element *site, point pos) {
    // if( !clip_overflow() )
    // WRONG, see layered: _draw_background( v, gfx, r, site ); // already
    // clipped
    // else
    {
      handle<gool::path> path;
      gool::rect         box;

      background_outline(v, gfx, r, box, path,site);

      if (path) {
        if (site && site->has_shadow_dom()) {
          gool::layer _(gfx, path);
          _draw_background(v, gfx, r, site);
          site->draw_shadow_dom(v, gfx, pos, 0);
        } else if (!_draw_background(v, gfx, path, site)) {
          gool::layer _(gfx, path);
          _draw_background(v, gfx, r, site);
        }
      }
      else {
        _draw_background(v, gfx, r, site);
        if (site && site->has_shadow_dom())
          site->draw_shadow_dom(v, gfx, pos, 0);
      }
    }
  }

  linef compute_cover(float a, rect rc) {
    pointf center = rc.pointOf(5);

    float ro = distance(center, pointf(rc.pointOf(3))); // center / e distance;

    pointf vb(std::cos(a) * ro, std::sin(a) * ro);
    pointf s, e, vp;

    pointf va;
    if(a < 0)
      va = pointf(rc.pointOf(1)) - center;
    else
      va = pointf(rc.pointOf(3)) - center;

    float l = (va * vb) / (vb * vb); // length of projection of va on vb, normalized to 0..1

    vp = l * vb;

    if (l > 0) {
      s = center - vp;
      e = center + vp;
    } else {
      s = center + vp;
      e = center - vp;
    }
    return linef(s, e);
  }

  linef compute_beam(float a, rectf rc, pointf origin) {
    pointf      corner;
    const float PI = real_traits<float>::pi();
    a              = fmodf(a, 2 * PI);
    if (a < PI / 2)
      corner = rc.pointOf(3);
    else if (a < PI)
      corner = rc.pointOf(1);
    else if (a < 3 * PI / 2)
      corner = rc.pointOf(7);
    else
      corner = rc.pointOf(9);

    float ro = distance(origin, corner); // center / corner distance;

    pointf va = corner - origin;
    pointf vb(std::cos(a) * ro, std::sin(a) * ro);
    pointf s, e, vp;

    float l = (va * vb) / (vb * vb); // length of projection of va on vb
    vp      = l * vb;
    // if( l > 0 )
    {
      s = origin;
      e = origin + vp;
    }
    // else
    //  { s = origin + vp; e = origin - vp; }
    return linef(s, e);
  }

  /*linef compute_cover(float a, rectf rc, pointf pos)
  {
     // dances with tambourines to find out intersection point of the
beam(lb.start,lb.angle) and bounding rc rectangle linef l(pos,pos); l.end.x =
pos.x + cosf(a) * (rc.width() + rc.height()); l.end.y = pos.y + sinf(a) *
(rc.width() + rc.height());

    switch(
gool::geom::intersect_at(l.s,l.end,rc.pointOf(7),rc.pointOf(9),l.end))
    {
      case gool::geom::DO_INTERSECT: goto FOUND;
      case gool::geom::COLLINEAR: l.end = l.s != rc.pointOf(7)?
rc.pointOf(7):rc.pointOf(9); goto FOUND;
    }
    switch(
gool::geom::intersect_at(l.s,l.end,rc.pointOf(7),rc.pointOf(1),l.end))
    {
      case gool::geom::DO_INTERSECT: goto FOUND;
      case gool::geom::COLLINEAR: l.end = l.s != rc.pointOf(7)?
rc.pointOf(7):rc.pointOf(1); goto FOUND;
    }
    switch(
gool::geom::intersect_at(l.s,l.end,rc.pointOf(9),rc.pointOf(3),l.end))
    {
      case gool::geom::DO_INTERSECT: goto FOUND;
      case gool::geom::COLLINEAR: l.end = l.s != rc.pointOf(9)?
rc.pointOf(9):rc.pointOf(3); goto FOUND;
    }
    switch(
gool::geom::intersect_at(l.s,l.end,rc.pointOf(1),rc.pointOf(3),l.end))
    {
      case gool::geom::DO_INTERSECT: goto FOUND;
      case gool::geom::COLLINEAR: l.end = l.s != rc.pointOf(1)?
rc.pointOf(1):rc.pointOf(3); goto FOUND;
    }
FOUND:
    return l;
  }*/

  void populate_stops(gool::gradient_brush &lb, const html::gradient *gr,
                      element *site) {
    const style *cs     = site ? site->c_style : element::null_style;
    auto         hstops = gr->stops();
    for (uint n = 0; n < hstops.length; ++n)
      lb.add_stop(hstops[n].position, hstops[n].color.to_argb());
  }

  handle<gool::brush> create_brush(view &v, graphics *gfx, const rect &r, element *site, const linear_gradient *plg)
  {
    gool::linear_brush* plb = new linear_brush();

    populate_stops(*plb, plg, site);

    point_v vdim;
    point_v vpos;

    vdim.x = plg->dim.x;
    vdim.y = plg->dim.y;
    vpos.x = plg->pos.x;
    vpos.y = plg->pos.y;

    sizef dim;

    if (vdim.x.is_defined() && vdim.y.is_defined()) {
      dim = sizef(pixels(v, site, vdim.x, r.size()).width_f(),
                  pixels(v, site, vdim.y, r.size()).height_f());
      if (vpos.x.is_percent()) {
        float percent = float(vpos.x.percent());
        float xi = (abs(dim.x) * percent) / 100.f;
        float xpos = (r.width() * percent) / 100.f;
        if (dim.x > 0) {
          plb->start.x = r.s.x + xpos - xi + 0.5f;
          plb->end.x = plb->start.x + dim.x;
        }
        else {
          plb->end.x = r.s.x + xpos - xi + 0.5f;
          plb->start.x = plb->end.x - dim.x;
        }
      }
      else {
        plb->start.x = r.s.x + pixels(v, site, vpos.x, r.size()).width_f() - 0.5f;
        plb->end.x = plb->start.x + pixels(v, site, vdim.x, r.size()).width_f();
      }
      if (vpos.y.is_percent()) {
        float percent = float(vpos.y.percent());
        float yi = (abs(dim.y) * percent) / 100.f;
        float ypos = (r.height() * percent) / 100.f;
        if (dim.y > 0) {
          plb->start.y = r.s.y + ypos - yi + 0.5f;
          plb->end.y = plb->start.y + dim.y;
        }
        else {
          plb->end.y = r.s.y + ypos - yi + 0.5f;
          plb->start.y = plb->end.y - dim.y;
        }
      }
      else {
        plb->start.y = r.s.y + pixels(v, site, vpos.y, r.size()).height_f() - 0.5f;
        plb->end.y = plb->start.y + pixels(v, site, vdim.y, r.size()).height_f();
      }

    }
    else if (vpos.x.is_defined() && vpos.y.is_defined()) {
      rectf rf = r;
      plb->start.x = rf.s.x + pixels(v, site, vpos.x, r.size()).width_f();
      plb->start.y = rf.s.y + pixels(v, site, vpos.y, r.size()).height_f();
      pointf center = rf.pointOf(5) + pointf(0.5f, 0.5f);
      plb->end = center + (center - plb->start);
    }
    if (plg->angle.is_defined()) {
      if (vpos.x.is_undefined() && vpos.y.is_undefined()) {
        linef l = compute_cover(plg->angle, r);
        plb->start = l.s;
        plb->end = l.e;
      }
      else if (vdim.x.is_defined() && vdim.y.is_defined()) {
        rectf rf(plb->start, plb->end);
        if (rf.s.x > rf.e.x) swap(rf.s.x, rf.e.x);
        if (rf.s.y > rf.e.y) swap(rf.s.y, rf.e.y);
        pointf center = rf.pointOf(5) + pointf(0.5f, 0.5f);
        plb->end.x = center.x + cosf(plg->angle) * rf.width() / 2;
        plb->end.y = center.y + sinf(plg->angle) * rf.height() / 2;
        plb->start = center + (center - plb->end);
      }
      else {
        linef l = compute_beam(plg->angle, r, plb->start);
        plb->start = l.s;
        plb->end = l.e;
      }
    }

    switch (site->get_style(v)->mapping.u.parts.background_image) {
    case mapping_left_to_right: swap(plb->start.x, plb->end.x); break;
    case mapping_top_to_right:
      // swap(plb->start.x,plb->end.x);
#pragma TODO("gradient: mapping_top_to_right")
      break;
    }

    return plb;
  }

  handle<gool::brush> create_brush(view &v, graphics *gfx, const rect &r, element *site, const radial_gradient *prg) {
    gool::radial_brush* prb = new gool::radial_brush();

    populate_stops(*prb, prg, site);

    point_v center;
    point_v radius;

    
    center.x = prg->center.x;
    center.y = prg->center.y;
    radius.x = prg->radius.x;
    radius.y = prg->radius.y;
  
    prb->center.x = r.s.x + pixels(v, site, center.x,r.size()).width_f() - 0.5f;
    prb->center.y = r.s.y + pixels(v, site, center.y,r.size()).height_f() - 0.5f;

    if (prg->shape == radial_gradient::ELLIPSE) {
      switch (prg->size) {
      case radial_gradient::CLOSEST_CORNER:
      case radial_gradient::CLOSEST_SIDE:
        prb->radius.x = min(prb->center.x - r.left(), r.right() - prb->center.x);
        prb->radius.y = min(prb->center.y - r.top(), r.bottom() - prb->center.y);
        if (prg->shape == radial_gradient::CIRCLE)
          prb->radius.x = prb->radius.y = min(prb->radius.x, prb->radius.y);
        break;
      case radial_gradient::FARTHEST_CORNER:
      case radial_gradient::FARTHEST_SIDE:
        prb->radius.x = max(prb->center.x - r.left(), r.right() - prb->center.x);
        prb->radius.y = max(prb->center.y - r.top(), r.bottom() - prb->center.y);
        if (prg->shape == radial_gradient::CIRCLE)
          prb->radius.x = prb->radius.y = max(prb->radius.x, prb->radius.y);
        break;
      }
      if (prg->size == radial_gradient::CLOSEST_CORNER ||
        prg->size == radial_gradient::FARTHEST_CORNER) {
        const float m_sqrt2 = 1.41421356f;
        prb->radius.x *= m_sqrt2;
        prb->radius.y *= m_sqrt2;
      }
    }
    else if (prg->shape == radial_gradient::CIRCLE) {
      float rad = 0;
      switch (prg->size) {
      case radial_gradient::CLOSEST_CORNER:
      case radial_gradient::FARTHEST_CORNER: {
        float d7 = gool::distance(prb->center, pointf(r.pointOf(7)));
        float d9 = gool::distance(prb->center, pointf(r.pointOf(9)));
        float d1 = gool::distance(prb->center, pointf(r.pointOf(1)));
        float d3 = gool::distance(prb->center, pointf(r.pointOf(3)));

        if (prg->size == radial_gradient::CLOSEST_CORNER)
          rad = float(min(min(d7, d9), min(d1, d3)));
        else
          rad = float(max(max(d7, d9), max(d1, d3)));
      } break;
      case radial_gradient::CLOSEST_SIDE: {
        float rh = min(prb->center.x - r.left(), r.right() - prb->center.x);
        float rv = min(prb->center.y - r.top(), r.bottom() - prb->center.y);
        rad = min(rh, rv);
      } break;
      case radial_gradient::FARTHEST_SIDE: {
        float rh = max(prb->center.x - r.left(), r.right() - prb->center.x);
        float rv = max(prb->center.y - r.top(), r.bottom() - prb->center.y);
        rad = max(rh, rv);
      } break;
      }
      prb->radius.x = prb->radius.y = rad;
    }
    else {
      prb->radius.x = pixels(v, site, radius.x, r.size()).width_f() - 0.5f;
      prb->radius.y = pixels(v, site, radius.y, r.size()).height_f() - 0.5f;
    }

    switch (site->get_style(v)->mapping.u.parts.background_image) {
    case mapping_left_to_right:
      // swap(lb.start.x,lb.end.x);
      prb->center.x = r.s.x + r.width() - (prb->center.x - r.s.x);
      break;
    case mapping_top_to_right:
      // swap(lb.start.x,lb.end.x);
#pragma TODO("gradient: mapping_top_to_right")
      break;
    }
    return prb;
  }

  handle<gool::brush> create_brush(view &v, graphics *gfx, const rect &r, element *site, const gradient *prg)
  {
    switch (prg->type()) {
    case gradient::LINEAR:
      return create_brush(v, gfx, r, site, static_cast<const linear_gradient*>(prg));
    case gradient::RADIAL:
      return create_brush(v, gfx, r, site, static_cast<const radial_gradient*>(prg));
    }
    return nullptr;
  }

  void element::do_draw_backdrop(view &v, graphics *pg, const rect& r, const filter_v& backdrop_filter)
  {
    // find backdrop root
    element* backdrop_root = v.doc();
    for (element* t = this->layout_parent(v); t; t = t->layout_parent(v))
    {
      backdrop_root = t;
      const style* ps = t->get_style(v);
      if (ps->has_solid_background()) break;
      unsigned opacity = unsigned(ps->opacity);
      if (opacity == 0) return;
      if (opacity < 255) break;
      if (ps->filter.has_items()) break;
      if (ps->backdrop_filter.has_items()) break;
    }
    assert(backdrop_root);

    pg->drawing_backdrop = true;
    {
      auto filter_producer = [&](gool::filter_graph_builder *target) -> bool
      {
        return produce_filter_graph(v, this, backdrop_filter.elements(), target);
      };
      gool::layer _(pg, r, 255, filter_producer);
      point pos = backdrop_root->view_pos(v);
      backdrop_root->draw(v, pg, pos);
    }
    pg->drawing_backdrop = false;

  }

  void style::_draw_background(view &v, graphics *gfx, const rect &r, element *site) {

#ifdef _DEBUG
    if (site && site->is_id_test())
      site = site;
#endif
    if (backdrop_filter.has_items() && v.app->supports_filters())
      site->do_draw_backdrop(v, gfx, r, backdrop_filter);

    if (has_background_color()) { gfx->fill(back_color.to_argb(), r); }
    if (has_background_gradient() && !r.empty()) {
      rect rr(r.size());
      handle<brush> br = create_brush(v, gfx, r, site, back_gradient);
      if (br) gfx->fill(br, r);
    }
    draw_image(v, gfx, this->back_image, r, false, site);
  }

  bool style::_draw_background(view &v, graphics *gfx, const path* p, element *site /*can be null*/)
  {
    if(back_image.id.img())
      return false;

    if (backdrop_filter.has_items() && v.app->supports_filters())
      site->do_draw_backdrop(v, gfx, p->bounds(), backdrop_filter);

    if (has_background_color()) {
      gool::state _(gfx);
      gfx->set_fill(back_color.to_argb());
      gfx->draw_path(p, false, true);
    }
    if (has_background_gradient()) {
      gool::state _(gfx);
      rect rr = p->bounds();
      handle<brush> br = create_brush(v, gfx, rr, site, back_gradient);
      if (br) {
        gfx->set_fill(br);
        gfx->draw_path(p, false, true);
      }
    }
    /* // if( has_background_image() )
    draw_image(v, gfx, this->back_image, r, false, site); */
    return true;
  }

  void style::draw_border(view &v, graphics *gfx, const rect &rcborder,
                          const rect &border_width, element *site, point pos) {


    if (has_rounded_corners()) {
      rect r  = rcborder;
      rect ir = r;
      ir <<= border_width;

      size rtl, rtr, rbr, rbl;
      get_rounded_corners(v, site,rtl, rtr, rbr, rbl, r.size());

      size irtl = (rtl - size(border_width.pointOf(7))).normalize(),
           irtr = (rtr - size(border_width.pointOf(9))).normalize(),
           irbr = (rbr - size(border_width.pointOf(3))).normalize(),
           irbl = (rbl - size(border_width.pointOf(1))).normalize();

      rectf fr  = r;
      rectf fir = ir;

      if (gfx->needs_half_pixel_adjustment()) {
        fr -= 0.5f;
        fir -= 0.5f;
      }

      handle<gool::path> path = v.app->create_path();
      path->add_round_rect(fr.s, fr.dimension(), rtl, rtr, rbr, rbl);
      path->add_round_rect(fir.s, fir.dimension(), irtl, irtr, irbr, irbl);

      argb c = (border_color[0].is_defined()
                   ? border_color[0].val(font_color)
                   : (border_color[1].is_defined()
                          ? border_color[1].val(font_color)
                          : (border_color[2].is_defined()
                                 ? border_color[2].val(font_color)
                                 : (border_color[3].val(font_color))))).to_argb();
      gfx->fill(c, path);
    } else
      _draw_border(v, gfx, rcborder, border_width, site);
  }

  inline size  noneg(size sz) { return size(max(0, sz.x), max(0, sz.y)); }
  inline point noneg(point sz) { return point(max(0, sz.x), max(0, sz.y)); }

  inline size positive(size sz) {
    return size(max(1, abs(sz.x)), max(1, abs(sz.y)));
  }

  inline point upoint(point p1, point p2) { return point(max(p1.x, p2.x), max(p1.y, p2.y)); }

  inline rect inner_box(rect rbox, size rtl, size rtr, size rbr, size rbl) {
    rbox.s.x += max(rtl.x, rbl.x);
    rbox.s.y += max(rtl.y, rtr.y);
    rbox.e.x -= max(rtr.x, rbr.x);
    rbox.e.y -= max(rbl.y, rbr.y);
    return rbox;
  }
  inline rect outer_box(rect rbox, size rtl, size rtr, size rbr, size rbl) {
    rbox.s.x -= max(rtl.x, rbl.x);
    rbox.s.y -= max(rtl.y, rtr.y);
    rbox.e.x += max(rtr.x, rbr.x);
    rbox.e.y += max(rbl.y, rbr.y);
    return rbox;
  }



  handle<shadow_bitmap> generate_shadow_outer(view &v, const rect &box,
                                              const box_shadow_params &params,
                                              bool rounded,
                                              bool gen_expandable)
  {
    int radius = params.radius + params.distance;

    float k = 1.0f;
    if (params.distance > 0)
      k = float(params.radius + params.distance) /
          max(float(params.radius), 0.25f);
    else if (params.distance < 0)
      k = max(float(params.radius), 0.25f) /
          float(params.radius - params.distance);

    size offset = params.offset;

    size blur_size = radius;

    rect abox = params.dim; // box.size();
    rect bbox = abox;
    bbox += offset;

    rect abox_exp = abox;
    abox_exp >>= blur_size;
    rect bbox_exp = bbox;
    bbox_exp >>= blur_size;

    rect sbox = abox_exp | bbox_exp;

    handle<shadow_bitmap> bmp = new shadow_bitmap(sbox.size());

    if (!bmp) return nullptr;

    argb w(255, 255, 255, 255);
    argb z(0, 0, 0, 0);

    size rtl = params.bradius[0], rtr = params.bradius[1],
         rbr = params.bradius[2], rbl = params.bradius[3];

    {
      handle<graphics> gfx = v.app->create_bitmap_bits_graphics(bmp, z);
      if (!gfx) return nullptr;

      gfx->offset(-sbox.s);

      if (rounded) {
        handle<gool::path> path = v.app->create_path();
        // gool::normailize_round_rect(boxb.size(),rtl,rtr,rbr,rbl);
        path->add_round_rect(bbox.s, bbox.size(), rtl, rtr, rbr, rbl);
        gfx->fill(w, path);
        if (gen_expandable) {

          bmp->section_defs.top = SRM_TILE;
          bmp->section_defs.bottom = SRM_TILE;
          bmp->section_defs.left = SRM_TILE;
          bmp->section_defs.right = SRM_TILE;

          rect zbox(0,0,0,0);
          rect rbox = outer_box(zbox, rtl, rtr, rbr, rbl);
          rbox >>= positive(offset);
          rbox >>= radius * 2;
          point m1 = zbox.s - rbox.s;
          point m2 = rbox.e - zbox.e;

          bmp->section_defs.margins.s = upoint(m1,m2);
          bmp->section_defs.margins.e = bmp->section_defs.margins.s;
        }
      } else {
        gfx->fill(w, bbox);
        if (gen_expandable) {
          bmp->section_defs.top = SRM_TILE;
          bmp->section_defs.bottom = SRM_TILE;
          bmp->section_defs.left = SRM_TILE;
          bmp->section_defs.right = SRM_TILE;

          rect zbox(0, 0, 0, 0);
          rect rbox = outer_box(zbox, rtl, rtr, rbr, rbl);
          rbox >>= positive(offset);
          rbox >>= radius * 2;
          point m1 = zbox.s - rbox.s;
          point m2 = rbox.e - zbox.e;

          bmp->section_defs.margins.s = upoint(m1, m2);
          bmp->section_defs.margins.e = bmp->section_defs.margins.s;
        }

      }
    }

#ifdef USE_CGX
    bool mirrored = mirrored_bitmaps();
    if(mirrored)
      bmp->top_to_bottom_inplace();
#endif

    array<argb> pixels = bmp->_pixels;

    bmp->blur(size(radius));

    size         dim = bmp->dim();
    gool::pixmap src(pixels.head(), dim);
    gool::pixmap dst(bmp->_pixels.head(), dim);

    rect cr  = dim;
    rect crs = cr - offset;
    crs &= cr;
    crs += offset;

    for (int y = crs.s.y; y < crs.e.y; ++y)
      for (int x = crs.s.x; x < crs.e.x; ++x) {
        uint  src_alpha = src.pixel(x, y).alfa;
        argb &dstp      = dst.pixel(x - offset.x, y - offset.y);

        if (src_alpha == 255)
          dstp = z;
        else if (src_alpha == 0)
          continue;
        else {
          byte a     = argb::channel_t((dstp.alfa * (255 - src_alpha)) >> 8);
          dstp.alfa  = a; // premultipled white.
          dstp.red   = a;
          dstp.green = a;
          dstp.blue  = a;
        }
      }
    // colorize
    {
      argb *pd  = bmp->_pixels.head();
      argb *pde = bmp->_pixels.tail();
      for (; pd < pde; ++pd) {
        byte a = (byte)min(uint(pd->red * k), 255u);
        *pd    = params.color.opacity(a).premultiply();
      }
    }

    bmp->drop_cache();
#ifdef USE_CGX
    if(mirrored)
      bmp->top_to_bottom_inplace();
#endif

    bmp->offset_origin = sbox.s - abox.s;
    bmp->offset_corner = abox.e - sbox.e;

    return bmp;
  }

  handle<shadow_bitmap> generate_shadow_inner(view &v, const rect &,
                                              const rect &border_width,
                                              const box_shadow_params &params,
                                              bool rounded,
                                              bool gen_expandable) {

    int radius = params.radius + params.distance;
    // int distance = 0;//params.distance;
    float k = 1.0f;
    if (params.distance > 0)
      k = float(params.radius + params.distance) /
          max(float(params.radius), 0.25f);
    else if (params.distance < 0)
      k = max(float(params.radius), 0.25f) /
          float(params.radius - params.distance);


    size offset = params.offset;

    size blur_size = radius;

    rect ibox = params.dim;
    ibox <<= border_width;

    rect obox = params.dim;
    obox >>= positive(offset);

    if (obox.empty())
      return nullptr;
    
    handle<shadow_bitmap> bmp = new shadow_bitmap(obox.size());
    if (!bmp) return nullptr;
        
    argb w(255, 255, 255, 255);
    argb w1(255, 0, 255, 255);
    argb z(0, 0, 0, 0);

    {
      handle<graphics> gfx = v.app->create_bitmap_bits_graphics(bmp, z);
      if (!gfx) return nullptr;

      gfx->offset(-obox.s);

      if (rounded) {
        rect ro = obox;
        rect ri = ibox;

        size irtl = (params.bradius[0] - size(border_width.pointOf(7))).normalize(),
             irtr = (params.bradius[1] - size(border_width.pointOf(9))).normalize(),
             irbr = (params.bradius[2] - size(border_width.pointOf(3))).normalize(),
             irbl = (params.bradius[3] - size(border_width.pointOf(1))).normalize();

        gool::normailize_round_rect(ri.size(), irtl, irtr, irbr, irbl);

        handle<gool::path> path = v.app->create_path();
        path->add_rect(ro);
        path->add_round_rect(ri.s, ri.dimension(), irtl, irtr, irbr, irbl);
        gfx->fill(w, path);

        if (gen_expandable) {
          rect box = params.dim;
          bmp->section_defs.top = SRM_TILE;
          bmp->section_defs.bottom = SRM_TILE;
          bmp->section_defs.left = SRM_TILE;
          bmp->section_defs.right = SRM_TILE;
          bmp->section_defs.center = SRM_UNDEFINED;
          rect rbox = inner_box(box, irtl, irtr, irbr, irbl);
          rbox <<= border_width;
          rbox <<= positive(offset);
          rbox <<= positive(offset);
          rbox <<= radius * 2;
          point m1 = rbox.s - box.s;
          point m2 = box.e - rbox.e;
          bmp->section_defs.margins.s = upoint(m1, m2);
          bmp->section_defs.margins.e = bmp->section_defs.margins.s;
        }

      } else {
        rect ro = obox;
        rect ri = ibox;

        handle<gool::path> path = v.app->create_path();
        path->add_rect(ro);
        path->add_rect(ri);
        gfx->fill(w, path);

        if (gen_expandable) {
          rect box = params.dim;
          bmp->section_defs.top = SRM_TILE;
          bmp->section_defs.bottom = SRM_TILE;
          bmp->section_defs.left = SRM_TILE;
          bmp->section_defs.right = SRM_TILE;
          bmp->section_defs.center = SRM_UNDEFINED;
          rect rbox = box;
          rbox <<= border_width;
          rbox <<= positive(offset);
          rbox <<= positive(offset);
          rbox <<= radius * 2;
          point m1 = rbox.s - box.s;
          point m2 = box.e - rbox.e;
          bmp->section_defs.margins.s = upoint(m1, m2);
          bmp->section_defs.margins.e = bmp->section_defs.margins.s;
        }

      }
    }

#ifdef USE_CGX
      bool mirrored = mirrored_bitmaps();
      if(mirrored)
          bmp->top_to_bottom_inplace();
#endif

    array<argb> pixels = bmp->_pixels;

    bmp->blur(size(radius));

    array<argb> nonshifted_pixels = bmp->_pixels;

    size         dim = bmp->dim();
    gool::pixmap src(nonshifted_pixels.head(), dim);
    gool::pixmap dst(bmp->_pixels.head(), dim);

    rect crs = rect(dim);
    rect crd = crs + offset;
    crd &= crs;
    crd -= offset;
    for (int y = crs.s.y; y < crs.e.y; ++y)
      for (int x = crs.s.x; x < crs.e.x; ++x) {
        point pt(x - offset.x, y - offset.y);
        argb  pc = crd.contains(pt) ? src.pixel(pt.x, pt.y) : w1;
        dst.pixel(x, y, pc);
      }

    {
      const argb *po  = pixels.head();
      argb *      pd  = bmp->_pixels.head();
      argb *      pde = bmp->_pixels.tail();
      for (; pd < pde; ++pd, ++po)
        if (po->green == 255 && po->red == 255)
          *pd = z;
        else {
          unsigned a = (byte)min(uint(pd->red * k), 255u);
          *pd = params.color.opacity(byte((uint(a) * (255 - po->alfa)) / 255))
                    .premultiply();
        }
    }

    bmp->drop_cache();

    bmp->offset_origin = -(ibox.s - obox.s - border_width.s);
    bmp->offset_corner = (obox.s - ibox.s + border_width.e);

#ifdef USE_CGX
      if(mirrored)
          bmp->top_to_bottom_inplace();
#endif
    return bmp;
  }

  handle<shadow_bitmap> generate_shadow(view &v, const rect &box,
                                        const rect &             border_width,
                                        const box_shadow_params &params,
                                        bool                     rounded,
                                        bool                     gen_expandable )
  {
    return params.inner
               ? generate_shadow_inner(v, box, border_width, params, rounded, gen_expandable)
               : generate_shadow_outer(v, box, params, rounded, gen_expandable);
  }

  void style::draw_box_shadow(view &v, graphics *sf, const rect &rcborder,
                              const rect &border_width, element *site) {
    for (shadow_def *it = box_shadow; it; it = it->next) {

      box_shadow_params params;
      bool              rounded = false;
      if (has_rounded_corners()) {
        get_rounded_corners(v, site, params.bradius[0], params.bradius[1],
                            params.bradius[2], params.bradius[3],
                            rcborder.size());
        rounded = true;
      }

      params.inner    = it->inset != 0;
      params.distance = pixels(v, site, it->spread, rcborder.size()).width();
      params.radius   = pixels(v, site, it->radius, rcborder.size()).width();
      params.offset.x = pixels(v, site, it->offset_x, rcborder.size()).width();
      params.offset.y = pixels(v, site, it->offset_y, rcborder.size()).height();
      params.color    = it->color.to_argb();

      int radius = params.radius + params.distance; // effective radius

      size cap_size = rcborder.size();
      bool gen_expandable = false;

      if (params.inner) {
        rect rc = rect(0, 0, 0, 0);
        rc >>= border_width;
        rc >>= radius * 2;
        rc >>= positive(params.offset);
        rc >>= positive(params.offset);
        rc = outer_box(rc, params.bradius[0], params.bradius[1], params.bradius[2], params.bradius[3]);
        size sz = rc.size() + size(8, 8); // 8x8 is the size of inner section
        if (cap_size.x >= sz.x && cap_size.y >= sz.y) {
          cap_size = sz;
          gen_expandable = true;
        }
      }
      else {
        rect rc = rect(0,0,0,0);
        rc >>= radius * 2;
        rc >>= positive(params.offset);
        rc = outer_box(rc, params.bradius[0], params.bradius[1], params.bradius[2], params.bradius[3]);
        size sz = rc.size() + size(radius, radius); // 8x8 is the size of inner section
        if (cap_size.x >= sz.x && cap_size.y >= sz.y) {
          cap_size = sz;
          gen_expandable = true;
        }
      }
      params.dim = cap_size;

      if (params.radius < 0) params.radius = 0;
      // if( params.radius + params.distance <= 0 )
      //  return;

      handle<shadow_bitmap> shadow;
      if (!v._box_shadow_cache.find(params, shadow)) {
        shadow = generate_shadow(v, rcborder, border_width, params, rounded,gen_expandable);
        if (!shadow) return;
        if (v._box_shadow_cache.size() >= BOX_SHADOW_CACHE_CAPACITY)
          v._box_shadow_cache.clear();
        v._box_shadow_cache[params] = shadow;
      }
      //shadow->draw(sf, rect(rcborder.s + shadow->offset, shadow->dim()),
      //             rect(shadow->dim()), 255);
      rect dst = rcborder;
      dst.s += shadow->offset_origin;// -border_width.s;
      dst.e -= shadow->offset_corner;// +border_width.e;

      if(gen_expandable)
        shadow->expand(sf, dst, shadow->section_defs);
      else
        shadow->draw(sf, rect(rcborder.s + shadow->offset_origin, shadow->dim()),
                     rect(shadow->dim()), 255);

    }
  }

  void style::_draw_border(view &v, graphics *sf, const rect &rcborder,
                           const rect &border_width, element *site) {

    if (has_border(0 /*left*/)) {
      rect brc;
      brc.s = rcborder.s;
      brc.e = rcborder.pointOf(1);
      brc.e.x += border_width.s.x;

      if( border_width.s.y && border_style[1] )
        brc.s.y += border_width.s.y;
      if( border_width.e.y && border_style[3] )
        brc.e.y -= border_width.e.y;

      argb c = border_color[0].val(font_color).to_argb();
      int  w = border_width.s.x;
      if (!w) w = w;
      if (w) switch (border_style[0].val())
      {
        case border_dotted: sf->draw_line_v(brc, c, w, 2 * w); break;
        case border_dashed: sf->draw_line_v(brc, c, 3 * w, 5 * w); break;
        case border_double:
        case border_solid:
        default: sf->fill(c, brc); break;
      }
    }

    if (has_border(2) /*right*/) {
      rect brc;
      brc.s = rcborder.pointOf(9);
      brc.e = rcborder.e;
      brc.s.x -= border_width.e.x;

      // int ps = border_style[2];
      argb c = border_color[2].val(font_color).to_argb();

      if( border_width.s.y && border_style[1] )
        brc.s.y += border_width.s.y;
      if( border_width.e.y && border_style[3] )
        brc.e.y -= border_width.e.y;

      int w = border_width.e.x;

      if (w) switch (border_style[2].val())
      {
        case border_dotted: sf->draw_line_v(brc, c, w, 2 * w); break;
        case border_dashed: sf->draw_line_v(brc, c, 3 * w, 5 * w); break;
        case border_double:
        case border_solid:
        default: sf->fill(c, brc); break;
      }
    }

    if (has_border(1 /*top*/)) {
      rect brc;
      brc.s = rcborder.s;
      brc.e = rcborder.pointOf(9);
      brc.e.y += border_width.s.y;

      argb c = border_color[1].val(font_color).to_argb();

      int w = border_width.s.y;

      if (w) switch (border_style[1].val())
      {
        case border_dotted: sf->draw_line_h(brc, c, w, 2 * w); break;
        case border_dashed: sf->draw_line_h(brc, c, 3 * w, 5 * w); break;
        case border_double:
        case border_solid:
        default: sf->fill(c, brc); break;
      }
    }

    if (has_border(3 /*bottom*/)) {
      rect brc;
      brc.s = rcborder.pointOf(1);
      brc.e = rcborder.e;
      brc.s.y -= border_width.e.y;

      color c = border_color[3].val(font_color).to_argb();
      int   w = border_width.e.y;

      if (w) switch (border_style[3].val())
      {
        case border_dotted: sf->draw_line_h(brc, c, w, 2 * w); break;
        case border_dashed: sf->draw_line_h(brc, c, 3 * w, 5 * w); break;
        case border_double:
        case border_solid:
        default: sf->fill(c, brc); break;
      }
    }
  }

  void style::draw_outline(view &v, graphics *sf, const rect &rcborder, element *site, point pos) {
    int w = pixels(v, site, outline_width).width();
    if (w <= 0) return;

    argb c =  (outline_color.is_defined() ? outline_color : font_color).to_argb();

    byte opacity_v = byte(opacity);

    if (opacity_v != 255) c.alfa = argb::channel_t((c.alfa * opacity_v) / 256);

    if (c.alfa == 0) return;

    int offset = 0;

    rect rcoutline       = rcborder;
    rect rcoutline_inner = rcborder;

    if (outline_offset.is_defined())
      rcoutline >>= (offset = pixels(v, site, outline_offset).width());

    rcoutline_inner = rcoutline;
    rcoutline >>= w;

    auto os = outline_style.val();

    if (os == border_glow) {

      box_shadow_params params;
      bool              rounded = false;
      if (has_rounded_corners()) {
        get_rounded_corners(v, site, params.bradius[0], params.bradius[1],
                            params.bradius[2], params.bradius[3],
                            rcborder.size());
        rounded = true;
      }

      params.inner    = false;
      params.distance = pixels(v, site, outline_offset, rcborder.size()).width();
      params.radius   = pixels(v, site, outline_width, rcborder.size()).width();
      params.offset.x = 0;
      params.offset.y = 0;
      params.color    = c;
      params.dim      = rcborder.size();

      if (params.radius < 0) params.radius = 0;
      if (params.radius + params.distance <= 0) return;

      size cap_size = rcborder.size();
      bool gen_expandable = false;

      {
        rect rc = rect(0, 0, 0, 0);
        rc >>= params.radius * 3 / 2;
        rc >>= positive(params.offset);
        rc = outer_box(rc, params.bradius[0], params.bradius[1], params.bradius[2], params.bradius[3]);
        size sz = rc.size() + size(8, 8); // 8x8 is the size of inner section
        if (cap_size.x >= sz.x && cap_size.y >= sz.y) {
          cap_size = sz;
          gen_expandable = true;
        }
      }
      params.dim = cap_size;

      handle<shadow_bitmap> shadow;
      if (!v._box_shadow_cache.find(params, shadow)) {
        shadow = generate_shadow_outer(v, rcborder, params, rounded, gen_expandable);
        if (!shadow) return;
        if (v._box_shadow_cache.size() >= BOX_SHADOW_CACHE_CAPACITY)
          v._box_shadow_cache.clear();
        v._box_shadow_cache[params] = shadow;
      }

      if (gen_expandable) {
        rect dst = rcborder;
        dst.s += shadow->offset_origin;// -border_width.s;
        dst.e -= shadow->offset_corner;// +border_width.e;
        shadow->expand(sf, dst, shadow->section_defs);
      }
      else
        shadow->draw(sf, rect(rcborder.s + shadow->offset_origin, shadow->dim()),
          rect(shadow->dim()), 255);

      return;
    } else if (os == border_nwse_hatch || os == border_nesw_hatch) {
      static himage nwse_hatch;
      static himage nesw_hatch;

      himage hatch;
      if (os == border_nwse_hatch) {
        if (!nwse_hatch) {
          bytes data = v.app->get_resource(W("nwse-hatch.png"));
          assert(data.length);
          nwse_hatch = image::create(data, "nwse-hatch.png");
        }
        hatch = nwse_hatch;
      } else if (os == border_nesw_hatch) {
        if (!nesw_hatch) {
          bytes data = v.app->get_resource(W("nesw-hatch.png"));
          assert(data.length);
          nesw_hatch = image::create(data, "nesw-hatch.png");
        }
        hatch = nesw_hatch;
      }

      auto_ptr<image_filter_colorize> tr(new image_filter_colorize());
      tr->src = c;

      handle<bitmap> bmp = tr->apply(hatch.ptr_of<bitmap>());

      rect rinner = rcoutline << w;

      polygonf p1;
      p1.set(rcoutline);
      polygonf p2;
      p2.set(rinner);
      handle<gool::path> path = v.app->create_path();
      path->set(p1, p2);
      sf->fill(bmp, path);

      return;
    }

    if (has_rounded_corners()) {

      rect r  = rcoutline;
      rect ir = rcoutline_inner;

      size rtl, rtr, rbr, rbl;
      get_rounded_corners(v, site, rtl, rtr, rbr, rbl, r.size());

      rtl += offset + w;
      rtl.normalize();
      rtr += offset + w;
      rtr.normalize();
      rbr += offset + w;
      rbr.normalize();
      rbl += offset + w;
      rbl.normalize();

      size irtl = (rtl - size(w)).normalize(),
           irtr = (rtr - size(w)).normalize(),
           irbr = (rbr - size(w)).normalize(),
           irbl = (rbl - size(w)).normalize();

      rectf fr  = r;
      rectf fir = ir;

      if (sf->needs_half_pixel_adjustment()) {
        fr -= 0.5f;
        fir -= 0.5f;
      }

      handle<gool::path> path = v.app->create_path();
      path->add_round_rect(fr.s, fr.dimension(), rtl, rtr, rbr, rbl);
      path->add_round_rect(fir.s, fir.dimension(), irtl, irtr, irbr, irbl);

      sf->fill(c, path);

      return;
    }

    // left
    {
      rect brc;
      brc.s = rcoutline.s;
      brc.e = rcoutline.pointOf(1);
      brc.e.x += w;

      switch (os) {
      case border_dotted: sf->draw_line_v(brc, c, w, 2 * w); break;
      case border_dashed: sf->draw_line_v(brc, c, 3 * w, 5 * w); break;
      case border_double:
      case border_solid:
      default:
        sf->fill(c, brc);
        break;
        // case border_dotted:    sf.shade(brc,c);      break;
        // case border_dashed:    sf.shade(brc,c,hatch_bits);    break;
        // case border_double:
        // case border_solid:
        // default:               sf.fill(brc,c);  break;
      }
    }
    // right
    {
      rect brc;
      brc.s = rcoutline.pointOf(9);
      brc.e = rcoutline.e;
      brc.s.x -= w;
      switch (os) {
      case border_dotted: sf->draw_line_v(brc, c, w, 2 * w); break;
      case border_dashed: sf->draw_line_v(brc, c, 3 * w, 5 * w); break;
      case border_double:
      case border_solid:
      default: sf->fill(c, brc); break;
      }
    }

    // top
    {
      rect brc;
      brc.s = rcoutline.s;
      brc.e = rcoutline.pointOf(9);
      brc.e.y += w;

      switch (os) {
      case border_dotted: sf->draw_line_h(brc, c, w, 2 * w); break;
      case border_dashed: sf->draw_line_h(brc, c, 3 * w, 5 * w); break;
      case border_double:
      case border_solid:
      default: sf->fill(c, brc); break;
      }
    }

    // bottom
    {
      rect brc;
      brc.s = rcoutline.pointOf(1);
      brc.e = rcoutline.e;
      brc.s.y -= w;

      switch (os) {
      case border_dotted: sf->draw_line_h(brc, c, w, 2 * w); break;
      case border_dashed: sf->draw_line_h(brc, c, 3 * w, 5 * w); break;
      case border_double:
      case border_solid:
      default: sf->fill(c, brc); break;
      }
    }
  }

  void style::draw_image(view &v, graphics *sf, const style::image_def &imd,
                         const rect &rc, bool foreground, element *site) {
    himage pimg;

    if (site)
      pimg = foreground ? site->get_fore_image(v) : site->get_back_image(v);

    if (pimg.is_null()) pimg = imd.id.img();

    if (pimg.is_null()) return;

    auto_state<style *> _0(sf->current_style, this);
    auto_state<element *> _1(sf->current_element, site);

#ifdef _DEBUG
    if (site && site->is_id_test())
      site = site;
#endif

    //auto_state<element *> _1(v.image_drawing_element, site);

    if (imd.filters.is_defined())
      pimg = pimg->transform(imd.filters.first);

    int mode = int(imd.repeat);

    if (site) {
      sf->pixels_scaling = v.pixels_scaling();

      if (pimg->is_animated(uint_ptr(site))) {
        animated_image *pani = pimg.ptr_of<animated_image>();
        if (imd.frame_no.is_undefined() || imd.frame_no.val(0) == FRAME_ANIMATE) {
          image_animator *itor = 0;
          if (foreground) {
            itor = site->get_animation_of_type<fore_image_animator>();
            if (!itor) v.add_animation(site, itor = new fore_image_animator());
          } else {
            itor = site->get_animation_of_type<back_image_animator>();
            if (!itor) v.add_animation(site, itor = new back_image_animator());
          }
          itor->himg = pani;
        } else {
          int fn = imd.frame_no.val(0);
          if (fn == FRAME_CURRENT) {
            const animated_image::state &st =
                pani->get_state_for(uint_ptr(site));
            fn = 1 + int(st.frame_no);
          } else if (fn == FRAME_LAST) {
            fn = pani->n_frames();
          } else if (fn == FRAME_FIRST) {
            fn = 1;
          }
          fn = fn - 1;
          pani->set_frame_no(fn, uint_ptr(site));
        }
        pimg = pani->get_bitmap(sf, rc.size(), uint_ptr(site));
      }
    }
    /*else if(site && pimg->is_of_type<animated_image>() &&
    imd.frame_no.is_defined())
    {
      animated_image* pani = pimg.ptr_of<animated_image>();
      pimg = pani->get_bitmap(sf,rc.size(),uint_ptr(site));
    }*/


    if (pimg.is_null()) return;

    rect  rcback = rc;
    rectf rcimg  = rc;

    if (imd.attachment == attachment_fixed) {
      rcback = rect(point(0, 0), v.dimension());
    } else if (imd.attachment == attachment_local && site) {
      scroll_data sd;
      site->get_scroll_data(v, sd);
      rcback  = rect(site->view_pos(v), site->client_rect(v).size());
      rcimg   = rcback;
      size sz = rcback.size();
      if (sz.x < sd.content_outline.e.x)
        sz.x = sd.content_outline.e.x;
      if (sz.y < sd.content_outline.e.y)
        sz.y = sd.content_outline.e.y;
      rect dr(rcback.s, sz);
      dr -= sd.pos;
      rcimg = dr;
    } else if (site && (site->c_style == this)) {
      // rcback = rcimg;
      rcback = foreground ? site->foreground_clip_box(v, element::TO_SELF)
                          : site->background_clip_box(v, element::TO_SELF);
      rcback += rc.s;
    }

    rect rclip = rcimg;
    if (site && (site->tag == tag::T_HTML || site->tag == tag::T_BODY))
      rclip = rect(point(0, 0), v.dimension());

    sizef ipsize = pimg->dim();

    /*if(zoom.is_defined()) {
      float mu = zoom.val(1.0f);
      ipsize.x = int(ipsize.x * mu + 0.5f);
      ipsize.y = int(ipsize.y * mu + 0.5f);
    }*/

    while (!imd.dim[0].undefined_or_auto() || !imd.dim[1].undefined_or_auto()) {
      float ratio = (ipsize.y == 0) ? 1.0f : float(ipsize.x) / float(ipsize.y);

      if (imd.dim[0].is_literal(size_v::special_values::$cover)) {
        float rcratio = (rclip.height() == 0)
                            ? 1.0f
                            : float(rclip.width()) / float(rclip.height());
        if (ratio > rcratio) {
          ipsize.y = float(rclip.height());
          ipsize.x = (ipsize.y * ratio + 0.5f);
        } else {
          ipsize.x = float(rclip.width());
          ipsize.y = (ipsize.x / ratio + 0.5f);
        }
        break;
      } else if (imd.dim[0].is_literal(size_v::special_values::$contain)) {
        float rcratio = (rclip.height() == 0)
                            ? 1.0f
                            : float(rclip.width()) / float(rclip.height());
        if (ratio > rcratio) {
          ipsize.x = float(rclip.width());
          ipsize.y = (ipsize.x / ratio);
        } else {
          ipsize.y = float(rclip.height());
          ipsize.x = (ipsize.y * ratio);
        }
        break;
      }

      if (imd.dim[0].is_ui_scale())
        ipsize.x = (ipsize.x * v.pixels_per_inch().x) / 96;
      else if (imd.dim[0].is_defined() && !imd.dim[0].is_auto() &&
               !imd.dim[0].is_spring())
        ipsize.x = pixels(v, site, imd.dim[0], rc.size()).width_f();
      if (imd.dim[1].is_ui_scale())
        ipsize.y = (ipsize.y * v.pixels_per_inch().y) / 96;
      else if (imd.dim[1].is_defined() && !imd.dim[1].is_auto() && !imd.dim[1].is_spring())
        ipsize.y = pixels(v, site, imd.dim[1], rc.size()).height_f();

      // if( imd.dim[0].undefined_or_auto() && imd.dim[1].undefined_or_auto() )
      //  /*use existing ipsize*/;
      // else
      if (imd.dim[0].undefined_or_auto())
        ipsize.x = (ipsize.y * ratio);
      else if (imd.dim[1].undefined_or_auto())
        ipsize.y = (ipsize.x / ratio);

      break;
    }

    if (mode & background_keep_ratio) {
      size  rcimg_size = rcimg.size();
      float xr         = float(rcimg_size.x) / float(ipsize.x);
      float yr         = float(rcimg_size.y) / float(ipsize.y);
      if (xr < yr) {
        ipsize.x = (ipsize.x * xr);
        ipsize.y = (ipsize.y * xr);
      } else {
        ipsize.x = (ipsize.x * yr);
        ipsize.y = (ipsize.y * yr);
      }
    }

    pointf org;
    pointf cor;

    if (imd.margin[0].is_defined() && imd.margin[2].undefined_or_auto()) {
      /*if (imd.margin[0].is_percent()) {
        int percent = imd.margin[0].percent();
        float xi      = (ipsize.x * percent) / 100.0f;
        float xw      = (rcback.width() * percent) / 100.0f;
        org.x         = xw - xi;
      } else
        org.x = imd.margin[0].pixels_width_f(v, site, rc.width());*/
      org.x = image_position_pixels(v, site, imd.margin[0], rcback.size(), ipsize).width_f();

    } else if (imd.margin[0].undefined_or_auto() &&
               imd.margin[2].is_defined()) {
      /*if (imd.margin[2].is_percent()) {
        int percent = imd.margin[2].percent();
        float xi      = (ipsize.x * percent) / 100.0f;
        float xw      = (rcback.width() * percent) / 100.0f;
        org.x       = rcback.width() - (xw - xi) - ipsize.x;
      } else if (imd.margin[2].is_auto())
        org.x = 0;
      else {
        org.x = rcback.width() - ipsize.x -
                imd.margin[2].pixels_width_f(v, site, rc.width());
      }*/
      org.x = rcback.width() - ipsize.x - image_position_pixels(v, site, imd.margin[2], rcback.size(), ipsize).width_f();

    } else if (!imd.margin[0].undefined_or_auto() &&
               !imd.margin[2].undefined_or_auto()) // both margins are defined
    {
      /*if (imd.margin[0].is_percent()) {
        int percent = imd.margin[0].percent();
        float xw      = (rcback.width() * percent) / 100.0f;
        org.x       = xw;
      } else if (imd.margin[0].is_auto())
        org.x = 0;
      else
        org.x = imd.margin[0].pixels_width_f(v, site, rc.width());
      if (imd.margin[2].is_percent()) {
        int percent = imd.margin[2].percent();
        float xw      = (rcback.width() * percent) / 100.0f;
        cor.x       = xw;
      } else if (imd.margin[2].is_auto())
        cor.x = 0;
      else {
        cor.x = imd.margin[2].pixels_width_f(v, site, rc.width());
      }*/
      org.x = image_position_pixels(v, site, imd.margin[0], rcback.size(), ipsize).width_f();
      cor.x = image_position_pixels(v, site, imd.margin[2], rcback.size(), ipsize).width_f();
    }

    if (imd.margin[1].is_defined() && imd.margin[3].undefined_or_auto()) {
      /*if (imd.margin[1].is_percent()) {
        int percent = imd.margin[1].percent();
        float yi      = (ipsize.y * percent) / 100.0f;;
        float yw      = (rcback.height() * percent) / 100.0f;;
        org.y       = yw - yi;
      } else if (imd.margin[1].is_auto())
        org.y = 0;
      else
        org.y = imd.margin[1].pixels_height_f(v, site, rc.height());*/
      org.y = image_position_pixels(v, site, imd.margin[1], rcback.size(), ipsize).height_f();

    } else if (imd.margin[1].undefined_or_auto() &&
               imd.margin[3].is_defined()) {
      /*if (imd.margin[3].is_percent()) {
        int percent = imd.margin[3].percent();
        float yi      = (ipsize.y * percent) / 100.0f;;
        float yw      = (rcback.height() * percent) / 100.0f;;
        org.y       = rcimg.height() - (yw - yi) - ipsize.y;
      } else if (imd.margin[3].is_auto())
        org.y = 0;
      else
        org.y = rcback.height() - ipsize.y -
                imd.margin[3].pixels_height_f(v, site, rc.height());*/
      org.y = rcback.height() - ipsize.y - image_position_pixels(v, site, imd.margin[3], rcback.size(), ipsize).height_f();

    } else if (!imd.margin[1].undefined_or_auto() &&
               !imd.margin[3].undefined_or_auto()) // both margins are defined
    {
      /*if (imd.margin[1].is_percent()) {
        int percent = imd.margin[1].percent();
        float yw      = (rcback.height() * percent) / 100.0f;;
        org.y       = yw;
      } else if (imd.margin[1].is_auto())
        org.y = 0;
      else
        org.y = imd.margin[1].pixels_height_f(v, site, rc.height());

      if (imd.margin[3].is_percent()) {
        int percent = imd.margin[3].percent();
        float yw      = (rcback.height() * percent) / 100.0f;
        cor.y       = yw;
      } else if (imd.margin[3].is_auto())
        cor.y = 0;
      else
        cor.y = imd.margin[3].pixels_height_f(v, site, rc.height());*/
      org.y = image_position_pixels(v, site, imd.margin[1], rcback.size(), ipsize).height_f();
      cor.y = image_position_pixels(v, site, imd.margin[3], rcback.size(), ipsize).height_f();
    }

    if (!pimg->is_vector()) // bitmaps are not subpixel positioned
    {
      org = point(org);
      cor = point(cor);
    }

    size srcsize = pimg->is_bitmap() ? pimg->dim() : size(ipsize);

    switch (mode & 0xF)
    {
    case background_repeat:
      if (imd.attachment.val() == attachment_fixed) {
        clipper _c(sf, rcimg);
        pimg->tile(sf, rcback, srcsize, org, ipsize);
      } else if (imd.attachment.val() == attachment_local) {
        clipper _c(sf, rcback);
        pimg->tile(sf, rcimg, srcsize, org, ipsize);
      } else
        pimg->tile(sf, rcimg, srcsize, org, ipsize);
      break;
    case background_no_repeat:
      if (imd.attachment.val() == attachment_fixed) {
        rcimg = rect(org, ipsize);
        clipper _c(sf, rclip);
        sf->draw(pimg, rcimg);
      } else {

        rcimg.s.y += org.y;
        rcimg.e.y = rcimg.s.y + ipsize.y;
        rcimg.s.x += org.x;
        rcimg.e.x = rcimg.s.x + ipsize.x;

        rect rx = rect(rcimg) & rcback;

        if (rx != rect(rcimg)) {
          clipper _c(sf, rc/*| rx -- WRONG*/);
          sf->draw(pimg, rcimg);
        } else
          sf->draw(pimg, rcimg);
      }
      break;

    case background_repeat_x:
      if (imd.attachment.val() == attachment_fixed) {
        rcimg          = rcback;
        rcimg.e.y = rcimg.s.y + ipsize.y;
        clipper _c(sf, rc);
        pimg->tile(sf, rcimg, rect(size(ipsize)), org);
      } else {
        rcimg.s.y += org.y;
        rcimg.e.y = rcimg.s.y + ipsize.y;
        clipper _c(sf, rc);
        pimg->tile(sf, rect(rcimg), rect(size(ipsize)), point(int(org.x), 0));
      }
      break;

    case background_repeat_y:
      if (imd.attachment.val() == attachment_fixed) {
        rcimg          = rcback;
        rcimg.e.x = rcimg.s.x + ipsize.x - 1;
        clipper _c(sf, rc);
        pimg->tile(sf, rect(rcimg), rect(size(ipsize)), org);
      } else {
        rcimg.s.x += org.x;
        rcimg.e.x = rcimg.s.x + ipsize.x - 1;
        clipper _c(sf, rc);
        pimg->tile(sf, rect(rcimg), rect(size(ipsize)), point(0, int(org.y)));
      }
      break;

    case background_stretch:
      if (mode & background_keep_ratio) {
        rcimg.s += org;
        rcimg.e = rcimg.s + ipsize - sizef(1);
      } else {
        rcimg.s += org;
        rcimg.e -= cor;
      }
      if (rcimg.empty()) break;
      sf->draw(pimg, rcimg);
      break;

    case background_expand: {
      SECTION_DEFS sd;
      sd.top = ((mode & background_stretch_top) == background_stretch_top)
                    ? SRM_STRETCH
                    : SRM_TILE;
      sd.bottom = ((mode & background_stretch_bottom) == background_stretch_bottom)
                    ? SRM_STRETCH
                    : SRM_TILE;
      sd.left = ((mode & background_stretch_left) == background_stretch_left)
                    ? SRM_STRETCH
                    : SRM_TILE;
      sd.right = ((mode & background_stretch_right) == background_stretch_right)
                    ? SRM_STRETCH
                    : SRM_TILE;
      sd.center = ((mode & background_stretch_center) == background_stretch_center)
                    ? SRM_STRETCH
                    : SRM_TILE;
      sd.margins = imd.calc_sections();

      if (imd.attachment == attachment_local) {
        clipper _c(sf, rcback);
        sf->expand(pimg, rcimg, sd);
      } else
        sf->expand(pimg, rcimg, sd);
    } break;
    default:
      // sf.fill(rcimg, back_image);
      assert(false);
      break;
    }

    if (site && foreground && site->ldata) {
      rcimg -= rc.s;
      switch (fore_image.clip) {
      case clip_content_box: break;
      case clip_padding_box: rcimg += site->ldata->padding_box().s; break;
      case clip_margin_box: rcimg += site->ldata->margin_box().s; break;
      default: rcimg += site->ldata->border_box().s; break;
      }

      site->ldata->foreground_box = rcimg;
    }
  }

  void element::draw_outlines(view &v, graphics *sf, point pos, bool children,
                              bool self, bool apply_transform) {
   
    auto draw = [&](element *b) -> bool {
      hstyle cs = b->get_style(v);
      if (!cs->display || cs->visibility != visibility_visible) return false;
      if (b->oof_positioned(v)) return false;
      if (b->state.popup()) return false;
        // rect r = b->rendering_box(v) + pos;
        // if( r && vr)
        //  b->draw(v, pg, b->pos() + pos);
#pragma TODO("better clipping!")
#pragma TODO("specialization for vertical layout!")
      point tp = b->pos() + pos;
      // if( cs->has_outline() )
      //  b->draw_outline(v,sf,tp);
      if (cs->overflow() < overflow_hidden)
        b->draw_outlines(v, sf, tp, true, true, true);
      else
        b->draw_outlines(v, sf, tp, false, true, true);
      return false;
    };

    const style *cs = get_style(v);
    if (apply_transform && cs->transforms) {
      gool::state  _(sf);
      affine_mtx_f mtx;
      compute_mtx(v, mtx, pos);
      if (mtx.is_valid()) {
        sf->transform(mtx);
        if (self && cs->has_outline()) draw_outline(v, sf, pos);
        if (children) {
          if (cs->overflow() < overflow_hidden) pos -= ldata->offset;
          each_ui_child(draw);
        }
      }
    } else {
      if (self && cs->has_outline()) draw_outline(v, sf, pos);
      if (children) {
        if (cs->overflow() < overflow_hidden) pos -= ldata->offset;
        each_ui_child(draw);
      }
    }
  }

  void style::draw_foreground(view &v, graphics *gfx, const rect &r, element *site, point pos) {
    auto do_draw = [&](const rect& box)
    {
      if (has_foreground_color()) {
        gfx->fill(fore_color.to_argb(), box);
      }
      if (has_foreground_gradient()) {
        rect rr(r.size());
        handle<brush> br = create_brush(v, gfx, r, site, fore_gradient);
        if (br) gfx->fill(br, r);
      }
      draw_image(v, gfx, fore_image, box, true, site);
    };

    if (fore_image.clip.is_defined())
    {
      handle<gool::path> path;
      gool::rect         box = r;

      background_outline(v, gfx, r, box, path, site);

      if (path) {
        gool::layer _(gfx, path);
        do_draw(box);
      }
      else
        do_draw(box);
    }
    else
      do_draw(r);
  }

  void style::draw_shape(view &v, graphics *sf, const gool::rect &rc, element *site) {
    // NOTE: rc is border box here
    rect border_width;
    size dim = rc.size();
    border_width.s.x = pixels(v, site, used_border_width(0), dim).width();
    border_width.e.x = pixels(v, site, used_border_width(2), dim).width();
    border_width.s.y = pixels(v, site, used_border_width(1), dim).height();
    border_width.e.y = pixels(v, site, used_border_width(3), dim).height();

    rect background_rc = rc;

    auto padding_width = [&]() -> rect {
      rect w;
      w.s.x = pixels(v, site, used_padding(0), dim).width();
      w.e.x = pixels(v, site, used_padding(2), dim).width();
      w.s.y = pixels(v, site, used_padding(1), dim).height();
      w.e.y = pixels(v, site, used_padding(3), dim).height();
      return w;
    };

    switch (back_image.clip) {
    case clip_content_box:
      background_rc <<= padding_width(); // fall through
    case clip_padding_box:
      background_rc <<= border_width; // fall through
    default: break;
    }

    draw_background(v, sf, background_rc, site, rc.s);

    if (box_shadow.is_defined()) draw_box_shadow(v, sf, rc, border_width, site);
    draw_border(v, sf, rc, border_width, site, rc.s);

    draw_foreground(v, sf, rc, site, rc.s);

    // assert(false);
  }

  static point tree_line_anchor(view &v, graphics *sf, element *b) {
    hstyle cs = b->get_style(v);

    rect iconr; // b->image_z_box(v, cs->fore_image);
    if (b->ldata) iconr = b->ldata->foreground_box;

    if (!iconr.empty()) return iconr.pointOf(5) + b->pos();

    int fly, flheight, flascent;

    b->get_first_line_metrics(v, fly, flheight, flascent);

    rect bp = b->border_box(v, element::TO_PARENT);

    point a;
    a.y = b->pos().y + fly + (flheight / 2) - b->border_distance(v).top();
    if (cs->direction == direction_ltr)
      a.x = bp.left();
    else
      a.x = bp.right();
    return a;
  }

  static bool is_last_displayed(view &v, element *b) {
    element *p = b->parent;
    if (!p) return false;
    int total = p->n_children();
    for (int n = b->index() + 1; n < total; ++n) {
      element *t = p->child(n);
      if (!t) continue;
      if (t->is_visible(v)) return false;
    }
    return true;
  }
  static bool is_first_visible_option(view &v, element *b) {
    if (!b->is_visible(v)) return false;
    element *p = b->parent;
    if (!p) return false;
    for (int n = b->index() - 1; n >= 0; --n) {
      element *t = p->child(n);
      if (!t) continue;
      if (t->tag == tag::T_OPTION) return false;
    }
    return true;
  }

  // void element::draw_bullet(view& v, graphics* sf, gool::point p ) {}
  void element::draw_bullet(view &v, graphics *sf, gool::point p) {
    // bullet drawing

    hstyle cs = get_style(v);

    argb color = cs->list_marker_color.val(cs->font_color).to_argb();

    // sf.font_metrics(height,ascent);
    int f_ascent = 0;
    int f_height = 0;
    int dy       = 0;

    get_first_line_metrics(v, dy, f_height, f_ascent);
    //assert(f_height && f_ascent);

    ustring s;
    int     idx = 0;
    size_v  fsize_x;
    himage  pimagebullet = cs->list_style_image.img();

    if (pimagebullet) switch (cs->mapping.u.parts.list_image) {
      case mapping_left_to_right:
        pimagebullet = pimagebullet->mapped_left_to_right();
        break;
      case mapping_top_to_right:
        pimagebullet = pimagebullet->mapped_top_to_right();
        break;
      }

    bool x_border_rel = true;
    int  xbp_width    = 0;

    bool ltr = cs->direction != direction_rtl;

    if (ltr) {
      xbp_width = ldata->margin_width.left();
      if (!xbp_width) { xbp_width = parent->ldata->margin_width.left(); }
      if (!xbp_width) { xbp_width = parent->ldata->padding_width.left(); }
      if (!xbp_width) {
        xbp_width    = ldata->padding_width.left();
        x_border_rel = false;
      }
    } else // rtl
    {
      xbp_width = ldata->margin_width.right();
      if (!xbp_width) { xbp_width = parent->ldata->margin_width.right(); }
      if (!xbp_width) { xbp_width = parent->ldata->padding_width.right(); }
      if (!xbp_width) {
        xbp_width    = ldata->padding_width.right();
        x_border_rel = false;
      }
    }
    if (xbp_width < 10) xbp_width = 10;

    auto list_style_type = cs->list_style_type.val();

    if (!pimagebullet.is_null())
      goto IMAGE_BULLET;
    else if (list_style_type >= list_style_decimal &&
             list_style_type < list_non_numeric) // ordered
      goto ORDERED_BULLET;
    else if (list_style_type < list_style_decimal) // unordered 1
      goto UNORDERED_BULLET;
    else if (list_style_type == list_style_tree_lines)
      goto TREE_LINES;
    else if (list_style_type && parent) // unordered, custom char bullet
    {
      s += wchar(list_style_type);
      goto ORDERED_BULLET_TEXT;
    }
    return;

  ORDERED_BULLET : {

    fsize_x = cs->font_size;
    if (cs->list_marker_size.is_defined()) {
      fsize_x = cs->list_marker_size;
      fsize_x.resolve(v, cs->font_size);
    }

    if (cs->list_style_type.val() == list_style_index)
      idx = 1 + this->index();
    else
      idx = get_list_index();

  ORDERED_BULLET_TEXT:
    rect rcborder = border_box(v) + p;
    rect rc       = content_box(v) + p;

    int x = ltr ? (x_border_rel ? rcborder.s.x : rc.s.x)
                : (x_border_rel ? rcborder.e.x : rc.e.x);

    list_marker *plm = ldata->lmarker;
    if (!plm) ldata->lmarker = plm = new list_marker;

    if (!plm->tl || plm->item_no != idx || plm->type != cs->list_style_type.val()) {
      plm->item_no = idx;
      plm->type    = cs->list_style_type.val();

      switch (cs->list_style_type.val()) {
      default:
      case list_style_decimal: s = ustring::format(W("%d."), idx); break;
      case list_style_index: s = ustring::format(W("%d"), idx); break;

      case list_style_lower_alpha:
      case list_style_upper_alpha:
        s = ustring(string::alpha(idx, cs->list_style_type.val() == list_style_upper_alpha));
        s += '.';
        break;

      case list_style_lower_roman:
      case list_style_upper_roman:
        s = ustring(string::roman(idx, cs->list_style_type.val() == list_style_upper_roman));
        s += '.';
        break;
      }

      plm->tl = this->create_text_block(v, s);
      /*if (cs->list_marker_color.is_defined()) {
        plm->tl->a_style = new style_prop_map();
        plm->tl->a_style->set(cssa_color, cs->list_marker_color);
        plm->tl->clear_style();
      }*/
      const style* mcs = plm->tl->get_style(v);
      plm->tl->set_width(v, plm->tl->max_content_width(v));
      plm->tl->set_height(v, plm->tl->max_content_height(v));
    }

    int y, height, ascent;
    plm->tl->get_first_line_metrics(v, y, height, ascent);

    if (!cs->list_style_position) {
      int twidth = plm->tl->dim().x;
      if (ltr) {
        int width = twidth + ascent / 2;
        x -= width;
      } else // rtl
      {
        int width = ascent / 2;
        x += width;
      }
    }
    plm->tl->set_default_text_color(color);
     //auto_state<argb> _(sf->text_color, color);
    // sf->draw( *(plm->tl), point(x, /*rcborder.s.y +*/ rc.s.y + dy +
    // f_ascent - int(plm->tl->ascent())), color);
    plm->tl->draw_content(v, sf, point(x, rc.s.y + dy + f_ascent - ascent), false);
    return;
  }

  UNORDERED_BULLET : {
    size_v fsize_x = cs->font_size;
    if (cs->list_marker_size.is_defined()) {
      fsize_x = cs->list_marker_size;
      fsize_x.resolve(v, cs->font_size);
    }

    rect rc        = (x_border_rel ? border_box(v) : content_box(v)) + p;
    int  marker_sz = pixels(v, this, fsize_x).height() / 4;

    pointf pos;

    if (ltr) {
      pos = rc.pointOf(7);
      if (!cs->list_style_position) pos.x -= (xbp_width + marker_sz) / 2;
    } else // ltr
    {
      pos = rc.pointOf(9);
      if (!cs->list_style_position) pos.x += (xbp_width - marker_sz) / 2;
    }
    pos.y = p.y + dy + f_ascent + (f_height - f_ascent - f_ascent) / 2 + 0.5f;

    // sf.draw( point(x, /*rcborder.s.y +*/ rc.s.y + dy + f_ascent),
    // s, s.length(), false);  graphics_state _(sf);

    switch (cs->list_style_type.val()) {
      case list_style_disc: {
        gool::state _(sf);
        sf->set_fill(color);
        sf->set_stroke();
        sf->draw_ellipse(pos, sizef(float(marker_sz), float(marker_sz)));
      } break;
      case list_style_circle: {
        gool::state _(sf);
        sf->set_fill();
        sf->set_stroke_width(1);
        sf->set_stroke(color);
        sf->draw_ellipse(pos,
                         sizef(float(marker_sz) - 0.5f, float(marker_sz) - 0.5f));
      } break;
      case list_style_square: {
        gool::state _(sf);
        float       m = max(1.0f, marker_sz / 1.5f);
        rectf       r(pos.x - m, pos.y - m, pos.x + m, pos.y + m);
        sf->set_fill(color);
        sf->set_stroke();
        sf->draw_rectangle(r.s, r.dimension());
        break;
      }
    }
    return;
    // s += uc;
  }

  IMAGE_BULLET : {
    size imsize = pimagebullet->dimension();

    rect rc = (x_border_rel ? border_box(v) : content_box(v)) + p;

    point pos;

    if (ltr) {
      pos = rc.pointOf(7);
      if (!cs->list_style_position) pos.x -= (xbp_width + imsize.x) / 2;
    } else // ltr
    {
      pos = rc.pointOf(9);
      if (!cs->list_style_position) pos.x += (xbp_width - imsize.x) / 2;
    }
    pos.y += dy + (f_height - imsize.y) / 2;

    sf->draw(pimagebullet, rect(pos, imsize));
    return;
  }

  TREE_LINES : {
    // draw tree
    rect bp = border_box(v, TO_PARENT);
    // int offset_x_ = ltr? bp.left() : bp.right();
    if (parent) {
      point anchor = tree_line_anchor(v, sf, this);

      rect hbar;
      hbar.e.y = (hbar.s.y = anchor.y) + 1;
      rect vbar                     = bp;
      bool has_icon                 = false;
      rect iconr = parent->image_z_box(v, parent->get_style(v)->fore_image);

      if (!iconr.empty()) {
        if (ltr) {
          hbar.s.x = iconr.left() + iconr.width() / 2;
          hbar.e.x = anchor.x;
          vbar.e.x = (vbar.s.x = hbar.s.x) + 1;
        } else {
          hbar.e.x = iconr.right() - iconr.width() / 2;
          hbar.s.x = anchor.x;
          vbar.e.x = (vbar.s.x = hbar.e.x) + 1;
        }
        has_icon = true;
      } else {
        if (ltr) {
          hbar.s.x = bp.left() - parent->padding_distance(v).left() / 2;
          hbar.e.x = anchor.x;
          vbar.e.x = vbar.s.x = hbar.s.x;
        } else {
          hbar.e.x = bp.right() + parent->padding_distance(v).right() / 2;
          hbar.s.x = anchor.x;
          vbar.e.x = vbar.s.x = hbar.e.x;
        }
      }
      if (is_last_displayed(v, this)) { vbar.e.y = hbar.s.y; }
      if (is_first_visible_option(v, this) && has_icon) {
        vbar.s.y = iconr.bottom(); // + parent->border_distance(v).bottom()
                                        // + border_distance(v).top();
        //= hbar.s.y;
        // hbar = rect();
      }
      int w             = max(1, pixels(v, this, cs->list_marker_size).width());
      int w1            = (w - 1) / 2;
      int w2            = w - w1 - 1;
      int outline_style = cs->list_marker_style.is_defined() ? cs->list_marker_style.val() : border_solid;
      color_v c = cs->list_marker_color.val(cs->font_color);
      if (!vbar.empty()) {
        vbar.s.x -= w1;
        vbar.e.x += w2;
        vbar += p - pos();
        switch (outline_style) {
        case border_dotted: sf->draw_line_v(vbar, c.to_argb(), w, 2 * w); break;
        case border_dashed:
          sf->draw_line_v(vbar, c.to_argb(), 3 * w, 5 * w);
          break;
        case border_double:
        case border_solid:
        default: sf->fill(c.to_argb(), vbar); break;
        }
      }
      if (!hbar.empty()) {
        hbar.s.y -= w1;
        hbar.e.y += w2;
        hbar += p - pos();
        switch (outline_style) {
        case border_dotted: sf->draw_line_h(hbar, c.to_argb(), w, 2 * w); break;
        case border_dashed:
          sf->draw_line_h(hbar, c.to_argb(), 3 * w, 5 * w);
          break;
        case border_double:
        case border_solid:
        default: sf->fill(c.to_argb(), hbar); break;
        }
      }
    }
  }
  } // draw_bullet

#pragma warning(push)
#pragma warning(disable : 4102) // warning C4102: 'START_ELLIPSIS' :
                                // unreferenced label - bug in MS compiler.

  void draw_decoration(view &v, const element* pel, const style *cs, gool::graphics *gfx, rect at,
                       float fthicknes, argb fore) {
    int thicknes = int(ceilf(fthicknes));

    argb clr;
    if (cs->text_decoration_color.is_undefined() ||
        cs->text_decoration_color.is_current())
      clr = fore;
    else
      clr = cs->text_decoration_color.to_argb();

    if (clr.alfa == 0) return;

    switch (cs->text_decoration_style.val()) {
    case text_decoration_style_solid: gfx->fill(clr, at); break;
    case text_decoration_style_double: gfx->fill(clr, at); break;
    case text_decoration_style_dotted:
      gfx->draw_line_h(at, clr, thicknes, 2 * thicknes);
      break;
    case text_decoration_style_dashed:
      gfx->draw_line_h(at, clr, 2 * thicknes, 4 * thicknes);
      break;
    case text_decoration_style_wavy: {
      static himage  wave1;
      static himage  wave2;
      handle<bitmap> wave;
      size           pixels = v.pixels_per_dip(size(1, 1));
      if (pixels.y == 1) {
        if (!wave1) {
          bytes data = v.app->get_resource(W("red-wave.png"));
          assert(data.length);
          wave1 = image::create(data, "wave.png");
        }
        wave = wave1.ptr_of<bitmap>();
      } else {
        if (!wave2) {
          bytes data = v.app->get_resource(W("red-wave-2x.png"));
          assert(data.length);
          wave2 = image::create(data, "wave.png");
        }
        wave = wave2.ptr_of<bitmap>();
      }

      static handle<bitmap> colored_wave;
      static bitmap *       proto_wave = nullptr;
      static argb           proto_color;

      if (!colored_wave || proto_wave != wave || proto_color != clr) {
        proto_wave  = wave;
        proto_color = clr;
        auto_ptr<image_filter_colorize> tr(new image_filter_colorize());
        tr->src = clr;
        assert(wave);
        if (wave)
          colored_wave = tr->apply(wave);
      }

      if (colored_wave) {
        at.e.y = at.s.y + colored_wave->dim().y;
        point off(colored_wave->dim().x - (at.s.x % colored_wave->dim().x),
                  0);
        gfx->fill(colored_wave, at, off);
      }
    } break;
    }
  }

#if 1

  struct glyph_runs_painter {
    view &                  v;
    element *               elem;
    const tflow::text_flow &tf;
    gool::graphics *        gfx;
    point                   at;
    tflow::sel_context *    sctx;

    handle<style> run_style;
    //handle<font>  font;
    color_v       fore_color;
    color_v       base_fore_color;
    argb          fore;

    color_v       back_color;
    handle<style> elem_style;
    bool          elem_rtl;
    bool          rtl            = false; // run rtl
    node *        run_node       = nullptr;
    bool          is_highlighted = false;

    int   draw_ellipsis = 0;
    int_v visible_lines;
    int   width;
    int   height;
    int   dx     = 0;
    float last_y = 0;
    float last_h = 0;

    glyph_runs_painter(view &_v, element *_elem, const tflow::text_flow &_tf, gool::graphics *_gfx, point _at, tflow::sel_context *_sctx)
        : v(_v), elem(_elem), tf(_tf), gfx(_gfx), at(_at), sctx(_sctx)
    {
      elem_style = elem->get_style(v);
      elem_rtl = elem_style->direction == direction_rtl;
      base_fore_color = _tf._default_text_color;
      if (base_fore_color.is_defined()) {
        fore_color = base_fore_color;
        fore = fore_color.to_argb();
      }
      auto rdim = elem->content_box(v).size();
#ifdef DEBUG
      if (_elem->is_id_test())
        _elem = _elem;
#endif
      width = rdim.x;
      height = rdim.y;
      if (elem->need_ellipsis(v)) {
        draw_ellipsis = elem_style->text_overflow;
        if (elem_rtl && elem->scroll_pos().x == 0 && elem_style->overflow_x > overflow_visible)
          dx = width - elem->ldata->dim_min.x;
      }
      else if (elem->need_multiline_ellipsis(v)) {
        draw_ellipsis = elem_style->text_overflow;
        visible_lines = 1;
        for (int ln = 0; ln < _tf._lines.size(); ++ln) {
          auto yln = _tf._lines[ln].yr();
          if (yln.e <= height)
            visible_lines = ln + 1;
          else
            break;
        }
      }
    }

    // helper functions:

    color_v get_back_color(const tflow::glyph_run &glyph_run) {
      if (glyph_run.pstyle.is_defined()) {
        color_v cv = glyph_run.pstyle->back_color;
        if (glyph_run.pstyle && glyph_run.pstyle->back_color.is_defined())
          return glyph_run.pstyle->back_color;
      }

      helement pel =
          glyph_run.node->get_ui_element(); //->is_element()?
                                            //pn->cast<element>():pn->owner;
      while (pel.is_defined()) {
        if (pel == elem) break;
        if (pel->is_box_element(v)) break;
        const style *cs = pel->get_style(v);
        color_v      cv = cs->back_color;
        if (cv.is_defined()) {
          // element* te = elem;
          return cv;
        }
        pel = pel->ui_parent(v);
      }
      return color_v();
    }

    bool get_highlightion_status(const tflow::glyph_run &glyph_run) {
      helement pel =
          glyph_run.node->get_ui_element(); //->is_element()?
                                            //pn->cast<element>():pn->owner;
      while (pel.is_defined()) {
        if (pel == elem) break;
        if (pel->is_box_element(v)) break;
        if (v.highlighted_ctl->el == pel) return true;
        pel = pel->ui_parent(v);
      }
      return false;
    };

    // painters

    void draw_glyph_back(pointf dst, const tflow::glyph_run &glyph_run,
                         color_v back_color) {
      float       width = width_of(tf, glyph_run);
      gool::font *pf    = v.get_font(run_style);
      rect rc = rtl ? rect(int(dst.x - width), int(dst.y - pf->ascent),
                           int(dst.x), int(dst.y + pf->descent))
                    : rect(int(dst.x), int(dst.y - pf->ascent),
                           int(dst.x + width), int(dst.y + pf->descent));

      v.draw_glyph_run_back(gfx, tf, glyph_run, rc, back_color.to_argb());
    };

    void draw_highlightion(pointf dst, const tflow::glyph_run &glyph_run) {
      float       width = width_of(tf, glyph_run);
      gool::font *pf    = v.get_font(run_style);
      rect rc = rtl ? rect(int(dst.x - width), int(dst.y - pf->ascent),
                           int(dst.x), int(dst.y + pf->descent))
                    : rect(int(dst.x), int(dst.y - pf->ascent),
                           int(dst.x + width), int(dst.y + pf->descent));

      gfx->fill(v.highlighted_ctl->get_fore(), rc);
    };

    void draw_text_decoration(pointf dst, const tflow::glyph_run &glyph_run) {
      float       width = width_of(tf, glyph_run);
      gool::font *pf    = v.get_font(glyph_run.get_style(v));
      gool::rectf rc;
      if (rtl) {
        rc.s.x = dst.x - width;
        rc.e.x = dst.x;
      } else {
        rc.s.x = dst.x;
        rc.e.x = dst.x + width;
      }

      const style* ps = run_style;
      element* el = glyph_run.get_element();

      auto thickness = [&]() -> float {
        if (ps->text_decoration_thickness.is_fixed())
          return pixels(v, el, ps->text_decoration_thickness).height_f();
        return ceilf(pf->size / 15.0f);
      };

      do {
          if (ps->text_decoration_line == 0) break;
          if (ps->text_decoration_line & text_decoration_underline) // includes text_decoration_underline
          {
            float thicknes = thickness();
            float shift = thicknes;
            rc.s.y = dst.y + shift;
            rc.e.y = dst.y + shift + thicknes;
            // gfx->fill(fore,rc);
            draw_decoration(v, el, ps, gfx, rc, thicknes, fore);
          }
          /*else if (ps->text_decoration_line & text_decoration_double) // includes text_decoration_underline
          {
            float thicknes = thickness();
            float shift = thicknes;
            rc.s.y = dst.y + shift;
            rc.e.y = dst.y + shift + thicknes - 1.f;
            // gfx->fill(fore, rc);
            draw_decoration(v, ps, gfx, rc, thicknes, fore);
            // if ((run_style->text_decoration_line &
            // text_decoration_double_underline) == text_decoration_double_underline)
            //{
            rc.s.y = dst.y + shift + 2 * thicknes;
            rc.e.y = dst.y + shift + 2 * thicknes + thicknes - 1.f;
            // gfx->fill(fore, rc);
            //}
          }*/

        if (ps->text_decoration_line & text_decoration_linethrough) {
          float thicknes = thickness();
          float shift = float(pf->ascent / 3);
          rc.s.y = dst.y - shift;
          rc.e.y = dst.y - shift + thicknes;
          // gfx->fill(fore,rc);
          draw_decoration(v, el, ps, gfx, rc, thicknes, fore);
        }
        if (ps->text_decoration_line & text_decoration_overline) {
          float thicknes = thickness();
          float shift = float(pf->ascent);
          rc.s.y = dst.y - shift;
          rc.e.y = dst.y - shift + thicknes;
          // gfx->fill(fore,rc);
          draw_decoration(v, el, ps, gfx, rc, thicknes, fore);
        }
        if (ps != el->get_style(v))
          ps = el->get_style(v);
        else {
          el = el->parent;
          if (!el) break;
          ps = el->get_style(v);
        }
    } while (el);
#if 0
       if (run_style->text_decoration_line == text_decoration_wavy)
       {
         float thicknes = ceilf(pf->size / 15.0f);
         size pixels = v.pixels_per_dip(size(1, 1));
         rc.s.y = dst.y + 1;
         rc.e.y = rc.s.y + 2 * pixels.y - 1;
         draw_decoration(run_style, gfx, rc, thicknes, fore);
         /*size pixels = v.pixels_per_dip(size(1,1));
         if( pixels.x < 2 ) {
         static handle<image> wavy;
         if( !wavy ) {
         bytes data = app()->get_resource(W("red-wave.png"));
         if(!data) return;
         wavy = image::create(data,"sciter:red-wave.png");
         if(!wavy) return;
         }
         rc.s.y = dst.y + 1;
         rc.e.y = rc.s.y + 2;
         gfx->fill(wavy,rc);
         } else {
         static handle<image> wavy;
         if( !wavy ) {
         bytes data = app()->get_resource(W("red-wave-2x.png"));
         if(!data) return;
         wavy = image::create(data,"sciter:red-wave-2x.png");
         if(!wavy) return;
         }
         rc.s.y = dst.y + 1;
         rc.e.y = rc.s.y + 5;
         gfx->fill(wavy,rc);
         }*/
       }
#endif
    }

    void draw_glyph_run(pointf dst, const tflow::glyph_run &run) {
      v.draw_glyph_run(gfx, tf, run, dst, fore, run_style);
      if (run_style->text_decoration_line != 0) draw_text_decoration(dst, run);
    }

    void draw_ime_composition_glyph_run(pointf                  dst,
                                        const tflow::glyph_run &glyph_run,
                                        const range &           yr) {
      pointf tdst = dst;

      uint rstart = sctx->ime_glyph_start < glyph_run.glyph_start
                        ? glyph_run.glyph_start
                        : sctx->ime_glyph_start;
      uint rend = sctx->ime_glyph_end >= glyph_run.glyph_end()
                      ? glyph_run.glyph_end()
                      : sctx->ime_glyph_end;

      if (rstart > glyph_run.glyph_start) // selection start is inside the run
      {
        tflow::glyph_run tdw = glyph_run;
        tdw.glyph_count      = rstart - glyph_run.glyph_start;
        draw_glyph_run(tdst, tdw);
        if (rtl)
          tdst.x -= width_of(tf, tdw);
        else
          tdst.x += width_of(tf, tdw);
      }

      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rstart - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = rend - rstart;
        if (tdw.glyph_end() <= tf._glyph_indices.length()) {
          float width = width_of(tf, tdw);
          if (width) {
            rect rc =
                rtl ? rect(int(tdst.x - width + 0.5f), yr.s, int(tdst.x), yr.e)
                    : rect(int(tdst.x), yr.s, int(tdst.x + width - 0.5f), yr.e);
            draw_glyph_run(tdst, tdw);
            if (rtl)
              tdst.x -= width_of(tf, tdw);
            else
              tdst.x += width_of(tf, tdw);
            rc.e.y += 1;
            rc.s.y = rc.e.y - 1;
            gfx->fill(sctx->selection_back_color, rc);
          }
        }
      }
      if (rend < glyph_run.glyph_end()) // selection start is inside the run
      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rend - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = glyph_run.glyph_end() - rend;
        draw_glyph_run(tdst, tdw);
      }
    }

    void draw_selected_glyph_run(pointf dst, const tflow::glyph_run &glyph_run,
                                 const range &yr) {
      pointf tdst = dst;
      // selection covers this glyph run
      uint rstart = sctx->sel_glyph_start < glyph_run.glyph_start
                        ? glyph_run.glyph_start
                        : sctx->sel_glyph_start;
      uint rend = sctx->sel_glyph_end >= glyph_run.glyph_end()
                      ? glyph_run.glyph_end()
                      : sctx->sel_glyph_end;

      if (rstart > glyph_run.glyph_start) // selection start is inside the run
      {
        tflow::glyph_run tdw = glyph_run;
        tdw.glyph_count      = rstart - glyph_run.glyph_start;
        // render_target->DrawGlyphRun(tdst, &tdw, fore, dw_measuring_mode);
        if (!v._draw_selection_only) draw_glyph_run(tdst, tdw);
        if (rtl)
          tdst.x -= width_of(tf, tdw);
        else
          tdst.x += width_of(tf, tdw);
      }

      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rstart - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = rend - rstart;
        float width     = width_of(tf, tdw);
        if (width) {
          rect rc =
              rtl ? rect(int(tdst.x - width), yr.s, int(tdst.x), yr.e)
                  : rect(int(tdst.x), yr.s, int(tdst.x + width), yr.e);
          gfx->fill(sctx->selection_back_color, rc);

          if (!sctx->selection_text_color.is_transparent()) {
            argb saved_fore = fore;
            fore = argb(sctx->selection_text_color);
            draw_glyph_run(tdst, tdw);
            fore = saved_fore;
          } else
            draw_glyph_run(tdst, tdw);

          if (rtl)
            tdst.x -= width_of(tf, tdw);
          else
            tdst.x += width_of(tf, tdw);
        }
      }
      if (rend < glyph_run.glyph_end()) // selection start is inside the run
      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rend - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = glyph_run.glyph_end() - rend;
        if (!v._draw_selection_only) draw_glyph_run(tdst, tdw);
      }
    }

    bool draw_one(const tflow::glyph_run &run) {
      const tflow::layout_line &line = tf._lines[run.line_no];

      assert(run.glyph_count < 10000);

      if (run.glyph_count == 0) return false; // continue;

      pointf dst;
      dst.x = at.x + run.x + dx;
      // if(lf_glyph_run.bidi_level & 1)
      //  dst.x -= width_of(tf,lf_glyph_run);

      html::element *el = run.get_inline_block_element(v);
      if (el) {
        //if (!el->is_it_visible(v)) return false;
        if (el->positioned(v)) return false;

        el->owner = elem; // enforce owner

        auto_state<selection_ctx *> _(v._selection_ctx, 0);

        if (sctx && (sctx->sel_glyph_start < sctx->sel_glyph_end) &&
            (sctx->sel_glyph_start <= run.glyph_start) &&
            (sctx->sel_glyph_end >= run.glyph_end())) {
          range yr = line.yr() + at.y;
          float w  = width_of(tf, run);
          rect  rc(int(dst.x), yr.s, int(dst.x + w - 1), yr.e);

          rect trc = el->border_box(v);

          gfx->fill(sctx->selection_back_color, rc);
          el->draw(v, gfx, at + el->pos());
          gfx->fill(sctx->selection_fore_color,
                    el->border_box(v, element::TO_PARENT) + at);
        } else
          el->draw(v, gfx, at + el->pos());
        return false;
      }

      if (tf._glyph_indices[run.glyph_start] == 0) return false;
      if (!run.node) return false;
      html::style *t_run_style = run.get_style(v);
      if (!t_run_style) return false;

      if (!run.node->get_element()->is_drawable(v, elem)) return false;

      rtl  = run.is_rtl();
      //font = run.font;

      if (run_style != t_run_style)
      {
        run_style = t_run_style;
        if (fore_color.is_undefined()) {
          fore_color = run_style->font_color;
          fore = fore_color.to_argb();
        }
        else if (fore_color != run_style->font_color) {
          if (run_style == elem_style && base_fore_color.is_defined()) {
            fore_color = base_fore_color;
            fore = fore_color.to_argb();
          }
          else {
            fore_color = run_style->font_color;
            fore = fore_color.to_argb();
          }
        }
        back_color = get_back_color(run);
      }
      if (run_node != run.node) {
        run_node = run.node;
        if (v.highlighted_ctl && v.highlighted_ctl->el)
          is_highlighted = get_highlightion_status(run);
      }

      /*if(run.marks)
      fore = argb(0xFF,0,0);
      else
      fore = argb(fore_color);*/

      switch (run.valign) {
      case valign_auto:
      case valign_baseline:
      case valign_sub:
      case valign_super: {
        dst.y       = float(line.baseline);
        element *el = run.get_element();
        if (el == elem) break;
        dst.y += el->get_baseline_shift(v, elem);
      } break;
      case valign_text_top:
      case valign_top: {
        dst.y = float(run.font->ascent);
        break;
      }
      case valign_middle: {
        int delta = (line.height - run.font->ascent - run.font->descent) / 2;
        dst.y     = float(delta + run.font->ascent);
      } break;
      case valign_text_bottom:
      case valign_bottom: dst.y = float(line.height - run.font->descent); break;
      }

      dst.y += at.y + line.y;
      last_y = dst.y;
      last_h = float(line.height);

#pragma TODO("position:relative; display:inline !")

      if (sctx && (sctx->sel_glyph_start < sctx->sel_glyph_end) &&
          (sctx->sel_glyph_start < run.glyph_end()) &&
          (sctx->sel_glyph_end >= run.glyph_start)) {
        if (back_color.is_defined()) draw_glyph_back(dst, run, back_color);
        draw_selected_glyph_run(dst, run, line.yr() + at.y);
      } else if (sctx && (sctx->ime_glyph_start < sctx->ime_glyph_end) &&
                 (sctx->ime_glyph_start < run.glyph_end()) &&
                 (sctx->ime_glyph_end >= run.glyph_start)) {
        draw_ime_composition_glyph_run(dst, run, line.yr() + int(at.y));
      }
#if 0
       else if (draw_ellipsis)
       {
         bool last_one = elem_rtl
           ? draw_glyph_run_ellipsis_rtl(dst, run, draw_ellipsis)
           : draw_glyph_run_ellipsis(dst, run, draw_ellipsis);
         if (last_one)
           break;
       }
#endif
      else if (
          !v._draw_selection_only /*&& !v.disable_glyph_run_draw(gfx,tf,run)*/) {
        if (back_color.is_defined()) draw_glyph_back(dst, run, back_color);
        draw_glyph_run(dst, run);
        if (is_highlighted) draw_highlightion(dst, run);
      }

      return false;
    }

    void draw_ellipsis_runs(slice<tflow::glyph_run> runs, int width) {

      handle<gool::font> font = v.get_font(elem_style);

      float ellipsis_width = font->size * 0.85f;

      float wme = width - ellipsis_width;

      slice<tflow::glyph_run> runs_to_draw = runs;

      float space_left_for_last_run = wme;
      int_v last_run_idx;

      for (int i = runs.size() - 1; i >= 0; --i) {
        auto glyph_run = runs[i];
        auto rx        = html::x_of(tf, glyph_run);
        if (rx.e < wme) {
          space_left_for_last_run = wme - rx.e;
          break;
        }
        runs_to_draw.length = uint(i);
        last_run_idx        = i;
      }

      runs_to_draw.each([this](const tflow::glyph_run &run) { return draw_one(run); });

      if (last_run_idx.is_defined()) {
        tflow::glyph_run last_run = runs[last_run_idx];

        bool  last_run_is_ltr = !last_run.is_rtl();
        float pos;
        if (last_run_is_ltr && elem_rtl) {
          float run_width = width_of(tf, last_run);
          last_run.reduce_to_fit(tf, space_left_for_last_run, runs_to_draw.length == 0);
          float new_run_width = width_of(tf, last_run);
          last_run.x += run_width - new_run_width;
          pos = float(at.x) + last_run.x - ellipsis_width + dx;
        } else {
          float rw = last_run.reduce_to_fit(tf, space_left_for_last_run, runs_to_draw.length == 0);
          pos      = rw + at.x + dx;
        }

        draw_one(last_run);

        static wchar ellipsis = 0x2026;

        handle<text_block> tl = elem->create_text_block(v, wchars(ellipsis));
        tl->set_width(v, tl->max_content_width(v));
        tl->set_height(v, tl->max_content_height(v));
        if (!last_run_is_ltr) pos -= tl->dim().x;
        // auto_state<argb> _(gfx->text_color, fore);
        int y, h, a;
        tl->get_first_line_metrics(v, y, h, a);
        tl->draw_glyphs(v, gfx, point(int(pos), int(last_y - a)));
      }
    }

    void draw_multiline_ellipsis_runs(slice<tflow::glyph_run> runs, int width) {

      handle<gool::font> font = v.get_font(elem_style);

      float ellipsis_width = font->size * 0.85f;

      float wme = width - ellipsis_width;

      slice<tflow::glyph_run> runs_to_draw = runs;

      float space_left_for_last_run = wme;
      int_v last_run_idx;

      for (int i = runs.size() - 1; i >= 0; --i) {
        auto glyph_run = runs[i];
        if (glyph_run.line_no >= (uint)visible_lines.val(0))
          continue;
        auto rx = html::x_of(tf, glyph_run);
        runs_to_draw.length = uint(i);
        last_run_idx = i;
        if (rx.e < wme) {
          space_left_for_last_run = wme - rx.s;
          break;
        }
      }

      runs_to_draw.each([this](const tflow::glyph_run &run) { return draw_one(run); });

      if (last_run_idx.is_defined()) {
        tflow::glyph_run last_run = runs[last_run_idx];

        bool  last_run_is_ltr = !last_run.is_rtl();
        float pos;
        if (last_run_is_ltr && elem_rtl) {
          float run_width = width_of(tf, last_run);
          last_run.reduce_to_fit(tf, space_left_for_last_run, runs_to_draw.length == 0);
          float new_run_width = width_of(tf, last_run);
          last_run.x += run_width - new_run_width;
          pos = float(at.x) + last_run.x - ellipsis_width + dx;
        }
        else {
          float rw = last_run.reduce_to_fit(tf, space_left_for_last_run, runs_to_draw.length == 0);
          pos = rw + at.x + dx;
        }

        draw_one(last_run);

        static wchar ellipsis = 0x2026;

        handle<text_block> tl = elem->create_text_block(v, wchars(ellipsis));
        tl->set_width(v, tl->max_content_width(v));
        tl->set_height(v, tl->max_content_height(v));
        if (!last_run_is_ltr) pos -= tl->dim().x;
        // auto_state<argb> _(gfx->text_color, fore);
        int y, h, a;
        tl->get_first_line_metrics(v, y, h, a);
        tl->draw_glyphs(v, gfx, point(int(pos), int(last_y - a)));
      }
    }


    void draw_ellipsis_runs_rtl(slice<tflow::glyph_run> runs, int width) {

      handle<gool::font> font = v.get_font(elem_style);

      float ellipsis_width = font->size * 0.85f;

      float wme = width - ellipsis_width;

      slice<tflow::glyph_run> runs_to_draw(runs.end(), 0);

      float space_left_for_last_run = wme;
      int_v last_run_idx;

      for (int i = runs.size() - 1; i >= 0; --i) {
        auto glyph_run = runs[i];
        auto rx        = html::x_of(tf, glyph_run);
        rx += float(dx);
        if (rx.s < ellipsis_width) {
          space_left_for_last_run = rx.e - ellipsis_width;
          last_run_idx = i;
          break;
        }
        --runs_to_draw.start;
        ++runs_to_draw.length;
      }

      runs_to_draw.each([this](const tflow::glyph_run &run) { return draw_one(run); });

      if (last_run_idx.is_defined()) {
        tflow::glyph_run last_run = runs[last_run_idx];
        //last_run.bidi_level = 0;
        bool  last_run_is_ltr = !last_run.is_rtl();
        float pos;
        if (last_run_is_ltr && elem_rtl) {
          float run_width = width_of(tf, last_run);
          last_run.reduce_to_fit(tf, space_left_for_last_run, runs_to_draw.length == 0);
          float new_run_width = width_of(tf, last_run);
          last_run.x += run_width - new_run_width;
          pos = float(at.x) + last_run.x - ellipsis_width + dx;
        } else {
          float rw = last_run.reduce_to_fit(tf, space_left_for_last_run, runs_to_draw.length == 0);
          pos = rw + at.x + dx;
        }

        draw_one(last_run);

        static wchar ellipsis = 0x2026;

        handle<text_block> tl = elem->create_text_block(v, wchars(ellipsis));
        tl->set_width(v, tl->max_content_width(v));
        tl->set_height(v, tl->max_content_height(v));
        if (!last_run_is_ltr) pos -= tl->dim().x;
        int y, h, a;
        tl->get_first_line_metrics(v, y, h, a);
        tl->draw_glyphs(v, gfx, point(int(pos), int(last_y - a)));
      }
    }

    bool draw_path_ellipsis_runs() {

      handle<gool::font> font = v.get_font(elem_style);

      buffer<tflow::glyph_run, 64> head_runs(tf._glyph_runs.size());
      buffer<tflow::glyph_run, 64> tail_runs(0);

      slice<tflow::glyph_run> runs = tf._glyph_runs();

      tflow::glyph_run tail;

      uint slash_gi      = font->glyph_index('/');
      uint back_slash_gi = font->glyph_index('\\');

      for (int i = runs.size() - 1; i >= 0; --i) {
        auto grun      = runs[i];
        auto glyphs    = tf._glyph_indices(grun.glyph_start, grun.glyph_end());
        int  slash_pos = glyphs.last_index_of(slash_gi);
        if (slash_pos < 0) slash_pos = glyphs.last_index_of(back_slash_gi);
        if (slash_pos >= 0) {
          head_runs                = runs(0, i + 1);
          head_runs[i].glyph_count = uint(slash_pos);
          tail_runs                = runs(i);
          tail_runs[0].glyph_start += slash_pos;
          tail_runs[0].glyph_count = uint(glyphs.length) - uint(slash_pos);
          int tail_width           = int(0.5f + width_of(tf, tail_runs()));
          draw_ellipsis_runs(head_runs(), width - tail_width);
          float x = float(width - tail_width);
          for (i = 0; i < tail_runs.size(); ++i) {
            tflow::glyph_run &run = tail_runs[i];
            run.x                 = x;
            x += width_of(tf, run);
            draw_one(run);
          }
          return true;
        }
      }
      return false;
    }

    void draw() {
#ifdef DEBUG
      if (this->elem->parent->is_id_test())
        elem = elem;
#endif
      // iterate through all the saved glyph runs
      // and draw each one.
      if (!sctx && draw_ellipsis) {

        v.get_x_space_at(0, elem); // TODO: side effect, need to create fctx (???), check test-cases/floats/ellipsis-floats.htm

        if (draw_ellipsis == text_overflow_path_ellipsis &&
            draw_path_ellipsis_runs())
          /*done, we have drawn it*/;
        else if(elem_rtl)
          draw_ellipsis_runs_rtl(tf._glyph_runs(), width);
        else if(visible_lines.is_defined())
          draw_multiline_ellipsis_runs(tf._glyph_runs(), width);
        else
          draw_ellipsis_runs(tf._glyph_runs(), width);
      } else if (elem_rtl) {
        // tf._glyph_runs().each_backward(draw_one);
        for (int n = tf._glyph_runs.last_index();
             n >= 0 && n < tf._glyph_runs.size(); --n)
          draw_one(tf._glyph_runs[n]);
      } else {
        // tf._glyph_runs().each(draw_one);
        for (int n = 0; n < tf._glyph_runs.size(); ++n)
        {
            draw_one(tf._glyph_runs[n]);

        }
      }
    }

  }; // struct glyph_runs_painter

  void draw_glyph_runs(view &v, element *elem, const tflow::text_flow &tf,
                       gool::graphics *gfx, point at,
                       tflow::sel_context *sctx) {
    glyph_runs_painter(v, elem, tf, gfx, at, sctx).draw();
  }

#else

  void draw_glyph_runs(view &v, element *elem, const tflow::text_flow &tf,
                       gool::graphics *gfx, point at,
                       tflow::sel_context *sctx) {
    // iterate through all the saved glyph runs
    // and draw each one.

    handle<style> run_style;
    handle<font>  font;
    color_v       fore_color = tf._default_text_color;
    argb          fore;
    if (fore_color.is_defined()) fore = fore_color.to_argb();

    color_v       back_color;
    handle<style> elem_style     = elem->get_style(v);
    bool          elem_rtl       = elem_style->direction == direction_rtl;
    bool          rtl            = false; // run rtl
    node *        run_node       = nullptr;
    bool          is_highlighted = false;

    int   draw_ellipsis = 0;
    int   width         = elem->content_box(v).width();
    int   dx            = 0;
    float last_y        = 0;
    float last_h        = 0;

    if (elem->ldata->dim_min.x > width) {
      draw_ellipsis = elem_style->text_overflow;
      if (elem->scroll_pos().x == 0 && elem_rtl &&
          elem_style->overflow_x > overflow_visible)
        dx = width - elem->ldata->dim_min.x;
    }

    // helper functions:
    auto get_back_color = [&](const tflow::glyph_run &glyph_run) -> color_v {
      if (glyph_run.pstyle.is_defined()) {
        color_v cv = glyph_run.pstyle->back_color;
        if (glyph_run.pstyle && glyph_run.pstyle->back_color.is_defined())
          return glyph_run.pstyle->back_color;
      }

      helement pel =
          glyph_run.node->get_ui_element(); //->is_element()?
                                            //pn->cast<element>():pn->owner;
      while (pel.is_defined()) {
        if (pel == elem) break;
        if (pel->is_box_element(v)) break;
        const style *cs = pel->get_style(v);
        color_v      cv = cs->back_color;
        if (cv.is_defined()) {
          // element* te = elem;
          return cv;
        }
        pel = pel->ui_parent(v);
      }
      return color_v();
    };

    auto get_highlightion_status =
        [&](const tflow::glyph_run &glyph_run) -> bool {
      helement pel =
          glyph_run.node->get_ui_element(); //->is_element()?
                                            //pn->cast<element>():pn->owner;
      while (pel.is_defined()) {
        if (pel == elem) break;
        if (pel->is_box_element(v)) break;
        if (v.highlighted_ctl->el == pel) return true;
        pel = pel->ui_parent(v);
      }
      return false;
    };

    auto draw_glyph_back = [&](pointf dst, const tflow::glyph_run &glyph_run,
                               color_v back_color) {
      float       width = width_of(tf, glyph_run);
      gool::font *pf    = v.get_font(run_style);
      rect rc = rtl ? rect(int(dst.x - width + 0.5f), int(dst.y - pf->ascent),
                           int(dst.x + 0.5f), int(dst.y + pf->descent))
                    : rect(int(dst.x + 0.5f), int(dst.y - pf->ascent),
                           int(dst.x + width + 0.5f), int(dst.y + pf->descent));

      v.draw_glyph_run_back(gfx, tf, glyph_run, rc, back_color.to_argb());

      // gfx->fill(argb(back_color),rc);
    };
    auto draw_highlightion = [&](pointf                  dst,
                                 const tflow::glyph_run &glyph_run) {
      float       width = width_of(tf, glyph_run);
      gool::font *pf    = v.get_font(run_style);
      rect rc = rtl ? rect(int(dst.x - width + 0.5f), int(dst.y - pf->ascent),
                           int(dst.x + 0.5f), int(dst.y + pf->descent))
                    : rect(int(dst.x + 0.5f), int(dst.y - pf->ascent),
                           int(dst.x + width + 0.5f), int(dst.y + pf->descent));

      gfx->fill(v.highlighted_ctl->get_fore(), rc);
    };

    auto draw_text_decoration = [&](pointf                  dst,
                                    const tflow::glyph_run &glyph_run) {
      float       width = width_of(tf, glyph_run);
      gool::font *pf    = v.get_font(elem_style);
      gool::rectf rc;
      if (rtl) {
        rc.s.x = dst.x - width + 1.f;
        rc.e.x = dst.x;
      } else {
        rc.s.x = dst.x;
        rc.e.x = dst.x + width - 1.f;
      }
      if (run_style->text_decoration_line &
          text_decoration_underline) // includes text_decoration_underline
      {
        float thicknes = ceilf(pf->size / 15.0f);
        float shift    = thicknes;
        rc.s.y    = dst.y + shift;
        rc.e.y    = dst.y + shift + thicknes - 1.f;
        // gfx->fill(fore,rc);
        draw_decoration(v, run_style, gfx, rc, thicknes, fore);
      } else if (run_style->text_decoration_line &
                 text_decoration_double) // includes text_decoration_underline
      {
        float thicknes = ceilf(pf->size / 15.0f);
        float shift    = thicknes;
        rc.s.y    = dst.y + shift;
        rc.e.y    = dst.y + shift + thicknes - 1.f;
        // gfx->fill(fore, rc);
        draw_decoration(v, run_style, gfx, rc, thicknes, fore);
        // if ((run_style->text_decoration_line &
        // text_decoration_double_underline) == text_decoration_double_underline)
        //{
        rc.s.y = dst.y + shift + 2 * thicknes;
        rc.e.y = dst.y + shift + 2 * thicknes + thicknes - 1.f;
        // gfx->fill(fore, rc);
        //}
      }

      if (run_style->text_decoration_line & text_decoration_linethrough) {
        float thicknes = ceilf(pf->size / 15.0f);
        float shift    = float(pf->ascent / 3);
        rc.s.y    = dst.y - shift;
        rc.e.y    = dst.y - shift + thicknes - 1.f;
        // gfx->fill(fore,rc);
        draw_decoration(v, run_style, gfx, rc, thicknes, fore);
      }
      if (run_style->text_decoration_line & text_decoration_overline) {
        float thicknes = ceilf(pf->size / 15.0f);
        float shift    = float(pf->ascent);
        rc.s.y    = dst.y - shift;
        rc.e.y    = dst.y - shift + thicknes - 1.f;
        // gfx->fill(fore,rc);
        draw_decoration(v, run_style, gfx, rc, thicknes, fore);
      }
#if 0
        if(run_style->text_decoration_line == text_decoration_wavy )
        {
          float thicknes = ceilf(pf->size / 15.0f);
          size pixels = v.pixels_per_dip(size(1, 1));
          rc.s.y = dst.y + 1;
          rc.e.y = rc.s.y + 2 * pixels.y - 1;
          draw_decoration(run_style, gfx, rc, thicknes, fore);
          /*size pixels = v.pixels_per_dip(size(1,1));
          if( pixels.x < 2 ) {
            static handle<image> wavy;
            if( !wavy ) {
              bytes data = app()->get_resource(W("red-wave.png"));
              if(!data) return;
              wavy = image::create(data,"sciter:red-wave.png");
              if(!wavy) return;
            }
		        rc.s.y = dst.y + 1;
		        rc.e.y = rc.s.y + 2;
            gfx->fill(wavy,rc);
          } else {
            static handle<image> wavy;
            if( !wavy ) {
              bytes data = app()->get_resource(W("red-wave-2x.png"));
              if(!data) return;
              wavy = image::create(data,"sciter:red-wave-2x.png");
              if(!wavy) return;
            }
		        rc.s.y = dst.y + 1;
		        rc.e.y = rc.s.y + 5;
            gfx->fill(wavy,rc);
          }*/
        }
#endif

    };

    auto draw_glyph_run = [&](pointf dst, const tflow::glyph_run &run) {
      v.draw_glyph_run(gfx, tf, run, dst, fore, run_style);
      if (run_style->text_decoration_line != 0) draw_text_decoration(dst, run);
    };

    auto draw_ime_composition_glyph_run = [&](pointf                  dst,
                                              const tflow::glyph_run &glyph_run,
                                              const range &           yr) {
      pointf tdst = dst;

      uint rstart = sctx->ime_glyph_start < glyph_run.glyph_start
                        ? glyph_run.glyph_start
                        : sctx->ime_glyph_start;
      uint rend = sctx->ime_glyph_end >= glyph_run.glyph_end()
                      ? glyph_run.glyph_end()
                      : sctx->ime_glyph_end;

      if (rstart > glyph_run.glyph_start) // selection start is inside the run
      {
        tflow::glyph_run tdw = glyph_run;
        tdw.glyph_count      = rstart - glyph_run.glyph_start;
        draw_glyph_run(tdst, tdw);
        if (rtl)
          tdst.x -= width_of(tf, tdw);
        else
          tdst.x += width_of(tf, tdw);
      }

      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rstart - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = rend - rstart;
        if (tdw.glyph_end() <= tf._glyph_indices.length()) {
          float width = width_of(tf, tdw);
          if (width) {
            rect rc =
                rtl ? rect(int(tdst.x - width + 0.5f), yr.l, int(tdst.x), yr.h)
                    : rect(int(tdst.x), yr.l, int(tdst.x + width - 0.5f), yr.h);
            draw_glyph_run(tdst, tdw);
            if (rtl)
              tdst.x -= width_of(tf, tdw);
            else
              tdst.x += width_of(tf, tdw);
            rc.e.y += 1;
            rc.s.y = rc.e.y - 1;
            gfx->fill(sctx->selection_back_color, rc);
          }
        }
      }
      if (rend < glyph_run.glyph_end()) // selection start is inside the run
      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rend - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = glyph_run.glyph_end() - rend;
        draw_glyph_run(tdst, tdw);
      }
    };

    auto draw_selected_glyph_run = [&](pointf                  dst,
                                       const tflow::glyph_run &glyph_run,
                                       const range &           yr) {
      pointf tdst = dst;
      // selection covers this glyph run
      uint rstart = sctx->sel_glyph_start < glyph_run.glyph_start
                        ? glyph_run.glyph_start
                        : sctx->sel_glyph_start;
      uint rend = sctx->sel_glyph_end >= glyph_run.glyph_end()
                      ? glyph_run.glyph_end()
                      : sctx->sel_glyph_end;

      if (rstart > glyph_run.glyph_start) // selection start is inside the run
      {
        tflow::glyph_run tdw = glyph_run;
        tdw.glyph_count      = rstart - glyph_run.glyph_start;
        // render_target->DrawGlyphRun(tdst, &tdw, fore, dw_measuring_mode);
        if (!v._draw_selection_only) draw_glyph_run(tdst, tdw);
        if (rtl)
          tdst.x -= width_of(tf, tdw);
        else
          tdst.x += width_of(tf, tdw);
      }

      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rstart - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = rend - rstart;
        float width     = width_of(tf, tdw);
        if (width) {
          rect rc =
              rtl ? rect(int(tdst.x - width + 0.5f), yr.l, int(tdst.x), yr.h)
                  : rect(int(tdst.x), yr.l, int(tdst.x + width - 0.5f), yr.h);
          gfx->fill(sctx->selection_back_color, rc);
          // render_target->DrawGlyphRun(tdst, &tdw,
          // gfx->get_brush(sctx->selection_text_color), dw_measuring_mode);
          argb saved_fore = fore;
          fore            = argb(sctx->selection_text_color);
          draw_glyph_run(tdst, tdw);
          fore = saved_fore;
          if (rtl)
            tdst.x -= width_of(tf, tdw);
          else
            tdst.x += width_of(tf, tdw);
        }
      }
      if (rend < glyph_run.glyph_end()) // selection start is inside the run
      {
        tflow::glyph_run tdw   = glyph_run;
        uint             delta = rend - glyph_run.glyph_start;
        tdw.glyph_start += delta;
        tdw.glyph_count = glyph_run.glyph_end() - rend;
        if (!v._draw_selection_only) draw_glyph_run(tdst, tdw);
      }
    };

    auto draw_one = [&](const tflow::glyph_run &run) -> bool {
      const tflow::layout_line &line = tf._lines[run.line_no];

      assert(run.glyph_count < 10000);

      if (run.glyph_count == 0) return false; // continue;

      pointf dst;
      dst.x = at.x + run.x + dx;
      // if(lf_glyph_run.bidi_level & 1)
      //  dst.x -= width_of(tf,lf_glyph_run);

      html::element *el = run.get_inline_block_element(v);
      if (el) {
        if (!el->is_it_visible(v)) return false;
        if (el->positioned(v)) return false;

        auto_state<selection_ctx *> _(v._selection_ctx, 0);

        if (sctx && (sctx->sel_glyph_start < sctx->sel_glyph_end) &&
            (sctx->sel_glyph_start <= run.glyph_start) &&
            (sctx->sel_glyph_end >= run.glyph_end())) {
          range yr = line.yr() + at.y;
          float w  = width_of(tf, run);
          rect  rc(int(dst.x), yr.l, int(dst.x + w - 1), yr.h);

          rect trc = el->border_box(v);

          gfx->fill(sctx->selection_back_color, rc);
          el->draw(v, gfx, at + el->ldata->pos);
          gfx->fill(sctx->selection_fore_color,
                    el->border_box(v, element::TO_PARENT) + at);
        } else
          el->draw(v, gfx, at + el->ldata->pos);
        return false;
      }

      if (tf._glyph_indices[run.glyph_start] == 0) return false;
      if (!run.node) return false;
      html::style *t_run_style = run.get_style(v);
      if (!t_run_style) return false;

      if (!run.node->get_element()->is_drawable(v, elem)) return false;

      rtl  = run.is_rtl();
      font = run.font;

      if (run_style != t_run_style) {
        run_style = t_run_style;
        if (tf._default_text_color.is_defined())
          ;
        else if (fore_color.is_undefined() ||
                 fore_color != run_style->font_color) {
          fore_color = run_style->font_color;
          fore       = fore_color.to_argb();
        }
        back_color = get_back_color(run);
      }
      if (run_node != run.node) {
        run_node = run.node;
        if (v.highlighted_ctl && v.highlighted_ctl->el)
          is_highlighted = get_highlightion_status(run);
      }

      /*if(run.marks)
        fore = argb(0xFF,0,0);
      else
        fore = argb(fore_color);*/

      switch (run.valign) {
      case valign_auto:
      case valign_baseline:
      case valign_sub:
      case valign_super: {
        dst.y       = float(line.baseline);
        element *el = run.get_element();
        if (el == elem) break;
        dst.y += el->get_baseline_shift(v, elem);
      } break;
      case valign_text_top:
      case valign_top: {
        dst.y = float(run.font->ascent);
        break;
      }
      case valign_middle: {
        int delta = (line.height - run.font->ascent - run.font->descent) / 2;
        dst.y     = float(delta + run.font->ascent);
      } break;
      case valign_text_bottom:
      case valign_bottom: dst.y = float(line.height - run.font->descent); break;
      }

      dst.y += at.y + line.y;
      last_y = dst.y;
      last_h = float(line.height);

#pragma TODO("position:relative; display:inline !")

      if (sctx && (sctx->sel_glyph_start < sctx->sel_glyph_end) &&
          (sctx->sel_glyph_start < run.glyph_end()) &&
          (sctx->sel_glyph_end >= run.glyph_start)) {
        draw_selected_glyph_run(dst, run, line.yr() + at.y);
      } else if (sctx && (sctx->ime_glyph_start < sctx->ime_glyph_end) &&
                 (sctx->ime_glyph_start < run.glyph_end()) &&
                 (sctx->ime_glyph_end >= run.glyph_start)) {
        draw_ime_composition_glyph_run(dst, run, line.yr() + int(at.y));
      }
#if 0
          else if(draw_ellipsis)
          {
            bool last_one = elem_rtl
                            ? draw_glyph_run_ellipsis_rtl(dst, run, draw_ellipsis)
                            : draw_glyph_run_ellipsis(dst, run, draw_ellipsis);
            if( last_one )
              break;
          }
#endif
      else if (
          !v._draw_selection_only /*&& !v.disable_glyph_run_draw(gfx,tf,run)*/) {
        if (back_color.is_defined()) draw_glyph_back(dst, run, back_color);
        draw_glyph_run(dst, run);
        if (is_highlighted) draw_highlightion(dst, run);
      }

      return false;

    };

    auto draw_ellipsis_runs = [&](slice<tflow::glyph_run> runs, int width) {

      handle<gool::font> font = v.get_font(elem_style);

      float ellipsis_width = font->size * 0.85f;

      float wme = width - ellipsis_width;

      slice<tflow::glyph_run> runs_to_draw = runs;

      float space_left_for_last_run = wme;
      int_v last_run_idx;

      if (!elem_rtl)
        for (int i = runs.size() - 1; i >= 0; --i) {
          auto glyph_run = runs[i];
          auto rx        = html::x_of(tf, glyph_run);
          if (rx.h < wme) {
            space_left_for_last_run = wme - rx.h;
            break;
          }
          runs_to_draw.length = uint(i);
          last_run_idx        = i;
        }
      else {
        float invisible = elem->ldata->inner_dim.x - wme;
        for (int i = 0; i < runs.size(); ++i) {
          auto glyph_run = runs[i];
          auto rx        = html::x_of(tf, glyph_run);
          if (rx.h > invisible) {
            space_left_for_last_run = rx.h - invisible;
            runs_to_draw.prune(1);
            last_run_idx = i;
            break;
          }
          runs_to_draw.prune(1);
          last_run_idx = i;
        }
      }

      if (elem_rtl)
        runs_to_draw.each_backward(draw_one);
      else
        runs_to_draw.each(draw_one);

      if (last_run_idx.is_defined()) {
        tflow::glyph_run last_run = runs[last_run_idx];

        bool  last_run_is_ltr = !last_run.is_rtl();
        float pos;
        if (last_run_is_ltr && elem_rtl) {
          float run_width = width_of(tf, last_run);
          last_run.reduce_to_fit(tf, space_left_for_last_run);
          float new_run_width = width_of(tf, last_run);
          last_run.x += run_width - new_run_width;
          pos = float(at.x) + last_run.x - ellipsis_width + dx;
        } else {
          float rw = last_run.reduce_to_fit(tf, space_left_for_last_run);
          pos      = rw + at.x + dx;
        }

        draw_one(last_run);

        static wchar ellipsis = 0x2026;

        handle<text_block> tl = elem->create_text_block(v, wchars(ellipsis));
        tl->set_width(v, tl->max_content_width(v));
        tl->set_height(v, tl->max_content_height(v));
        if (!last_run_is_ltr) pos -= tl->dim().x;
        // auto_state<argb> _(gfx->text_color, fore);
        int y, h, a;
        tl->get_first_line_metrics(v, y, h, a);
        tl->draw_glyphs(v, gfx, point(int(pos), int(last_y - a)));
      }

    };

    auto draw_path_ellipsis_runs = [&]() -> bool {

      handle<gool::font> font = v.get_font(elem_style);

      buffer<tflow::glyph_run, 64> head_runs(tf._glyph_runs.size());
      buffer<tflow::glyph_run, 64> tail_runs(0);

      slice<tflow::glyph_run> runs = tf._glyph_runs();

      tflow::glyph_run tail;

      uint slash_gi      = font->glyph_index('/');
      uint back_slash_gi = font->glyph_index('\\');

      for (int i = runs.size() - 1; i >= 0; --i) {
        auto grun      = runs[i];
        auto glyphs    = tf._glyph_indices(grun.glyph_start, grun.glyph_end());
        int  slash_pos = glyphs.last_index_of(slash_gi);
        if (slash_pos < 0) slash_pos = glyphs.last_index_of(back_slash_gi);
        if (slash_pos >= 0) {
          head_runs                = runs(0, i + 1);
          head_runs[i].glyph_count = uint(slash_pos);
          tail_runs                = runs(i);
          tail_runs[0].glyph_start += slash_pos;
          tail_runs[0].glyph_count = uint(glyphs.length) - uint(slash_pos);
          int tail_width           = int(0.5f + width_of(tf, tail_runs()));
          draw_ellipsis_runs(head_runs(), width - tail_width);
          float x = float(width - tail_width);
          for (i = 0; i < tail_runs.size(); ++i) {
            tflow::glyph_run &run = tail_runs[i];
            run.x                 = x;
            x += width_of(tf, run);
            draw_one(run);
          }
          return true;
        }
      }
      return false;
    };

    if (!sctx && draw_ellipsis) {
      if (draw_ellipsis == text_overflow_path_ellipsis &&
          draw_path_ellipsis_runs())
        /*done, we have drawn it*/;
      else
        draw_ellipsis_runs(tf._glyph_runs(), width);
    } else if (elem_rtl) {
      // tf._glyph_runs().each_backward(draw_one);
      for (int n = tf._glyph_runs.last_index();
           n >= 0 && n < tf._glyph_runs.size(); --n)
        draw_one(tf._glyph_runs[n]);
    } else {
      // tf._glyph_runs().each(draw_one);
      for (int n = 0; n < tf._glyph_runs.size(); ++n)
        draw_one(tf._glyph_runs[n]);
    }
  }
#endif

#pragma warning(pop)

  bool produce_filter_graph(view &v, element *element, slice<value> filters,filter_graph_builder *dest, bool as_ppx) {

    element_context ctx(v, element);

    for (size_t n = 0; n < filters.length; ++n) {

      value el = filters[n];

      if (el.is_function(WCHARS("blur"))) {
        const function_value *pf          = el.get_function();
        size_v                blur_radius = ctx.length_value(pf->params.value(0));
        float deviation = as_ppx ? pixels(v, element, blur_radius).width_f()
                                 : dips(v, element, blur_radius).width_f();
        dest->add_blur(deviation);
      } else if (el.is_function(WCHARS("brightness"))) {
        const function_value *pf = el.get_function();
        float                 p  = pf->params.value(0).get_percent();
        dest->add_brightness(p);
      } else if (el.is_function(WCHARS("contrast"))) {
        const function_value *pf = el.get_function();
        float                 p  = pf->params.value(0).get_percent();
        dest->add_contrast(p);
      } else if (el.is_function(WCHARS("grayscale"))) {
        const function_value *pf = el.get_function();
        float                 p  = pf->params.value(0).get_percent();
        dest->add_grayscale(p);
      } else if (el.is_function(WCHARS("hue-rotate"))) {
        const function_value *pf    = el.get_function();
        float                 angle = pf->params.value(0).get_angle();
        dest->add_hue_rotate(angle);
      } else if (el.is_function(WCHARS("invert"))) {
        dest->add_invert();
      } else if (el.is_function(WCHARS("opacity"))) {
        const function_value *pf = el.get_function();
        float                 p  = pf->params.value(0).get_percent();
        dest->add_opacity(p);
      } else if (el.is_function(WCHARS("saturate"))) {
        const function_value *pf = el.get_function();
        float                 p  = pf->params.value(0).get_percent();
        dest->add_saturate(p);
      } else if (el.is_function(WCHARS("sepia"))) {
        const function_value *pf = el.get_function();
        float                 p  = pf->params.value(0).get_percent();
        dest->add_sepia(p);
      } else if (el.is_function(WCHARS("drop-shadow"))) // drop-shadow( <length>{2,4} <color>? )
      {
        const function_value *pf = el.get_function();
        int                   ln = 0;
        size_v                offset_x, offset_y;
        size_v                blur_radius, spread_radius;
        argb c = element->get_style(v)->font_color.to_argb();
        for (int n = 0; n < pf->params.size(); ++n) {
          value val = pf->params.value(n);
          if (val.is_length()) switch (ln++) {
            case 0: offset_x = ctx.length_value(val); break;
            case 1: offset_y = ctx.length_value(val); break;
            case 2: blur_radius = ctx.length_value(val); break;
            case 3: spread_radius = ctx.length_value(val); break;
            default: return false;
            }
          else if (val.is_color())
            c = ctx.color_value(val);
          else
            return false;
        }
        // all params are ok
        dest->add_drop_shadow(
            as_ppx ? pixels(v, element, offset_x).width_f() : dips(v, element, offset_x).width_f(),
            as_ppx ? pixels(v, element, offset_y).width_f() : dips(v, element, offset_y).width_f(),
            blur_radius.is_defined() ? (as_ppx ? pixels(v, element, blur_radius).width_f(): dips(v, element, blur_radius).width_f()) : 0.0f,
            spread_radius.is_defined() ? (as_ppx ? pixels(v, element, spread_radius).width_f(): dips(v, element, spread_radius).width_f()) : 0.0f,
            c);
      }
    }
    return true;
  }

} // namespace html
