#ifndef __html_dom_traversal_h__
#define __html_dom_traversal_h__

#include "tool/tool.h"
#include "html-dom.h"
#include "html-layout.h"

namespace html {

  // typedef bool element_visitor_filter(view& v, element* pe);
  typedef function<bool(view &, element *)> element_visitor_filter;

  namespace walk {
    INLINE helement next(helement el, helement root) {
      helement t = el->next_element();
      if (t) return t;
      while (el) {
        el = el->parent;
        if ((el == root) || !el) return 0;
        t = el->next_element();
        if (t) return t;
      }
      assert(0);
      return nullptr;
    }
    INLINE helement first_or_next(helement el, helement root) {
      helement t = el->first_element();
      return t ? t : next(el, root);
    }
    INLINE helement prev(helement el, helement root) {
      helement t = el->prev_element();
      if (t) return t;
      while (el) {
        el = el->parent;
        if ((el == root) || !el) return 0;
        t = el->prev_element();
        if (t) return t;
      }
      assert(0);
      return nullptr;
    }
    INLINE helement last_or_prev(helement el, helement root) {
      helement t = el->last_element();
      return t ? t : prev(el, root);
    }

    INLINE helement next_ui(helement el, helement root) {
      helement t = el->next_ui_element();
      if (t) return t;
      while (el) {
        el = el->get_owner();
        if (!el || (el == root)) return 0;
        t = el->next_ui_element();
        if (t) return t; 
      }
      assert(0);
      return nullptr;
    }

    INLINE helement first_or_next_ui(helement el, helement root) {
      helement t = el->first_ui_element();
      return t ? t : next_ui(el, root);
    }

    INLINE hnode next(hnode n, hnode root) {
      hnode t = n->next_node();
      if (t) return t;
      while (n) {
        n = n->parent;
        if ((n == root) || !n) return 0;
        t = n->next_node();
        if (t) return t;
      }
      assert(0);
      return nullptr;
    }
    INLINE hnode first_or_next(hnode n, hnode root) {
      hnode t = n->first_node();
      return t ? t : next(n, root);
    }

    INLINE hnode prev(hnode n, hnode root) {
      hnode t = n->prev_node();
      if (t) return t;
      while (n) {
        n = n->parent;
        if (n == root || !n) return 0;
        t = n->prev_node();
        if (t) return t;
      }
      assert(0);
      return nullptr;
    }
    INLINE hnode last_or_prev(hnode n, hnode root) {
      hnode t = n->last_node();
      return t ? t : prev(n, root);
    }

    INLINE helement very_last_ui(helement el, helement dv = 0) {
      helement t = dv;
      while (el) {
        el = el->last_ui_element();
        if (!el) break;
        t = el;
      }
      return t;
    }

    INLINE helement prev_ui(helement el, helement root) {
      helement t = el->prev_ui_element();
      if (t) return t;
      while (el) {
        el = el->get_owner();
        if (!el || el == root) return 0;
        t = el->prev_ui_element();
        if (t) return t;
      }
      assert(0);
      return nullptr;
    }

    INLINE helement last_or_prev_ui(helement el, helement root) {
      helement t = el->prev_ui_element();
      if (t) return very_last_ui(t, t);
      el = el->get_owner();
      return el == root ? 0 : el;
    }

  } // namespace walk

  $generator(element_iterator) {
    view &                 v;
    bool                   forward; // direction
    helement               root;
    helement               b;
    element_visitor_filter pfilter;
    element_visitor_filter pskiper;
    bool shallow; // if true - does not go inside accepted element
    element_iterator(
        view & v_, element * root_blk, element_visitor_filter pf = nullptr,
        element_visitor_filter ps = nullptr, bool shallow_walk = false)
        : v(v_), forward(true), root(root_blk), pfilter(pf), pskiper(ps),
          shallow(shallow_walk) {}

    element_iterator(bool fwd, view &v_, element *root_blk,
                     element_visitor_filter pf           = nullptr,
                     element_visitor_filter ps           = nullptr,
                     bool                   shallow_walk = false)
        : forward(fwd), v(v_), root(root_blk), pfilter(pf), pskiper(ps),
          shallow(shallow_walk) {}

    inline bool skip(element * b) {
      if (!pskiper) return false;
      return pskiper(v, b);
    }
    inline bool accept(element * b) {
      if (!pfilter) return true;
      return pfilter(v, b);
    }

    inline void rewind(element *t = 0) {
      assert(!t || t->belongs_to(root));
      b = t;
    }

    $emit(element *) if (forward) {
      if (!b)
        b = root->first_element();
      else
        b = walk::first_or_next(b, root);
      for (; b;) {
        if (skip(b)) {
          b = walk::next(b, root);
          continue;
        }
        if (accept(b)) {
          $yield(b);
          if (!b) break;
          if (shallow) {
            b = walk::next(b, root);
            continue;
          }
        }
        b = walk::first_or_next(b, root);
      }
    }
    else {
      if (!b)
        b = root->last_element();
      else
        b = walk::last_or_prev(b, root);
      for (; b;) {
        if (skip(b)) {
          b = walk::prev(b, root);
          continue;
        }
        if (accept(b)) {
          $yield(b);
          if (!b) break;
          if (shallow) {
            b = walk::prev(b, root);
            continue;
          }
        }
        b = walk::last_or_prev(b, root);
      }
    }
    $stop
  };

  $generator(element_ui_iterator) {
    view &                 v;
    bool                   forward; // direction
    helement               root;
    helement               b;
    element_visitor_filter pfilter;
    element_visitor_filter pskiper;
    bool shallow; // if true - does not go inside accepted element
    element_ui_iterator(
        view & v_, element * root_blk, element_visitor_filter pf = nullptr,
        element_visitor_filter ps = nullptr, bool shallow_walk = false)
        : v(v_), forward(true), root(root_blk), pfilter(pf), pskiper(ps),
          shallow(shallow_walk) {}

    element_ui_iterator(bool fwd, view &v_, element *root_blk,
                        element_visitor_filter pf           = nullptr,
                        element_visitor_filter ps           = nullptr,
                        bool                   shallow_walk = false)
        : forward(fwd), v(v_), root(root_blk), pfilter(pf), pskiper(ps),
          shallow(shallow_walk) {}

    inline bool skip(element * b) {
      if (!pskiper) return false;
      return pskiper(v, b);
    }
    inline bool accept(element * b) {
      if (!pfilter) return true;
      return pfilter(v, b);
    }

    inline void rewind(element *t = 0) {
      assert(!t || t->owned_by(root));
      b = t;
    }

    $emit(element *) if (forward) {
      if (!b)
        b = root->first_ui_element();
      else
        b = walk::first_or_next_ui(b, root);
      for (; b;) {
        if (skip(b)) {
          b = walk::next_ui(b, root);
          continue;
        }
        if (accept(b)) {
          $yield(b);
          if (!b) break;
          if (shallow) {
            b = walk::next_ui(b, root);
            continue;
          }
        }
        b = walk::first_or_next_ui(b, root);
      }
    }
    else {
      if (!b)
        b = root->last_ui_element();
      else
        b = walk::last_or_prev_ui(b, root);
      for (; b;) {
        if (skip(b)) {
          b = walk::prev_ui(b, root);
          continue;
        }
        if (accept(b)) {
          $yield(b);
          if (!b) break;
          if (shallow) {
            b = walk::prev_ui(b, root);
            continue;
          }
        }
        b = walk::last_or_prev_ui(b, root);
      }
    }
    $stop
  };

  $generator(each_element) {
    helement root;
    helement b;
    bool     dont_go_inside;
    each_element(element * root_blk) : root(root_blk), dont_go_inside(false) {}

    void skip() { dont_go_inside = true; }

    $emit(element *) if (!b) b = root->first_element();
    else b                     = walk::first_or_next(b, root);
    for (; b;) {
      $yield(b);
      if (!b) break;
      if (dont_go_inside) {
        dont_go_inside = false;
        b              = walk::next(b, root);
      } else
        b = walk::first_or_next(b, root);
    }
    $stop
  };

  $generator(each_element_backward) {
    helement root;
    helement b;
    bool     dont_go_inside;
    each_element_backward(element * root_blk)
        : root(root_blk), dont_go_inside(false) {}

    void skip() { dont_go_inside = true; }

    $emit(element *) if (!b) b = root->last_element();
    else b                     = walk::last_or_prev(b, root);
    for (; b;) {
      $yield(b);
      if (!b) break;
      if (dont_go_inside) {
        dont_go_inside = false;
        b              = walk::prev(b, root);
      } else
        b = walk::last_or_prev(b, root);
    }
    $stop
  };

  $generator(each_ui_element) {
    helement root;
    helement b;
    bool     dont_go_inside;
    each_ui_element(element * root_blk)
        : root(root_blk), dont_go_inside(false) {}

    void skip() { dont_go_inside = true; }

    $emit(element *) if (!b) b = root->first_ui_element();
    else b                     = walk::first_or_next_ui(b, root);
    for (; b;) {
      $yield(b);
      if (!b) break;
      if (dont_go_inside) {
        dont_go_inside = false;
        b              = walk::next_ui(b, root);
      } else
        b = walk::first_or_next_ui(b, root);
    }
    $stop
  };

  $generator(each_ui_element_backward) {
    helement root;
    helement b;
    bool     dont_go_inside;
    each_ui_element_backward(element * root_blk)
        : root(root_blk), dont_go_inside(false) {}

    void skip() { dont_go_inside = true; }

    $emit(element *) if (!b) b = walk::very_last_ui(root);
    else b                     = walk::last_or_prev_ui(b, root);
    for (; b;) {
      $yield(b);
      if (!b) break;
      if (dont_go_inside) {
        dont_go_inside = false;
        b              = walk::prev_ui(b, root);
      } else
        b = walk::last_or_prev_ui(b, root);
    }
    $stop
  };

  $generator(each_node) 
  {
    hnode root;
    hnode n;
    bool  dont_go_inside;
    bool  forward;
    each_node(node * root_blk, bool f = true) : root(root_blk), dont_go_inside(false), forward(f) {}

    void skip() { dont_go_inside = true; }

    $emit(hnode) 
      if (!n)
        n = forward ? root->first_node() : root->last_node();
      while (n) {
        $yield(n);
        if (forward) {
          if (dont_go_inside) {
            dont_go_inside = false;
            n = walk::next(n, root);
          }
          else
            n = walk::first_or_next(n, root);
        }
        else {
          if (dont_go_inside) {
            dont_go_inside = false;
            n = walk::prev(n, root);
          }
          else
            n = walk::last_or_prev(n, root);
        }
      }
    $stop
  };

  struct each_node_backward : public each_node {
    each_node_backward(node * root_blk) : each_node(root_blk,false) {}
  };

  struct element_visitor_context {
    virtual bool accept(view &v, element *pe) = 0;
    virtual bool skip(view &v, element *pe) { return false; }
  };

  $generator(element_iterator_ctx) {
    view &                   v;
    helement                 root;
    helement                 b;
    element_visitor_context &ctx;
    bool shallow; // if true - does not go inside accepted element
    element_iterator_ctx(view & v_, const element *root_blk,
                         element_visitor_context &vctx,
                         bool                     shallow_walk = false)
        : v(v_), root(root_blk), ctx(vctx), shallow(shallow_walk) {}

    inline bool skip(element * b) { return ctx.skip(v, b); }
    inline bool accept(element * b) { return ctx.accept(v, b); }

    $emit(element *) for (b = root->first_element(); b;) {
      //b->dbg_report("element_iterator");
      if (skip(b)) {
        b = walk::next(b, root);
        continue;
      }
      if (accept(b)) {
        $yield(b);
        if (shallow) {
          b = walk::next(b, root);
          continue;
        }
      }
      b = walk::first_or_next(b, root);
    }
    $stop
  };

  // immediate child iterator
  $generator(each_child) {
    helement root;
    helement b;

    each_child(const element *root_blk) : root(root_blk) {}

    void rewind(element *t = 0) { b = t; }

    $emit(element *) for (b = b ? b : b = root->first_element(); b;
                          b             = b->next_element()) {
      $yield(b);
    }
    $stop
  };

  // extended immediate child iterator
  $generator(child_iterator) {
    view &                 v;
    helement               root;
    helement               b;
    element_visitor_filter pfilter;
    element_visitor_filter pskiper;

    child_iterator(view & vv, const element *root_blk,
                   element_visitor_filter filter = nullptr,
                   element_visitor_filter skiper = nullptr)
        : v(vv), root(root_blk), pfilter(filter), pskiper(skiper) {}

    void rewind(element *t = 0) { b = t; }

    $emit(element *) if (pfilter || pskiper) for (b = b ? b->next_element()
                                                        : root->first_element();
                                                  b; b = b->next_element()) {
      if (pskiper && pskiper(v, b)) continue;
      if (pfilter) {
        if (pfilter(v, b)) { $yield(b); }
      } else {
        $yield(b);
      }
    }
    else for (b = b ? b : b = root->first_element(); b; b = b->next_element()) {
      $yield(b);
    }
    $stop
  };

  $generator(child_iterator_backward) {
    view &                 v;
    helement               root;
    helement               b;
    element_visitor_filter pfilter;
    element_visitor_filter pskiper;

    child_iterator_backward(view & vv, const element *root_blk,
                            element_visitor_filter filter = 0,
                            element_visitor_filter skiper = 0)
        : v(vv), root(root_blk), pfilter(filter), pskiper(skiper) {}

    void rewind(element *t = 0) { b = t; }

    $emit(element *) if (pfilter || pskiper) for (b = b ? b->prev_element()
                                                        : root->last_element();
                                                  b; b = b->prev_element()) {
      if (pskiper && pskiper(v, b)) continue;
      if (pfilter) {
        if (pfilter(v, b)) { $yield(b); }
      }
    }
    else for (b = b ? b : b = root->last_element(); b; b = b->prev_element()) {
      $yield(b);
    }
    $stop
  };

  $generator(pos_iterator) {
    bookmark   start;
    bookmark   end;
    array<int> end_stack;
    bookmark   bm;
    bookmark   t;
    bool       is_forward;
    wchar      char_code;

    pos_iterator(bookmark s, bookmark e, bool forward = true)
        : start(s), end(e), is_forward(forward), char_code(0) {
      if (start > end) swap(start, end);
      start.linearize();
      end.linearize();
      end.stack(end_stack);
    }

    $emit(bookmark) 
      if (is_forward) for (bm = start; bm.valid() && bm < end_stack; bm.advance_forward(char_code)) 
      {
        t = bm;
        $yield(bm);
        if (t != bm && bm >= end) break;
      }
      else for (bm = end; bm.advance_backward(char_code) && bm.valid();) 
      {
        t = bm;
        $yield(bm);
        if (bm == start) break;
        if (t != bm && bm < start) break;
      }
    $stop
  };

  $generator(pos_ui_iterator) {
    view &       v;
    bookmark     start;
    bookmark     end;
    bookmark     bm;
    bool         is_forward;
    array<wchar> char_buf;

    pos_ui_iterator(view & v_, bookmark s, bookmark e, bool forward = true)
        : v(v_), start(s), end(e), is_forward(forward) {
      if (start > end) swap(start, end);
      start.linearize();
      end.linearize();
      /*if( !start.at_caret_pos() && !start.at_element_start() &&
      !start.at_element_end() )
      {
        assert(false);
        start = end = bookmark();
      }
      else if( !end.at_caret_pos() && !end.at_element_start() &&
      !end.at_element_end() )
      {
        assert(false);
        start = end = bookmark();
      }*/
    }

    $emit(bookmark) if (is_forward) for (bm = start; bm.valid() && bm <= end;
                                         bm.advance_caret_pos(v, ADVANCE_NEXT,
                                                              char_buf)) {
      $yield(bm);
    }
    else for (bm = end;
              bm.valid() && bm.advance_caret_pos(v, ADVANCE_PREV, char_buf);) {
      if (!bm.valid()) break;
      $yield(bm);
      if (bm < start) break;
    }
    $stop
  };

  /*
  $generator(element_shallow_iterator)
  {
    helement root;
    helement c;
    helement b;
    element_visitor_filter* pfilter;
    element_visitor_filter* pskiper;
    int    n;
    element_shallow_iterator( element* root_blk, element_visitor_filter* pf = 0,
element_visitor_filter* ps = 0 ): c(0), root(root_blk), pfilter( pf ),
pskiper(ps) { }

    void rewind() {
      _generator::rewind();
      c = 0;
      n = 0;
    }

    inline bool skip(element *b)
    {
      if(!b) return true;
      if(b == root) return false;
      if(!pskiper) return false;
      return pskiper(b);
    }
    inline bool accept(element *b)
    {
      if(!pfilter) return true;
      return pfilter(b);
    }

    $emit(element*)
       n = 0; c = root;
       while( c )
       {
         while( n < c->subs.size() )
         {
           b = c->subs[n];
           if( !b ) { ++n; continue; }
           if( skip( b ) ) { ++n; continue; }
           if( accept(b) ) { $yield(b); ++n; continue; }
           else { n = 0; c = b; }
         }
         if( c == root )
           break;
         n = c->index + 1; c = c->owner;
       }
    $stop
  };


  // immediate child iterator
  $generator(child_iterator)
  {
    helement root;
    element* b;
    element_visitor_filter* pfilter;
#ifdef PLATFORM_WINCE
    volatile int    n; // without it optimizer generates incorrect code
#else
    int    n;
#endif
    int    k;

    child_iterator( element* root_blk, element_visitor_filter* pf = 0 ):
       root(root_blk), pfilter( pf ) { k = root->subs.size(); }

    inline bool accept(element *b)
    {
      if(!pfilter) return true;
      return pfilter(b);
    }
    $emit(element*)
       for( n = 0; n < k; ++n)
       {
         b = root->subs[n];
         if( accept(b) )
            $yield(b);
       }
    $stop
  };
  */

  extern bool is_document_filter(view &v, element *el);

} // namespace html

#endif
