#ifndef __html_dom_merge_h__
#define __html_dom_merge_h__

#include "tool/tool.h"
#include "html-view.h"
#include "html-dom.h"

namespace html {

  inline tool::ustring node_key(html::context&, node* pn) {
    if (pn->is_element()) {
      ustring key = pn->cast<element>()->attr_id();
      if (key.is_defined())
        return key;
      key = pn->cast<element>()->attr_key();
      if (key.is_defined())
        return key;
      if (pn->cast<element>()->tag == tag::T_OPTION || pn->cast<element>()->tag == tag::T_BUTTON) {
        key = pn->cast<element>()->attr_value();
        if (key.is_defined())
          return key;
      }
      key = pn->cast<element>()->attr_name();
      if (key.is_defined())
        return key;
    }
    return ustring();
  }

  inline html::node::NODE_TYPE node_type(html::context&, node* pn) { return pn->node_type(); }
  inline uint      node_tag(html::context&, node* pn) {
    if (pn->is_element()) 
      return uint(pn->cast<element>()->tag);
    return uint(0);
  }
  inline ustring_chars node_text(context& pv, node* pn) { 
#ifdef DEBUG
    if (!pn->is_text())
      pn = pn;
    if(pn->cast<text>()->chars() == WCHARS("pythonw.exe"))
      pn = pn;
#endif // DEBUG
    return ustring_chars(pn->is_text() ? pn->cast<text>()->chars() : wchars()); 
  }
  inline uint      node_size(html::context&, node* pn) { return pn->is_element() ? uint(pn->cast<element>()->nodes.length()) : 0; }
  inline node*     node_child(context& pv, hnode &pn, uint i) {
    if (!pn) return nullptr;
    if (i >= node_size(pv,pn)) return nullptr;
    return pn->is_element() ? pn->cast<element>()->nodes[i] : nullptr;
  }
  inline node*     node_child_ex(context& pv, hnode &pn, uint i, hnode parent) {
    if (!pn) return nullptr;
    if (i >= node_size(pv, pn)) return nullptr;
    return pn->is_element() ? pn->cast<element>()->nodes[i] : nullptr;
  }

  inline hnode     node_new_to_old(context& pv, html::hnode nn, html::hnode parent) {
    return nn; 
  }

  inline bool node_new_to_old_ex(context& pv, hnode &left_parent, uint left_child_index, html::hnode right_parent, hnode& dom_child) {
    return node_child(pv,left_parent, left_child_index);
  }

  inline void      node_remove_child(html::context&, node* pn, uint i) {
    if (pn->is_element()) pn->cast<element>()->nodes[i]->remove(false);
  }


  // don't update children
  inline node*  node_for_children_update(html::context&, node* pn) {
    if (pn->is_element())
      return pn->cast<element>()->reconciliation_container();
    return pn;
  }

  inline const attribute_bag& node_atts(html::context&, node* pn) {
    static attribute_bag zero;
    return pn->is_element() ? pn->cast<element>()->atts : zero;
  }

  inline bool node_states(html::context&, node* pn, attribute_bag_v&) { // not used
    return false;
  }

  inline node* node_element_expand(context& pv, node* new_node, hnode& old_node, html::hnode parent) { return new_node; }

  template<typename pvnode> inline bool is_real_dom(pvnode pn) { return false; }

  template<> inline bool is_real_dom(node* pn) { return true; }

  inline void update_element_atts_states(context& pv, element* old_node, node* new_node, bool is_root) {
    auto newatts = node_atts(pv,new_node);
    if((node_atts(pv,old_node) != newatts) /*&& opts->update_atts(old_node, newatts)*/)
    {
      old_node->set_attributes(newatts, pv.pview(), is_root ? false : true);
    }
    attribute_bag_v newstates;
    if (node_states(pv,new_node, newstates)) {
      old_node->set_states(newstates, pv.pview(), true);
    }
  }

  uint64 node_factory(context& pv, node* pn);

  inline bool node_equal(context& pv, html::hnode a, html::hnode b) {
    html::node::NODE_TYPE nta = node_type(pv,a);
    if (nta != node_type(pv,b))
      return false;

    if (nta == html::node::NODE_TYPE::TEXT)
      return node_text(pv, a) == node_text(pv, b);

    if (nta != html::node::NODE_TYPE::ELEMENT)
      return false;

    if (node_tag(pv,a) != node_tag(pv,b))
      return false;

    ustring aid = node_key(pv,a);
    ustring bid = node_key(pv,b);
    if ((aid.is_defined() || bid.is_defined()) && (aid != bid))
      return false;

    if (node_size(pv,a) != node_size(pv,b))
      return false;

    for (uint n = 0; n < node_size(pv,a); ++n)
      if (!node_equal(pv, node_child(pv,a, n), node_child(pv,b, n)))
        return false;

    return true;

  }
  
  struct node_protector {
    node_protector(view* pv, hnode &n) {}
  };

  template<typename pvnode, typename NODE_PROTECTOR>
  class morph_t
  {
  public:

    typedef html::node::NODE_TYPE NODE_TYPE;

    morph_t(context& c) : pv(c) {}

  public:

    morph_options* opts = nullptr;
    context& pv;

    // Morph one tree into another tree
    //
    // no parent
    //   -> same: diff and walk children
    //   -> not same: replace and return
    // old node doesn't exist
    //   -> insert new node
    // new node doesn't exist
    //   -> delete old node
    // nodes are not the same
    //   -> diff nodes and apply patch to old node
    // nodes are the same
    //   -> walk all child nodes and append to old node

    // NOTE: will modify new_tree so it shall be discarded

    static helement exec(html::context& hc, helement old_tree, pvnode new_tree, morph_options* opts) {
      morph_t instance(hc);
      instance.opts = opts;

      //if (!instance.pv) instance.pv = view::get_current();
      //if (!instance.pv) return nullptr;

      mutator_ctx _(old_tree, instance.pv.pview());

      if (opts->children_only) {
        instance.update_children(new_tree, old_tree);
        return old_tree;
      }

      hnode n = instance.walk(new_tree, old_tree,true);

      return helement(n.ptr_of<element>());
    }

  protected:

    bool same_node_type(pvnode nnode, hnode onode) {
      return node_type(pv,nnode) == node_type(pv,onode);
    }

    bool are_similar(pvnode nnode, hnode onode) {
      if (!same_node_type(nnode, onode))
        return false;
      if (node_type(pv,nnode) == NODE_TYPE::ELEMENT) {
        if (node_tag(pv,nnode) != node_tag(pv,onode))
          return false;
        ustring aid = node_key(pv,nnode);
        if (aid.is_defined() && (aid != node_key(pv,onode)))
          return false;
      }
      else
        return node_text(pv, nnode) == node_text(pv, onode);

      return true;
    };

    // walk and morph a dom tree
    hnode walk(pvnode new_node, hnode old_node, bool is_root = false) {
      if (!old_node)
        return new_node ? node_new_to_old(pv,new_node, nullptr) : nullptr;
      else if (!new_node)
        return nullptr;

      if (node_type(pv,new_node) == NODE_TYPE::ELEMENT) {
        new_node = node_element_expand(pv,new_node, old_node, old_node->parent.ptr());
        if (!new_node)
          return nullptr;
      }


      NODE_PROTECTOR _(pv, new_node);

      /*if (!are_similar(new_node, old_node) && !is_root)
        return node_new_to_old(pv,new_node, old_node->parent.ptr());
      else */
      {
        NODE_TYPE nt = node_type(pv,new_node);
        if (nt == NODE_TYPE::ELEMENT && nt == node_type(pv,old_node)) {
          mutator_ctx _(old_node.ptr_of<element>(), pv);
          update_children(new_node, old_node);
          morph_node(new_node, old_node, is_root);
        }
        else {
          mutator_ctx _(old_node->parent, pv);
          morph_node(new_node, old_node);
        }
        return old_node;
      }
    }

    void morph_node(pvnode& new_node, hnode old_node, bool is_root = false) {

      if (node_type(pv,old_node) == NODE_TYPE::ELEMENT)
      {
        update_element_atts_states(pv, old_node.ptr_of<element>(), new_node, is_root);
      }
      else { // text or comment
        ustring_chars ntext = node_text(pv,new_node);
        ustring_chars otext = node_text(pv,old_node);
        if (ntext != otext && opts->change_text(old_node.ptr_of<text>(), ntext))
        {
          old_node.ptr_of<text>()->set_text(ntext, pv);
        }
      }
    }

    /*node *get_child(node* el, uint index) {
      if (index < 0 || index >= node_size(el)) return nullptr;
      return el->nodes[index];
    };*/

    // Update the children of elements

    inline bool safe2morph(context& pv, pvnode new_child, hnode old_child) {
      if (!node_key(pv,new_child).is_undefined()) return false;
      if (!node_key(pv,old_child).is_undefined()) return false;
      auto fn = node_factory(pv,new_child);
      auto fo = node_factory(pv,old_child);
      return fn && (fn == fo);
    }
    

    void update_children(pvnode& new_node, hnode old_node)
    {
#ifdef DEBUG
      //if (old_node.ptr_of<element>()->is_id_test())
      //  old_node = old_node;
#endif

      new_node = node_for_children_update(pv,new_node);
      old_node = node_for_children_update(pv,old_node);

      if (!new_node || !old_node)
        return;

      hnode old_child, morphed, old_match;
      pvnode new_child, new_match;

      // the offset is only ever increased, and used for [i - offset] in the loop
      int offset = 0;

      for (uint i = 0; ; ++i) {

        old_child = node_child(pv, old_node, i);
        new_child = node_child_ex(pv, new_node, i - offset, old_node);

        // both nodes are empty, do nothing
        if (!old_child && !new_child)
          break;

        //if(new_child && old_child) BAD, too early!
        //  new_child = node_element_expand(pv, new_child, old_child, old_node);

        if (!new_child) { // there is no new child, remove old
          if (opts->remove_node(old_child)) {
            old_child->remove(true,pv.pview());
            --i;
          }
        }
        else if (!old_child) { // there is no old child, add new
          uint index = node_size(pv,old_node);
          hnode hn;
          if (node_new_to_old_ex(pv, new_node, i - offset, old_node, hn)) {
            --i;
            continue;
          }
          else if (hn && opts->insert_node(old_node, index, hn)) {
            node_remove_child(pv,new_node, i - offset);
            old_node.ptr_of<element>()->insert(index,hn,pv);
            ++offset;
          } 
          //else assert(false);
        }
        else if (same(pv,new_child, old_child)) { // both nodes are the same, morph
          if (node_type(pv,new_child) == NODE_TYPE::ELEMENT) {
            morphed = walk(new_child, old_child);
            if (morphed != old_child) {
              if (opts->replace_node(old_child, morphed))
                old_node.ptr_of<element>()->replace_child(old_child, morphed,pv);
              //WRONG:++offset;
            }
          }
        } else { // both nodes do not share an ID or a placeholder, try reorder

          old_match = nullptr;

          // try and find a similar node further in the tree
          for (uint j = i + 1; j < node_size(pv,old_node); j++)
          {
            auto tn = node_child(pv,old_node, j);
            if (same(pv,new_child, tn)) {
              old_match = tn;
              break;
            }
          }

          if (!old_match) {
            new_match = 0;
            for (uint j = i - offset + 1; j < node_size(pv,new_node); ++j)
            {
              if (same(pv,node_child(pv,new_node,j), old_child)) {
                new_match = node_child(pv, new_node, j);
                break;
              }
            }
            if (new_match) { // we've found match in new_node for our old_child so just insert that new one here
              hnode nnchild = node_new_to_old(pv, new_child, old_node);
              if (opts->insert_node(old_node, old_child->node_index, nnchild)) {
                node_remove_child(pv, new_node, i - offset);
                if (nnchild) {
                  old_node.ptr_of<element>()->insert(old_child->node_index, nnchild, pv);
                  ++offset;
                }
                continue;
              }
            }
          }

          if (old_match) { // If there was a node with the same ID or placeholder in the old list
            morphed = walk(new_child, old_match);
            //if (morphed != old_match) ??????
            //  ++offset;
            if (opts->reposition_node(old_node, old_child->node_index, morphed)) {
              old_node.ptr_of<element>()->insert(old_child->node_index, morphed, pv);
            }
          }
          else if (safe2morph(pv,new_child, old_child)) { // it's safe to morph two nodes in-place if neither of them have an ID or factory
            morphed = walk(new_child, old_child);
            if (morphed != old_child) {
              if (opts->replace_node(old_child, morphed))
                old_node.ptr_of<element>()->replace_child(old_child, morphed,pv);
              //WRONG: ++offset;
            }
          }
          else { 
            hnode nnchild;
            if (node_new_to_old_ex(pv, new_node, i - offset, old_node, nnchild)) 
            {
              --i; // to stay on the same index
              continue;
            }
            else if(nnchild)
            {
              // no new match and no old match - replace node at the index
              if (opts->replace_node(old_child, morphed))
                old_node.ptr_of<element>()->replace_child(old_child, nnchild, pv);
            }
            else {
              if (opts->remove_node(old_child)) {
                old_child->remove(true, pv.pview());
                --i;
              }
            }

          }
        }
      }
    }

    bool same(context& pv, pvnode a, node* b)
    {
      return node_equal(pv, a, b);
    }

  };

  typedef morph_t<html::hnode,node_protector> morph;


} // namespace html

#endif
