﻿#include "html.h"

#ifdef SCITERJS 
  #include "xdomjs/xcontext.h"
#endif

namespace html {

  data_loader *application::primordial_loader = 0;

  document_context::document_context(const element* pe) { _pdoc = pe->doc(); _pview = _pdoc->pview(); }

  selector_context::selector_context(element *root_el, wchars selector,
                                     bool do_visible_only, bool do_deep)
      : root(root_el), deep(do_deep), only_visible(do_visible_only) {
    while (root_el && selector[0] == '<') {
      ++selector;
      root = root->parent;
    }

    css_istream s(selector, string());
    html::style_def::parse_list(nullptr, s, sds);
    if (sds.size() == 0) {
      view::debug_printf(OT_DOM, OS_ERROR, "bad selector %S\n", selector.start);
      assert(0);
    }
  }

  document::~document() {
    _images.clear();
  }

  document::document(const document *src)
      : super(src->tag)
      , _view(src->_view)
      , _media_vars_provider(src->_media_vars_provider), _isolated(true)
      , style_serial_no(src->style_serial_no)
#ifdef SCITER
      , ns(0) // namespace of loaded document
#endif
      , _uri(src->_uri)
      , _images(src->_images), _image_maps(src->_image_maps)
      , _entities(src->_entities), app(src->app)

  {
    atts   = src->atts;
    int nn = src->nodes.size();
    nodes.size(nn);
    for (int n = 0; n < nn; ++n) {
      node *tn = src->nodes[n]->clone();
      tn->got_parent(this, n);
      nodes[n] = tn;
    }
    flags.indexes_are_valid = 0;
    state                   = state.clone();
    drop_layout();
  }

  void document::pview(view *pv) {
    assert(parent == 0);
    _view = pv;
    if (_view) app = _view->app;
#if defined(SCITERJS)
    else 
      ns = nullptr;
#endif
  }

  element* document::get_body()  {
    if (!_body)
      this->each_child([this](element* pel) -> bool {
        if (pel->tag == tag::T_BODY) { _body = pel; return true; }
        return false;
      });
    return _body;
  }

  element* document::get_head() {
    if (!_head)
      this->each_child([this](element* pel) -> bool {
      if (pel->tag == tag::T_HEAD) { _head = pel; return true; }
      return false;
    });
    return _head;
  }


  void document::setup_layout(view &v) {

    auto_state<handle<document>> _(v.current_doc, this);

    block_vertical::setup_on(v, this);

    slice<hnode> nodes   = this->nodes();
    slice<hnode> inlines = scan_inlines(v, nodes);
    if (inlines.length && is_space_seq(v, inlines)) inlines.length = 0;

    handle<layout_data> ld = ldata.ptr_of<layout_data>();
    ld->drop();

    if (nodes.length == 0) // all nodes are inlines - text container
    {
      helement anonymous_text_block            = new element(tag::T_TEXT);
      anonymous_text_block->flags.is_synthetic = 1;
      anonymous_text_block->owner              = this;
      anonymous_text_block->nodes              = inlines;
      text_block::setup_on(v, anonymous_text_block, inlines);
      ld->push(v, this, anonymous_text_block);
      ld->is_initialized = true;

      return;
    }

    // otherwise it has blocks

    while (nodes.length || inlines.length) {
      if (inlines.length == 0) // bumped into block
      {
        ld->push(v, this, nodes[0].ptr_of<element>());
        nodes.prune(1);
      } else if (!is_space_seq(v, inlines)) {
        helement anonymous_text_block = new element(tag::T_TEXT);
        anonymous_text_block->owner   = this;
        anonymous_text_block->nodes   = inlines;
        ld->push(v, this, anonymous_text_block);
      }
      inlines = scan_inlines(v, nodes);
    }
    ld->is_initialized = true;
  }

  element *document::drop_layout(view *pv) {
    if (pv && is_layout_valid()) pv->refresh(this);
    if (ldata) ldata->drop();
    flags.layout_ctl_valid = 0;
    drop_minmax_dim();
    clear_style();
    return this;
  }

  style *document::get_base_style(view &v) {
    //return v.get_default_style();
    return super::get_base_style(v);
  }

  void document::stray(view &v) {
    operational = false;

    FOREACH(i, _synthetic_elements)
      _synthetic_elements[i]->stray(v);
    _synthetic_elements.clear();

    _styles.clear();
    _actions.clear();
    _running_actions.clear();
    _images.clear();
    _image_maps.clear(); // @image-map things
    _cursors.clear();

    super::stray(v);

#if defined(SCITERJS)
    ns = nullptr;
#endif

  }

  style_bag *document::get_named_style_set(const string &set_name) {
    if (set_name == CHARS("none")) return 0;
    style_bag *sb = styles().get_named_set(set_name);
    if (!sb) sb = application::stock_styles().get_named_set(set_name);
    return sb;
  }

  /*bool document::drawing_context(view* &pv, graphics* &pgfx)
  {
    if( parent ) return super::drawing_context(pv,pgfx);
    if( !_view ) return false;
    pv = _view;
    pgfx = _view->surface();
    return true;
  }*/

  bool document::a11y_get_name(view &v, ustring &name) {
    element *pt = find_first(v, this, WCHARS("head>title"));
    if (pt) {
      name = pt->get_text(v);
      if (name.length()) return true;
    }
    return super::a11y_get_name(v, name);
  }


  void document::measure(view &v, size sz) {
    // if( !lctl )
    //  setup_layout(v,this);
    get_style(v);
    check_layout(v);
    if (!sz.empty()) {
      auto_state<helement> BFC(v.bfc, this);
      auto_state<handle<document>> CD(v.current_doc, this);
      if (parent) {
        replace_h(v, this, sz.x, false);
        replace_v(v, this, sz.y, false);
      } else {
        //dbg_printf("sz.y %d, client_dim.h %d\n", sz.y, v.client_dim().y);
        measure_borders_x(v, sz);
        measure_borders_y(v, sz);
        set_border_pos(point(0, 0));
        set_border_width(v, sz.x);
        set_border_height(v, sz.y);
      }
    }
  }

  iwindow *document::window(view &v) {
    if (!parent && _view) return _view;
    return super::window(v);
  }

  ustring document::get_lang() const {
    ustring l = element::get_lang();
    if (l.length()) return l;
    if (_view) return _view->get_lang();
    return ustring();
  }

  ustring document::get_theme() const {
    ustring l = element::get_theme();
    if (l.length()) return l;
    if (_view) return _view->get_theme();
    return ustring();
  }


  void document::view_mtx(view &v, affine_mtx_f &vmx) {
    element *owner = get_owner();
    if (!_isolated && !owner)
      vmx.translate(point(ldata->borpad_left(), ldata->borpad_top()));
    else
      super::view_mtx(v, vmx);
  }

  point document::view_pos(view &v) {
    if (_isolated) return point(0, 0);
    // if(owner) return owner->view_pos(v) + super::doc_pos(v);
    element *owner = get_owner();
    if (owner) {
      document *pd = owner->doc();
      if (pd) return pd->view_pos(v) + super::doc_pos(v);
    } else
      return point(ldata->borpad_left(), ldata->borpad_top());

    return doc_pos(v);
  }

  image_ref document::register_image(string &image_url) {
    if (image_url.length() == 0) return image_ref::none();

#ifdef _DEBUG
      /*{ // flat url as a base
        string base("schema:domain/filename.ext");
        string rel1("test.gif");
        string rel2("folder/test.gif");
        string abs1 = combine_url(base,rel1);
        string abs2 = combine_url(base,rel2);
        assert(abs1 == "schema:test.gif");
        assert(abs2 == "schema:folder/test.gif");
      }

      { // hierachical url as a base
        string base("schema://domain/filename.ext");
        string rel1("test.gif");
        string rel2("folder/test.gif");
        string rel3("/folder/test.gif");
        string abs1 = combine_url(base,rel1);
        string abs2 = combine_url(base,rel2);
        string abs3 = combine_url(base,rel3);
        assert(abs1 == "schema://domain/test.gif");
        assert(abs2 == "schema://domain/folder/test.gif");
        assert(abs3 == "schema://domain/folder/test.gif");
      }*/
#endif

    image_url = combine_url(_uri.src, image_url);
    // image_url.to_lower();
    image_url.replace_all('\\', '/');

    tool::url u;
    u.parse(image_url);

    view *pv = pview();

    if (parent && pv) // this is doc inside <frame>
    {
      // try to get image from the root document cache first
      handle<image_rec> im_rec;
      if (pv->doc()->_images.find(image_url, im_rec)) return image_ref(im_rec);
    }

    handle<image_rec> im_rec;
    if (_images.find(image_url, im_rec)) return image_ref(im_rec);
    return image_ref::create(_images, image_url);
  }

  void document::release_image(image_ref &imr) {
    if (!imr.rec) return;
    if (imr.rec->get_ref_count() <= 2) // it is held by _images and the imr
      _images.remove(imr.rec->url);
    imr.clear();
  }

  bool document::register_image_map(ustring                  map_name,
                                    slice<pair<string, int>> urls_and_DPIs,
                                    size cells, slice<value> items) {
    // if( cells.empty() )
    //  return false;
    cells.x = max(1, cells.x);
    cells.y = max(1, cells.y);

    handle<image_map> im = new image_map();
    if (!im) return false;
    im->cols = max(1, cells.x);
    im->rows = max(1, cells.y);
    uint sn  = 0;
    for (uint n = 0; n < items.length; ++n) {
      value pv = items[n];
      if (!pv.is_array()) return false;
      slice<value> vals = pv.values();
      ustring      name;
      rect         ar;
      if (vals.length == 1 && vals[0].is_string()) {
        name        = vals[0].get(W(""));
        ar.s.x = sn % im->cols;
        ar.s.y = sn / im->cols;
        // assert(ar.s.y > im->rows);
        // if( ar.s.y > im->rows )
        //  return false;
        ar.e = ar.s + size(1);
        ++sn;
      } else if (vals.length == 2 && vals[0].is_string() && vals[1].is_int()) {
        name        = vals[0].get(W(""));
        sn          = vals[1].to_int() - 1;
        ar.s.x = sn % im->cols;
        ar.s.y = sn / im->cols;
        ar.e   = ar.s + size(1);
        // reset sn to next image
        ++sn;
      } else if (vals.length == 3 && vals[0].is_string() && vals[1].is_int() &&
                 vals[2].is_int()) {
        name        = vals[0].get(W(""));
        ar.s.x = vals[1].to_int() - 1;
        ar.s.y = vals[2].to_int() - 1;
        ar.e   = ar.s + size(1);
        // reset sn to next image
        sn = ar.s.y * cells.x + ar.s.x + 1;
      } else if (vals.length == 5 && vals[0].is_string() && vals[1].is_int() &&
                 vals[2].is_int() && vals[3].is_int() && vals[4].is_int()) {
        name        = vals[0].get(W(""));
        ar.s.x = vals[1].to_int() - 1;
        ar.s.y = vals[2].to_int() - 1;
        size sz;
        sz.x      = vals[3].to_int();
        sz.y      = vals[4].to_int();
        ar.e = ar.s + sz;
      } else
        return false;

      assert(!ar.empty());
      if (ar.empty()) return false;
      assert(ar.s.x >= 0 && ar.s.x < im->cols);
      assert(ar.s.y >= 0 && ar.s.y < im->rows);
      if (ar.s.x < 0 || ar.s.y < 0 || ar.s.x > im->cols || ar.s.y > im->rows)
        return false;
      assert(ar.e.x >= 0 && ar.e.x <= im->cols);
      assert(ar.e.y >= 0 && ar.e.y <= im->rows);
      if (ar.e.x < 0 || ar.e.y < 0 || ar.e.x > im->cols || ar.e.y > im->rows)
        return false;

      im->parts[name] = ar;
    }

    im->parts[""] = rect(cells); // part with empty name represents the whole image

    im->irefs.length(urls_and_DPIs.length);
    for (int n = 0; n < urls_and_DPIs.size(); ++n) {
      string url        = urls_and_DPIs[n].first;
      int    dpi        = urls_and_DPIs[n].second;
      im->irefs[n].iref = register_image(url);
      im->irefs[n].dpi  = dpi ? dpi : limits<int>::max_value();
    }
    // sort irefs in dpi ascending order
    sort(im->irefs.head(), im->irefs.size(),
         [](const image_map::image_ref_dpi &v1,
            const image_map::image_ref_dpi &v2) { return v1.dpi < v2.dpi; });

    _image_maps[map_name] = im;
    // im->iref.fetch(v,this);
    return true;
  }
  image_ref document::register_image_fragment(ustring map_name,
                                              ustring part_name) {
    handle<image_map> im;
    string            url = string::format("%S(%S)", map_name.c_str(), part_name.c_str());

    handle<image_rec> im_rec;

    if (_images.find(url, im_rec)) return image_ref(im_rec);

    if (_image_maps.find(map_name, im)) {
      handle<image_map_fragment> imf = new image_map_fragment(im, part_name);
      image_ref                  ir  = image_ref::create(_images, url);
      ir.img(imf);
      return ir;
    }
    else if (this != application::stock_styles_doc) {
      if (application::stock_styles_doc->_image_maps.find(map_name, im)) {
        handle<image_map_fragment> imf = new image_map_fragment(im, part_name);
        image_ref                  ir = image_ref::create(_images, url);
        ir.img(imf);
        return ir;
      }
    }
    return image_ref();
  }

  image *document::get_image(const string &url) {
    handle<image_rec> im_rec;
    if (_images.find(url, im_rec)) return im_rec->img;
    return nullptr;
  }

  bool document::drop_image(const string &url) {
    handle<image_rec> im_rec;
    return _images.remove(url);
  }
  
  cursor *document::load_cursor(const string &cursor_url, size *hotspot) {
    view *pv = pview();

    if (pv == 0 || cursor_url.length() == 0) return 0;

    string curl = combine_url(_uri.src, cursor_url);
    curl.replace_all('\\', '/');

    handle<cursor> cur;
    if (_cursors.find(curl, cur)) return cur;

    handle<pump::request> rq =
        new pump::request(curl, hotspot ? DATA_IMAGE : DATA_CURSOR);
    if (load_data(rq, pv, true)) {
      if (hotspot) {
        handle<image> pimg = image::create(rq->data, curl, this);
        if (pimg && pimg->is_bitmap())
          cur = cursor::from_bitmap(pimg.ptr_of<bitmap>(), curl, *hotspot);
      } else
        cur = cursor::from_data(rq->data(), rq->url);
      _cursors[curl] = cur;
      return cur;
    }
    return 0;
  }

  image_ref &image_map::iref() const {
    assert(irefs.size());
    view *pv  = view::get_current();
    int   dpi = pv ? pv->pixels_per_inch().x
                   : resolution_provider::desktop().pixels_per_inch().x;
    for (int n = 0; n < irefs.last_index(); ++n) {
      if (dpi <= irefs[n].dpi) {
        return const_cast<image_map *>(this)->irefs[n].iref;
      }
    }
    return const_cast<image_map *>(this)->irefs.last().iref;
  }

  bool image_map::fetch(view &v, document *pd) { return iref().fetch(v, pd); }

  void image_map::fragment_draw(const image_map_fragment *fr, gool::graphics *gfx,
    rect dst, rect src, byte opacity) {
    rect cr = part_area(fr->name, fr->mapping);
    if (cr.empty()) return;
    base(fr->filters, fr->mapping)->draw(gfx, dst, src + cr.s, opacity);
  }
  void image_map::fragment_draw(const image_map_fragment *fr, gool::graphics *gfx,
    rectf dst, rect src, byte opacity) {
    rect cr = part_area(fr->name, fr->mapping);
    if (cr.empty()) return;
    base(fr->filters, fr->mapping)->draw(gfx, dst, src + cr.s, opacity);
  }


  bool image_map_fragment::fetch(view &v, document *pd) {
    if (map) return map->fetch(v, pd);
    return false;
  }

  image_ref image_ref::create(image_bag &ib, const tool::string &url,
                              gool::image *pimg) {
    int               hash_pos = url().index_of('#');
    handle<image_rec> im_rec_base;
    if (hash_pos >= 0) {
      string url_base = url().sub(0, hash_pos);
      if (!ib.find(url_base, im_rec_base)) {
        im_rec_base      = new image_rec;
        im_rec_base->url = url_base;
        ib[url_base]     = im_rec_base;
      }
      assert(pimg == nullptr);
    }

    handle<image_rec> im_rec = new image_rec;
    im_rec->img              = pimg;
    im_rec->url              = url;

    ib[url]     = im_rec;
    im_rec->img = pimg;
    return image_ref(im_rec, im_rec_base);
  }

  bool image_ref::fetch(view &v, document *pd) {
    if (is_empty()) return false;

    // handle<image_rec> irec = rec;
    // handle<image_rec> irec_base = nullptr;

    if (rec->img) {
      if (rec->img->is_of_type<image_map_fragment>())
        return rec->img.ptr_of<image_map_fragment>()->fetch(v, pd);
      else if (rec->img->is_of_type<image_map>())
        return rec->img.ptr_of<image_map>()->fetch(v, pd);
    }

    if (rec->requested ||
        rec->img) // already requested, don't need to do anything.
      return rec->img.is_undefined();

    rec->requested = true;

    if (base_rec.is_defined()) base_rec->requested = true;

    /*if( rec->img.ptr() )
    {
      assert(false);
      return false;
    }*/

    string target_url = rec->url;

    if (target_url.is_empty()) {
      rec->requested = true;
      return false;
    }

    tool::url u;
    u.parse(target_url);

    if (u.anchor.length()) target_url = u.compose_without_anchor();

#ifdef THEMES_SUPPORT
    if (u.protocol == CHARS("theme")) {
      rec->img = gool::theme::current()->get_image(u.filename);
      return false;
    } else
#endif
   if (u.protocol == CHARS("stock")) {
      rec->img = stock_image::get(u.filename);
      return false;
    }

    if (u.protocol == CHARS("path")) {
      rec->img = new path_image(url::unescape(u.filename));
      return false;
    }

    if (base_rec && base_rec->img) {
      rec->img = base_rec->img->get_fragment(u.anchor);
      return true;
    }

    handle<pump::request> prq = new pump::request(rec->url, DATA_IMAGE);
    prq->dst                  = pd;

    if (load_data(prq, &v)) {
      if (prq->data.size()) {
        if (base_rec) {
          base_rec->img = image::create(prq->data, rec->url, pd);
          if (base_rec->img) rec->img = base_rec->img->get_fragment(u.anchor);
        } else
          rec->img = image::create(prq->data, rec->url, pd);
        prq->consumed = true;
        pd->on_data_arrived(v, prq);
      }
    }
    return !prq->ready_flag;
  }

  void document::drop_cache() {
    int nm = _images.size();
    for (int n = 0; n < nm; ++n) {
      image *pimg = _images.value(n)->img;
      if (pimg) pimg->drop_cache();
    }
    _styles.drop_caches();
  }

  bool document::load_style(element *style_or_link, wchars text) {
    if (!pview()) return false;
    if (!allow_own_styles) return false;

    string id      = style_or_link->sequential_id();
    if (!id) {
      id = string(char(++style_serial_no), 1);
      style_or_link->sequential_id(id);
    }

    int    line_no = style_or_link->line_no().val(1);

    string href = style_or_link->atts.get_url(
        uri().src(),
        style_or_link->tag == tag::T_LINK ? attr::a_href : attr::a_src);

    ustring lang    = style_or_link->get_lang();
    ustring forlang = style_or_link->atts(attr::a_forlang);
    if (forlang.is_defined() && lang.is_defined()) {
      if (!match_list_ci(lang(), forlang(), WCHARS(";,"))) {
        view::debug_printf(
            OT_CSS, OS_WARNING,
            "style declaration skipped, language %S mismatch %S at (%s(%d))\n",
            lang.c_str(), forlang.c_str(), uri().src.c_str(), line_no);
        return true;
      }
    }

    ustring mtype = style_or_link->atts(attr::a_media);

    ustring wdata;

    if (href.is_empty()) {
      href = uri().src();
    } else {
      handle<pump::request> hrq = new pump::request(href, DATA_STYLE);
      hrq->rq_id                = id;
      hrq->dst                  = this;
      if (!load_data(hrq, pview())) return false;
      if (!hrq->data.size()) return false;
      wdata = u8::cvt(hrq->data());
      text  = wdata;
      line_no = 1;
    }

    style_parser stp(id, text, this, href, line_no);
    stp.parse(mtype);

    pview()->add_to_update(this, true);
    return true;
  }

  bool document::load_style(const string& url)
  {
    if (!pview()) return false;
    if (!allow_own_styles) return false;

    ustring wdata;
    string id = "z";

    handle<pump::request> hrq = new pump::request(url, DATA_STYLE);
    hrq->rq_id = id;
    hrq->dst = this;
    if (!load_data(hrq, pview())) return false;
    if (!hrq->data.size()) return false;
    wdata = u8::cvt(hrq->data());

    style_parser stp(id, wdata(), this, url, 1);
    stp.parse(wchars());

    return true;
  }

  void document::reset_styles(view &v) { super::reset_styles(v); }

  void document::handle_style(element *style_el) {
    if (is_fragment_document()) return;
    if (!allow_own_styles) return;
    array<wchar> style_text;
    if (style_el->is_empty()) {
      string src;
      if (style_el->tag == tag::T_STYLE)
        src = string(style_el->atts(attr::a_src));
      else if (style_el->tag == tag::T_LINK)
        src = string(style_el->atts(attr::a_href));
      if (src.length() == 0) return;

      if (style_el->atts.exist(attr::a_disabled)) return;

      load_style(style_el, wchars());

      return;
    }

    node *tn = style_el->first_node();
    if (tn && tn->is_text()) {
      load_style(style_el, static_cast<text *>(tn)->chars());
    }
  }

#if 0
  bool document::has_pseudo_classes_for(element *el, ui_state st, bool deep) {
    const string & set_name = el->get_style()->style_set;
    const element *r = 0;
    if (set_name.length()) {
      r = el;
      const element *t = el->parent;
      for (; t && t->get_style()->style_set == set_name; t = t->parent) r = t;
      if (r == el) st.root(true);
    }
    if (styles().has_pseudo_classes_for(el, st, deep, r) ||
      application::stock_styles().has_pseudo_classes_for(el, st, deep, r))
      return true;

    const style_set_holder* pssh = el->forced_style_set();
    if (pssh && pssh->style_set) {
      st.root(false);
      const element *r = nullptr;
      r = el;
      const element *t = el->parent;
      for (; t && t->forced_style_set() == pssh; t = t->parent) r = t;
      if (r == el) st.root(true);
      if (pssh->style_set->has_pseudo_classes_for(el, st, deep, r))
        return true;
    }
    return false;
  }
#endif

  void document::cursor_data_arrived(view &v, handle<pump::request> rq) {
    _cursors[rq->url] = cursor::from_data(rq->data(), rq->url);
  }

  bool
  document::on_data_arrived(view &         v,
                            pump::request *prq) // requested data just arrived
  {
    if (!prq->consumed) {
      switch (prq->data_type) {
      case DATA_IMAGE: image_data_arrived(v, prq); goto CHECK_COMPLETE;
      case DATA_STYLE: style_data_arrived(v, prq); goto CHECK_COMPLETE;
      case DATA_CURSOR:
        cursor_data_arrived(v, prq);
        goto CHECK_COMPLETE;
        // case DATA_SCRIPT_DATA:
      case DATA_RAW_DATA:
        break; // return false; // sync request
      default:
        // assert(false);
        break;
      }
      return super::on_data_arrived(v, prq);
    CHECK_COMPLETE:
      prq->consumed           = true;
      num_resources_requested = num_resources_requested - 1;
      if (num_resources_requested == 0) v.on_document_complete(this);
    }
    return super::on_data_arrived(v, prq);
  }

  void document::image_data_arrived(view &v, handle<pump::request> hrq) {
    if (parent && !_images.exists(hrq->url) && v.doc() != this) {
      v.doc()->image_data_arrived(v, hrq);
      return;
    }

    handle<image> img = image::create(hrq->data, hrq->url, this);

    handle<image_rec> irec;

    if (!_images.find(hrq->url, irec)) {
      irec              = new image_rec;
      irec->url         = hrq->url;
      _images[hrq->url] = irec;
    }

    irec->img       = img;
    irec->requested = true;

    if (v.is_measuring) return;

    v.refresh(this);

    image_ref        ir(irec);
    element_iterator it(v, this);
    for (element *b; it(b);) {
      b->accept_image(v, ir);
    }
  }

  void document::image_arrived(view &v, image *pimg) {
    handle<image_rec> irec;

    if (!_images.find(pimg->get_url(), irec)) {
      irec                     = new image_rec;
      irec->url                = pimg->get_url();
      _images[pimg->get_url()] = irec;
    }
    irec->img = pimg;

    if (v.is_measuring) return;

    v.refresh(this);

    image_ref        ir(irec);
    element_iterator it(v, this);
    for (element *b; it(b);)
      b->accept_image(v, ir);
  }

  void document::style_data_arrived(view &v, handle<pump::request> hrq) {
    if (hrq->data.size()) {
      string      url = hrq->url;
      array<byte> data;
      data.swap(hrq->data);
      parse_style_sheet(hrq->rq_id, u8::cvt(data()), this, url,
                        hrq->mediaq);
      // if(!v.is_measuring)
      //  v.add_to_update(this,CHANGES_MODEL);
      drop_layout_tree(&v);
      resolve_styles(v);
      commit_measure(v);
    }
    // dprintf("style data arrived:%s\n", (const char *)uri);
  }

  bool document::is_large() const {
    helement body;
    const_cast<document *>(this)->each_child([&body](element *el) -> bool {
      if (el->tag == tag::T_BODY) {
        body = el;
        return true;
      }
      return false;
    });
    return body && body->nodes.size() > 64;
  }

  bool document::get_scroll_data(view &v, scroll_data &sd) {
#if defined(NATIVE_ROOT_SCROLLBARS)
    if (!is_root_document() || !window(v)) return super::get_scroll_data(v, sd);
    sd.content_outline = rect(size(ldata->dim_min.x, ldata->dim_min.y));
    sd.pos             = scroll_pos();
    sd.dim_view        = v.dimension(); // ldata->dim;
    return true;
#else
    return super::get_scroll_data(v, sd);
#endif
  }

  void document::update_scrollbars(view &v) {
#if defined(NATIVE_ROOT_SCROLLBARS)
    if (!is_root_document() || !window(v) || v.is_layered()) {
      super::update_scrollbars(v);
      return;
    }

    int overflow_x = c_style->overflow_x;
    int overflow_y = c_style->overflow_y;

    bool hsb_needed =
        (overflow_x == overflow_auto || overflow_x == overflow_scroll);
    bool vsb_needed =
        (overflow_y == overflow_auto || overflow_y == overflow_scroll);

    if (!hsb_needed && !vsb_needed) return;

    scroll_data scd;
    get_scroll_data(v, scd);
    point saved_scroll_pos = scd.pos;

    for (int i = 0; i < 2; ++i) {
      size dim = v.client_dim();

      range cx = scd.content_outline.x(); // scrollable_content_width(v);
      // bool relayout_y  =  !hsb_needed?
      //                    sb.remove_h():
      //                    sb.set_h(/*saved_scroll_pos.x,*/ cx, scd.dim_view.x,
      //                    overflow_x == overflow_scroll,is_rtl);
      bool relayout_y =
          v.set_h_scrollbar(cx, overflow_x == overflow_scroll, dim);
      if (relayout_y) {
        layout_height(v, dim.y);
        get_scroll_data(v, scd);
      }
      range cy = scd.content_outline.y(); // scrollable_content_height(v);
      bool  relayout_x =
          v.set_v_scrollbar(cy, overflow_y == overflow_scroll, dim);
      // bool relayout_x  = !vsb_needed?
      //                    sb.remove_v():
      //                    sb.set_v(/*saved_scroll_pos.y,*/ cy, scd.dim_view.y,
      //                    overflow_y == overflow_scroll);
      if (relayout_x) {
        layout_width(v, dim.x);
        layout_height(v, dim.y);
        get_scroll_data(v, scd);
      } else if (cx == scd.content_outline.x())
        break;
    }
    ldata->inner_dim = client_rect(v).size();
    // sb.refresh(v, this);
#else
    super::update_scrollbars(v);
#endif
  }

  void document::scroll_pos(view &v, point off, bool allow_out_of_bounds) {
#if defined(NATIVE_ROOT_SCROLLBARS)
    if (!is_root_document() || !window(v) || v.is_layered()) {
      super::scroll_pos(v, off, allow_out_of_bounds);
      return;
    }
    v.set_scroll_pos(off);
#else
    super::scroll_pos(v, off, allow_out_of_bounds);
#endif
  }
  point document::scroll_pos() const {
#if defined(NATIVE_ROOT_SCROLLBARS)
    if (!is_root_document() /* || !const_cast<document*>(this)->window(v)*/ ||
        _view->is_layered())
      return super::scroll_pos();
    return _view->get_scroll_pos();
#else
    return super::scroll_pos();
#endif
  }

  bool document::can_scroll_h(view &v) const {
#if defined(NATIVE_ROOT_SCROLLBARS)
    bool yes = false;
    if (is_root_document() && v.has_h_scrollbar(yes)) return yes;
    return super::can_scroll_h(v);
#else
    return super::can_scroll_h(v);
#endif
  }
  bool document::can_scroll_v(view &v) const {
#if defined(NATIVE_ROOT_SCROLLBARS)
    bool yes = false;
    if (is_root_document() && v.has_v_scrollbar(yes)) return yes;
    return super::can_scroll_v(v);
#else
    return super::can_scroll_v(v);
#endif
  }

  element *document::get_element_by_id(const ustring &id) {
    weak_handle<element> el;
    if (id2element.find(id, el) && el)
      return el.ptr();
    else {
      each_element it(this);
      for (element* pel; it(pel);) {
        ustring eid = pel->attr_id();
        if (eid.is_defined()) {
          weak_handle<element> wel = pel;
          id2element[eid] = wel;
          if (eid == id)
            return pel;
        }
        if (pel->is_document())
          it.skip();
      }
    }
    return nullptr;
  }
  void document::add_element_id(element* pel, const ustring &id) {
    element *other = get_element_by_id(id);
    if (other != this) {
      weak_handle<element> el = pel;
      id2element[id] = pel;
    }
  }
  void document::remove_element_id(element* pel, const ustring &id) {
    element *other = get_element_by_id(id);
    if (other == this)
      id2element.remove(id);
  }


  bool element::is_sensitive_attr(const name_or_symbol &ns, bool &rem) {
    bool r1 = false, r2 = false;

    rem = false;

    bool is_known = false;

    document* pd = doc();

    if ((pd && pd->styles().is_sensitive_attr(ns, r1)) || application::stock_styles().is_sensitive_attr(ns, r2)) {
      rem      = r1 || r2;
      is_known = true;
      // return true;
    }

    switch (ns.att_sym) {
    case attr::a_class:
    case attr::a_src:
    case attr::a_style: rem = true; return true;
    case attr::a_width:
    case attr::a_height: rem = true; return true;

    case attr::a_id: return true;

    case attr::a_placeholder:
    case attr::a_novalue:
      rem = true; return true;
    case attr::a_bordercolor:
    case attr::a_background:
    case attr::a_color:
    case attr::a_bgcolor: return true;

    case attr::a_vspace:
    case attr::a_hspace:
    case attr::a_border:

    case attr::a_cellpadding:
    case attr::a_cellspacing:

    case attr::a_fixedrows:
    case attr::a_fixedcols:

    case attr::a_fixedlayout:

    case attr::a_align:
    case attr::a_valign: rem = true; return true;

    case attr::a_nowrap: rem = true; return true;

    case attr::a_expanded:
    case attr::a_collapsed:
    case attr::a_visible:
    case attr::a_hidden: rem = true; return true;

    case attr::a_dir: rem = true; return true;
    case attr::a_lang: rem = true; return true;
    case attr::a_theme: rem = true; return true;

    case attr::a_fill:
    case attr::a_fill_opacity:
    case attr::a_fill_rule:
    case attr::a_stroke:
    case attr::a_stroke_width:
    case attr::a_stroke_linecap:
    case attr::a_stroke_linejoin:
    case attr::a_stroke_miterlimit:
    case attr::a_stroke_dasharray:
    case attr::a_stroke_dashoffset:
    case attr::a_stroke_opacity:

    case attr::a_stop_color:
    case attr::a_stop_opacity:

    case attr::a_gradienttransform:
    case attr::a_transform:

      return true;

      /*case attr::a_step:
      case attr::a_min:
      case attr::a_max:
      case attr::a_minvalue:
      case attr::a_maxvalue:
        rem = true; return true;*/
    }

    return is_known;
  }

  bool document::find_media_var(wchars name, value &v) {
    if (_media_vars_provider &&
        _media_vars_provider->media_vars().find(name, v))
      return true;
    if (_isolated) return false;
    view *pv = pview();
    if (!pv) return false;
#ifdef _DEBUG
      /*if(!pv->media_vars().find(name,v))
      {
        for( int i = 0; i < pv->media_vars().size(); ++i ) {
          ustring us = pv->media_vars().key(i);
          dbg_printf("media_var:%S\n",us.c_str());
        }
        return false;
      }*/
#endif

    return pv->media_vars().find(name, v);
  }

  bool document::start_eval(element *b, eval::conduit *prg) {
    running_action ra;
    ra.el  = b;
    ra.prg = prg;
    for (int n = _running_actions.last_index(); n >= 0; --n) {
      if (_running_actions[n] == ra) return false;
    }
    _running_actions.push(ra);
    return true;
  }
  void document::stop_eval(element *b, eval::conduit *prg) {
    running_action ra;
    ra.el  = b;
    ra.prg = prg;
    for (int n = _running_actions.last_index(); n >= 0; --n) {
      if (_running_actions[n] == ra) {
        _running_actions.remove(n);
        return;
      }
    }
    assert(false);
  }

  bool exec_command(view &v, element *source, element *target,
                    const ustring &command, const value &params) {
    traverser<event_command> trv(v);
    event_command            cevt(target, source);
    cevt.cmd     = event_command::CHECK;
    cevt.command = command;
    cevt.data    = params;
    if (trv.traverse(target, cevt)) {
      cevt.cmd = event_command::EXEC;
      return trv.traverse(target, cevt);
    }
    return false;
  }

  pair<bool, value> query_command(view &v, element *source, element *target,
                                  const ustring &command, const value &params) {
    traverser<event_command> trv(v);
    event_command            cevt(target, source);
    cevt.cmd     = event_command::CHECK;
    cevt.command = command;
    cevt.data    = params;
    if (trv.traverse(target, cevt)) return pair<bool, value>(true, cevt.result);
    return pair<bool, value>(false, value());
  }

  bool value_to_attribute_bag(const value &v, attribute_bag &atts) {
    if (v.is_string()) {
      atts.set(string(v.get_string()));
      return true;
    } else if (v.is_array_like() && v.size() == 2) {
      string  an = v[0].to_string();
      ustring av = v[1].to_string();
      atts.set(an, av);
      return true;
    } else if (v.is_map() || v.is_object() || v.is_proxy_of_object()) {
      auto oneach = [&](const tool::value &k, const tool::value &v) -> bool {
        string  an = k.to_string();
        ustring av = v.to_string();
        atts.set(an, av);
        return true;
      };
      v.visit(oneach);
      return true;
    }
    return false;
  }

  bool document::on(view &v, event_focus &evt) {
#ifdef _DEBUG
    /*if (!parent) {
      if (evt.cmd == (FOCUS_IN | EVENT_HANDLED))
        dbg_report("focusin-handled");
      else if (evt.cmd == (FOCUS_OUT | EVENT_HANDLED))
        dbg_report("focusout-handled");
      else if (evt.cmd == (FOCUS_IN))
        dbg_report("focusin");
      else if (evt.cmd == (FOCUS_OUT))
        dbg_report("focusout");
    }*/
#endif // _DEBUG
    return super::on(v, evt);
  }

  bool document::on(view &v, event_key &evt) {
    // tooltip_cmd(&v,REMOVE_TOOLTIP);

    if (super::on(v, evt)) return true;

    if (evt.cmd != KEY_DOWN /* || evt.alt_state != 0*/) return false;

    if (parent) return false; // functionality below is for root document only.

    v.remove_tooltips();

    int_v scroll_op;
    bool  vertical = true;
    switch (evt.key_code) {
    case KB_NEXT: scroll_op = SCROLL_PAGE_PLUS; break;
    case KB_PRIOR: scroll_op = SCROLL_PAGE_MINUS; break;
    case KB_UP: scroll_op = SCROLL_STEP_MINUS; break;
    case KB_DOWN: scroll_op = SCROLL_STEP_PLUS; break;
    case KB_HOME: scroll_op = SCROLL_HOME; break;
    case KB_END: scroll_op = SCROLL_END; break;
    case KB_LEFT:
      scroll_op = SCROLL_STEP_MINUS;
      vertical  = false;
      break;
    case KB_RIGHT:
      scroll_op = SCROLL_STEP_PLUS;
      vertical  = false;
      break;
    default: return false;
    }

    if (scroll_op.is_defined()) {
      traverser<event_scroll> trv(v);
      element *               target = v.focus_element ? v.focus_element : this;
      event_scroll            sevt(target, scroll_op, vertical, 0, SCROLL_SOURCE_KEYBOARD, evt.key_code);
      return trv.traverse(target, sevt);
    }
    return false;
  }

  bool document::on(view &v, event_behavior &evt) {
#ifdef DEBUG
    if ((evt.cmd == BUTTON_STATE_CHANGED) && parent)
      evt.cmd = evt.cmd;
#endif

    if (element::on(v, evt)) return true;

    if (evt.cmd == DOCUMENT_CLOSE_REQUEST && evt.target == this) {
      event_behavior bevt(this, this, CONTAINER_CLOSE_REQUEST, 0);
      bevt.data = evt.data;
      broadcast(v,bevt);
      if (bevt.is_canceled()) {
        evt.cancel_event();
        return true;
      }
    }
    else if (evt.cmd == DOCUMENT_CLOSING && evt.target == this) {
      event_behavior bevt(this, this, CONTAINER_CLOSING, 0);
      broadcast(v,bevt);
    }
    else if (evt.cmd == HYPERLINK_CLICK && evt.source) {
      string uri = evt.target->atts.get_url(this->uri().src, attr::a_href);

      ustring target;
      evt.target->get_attr("-target", target);

      if (uri.is_empty() || uri == CHARS("#")) return false;

      tool::url u;
      u.parse(uri);
      if (u.anchor_only()) {
        element *anchor =
            find_first(v, this,
                       ustring::format(W("[id='%S'],[name='%S']"),
                                       u.anchor.c_str(), u.anchor.c_str()));
        if (anchor) // found
        {
          v.ensure_visible(anchor, true);
          return true;
        }
      }
      if (!target.is_empty()) {
        if (target == W("@system")) {
          ustring path = url::file_url_to_path(uri);
          return tool::launch(path);
        }
        if (target.like(W("_*"))) return false;
        ustring selector = ustring::format(
            W("frame[id='%s'],frame[name='%s'],iframe[id='%s'],iframe[name='%s']"),
            target.c_str(), target.c_str(), target.c_str(), target.c_str());
        document *pd = this;
        while (pd) {
          element *frame = find_first(v, pd, selector, true, true);
          if (frame) // frame
          {
            handle<pump::request> prq = new pump::request(uri, DATA_HTML);
            prq->dst                  = frame;
            prq->initiator            = this;
            v.request_data(prq);
            return true;
          }
          if (!pd->parent) break;
          pd = pd->parent->doc();
        }
      }

      if (_view) // this is super super root
        return v.load_url(uri);
      else if (parent) // this is frame
      {
        handle<pump::request> rq = new pump::request(uri, DATA_HTML);
        rq->dst                  = parent;
        v.request_data(rq);
        return true;
      }
      return false;
    }

    if ((evt.cmd == MENU_ITEM_CLICK || evt.cmd == BUTTON_CLICK) && evt.source) {
      ustring cmd = evt.target->atts(attr::a_command);
      if (cmd.length()) {
        traverser<event_command> trv(v);
        event_command            cevt(evt.source, evt.target);
        cevt.command = cmd;
        cevt.cmd     = event_command::CHECK;
        if (trv.traverse(evt.source, cevt)) {
          cevt.cmd = event_command::EXEC;
          return trv.traverse(evt.source, cevt);
        }
      }
      return false;
    }

    return false;
  }

  bool document::on(view &v, event_mouse &evt) {
    // if((evt.cmd == (MOUSE_DOWN | EVENT_SINKING | EVENT_HANDLED)))
    //  evt.cmd = evt.cmd;

    if (v.tooltip_owner_element &&
        ((evt.cmd == (MOUSE_MOVE | EVENT_SINKING)) ||
         (evt.cmd == (MOUSE_LEAVE | EVENT_SINKING)))) {
      element *b = v.find_element(evt.pos_view); // to allow tooltips on disabled elements.
      if (!b) {
        v.tooltip_owner_element = nullptr;
        v.remove_tooltips();
      } else if (v.tooltip_owner_element && !b->belongs_to(v,v.tooltip_owner_element, true)) {
        v.tooltip_owner_element = nullptr;
        v.remove_tooltips();
      }
    }

    if (evt.cmd & EVENT_HANDLED) return super::on(v, evt);

    if (v.tooltip_owner_element) return super::on(v, evt);

    // static point tooltip_pos;

    // if(evt.cmd_no_flags() != MOUSE_CHECK && evt.pos_view != tooltip_pos)
    //  v.remove_tooltips();

    // WRONG: if(distance(evt.pos_view,tooltip_pos) > 2) v.remove_tooltips();

    point offset = size(v.pixels_per_dip(sizef(0, 24)));

    if (evt.cmd == MOUSE_IDLE && evt.button_state == 0 /*&& v.is_active() - showing tooltips on incative window*/) {
      element *b = v.find_element(evt.pos_view); // to allow tooltips on disabled elements.
      while (b) {
        if (b->state.popup()) break;
        if (b->state.owns_popup()) break;

        ustring t = b->attr_title();

        event_behavior tooltip_evt(0, evt.target, TOOLTIP_REQUEST, 0);
        if (b->on(v, tooltip_evt)) {
          if (tooltip_evt.source) {
            tooltip_evt.source->atts.set("role", WCHARS("tooltip"));
            v.tooltip_owner_element = b;
            point vpt = evt.pos_view + offset;
            v.show_popup(tooltip_evt.source, b, TOOLTIP_WINDOW, 0x20, vpt);
            return true;
          } else if (tooltip_evt.data.is_string()) {
            v.tooltip_owner_element = b;
            t = tooltip_evt.data.get<ustring>();
          } else
            //return super::on(v, evt);
            return true; // suppress tooltip show
        }

        if (t.length()) {
          // PLAIN_TEXT_TOOLTIP:
          helement tooltip = create_tooltip_element(v, t, PLAIN_TOOLTIP, b);
          if (tooltip) {
            tooltip->atts.set("role", WCHARS("tooltip"));
            v.tooltip_owner_element = b;
            point vpt = evt.pos_view + offset;
            v.show_popup(tooltip, b, TOOLTIP_WINDOW, 0x20, vpt);
          }
          return true;
        }
        t = b->attr_tooltip();
        if (t.length()) {
          helement tooltip = create_tooltip_element(v, t, HTML_TOOLTIP, b);
          if (tooltip) {
            tooltip->atts.set("role", WCHARS("tooltip"));
            v.tooltip_owner_element = b;
            v.show_popup(tooltip, b, TOOLTIP_WINDOW, 0x20, evt.pos_view + offset);
          }
          return true;
        }
        t = b->attr_titleid();
        if (t.length()) {
          helement tooltip = create_tooltip_element(v, t, ID_OF_TOOLTIP, b);
          if (tooltip) {
            tooltip->atts.set("role", WCHARS("tooltip"));
            v.tooltip_owner_element = b;
            v.show_popup(tooltip, b, TOOLTIP_WINDOW, 0x20,
                         evt.pos_view + offset);
          }
          return true;
        }

        hstyle st = b->get_style(v);

        if (b->need_ellipsis(v)) {
          helement tooltip =
              create_tooltip_element(v, W(""), ELEMENT_TEXT_TOOLTIP, b);
          if (tooltip) {
            tooltip->atts.set("role", WCHARS("overflow-tooltip"));
            v.tooltip_owner_element = b;
            v.show_popup(tooltip, b, TOOLTIP_WINDOW,
                         st->direction == direction_ltr ? 4 : 6);
          }
          return true;
        }

        else if (b->need_multiline_ellipsis(v)) {
          helement tooltip =
            create_tooltip_element(v, W(""), ELEMENT_TEXT_TOOLTIP, b);
          if (tooltip) {
            tooltip->atts.set("role", WCHARS("overflow-multiline-tooltip"));
            v.tooltip_owner_element = b;
            v.show_popup(tooltip, b, TOOLTIP_WINDOW,
              st->direction == direction_ltr ? 7 : 9);
          }
          return true;
        }


        /*t = b->atts(attr::a_alt);
        if( t.length() )
          goto PLAIN_TEXT_TOOLTIP;*/

        b = b->parent;
      }
    }
    return super::on(v, evt);
  }

  bool document::each_ui_child(function<bool(element *)> func) {
    handle<element> holder = this;
    if (super::each_ui_child(func)) return true;
    FOREACH(n, _synthetic_elements)
    {
      helement pup = _synthetic_elements[n];
      if (pup && (pup->doc() == this)) {
        if (func(pup)) return true;
      } else {
        _synthetic_elements.remove(n);
      }
    }
    return false;
  }

  bool document::debug_mode() const {
    if (_debug_mode) return true;
    if (parent) {
      document* pd = parent->doc();
      if(pd)
        return pd->debug_mode();
    }
    else {
      view* pv = pview();
      if (pv) 
        return pv->debug_mode();
    }
    return false;
  }

  element *document::create_tooltip_element(view &v, ustring text,
                                            TOOLTIP_TYPE pupt, element *b) {
    helement popup;
    switch (pupt) {
    case ID_OF_TOOLTIP: popup = find_by_id(text); break;
    case ELEMENT_TEXT_TOOLTIP:
      popup = new element(tag::T_POPUP);
      popup->node_index = SYNTHETIC_NODE_INDEX;
      _synthetic_elements.push(popup);
      popup->parent = b;
      assert(b);
      {
        array<wchar> overflow_text;
        b->get_ui_text(v, overflow_text);
        popup->append(new html::text(overflow_text()));
      }
      break;
    case PLAIN_TOOLTIP:
      popup = new element(tag::T_POPUP);
      _synthetic_elements.push(popup);
      popup->node_index = SYNTHETIC_NODE_INDEX;
      popup->parent = b;
      assert(b);
      popup->append(new html::text(text));
      break;
    case HTML_TOOLTIP:
      popup = new element(tag::T_POPUP);
      _synthetic_elements.push(popup);
      popup->node_index = SYNTHETIC_NODE_INDEX;
      popup->parent = b;
      v.set_element_html(popup, text, SE_REPLACE);
      break;
    }
    if (popup) popup->atts.set(attr::a_role, WCHARS("tooltip"));
    return popup;
  }

  void view::remove_tooltips() {
    for (int n = windows.size() - 1; n >= 0; --n) {
      handle<iwindow> iw = windows[n];
      if (!iw || !iw->is_tooltip()) continue;
      // helement el = iw->root;
      iw->dismiss();
    }
  }

  void document::register_popup(view &v, element *popup) {
    v.remove_tooltips();
    register_service_block(popup);
  }

  void document::register_service_block(element *popup) {
    _synthetic_elements.push(popup);
    popup->node_index = SYNTHETIC_NODE_INDEX;
    if (!popup->parent) {
      popup->parent = this;
      popup->owner  = this;
    }
  }

  // <!ENTITY ...>
  bool document::resolve_entity(chars name, ustring &val) {
    if (_entities.find(name, val)) return true;
    if (parent) {
      document *pd = parent->doc();
      return pd->resolve_entity(name, val);
    }

    if (name == CHARS("platform-cmd-mod")) {
#ifdef OSX
      static ustring t = WCHARS("\u2318"); // the command key symbol,
#else
      static ustring t = WCHARS("Ctrl");
#endif
      val = t;
      return true;
    } else if (name == CHARS("platform-shift-mod")) {
#ifdef OSX
      static ustring t = WCHARS("\u21E7"); // the shift Key symbol
#else
      static ustring t = WCHARS("Shift");
#endif
      val = t;
      return true;
    }

    return false;
  }
  void document::register_entity(chars name, wchars val) {
    string sname = name;
    if (_entities.exists(sname))
      return; // entities are "constants" - first defined stays intact.
    _entities[sname] = val;
  }

  int document::paginate(view &v, gool::size page_content_size,
                         array<gool::range> &pages) {
    int y = 0;

    pages.clear();

    size dim = this->content_dim();

    //;y < dim.y &&  page_count < 1000 ;++page_count

    drop_pagination(v);

    gool::range pg;
    gool::range ym;

    int nelements = 0; // on the page

    auto scan = [&](element *b) -> bool {
      nelements = b->paginate(v, pg, ym, nelements, pages.size() + 1);
      return false;
    };

    while (y < dim.y) {
      if (pages.size() >= 1000) break;
      pg = gool::range(y, y + page_content_size.y - 1);
      y += page_content_size.y;
      ym        = gool::range(y, y);
      nelements = 0;
      each_ui_child(scan);
      if (ym.length() < pg.length() && nelements > 0) {
        pg.e = ym.s;
        y = ym.s;
      }
      // else cannot satisfy constraints - simply ignore breakage
      pages.push(pg);
    };

    if (pages.size() == 0) pages.push(gool::range(0, page_content_size.y - 1));

    return pages.size();
  }

  bool document::on_timer(view &v, timer_def& td) {
#if defined(USE_INCREMENTAL_INIT)
    if (id == DELAYED_INIT_TIMER_ID && kind == INTERNAL_TIMER) {
      return !process_delayed_init(v);
      //return false; // stop it
    }
#endif
    return super::on_timer(v, td);
  }

#if defined(USE_INCREMENTAL_INIT)
  bool document::request_delayed_init(view& v, element* pel)
  {
    const int CAP = 100;
    if ((++_processed_text_blocks > CAP) && !pel->flags.delayed_layout)
    {
      pel->flags.delayed_layout = 1;
      if(_delayed_initialization_queue.length() == 0)
        v.start_timer(this, DELAYED_INIT_TIMER_MS, DELAYED_INIT_TIMER_ID, INTERNAL_TIMER);
      _delayed_initialization_queue.push(pel);
      //v.post(delegate(&v, &view::process_delayed_elements, handle<document>(this)), true);
      return true;
    }
    return false;
  }

  bool document::process_delayed_init(view& v)
  {
    //uint processed_text_blocks = 0;
    //array<weak_handle<element>> delayed_text_blocks;
    uint n = 0;
    uint nm = min((uint)_delayed_initialization_queue.length(),100u);
      //(uint)_delayed_initialization_queue.length();
    array<weak_handle<element>> delayed_initialization_queue = _delayed_initialization_queue(0, nm);
    _delayed_initialization_queue.remove(0, nm);
    _processed_text_blocks = 0;

    //dbg_printf("process_delayed_elements %d, left %d\n", delayed_initialization_queue.size(), _delayed_initialization_queue.size());
    for (n = 0; n < nm; ++n) {
      element* p = delayed_initialization_queue[n];
      if (!p) continue;
      assert(p->flags.delayed_layout);
      p->flags.delayed_layout = 0;
      v.add_to_update(p, true);
    }
    return _delayed_initialization_queue.size() == 0;
  }
#endif

} // namespace html
