#include "html.h"

namespace html {

  int target_y_pos(element *b);
  int target_x_pos(element *b);

  namespace behavior {

    struct vlist_ctl_factory : ctl_factory {
      vlist_ctl_factory() : ctl_factory("virtual-list") {}
      virtual ctl *create(element *el) override;
    };

    static vlist_ctl_factory *_vlist_ctl_factory = 0;

    struct vlist_ctl : ctl, animation
    {
      weak_handle<element>  self;
      uint      sliding_window_size = 160;
      uint      buffer_chunk_size = 80;
      size      overscroll;
      int       n_more_top = 0;
      int       n_more_bottom = 0;
      bool      initialized = false;

      weak_handle<element> first_in_set; // reference of very first/last elements 
      weak_handle<element> last_in_set;  // in list - represent first and last row in set
      
      bool      touch_scrolling = false; // true MOUSE_WHEEL was synthesized (touch, touchpad) and false if mouse wheel click  

      virtual CTL_TYPE get_type() { return CTL_LIST; }

      vlist_ctl() {
        subscription = uint(-1);
      }
      virtual const string &behavior_name() const override {
        return _vlist_ctl_factory->name;
      }
      virtual bool focusable(const element *self) override { return true; }

      bool allow_overscroll() const {
        return touch_scrolling;
      }

      virtual void detach(view &v, element *self) override {
        self = nullptr;
        //content_provider.clear();
      }
      virtual bool attach(view &v, element *self) override {
        this->self = self;
        self->flags.virtual_v_scrollbar = true;
        handle<vlist_ctl> _this = this;
        v.post([_this,&v]() -> bool {
          _this->init(v);
          return true;
        });
        return true;
      }

      virtual element* r13n_container(element* self) {
        return self; 
      }


      bool bottomed = false;

      void init(view& v) {
        if (initialized || !self) return;
        initialized = true;
        style* cs = self->get_style(v);
        bottomed = cs->vertical_align.val() == valign_e::valign_bottom;
        point sp;
        if (bottomed) {
          scroll_up(v, 0, int_v(), sliding_window_size);
          if (self->content_dim().y > self->dim().y)
            sp.y = self->content_dim().y - self->dim().y;
          perform_scroll_to(v, sp);
        }
        else {
          scroll_down(v, 0, int_v(), sliding_window_size);
        }
      }
      
      virtual void on_size_changed(view &v, element *self) override {
        if (!self || !self->n_children())
          return;
                  
        handle<vlist_ctl> _this = this;
        v.post([_this, &v]() -> bool {
          if (!_this->self) return true;
          point sp = _this->get_scroll_pos(v);
          if (sp.y + _this->self->dim().y > _this->self->content_dim().y) {
            sp.y = _this->self->content_dim().y - _this->self->dim().y;
            _this->perform_scroll_to(v, sp);
          }
          else if (sp.y < 0) {
            sp.y = 0;
            _this->perform_scroll_to(v, sp);
          }
          else
            _this->perform_scroll_to(v, sp);

          return true;
        }, true);
      }

      struct stroke_item {
        size advance;   // in "pixels"
        uint duration;  // in ticks
      };

      circular_buffer<stroke_item,8> strokes;
      uint   last_time = 0;
      sizef  speed; // pixels per ms
      size   rollback_step; // pixels

      void push_stroke(view &v, size advance) {
        uint tm = v.get_ticks();
        if (last_time) {
          stroke_item si;
          si.advance = advance;
          si.duration = max(ANIMATION_TIMER_SPAN/2,tm - last_time);
          strokes.push_front(si);
        }
        else {
          stroke_item si;
          si.advance = advance;
          si.duration = 16;
          strokes.push_front(si);
        }
        if (!touch_scrolling) {
          //speed = calc_average_speed(); // pixels per 1 millisecond
          speed.y += float(advance.y) / (400 * 2);

          //dbg_printf("speed %f %f\n", speed.x, speed.y);

          bool  scrollable_h = self->can_scroll_h(v);
          bool  scrollable_v = self->can_scroll_v(v);
          if (!scrollable_h) speed.x = 0;
          if (!scrollable_v) speed.y = 0;
          //dbg_printf("advance %f %f\n", speed.x, speed.y);
          if (!is_animating) {
            v.add_animation(self, this);
          }
        }

        last_time = tm;
      }

      sizef calc_average_speed(int m = 2) {

        int   n = 0;
        sizef sp;
        for (int i = 0; i < strokes.size(); ++i) {
          if (m-- == 0) break;
          const stroke_item &si = strokes[i];
          {
            sizef t = sizef(si.advance) / 16.0f; //float(si.duration);
            sp += t;
            ++n;
          }
        }
        if (n) {
          sp /= float(n);
          if (abs(sp.y) < abs(speed.y) && signbit(sp.y) == signbit(speed.y)) { sp.y = speed.y; }
          if (abs(sp.x) < abs(speed.x) && signbit(sp.x) == signbit(speed.x)) { sp.x = speed.x; }

          sizef discriminant = max_speed();

          sp.x = limit(sp.x, -discriminant.x, discriminant.x);
          sp.y = limit(sp.y, -discriminant.y, discriminant.y);

          dbg_printf("speed %f %f\n", sp.x, sp.y);
          /*if(sp.x != sp.x)
            calc_average_speed();
          if (abs(speed.y) > 10 && abs(sp.y) > 2 * abs(speed.y))
            calc_average_speed();*/

          return sp;
        }
        return sizef(0,0);
      }


      void stop_strokes(view &v, bool is_touch) 
      {
        speed = calc_average_speed(); // pixels per 1 millisecond
        
        bool scrollable_h = self->can_scroll_h(v);
        bool scrollable_v = self->can_scroll_v(v);
        if (!scrollable_h) speed.x = 0;
        if (!scrollable_v) speed.y = 0;

        if ((abs(speed.x) > 0.01f || abs(speed.y) > 0.01f) && !is_animating)
          v.add_animation(self, this);
        else
          check_stop(v, self);
      }

#ifdef USE_TOUCH
      enum OVERSCROLL {
        NONE = 0,
        AT_TOP = 1,
        AT_BOTTOM = 2,
      };
      OVERSCROLL is_rollback_animating = NONE;
#endif

      bool       is_animating = false;
      
      sizef max_speed() {
        return sizef(self->dim()) / 16.f;
      }

      virtual uint start(view& v, element* self, const style* /*nst*/, const style* /*ost*/)  override {
        is_animating = true;
        last_time = start_clock - ANIMATION_TIMER_SPAN;
        v.close_popup_tree(self);
        return ANIMATION_TIMER_SPAN;
      }
      virtual uint step(view& v, element* self, uint current_clock) override {
        v.close_popup_tree(self);
#ifdef USE_TOUCH
        if (is_rollback_animating)
          return step_rollback(v, self, current_clock);
#endif
        point delta = size(speed * 16.f); // float(max(1,current_clock - last_time)); - not reliable?

        last_time = current_clock;

        speed *= touch_scrolling ? 0.95f : 0.91f;

        if (abs(delta.x) < 1 && abs(delta.y) < 1)
          return check_stop(v, self);

        bool r;
        if (delta.y < 0)
          r = scroll_down(v, -delta.y);
        else
          r = scroll_up(v, delta.y);

        dbg_printf("scroll animation\n");

        return r ? ANIMATION_TIMER_SPAN : 0;
      }
      virtual void stop(view& v, element * self)  override  {
        is_animating = false;
#ifdef USE_TOUCH
        is_rollback_animating = NONE;
#endif
        speed = sizef(0, 0);
        overscroll = size();
        rollback_step = size();
        last_time = 0;
        strokes.clear();
        dbg_printf("STOP scroll animation\n");
      }

      uint check_stop(view& v, element * self) {
#ifdef USE_TOUCH

        if (!touch_scrolling)
          return 0;

        point offset;

        if (is_at_top(v, offset.y) && offset.y > 0) {
          is_rollback_animating = AT_TOP;
          if (!is_animating)
            v.add_animation(self, this);
          overscroll.y = offset.y;
          rollback_step.y = 0;
          return ANIMATION_TIMER_SPAN;
        }
        if (is_at_bottom(v, offset.y) && offset.y > 0) {
          is_rollback_animating = AT_BOTTOM;
          if (!is_animating)
            v.add_animation(self, this);
          overscroll.y = offset.y;//-get_scroll_pos(v).y;
          rollback_step.y = 0;
          return ANIMATION_TIMER_SPAN;
        }
        /*if (abs(speed.y) > 0.001f || abs(speed.x) > 0.001f)
        {
          //dbg_printf("speed.y=%f\n",speed.y);
          //assert(false);
          return ANIMATION_TIMER_SPAN;
        }*/
        dbg_printf("SCROLL STOP\n");
#endif
        return 0;
      }

#ifdef USE_TOUCH
      uint step_rollback(view& v, element* self, uint current_clock) {

        switch (is_rollback_animating) {
          case AT_TOP: {
            point sp = get_scroll_pos(v);
            int coverscroll = -sp.y;
            if (coverscroll == 0)
              return 0;
            assert(overscroll.y > 0);
            assert(coverscroll > 0);
            assert(rollback_step.y >= 0);
            if (coverscroll >= overscroll.y / 2) {
              rollback_step.y += 2;
            }
            else {
              rollback_step.y = max(1,rollback_step.y - 2);
            }
            sp += rollback_step;
            if (sp.y >= 0) {
              sp.y = 0;
              rollback_step.y = 0;
              perform_scroll_to(v, sp);
              dbg_printf("ROLLBACK STOP\n");
              return 0;
            } else
              perform_scroll_to(v, sp);
            break;
          }
          case AT_BOTTOM: {
            point sp = get_scroll_pos(v);
            point spe = get_end_pos(v);
            int coverscroll = sp.y - spe.y;
            //is_at_bottom(v, coverscroll);
            //if (coverscroll == 0) {
            //  dbg_printf("ROLLBACK STOP\n");
            //  return 0;
            //}
            assert(overscroll.y > 0);
            assert(coverscroll > 0);
            assert(rollback_step.y >= 0);
            if (coverscroll >= overscroll.y / 2) {
              rollback_step.y += 2;
            }
            else {
              rollback_step.y = max(1, rollback_step.y - 2);
            }
            sp -= rollback_step;
            if (sp.y <= spe.y) {
              sp.y = spe.y;
              rollback_step.y = 0;
              perform_scroll_to(v, sp);
              dbg_printf("ROLLBACK STOP\n");
              return 0;
            }
            else
              perform_scroll_to(v, sp);
            break;
          }
        }

        return ANIMATION_TIMER_SPAN;
      }
#endif

      size click_delta(view& v, element* self, size delta) {
        size dim = self->dim();
        style* cs = self->get_style(v);
#ifdef USE_TOUCH
        size step;
        step.y = pixels(v, self, cs->wheel_step(true, size_v(dim.y, size_v::unit_type::ppx)), dim).height();
        step.x = pixels(v, self, cs->wheel_step(false, size_v(dim.y, size_v::unit_type::ppx)), dim).width();
        delta.x = delta.x < 0 ? -step.x : step.x;
        delta.y = delta.y < 0 ? -step.y : step.y;
#else
        delta = v.dip_to_ppx(delta) * 2;
#endif
        return delta;
      }

      size na_click_delta(view& v, element* self, size delta) {
        /*size dim = self->dim();
        delta.x = (delta.x < 0 ? -dim.x : dim.x) / 8;
        delta.y = (delta.y < 0 ? -dim.y : dim.y) / 8;*/

        size dim = self->dim();
        style* cs = self->get_style(v);
        size step;
        step.y = pixels(v, self, cs->wheel_step(true, size_v(dim.y / 8, size_v::unit_type::ppx)), dim).height();
        step.x = pixels(v, self, cs->wheel_step(false, size_v(dim.y / 8, size_v::unit_type::ppx)), dim).width();
        delta.x = delta.x < 0 ? -step.x : step.x;
        delta.y = delta.y < 0 ? -step.y : step.y;

        return delta;
      }


      virtual bool on(view& v, element* self, event_mouse& evt) override {
        switch (evt.cmd) {
#ifdef USE_TOUCH
          case MOUSE_TOUCH_START:
            v.set_capture(self);
            if (is_animating) {
              v.remove_animation(self, this);
            }
            return true;

          case MOUSE_TOUCH_END | EVENT_SINKING:
            v.set_capture(nullptr);
            stop_strokes(v,true);
            return true;
#endif

          case MOUSE_WHEEL: {
#ifdef USE_TOUCH
            if (!evt.is_physical_scroll())
              return true;
            if (is_rollback_animating)
              return true;
            if (evt.is_control())
              return false;
            touch_scrolling = !evt.is_real_wheel_tick();
#endif
            bool animated_scroll = self->get_style(v)->smooth_scroll(true);

            size advance = touch_scrolling ? evt.wheel_delta_xy() 
              : !animated_scroll ? na_click_delta(v, self, evt.wheel_delta_xy())
              : click_delta(v, self, evt.wheel_delta_xy()); //v.pixels_per_dip(evt.wheel_delta_xy());

            if(animated_scroll)
              push_stroke(v, advance);

            if (touch_scrolling || !animated_scroll) {
              int wheel_advance = advance.y;
              if (wheel_advance < 0)
                scroll_down(v, -wheel_advance);
              else if (wheel_advance > 0)
                scroll_up(v, wheel_advance);
            }
            return true;
          }
        }
        return false;
      }

      int n_visible(view &v, element *self) {
        auto pa = get_visible_range(v);
        if (pa.first && pa.second)
          return pa.second->index() - pa.first->index() + 1;
        return 0;
      }

      virtual bool on(view &v, element *self, event_scroll &evt) override 
      { 
        if (!evt.vertical)
          return false;
        if(evt.source == SCROLL_SOURCE_ANIMATOR)
          return false;
        switch (evt.cmd)
        {
        case SCROLL_HOME:
FIRST:
          return go_to_item(v, 0);
        case SCROLL_END: {
LAST:
          self->clear();
          n_more_bottom = 0;
          n_more_top = 0;
          scroll_up(v, 0);
          point sp;
          sp.y = self->content_dim().y - self->dim().y;
          perform_scroll_to(v, sp);
          return true;
        }
        case SCROLL_STEP_PLUS: {
          auto pa = get_visible_range(v);
          if (pa.first && pa.second) {
            int index = n_more_top + pa.first->index() + 1;
            if (index + n_visible(v, self) >= total_items())
              goto LAST;
            return go_to_item(v, index);
          }
        }
        case SCROLL_STEP_MINUS: {
          auto pa = get_visible_range(v);
          if (pa.first && pa.second) {
            int index = n_more_top + pa.first->index() - 1;
            if (index < 0)
              goto FIRST;
            return go_to_item(v, index);
          }
        }
        case SCROLL_PAGE_MINUS: {
            auto pa = get_visible_range(v);
            if (pa.first && pa.second) {
              int index = max(0,n_more_top - max(1,pa.second->index() - pa.first->index()));
              return go_to_item(v, index);
            }
            return false;
        }
        case SCROLL_PAGE_PLUS: {
          auto pa = get_visible_range(v);
          if (pa.first && pa.second) {
            int index = n_more_top + max(1,pa.second->index());
            if (index + n_visible(v, self) >= total_items())
              goto LAST;
            return go_to_item(v, index);
          }
          return false;
        }
        case SCROLL_POS:
          return go_to_scroll_pos(v, evt.pos);
        }
        return false; 
      }

      int total_items() {
        return n_more_top + self->n_children() + n_more_bottom;
      }

      bool go_to_scroll_pos(view& v, int pos) {
        auto details = get_scroll_details(v);
        if (pos < details.before_pixels)
          return go_to_item(v, pos / details.average_item_height, float(pos % details.average_item_height)/ details.average_item_height);
        if (pos >= details.before_pixels && pos < details.before_pixels + details.window_pixels) {
          perform_scroll_to(v, point(0, pos - details.before_pixels), true);
          return true;
        }
        pos -= details.before_pixels + details.window_pixels;
        int item_no = pos / details.average_item_height + n_more_top + self->n_children();
        return go_to_item(v, item_no);
      }

      bool go_to_item(view& v, int item_no, float offset = 0) {
        initialized = true;
        int new_elements = this->request_elements(v, buffer_chunk_size, 0, item_no);

        if (!self->is_layout_valid())
          self->measure_inplace(v);

        v.update_element(self);

        point pos(0, 0);
        if (helement first = self->first_element()) {
          pos.y = int(first->border_box(v).height() * offset);
        }

        perform_scroll_to(v, pos,true);

        return true;
      }

      point get_scroll_pos(view &v) {
        if (!self)
          return point();
        if (!self->is_layout_valid())
          self->measure_inplace(v);
        return self->scroll_pos();
      }

      point get_end_pos(view &v) {
        if (!self)
          return point();
        if (!self->is_layout_valid())
          self->measure_inplace(v);
        scroll_data sd;
        self->get_scroll_data(v, sd);
        return sd.content_outline.e - sd.dim_view + point(1,1);
      }

      element* immediate_child_of(element* container, element* child) {
        if (!child)
          return nullptr;
        if (child->parent == container)
          return child;
        return immediate_child_of(container, child->parent);
      }

      pair<helement,helement> get_visible_range(view& v) {
        pair<helement, helement> pa;
        if (!self)
          return pair<helement, helement>();
        point sp = get_scroll_pos(v);
        /*
          pa.first = immediate_child_of(self,self->find_child_element(v, sp, false));
          if (!pa.first) pa.first = self->first_element();
          sp.y += self->dim().y - 1;
          pa.second = immediate_child_of(self,self->find_child_element(v, sp, false));
          if (!pa.second) pa.second = self->last_element();
          //assert(pa.first);
        */
        helement etc;
        range y = self->client_box(v).y();
        y += sp.y;
        each_child it(self);
        for (element* child; it(child);) {
          if (!child->is_layout_valid())
            child->measure_inplace(v);
          range cy = child->border_box(v, element::TO_PARENT).y();
          if (cy.empty())
            continue;
          if (y.covers(cy)) {
            if (!pa.first) pa.first = child;
            pa.second = child;
          }
          else if (y.overlaps_with(cy))
            etc = child;
        }
        if (!pa.first)
          pa.first = pa.second = etc;
        return pa;
      }

      struct scroll_details {
        int average_item_height = 15;
        int before_pixels = 0;
        int after_pixels = 0;
        int window_pixels = 0;
        int view_pixels = 0;
        int content_pixels() const { return before_pixels + window_pixels + after_pixels; }
      };

      scroll_details get_scroll_details(view &v) {

        scroll_details details;

        int n = self->n_children();
        scroll_data sd;
        self->get_scroll_data(v, sd);
        details.window_pixels = sd.content_outline.height();
        if(n) details.average_item_height = details.window_pixels / n;
        details.before_pixels = n_more_top * details.average_item_height;
        details.after_pixels = n_more_bottom * details.average_item_height;
                
        details.view_pixels = sd.dim_view.y;
        return details;
      }
      
      void perform_scroll_to(view &v, point pos, bool update = false) {
        if (!self->is_layout_valid())
          self->measure_inplace(v);

        if(update)
          v.update_element(self);

        self->scroll_pos(v, pos, allow_overscroll());

        scroll_details details = get_scroll_details(v);
        
        range ra(0, details.content_pixels());

        int   spos = details.before_pixels + pos.y;

        self->ldata->sb.set_v(v, self, ra, details.view_pixels, spos, SB_MODE::SBM_PER_CSS);
        v.refresh(self);

        event_scroll evt(self,SCROLL_POS,true,spos,SCROLL_SOURCE::SCROLL_SOURCE_UNKNOWN,0);
        v.handle_element_event(self, evt);
      }

      // true if sliding window contains very first item
      bool is_at_top(view &v, int& offset) { 
        if (!first_in_set) 
          return false;
        if (!self->is_layout_valid())
          self->measure_inplace(v);
          //v.update();

        offset = first_in_set->border_box(v,element::TO_VIEW).s.y 
               - self->content_box(v, element::TO_VIEW).s.y;

        return true;
      }

      // true if sliding window contains very last item
      bool is_at_bottom(view &v, int& offset) {
        if (!last_in_set) 
          return false;
        if (!self->is_layout_valid())
          self->measure_inplace(v);
          //v.update();

        offset = self->content_box(v, element::TO_VIEW).e.y 
               - last_in_set->border_box(v, element::TO_VIEW).e.y;
        return true;
      }

      bool is_at_top_overscroll(view &v) {
        int offset;
        if (!is_at_top(v, offset))
          return false;
        return offset > 0;
      }
      bool is_at_bottom_overscroll(view &v) {
        int offset;
        if (!is_at_bottom(v, offset))
          return false;
        return offset > 0;
      }

      int request_elements(view& v, uint n, int where, int_v from_index) {
        int n_before = self->n_children();
        event_behavior evt(self, self, CONTENT_REQUIRED, n);
        evt.event_bubbling(false);
        evt.data = 
          tool::value::make_map({ 
            {"length",value(n)}, 
            {"start", from_index.is_undefined() ? value() : value(from_index.val(0))},
            {"where", value(where)}
          });
                                  
        bool handled = self->on(v, evt);
        int n_after = self->n_children();
        int n_fetched = max(0,n_after - n_before);
        if(handled || (n_fetched > 0))
        {
          if (evt.data.is_map_alike()) {
            if (where > 0) {
              this->n_more_bottom = uint(max(0, evt.data.get_prop("moreafter").get_int()));
              value morebefore = evt.data.get_prop("morebefore");
              if(morebefore.is_number())
                this->n_more_top = uint(max(0, morebefore.get_int()));
            }
            else if (where < 0) {
              this->n_more_top = uint(max(0, evt.data.get_prop("morebefore").get_int()));
              value moreafter = evt.data.get_prop("moreafter");
              if (moreafter.is_number())
                this->n_more_bottom = uint(max(0, moreafter.get_int()));
            }
            else {
              value morebefore = evt.data.get_prop("morebefore");
              if (morebefore.is_number())
                this->n_more_top = uint(max(0, morebefore.get_int()));
              value moreafter = evt.data.get_prop("moreafter");
              if (moreafter.is_number())
                this->n_more_bottom = uint(max(0, moreafter.get_int()));
            }
          }
          return n_fetched;
        }
        return 0;
      }

      bool scroll_up(view &v, int delta_y, int_v index = int_v(), int_v chunk_size = int_v()) {

        initialized = true;
        point scroll_pos = get_scroll_pos(v);
      
        point scroll_to = scroll_pos;
        scroll_to.y -= delta_y;

        auto fetch_more = [&]() -> bool {

          helement first = self->first_element();
          int prev_top = first ? first->pos().y : 0;

          if (index.is_undefined() && delta_y)
            index = n_more_top - 1;

          int new_elements = this->request_elements(v, chunk_size.val(buffer_chunk_size), -1, index);
          if (new_elements == 0)
            return true;

          self->drop_layout();
          self->measure_inplace(v);

          first = self->child(new_elements);
          if (!first) 
            return false; // TODO: error
          
          int new_top = first->pos().y;
          scroll_pos.y += new_top - prev_top; // adjust scroll position.
          
          if (uint(self->n_children()) > sliding_window_size) {
            helement pn = self->child(sliding_window_size);
            int last_ni = pn->node_index;
            int n_removed = self->nodes.size() - last_ni;
            self->remove_nodes(last_ni, self->nodes.size(), &v);
            self->drop_layout();
            self->measure_inplace(v);
            n_more_bottom += n_removed;
          }
          return false;
        };

        if (scroll_to.y <= 0 || delta_y == 0)
        {
          // we need to pump more items in this virtual list:      
          bool out = fetch_more();
          if (out && !allow_overscroll()) {
            scroll_to.y = 0;
            first_in_set = self->first_element();
          }
          else if (out) {
            float max_over = self->dim().y / 4.0f;
            float mul = max(0, max_over - abs(scroll_pos.y)) / max_over;

            scroll_to.y = scroll_pos.y - int(delta_y * mul);
            assert(mul <= 1.0f);
            speed.y *= mul;
            first_in_set = self->first_element();
          }
          else {
            scroll_to.y = scroll_pos.y - delta_y;
          }
        }
        if (scroll_to == scroll_pos && !touch_scrolling)
          return false;
        perform_scroll_to(v, scroll_to);
        return true;
      }

      bool scroll_down(view &v, int delta_y, int_v index = int_v(), int_v chunk_size = int_v()) {

        initialized = true;

        point scroll_pos = get_scroll_pos(v);
        
        int content_height = self->content_dim().y;
        int scroll_bottom = content_height - self->dim().y - scroll_pos.y;

        point scroll_to = scroll_pos;
        scroll_to.y += delta_y;

        auto fetch_more = [&]() -> bool {

          helement last = self->last_element();
          //if(!last)
          //  return true; // no content

          if (index.is_undefined() && delta_y)
            index = n_more_top + self->n_children();

          int new_elements = this->request_elements(v, chunk_size.val(buffer_chunk_size), +1, index);
          if (new_elements == 0)
            return true; // nothing was inserted - no more records. true - to mark end reached

          self->drop_layout();
          self->measure_inplace(v);

          content_height = self->content_dim().y;

          // drop first items that exceed sliding_window_size
          int n_elements_to_remove = max(0,self->n_children() - sliding_window_size);
          if (n_elements_to_remove) {
            helement pn = self->child(n_elements_to_remove);
            int first_ni = pn->node_index;
            self->remove_nodes(0, first_ni, &v);
            n_more_top += n_elements_to_remove;
            self->drop_layout();
            self->measure_inplace(v);
          }

          int content_height_after = self->content_dim().y;       // adjust scroll position as
          scroll_pos.y += content_height_after - content_height;  // we've removed first items.
          content_height = content_height_after;
          return false;
        };

        if (scroll_bottom < delta_y)
        {
          // we need to pump more items in this virtual list:      
          bool out = fetch_more();
          if (out && !allow_overscroll()) {
            scroll_to.y = content_height - self->dim().y;
            last_in_set = self->last_element();
          }
          else if(out) {
            //scroll_to.y = scroll_pos.y + delta_y / 2;
            //last_in_set = self->last_element();
            float max_over = self->dim().y / 4.0f;
            int last_scroll_pos = content_height - self->dim().y;

            float mul = 1.0f;
            if (abs(scroll_pos.y - last_scroll_pos) > max_over)
              mul = 0;
            else
              mul = (max_over - abs(scroll_pos.y - last_scroll_pos)) / max_over;

            //assert(mul <= 1.0f);
            //if (mul > 1.0f)
            //  mul = mul;

            scroll_to.y = scroll_pos.y + int(delta_y * mul);
            speed.y *= mul;
            last_in_set = self->last_element();
          }
          else {
            scroll_to.y = scroll_pos.y + delta_y;
          }
        }

        if (scroll_to == scroll_pos && !touch_scrolling)
          return false;
        perform_scroll_to(v, scroll_to);
        return true;
      }

      bool advance_to(view &v, int index) {

        initialized = true;

        auto fetch_down = [&]() -> bool {

          //if(!last)
          //  return true; // no content

          int new_elements = this->request_elements(v, buffer_chunk_size, +1, index);
          if (new_elements == 0)
            return false; // nothing was inserted - no more records. true - to mark end reached

          self->drop_layout();
          self->measure_inplace(v);

          helement last = self->last_element();
          if (!last)
            return false; // TODO: error

          // drop first items that exceed sliding_window_size
          int n_elements_to_remove = max(0, self->n_children() - sliding_window_size);
          if (n_elements_to_remove) {

            point scroll_pos = get_scroll_pos(v);
            int before = last->border_box(v, element::RELATIVE_TO::TO_PARENT).top();

            helement pn = self->child(n_elements_to_remove);
            int first_ni = pn->node_index;
            self->remove_nodes(0, first_ni, &v);
            n_more_top += n_elements_to_remove;
            self->drop_layout();
            self->measure_inplace(v);

            int after = last->border_box(v, element::RELATIVE_TO::TO_PARENT).top();
            scroll_pos.y -= before - after;

            self->set_scroll_pos(v, scroll_pos, false, true);

          }
                    
          return true;
        };

        auto fetch_up = [&]() -> bool {

          point scroll_pos = get_scroll_pos(v);

          helement first = self->first_element();
          if (!first)
            return false; // TODO: error

          int before = first->border_box(v, element::RELATIVE_TO::TO_PARENT).top();

          int new_elements = this->request_elements(v, buffer_chunk_size, -1, index);
          if (new_elements == 0)
            return false;

          self->drop_layout();
          self->measure_inplace(v);

          first = self->child(new_elements);
          if (!first)
            return false; // TODO: error

          int after = first->border_box(v, element::RELATIVE_TO::TO_PARENT).top();

          scroll_pos.y += after - before;

          self->set_scroll_pos(v, scroll_pos, false, true);

          if (uint(self->n_children()) > sliding_window_size) {
            helement pn = self->child(sliding_window_size);
            int last_ni = pn->node_index;
            int n_removed = self->nodes.size() - last_ni;
            self->remove_nodes(last_ni, self->nodes.size(), &v);
            self->drop_layout();
            self->measure_inplace(v);
            n_more_bottom += n_removed;
          }
          return true;
        };


        if (index >= self->n_children() + n_more_top) {
          if (!fetch_down())
            return false;
        }
        else if (index < n_more_top) {
          if (!fetch_up())
            return false;
        }

        helement ch = self->child(index - n_more_top);
        if (ch) v.ensure_visible(ch, false, SCROLL_SMOOTH);

        return true;
        
      }

      void scroll_to(view &v, int index) {
        initialized = true;
        if (!self) return;

        int new_elements = this->request_elements(v, buffer_chunk_size, 0, index);
        //if (new_elements == 0)
        //  return; // nothing was inserted - no more records. true - to mark end reached

        self->drop_layout();
        self->measure_inplace(v);

        perform_scroll_to(v, point(0,0), true);
      }

      //bool get_overscroll() { return this->allow_overscroll(); }
      //bool set_overscroll(bool onoff) { this->allow_overscroll() = onoff; return true; }

      //uint get_list_items() { return this->n_list_items; }
      //bool set_list_items(uint n) { this->n_list_items = max(n,self->n_children()); return true; }

      //value get_first_visible() { return value(); }
      //value get_last_visible() { return value(); }

      bool navigate_to(value to, int_v index) {
        if (!self) return false;
        html::view* pv = self->pview();
        if (!pv) return false;
        if (to.is_int()) {
          html::event_scroll evt(self, SCROLL_POS, true, to.get_int(), SCROLL_SOURCE::SCROLL_SOURCE_UNKNOWN, 0);
          return this->on(*pv, self, evt);
        }
        else if (to.is_string()) {
          html::event_scroll evt(self, 0, true, 0, SCROLL_SOURCE::SCROLL_SOURCE_UNKNOWN, 0);
          string cmd = to.get_string();
          if (cmd == CHARS("start")) evt.cmd = SCROLL_HOME;
          else if (cmd == CHARS("end")) evt.cmd = SCROLL_END;
          else if (cmd == CHARS("pagenext")) evt.cmd = SCROLL_PAGE_PLUS;
          else if (cmd == CHARS("pageprior")) evt.cmd = SCROLL_PAGE_MINUS;
          else if (cmd == CHARS("itemnext")) evt.cmd = SCROLL_STEP_PLUS;
          else if (cmd == CHARS("itemprior")) evt.cmd = SCROLL_STEP_MINUS;
          else if ((cmd == CHARS("advance") || cmd == CHARS("position")) && index.is_defined()) return advance_to(*pv, index.val(0));
          //else if (cmd == CHARS("replace")) return replace(*pv);
          else goto BAD_PARAM;
          return this->on(*pv, self, evt);
        }
      BAD_PARAM:
        return false;
      }

      value get_first_visible() {
        if (!self) return value();
        html::view* pv = self->pview();
        if (!pv) return value();
        pair<helement, helement> ra = this->get_visible_range(*pv);
        return ra.first ? value::wrap_resource(ra.first) : value();
      }

      value get_last_visible() {
        if (!self) return value();
        html::view* pv = self->pview();
        if (!pv) return value();
        pair<helement, helement> ra = this->get_visible_range(*pv);
        //return value::wrap_resource(ra.second);
        return ra.second ? value::wrap_resource(ra.second) : value();
      }

      value get_buffer_first_index() {
        if (!self) return value();
        if (!self->n_children())
          return value();
        return value(n_more_top);
      }

      value get_buffer_last_index() {
        if (!self) return value();
        if (int n = self->n_children())
          return value(n_more_top + n - 1);
        return value();
      }

      int get_items_before() {
        if (!self) return 0;
        return n_more_top;
      }

      int get_items_total() {
        if (!self) return 0;
        return n_more_top + self->n_children() + n_more_bottom;
      }

      bool set_items_before(int n) {
        if (!self) return false;
        html::view* pv = self->pview();
        if (!pv) return false;
        if (n_more_top == n) return false;

        n_more_top = n;

        range ra(0, n_more_top + self->n_children() + n_more_bottom);
        int   spos = n_more_top;
        int   page = 1;
        auto pa = get_visible_range(*pv);
        if (pa.first && pa.second) {
          spos += pa.first->index();
          page = max(1, pa.second->index() - pa.first->index() + 1);
        }
        self->ldata->sb.set_v(*pv, self, ra, page, spos, SB_MODE::SBM_PER_CSS);
        pv->refresh(self);
        return true;
      }

      int get_items_after() {
        return n_more_bottom;
      }

      int get_sliding_window_size() {
        return sliding_window_size;
      }
      bool set_sliding_window_size(int ws) {
        if (ws) {
          sliding_window_size = uint(ws);
          buffer_chunk_size = sliding_window_size / 2;
        }
        else {
          sliding_window_size = 160;
          buffer_chunk_size = 80;
        }
        return true;
      }

      bool set_items_after(int n) {
        if (!self) return false;
        html::view* pv = self->pview();
        if (!pv) return false;
        if (n_more_bottom == n) return false;

        n_more_bottom = n;

        range ra(0, n_more_top + self->n_children() + n_more_bottom);
        int   spos = n_more_top;
        int   page = 1;
        auto pa = get_visible_range(*pv);
        if (pa.first && pa.second) {
          spos += pa.first->index();
          page = max(1, pa.second->index() - pa.first->index() + 1);
        }
        self->ldata->sb.set_v(*pv, self, ra, page, spos, SB_MODE::SBM_PER_CSS);
        pv->refresh(self);
        return true;
      }


      SOM_PASSPORT_BEGIN_EX(vlist, vlist_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(navigate, navigate_to)
        )
        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(firstVisibleItem, get_first_visible),
          SOM_RO_VIRTUAL_PROP(lastVisibleItem, get_last_visible),
          SOM_RO_VIRTUAL_PROP(firstBufferIndex, get_buffer_first_index),
          SOM_RO_VIRTUAL_PROP(lastBufferIndex, get_buffer_last_index),
          SOM_VIRTUAL_PROP(itemsBefore, get_items_before, set_items_before),
          SOM_VIRTUAL_PROP(itemsAfter, get_items_after, set_items_after),
          SOM_RO_VIRTUAL_PROP(itemsTotal, get_items_total),
          SOM_VIRTUAL_PROP(slidingWindowSize, get_sliding_window_size, set_sliding_window_size),

        )
      SOM_PASSPORT_END

    };

    ctl *vlist_ctl_factory::create(element *) {
      return new vlist_ctl();
    }

    void init_virtual_list() {
      ctl_factory::add(_vlist_ctl_factory = new vlist_ctl_factory());
    }

  }
}
