#include "html.h"

namespace html {
  namespace behavior {

    struct textarea_ctl_factory : public ctl_factory {
      textarea_ctl_factory() : ctl_factory("textarea") {}
      virtual ctl *create(element *el);
    };

    static textarea_ctl_factory *_textarea_ctl = 0;

    const string &textarea_ctl::behavior_name() const {
      return _textarea_ctl->name;
    }

    static bool is_inside_focus(const element *self) {
      if (self->state.focus()) return true;
      // if( self->state.current() )
      //  return true;
      if ((self->tag == tag::T_CAPTION || self->tag == tag::T_TEXT) &&
          self->parent && self->parent->state.focus())
        return true;
      return false;
    }

    void textarea_ctl::check_empty(view &v, element *self) {
      ustring tv;
      get_text(v, self,tv);
      bool   is_empty_now = tv.length() == 0;
      if (_is_empty != is_empty_now) {
        _is_empty = is_empty_now;
        v.add_to_update(self, CHANGES_VISUAL);
      }
    }

    bool textarea_ctl::attach(view &v, element *_this) {
      if (_this->get_style(v)->collapse_ws()) {
        view::debug_printf(
            OT_DOM, OS_ERROR,
            "behavior:textarea requires white-space:pre or pre-wrap\n");
        return false;
      }

      edit_buffer(v, _this);

      _this->check_layout(v);
      if (_this->layout_type() != flow_text)
        text_block::setup_on(v, _this, _this->nodes());
      assert(_this->layout_type() == flow_text);
      self = _this->layout_type() == flow_text ? _this->cast<text_block>() : 0;
      if (!self) return false;

      check_empty(v, self);
      rq_spell_check(v);

      return true;
    }

    void textarea_ctl::detach(view &v, element *_this) {
      this->self = 0;
      anchor = caret = bookmark();
      _to_undo       = 0;
      _to_redo       = 0;
    }

    bool textarea_ctl::get_auto_width(view &v, element *self, int &value) {
      int sz = self->atts.get_int(attr::a_cols, 0);
      if (sz) {
        value = sz * (pixels(v, self, self->get_style(v)->font_size).height() * 3) / 4;
      } else
        value = 300;
      return true;
    }
    bool textarea_ctl::get_auto_height(view &v, element *self, int &value) {
      int sz = self->atts.get_int(attr::a_rows, 0);
      if (sz) {
        style *cs = self->get_style(v);
        int    h  = pixels(v, self, cs->font_size).height();
        if (cs->line_height.is_defined())
          h = cs->line_height.pixels(cs->font_size, 0);
        value = sz * h * 4 / 3;
      } else
        value = 150;
      return true;
    }

    html::text *textarea_ctl::text_node() const {
      for (int n = 0; n < self->nodes.size(); ++n)
        if (self->nodes[n]->is_text()) return self->nodes[n].ptr_of<text>();
      return 0;
    }

    wchars textarea_ctl::chars() const {
      text *pt = text_node();
      if (pt) return pt->chars();
      return wchars();
    }
    /*wchars textarea_ctl::chars() const
    {
      self->ldata.ptr_of<text_block::layout_data>()->chars();
    }*/

    argb textarea_ctl::selection_back_color(view &v) {
      const style *cs = self->get_style(v);
      argb c = cs->text_selection_background_color.to_argb();
      return c;
    }
    argb textarea_ctl::selection_fore_color(view &v) {
      const style *cs = self->get_style(v);
      argb c = cs->text_selection_background_color.to_argb();
      c.alfa = 127;
      return c;
    }
    argb textarea_ctl::selection_text_color(view &v) {
      const style *cs = self->get_style(v);
      argb c = cs->text_selection_color.to_argb();
      return c;
    }
    argb textarea_ctl::selection_caret_color(view &v) {
      const style *cs = self->get_style(v);

      if (cs->text_selection_caret_color.is_defined())
        return cs->text_selection_caret_color.to_argb();

      argb c = cs->back_color.is_defined() ? cs->back_color.to_argb()
                                           : argb(255, 255, 255);
      if (c.alfa == 0) return argb(0, 0, 0);
      c.alfa = 255;
      return c.inverse();
    }

    bool textarea_ctl::draw_content(view &v, element *self, graphics *pg,
                                    point pos) {
      auto_state<selection_ctx *> _(v._selection_ctx, this);
      self->draw_content(v, pg, pos);
      return true;
    }

    bool textarea_ctl::allow_text_selection_on(text_block *tb) {
      element *t = tb;
      while (t) {
        if (t->flags.disable_text_selection) return false;
        if (t == self) break;
        t = t->parent;
      }
      return true;
    }

    ustring textarea_ctl::get_filter_attr() {
      if (self->tag == tag::T_CAPTION && self->parent && self->state.synthetic())
        return get_attr(self->parent, "-filter");
      else
        return get_attr(self, "-filter");
    }

    bool textarea_ctl::check_char(wchar c, bool signal) {
      if (c == '\r') {
        if(is_multiline())
          return true;
        else 
          return false;
      }
      if (c >= ' ') {
        ustring f = get_filter_attr();

        if (f.length() != 0) {
          tool::charset<wchar, wchar('~'), wchar(0)> cset;
          const wchar *                              pc = f;
          cset.parse(pc);
          if (cset.valid(c))
            return true; // ok
          else           // not valid
          {
            if (signal) beep();
            return false;
          }
        }
        return true;
      } else if (signal)
        beep();
      return false;
    }

    bool textarea_ctl::check_chars(array<wchar> &txt) {
      ustring f = get_filter_attr();
      if (f.length() != 0) {
        tool::charset<wchar, wchar('~'), wchar(0)> cset;
        const wchar *                              pc = f.c_str();
        cset.parse(pc);
        for (int i = 0; i < txt.size();) {
          wchar c = txt[i];
          if ((c == '\r' || c == '\n') && is_multiline())
            ++i;
          else if (c >= ' ' && cset.valid(c))
            ++i;
          else
            txt.remove(i);
        }
      } else {
        for (int i = 0; i < txt.size();) {
          wchar c = txt[i];
          if ((c == '\r' || c == '\n') && is_multiline())
            ++i;
          else if (c >= ' ')
            ++i;
          else if (c == '\t') {
            ++i; // assert(false); txt.remove(i);
          } else
            txt.remove(i);
        }
      }
      return txt.length() > 0;
    }

    bool textarea_ctl::notify_changed(view &v, uint reason) {
      self->get_style(v);

      if (_to_undo && reason != CHANGE_BY_UNDO_REDO) _to_undo->endpos = caret;

      check_empty(v, self);

      event_behavior evt(self, self, EDIT_VALUE_CHANGED, reason);
      v.post_behavior_event(evt,true);

      return true;
    }

    bool textarea_ctl::notify_changing(view &v, uint reason) {
      event_behavior evt(self, self, EDIT_VALUE_CHANGING, reason);
      return v.send_behavior_event(evt);
    }

    bool textarea_ctl::notify_changing(view &v, uint reason,
                                       array<wchar> &text) {
      event_behavior evt(self, self, EDIT_VALUE_CHANGING, reason);
      evt.data = value(text());
      if (v.send_behavior_event(evt)) {
        if (evt.data.is_string()) text = evt.data.get(W(""));
        return true;
      }
      return false;
    }

    bool textarea_ctl::move_caret(view &v, ADVANCE_DIR dir, bool keep_anchor,
                                  uint n) {
      bool     r  = false;
      bookmark bm = caret;
      if (dir == ADVANCE_UP || dir == ADVANCE_DOWN) {
        r = self->advance(v, bm, dir, _caret_pos + self->view_pos(v) - self->scroll_pos());
        int save_x = _caret_pos.x;
        move_caret_to(v, bm, keep_anchor);
        _caret_pos.x = save_x;
      } else {
        self->advance(v, bm, dir);
        r = move_caret_to(v, bm, keep_anchor);
        rect cr;
        if (self->is_visible(v)) {
          get_caret_place(v, self, cr);
          _caret_pos = cr.pointOf(5);
        }
      }
      return r;
    }

    bool textarea_ctl::on(view &v, element *self, event_focus &evt) {
      if (evt.cmd == LOST_FOCUS && evt.target == self) {
        show_caret(v, false);
        v.refresh(self);
        return false;
      } else if (evt.cmd == GOT_FOCUS && evt.target == self) {
        self->check_layout(v);
        if (evt.by_key() && !is_multiline()) {
          move_caret(v, ADVANCE_FIRST, false);
          move_caret(v, ADVANCE_LAST, true);
        } else {
          if (caret.valid())
            show_caret(v, true);
          else
            move_caret(v, ADVANCE_FIRST, false);
        }
        evt.need_ime = need_ime(self);
        v.refresh(self);
        return false;
      }
      /*    if (evt.cmd == FOCUS_OUT && evt.target == self->parent)
          {
            show_caret(v, false);
            v.refresh(self);
            return false;
          }
          else if (evt.cmd == GOT_FOCUS && evt.target == self->parent)
          {
            self->check_layout(v);
            if (caret.valid())
              show_caret(v, true);
            else
              move_caret(v, ADVANCE_FIRST, false);
            evt.need_ime = need_ime(self);
            v.refresh(self);
            return false;
          } */

      return false;
    }

    bool textarea_ctl::perform_drag(view &v, element *self, event_mouse &evt) {
      int s = anchor.linear_pos();
      int e = caret.linear_pos();
      if (s > e) swap(s, e);
      wchars                  txt = edit_buffer(v, self)(s, e);
      handle<clipboard::data> cd  = new clipboard::data();
      cd->add(new html::clipboard::text_item(txt));

      rect vbox = self->content_box(v, element::TO_VIEW);
      rect vsrc = html::rbox(v, anchor, caret);
      assert(!vsrc.empty());
      if (vsrc.empty()) return false;

      handle<bitmap> drag_image = new bitmap(vsrc.size(), true, false);
      if (drag_image) {
        handle<graphics> sfi =
            v.app->create_bitmap_bits_graphics(drag_image, argb(0, 0, 0, 0));
        if (!sfi) return false;
        sfi->offset(-vsrc.s);
        // sfi->set_clip_rc(vsrc);
        auto_state<graphics *>      _1(v.drawing_surface, sfi);
        auto_state<selection_ctx *> _2(v._selection_ctx, this);
        auto_state<tristate_v>      _3(v._draw_selection_only, TRUE);

        point vpt = self->view_pos(v);
        self->draw_content(v, sfi, vpt, false);
        v.commit_drawing_buffers();
      }
      uint ddm = is_readonly(self) ? dd_copy : (dd_move | dd_copy);
      if (v.exec_drag(cd, ddm, self, drag_image, evt.pos_view - vsrc.s) && ddm == dd_move)
        delete_char(v, 0);
      return true;
    }

    bool textarea_ctl::on(view &v, element *self, event_mouse &evt) {
      switch (evt.cmd) {
      case MOUSE_DCLICK:
        if (evt.is_point_button()) {
          v.refresh(self);
          v.set_focus(self, BY_MOUSE);
          if (evt.is_shortcut()) {
            move_caret(v, ADVANCE_HOME, false);
            move_caret(v, ADVANCE_END, true);
          } else {
            move_caret(v, ADVANCE_WORD_START, false);
            move_caret(v, ADVANCE_WORD_END, true);
          }
          return true;
        }
        break;
      case MOUSE_TCLICK:
        if (evt.is_point_button()) {
          v.refresh(self);
          v.set_focus(self, BY_MOUSE);
          move_caret(v, ADVANCE_HOME, false);
          move_caret(v, ADVANCE_END, true);
          return true;
        }
        break;
      case MOUSE_DOWN:
        if (evt.is_point_button()) {
          v.refresh(self);
          v.set_focus(self, BY_MOUSE);
          v.set_capture_strict(self);
          bookmark bm = self->find_text_pos(v, evt.pos);
          _mouse_down_on_selection = bm.valid() && bm.is_between(selection_ctx::normalized());
          if (!_mouse_down_on_selection) 
            move_caret_to(v, evt.pos, evt.is_shift());
          return true;
        } else { // if(!caret.valid())
          bookmark bm = self->find_text_pos(v, evt.pos);
          bool     on_sel =
              bm.valid() && bm.is_between(selection_ctx::normalized());
          if (!on_sel) move_caret_to(v, evt.pos, false);
        }
        break;
      case MOUSE_UP:
        if (evt.is_point_button() && v.mouse_down_element &&
            v.mouse_down_element->belongs_to(self, true)) {
          v.refresh(self);
          v.set_capture(0);

          bookmark bm = self->find_text_pos(v, evt.pos);
          if (_mouse_down_on_selection) move_caret_to(v, evt.pos, false);

          if (self->state.pressed() && self->state.hover()) {
            self->state.pressed(false);
            return true;
          } else
            self->state.pressed(false);

          _mouse_down_on_selection.clear();
        }
        break;

      case MOUSE_DRAG_REQUEST:
        if (_mouse_down_on_selection) {
          //_mouse_down_on_selection = false;
          perform_drag(v, self, evt);
        }
        break;

      case MOUSE_MOVE:
        if (evt.is_point_button()) {
          if (!_mouse_down_on_selection) move_caret_to(v, evt.pos, true);
          return true;
        }
        // evt.cursor_type = self->get_style(v)->cursor;
        break;
      // case MOUSE_IDLE:
      // case MOUSE_CHECK:
      //  evt.cursor_type = self->get_style(v)->cursor;
      //  break;
      case MOUSE_LEAVE: {
        self->state.pressed(false);
      } break;
      }
      return false;
    }

    bool textarea_ctl::on(view &v, element *, event_key &evt) {
      if (!evt.target->belongs_to(self, true) && !self->state.current())
        return false; // that is a key targeted to something else (menu.context
                      // for example).

      // edit_auto_state _(this,b,v);

      if (evt.cmd == KEY_CHAR) {
        if (evt.key_code == 127 /* Ctrl+Backspace! */) return true;
        if (!is_editable(self)) return false;
        if (evt.is_alt()) return false;

        /*if( evt.key_code == '\r' ) - moved to KEY_DOWN handler
        {
          if( check_char(wchar(evt.key_code), true ) &&
        insert_range(v,WCHARS("\n")) ) return true;
        }
        else */

        if (!is_alnum(evt.key_code))
          rq_spell_check(v);
        else if (caret.valid() && caret.marks_contains(CHARS("misspell")))
          rq_spell_check(v);

        if (evt.key_code >= ' ') {
          if (check_char(wchar(evt.key_code), true) &&
              insert_char(v, (wchar)evt.key_code))
            return true;
        }

        return false;
      } else if (evt.cmd == KEY_DOWN) {
        bool isCtl   = (evt.alt_state & ALT_SHORTCUT) != 0;
        bool isShift = (evt.alt_state & ALT_SHIFT) != 0;
        bool isAlt   = (evt.alt_state & ALT_ALT) != 0;
        if (isAlt) return false;

        /*if(handle_key_at(v,caret.tb_pos, evt.key_code, isCtl, isShift))
        {
           //_anchor_pos = _caret_pos;
           //update_caret(v);
           move_caret_to(v,caret,false);
           v.refresh(self);
           notify_changed(v,CHANGE_BY_INS_CHAR);
           return true;
        }*/

        switch (evt.key_code) {
        case KB_LEFT:
          if (!isShift && (caret != anchor))
            return move_caret_to(v, caret < anchor ? caret : anchor, false);
          else
            return move_caret(v, isCtl ? ADVANCE_WORD_START : ADVANCE_LEFT,
                              isShift);
        case KB_RIGHT:
          if (!isShift && (caret != anchor))
            return move_caret_to(v, caret > anchor ? caret : anchor, false);
          else
            return move_caret(v, isCtl ? ADVANCE_WORD_END : ADVANCE_RIGHT,
                              isShift);
        case KB_UP: return move_caret(v, ADVANCE_UP, isShift);
        case KB_DOWN: return move_caret(v, ADVANCE_DOWN, isShift);
        case KB_HOME:
          return move_caret(v, isCtl ? ADVANCE_FIRST : ADVANCE_HOME, isShift);
        case KB_END:
          return move_caret(v, isCtl ? ADVANCE_LAST : ADVANCE_END, isShift);

        case KB_RETURN:
          if (is_editable(self) && check_char(wchar('\r'), true) &&
              insert_range(v, WCHARS("\n")))
            return true;
          break;

          // return move_caret(v,isCtl? ADVANCE_LAST:ADVANCE_END, isShift);

          /*        case KB_BACK:
                    if(isAlt)
                      return is_editable(self) && undo(v);
                    else if( isCtl )
                    {
                      if(move_caret(v,ADVANCE_WORD_START, true))
                        return is_editable(self) && delete_char(v, 0);
                    }
                    return is_editable(self) && delete_char(v, -1);

                  case 'A':
                    if(isCtl)
                    {
                      select_all(v);
                      return true;
                    }
                    break;
                  case KB_DELETE:
                    if(isShift)
                      return is_editable(self) && cut(v);
                    else if( isCtl )
                    {
                      if(move_caret(v,ADVANCE_WORD_END, true))
                        return is_editable(self) && delete_char(v,0);
                    }
                    return is_editable(self) && delete_char(v,1);
                  case KB_INSERT:
                    if(isShift)
                      return is_editable(self) && paste(v);
                    //else if(isCtl)
                    //  return copy(v,self);
                    //break; - fall through
                  case 'C':
                    if(isCtl && copy(v)) return true;
                    break;
                  case 'X':
                    if(isCtl && is_editable(self) && cut(v)) return true;
                    break;
                  case 'V':
                    if(isCtl && is_editable(self) && paste(v)) return true;
                    break;
                  case 'Z':
                    if(isCtl) return is_editable(self) && undo(v);
                    break;
                  case 'Y':
                                                  if(isCtl) return
             is_editable(self) && redo(v); break; */
        case KB_SHIFT:
          if (isCtl && self->get_style(v)->direction.is_defined()) {
            bool to_rtl = (evt.alt_state & ALT_RIGHT_SHIFT) != 0;
            self->atts.set(attr::a_dir, to_rtl ? "rtl" : "ltr");
            self->drop_style(&v);
            // self->commit_measure(v);
            // self->measure_and_refresh(*v);
            // v.refresh(self);
            update(v, self);
            move_caret_to(v, caret, false);
            return true;
          }
          break;
        }
        return false;
      }
      // else if( KEY_RESULT_STRING:KEY_COMP_STRING )
      else if (evt.cmd == KEY_COMP_STRING) {
        if (!is_editable(self) || !need_ime(self)) return true;
        if (evt.ime.composition.length) delete_char(v, 0);

        clear_comp_chars(v);
        if (evt.ime.composition.length)
          insert_ime_range(v, evt.ime.composition, evt.ime.caret_pos);
        v.refresh(self);
        return true;
      } else if (evt.cmd == KEY_RESULT_STRING) {
        if (!is_editable(self) || !need_ime(self)) return true;
        delete_char(v, 0);
        clear_comp_chars(v);
        notify_changing(v, CHANGE_BY_INS_CHARS);
        if (evt.ime.composition.length) insert_range(v, evt.ime.composition);
        // move_caret(v,ADVANCE_RIGHT,false,evt.ime.composition.length);
        notify_changed(v, CHANGE_BY_INS_CHARS);
        v.refresh(self);
        return true;
      }
      return false;
    }

    bool textarea_ctl::on_timer(view &v, element *self, timer_id tid,
                                TIMER_KIND kind) {
      if (kind != INTERNAL_TIMER) return false;
      if (tid == CARET_TIMER_ID) {
        if (needs_show_caret(v,self)) {
          if (caret != anchor || !is_editable(self)) {
            caret_state = 0;
            refresh_caret(v);
            return false;
          } else if (caret_state == 2) {
            caret_state = 1;
            refresh_caret(v);
          } else if (caret_state == 1) {
            caret_state = 2;
            refresh_caret(v);
          }
          return true;
        } else {
          caret_state = 0;
          refresh_caret(v);
          // v.refresh(self);
          return false;
        }
      } else if (tid == SPELL_CHECK_TIMER_ID) {
        spell_check(v);
        return false;
      }
      return false;
    }

    bool can_continue(view &v, textarea_ctl *ta, textarea_ctl::E_OP op,
                      bookmark cp, bookmark cpe = bookmark()) {
      if (!ta->_to_undo) return false;
      if (ta->_to_undo->op != op) return false;
      if (op == textarea_ctl::INSERT_CHARS) {
        bookmark pos = ta->_to_undo->endpos;
        return pos == cp;
      } else if (op == textarea_ctl::REMOVE_CHARS) {
        if (cp == ta->_to_undo->endpos) return true;
        if (cpe == ta->_to_undo->pos) return true;
      }

      return true;
    }

    bool textarea_ctl::insert_chars_at(view &v, bookmark &cp, wchars s,
                                       bool save_state) {
      if (!s)
        return true;

      wchars bf = edit_buffer(v, self)();

      int ins_pos = limit<int>(cp.linear_pos(), 0, int(bf.length));

      edit_buffer(v, self).insert(ins_pos, s.start, s.length);

      if (save_state) {
        if (!can_continue(v, this, INSERT_CHARS, cp)) {
          e_state *ps = new e_state;
          ps->op      = INSERT_CHARS;
          ps->text    = s;
          ps->pos     = cp;
          ps->next    = _to_undo;
          _to_undo    = ps;
          _to_redo    = 0;
        } else {
          _to_undo->text.push(s);
        }
      }

      update(v, self);

      cp.node     = self->nodes.first();
      cp.pos      = ins_pos + s.size() - 1;
      cp.after_it = true;

      return true;
    }

    bookmark textarea_ctl::remove_chars_at(view &v, bookmark start,
                                           bookmark end, bool save_state) {

      bookmark oend = end;

      start.linearize();
      end.linearize();

      if (start > end) swap(start, end);
      if (start.node != end.node) return oend;

      int s = start.pos;
      int e = end.pos;

      array<wchar> &buffer = edit_buffer(v, self);
      s                    = limit(s, 0, buffer.size());
      e                    = limit(e, 0, buffer.size());
      wchars chars         = buffer()(s, e);

      if (chars.length == 0) return oend;

      if (save_state) {
        if (!can_continue(v, this, REMOVE_CHARS, start, end)) {
          e_state *ps = new e_state;
          ps->op      = REMOVE_CHARS;
          ps->text    = chars;
          ps->pos     = start;
          ps->next    = _to_undo;
          _to_undo    = ps;
          _to_redo    = 0;
        } else {
          if (start == _to_undo->endpos) {
            _to_undo->text.push(chars);
            _to_undo->endpos = end;
          } else if (end == _to_undo->pos) {
            _to_undo->text.insert(0, chars.start, chars.length);
            _to_undo->pos = start;
          }
        }
      }

      buffer.remove(s, e - s);
      update(v, self);

      rq_spell_check(v);

      start.normalize();
      return start;
    }

    struct transaction // folds recent operations
    {
      handle<textarea_ctl::e_state> top;
      textarea_ctl *                pta;
      transaction(textarea_ctl *ta) {
        pta = ta;
        top = pta->_to_undo;
      }
      ~transaction() {
        if (!pta->_to_undo) return;
        if (!pta->_to_undo->next) return;
        if (pta->_to_undo->next->next != top) return;
        handle<textarea_ctl::e_state> t  = pta->_to_undo;
        handle<textarea_ctl::e_state> td = t->next;
        assert(td->op == textarea_ctl::REMOVE_CHARS);
        t->next    = top;
        t->initial = td;
        td->next   = 0;
      }
    };

    bool textarea_ctl::insert_char(view &v, wchar c) {
      if (is_readonly(self)) return false;

      clear_comp_chars(v);

      transaction _(this);

      delete_char(v, 0);

      int ml = maxlength();

      if (ml && int(chars().length) >= ml || !caret.valid()) {
        // MessageBeep(MB_ICONEXCLAMATION);
        beep();
        return true;
      }

      array<wchar> t;
      t.push(c);

      notify_changing(v, CHANGE_BY_INS_CHAR, t);
      if (!self)
        return false;

      bookmark at =
          caret.node->node_index == 0 ? caret : self->first_node()->end_pos();

      if (!is_valid_char_at(c, at.pos) || !insert_chars_at(v, at, t())) {
        beep();
        return false;
      }
      anchor = caret = at;

      notify_changed(v, CHANGE_BY_INS_CHAR);
      if (!self)
        return false;

      // self->commit_measure(v);
      update_caret(v);
      return true;
    }

    array<wchar> &textarea_ctl::edit_buffer(view &v, element *self) {
#if 1
      // we need ghost "\n" node at the end to properly represent CRLF typed at
      // the end
      if (self->nodes.size() != 2 || !self->nodes[0]->is_text() ||
          !self->nodes[1]->is_text()) {
        //bool ldata_valid = self->is_layout_valid();
        // self->clear(v);
        array<wchar> t;
        self->emit_text(t);
        for (helement c = self->first_element(); c; c = c->next_element())
          c->stray(v);
        self->nodes.clear();
        self->append(new text(t()));
        self->append(new text(WCHARS("\n")));
        self->drop_style(&v);
        self->drop_layout(&v);
        self->setup_layout(v);
      }
#else
      if (self->nodes.size() == 0 || !self->nodes[0]->is_text()) {
        bool         ldata_valid = self->is_layout_valid();
        array<wchar> t;
        self->emit_text(t);
        for (helement c = self->first_element(); c; c = c->next_element())
          c->stray(v);
        self->nodes.clear();
        self->append(new text(t()));
        if (ldata_valid) {
          self->drop_layout(&v);
          self->setup_layout(v);
        }
      }
#endif
      return self->nodes[0]->cast<text>()->chars;
    }

    void textarea_ctl::update(view &v, element *self) {
      hstyle cs = self->get_style(v);
      if (cs->width_depends_on_intrinsic() ||
          cs->height_depends_on_intrinsic()) {
        self->drop_layout(&v);
        self->check_layout(v);
        check_empty(v, self);
        v.add_to_update(self, CHANGES_DIMENSION);
      } else {
        v.refresh(self);
        self->drop_layout(&v);
        self->check_layout(v);
        check_empty(v, self);
        self->commit_measure(v);
      }
      self->check_layout(v);
    }

    bool textarea_ctl::set_value(view &v, element *self, const value &val) {
      ustring s;
      if (val.is_defined() && !val.is_null()) 
        s = val.to_string();
      return set_text(v, self, s());
    }
    bool textarea_ctl::get_value(view &v, element *self, value &val) {
      ustring s;
      get_text(v, self,s);
      val = value(s);
      return true;
    }

    bool textarea_ctl::get_text(view &v, element *self, ustring &val) {
      val = edit_buffer(v, self)();
      return true;
    }

    bool textarea_ctl::set_text(view &v, element *self, wchars val) { 

      if (edit_buffer(v, self)() == val)
        return true;

      array<wchar> t = val;
      check_chars(t);

      clear_comp_chars(v);
      _to_undo = 0;
      _to_redo = 0;

      int ml = maxlength();
      if (ml && t.size() >= ml) t.size(ml);

      edit_buffer(v, self) = t;

      update(v, self);
      spell_check(v);

      caret = anchor = self->first_node()->start_pos();
      self->scroll_pos(v, point(0, 0));

      return true;
    }

    bool textarea_ctl::insert_range(view &v, wchars s, bookmark at,
                                    bool delete_selection) {
      // edit_auto_state _(this,b,v);
      if (s.length == 0) return false;
      if (!caret.valid() && !at.valid()) return false;

      clear_comp_chars(v);

      transaction _(this);

      if (delete_selection && at.valid()) {
        bookmark start = caret;
        bookmark end   = anchor;
        start.linearize();
        end.linearize();
        if (start > end) swap(start, end);
        if (at > start) at.pos = at.pos - (end.pos - start.pos);
      }
      if (delete_selection) delete_char(v, 0);

      array<wchar> t = s;
      if (!check_chars(t)) return true;

      int  ml         = maxlength();
      auto this_chars = chars();

      if (ml && (this_chars.size() + t.size()) > ml) {
        t.size(max(0, ml - this_chars.size()));
        beep();
        if (t.length() == 0) return true;
      }

      notify_changing(v, CHANGE_BY_INS_CHAR, t);

      bookmark pos = at.valid() ? at : caret;

      if (!insert_chars_at(v, pos, t())) {
        beep();
        return false;
      }
      notify_changed(v, CHANGE_BY_INS_CHAR);
      move_caret_to(v, pos, false);

      return true;
    }

    bool textarea_ctl::insert_ime_range(view &v, wchars s, int_v caret_pos) {
      // edit_auto_state _(this,b,v);
      if (s.length == 0 || !caret.valid()) return false;

      clear_comp_chars(v);

      transaction _(this);

      delete_char(v, 0);

      bookmark pos;
      ime_start = ime_end = caret;
      bool is_empty       = false;
      this->is_empty(self, is_empty);

      if (!insert_chars_at(v, ime_end, s, false)) {
        beep();
        return false;
      }

      if (caret_pos.is_defined()) {
        int n = caret_pos;
        if (is_empty)
          ime_start = pos = bookmark(self->nodes.first(), 0, false);
        else
          pos = ime_start;
        while (n--)
          self->advance(v, pos, ADVANCE_NEXT);
      } else
        pos = ime_end;

      move_caret_to(v, pos, false);
      // move_caret_to(v,ime_start,false);

      return true;
    }

    bool textarea_ctl::delete_char(view &v, int dir) {
      if (is_readonly(self)) return false;

      if (selection_exists()) {
        notify_changing(v, CHANGE_BY_DEL_CHARS);
        // int pos = selection_start();
        // int end = selection_end();
        //_anchor_pos = _caret_pos = delete_range(v,pos,end);
        caret = anchor = delete_range(v, caret, anchor);
        notify_changed(v, CHANGE_BY_DEL_CHARS);
      } else if (dir == 0)
        return false;
      else {
        if (dir < 0) {
          // if(caret.tb_pos == 0) return false;
          bookmark pbm = caret;
          if (!move_caret(v, ADVANCE_LEFT, false) || !caret.valid()) {
            caret = anchor = pbm;
            return false;
          }
          // if(_caret_pos >= int(chars().length)) return false;
          notify_changing(v, CHANGE_BY_DEL_CHAR);
          caret = anchor = delete_range(v, pbm, caret);
        } else {
          bookmark pbm = caret;
          if (!move_caret(v, ADVANCE_RIGHT, false) || !caret.valid()) {
            caret = anchor = pbm;
            return false;
          }
          notify_changing(v, CHANGE_BY_DEL_CHAR);
          caret = anchor = delete_range(v, pbm, caret);
        }
        notify_changed(v, CHANGE_BY_DEL_CHAR);
      }
      v.refresh(self);
      rq_spell_check(v);
      self->drop_layout(&v);
      self->check_layout(v);
      self->commit_measure(v);
      update_caret(v);
      return true;
    }

    bool textarea_ctl::clear_comp_chars(view &v) {
      if (is_readonly(self)) return false;

      if (ime_start.valid() && ime_end.valid()) {
        caret = anchor = remove_chars_at(v, ime_start, ime_end, false);
        // this->self->ldata->drop();
        // this->self->init(v,this->self->nodes());
        ime_start = ime_end = bookmark();
        v.refresh(self);
        update_caret(v);
      }
      return true;
    }

    bool textarea_ctl::get_caret_place(view &v, element *self, rect &rc) {
      if (caret.valid()) {

        // get_caret_place
        // self->get_caret_rect(v,rf, this);
        caret_metrics gm;
        if (caret.get_caret_metrics(v, gm)) {
          rc = gm.caret_v_bar() + gm.elem->rel_pos(v, self);
          return true;
          // rc.s.x
        }
      }
      return false;
    }

    bool textarea_ctl::draw_caret(view &v, graphics *pg,
                                  const caret_metrics &cm) {
      /*if(!cm.glyph_rc.empty())
      {
        argb caret_tail_color = selection_back_color(v);
        if( caret_tail_color.alfa ) {
          argb tail_color = selection_back_color(v);
          tail_color.alfa = 128;
          pg->fill(tail_color, cm.glyph_rc);
        }
      }*/
      if (caret_state == 2 || target.valid()) {
        argb caret_color = selection_caret_color(v);
        pg->fill(caret_color, cm.caret_v_bar());
      }
      return true;
    }

    void textarea_ctl::update_caret(view &v) {
      if (self->is_visible(v)) {
        scroll_to_view(v);
        show_caret(v, true);
        rect cr;
        get_caret_place(v, self, cr);
        _caret_pos = cr.pointOf(5);
      }
    }

    void textarea_ctl::show_caret(view &v, bool onOff) {
      if (onOff) {
        if (needs_show_caret(v,self)) {
          caret_state = 2;
          v.start_timer(self, CARET_TIMER_MS, CARET_TIMER_ID, INTERNAL_TIMER);
        } else
          caret_state = 1;
      } else {
        caret_state = 0;
        v.stop_timer(self, CARET_TIMER_ID, INTERNAL_TIMER);
      }
      refresh_caret(v);
    }

    void textarea_ctl::refresh_caret(view &v) {
      v.refresh(self);
      // rect r = get_caret_place(v);
      // v.refresh(self,r - self->scroll_pos());
    }

    void textarea_ctl::scroll_to_view(view &v) {
      // rect place = edit_place();
      // if(place.empty())
      //  return;

      rect r;
      if (!get_caret_place(v, self, r)) return;

      if (r.s.x > r.e.x) swap(r.s.x, r.e.x);

      point sp  = self->scroll_pos();
      point nsp = sp;

      r -= sp;

      rect cr = self->client_rect(v);
      //r += cr.s;

      if (r.left() < cr.left())
        nsp.x -= cr.left() - r.left() + 1;
      else if (r.right() > cr.right())
        nsp.x += r.right() - cr.right() + 1;

      if(self->ldata->dim_min.x <= cr.width())
        nsp.x = 0;

      int wc = self->ldata->dim_min.x + r.width();

      if (wc > cr.width()) nsp.x = limit<int>(nsp.x, 0, wc - cr.width() + 1);

      if (r.top() <= cr.top() && r.bottom() >= cr.bottom())
        nsp.y = sp.y; // caret is taller than client rect, don't change position
      else if (r.top() < cr.top())
        nsp.y -= cr.top() - r.top() + 1;
      else if (r.bottom() > cr.bottom())
        nsp.y += r.bottom() - cr.bottom() + 1;

      nsp.y = limit<int>(nsp.y, 0, self->ldata->dim_min.y - cr.height() + 1);

      if (sp != nsp) {
        self->scroll_pos(v, nsp, true);
        v.refresh(self);
      }
    }

    bool textarea_ctl::move_caret_to(view &v, point pt, bool keep_anchor) {
#if _DEBUG
    // if(pt.x < 0)
    //  pt.x = pt.x;
#endif
      bookmark bm = self->find_text_pos(v, pt);
      return move_caret_to(v, bm, keep_anchor);
    }

    bool textarea_ctl::move_caret_to(view &v, bookmark pos, bool keep_anchor) {
      refresh_caret(v);
      if (!pos.valid())
        ; //self->advance(v, pos, ADVANCE_FIRST);
      else {
        if (!pos.node->belongs_to(self)) return false;
        if (pos.node->node_index != 0 && self->nodes.size())
          pos = self->nodes[0]->end_caret_pos(v);
        pos.normalize();
      }
      caret = pos;
      if (!keep_anchor) anchor = caret;
      update_caret(v);
      return true;
    }

    bool textarea_ctl::move_target_to(view &v, bookmark pos) {
      refresh_caret(v);
      if (pos.valid()) pos.normalize();
      target = pos;
      refresh_caret(v);
      return true;
    }

    bool textarea_ctl::cut(view &v) {
      if (!is_readonly(self) && copy(v) && can_cut(v)) {
        // edit_auto_state _(this,b,v);
        delete_char(v, 0);
        return true;
      }
      return false;
    }

    bool textarea_ctl::copy(view &v) {
      if (!caret.valid() || caret == anchor || !can_copy(v)) return false;
      // clipboard::set(v,anchor,caret);
      int s = anchor.linear_pos();
      int e = caret.linear_pos();
      if (s > e) swap(s, e);
      clipboard::empty();
      clipboard::set_text(edit_buffer(v, self)(s, e));
      return true;
    }

    bool textarea_ctl::paste(view &v) {
      ustring us;
      if (clipboard::get(us) && can_paste(v)) return paste(v, us);
      return false;
    }

    bool textarea_ctl::paste(view &v, const ustring &s) {
      if (insert_range(v, s)) {
        rq_spell_check(v);
        return true;
      }
      return false;
    }

    bool textarea_ctl::select_all(view &v) {
      move_caret(v, ADVANCE_LAST, false);
      move_caret(v, ADVANCE_FIRST, true);
      // if( !caret.valid() || caret == anchor )
      //  return false;
      // clipboard::set(v,anchor,caret);
      // return true;
      return true;
    }

    bool textarea_ctl::restore_state(view &v, textarea_ctl::e_state *ps) {
      if (ps->op == INSERT_CHARS) {
        bookmark start = ps->pos;
        bookmark end   = start;
        end.pos        = end.pos + ps->text.size();
        remove_chars_at(v, start, end, false);
        notify_changed(v, CHANGE_BY_UNDO_REDO);
        move_caret_to(v, start, false);
        return true;
      } else if (ps->op == REMOVE_CHARS) {
        bookmark pos  = ps->endpos;
        wchars   text = ps->text();
        move_caret_to(v, pos, false);
        insert_chars_at(v, pos, text, false);
        notify_changed(v, CHANGE_BY_UNDO_REDO);
        move_caret_to(v, pos, true);
        return true;
      }
      return false;
    }

    bool textarea_ctl::undo(view &v) {
      if (!_to_undo) return false;

      restore_state(v, _to_undo);
      if (_to_undo->initial) restore_state(v, _to_undo->initial);

      handle<e_state> n = _to_undo->next;
      _to_undo          = n;

      update_caret(v);
      return true;
    }
    bool textarea_ctl::can_undo(view &v) { return !!_to_undo; }
    bool textarea_ctl::redo(view &v) {
      assert(NOT_YET);
      return true;
    }
    bool textarea_ctl::can_redo(view &v) { return !!_to_redo; }

    void textarea_ctl::clear(view &v) { assert(NOT_YET); }

    bool textarea_ctl::on(view &v, element *self, event_behavior &evt) {
      if (evt.cmd == INPUT_LANGUAGE_CHANGED) {
        handle<spell_checker> sc = app()->get_spell_checker(v.get_input_lang());
        if (sc) {
          _spell_checker = sc;
          spell_check(v);
        }
      } else if (evt.cmd == CONTEXT_MENU_SETUP) {

        if (evt.data.get_prop("x").is_undefined()) {
          // seems like context menu is invoked by kbd
          rect loc;
          if (get_caret_place(v, self, loc)) {
            loc += self->view_pos(v);
            evt.data.set_prop("x", value(loc.e.x));
            evt.data.set_prop("y", value(loc.e.y));
          }
        }

        if (_spell_checker) {
          // bool get_marks_span(bookmark& start, bookmark& end, chars mark)
          bookmark start = caret;
          bookmark end;
          ustring  misspelled_word;
          if (get_marks_span(start, end, misspelled_word, CHARS("misspell"))) {
            this->select(v, end, start);
            auto list = _spell_checker->get_suggestions_for(misspelled_word());
            if (list.size()) {
              handle<element> menu = evt.source;
              menu->insert(0, new element(tag::T_HR));
              for (int n = list.last_index(); n >= 0; --n) {
                handle<element> repl = new element(tag::T_LI);
                repl->append(new text(list[n]()));
                repl->atts.set(attr::a_class, WCHARS("replacement"));
                menu->insert(0, repl);
                // dbg_printf("repl: %S\n", list[i].c_str());
              }
            }
          }
        }
      } else if (evt.cmd == MENU_ITEM_CLICK &&
                 evt.target->attr_class() == WCHARS("replacement") &&
                 this->can_paste(v)) {
        this->paste(v, evt.target->first_node()->cast<text>()->chars());
        return true;
      }
      return super::on(v, self, evt);
    }

    bool textarea_ctl::on(view &v, element *self, event_exchange &evt) {
      if (!is_editable(self)) return false;

      switch (evt.cmd) {
      case X_DRAG: {
        bookmark bm = self->find_text_pos(v, evt.pos);
        if (evt.dd_source == self) {
          auto selection = selection_ctx::normalized();
          selection.first.linearize();
          selection.second.linearize();
          // selection.first.advance_caret_pos(v,html::ADVANCE_PREV);
          // selection.second.advance_caret_pos(v,html::ADVANCE_NEXT);
          if (bm.is_between(selection.first, selection.second)) return false;
        }
        move_target_to(v, bm);
        v.refresh(self);
      }
        return true;
      case X_DRAG_ENTER: return true;
      case X_DRAG_LEAVE: move_target_to(v, bookmark()); return true;
      case X_DROP: {
        handle<clipboard::text_item> pti =
            evt.data->get<clipboard::text_item>();
        if (pti) {

          bookmark bm = self->find_text_pos(v, evt.pos);
          if ((evt.dd_source == self) && bm.is_between(caret, anchor))
            return false;
          insert_range(v, pti->val, bm,
                       evt.dd_source == self && evt.dd_modes == dd_move);
          v.set_focus(self, BY_MOUSE);
        }
        move_target_to(v, bookmark());
      }
        return true;
      }
      return false;
    }

    void enable_mi(view &v, element *m, wchars selector, bool onoff) {
      element *mi = find_first(v, m, selector);
      if (mi) {
        mi->state.disabled(!onoff);
        mi->flags.state_initialized = 1;
        mi->drop_styles(v);
      }
    }

    /*bool textarea_ctl::setup_context_menu(view& v, element* self,
    event_behavior& evt)
    {

      helement m = evt.source;

      if(!m) return false;
      bool editable = is_editable(self);

      enable_mi(v,m, WCHARS("[command='edit:cut']"), can_cut(v));
      enable_mi(v,m, WCHARS("[command='edit:copy']"), can_copy(v));
      enable_mi(v,m, WCHARS("[command='edit:paste']"), can_paste(v) &&
    (clipboard::available_formats() & clipboard::cf_text) != 0); enable_mi(v,m,
    WCHARS("[command='edit:selectall']"), self->get_text(v).length != 0);
      enable_mi(v,m, WCHARS("[command='edit:undo']"), editable && can_undo(v));
      enable_mi(v,m, WCHARS("[command='edit:redo']"), editable && can_redo(v));

      return true;
    }*/

    bool textarea_ctl::on(view &v, element *self, event_command &evt) {
      if (!evt.doit() && !evt.checkit()) return false;

      ustring cmd = evt.command;
      if (cmd == event_command::EDIT_CUT()) {
        if (evt.doit()) return cut(v);
        evt.result = can_cut(v) ? CMD_AVAILABLE : CMD_DISABLED;
        return true;
      } else if (cmd == event_command::EDIT_COPY()) {
        if (evt.doit()) return copy(v);
        evt.result = can_copy(v) ? CMD_AVAILABLE : CMD_DISABLED;
        return true;
      } else if (cmd == event_command::EDIT_PASTE()) {
        if (evt.doit()) {
          if(evt.data.is_string())
            return paste(v,evt.data.get_string());
          else 
            return paste(v);
        }
        evt.result = (can_paste(v) && (clipboard::available_formats() &
                                       clipboard::cf_text) != 0)
                         ? CMD_AVAILABLE
                         : CMD_DISABLED;
        return true;
      } else if (cmd == event_command::EDIT_SELECT_ALL()) {
        if (evt.doit()) return select_all(v);
        ustring us; get_text(v, self, us);
        evt.result = us.length() > 0 ? CMD_AVAILABLE : CMD_DISABLED;
        return true;
      } else if (cmd == event_command::EDIT_UNDO()) {
        if (evt.doit()) return undo(v);
        evt.result =
            is_editable(self) && can_undo(v) ? CMD_AVAILABLE : CMD_DISABLED;
        return true;
      } else if (cmd == event_command::EDIT_REDO()) {
        if (evt.doit()) return redo(v);
        evt.result =
            is_editable(self) && can_redo(v) ? CMD_AVAILABLE : CMD_DISABLED;
        return true;
      } else if (cmd == event_command::EDIT_DELETE_WORD_PREV()) {
        if (!evt.doit()) return true;
        if (move_caret(v, ADVANCE_WORD_START, true))
          return is_editable(self) && delete_char(v, 0);
        return false;
      } else if (cmd == event_command::EDIT_DELETE_PREV()) {
        if (!evt.doit()) return true;
        return is_editable(self) && delete_char(v, -1);
      } else if (cmd == event_command::EDIT_DELETE_WORD_NEXT()) {
        if (!evt.doit()) return true;
        if (move_caret(v, ADVANCE_WORD_END, true))
          return is_editable(self) && delete_char(v, 0);
        return false;
      } else if (cmd == event_command::EDIT_DELETE_NEXT()) {
        if (!evt.doit()) return true;
        return is_editable(self) && delete_char(v, 1);
      }
      return false;
    }

    bool textarea_ctl::select(view &v, bookmark c, bookmark a) {
      if (a.valid() && c != a) {
        move_caret_to(v, a, false);
        move_caret_to(v, c, true);
      } else
        move_caret_to(v, c, false);
      return true;
    }

    bool textarea_ctl::select(view &v, int start, int end) {
      int length = chars().size();

      if (start < 0 || end < 0) {
        self->scroll_pos(v, point());
        return select(v, bookmark(), bookmark());
      }

      bookmark bm_start(text_node(), 0, false);
      bookmark bm_end(bm_start);
      bm_start.pos = limit(start, 0, length);
      bm_end.pos   = limit(end, 0, length);

      return select(v, bm_end, bm_start);
    }

    bool textarea_ctl::rq_spell_check(view &v) {

      if (can_spell_check(v))
        v.start_timer(self, SPELL_CHECK_TIMER_MS, SPELL_CHECK_TIMER_ID,
                      INTERNAL_TIMER);
      return true;
    }

    bool textarea_ctl::can_spell_check(view &v) {
      if (!self) return false;
      string dis = self->atts.get_string(attr::a_spellcheck);
      if (dis == CHARS("disable") || dis == CHARS("no") ||
          dis == CHARS("false"))
        return false;
      return true;
    }

    bool textarea_ctl::spell_check(view &v) {
      if (!can_spell_check(v)) 
        return false;
      if (!_spell_checker) {
        _spell_checker = app()->get_spell_checker(v.get_input_lang());
        if (!_spell_checker) return false;
      }
      self->check_spelling(v, _spell_checker);
      return false;
    }

    bool textarea_ctl::do_selectAll()       { return select_all(*self->pview()); }

    int     textarea_ctl::get_selectionStart() { return (int)selection_start(); }
    int     textarea_ctl::get_selectionEnd()   { return (int)selection_end(); }
    bool    textarea_ctl::do_selectRange(int_v start, int_v end) { return select(*self->pview(), start.val(-1), end.val(-1)); }
    ustring textarea_ctl::get_selectionText()  { int start = selection_start(); int end = selection_end(); return ustring(chars()(start, end)); }
    bool    textarea_ctl::do_insertText(ustring text) { return paste(*self->pview(), text); return true; }
    bool    textarea_ctl::do_appendText(ustring text) { int last = chars().size(); select(*self->pview(), last, last); paste(*self->pview(), text); return true; }
    bool    textarea_ctl::do_removeText() { return delete_char(*self->pview(), 0); }
               
    bool textarea_ctl::on_x_method_call(view &v, element *self,
                                        const char *name, const value *argv,
                                        size_t argc, value &retval) {
      // edit_auto_state _(this,self,v);
      tool::chars fname = chars_of(name);
#define ACTION(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
#define IFDO() if (argv[0].get(false))

      ACTION(1, undo) {
        IFDO() {
          bool r = undo(v);
          retval = value(r);
          v.set_focus(self, BY_CODE);
        }
        else {
          bool r = is_editable(self) && can_undo(v);
          retval = value(r);
        }
        return true;
      }
      ACTION(0, canUndo) {
        bool r = is_editable(self) && can_undo(v);
        retval = value(r);
        return true;
      }
      ACTION(0, doUndo) {
        bool r = undo(v);
        retval = value(r);
        v.set_focus(self, BY_CODE);
        return true;
      }
      ACTION(1, redo) {
        IFDO() {
          bool r = redo(v);
          retval = value(r);
          v.set_focus(self, BY_CODE);
        }
        else {
          bool r = is_editable(self) && can_redo(v);
          retval = value(r);
        }
        return true;
      }
      ACTION(0, canRedo) {
        bool r = is_editable(self) && can_redo(v);
        retval = value(r);
        return true;
      }
      ACTION(0, doRedo) {
        bool r = redo(v);
        retval = value(r);
        return true;
      }

      ACTION(1, cut) {
        IFDO() {
          bool r = cut(v);
          retval = value(r);
          v.set_focus(self, BY_CODE);
        }
        else {
          bool r = is_editable(self) && selection_exists() && can_cut(v);
          retval = value(r);
        }
        return true;
      }

      ACTION(0, canCut) {
        bool r = is_editable(self) && selection_exists() && can_copy(v);
        retval = value(r);
        return true;
      }
      ACTION(0, doCut) {
        bool r = cut(v);
        retval = value(r);
        return true;
      }

      ACTION(1, copy) {
        IFDO() {
          bool r = copy(v);
          retval = value(r);
          v.set_focus(self, BY_CODE);
        }
        else {
          bool r = selection_exists() && copy_allowed();
          retval = value(r);
        }
        return true;
      }

      ACTION(0, canCopy) {
        bool r = selection_exists() && copy_allowed();
        retval = value(r);
        return true;
      }
      ACTION(0, doCopy) {
        bool r = copy(v);
        retval = value(r);
        return true;
      }

      ACTION(1, paste) {
        IFDO() {
          bool r = paste(v);
          retval = value(r);
          v.set_focus(self, BY_CODE);
        }
        else {
          bool r = is_editable(self) &&
                   (clipboard::available_formats() & clipboard::cf_text) != 0;
          retval = value(r);
        }
        return true;
      }
      ACTION(0, canPaste) {
        bool r = is_editable(self) &&
                 (clipboard::available_formats() & clipboard::cf_text) != 0;
        retval = value(r);
        return true;
      }
      ACTION(0, doPaste) {
        bool r = paste(v);
        retval = value(r);
        return true;
      }

      ACTION(1, selectAll) {
        IFDO()
        return select_all(v);
        else {
          bool r = chars().length != 0;
          retval = value(r);
        }
        return true;
      }
      ACTION(0, canSelectAll) {
        bool r = chars().length != 0;
        retval = value(r);
        return true;
      }
      ACTION(0, doSelectAll) {
        bool r = select_all(v);
        retval = value(r);
        return true;
      }

      ACTION(0, selectionStart) {
        retval = value((int)selection_start());
        return true;
      }
      ACTION(0, selectionEnd) {
        retval = value((int)selection_end());
        return true;
      }
      ACTION(2, setSelection) {
        return select(v, argv[0].get(0), argv[1].get(0));
      }
      ACTION(0, clearSelection) { return select(v, bookmark(), bookmark()); }

      ACTION(0, selectionText) {
        int start = selection_start();
        int end   = selection_end();
        retval    = ustring(chars()(start, end));
        return true;
      }
      ACTION(1, insertText) {
        ustring us = argv[0].get(W(""));
        paste(v, us);
        return true;
      }
      ACTION(1, appendText) {
        ustring us = argv[0].get(W(""));
        // move_caret_to(v,self,mutable_text(self).size(),false);
        int last = chars().size();
        select(v, last, last);
        paste(v, us);
        return true;
      }
#undef ACTION
      return ctl::on_x_method_call(v, self, name, argv, argc, retval);
    }

    ctl *textarea_ctl_factory::create(element *el) {
      // if( el->layout_type() == flow_text )
      return new textarea_ctl();
      // return 0;
    }

    void init_edit_ctl();

    void init_edits() {
      ctl_factory::add(_textarea_ctl = new textarea_ctl_factory());
      init_edit_ctl();
    }

  } // namespace behavior
} // namespace html
