#include "html.h"

namespace html {
  namespace behavior {

    struct swipe_touch_ctl_factory : ctl_factory {
      swipe_touch_ctl_factory() : ctl_factory("swipe-touch") {}
      virtual ctl *create(element *el);
    };

    struct swipe_ctl_factory : ctl_factory {
      swipe_ctl_factory() : ctl_factory("swipe") {}
      virtual ctl *create(element *el);
    };

    static swipe_touch_ctl_factory *_swipe_touch_ctl_factory = 0;
    static swipe_ctl_factory *      _swipe_ctl_factory       = 0;

    struct swipe_touch_ctl : ctl {
      bool  swipe_tracking;
      bool  scroll_tracking;
      bool  got_swipe;
      bool  touch_device; // true if touch device is seen
      point prev_pos;
      uint  prev_time;
      struct stroke_item {
        size delta_pos;
        uint delta_time;
      };
      circular_buffer<stroke_item> strokes;
      helement                     started_in;

      handle<scroll_animator> scroller;

      swipe_touch_ctl()
          : swipe_tracking(false), scroll_tracking(false), got_swipe(false),
            touch_device(false), prev_time(0), strokes(16) {}

      virtual CTL_TYPE get_type() { return CTL_LIST; /*???*/ }

      virtual const string &behavior_name() const {
        return _swipe_touch_ctl_factory->name;
      }

      //? virtual bool focusable(const element* self) { return
      //!self->attr_disabled(); }

      virtual void detach(view &v, element *self) {}
      virtual bool attach(view &v, element *self) { return true; }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }

      void push_stroke(view &v, point at) {
        stroke_item si;
        uint        tm = v.get_ticks();
        si.delta_pos   = prev_pos - at;
        prev_pos       = at;
        si.delta_time  = tm - prev_time;
        prev_time      = tm;
        strokes.push_front(si);
        // dbg_printf("st %d,%d %d\n",si.delta_pos.x,si.delta_pos.y,
        // si.delta_time);
      }
      void push_stroke(view &v, size dp, uint dt) {
        stroke_item si;
        si.delta_pos  = dp;
        si.delta_time = dt;
        strokes.push_front(si);
      }

      bool check_scroll_gesture(view &v, element *self, bool scrollable_h,
                                bool scrollable_v) {
        if (strokes.size() == 0) 
          return 0;
        point sp              = self->scroll_pos();
        point spold           = sp;
        scroll_tracking       = false;
        const stroke_item &si = strokes[0];
        if (scrollable_h && abs(si.delta_pos.x) > 0 &&
            abs(si.delta_pos.x) > 3 * abs(si.delta_pos.y))
          sp.x += si.delta_pos.x;
        if (scrollable_v && abs(si.delta_pos.y) > 0 &&
            abs(si.delta_pos.y) > 3 * abs(si.delta_pos.x))
          sp.y += si.delta_pos.y;
        if (spold == sp) return false;
        v.scroll_window(sp, self, SCROLL);
        v.update_element(self);
        scroll_tracking = true;
        // self->scroll_pos(v,sp,false);
        return true;
      }

      bool check_swipe_gesture(view &v, element *self, bool scrollable_h,
                               bool scrollable_v) {
        if (!scrollable_h && strokes.size() > 3) // try horizontal swipe
        {
          bool hstroke  = false;
          int  distance = 0;
          for (int n = 0; n < strokes.size(); ++n) {
            const stroke_item &si = strokes[n];
            if (si.delta_time > 800) {
              hstroke = false;
              break;
            }
            if (abs(si.delta_pos.x) > 3 * abs(si.delta_pos.y))
              hstroke = true;
            else {
              hstroke = false;
              break;
            }
            distance -= si.delta_pos.x;
          }
          if (hstroke) {
            event_behavior eb(started_in, self, SWIPE, distance > 0 ? 6 : 4);
            v.post_behavior_event(eb, true);
            got_swipe = true;
            return true;
          }
        }
        if (!scrollable_v && strokes.size() > 3) // try vertical swipe
        {
          bool vstroke  = false;
          int  distance = 0;
          for (int n = 0; n < strokes.size(); ++n) {
            const stroke_item &si = strokes[n];
            if (si.delta_time > 200) {
              vstroke = false;
              break;
            }
            if (abs(si.delta_pos.y) > 3 * abs(si.delta_pos.x))
              vstroke = true;
            else {
              vstroke = false;
              break;
            }
            distance -= si.delta_pos.y;
          }
          if (vstroke) {
            event_behavior eb(started_in, self, SWIPE, distance > 0 ? 2 : 8);
            v.post_behavior_event(eb, true);
            got_swipe = true;
            return true;
          }
        }
        return false;
      }

      bool check_click_gesture(view &v, element *self, event_mouse &evt) {
        /*size  distance = 0;
        for(int n = 1; n < strokes.size(); ++ n)
        {
          const stroke_item& sip = strokes[n-1];
          const stroke_item& si = strokes[n];
          if( (si.tm - sip.tm) > 200)
          {
            break;
          }
          if( abs(si.pt.y - sip.pt.y) > 3*abs(si.pt.x - sip.pt.x) )
            ;//vstroke = true;
          else
          {
            //vstroke = false;
            break;
          }
          distance += si.pt.y - sip.pt.y;
        }*/
        return false;
      }

      void stop_swipe_tracking(view &v, element *self) {
        swipe_tracking = false;
        started_in     = 0;
        // strokes.clear();
      }

      sizef calc_average_speed(int m = 3) {
        int   n = 0;
        sizef sp;
#if 1
        for (int i = 0; i < strokes.size(); ++i) {
          if (m-- == 0) break;
          const stroke_item &si = strokes[i];
          {
            sizef t = sizef(si.delta_pos) / float(max(1, si.delta_time));
            t.x = limit(t.x, -3.f, 3.f);
            t.y = limit(t.y, -3.f, 3.f);
            sp += t;
            ++n;
          }
        }
        if (n) {
          sp /= float(n);
          return sp;
        }
#else 
        for (int i = 0; i < strokes.size(); ++i) {
          const stroke_item &si = strokes[i];
          {
            sizef t = sizef(si.delta_pos) / float(max(1, si.delta_time));
            t.x     = limit(t.x, -2.f, 2.f);
            t.y     = limit(t.y, -2.f, 2.f);
            if (abs(t.x) > abs(sp.x)) sp.x = t.x;
            if (abs(t.y) > abs(sp.y)) sp.y = t.y;
            ++n;
          }
        }
        return sp;
#endif
        return sizef();
      }

      void stop_scroll_tracking(view &v, element *self) {
        if (scroll_tracking && !got_swipe) {
          point sp        = self->scroll_pos();
          point spold     = sp;
          scroll_tracking = false;

          sizef speed = calc_average_speed(); // pixels per 1 millisecond
          bool  scrollable_h = self->can_scroll_h(v);
          bool  scrollable_v = self->can_scroll_v(v);
          if (!scrollable_h) speed.x = 0;
          if (!scrollable_v) speed.y = 0;

          if (abs(speed.x) > 0.f || abs(speed.y) > 0.f) {
            // v.scroll_window(sp,self,SCROLL_PAN);
            scroller = new scroll_animator(v,point(0, 0), scroll_animator::INERTIAL);
            scroller->speed = speed;
            // psa->prev_clock =
            v.add_animation(self, scroller);
          }
        }
        scroll_tracking = false;
        started_in      = 0;
        // strokes.clear();
      }

      virtual bool on(view &v, element *self, event_gesture &evt) override {
        switch (evt.cmd) {
        case GESTURE_REQUEST:
          evt.flags = GESTURE_FLAG_PAN_VERTICAL | GESTURE_FLAG_PAN_HORIZONTAL |
                      GESTURE_FLAG_TAP1 | GESTURE_FLAG_TAP2 |
                      GESTURE_FLAG_PAN_WITH_GUTTER;
          touch_device = true;
          return true;
        case GESTURE_START:
          prev_pos  = evt.pos;
          prev_time = v.get_ticks();
          v.set_focus(self, BY_MOUSE);
          v.set_capture(self);
          started_in = evt.target;
          strokes.clear();
          return true;
        case GESTURE_PAN:
          if (evt.flags & GESTURE_STATE_BEGIN) {
            swipe_tracking = true;
            got_swipe      = false;
            // push_stroke(v,evt.delta_xy,evt.delta_time);
            // dbg_printf("++st %d,%d %d\n",evt.delta_xy.x,evt.delta_xy.y,
            // evt.delta_time);  push_stroke(v,evt.pos);
          } else if (evt.flags & GESTURE_STATE_END) {
            // push_stroke(v,evt.delta_xy,evt.delta_time);
            stop_swipe_tracking(v, self);
            stop_scroll_tracking(v, self);
          } else {
            // push_stroke(v,evt.delta_xy,evt.delta_time);
            push_stroke(v, evt.pos);
            bool scrollable_h = self->can_scroll_h(v);
            bool scrollable_v = self->can_scroll_v(v);
            if (swipe_tracking) {
              if (check_swipe_gesture(v, self, scrollable_h, scrollable_v))
                stop_swipe_tracking(v, self);
              // return true;
            }
            check_scroll_gesture(v, self, scrollable_h, scrollable_v);
          }

          // dbg_printf("PAN %d,%d\n",evt.delta_xy.x,evt.delta_xy.y);
          return true;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override {
        if (evt.cmd == ANIMATION && evt.reason == 0 && evt.target == self) {
          scroller = nullptr;
        }
        return false;
      }
      
    };

    struct swipe_ctl : swipe_touch_ctl {
      virtual const string &behavior_name() const {
        return _swipe_ctl_factory->name;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) {
        if (touch_device) return false;
        switch (evt.cmd) {
        case MOUSE_ENTER: return false;
        case MOUSE_LEAVE: return false;

        case MOUSE_DOWN:
        case MOUSE_DCLICK:
          if (!evt.is_point_button()) return false;
          swipe_tracking = true;
          got_swipe      = false;
          v.set_focus(self, BY_MOUSE);
          v.set_capture(self);
          started_in = evt.target;
          strokes.clear();
          // push_stroke(v,evt.pos);
          prev_time = v.get_ticks();
          prev_pos  = evt.pos;
          if (scroller) 
            v.remove_animation(self, scroller);
          return false;
        case MOUSE_UP:
          v.stop_capture(self);
          if (!evt.is_point_button()) return false;
          push_stroke(v, evt.pos);
          stop_swipe_tracking(v, self);
          stop_scroll_tracking(v, self);
          return false;
        case MOUSE_MOVE:
          if (evt.is_point_button()) {
            push_stroke(v, evt.pos);
            bool scrollable_h = self->can_scroll_h(v);
            bool scrollable_v = self->can_scroll_v(v);
            if (swipe_tracking) {
              if (check_swipe_gesture(v, self, scrollable_h, scrollable_v))
                stop_swipe_tracking(v, self);
              // return true;
            }
            check_scroll_gesture(v, self, scrollable_h, scrollable_v);
          }

          break;
        case MOUSE_WHEEL: return false;
        }
        return false;
      }
    };

    ctl *swipe_touch_ctl_factory::create(element *el) {
      return new swipe_touch_ctl();
    }
    ctl *swipe_ctl_factory::create(element *el) { return new swipe_ctl(); }

    void init_swipe_ctl() {
      ctl_factory::add(_swipe_ctl_factory = new swipe_ctl_factory());
      ctl_factory::add(_swipe_touch_ctl_factory = new swipe_touch_ctl_factory());
    }

    void init_gestures() { init_swipe_ctl(); }

  } // namespace behavior

  void setup_touch_scroll_handler(view &v, element *b) {
    if (b->ldata->sb.visible()) {
      if (b->get_named_behavior(behavior::_swipe_ctl_factory->name)) return;
      b->attach_behavior(v, behavior::_swipe_ctl_factory->create(b));
    } else {
      b->detach_named_behavior(v, behavior::_swipe_ctl_factory->name);
    }
  }

} // namespace html
