#include "html.h"

namespace html {
  namespace behavior {

    struct select_ctl_factory : public ctl_factory {
      select_ctl_factory() : ctl_factory("select") {}
      virtual ctl *create(element *el);
    };
    static select_ctl_factory *_select_ctl = 0;

    struct select_multiple_ctl_factory : public ctl_factory {
      select_multiple_ctl_factory() : ctl_factory("select-multiple") {}
      virtual ctl *create(element *el);
    };
    static select_multiple_ctl_factory *_select_multiple_ctl = 0;

    struct select_checkmarks_ctl_factory : public ctl_factory {
      select_checkmarks_ctl_factory() : ctl_factory("select-checkmarks") {}
      virtual ctl *create(element *el);
    };
    static select_checkmarks_ctl_factory *_select_checkmarks_ctl = 0;

    ustring role_option = WCHARS("option");
    // static ustring role_options = L"options";

    value option_value(view &v, function<value(const ustring &)> coerce,
                       element *opt) {
      ustring us;
      if (opt->attr_value(us)) {
        if (us.length())
          // return value::parse(us);
          // return value(us);
          return coerce(us);
        else
          return value(W(""));
      }
      else if (opt->attr_key(us))
        return value(us);
      return value(trim(opt->get_text(v)()));
    }

    bool is_option_filter(view &v, element *b) // option disabled or enabled
    {
      return (b->tag == tag::T_OPTION || b->attr_role() == role_option) && !b->state.disabled();
    }

    bool is_select_option_filter(view &   v,
                                 element *b) // option disabled or enabled
    {
      return (b->tag == tag::T_OPTION || b->attr_role() == role_option) && b->is_it_visible(v) && !b->state.disabled();
    }

    bool _is_select_option(view &v, element *b) // option disabled or enabled
    {
      return (b->tag == tag::T_OPTION || b->attr_role() == role_option) && !b->state.disabled();
    }

    bool is_select_leaf_filter(view &   v,
                               element *b) // option disabled or enabled
    {
      if (b->tag == tag::T_OPTION && !b->state.node() && b->is_visible(v))
        return true;
      element *p = b->get_owner();
      if (!p) return false;
      if (p->tag == tag::T_OPTION && p->state.node() &&
          p->first_ui_element() == b && b->is_visible(v))
        return true; // option's caption
      return false;
    }

    select_ctl::select_ctl() : _in_set_value(false) {
      _coerce          = value::parse;
      is_select_option = is_select_option_filter;
    }

    bool select_ctl::attach(view &v, element *self) {
      // self->atts.set(attr::a_type,L"select");
      _self = self;
      _current_option    = 0;
      element *    first = 0;
      each_element ee(self);

      bool is_dd_list = is_select_on_popup(v, self);

      ustring as = (is_dd_list ? self->parent.ptr() : self)->atts(attr::a_as);

      if (as == WCHARS("string"))
        _coerce = value::parse_string;
      else if (as == WCHARS("integer"))
        _coerce = value::parse_integer;
      else if (as == WCHARS("float"))
        _coerce = value::parse_float;
      else if (as == WCHARS("numeric"))
        _coerce = value::parse_numeric;
      else if (as == WCHARS("auto"))
        _coerce = value::parse;
      else
#if defined(SCITERJS) 
        _coerce = value::parse_string;
#else
        _coerce = value::parse;
#endif

      for (element *el; ee(el);) {
        if (!is_select_option(v, el)) continue;
        if (el->atts.exist(attr::a_selected)) {
          el->state.current(true);
          el->state.checked(true);
          // if(!_current_option)
          _current_option             = el;
          _value_option               = el;
          el->flags.state_initialized = 1;
          el->reset_styles(v);
          self->check_layout(v);
          return true;
        } else if (!first)
          first = el;
      }
      // !!!!! this must be here otherwise sel.value = XX will not work properly
      // after initial initialization.
      // !!!!! before/after changing this check
      // /test-cases/inputs/select-dd-set-value.htm

      if (first && is_dd_list /*self->tag == tag::T_POPUP*/) // popup dropdown list
      {
        first->state.current(true);
        _current_option                = first;
        first->flags.state_initialized = 1;
        first->reset_styles(v);
      }
      self->check_layout(v);
      return true;
    }

    void select_ctl::detach(view &v, element *self) {
      _current_option = 0;
      _value_option   = 0;
      _self = nullptr;
    }

    bool select_ctl::on(view &v, element *self, event_behavior &evt) {
      if (evt.cmd == (CONTENT_CHANGED | EVENT_SINKING) || evt.cmd == (CONTENT_CHANGED | EVENT_SINKING | EVENT_HANDLED)) {
        if (_current_option && !_current_option->belongs_to(self)) {
          _current_option->state.current(false);
          _current_option = 0;
        }
        if (_value_option && !_value_option->belongs_to(self)) {
          _value_option->state.checked(false);
          _value_option = 0;
        }
      }
      else if (select_type(self) == SINGLE &&
              (evt.cmd_bubbling() == CURRENT_ELEMENT_CHANGED) && 
              is_option_filter(v, evt.source)) 
      {
        auto_state<bool> _(this->_in_set_value, true);
        select_option(v, self, evt.source, 0, SET_VALUE);
        return true;
      }
      else if (select_type(self) == SINGLE &&
        (evt.cmd == ACTIVATE_CHILD) &&
        is_option_filter(v, evt.target))
      {
        if(select_option(v, self, evt.target, 0, SET_VALUE))
          notify_change(v, self, evt.target, CLICK_SYNTHESIZED);
        return true;
      }
      return false;
    }

    bool select_ctl::on(view &v, element *self, event_command &evt) {
      if (evt.command == WCHARS("set-current")) {
        element *opt = get_target_option(v, self, evt.target);
        if (!opt || opt->is_disabled()) return false;

        if (evt.cmd == event_command::CHECK) return true;

        if (evt.cmd == event_command::EXEC) {
          _in_set_value = true;
          v.reset_current_in(self);
          select_option(v, self, opt, 0, SET_VALUE);
          _in_set_value = false;
          return true;
        }
      }
      return false;
    }

    element *select_ctl::get_first_option(view &v, element *self) {
      each_ui_element ee(self);
      for (element *c; ee(c);) {
        if (!is_select_option(v, c)) continue;
        return c;
      }
      return 0;
    }

    element *get_option_caption(view &v, element *opt) {
      if (opt->has_children_of_type(tag::T_OPTION))
        // that is option group
        return opt->first_ui_element();
      return opt;
    }

    element *select_ctl::get_target_option(view &v, element *self,
                                           element *target) {
      while (target) {
        if (target == self) break;
        if (is_select_option(v, target)) return target;
        target = target->get_owner();
      }
      return 0;
    }

    bool select_ctl::get_auto_height(view &v, element *self, int &value) {
      int sz = self->parent && self->tag == tag::T_POPUP 
        ? self->parent->atts.get_int(attr::a_size, 40)
        : self->atts.get_int(attr::a_size, 4);
      if (sz) {
        element *opt = get_first_option(v, self);
        if (opt) {
          rect bd = opt->border_distance(v);
          int  h  = opt->computed_height(v, self->content_box(v).height()) +
                  bd.s.y + bd.e.y;
          value = sz * h;
        } else
          value = sz * pixels(v, self, self->get_style(v)->font_size).height();

        if (self->parent && self->tag == tag::T_POPUP) {
          size psz = self->parent->border_box(v).size();
          size src = (v.screen_workarea().size() - psz) * 2 / 5;
          value    = limit(value, psz.y, src.y);
        }
      }
      return true;
    }
    bool select_ctl::get_auto_width(view &v, element *self, int &value) {
      value = self->max_content_width(v) 
#ifdef THEMES_SUPPORT
        + theme::current()->scrollbar_width()
#endif
        ;
      return true;
    }

    bool select_ctl::set_value(view &v, element *self, const value &val) {
      auto_state<bool> _(_in_set_value, true);
                  
      helement found;

      if (val.is_defined()) {
        auto onel = [&](element *el) -> bool {
          value t = option_value(v, _coerce, el);
          if (val.is_string()) t = t.to_string();
          if (val == t) {
            //select_option(v, self, el, 0, SET_VALUE);
            found = el;
            return true;
          }
          return false;
        };
        find_all(v, self, WCHARS("option,[role='option']"), onel);
      }

      if ((found != _value_option) || !found) {
        drop_selection(v, self);
        if(found)
          select_option(v, self, found, 0, SET_VALUE);
      }

      /*if (_current_option)
        clear_option_state(v, self, _current_option, S_CURRENT | S_CHECKED);
      if ((_value_option != _current_option) && (_value_option != 0))
        clear_option_state(v, self, _value_option, S_CURRENT | S_CHECKED);
      _current_option = _value_option = 0;*/
      
      return true;
    }
    bool select_ctl::get_value(view &v, element *self, value &val) {
      // element* b = find_first(v,self,
      // WCHARS("option:current,[role='option']:current"));
      if (_value_option) {
        val = option_value(v, _coerce, _value_option);
        return true;
      }
      val = value();
      return true;
    }

    bool select_ctl::set_value_multiple(view &v, element *self,
                                        const value &val_) {
      auto_state<bool> _(_in_set_value, true);

      if (_current_option)
        clear_option_state(v, self, _current_option, S_CURRENT);

      value val;

      auto onel = [&](element *el) -> bool {
        value t = option_value(v, _coerce, el);
        if (val.is_string()) t = t.to_string();
        if (val == t) {
          set_option_state(v, self, el, S_CHECKED);
          return true;
        }
        return false;
      };

      auto set_one_val = [&](const value &opt_val) {
        val = opt_val;
        find_all(v, self,
                 WCHARS("option:not(:node),[role='option']:not(:node)"), onel);
      };

      drop_selection(v, self);

      if (val_.is_array_like()) {
        for (uint i = 0; i < val_.size(); ++i)
          set_one_val(val_.get_element(i));
      } else
        set_one_val(val_);
      return true;
    }

    bool select_ctl::get_value_multiple(view &v, element *self, value &val) {
      // val = value::make_array(0);
      val.clear();
      auto onel = [&](element *el) -> bool {
        val.push(option_value(v, _coerce, el));
        return false;
      };
      find_all(
          v, self,
          WCHARS(
              "option:not(:node):checked,[role='option']:not(:node):checked"),
          onel);
      return true;
    }

    bool select_ctl::on(view &v, element *self, event_mouse &evt) {
      switch (evt.cmd) {
      case MOUSE_DOWN: {
        if (evt.is_point_button()) {
          v.set_capture(self);
          v.set_focus(self, BY_MOUSE);
        }
        element *opt = get_target_option(v, self, evt.target);
        if (opt) return on_option(v, self, evt, opt);
      } break;
      case MOUSE_UP:
        if (self->state.pressed()) {
          v.set_capture(0);
          element *opt = get_target_option(v, self, evt.target);
          //if (!opt) opt = this->_current_option; -- WRONG, :disabled !
          if (opt) return on_option(v, self, evt, opt);
        }
        break;
      case MOUSE_DCLICK:
        if (self->state.pressed()) {
          element *opt = get_target_option(v, self, evt.target);
          if (opt) return on_option(v, self, evt, opt);
        }
        break;
      case MOUSE_MOVE:
        if (evt.is_point_button() && self->state.pressed()) {
          element *opt = get_target_option(v, self, evt.target);
          if (opt) return on_option(v, self, evt, opt);
        }
        break;
        /* too fancy
        case MOUSE_WHEEL:
         if(self->state.focus())
         {
            int t = evt.get_wheel_delta();
            bool r = false;
            if( t < 0 )  while( t++ )  {
        if(select_next_option(v,self,+1,evt.alt_state,CLICK_BY_MOUSE))  r =
        true; } else         while( t-- )  {
        if(select_next_option(v,self,-1,evt.alt_state,CLICK_BY_MOUSE))  r =
        true; } return r;
         }
         break; */
      }
      return false;
    }

    bool select_ctl::on_option(view &v, element *self, event_mouse &evt,
                               element *opt) {
      switch (evt.cmd) {
      case MOUSE_DOWN:
        if (evt.is_point_button()) {
          if (select_type(self) == MULTIPLE)
          {
            if(select_option(v, self, opt, evt.alt_state))
              notify_change(v, self, opt, CLICK_BY_MOUSE);
          }
          else if (opt != _value_option) {
            select_option(v, self, opt, evt.alt_state);
            // if( self->state.popup() )
            //  notify_changing(v,self,opt, CLICK_BY_MOUSE);
            // else
            notify_change(v, self, opt,
                          evt.is_on_icon ? CLICK_BY_MOUSE_ON_ICON
                                         : CLICK_BY_MOUSE);
          }
          /* attempt to implement click on current option to "un-current" it .
             does not work due to mouse_up below else // click on _value_option
                    {
                      if (_current_option)
                        clear_option_state(v, self, _current_option, S_CURRENT |
             S_CHECKED); if (_value_option != _current_option && _value_option
             != 0) clear_option_state(v, self, _value_option, S_CURRENT |
             S_CHECKED); _current_option = _value_option = 0;
                    } */
          return true;
        }
        break;
      case MOUSE_MOVE: {
        if (evt.is_point_button() && opt != _current_option) {
          select_option(v, self, opt, evt.alt_state,
                        evt.is_point_button() ? SET_VALUE : SET_ONLY_CURRENT);
          // if(self->state.popup())
          //  notify_changing(v,self,opt, CLICK_BY_MOUSE);
          // else
          notify_change(v, self, opt, CLICK_BY_MOUSE);
          return true;
        }
      } break;
      case MOUSE_UP:
        /* WRONG: MOUSE_UP arrives after MOUSE_DOUBLECLICK
        if (evt.is_point_button()) {
          if (opt != _value_option) {
            select_option(v, self, opt, evt.alt_state);
            notify_change(v, self, opt,
                          evt.is_on_icon ? CLICK_BY_MOUSE_ON_ICON
                                         : CLICK_BY_MOUSE);
            return true;
          }
        }*/
        return false;
      }
      return false;
    }

    bool select_ctl::drop_selection(view &v, element *self,
                                    bool including_anchor) {
      each_element ee(self);
      int             n             = 0;
      uint            flags_to_drop = selected_state_flags();
      if (including_anchor) flags_to_drop |= S_ANCHOR;
      for (element *c; ee(c);) {
        if (!is_select_option(v, c)) continue;
        if (c->state.anchor() || c->state.current() || c->state.checked()) {
          clear_option_state(v, self, c, flags_to_drop);
          // c->state_off(v,flags_to_drop);
          ++n;
        }
      }
      _current_option = 0;
      _value_option   = 0;
      return n > 0;
    }

    bool select_ctl::select_option(view &v, element *self, element *opt,
                                   uint alts, SET_WHAT what) {
      uint64 flags = what == SET_ONLY_CURRENT ? S_CURRENT : (S_CURRENT | S_CHECKED);
      if (_current_option) {
        if ((opt == _current_option) && (_current_option->state.data & flags) == flags)
          return true;
        clear_option_state(v, self, _current_option, flags);
      }
      if (what == SET_VALUE && _value_option != _current_option && _value_option != 0)
        clear_option_state(v, self, _value_option, flags);

      _current_option = opt;
      if (what == SET_VALUE) 
        _value_option = _current_option;

      if (_current_option) 
      {
        find_all(v, self, WCHARS(":current"), [&](element* c) { c->state_off(v, S_CURRENT); return false; });
        set_option_state(v, self, _current_option, flags);
        if (_current_option->state.node())
          v.ensure_visible(_current_option->first_ui_element(), false, SCROLL );
        else
          v.ensure_visible(_current_option, false, SCROLL);
      }
      return true;
    }

    bool is_option_visible(view &v, element *self, element *opt) {
      rect src = self->content_box(v, element::TO_VIEW);
      rect orc = opt->content_box(v, element::TO_VIEW);
      if (orc.height() <= src.height())
        return (src.y() & orc.y()) == orc.y();
      else
        return (src.y() & orc.y()) == src.y();
    }

    bool select_ctl::select_next_option(view &v, element *self, int cmd,
                                        uint alts, uint reason) {
      element *nc           = 0;
      bool     only_current = false;

      if (_current_option && !_current_option->belongs_to(self))
        _current_option = 0;

      switch (cmd) {
      case 0: {
        assert(_current_option == 0);
        each_ui_element ee(self);
        // ee.b = _current_option;
        element *fe = 0;
        element *se = 0;
        for (element *el; ee(el);) {
          if (!is_select_option(v, el)) continue;
          if (!fe) fe = el;
          if (fe->atts.exist(attr::a_selected)) {
            se = el;
            break;
          }
        }
        if (se)
          nc = se;
        else if (fe) {
          nc           = fe;
          only_current = true;
        } else
          return false;

      } break;

      case -1: {
        if (!_current_option) goto END;
        each_ui_element_backward ee(self);
        ee.b = _current_option;
        element *fe;
        while (ee(fe)) {
          if (!is_select_option(v, fe)) continue;
          nc = fe;
          break;
        }
        /*if( !nc )
        {
          self->scroll_pos(v,point(0,0));
          v.refresh(self);
          return true;
        }*/
      } break;
      case -2: {
        if (!_current_option) goto END;
        each_ui_element_backward ee(self);
        ee.b        = _current_option;
        element *fe = 0;
        while (ee(fe)) {
          if (!is_select_option(v, fe)) continue;
          if (is_option_visible(v, self, fe))
            nc = fe;
          else
            break;
        }
        if (nc && (nc != _current_option)) break;
        if (!fe) break;
        // ee.b = nc;
        int y  = _current_option->border_box(v, element::TO_VIEW).bottom();
        int ly = y - self->content_box(v, element::TO_VIEW).height();
        do {
          if (!is_select_option(v, fe)) continue;
          nc = fe;
          if (nc->border_box(v, element::TO_VIEW).top() <= ly) break;
        } while (ee(fe));

      } break;
      case -3: {
      HOME:
        each_ui_element ee(self);
        element *       fe;
        while (ee(fe)) {
          if (!is_select_option(v, fe)) continue;
          nc = fe;
          break;
        }
        /*if( nc && nc == _current_option )
        {
          v.ensure_visible(self->first_ui_element(),true);
          return true;
        }*/
      } break;
      case +1: {
        if (!_current_option) goto HOME;
        each_ui_element ee(self);
        ee.b = _current_option;
        element *fe;
        while (ee(fe)) {
          if (!is_select_option(v, fe)) continue;
          nc = fe;
          break;
        }
      } break;
      case +2: // page down
      {
        if (!_current_option) goto HOME;
        each_ui_element ee(self);
        ee.b        = _current_option;
        element *fe = 0;
        while (ee(fe)) {
          if (!is_select_option(v, fe)) continue;
          if (is_option_visible(v, self, fe))
            nc = fe;
          else
            break;
        }
        if (nc && (nc != _current_option)) break;
        if (!fe) break;
        // ee.b = nc;
        int y  = _current_option->border_box(v, element::TO_VIEW).top();
        int ly = y + self->content_box(v, element::TO_VIEW).height();
        do {
          if (!is_select_option(v, fe)) continue;
          nc = fe;
          if (nc->border_box(v, element::TO_VIEW).bottom() >= ly) break;
        } while (ee(fe));
      } break;
      case +3: {
      END:
        each_ui_element_backward ee(self);
        element *                fe;
        while (ee(fe)) {
          if (!is_select_option(v, fe)) continue;
          nc = fe;
          break;
        }
      } break;

      case +10: // right
      {
        if (!_current_option) goto END;
        int row, col;
        if (!self->row_col_of(_current_option, row, col)) goto END;

        element *tc = self->at(row, col + 1);
        if (!tc) goto END;
        if (is_select_option(v, tc)) nc = tc;

      } break;

      case -10: // left
      {
        if (!_current_option) goto HOME;
        int row, col;
        if (!self->row_col_of(_current_option, row, col) || col <= 0) goto HOME;

        element *tc = self->at(row, col - 1);
        if (!tc) goto HOME;
        if (is_select_option(v, tc)) nc = tc;

      } break;

      case +11: // down
      {
        if (!_current_option) goto END;
        int row, col;
        if (!self->row_col_of(_current_option, row, col)) goto END;

        element *tc = self->at(row + 1, col);
        if (!tc) goto END;
        if (is_select_option(v, tc)) nc = tc;

      } break;

      case -11: // up
      {
        if (!_current_option) goto HOME;
        int row, col;
        if (!self->row_col_of(_current_option, row, col) || col <= 0) goto HOME;

        element *tc = self->at(row - 1, col);
        if (!tc) goto HOME;
        if (is_select_option(v, tc)) nc = tc;

      } break;
      }
      if (nc) {
        select_option(v, self, nc, alts, only_current ? SET_ONLY_CURRENT : SET_VALUE);
        if (!only_current && cmd) notify_change(v, self, nc, reason);
        return true;
      } //else if (_current_option) ???
        //return true;
      return false;
    }

    void select_ctl::notify_state_changing(view &v, element *self,
                                          element *option, uint reason) {
      if (_in_set_value) return;
      auto_state<bool> _(_in_set_value, true);
      event_behavior evt(self, self, SELECT_SELECTION_CHANGING, reason);
      v.send_behavior_event(evt);
    }
    void select_ctl::notify_change(view &v, element *self, element *option,
                                   uint reason) {
      if (_in_set_value) return;
      if (notify_changing(v, self, option, reason))
        return;
      {
        event_behavior evt(self, self, is_select_on_popup(v, self) ? POPUP_SELECT_VALUE_CHANGED : SELECT_VALUE_CHANGED, reason);
        v.post_behavior_event(evt,true);
      }
    }
    
    bool select_ctl::a11y_get_focus(view &v, element *self, helement& hf) {
      assert(self->state.focus()); 
      if (_current_option) {
        hf = _current_option;
        return true;
      }
      return false;
    }

    bool select_ctl::notify_changing(view &v, element *self, element *option,
                                     uint reason) {
      if (_in_set_value)
          return false;
      event_behavior evt(option, self, SELECT_SELECTION_CHANGING, reason);
      return v.send_behavior_event(evt);
    }

    void select_checkmarks_ctl::notify_change(view &v, element *self,
                                              element *option, uint reason) {
      if (_in_set_value) return;
      event_behavior evt(option, self, CLICK, reason);
      v.post_behavior_event(evt);
    }

    bool select_ctl::on(view &v, element *self, event_key &evt) {
      if (evt.cmd == KEY_DOWN)
        goto KEY_DOWN;
      else if (evt.cmd == KEY_CHAR)
        goto KEY_CHAR;
      return false;
    KEY_DOWN:
      switch (evt.key_code) {
      case KB_LEFT:
        if (self->is_horizontal_layout())
          return select_next_option(v, self, -1, evt.alt_state, CLICK_BY_KEY);
        else
          return select_next_option(v, self, -10, evt.alt_state, CLICK_BY_KEY);
        break;
      case KB_RIGHT:
        if (self->is_horizontal_layout())
          return select_next_option(v, self, +1, evt.alt_state, CLICK_BY_KEY);
        else
          return select_next_option(v, self, +10, evt.alt_state, CLICK_BY_KEY);
        break;

      case KB_UP:
        if (self->is_horizontal_layout())
          return select_next_option(v, self, -11, evt.alt_state, CLICK_BY_KEY);
        else
          return select_next_option(v, self, -1, evt.alt_state, CLICK_BY_KEY);
      case KB_DOWN:
        if (self->is_horizontal_layout())
          return select_next_option(v, self, +11, evt.alt_state, CLICK_BY_KEY);
        else
          return select_next_option(v, self, +1, evt.alt_state, CLICK_BY_KEY);
      case KB_PRIOR:
        return select_next_option(v, self, -2, evt.alt_state, CLICK_BY_KEY);
      case KB_NEXT:
        return select_next_option(v, self, +2, evt.alt_state, CLICK_BY_KEY);
      case KB_HOME:
        return select_next_option(v, self, -3, evt.alt_state, CLICK_BY_KEY);
      case KB_END:
        return select_next_option(v, self, +3, evt.alt_state, CLICK_BY_KEY);
      }
      return false;
    KEY_CHAR:
      if (evt.has_modifiers()) return false;
      return select_by_char(v, self, evt.key_code);
    }

    bool select_ctl::select_by_char(view &v, element *self, uint key_code) {
      key_code = to_upper(wchar(key_code));

      // if( b->state.current() )
      element *first_matching = 0;
      element *found          = 0;

      auto onel = [&](element *el) -> bool {
        ustring text = el->get_text(v);
        wchar first_char = text.length() ? text[0] : 0;
        if (key_code == to_upper(first_char)) {
          if (!first_matching) first_matching = el;

          if (el->state.current())
            found = 0;
          else if (!found)
            found = el;
        }
        return false;
      };

      find_all(v, self, WCHARS("option:not(:node),[role='option']:not(:node)"),
               onel);

      auto set_value = [&](element *el) {
        select_option(v, self, el, 0, SET_VALUE);
        notify_change(v, self, el, CLICK_BY_KEY);
      };

      if (found)
        set_value(found);
      else if (first_matching)
        set_value(first_matching);
      else
        return false;

      return true;
    }

    bool select_ctl::on(view &v, element *self, event_focus &evt) {
      if (evt.cmd == FOCUS_IN) {
        // WRONG, gives visual clue that value is set when it is not:
        // if(!_current_option)
        //{
        //  select_next_option(v,self, 0, 0, BY_CODE);
        //}
        return true;
      } else if (evt.cmd == FOCUS_OUT) {
        return true;
      }
      return false;
    }
    bool select_ctl::on_method_call(view &v, element *self, method_params *p) {
      return false;
    }

    void select_ctl::set_option_state(view &v, element *self, element *opt, uint64 state, uint64 state_to_clear) {
      if ((opt->state.data & state) != state) {
        if (state & S_CURRENT) {
          if (_current_option) {
            _current_option->state.current(false);
            _current_option->drop_styles(v);
          }
          _current_option = opt;
        }
        opt->state.data |= state;
        opt->drop_styles(v);
        //opt->state_on(v, state); //- this will post CURRENT_ELEMENT_CHANGED - not desirable
        if (state & S_CHECKED) // but not S_CURRENT
          notify_state_changing(v, self, opt, 1);
      }
      if ((opt->state.data & state_to_clear) != 0) {
        opt->state.data &= ~state_to_clear;
        opt->drop_styles(v);
        //opt->state_off(v, state_to_clear);
        if (state_to_clear & S_CHECKED) // but not S_CURRENT
          notify_state_changing(v, self, opt, 0);
        if (state_to_clear & S_CURRENT) {
          if (_current_option) {
            _current_option->state.current(false);
            _current_option->drop_styles(v);
          }
          _current_option = nullptr;
        }
      }
    }
    void select_ctl::clear_option_state(view &v, element *self, element *opt,
                                        uint64 state) {
      set_option_state(v, self, opt, 0, state);
    }

    void select_ctl::attach_multiple(view &v, element *self) {
      _current_option = 0;
      each_ui_element ee(self);
      for (element *el; ee(el);) {
        if (!is_select_option(v, el)) continue;
        if (el->atts.exist(attr::a_selected)) {
          el->state.current(true);
          el->state.checked(true);
          if (!_current_option) _current_option = el;
          el->flags.state_initialized = 1;
        }
      }
      self->check_layout(v);
    }

    bool select_ctl::select_option_multiple(view &v, element *self,
                                            element *opt, uint alts,
                                            SET_WHAT what) {
      if (alts == ALT_SHORTCUT) {
        if (opt->state.checked())
          set_option_state(v, self, opt, S_CURRENT | S_ANCHOR, S_CHECKED);
        else
          set_option_state(v, self, opt, S_CHECKED | S_ANCHOR | S_CURRENT);
      }
      else if (alts & ALT_SHIFT) {
        helement anchor;
        each_ui_element ee(self);
        element *c;
        while (ee(c)) {
          if (!is_select_option(v, c))
            continue;
          if (c->state.anchor()) {
            anchor = c;
            break;
          }
        }
        if((alts & ALT_SHORTCUT) == 0)
          drop_selection(v, self,false);
        if( !anchor)
          set_option_state(v, self, opt, S_CURRENT, S_CHECKED);
        else
          fill_selection_multiple(v, self, anchor, opt);
      } else {
        drop_selection(v, self);
        set_option_state(v, self, opt, S_CHECKED | S_ANCHOR | S_CURRENT);
      }

      // opt->state_on(v, flags);

      _current_option = opt;
      _value_option   = opt;
      v.ensure_visible(opt, false);
      return true;
    }

    void select_ctl::fill_selection_multiple(view &v, element *self, element *opt1, element* opt2) {
      // uint flags_to_set = S_CHECKED;
      // bool is_inside_selection = false;

      if (opt1->start_pos() > opt2->start_pos())
        swap(opt1, opt2);

      bool inside_selection = false;

      each_ui_element ee(self);
      element *       c;
      while (ee(c)) {
        if (!is_select_option(v, c))
          continue;
       if (c == opt1)
          inside_selection = true;
       if (inside_selection)
            set_option_state(v, self, c, S_CHECKED);
       if (c == opt2)
         break;
      }
    }

    void init_dd_selects();

    const string &select_ctl::behavior_name() const {
      return _select_ctl->name;
    }
    ctl *select_ctl_factory::create(element *el) { return new select_ctl(); }

    const string &select_multiple_ctl::behavior_name() const {
      return _select_multiple_ctl->name;
    }
    ctl *select_multiple_ctl_factory::create(element *el) {
      return new select_multiple_ctl();
    }

    const string &select_checkmarks_ctl::behavior_name() const {
      return _select_checkmarks_ctl->name;
    }
    ctl *select_checkmarks_ctl_factory::create(element *el) {
      return new select_checkmarks_ctl();
    }

    bool select_checkmarks_ctl::attach(view &v, element *self) {
      // self->atts.set(attr::a_type,L"select");
      _current_option = 0;
      each_ui_element ee(self);
      for (element *el; ee(el);) {
        if (!is_select_option(v, el)) continue;
        if (el->atts.exist(attr::a_selected)) {
          el->state.checked(true);
          if (!_current_option) {
            el->state.current(true);
            _current_option = el;
          }
          el->flags.state_initialized = 1;
          el->drop_styles(v);
        }
      }
      self->check_layout(v);
      return true;
    }

    bool select_checkmarks_ctl::on_option(view &v, element *self,
                                          event_mouse &evt, element *opt) {
      switch (evt.cmd) {
      case MOUSE_UP:
        if (evt.is_point_button()) {
          if (!opt->state.checked())
            set_option_state(v, self, opt, S_CHECKED);
          else
            clear_option_state(v, self, opt, S_CHECKED);
        }
        break;
      }
      return select_ctl::on_option(v, self, evt, opt);
    }
    bool select_checkmarks_ctl::on(view &v, element *self, event_key &evt) {
      if (evt.cmd == KEY_DOWN && evt.key_code == KB_SPACE && _current_option) {
        if (!_current_option->state.checked())
          set_option_state(v, self, _current_option, S_CHECKED);
        else
          clear_option_state(v, self, _current_option, S_CHECKED);
        notify_change(v, self, _current_option, CLICK_BY_KEY);
        return true;
      } else if (evt.cmd == KEY_DOWN && evt.key_code == 'A' &&
                 evt.is_shortcut()) {
        element *       first = 0;
        bool            on    = false;
        each_ui_element ee(self);
        for (element *c; ee(c);)
          if (is_select_option(v, c)) {
            if (!first) {
              first = c;
              on    = !c->state.checked();
            }
            if (on)
              set_option_state(v, self, c, S_CHECKED);
            else
              clear_option_state(v, self, c, S_CHECKED);
          }
        return true;
      }
      return select_ctl::on(v, self, evt);
    }

    //|
    //| TREE
    //|

    struct tree_ctl_factory : public ctl_factory {
      tree_ctl_factory() : ctl_factory("tree") {}
      virtual ctl *create(element *) override;
    };
    tree_ctl_factory *_tree_ctl;

    struct tree_ctl : public select_ctl {
      typedef select_ctl super;
      tree_ctl() : select_ctl() { is_select_option = is_select_leaf_filter; }

      virtual const string &behavior_name() const { return _tree_ctl->name; }
      virtual CTL_TYPE      get_type() { return CTL_SELECT_SINGLE; }

      /*virtual bool attach(view& v, element* self) override
      {
        //init_options(v,self);
        super::attach(v,self);
      }*/

      bool toggle(view &v, element *self, element *leaf, bool select_it) {
        if (leaf->flags.ui_index != 0 || !leaf->get_owner() ||
            !leaf->get_owner()->state.node())
          return false;
        if (leaf->get_owner()->state.expanded())
          return collapse_option(v, self, leaf, select_it);
        else
          return expand_option(v, self, leaf, select_it);
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        element *target = evt.is_on_icon;
        if ((evt.cmd == MOUSE_DOWN || evt.cmd == MOUSE_DCLICK) && target != 0 &&
            target->state.node())
          return toggle(v, self, target->first_ui_element(), false);
        return super::on(v, self, evt);
      }

      virtual bool on_option(view &v, element *self, event_mouse &evt,
                             element *opt) override {
        switch (evt.cmd) {
        case MOUSE_DCLICK:
          if (evt.is_point_button()) {
            if (opt->get_owner()->state.node()) return toggle(v, self, opt, true);
          }
          break;
        }
        return super::on_option(v, self, evt, opt);
      }

      bool is_expanded_option_caption(element *el) {
        element *owner = el->get_owner();
        return el->flags.ui_index == 0 && owner->state.node() && owner->state.expanded();
      }

      virtual bool on(view &v, element *self, event_key &evt) override {

        if (evt.cmd == KEY_DOWN) {
          switch (evt.key_code) {
          case KB_LEFT:
            if (!self->state.rtl())
              return collapse_option(v, self, _current_option, true);
            else
              return expand_option(v, self, _current_option, true);
            break;
          case KB_RIGHT:
            if (!self->state.rtl())
              return expand_option(v, self, _current_option, true);
            else
              return collapse_option(v, self, _current_option, true);
            break;
          }
        }
        return super::on(v, self, evt);
      }

      bool expand_option(view &v, element *self, element *leaf,
                         bool select_it = false) {
        if (!leaf) return false;
        if (leaf->tag == tag::T_OPTION) return false;

        element *nc = find_first_parent(v, leaf, WCHARS("option:node"));
        if (!nc || !nc->belongs_to(self)) return false;

        if (nc->state.collapsed()) {
          nc->state_on(v, S_EXPANDED);
          event_behavior evt(nc, nc, ELEMENT_EXPANDED, 0);
          v.post_behavior_event(evt);
        } else if (select_it)
          return select_next_option(v, self, +1, 0, 0);
        else
          return false;

        // self->commit_measure(v);

        if (select_it)
          select_option(v, self, nc->first_ui_element(), 0, SET_ONLY_CURRENT);
        return true;
      }
      bool collapse_option(view &v, element *self, element *leaf,
                           bool select_it = false) {
        if (!leaf) return false;

        element *nc =
            find_first_parent(v, leaf, WCHARS("option:node:expanded"));
        if (!nc || !nc->belongs_to(self)) return false;

        if (!is_expanded_option_caption(leaf)) {
          select_option(v, self, nc->first_ui_element(), 0, SET_ONLY_CURRENT);
          return true;
        }

        if (nc->state.expanded()) {
          nc->state_on(v, S_COLLAPSED);
          event_behavior evt(nc, nc, ELEMENT_COLLAPSED, 0);
          v.post_behavior_event(evt);
        } else if (select_it) {
          element *nnc =
              find_first_parent(v, nc->parent, WCHARS("option:node"));
          if (!nnc || !nnc->belongs_to(self)) return false;
          return collapse_option(v, self, nnc->first_ui_element(), true);
        } else
          return false;

        // self->commit_measure(v);
        if (select_it)
          select_option(v, self, nc->first_ui_element(), 0, SET_ONLY_CURRENT);
        return true;
      }
    };

    ctl *tree_ctl_factory::create(element *el) { return new tree_ctl(); }

    //|
    //| TREE-VIEW
    //|

    bool is_visible_option(view &v, element *b) // option disabled or enabled
    {
      return b->tag == tag::T_OPTION && b->is_visible(v);
    }

    struct tree_view_ctl_factory : public ctl_factory {
      tree_view_ctl_factory() : ctl_factory("tree-view") {}
      virtual ctl *create(element *) override;
    };
    tree_view_ctl_factory *_tree_view_ctl;

    struct tree_view_ctl : public select_ctl {
      typedef select_ctl super;
      tree_view_ctl() : select_ctl() { is_select_option = is_visible_option; }

      virtual const string &behavior_name() const {
        return _tree_view_ctl->name;
      }
      virtual CTL_TYPE get_type() { return CTL_SELECT_SINGLE; }

      /*virtual bool attach(view& v, element* self) override
      {
        //init_options(v,self);
        super::attach(v,self);
      }*/

      bool is_option(element *b) {
        return (b->tag == tag::T_OPTION || b->attr_role() == role_option);
      }

      element *target_option(element *self, element *el) {
        if (!el) return 0;
        if (el == self) return 0;
        if (is_option(el)) return el;
        return target_option(self, el->parent);
      }

      bool is_on_node_option_caption(element *self, element *b) {
        element *opt = target_option(self, b);
        return opt && opt->state.node() &&
               b->belongs_to(opt->first_ui_element(), true);
      }

      bool toggle(view &v, element *self, element *option, bool select_it) {
        if (option->state.expanded())
          return collapse_option(v, self, option, select_it);
        else
          return expand_option(v, self, option, select_it);
      }

      enum OPTION_HIT_AREA {
        CONTENT,
        PLUS_MINUS_SIGN,
        CHECK_BOX
      };

      OPTION_HIT_AREA hit_test(const event_mouse &evt) {
        if (element *target = evt.is_on_icon) {
          if (is_option(target))
            return is_node(target) ? PLUS_MINUS_SIGN : CHECK_BOX;
          if (is_option(target->parent) && (target == target->parent->first_ui_element()))
            return CHECK_BOX;
        }
        return CONTENT;
      }
      
      virtual bool on(view &v, element *self, event_mouse &evt) override {
        element *target = evt.is_on_icon;
        if (target) {
          if (evt.cmd == MOUSE_DOWN || evt.cmd == MOUSE_DCLICK) {
            element *option = target_option(self, target);
            if (option && option->state.node())
              return toggle(v, self, target, false);
          }
          if (evt.cmd == MOUSE_UP && hit_test(evt) == PLUS_MINUS_SIGN) {
            return true;
          }
        }
        return super::on(v, self, evt);
      }

      virtual bool on_option(view &v, element *self, event_mouse &evt,
                             element *opt) override {
        switch (evt.cmd) {
        case MOUSE_DCLICK:
          if (evt.is_point_button()) {
            if (opt->state.node() &&
                is_on_node_option_caption(self, evt.target))
              return toggle(v, self, opt, true);
          }
          break;
        }
        return super::on_option(v, self, evt, opt);
      }

      bool is_node(element *option) { return option->get_state(true).node(); }
      bool is_leaf(element *option) { return !option->get_state(true).node(); }

      virtual bool on(view &v, element *self, event_key &evt) override {

        if (evt.cmd == KEY_DOWN) {
          switch (evt.key_code) {
          case KB_LEFT:
            if (!self->state.rtl())
              return collapse_option(v, self, _current_option, true);
            else
              return expand_option(v, self, _current_option, true);
            break;
          case KB_RIGHT:
            if (!self->state.rtl())
              return expand_option(v, self, _current_option, true);
            else
              return collapse_option(v, self, _current_option, true);
            break;
          }
        }
        return super::on(v, self, evt);
      }

      bool expand_option(view &v, element *self, element *option,
                         bool select_it = false) {
        if (!option) return false;

        if (!is_node(option)) return false;

        if (!option->belongs_to(self)) return false;

        //if (select_it) select_option(v, self, option, 0, SET_ONLY_CURRENT);

        if (option->state.collapsed()) {
          option->state_on(v, S_EXPANDED);
          event_behavior evt(option, option, ELEMENT_EXPANDED, 0);
          v.post_behavior_event(evt);
        } else if (select_it)
          return select_next_option(v, self, +1, 0, 0);
        else
          return false;

        self->commit_measure(v);

        return true;
      }
      bool collapse_option(view &v, element *self, element *option,
                           bool select_it = false) {
        if (!option) return false;
        if (!option->belongs_to(self)) return false;

        //if (!is_node(option)) return false;
        //if (is_leaf(option)) return true;

        if (option->state.expanded()) {
          option->state_on(v, S_COLLAPSED);
          event_behavior evt(option, option, ELEMENT_COLLAPSED, 0);
          v.post_behavior_event(evt);
        } else if (select_it) {
          element *nnc = find_first_parent(v, option->parent, WCHARS("option:node"));
          if (!nnc || !nnc->belongs_to(self)) return false;
          //return collapse_option(v, self, nnc, true);
          return select_option(v, self, nnc, 0, SET_ONLY_CURRENT);
        } else
          return false;

        self->commit_measure(v);
        return true;
      }
    };

    ctl *tree_view_ctl_factory::create(element *el) {
      return new tree_view_ctl();
    }

    // TREE-VIEW with CHECKMARKS

    struct tree_checkmarks_ctl_factory : public ctl_factory {
      tree_checkmarks_ctl_factory() : ctl_factory("tree-checkmarks") {}
      virtual ctl *create(element *el) override;
    };
    tree_checkmarks_ctl_factory *_tree_checkmarks_ctl;

    struct tree_checkmarks_ctl : public tree_view_ctl {
      typedef tree_view_ctl super;
      tree_checkmarks_ctl() {}
      virtual const string &behavior_name() const {
        return _tree_checkmarks_ctl->name;
      }
      virtual CTL_TYPE get_type() { return CTL_SELECT_MULTIPLE; }

      virtual bool select_option(view &v, element *self, element *opt,
                                 uint alts, SET_WHAT what = SET_VALUE) {
        return select_ctl::select_option(v, self, opt, alts, SET_ONLY_CURRENT);
      }

      virtual bool attach(view &v, element *self) override {
        super::attach_multiple(v, self);
        init_options(v, self);
        return true;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        auto_state<bool> _(_in_set_value, true);
        if (set_value_multiple(v, self, val)) {
          init_options(v, self);
          return true;
        }
        return false;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        return get_value_multiple(v, self, val);
      }

      virtual void notify_change(view &v, element *self, element *opt,
        uint reason) override {
          if (_in_set_value) return;
          event_behavior evt(opt, self, CLICK, reason);
          v.post_behavior_event(evt);
      }


      enum NODE_STATE {
        NODE_STATE_UNKNOWN = 0,
        NODE_OFF           = 1,
        NODE_ON            = 2,
        NODE_MIXED         = 3
      };

      static bool is_node(element *t) {
        return (t->tag == tag::T_OPTION) && t->state.node();
      }
      static bool is_leaf(element *t) {
        return (t->tag == tag::T_OPTION) && !t->state.node();
      }

      void set_state(view &v, element *node, NODE_STATE st) {
        NODE_STATE old_st = get_state(node);
        if (old_st == st) return;

        v.refresh(node);
        // uint state = 0;
        switch (st) {
        case NODE_ON:
          node->state.incomplete(false);
          node->state.checked(true);
          break;
        case NODE_MIXED:
          node->state.incomplete(true);
          node->state.checked(false);
          break;
        case NODE_OFF:
          node->state.incomplete(false);
          node->state.checked(false);
          break;
        }
        node->reset_styles(v);

        // node->drop_state_styles(true);
        // node->current_style(*v);
        // v->refresh(node);
      }
      NODE_STATE get_state(element *node) {
        if (node->state.incomplete()) return NODE_MIXED;
        if (node->state.checked()) return NODE_ON;
        return NODE_OFF;
      }

      NODE_STATE init_options(view &v, element *node) {
        int n_off   = 0;
        int n_on    = 0;
        int n_total = 0;

        child_iterator each(v, node, is_option_filter);
        for (element *t; each(t);) {
          NODE_STATE t_state;
          if (is_node(t))
            t_state = init_options(v, t);
          else if (is_leaf(t))
            t_state = get_state(t);
          else
            continue;

          set_state(v, t, t_state);

          switch (t_state) {
          case NODE_STATE_UNKNOWN:
          case NODE_OFF: ++n_off; break;
          case NODE_ON: ++n_on; break;
          }
          ++n_total;
        }

        if (n_off == n_total) return NODE_OFF;
        if (n_on == n_total) return NODE_ON;
        return NODE_MIXED;
      }

      virtual bool on_option(view &v, element *self, event_mouse &evt,
                             element *opt) override {
        switch (evt.cmd) {
        case MOUSE_UP:
          if (evt.is_point_button() && is_node(opt) &&
              evt.is_on_icon == opt->first_ui_element()) {
            toggle_checkmark(v, self, opt);
            return true;
          } else if (evt.is_point_button() && is_leaf(opt) &&
                     evt.is_on_icon == opt) {
            toggle_checkmark(v, self, opt);
            return true;
          }
          break;
          /*case MOUSE_DCLICK:
            if( evt.is_point_button() && opt->tag == tag::T_OPTION)
            {
              toggle_checkmark( v, self, opt );
              return true;
            }
            break;*/
        }
        return super::on_option(v, self, evt, opt);
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.cmd != KEY_DOWN || evt.key_code != KB_SPACE)
          return super::on(v, self, evt);

        // ok, here we have spacebar up
        if (_current_option) {
          toggle_checkmark(v, self, _current_option);
          return true;
        }
        return super::on(v, self, evt);
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == BUTTON_CLICK) {
          toggle_checkmark(v, self, evt.target);
          return true;
        } else if (evt.cmd == CONTENT_CHANGED) {
          init_options(v, self);
        }

        return select_ctl::on(v, self, evt);
      }

      void toggle_checkmark(view &v, element *self, element *leaf) {
        element *node = find_first_parent(v, leaf, WCHARS("option"));
        if (!node || !node->belongs_to(self)) return;

        NODE_STATE old_state = get_state(node);
        NODE_STATE new_state;
        if (old_state == NODE_ON)
          new_state = NODE_OFF;
        else
          new_state = NODE_ON;

        if (is_node(node)) {
          // non-terminal node case
          // do it for all descendants
          each_element each(node);
          for (element *t; each(t);) {
            if (is_option_filter(v, t)) set_state(v, t, new_state);
          }
          // and for itself
          set_state(v, node, new_state);
        } else if (is_leaf(node)) {
          set_state(v, node, new_state);
        }
        // as state was changed we need to update parent chain here too.
        element *p = node->parent;
        while (p && p != self) {
          if (is_node(p)) setup_node(v, p);
          p = p->parent;
        }
        notify_state_changing(v, self, node, new_state == NODE_ON);
      }

      element *find_subitem_with_state(view &v, element *node, NODE_STATE st) {
        element_iterator each(v, node, is_option_filter);
        for (element *t; each(t);)
          if (get_state(t) == st) return t;
        return 0;
      }

      void setup_node(view &v, element *node) {
        element *on    = find_subitem_with_state(v, node, NODE_ON);
        element *off   = find_subitem_with_state(v, node, NODE_OFF);
        element *mixed = find_subitem_with_state(v, node, NODE_MIXED);
        if (mixed || (on && off)) {
          set_state(v, node, NODE_MIXED);
        } else if (on) {
          set_state(v, node, NODE_ON);
        } else {
          set_state(v, node, NODE_OFF);
        }
      }

      /*virtual void set_selected_state(view* v, element* self, element* opt,
      bool current, bool trigger = false, bool animate = true)
      {
          uint cflag = current? S_CURRENT:0;

          if(current)
            ensure_visible(v, self, opt, animate);

          if( trigger )
          {
            if( opt->subs.size() > 1 )
            {
              if( opt->state.expanded() )
                super::set_state(v, self, opt, S_COLLAPSED | cflag);
              else
                super::set_state(v, self, opt, S_EXPANDED | cflag);

            }
            else
              clear_state(v,self,opt,S_EXPANDED | S_COLLAPSED);
          }
          if( !opt->state.current() && current )
          {
            super::set_state(v, self, opt, cflag);
          }
      }

      virtual void reset_selected_state(view* v, element* self, element* opt)
      {
          if( opt->state.current())
          {
            super::clear_state(v,self,opt,S_CURRENT);
          }
      }*/
    };

    ctl *tree_checkmarks_ctl_factory::create(element *el) {
      return new tree_checkmarks_ctl();
    }

    void init_selects() {
      ctl_factory::add(_select_ctl = new select_ctl_factory());
      ctl_factory::add(_select_multiple_ctl =
                           new select_multiple_ctl_factory());
      ctl_factory::add(_select_checkmarks_ctl =
                           new select_checkmarks_ctl_factory());
      ctl_factory::add(_tree_ctl = new tree_ctl_factory());
      ctl_factory::add(_tree_view_ctl = new tree_view_ctl_factory());
      ctl_factory::add(_tree_checkmarks_ctl =
                           new tree_checkmarks_ctl_factory());
      init_dd_selects();
    }

  } // namespace behavior
} // namespace html
