
#include "html.h"

namespace html {
  namespace behavior {

    struct progress_ctl_factory : ctl_factory {
      progress_ctl_factory() : ctl_factory("progress") {}
      virtual ctl *create(element *el) override;
    };

    static progress_ctl_factory *_progress_ctl_factory = 0;

#if defined(SCITERJS) 

    struct progress_ctl : ctl 
    {
      double progress;

      progress_ctl() : progress(0.0) {}

      virtual CTL_TYPE get_type() { return CTL_PROGRESS; }

      virtual const string &behavior_name() const {
        return _progress_ctl_factory->name;
      }

      virtual bool focusable(const element *self) override {
        return false; /*!self->attr_disabled();*/
      }

      virtual bool attach(view &v, element *self) override {
        if (is_indefinite(self))
          self->state_on(v, S_BUSY);
        else {
          self->state_off(v, S_BUSY);
          update_progress(v,self,self->atts.get_float(attr::a_value, 0.0));
        }
        return true;
      }

      double get_max(const element *self) {
        double t = self->atts.get_float("max", 1.0);
        t = self->atts.get_float("maxvalue", t);
        if (t < 0.0000001) t = 0.0000001;
        return t;
      }

      virtual void on_attr_change(view &v, element *self, const name_or_symbol &nm) {
        switch (nm.att_sym) {
          case attr::a_max:
          case attr::a_value: v.post([&]() -> bool { this->attach(v, self); return true; }); break;
          default: break;
        }
      }

      void update_progress(view &v, element *self, double p) {
        double _max = get_max(self);
        if (p <= 0) p = 0;
        if (p >= _max) p = _max;
        if (progress != p) {
          progress = p;
          self->set_style_variable(v, "progress-percent", value::make_percent_length(int(p / _max * 100)));
          v.refresh(self);
        }
      }

      virtual bool set_value(view &v, element *self, const value &val) override {

        bool indefinite = val.is_undefined() || val.is_null();
        
        if (self->state.busy() == indefinite);
        else if (indefinite) self->state_on(v, S_BUSY);
        else self->state_off(v, S_BUSY);
        
        update_progress(v,self,val.get(0.0));
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        val = value(progress);
        return true;
      }

      bool is_indefinite(element *self) {
        return !self->atts.exist(attr::a_max) && !self->atts.exist(attr::a_value);
      }
    };

#else 

    struct progress_ctl : ctl {
      int_v  phase; // animation phase
      double progress;

      progress_ctl() : progress(0.0) {}

      virtual CTL_TYPE get_type() { return CTL_PROGRESS; }

      virtual const string &behavior_name() const {
        return _progress_ctl_factory->name;
      }

      virtual bool focusable(const element *self) override {
        return false; /*!self->attr_disabled();*/
      }

      virtual void detach(view &v, element *self) override {
        if (animator) v.remove_animation(self, animator);
      }
      virtual bool attach(view &v, element *self) override {
        progress = self->atts.get_float(attr::a_value, 0.0);
        // self->state.busy(true);
        return true;
      }

      struct phase_animator : public animation {
        virtual const char *name() { return "progress"; }
        progress_ctl *      prog;
        phase_animator(progress_ctl *p) : prog(p) {}
        virtual uint start(view &v, element *el, const style *nst,
                           const style *ost) {
          return prog->animation_step(v, el);
        }
        virtual uint step(view &v, element *el, uint current_clock) {
          return prog->animation_step(v, el);
        }
        virtual void stop(view &v, element *el) { prog->animator = 0; }
      };

      handle<phase_animator> animator;

      enum BAR_TYPE { BAR_RESIZE = 0, BAR_CLIP = 1, BAR_REPEAT = 2 };

      BAR_TYPE bar_type(const element *self) {
        ustring v;
        if (self->get_attr("-bar", v)) {
          if (v == WCHARS("clip")) return BAR_CLIP;
          if (v == WCHARS("repeat")) return BAR_REPEAT;
        }
        return BAR_RESIZE;
      }

      double get_max(const element *self) {
        double t = self->atts.get_float("max", 1.0);
        t        = self->atts.get_float("maxvalue", t);
        // progress = self->atts.get_float(attr::a_value,0.0);
        return t;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        double dv   = val.get(0.0);
        double _max = get_max(self);
        self->state.busy(true);
        if (dv <= 0) {
          dv = 0;
          self->state.busy(false);
          phase.clear();
        }
        if (dv >= _max) {
          dv = _max;
          self->state.busy(false);
        }
        progress = dv;
        v.refresh(self);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        val = value(progress);
        return true;
      }

      bool is_indefinite(element *self) {
        return !self->atts.exist(attr::a_max) &&
               !self->atts.exist(attr::a_value);
      }

      virtual bool draw_foreground(view &v, element *self, graphics *sf,
                                   point pos) override {
        hstyle cs   = self->get_style(v);
        rect   base = self->foreground_clip_box(v) + pos;
        rect   fill = base;
        rect   t    = base;
        if (cs->fore_image.id.is_defined() || cs->fore_color.is_defined()) {
          if (is_indefinite(self)) 
            goto DRAW_OVERLAY;
          {
            double vmax = get_max(self);
            // double vmax = max(_max,progress);
            if (vmax <= 0.0) vmax = 1.0;
            bool is_rtl = cs->direction == direction_rtl;

            if (bar_type(self) == BAR_CLIP) {
              rect frc = t;
              if (t.width() > t.height()) // horizontal
              {
                if (is_rtl)
                  t.s.x =
                      t.e.x - int((t.width() * progress) / vmax);
                else
                  t.e.x =
                      t.s.x + int((t.width() * progress) / vmax);
              } else {
                t.s.y =
                    t.e.y - int((t.height() * progress) / vmax);
              }
              gool::clipper _(sf, t);
              // self->draw_foreground(v,sf,frc);
              cs->draw_foreground(v, sf, fill = frc, self, pos);

            } else {
              if (t.width() > t.height()) // horizontal
              {
                if (is_rtl)
                  t.s.x =
                      t.e.x - int((t.width() * progress) / vmax);
                else
                  t.e.x =
                      t.s.x + int((t.width() * progress) / vmax);
              } else {
                t.s.y =
                    t.e.y - int((t.height() * progress) / vmax);
              }
              cs->draw_foreground(v, sf, fill = t, self, pos);
              // self->draw_foreground(sf,*v,t);
            }
          }

        DRAW_OVERLAY:
          if ((self->state.busy() || phase.is_defined()) &&
              cs->list_style_image.is_defined()) {
            gool::image *pimg = cs->list_style_image.img();
            if (!pimg) return true;
            gool::size sz = pimg->dimension();
            if (phase.is_undefined()) phase = -sz.x;
            if (!animator) {
              animator = new phase_animator(this);
              v.add_animation(self, animator);
              phase = -sz.x;
            } else if (phase > base.width()) {
              phase.clear();
              return true;
            }

            gool::clipper _(sf, fill);
            gool::point   offset(phase, 0);
            sz.y = fill.height();
            sf->draw(pimg, rect(fill.start() + offset, sz));
          } else if (animator)
            v.remove_animation(self, animator);

          return true;
        }
        return false;
      }

      uint animation_step(view &v, element *self) {
        if (!self->state.busy() && !phase.is_defined()) return 0;
        if (!self->get_style(v)->list_style_image.is_defined()) return 0;
        if (!self->is_drawable(v)) return 0;
        v.refresh(self);
        if (phase.is_defined()) {
          phase = phase + int(v.pixels_per_dip(size(4)).x);
          return ANIMATION_TIMER_SPAN * 2;
        } else
          return 1000;
      }
    };

#endif

    ctl *progress_ctl_factory::create(element *el) {
      return new progress_ctl();
    }

    void init_progress_ctl() {
      ctl_factory::add(_progress_ctl_factory = new progress_ctl_factory());
    }

  } // namespace behavior
} // namespace html
