
#include "html.h"

namespace html {
  namespace behavior {

    datetime_t get_time_zone_shift(view &v, element *self);

    struct output_ctl_factory : ctl_factory {
      output_ctl_factory() : ctl_factory("output") {}
      virtual ctl *create(element *el);
    };

    static output_ctl_factory *_output_ctl_factory = 0;

    struct output_ctl : ctl {
      typedef ctl super;
      value       val;

      output_ctl() {}

      virtual CTL_TYPE get_type() { return CTL_LABEL; }

      virtual const string &behavior_name() const {
        return _output_ctl_factory->name;
      }

      virtual bool focusable(const element *self) { return false; }

      virtual void detach(view &v, element *self) {
        return super::detach(v, self);
      }
      virtual bool attach(view &v, element *self) {
        ustring sval = self->attr_value();
        if (sval.is_defined()) this->set_value(v, self, value(sval));
        return super::attach(v, self);
      }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override {
        chars fname = chars_of(name);
#define ACTION(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        /*        ACTION(0, min)
                {
                  retval = tool::value(value_min);
                  return true;
                }
                ACTION(0, max)
                {
                  retval = tool::value(value_max);
                  return true;
                }
                ACTION(0, step)
                {
                  retval = tool::value(step);
                  return true;
                } */
#undef ACTION
        return super::on_x_method_call(v, self, name, argv, argc, retval);
      }

      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) {
        return ctl::draw_content(v, self, sf, pos);
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        ustring type = self->attr_type();
        output(v, self, val, type);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        val = this->val;
        return true;
      }

      virtual bool set_text(view &v, element *self, wchars t) override {
        bool         model_changed = false;
        handle<text> tn;
        if (self->nodes.size() != 1 || !self->nodes[0]->is_text()) {
          self->clear();
          model_changed = true;
        }
        if (self->nodes.size() == 0) {
          tn = new text(t);
          self->append(tn);
          model_changed = true;
        } else {
          tn = self->nodes[0].ptr_of<text>();
          if (tn->chars() != t) {
            tn->chars     = t;
            model_changed = true;
          }
        }
        if (model_changed) v.add_to_update(self, CHANGES_MODEL);
        return true;
      }

      typedef function<ustring(element *self, value val)> formatter;

      void output(view &v, element *self, value val, const ustring &type) {
        tristate_v is_invalid;
        tristate_v is_negative;
        ustring    lang     = self->get_lang();
        ustring    no_value = self->attr_novalue();
        ustring    format   = self->get_attr("-format");

        this->val = val;

        auto simple_formatter = [](element *self, value val) -> ustring {
          return val.to_string();
        };
        auto integer_formatter = [&](element *self,
                                            value    val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_number()) {
            is_invalid = true;
            return no_value;
          }
          int i       = val.to_int();
          is_invalid  = false;
          is_negative = i < 0;
          return itow(i);
        };
        auto decimal_formatter = [&](element *self, value val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_number() && !val.is_duration() && val.is_angle()) {
            is_invalid = true;
            return no_value;
          }
          double d     = val.to_float();
          is_invalid   = false;
          is_negative  = d < 0;
          ustring fval = ftow(d, W(""), 6);
          if (format.length()) {
            tool::number_format_def nf;
            if (parse_number_format_def(format, nf))
              return tool::format_number_str(lang, fval, &nf);
          }
          return tool::format_number_str(lang, fval);
        };
        auto currency_formatter = [&](element *self, value val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_number()) {
            is_invalid = true;
            return no_value;
          }
          double d     = val.to_float();
          is_invalid   = false;
          is_negative  = d < 0;
          ustring fval = ftow(d, W(""), 6);
          if (format.length()) {
            tool::number_format_def nf;
            if (parse_number_format_def(format, nf))
              return tool::format_currency_str(lang, fval, &nf);
          }
          return tool::format_currency_str(lang, fval);
        };
        auto date_formatter = [&](element *self, value val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_date()) {
            is_invalid = true;
            return no_value;
          }
          is_invalid   = false;
          date_time dt = val.get_date();
          dt.to_timezone(get_time_zone_shift(v, self));
          return tool::format_date(lang, dt, format);
        };
        auto date_local_formatter = [&](element *self, value val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_date()) {
            is_invalid = true;
            return no_value;
          }
          date_time dt = val.get_date();
          is_invalid   = false;
          dt.to_local();
          return tool::format_date(lang, dt, format);
        };
        auto time_formatter = [&](element *self, value val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_date()) {
            is_invalid = true;
            return no_value;
          }
          date_time dt = val.get_date();
          dt.to_timezone(get_time_zone_shift(v, self));
          is_invalid = false;
          return tool::format_time(lang, dt, format);
        };
        auto time_local_formatter = [&](element *self, value val) -> ustring {
          if (val.is_string()) val = value::parse(val.get(W("")));
          if (!val.is_date()) {
            is_invalid = true;
            return no_value;
          }
          date_time dt = val.get_date();
          dt.to_local();
          is_invalid = false;
          return tool::format_time(lang, dt, format);
        };
        auto enum_formatter = [&](element *self, value val) -> ustring {
          return val.to_string();
        };

      formatter fmt = simple_formatter;
      
      //clang-format off 
      fmt   = type == WCHARS("")            ? simple_formatter 
            : type == WCHARS("text")        ? simple_formatter 
            : type == WCHARS("integer")     ? integer_formatter 
            : type == WCHARS("decimal")     ? decimal_formatter 
            : type == WCHARS("currency")    ? currency_formatter 
            : type == WCHARS("date")        ? date_formatter 
            : type == WCHARS("date-local")  ? date_local_formatter 
            : type == WCHARS("time")        ? time_formatter 
            : type == WCHARS("time-local")  ? time_local_formatter 
            : type == WCHARS("enum")        ? enum_formatter 
            : fmt = simple_formatter; 
      //clang-format on 

        set_text(v, self, fmt(self, val));
        if (is_invalid.is_defined())
          is_invalid ? self->state_on(v, S_INVALID) :
                       self->state_off(v, S_INVALID);
        if (is_negative.is_defined())
          is_negative ? self->set_attr(v, "negative", ustring()) :
                        self->remove_attr(v, "negative");
      }

      /*virtual bool max_intrinsic_width(view& v, element* self, int& value)
        {
          return min_intrinsic_width(v,self,value);
        }
      virtual bool min_intrinsic_width(view& v, element* self, int& value)
        {
          return false;
        }
      virtual bool max_intrinsic_height(view& v, element* self, int& value)
        {
           return min_intrinsic_height(v,self,value);
        }
      virtual bool min_intrinsic_height(view& v, element* self, int& value)
        {
          return false;
        } */
    };

    ctl *output_ctl_factory::create(element *el) {
        return new output_ctl();
    }

    void init_output() {
      ctl_factory::add(_output_ctl_factory = new output_ctl_factory());
    }

  } // namespace behavior
} // namespace html
