#include "html.h"

namespace html {
  namespace behavior {

    struct numeric_ctl_factory : ctl_factory {
      numeric_ctl_factory() : ctl_factory("number") {}
      virtual ctl *create(element *el);
    };

    static numeric_ctl_factory *_numeric_ctl_factory = 0;

    struct numeric_ctl : ctl {
      value       current;
      value       step;
      value       minval;
      value       maxval;
      typedef ctl super;

      numeric_ctl() {}

      virtual bool focusable(const element *self) { return true; }

      virtual const string &behavior_name() const {
        return _numeric_ctl_factory->name;
      }

      virtual CTL_TYPE get_type() { return CTL_NUMERIC; }

      virtual bool attach(view &v, element *self) override {
        ctl::attach(v, self);
        child_iterator cid(v, self);
        element *      b = 0;

        value current = attr_value(self);
        value stp     = attr_step(self);

        ustring usmin = get_attr(self, "-min");
        ustring usmax = get_attr(self, "-max");

        ustring val;
        if (current.is_number()) val = current.to_string();

        if (stp.is_defined()) {

          if (self->n_children() != 3 || (cid(b), b->tag != tag::T_CAPTION) ||
              (cid(b), b->tag != tag::T_BUTTON) ||
              (cid(b), b->tag != tag::T_BUTTON)) {
            self->clear(&v);
            element *cap = new element(tag::T_CAPTION);
            cap->append(new text(val));
            if (is_integer())
              cap->atts.set(attr::a_maxlength,
                            itow(max(usmin.size(), usmax.size())));
            // cap->atts
            self->append(cap);
            cap->state.wants_no_focus(true);
            cap->get_style(v);
            cap->state.synthetic(true);
            cap->flags.state_initialized = 1;
            element *b1                  = new element(tag::T_BUTTON);
            b1->state.synthetic(true);
            b1->atts.set(attr::a_class, WCHARS("plus"));
            b1->state.wants_no_focus(true);
            self->append(b1);
            element *b2 = new element(tag::T_BUTTON);
            b2->state.synthetic(true);
            b2->atts.set(attr::a_class, WCHARS("minus"));
            b2->state.wants_no_focus(true);
            self->append(b2);
            // cap->get_style(v);
            // v.add_to_update(self,CHANGES_DIMENSION);
          }

        } else { // no step

          if (self->n_children() != 1 || (cid(b), b->tag != tag::T_CAPTION)) {
            self->clear(&v);
            element *cap = new element(tag::T_CAPTION);
            cap->append(new text(val));
            if (is_integer())
              cap->atts.set(attr::a_maxlength,
                            itow(max(usmin.size(), usmax.size())));
            cap->state.synthetic(true);
            cap->flags.state_initialized = true;
            self->append(cap);
            cap->state.wants_no_focus(true);
            cap->get_style(v);
            // v.add_to_update(self,CHANGES_DIMENSION);
          }
        }
        if (current.is_number()) show_value(v, self, current);
        // self->drop_layout_tree(&v);
        validate(v, self);

        return true;
      }

      virtual bool is_empty(const element *self, bool &yes) override {
        yes = current.is_undefined();
        return true;
      }

      virtual tristate_v is_integer() { return tristate_v(); }

      virtual bool coerce(value &v) {
        if (v.is_number()) {
          tristate_v type = is_integer();

          if (type.is_defined() && type)
            v = v.to_int();
          else if (type.is_defined() && !type)
            v = v.to_float();
          else {
            double d = v.to_float();
            double c = abs(d) - floor(abs(d));
            if (c <= 0.00001)
              v = int(d);
            else
              v = d;
          }
          return true;
        }
        return false;
      }

      virtual void detach(view &v, element *self) override {
        ctl::detach(v, self);
      }

      void notify_changed(view &v, element *self, uint_ptr reason) {
        event_behavior evt(self, self, EDIT_VALUE_CHANGED, reason);
        v.post_behavior_event(evt);
      }

      bool get_auto_height(view &v, element *self, int &value) override {
        return false;
      }
      bool get_auto_width(view &v, element *self, int &value) {
        ustring usm = get_attr(self, "-max");
        ustring usv = get_attr(self, "-value");
        ustring us(L'0', max(usm.size(), usv.size()));
        if (us.length()) {
          element *cap = get_caption(v, self);
          if (cap) {
            handle<text_block> tl = self->create_text_block(v, us);
            value                 = tl->max_content_width(v) + 6;
            rect md               = cap->margin_distance(v);
            value += md.right() + md.left();
          }
          element *btn = get_minus_button(self);
          if (btn) value += btn->max_width(v) + btn->outer_int_x_extra(v);
        } else
          value = 100;
        /*if(_caption && _button && _popup)
        {
          value = _popup->max_width_outer(v) + _button->max_width_outer(v);
          rect md = _caption->margin_distance(v);
          value += max(md.right(),md.left());
        }*/
        return true;
      }

      /*virtual bool get_value_from_attr(element* self, value& dt)
      {
        string str = self->attr_value();
        if(str.length())
        {
          dt = value::parse(str);
          return coerce(dt);
        }
        return false;
      }*/

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        value t = val;
        coerce(t);
        set_caption_value(v, self, t);
        validate(v, self);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        get_caption_value(v, self, val);
        return true;
      }

      element *get_caption(view &v, element *self) {
        child_iterator cit(v, self);
        for (element *b; cit(b);) {
          if (b->tag == tag::T_CAPTION) return b;
        }
        return 0;
      }

      element *get_minus_button(element *self) {
        element *b = self->child(2);
        assert(!b || b->tag == tag::T_BUTTON);
        if (!b || b->tag != tag::T_BUTTON) return 0;
        return b;
      }
      element *get_plus_button(element *self) {
        element *b = self->child(1);
        assert(!b || b->tag == tag::T_BUTTON);
        if (!b || b->tag != tag::T_BUTTON) return 0;
        return b;
      }

      void set_caption_value(view &v, element *self, const value &val) {
        v.refresh(self);
        show_value(v, self, val);
        
        self->drop_styles(v);
        self->commit_measure(v);
      }

      /*void set_caption_value(view& v, element* self)
      {
        element *cd = find_first(self,"td:current");
        if( cd )
        {
          ustring val = cd->attr_value();
          set_caption_value(pv, self, val );
        }
      }*/

      virtual bool child_focusable(const element *self, const element *child) {
        return false;
      }

      /*    virtual bool on_x_method_call(view& v, element *self, const char*
      name, const value* argv, size_t argc, value& retval)
          {
              chars fname = chars_of(name);
      #define ACTION(ARGC, NAME) if( argc == ARGC && fname == CHARS(#NAME))
              ACTION(0, now)
              {
                 time dt; dt.set_now();
                 set_caption_value(v, self, dt);
                 return true;
              }
      #undef ACTION
               return ctl::on_x_method_call(v, self, name, argv, argc, retval);
          }*/

      void show_value(view &v, element *self, const value &val) {
        element *cap = get_caption(v, self);
        if (!cap) return;
        current = val;
        if(current.is_defined())
          coerce(current);
        cap->set_value(v, current);
      }

      bool get_caption_value(view &v, element *self, value &val) {
        element *cap = get_caption(v, self);
        if (!cap) return false;

        cap->get_value(v, val);
        // wchars text = cap->get_text(v);
        val = value::parse(val.to_string());
        return val.is_number();
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        if (evt.cmd == MOUSE_DOWN || evt.cmd == (MOUSE_DOWN | EVENT_HANDLED) ||
            evt.cmd == MOUSE_DCLICK ||
            evt.cmd == (MOUSE_DCLICK | EVENT_HANDLED)) {
          if (evt.target->belongs_to(self, true)) {
            element *cap = get_caption(v, self);
            v.set_focus(self, BY_MOUSE);
            if (cap && evt.target == cap) {
              activate_caption(v, self, true, BY_MOUSE);
              return true;
            }
          }
        } else if (evt.cmd == MOUSE_WHEEL && self->is_safe_to_wheel(v)) {
          do_increment(v, self, evt.get_wheel_delta() > 0, CLICK_BY_MOUSE);
          return true;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_command &evt) override {
        element *cap = get_caption(v, self);
        if (cap) return cap->on(v, evt);
        return false;
      }

      void activate_caption(view &v, element *self, bool on,
                            FOCUS_CAUSE cause) {
        element *cap = get_caption(v, self);
        if (!cap) return;
        event_focus evt(cap, on ? GOT_FOCUS : LOST_FOCUS, cause, true);
        cap->state.current(on);
        cap->on(v, evt);
        // cap->drop_style(&v);
        // cap->commit_measure(v);
        /*if(on)
        {
          //dbg_printf("caption activated\n");
          v.set_focus(self,cause);
        }*/
      }

      virtual bool on(view &v, element *self, event_focus &evt) {
        if (evt.cmd == FOCUS_IN || evt.cmd == (FOCUS_IN | EVENT_HANDLED))
          activate_caption(v, self, true, evt.cause);
        else if (evt.cmd == FOCUS_OUT || evt.cmd == (FOCUS_OUT | EVENT_HANDLED))
          activate_caption(v, self, false, evt.cause);
        return false;
      }

      value attr_step(element *self) {
        ustring us = get_attr(self, "-step");
        value   t  = value::parse(us);
        coerce(t);
        return t;
      }
      value attr_min(element *self) {
        ustring us = get_attr(self, "-min");
        value   t  = value::parse(us);
        coerce(t);
        return t;
      }
      value attr_max(element *self) {
        ustring us = get_attr(self, "-max");
        value   t  = value::parse(us);
        coerce(t);
        return t;
      }
      value attr_value(element *self) {
        ustring us = get_attr(self, "-value");
        value   t  = value::parse(us);
        coerce(t);
        return t;
      }

      bool do_increment(view &v, element *self, bool plus, int reason) {
        value stp = attr_step(self);
        if (!stp.is_number()) return false;

        value val;
        if (!get_caption_value(v, self, val)) val = value(0);

        value pval = val;

        value minv = attr_min(self);
        value maxv = attr_max(self);

        if (is_integer()) {
          int n = val.get_int();
          int s = stp.to_int();
          if (plus)
            n += s;
          else
            n -= s;
          val = value(n);
          if (maxv.is_number() && n > maxv.to_int()) val = maxv;
          if (minv.is_number() && n < minv.to_int()) val = minv;
        } else {
          double n = val.to_float();
          double s = stp.to_float();
          if (plus)
            n += s;
          else
            n -= s;
          val = value(n);
          if (maxv.is_number() && n > maxv.to_float()) val = maxv;
          if (minv.is_number() && n < minv.to_float()) val = minv;
        }
        set_caption_value(v, self, val);
        validate(v, self);
        if (pval != val) notify_changed(v, self, reason);
        return true;
      }

      void validate(view &v, element *self) {
        value val, minval = attr_min(self), maxval = attr_max(self);
        get_caption_value(v, self, val);
        bool invalid = !val.is_number();
        if (minval.is_defined() && val.to_float() < minval.to_float())
          invalid = true;
        if (maxval.is_defined() && val.to_float() > maxval.to_float())
          invalid = true;

        bool empty = val.is_undefined();
        current    = val;

        if (!empty) {
          if (!invalid && self->state.invalid())
            self->state_off(v, S_INVALID);
          else if (invalid && !self->state.invalid())
            self->state_on(v, S_INVALID);
        } else if (self->state.invalid())
          self->state_off(v, S_INVALID);

        if (empty && !self->state.empty())
          self->state_on(v, S_EMPTY);
        else if (!empty && self->state.empty())
          self->state_off(v, S_EMPTY);
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == BUTTON_PRESS && evt.target == get_minus_button(self))
          return do_increment(v, self, false, CLICK_SYNTHESIZED);
        else if (evt.cmd == BUTTON_PRESS && evt.target == get_plus_button(self))
          return do_increment(v, self, true, CLICK_SYNTHESIZED);
        else if (evt.cmd == EDIT_VALUE_CHANGED &&
                 evt.target == get_caption(v, self)) {
          validate(v, self);
          notify_changed(v, self, evt.reason);
          return true;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.cmd == KEY_DOWN && evt.get_key_code() == KB_DOWN)
          return do_increment(v, self, false, CLICK_BY_KEY);
        else if (evt.cmd == KEY_DOWN && evt.get_key_code() == KB_UP)
          return do_increment(v, self, true, CLICK_BY_KEY);

        element *cap = get_caption(v, self);
        if (cap) {
          cap->on(v, evt);
          value val;
          get_caption_value(v, self, val);
          v.refresh(self);
          self->drop_styles(v);
          current = val;
          v.refresh(self);
        }
        return false;
      }
    };

    ctl *numeric_ctl_factory::create(element *el) { return new numeric_ctl(); }

    struct integer_ctl_factory : ctl_factory {
      integer_ctl_factory() : ctl_factory("integer") {}
      virtual ctl *create(element *el);
    };
    static integer_ctl_factory *_integer_ctl_factory = 0;

    struct integer_ctl : numeric_ctl {
      virtual const string &behavior_name() const override {
        return _integer_ctl_factory->name;
      }
      virtual tristate_v is_integer() { return tristate_v(1); }
    };
    ctl *integer_ctl_factory::create(element *el) { return new integer_ctl(); }

    struct decimal_ctl_factory : ctl_factory {
      decimal_ctl_factory() : ctl_factory("decimal") {}
      virtual ctl *create(element *el);
    };
    static decimal_ctl_factory *_decimal_ctl_factory = 0;

    struct decimal_ctl : numeric_ctl {
      virtual const string &behavior_name() const override {
        return _decimal_ctl_factory->name;
      }
      virtual tristate_v is_integer() { return tristate_v(0); }
    };
    ctl *decimal_ctl_factory::create(element *el) { return new decimal_ctl(); }

    void init_numeric_edit() {
      ctl_factory::add(_numeric_ctl_factory = new numeric_ctl_factory());
      ctl_factory::add(_integer_ctl_factory = new integer_ctl_factory());
      ctl_factory::add(_decimal_ctl_factory = new decimal_ctl_factory());
    }

  } // namespace behavior
} // namespace html
