#include "html.h"
#include "html-behaviors.h"

#if defined(USE_LOTTIE)

#include "external/rlottie/inc/rlottie.h"

extern "C" unsigned char *lottie_image_load(char const *filename, int *x, int *y, int *comp, int req_comp)
{
  return nullptr;
}
extern "C" unsigned char *lottie_image_load_from_data(const char *imageData, int len, int *x, int *y, int *comp, int req_comp)
{
  tool::handle<gool::image> pimg = gool::image::create(tool::bytes((const byte*)imageData,len), tool::string());
  if (!pimg) return nullptr;
  if (!pimg->is_bitmap()) return nullptr;
  gool::bitmap* pbmp = pimg.ptr_of<gool::bitmap>();
  auto bytes = pbmp->pixels_as_bytes();
  byte* pdata = (byte*)malloc(bytes.length);
  tool::target_bytes(pdata, bytes.length).copy(bytes);
  *x = pimg->dim().x;
  *y = pimg->dim().y;
  *comp = 0;
  return pdata;
}
extern "C" void lottie_image_free(unsigned char *data)
{
  if (data)
    free(data);
}

namespace html 
{
  namespace behavior 
  {

    //|
    //| lottie animation
    //|
    struct lottie_ctl_factory : public ctl_factory {
      lottie_ctl_factory() : ctl_factory("lottie") {}
      virtual ctl *create(element *) override;
    };

    static lottie_ctl_factory *_lottie_ctl = 0;

    struct lottie_ctl : public ctl, public animation 
    {
      view*    pv = nullptr;
      element* self = nullptr;

      string url;
      
      bool  loop = false;
      bool  autoplay = false;
      float speed = 1.0f;

      bool playing = false;
      bool forward = true;

      //bool use_cache = true;

      uint frame_no = 0;
      uint total_frames = 0;
      uint first_frame = 0; // range to play
      uint last_frame = 0;  // range to play
      
      uint frame_duration = 0; // ms
      uint total_duration = 0; // ms
      size frame_size;
      
      std::unique_ptr<rlottie::Animation> lottie_animation;
      std::future<rlottie::Surface>       lottie_future_surface;
      rlottie::Surface                    lottie_surface;
      handle<bitmap> current_frame;
      handle<bitmap> future_frame;

      array<handle<gool::bitmap>> cached_frames;

      virtual CTL_TYPE      get_type() override { return CTL_IMAGE; }
      virtual const string &behavior_name() const override { return _lottie_ctl->name; }

      virtual bool is_finite(view &, element *) override { return false; }

      virtual bool attach(view &v, element *self) override {
        this->pv = &v;
        this->self = self;
        ctl::attach(v, self);
        process_attributes(v, self);
        return true;
      }

      virtual void detach(view &v, element *self) {
        if (lottie_future_surface.valid())
          lottie_future_surface.get();
        v.remove_animation(self, this);
        self = nullptr;
        pv = nullptr;
        ctl::detach(v, self);
      }

      void calc_frame_size() {

        if (!lottie_animation)
          return;

        size_t w = 0, h = 0;
        lottie_animation->size(w, h);

        size frame_size;
        
        float ratio = (h == 0) ? 1.0f : float(w) / float(h);

        size rcdim = self->dim();
        if (rcdim.empty()) {
          rcdim.x = self->declared_width(*pv, 0);
          rcdim.y = self->declared_height(*pv, 0);
          if (rcdim.empty()) {
            rcdim.x = int(w);
            rcdim.y = int(h);
          }
        }

        const style* cs = self->get_style(*pv);

        if (cs->fore_image.dim[0].is_literal(size_v::special_values::$cover)) {
          float rcratio = (rcdim.y == 0)
            ? 1.0f
            : float(rcdim.x) / float(rcdim.y);
          if (ratio > rcratio) {
            frame_size.y = rcdim.y;
            frame_size.x = int(frame_size.y * ratio + 0.5f);
          }
          else {
            frame_size.x = rcdim.x;
            frame_size.y = int(frame_size.x / ratio + 0.5f);
          }
        }
        else if (cs->fore_image.dim[0].is_literal(size_v::special_values::$contain)) {
          float rcratio = (rcdim.y == 0)
            ? 1.0f
            : float(rcdim.x) / float(rcdim.y);
          if (ratio > rcratio) {
            frame_size.x = rcdim.x;
            frame_size.y = int(frame_size.x / ratio);
          }
          else {
            frame_size.y = rcdim.y;
            frame_size.x = int(frame_size.y * ratio);
          }
        }
        else {
          frame_size.x = int(w);
          frame_size.y = int(h);
          frame_size = pv->pixels_per_dip(frame_size);
        }

        if (frame_size != this->frame_size) {
          reset_cache();
          this->frame_size = frame_size;
        }
      }

      //handle<gool::bitmap> get_cached_frame(uint fno) {
      //
      //}

      void reset_cache() {
        cached_frames.clear();
        if (need_cache())
          cached_frames.size(total_frames);
      }

      bool need_cache() const { return loop; }
      
      handle<gool::bitmap> get_cached(uint fno) {
        return need_cache() ? cached_frames[fno] : handle<gool::bitmap>();
      }

      void set_cached(uint fno, handle<gool::bitmap> bmp) {
        if (need_cache())
          cached_frames[fno] = bmp;
      }
      
      virtual void on_size_changed(view &v, element *self) override {
        calc_frame_size();
        reset_cache();
      }

      // animation::
      virtual gool::image *provide_fore_image() override { 
        return current_frame; 
      }

      // ctl::
      virtual gool::image *get_fore_image(view &v, element *self) override {
        return current_frame;
      }

      bool start_animation() {
        if (!self) return true;
        if (playing) return false;
        view *pv = self->pview();
        stop_animation();
        show_frame(frame_no);
        if (pv) pv->add_animation(self, this);
        return true;
      }

      bool stop_animation() {
        if (!self) return true;
        if (!playing) return false;
        view *pv = self->pview();
        if (pv) pv->remove_animation(self, this);
        reset_cache();
        return true;
      }

      virtual void on_attr_change(view &v, element *self, const name_or_symbol &nm) {
        event_behavior evt(self, self, CONTENT_MODIFIED, 0);
        v.post_behavior_event(evt, true);
      }

      void process_attributes(view &v, element *self) {
        autoplay = self->atts.get_bool("autoplay");
        loop = self->atts.get_bool("loop");
        speed = self->atts.get_float("speed",1.0f);
        if (auto pd = self->doc()) {
          string url = self->atts.get_url(pd->uri().src, attr::a_src);
          load(url());
        }
      }

      void handle_request(request* prq) {
        if (!self)
          return;
        std::string json = std::string((const char*)prq->data.cbegin(), prq->data.length());
        std::string url = std::string(prq->used_url().src);
        std::unique_ptr<rlottie::Animation> la = rlottie::Animation::loadFromData(json, url, url);
        if (la) {
          lottie_animation.swap(la);
          total_frames = (uint)lottie_animation->totalFrame();
          frame_no = 0;
          frame_duration = (uint)(1000 / lottie_animation->frameRate());
          total_duration = (uint)(lottie_animation->duration() * 1000);
          stop_animation();
          reset_cache();
          calc_frame_size();
          current_frame = future_frame = nullptr;
          first_frame = 0;
          last_frame = total_frames - 1;
          process_children();
          if (autoplay) {
            if (forward)
              show_frame(first_frame);
            else
              show_frame(last_frame);
            start_animation();
          }
          else
            show_frame(0);
        }
      }

      bool   update(wchars path, ustring prop, value val)
      {
#if 0
        FillColor,     /*!< Color property of Fill object , value type is rlottie::Color */
          FillOpacity,   /*!< Opacity property of Fill object , value type is float [ 0 .. 100] */
          StrokeColor,   /*!< Color property of Stroke object , value type is rlottie::Color */
          StrokeOpacity, /*!< Opacity property of Stroke object , value type is float [ 0 .. 100] */
          StrokeWidth,   /*!< stroke with property of Stroke object , value type is float */
          TrAnchor,      /*!< Transform Anchor property of Layer and Group object , value type is rlottie::Point */
          TrPosition,    /*!< Transform Position property of Layer and Group object , value type is rlottie::Point */
          TrScale,       /*!< Transform Scale property of Layer and Group object , value type is rlottie::Size. range[0 ..100] */
          TrRotation,    /*!< Transform Scale property of Layer and Group object , value type is float. range[0 .. 360] in degrees*/
          TrOpacity      /*!< Transform Opacity property of Layer and Group object , value type is float [ 0 .. 100] */
#endif

        if (!lottie_animation)
          return false;
        std::string apath = u8::cvt(path).to_std();
        prop.to_lower();
        val.isolate();

        element_context ctx(*pv, self);
        val = resolve_var(ctx, val);

        if (val.is_color()) {
          //color_v c;
          //color_value(c, val);
          //gool::argb rlc = c.to_argb();
          gool::argb rlc = ctx.color_value(val);
          rlottie::Color lc(rlc.red / 255.0f, rlc.green / 255.0f, rlc.blue / 255.0f);
          if (prop == WCHARS("fillcolor"))
            lottie_animation->setValue<rlottie::Property::FillColor>(apath, lc);
          else if (prop == WCHARS("strokecolor"))
            lottie_animation->setValue<rlottie::Property::StrokeColor>(apath, lc);
          else
            return false;
        }
        else if (val.is_number()) {
          float lf = val.get_float();
          if (prop == WCHARS("fillopacity"))
            lottie_animation->setValue<rlottie::Property::FillOpacity>(apath, lf);
          else if (prop == WCHARS("strokeopacity"))
            lottie_animation->setValue<rlottie::Property::StrokeOpacity>(apath, lf);
          else if (prop == WCHARS("strokewidth"))
            lottie_animation->setValue<rlottie::Property::StrokeWidth>(apath, lf);
          else if (prop == WCHARS("tropacity"))
            lottie_animation->setValue<rlottie::Property::TrOpacity>(apath, lf);
          else if (prop == WCHARS("trrotation"))
            lottie_animation->setValue<rlottie::Property::TrRotation>(apath, lf);
          else
            return false;
        }
        else if (val.is_angle()) {
          float lf = val.get_angle() * 57.2958f;
          if (prop == WCHARS("trrotation"))
            lottie_animation->setValue<rlottie::Property::TrRotation>(apath, lf);
          else
            return false;
        }
        else if (val.is_array(2)) {
          if (prop == WCHARS("tranchor")) {
            rlottie::Point pt = rlottie::Point(val[0].get_float(), val[1].get_float());
            lottie_animation->setValue<rlottie::Property::TrAnchor>(apath, pt);
          }
          else if (prop == WCHARS("trposition")) {
            rlottie::Point pt = rlottie::Point(val[0].get_float(), val[1].get_float());
            lottie_animation->setValue<rlottie::Property::TrPosition>(apath, pt);
          }
          else if (prop == WCHARS("trscale")) {
            rlottie::Size sz = rlottie::Size(val[0].get_float(), val[1].get_float());
            lottie_animation->setValue<rlottie::Property::TrScale>(apath, sz);
          }
          else
            return false;
        }
        else
          return false;
        return true;
      }


      void process_children() {
        document* pd = self->doc();
        if (!pd)
          return;
        self->each_child([this,pd](element* pel) -> bool {
          if (pel->tag != tag::T_PARAM)
            return false;
          ustring path = pel->atts.get_ustring("path");
          if (path.is_undefined()) return false;
          ustring prop = pel->atts.get_ustring("prop");
          if (prop.is_undefined()) return false;
          ustring sval = pel->atts.get_ustring(attr::a_value);
          if (sval.is_undefined()) return false;
          value val;
          css_istream input(sval(), url);
          if (parse_variable_value(pd, url, input, val))
            this->update(path, prop, val);
          return false; // continue
        });
      }

      bool load(chars url) {
        if (this->url == url) 
          return false;
        this->url = url;
        if (!url) {
          current_frame = nullptr;
          pv->remove_animation(self, this);
          return true;
        }
        handle<lottie_ctl> _this = this;
        handle<request> rq = new request(url, DATA_RAW_DATA);
        rq->add([_this](request* prq) -> bool {
          _this->handle_request(prq);
          return true;
        });
        return load_data(rq, pv);
      }

      bool rq_next_frame() 
      {
        if (forward) {
          if (frame_no >= last_frame + 1) {
            if (loop && pv && self) {
              frame_no = first_frame;
              event_behavior evt(self, self, ANIMATION, 2); // animationloop
              evt.data = value(this->name());
              pv->post_behavior_event(evt, true);
            }
            else
              return false;
          }
        }
        else if (frame_no == first_frame) // backward
          return false;
        if (lottie_future_surface.valid())
          lottie_future_surface.get();
       
        if (frame_size.empty()) {
          reset_cache();
          calc_frame_size();
        }

        if (future_frame) {
          current_frame = future_frame;
        }

        if (!get_cached(frame_no)) {
          set_cached(frame_no, future_frame = new gool::bitmap(frame_size));
          rlottie::Surface surface((uint32_t*)future_frame->pixels_as_target_bytes().start, frame_size.x, frame_size.y, future_frame->stride());
          lottie_future_surface = lottie_animation->render(frame_no, surface);
        }
        else
          future_frame = get_cached(frame_no);
        if (forward)
          ++frame_no;
        else {
          if (--frame_no == first_frame) {
            if (loop) {
              frame_no = last_frame;
              event_behavior evt(self, self, ANIMATION, 2); // animationloop
              evt.data = value(this->name());
              pv->post_behavior_event(evt, true);
            }
            else
              return true;
          }
        }
        return true;
      }

      bool show_frame(uint fno) 
      {
        fno = fno % total_frames;
        frame_no = fno;
        current_frame = future_frame = nullptr;
        if (auto fr = get_cached(frame_no))
          current_frame = fr;
        else {
          set_cached(frame_no,current_frame = new gool::bitmap(frame_size));
          rlottie::Surface surface((uint32_t*)current_frame->pixels_as_target_bytes().start, frame_size.x, frame_size.y, current_frame->stride());
          lottie_animation->renderSync(frame_no, surface);
        }
        pv->refresh(self);
        return true;
      }

      virtual const char *name() { return "lottie"; }

      virtual uint start(view &v, element *el, const style *nst,
                         const style *ost) override 
      {
        playing = true;
        if (!rq_next_frame())
          return 0;
        v.refresh(self);
        return max(ANIMATION_TIMER_SPAN,uint(frame_duration / max(0.001f,speed)));
      }

      // animation::
      virtual uint step(view &v, element *self, uint current_clock) override {
        if(!rq_next_frame())
          return 0;
        v.refresh(self);
        return max(ANIMATION_TIMER_SPAN,uint(frame_duration / max(0.001f, speed)));
      }

      virtual void stop(view &v, element *self) override {
        v.refresh(self);
        if (lottie_future_surface.valid())
          lottie_future_surface.get();
        playing = false;
      }

      bool api_play(value first, value last) { 
        if (!lottie_animation)
          return false;
        
        if (first.is_int())         first_frame = first.get<uint>() % total_frames;
        else if (first.is_double()) first_frame = (uint)lottie_animation->frameAtPos(first.get<double>()) % total_frames;
        else                        first_frame = 0;
        
        if (last.is_int())          last_frame = last.get<uint>() % total_frames - 1;
        else if (last.is_double())  last_frame = (uint)lottie_animation->frameAtPos(last.get<double>()) % total_frames - 1;
        else                        last_frame = total_frames - 1;
        
        if (first.is_defined())
          show_frame(first_frame);
        return start_animation();
      }
      bool api_stop() {
        return stop_animation();
      }

      bool api_load(ustring val) {
        return load( u8::cvt(val) );
      }

      bool   api_update(ustring path, ustring prop, value val)
      {
        if (!update(path, prop, val))
          return false;
        pv->refresh(self);
        reset_cache();
        return true;
      }

      value  api_get_duration() {
        return value::make_duration(total_duration / 1000.0);
      }
      int    api_get_frame() {
        return int(frame_no);
      }

      int    api_get_frames() {
        return int(total_frames);
      }

      bool   api_set_frame(int fno) {
        stop_animation();
        return show_frame(uint(fno));
      }
      double api_get_position() {
        return double(frame_no) / total_frames;        
      }
      bool   api_set_position(double progress) {
        if (!lottie_animation) return false;
        stop_animation();
        return show_frame((uint)lottie_animation->frameAtPos(progress));
      }

      value api_get_markers() {
        if (!lottie_animation) return value();
#if 1
        array<value> list;
        for (auto mdef : lottie_animation->markers()) {
          array<value> def(3);
          def[0] = ustring(chars_of(std::get<0>(mdef)));
          def[1] = std::get<1>(mdef);
          def[2] = std::get<2>(mdef);
          list.push(value::make_array(def()));
        }
        return value::make_array(list());
#else
        value map;
        for (auto mdef : lottie_animation->markers()) {
          value def;
          def.set_item("first", value(std::get<1>(mdef)));
          def.set_item("last", value(std::get<2>(mdef)));
          map.set_item(value(ustring(chars_of(std::get<0>(mdef)))),def);
        }
        return map;
#endif
      }

      bool api_playing() {
        return playing;
      }

      SOM_PASSPORT_BEGIN_EX(lottie, lottie_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(load, api_load),
          SOM_FUNC_EX(play, api_play),
          SOM_FUNC_EX(stop, api_stop),
          SOM_FUNC_EX(update, api_update)
        )
        SOM_PROPS(
          SOM_PROP(speed),
          SOM_PROP(loop),
          SOM_PROP(forward),
          SOM_RO_VIRTUAL_PROP(playing, api_playing),
          SOM_RO_VIRTUAL_PROP(duration, api_get_duration),
          SOM_RO_VIRTUAL_PROP(markers, api_get_markers),
          SOM_VIRTUAL_PROP(frame, api_get_frame, api_set_frame),
          SOM_RO_VIRTUAL_PROP(frames, api_get_frames),
          SOM_VIRTUAL_PROP(position, api_get_position, api_set_position)
        )
      SOM_PASSPORT_END


    };

    ctl *lottie_ctl_factory::create(element *) { return new lottie_ctl(); }

    void init_lottie() {
      ctl_factory::add(_lottie_ctl = new lottie_ctl_factory());
    }

  } // namespace behavior
} // namespace html

#endif