#include "html.h"

namespace html {
  namespace behavior {

    struct expandable_ctl_factory : public ctl_factory {
      expandable_ctl_factory() : ctl_factory("expandable-list") {}
      virtual ctl *create(element *e) override;
    };

    static expandable_ctl_factory *_expandable_ctl_factory = 0;

    struct expandable_ctl : ctl {
      virtual bool focusable(const element *self) override { return true; }

      virtual CTL_TYPE get_type() override { return CTL_LIST; }

      virtual const string &behavior_name() const override {
        return _expandable_ctl_factory->name;
      }

      virtual void detach(view &v, element *self) override {}

      virtual bool attach(view &v, element *self) override {
        bool           got_one = false;
        child_iterator cit(v, self);
        for (element *t; cit(t);) {
          if (t->atts.exist("default") && !got_one) {
            t->state.expanded(true); //_on(*v,S_EXPANDED); // set state flags
            got_one = true;
          } else
            t->state.collapsed(true);
          t->reset_styles(v);
        }
        ctl::attach(v, self);
        return true;
      }

      void set_current_item(view &v, element *self, element *item) {
        // get previously selected item:
        element *prev_current = find_first(v, self, WCHARS(":root>:current"));
        element *prev         = find_first(v, self, WCHARS(":root>:expanded"));

        if (prev_current && prev_current != item)
          prev_current->state_off(v, S_CURRENT);

        if (prev) {
          if (prev == item) return; // already here, nothing to do.
          prev->state_off(v, S_CURRENT | S_EXPANDED); // drop state flags
          event_behavior evt(prev, prev, ELEMENT_COLLAPSED, 0);
          v.post_behavior_event(evt, true);
        }
        if (item) {
          item->state_on(v, S_CURRENT | S_EXPANDED); // set state flags
          event_behavior evt(item, item, ELEMENT_EXPANDED, 0);
          v.post_behavior_event(evt, true);
        }
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        if (evt.cmd != MOUSE_DOWN && evt.cmd != MOUSE_DCLICK) return false;
        if (!evt.is_point_button()) return false;

        element *item = target_item(self, evt.target);

        if (item) // click on the item caption
          set_current_item(v, self, item);
        return true;
      }

      element *target_item(element *self, element *target) {
        if (target == self) return 0;

        if (!target) return 0;

        element *target_parent = target->parent;
        if (!target_parent) return target;

        if ((target->tag == tag::T_CAPTION ||
             target->attr_role() == WCHARS("caption") ||
             target->has_class(WCHARS("caption"))) &&
            (target_parent->parent == self))
          return target_parent; // only if click on "caption" element of
                                // immediate child of self.

        return target_item(self, target->parent);
      }

      virtual bool on(view &v, element *self, event_key &evt) override {

        if (evt.cmd != KEY_DOWN) return false;

        switch (evt.key_code) {
        case KB_DOWN: {
          element *c = find_first(v, self, WCHARS(":root>:expanded"));
          if (c) {
            element *nc = c->next_element();
            if (nc) set_current_item(v, self, nc);
          }
        }
          return true;
        case KB_UP: {
          element *c = find_first(v, self, WCHARS(":root>:expanded"));
          if (c) {
            element *nc = c->prev_element();
            if (nc) set_current_item(v, self, nc);
          }
        }
          return true;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_focus &evt) override {
        if (evt.cmd == FOCUS_IN)
          return true;
        else if (evt.cmd == FOCUS_OUT)
          return true;
        return false;
      }
      // virtual bool on_measure (view& v, element* self ) override { return
      // false; }  virtual bool on_timer   (view& v, element* self, uint timer_id,
      // TIMER_KIND kind ) { return false; /*stop this timer*/ }

      virtual bool on_method_call(view &v, element *self, method_params *p) {
        /*switch(p->method_id)
        {
          case DO_CLICK:
              return do_click(v,self,self,false);
        }*/
        return false;
      }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }
    };

    struct collapsible_ctl_factory : public ctl_factory {
      collapsible_ctl_factory() : ctl_factory("collapsible-list") {}
      virtual ctl *create(element *e) override;
    };

    static collapsible_ctl_factory *_collapsible_ctl_factory = 0;

    struct collapsible_ctl : public expandable_ctl {
      virtual const string &behavior_name() const {
        return _collapsible_ctl_factory->name;
      }

      // set current item
      virtual void set_current_item(view &v, element *self, element *item) {
        // get previously expanded item:
        element *prev         = find_first(v, self, WCHARS(":root>:expanded"));
        element *prev_current = find_first(v, self, WCHARS(":root>:current"));

        if (prev_current != item && prev_current)
          prev_current->state_off(v, S_CURRENT);

        if (prev == item) {
          prev->state_on(v, S_CURRENT | S_COLLAPSED);
          event_behavior evt(prev, prev, ELEMENT_COLLAPSED, 0);
          v.post_behavior_event(evt, true);
        } else {
          if (prev_current) prev_current->state_off(v, S_CURRENT | S_EXPANDED);
          item->state_on(v, S_CURRENT | S_EXPANDED); // set new expanded.
          event_behavior evt(item, item, ELEMENT_EXPANDED, 0);
          v.post_behavior_event(evt, true);
        }
      }
    };

/////////////////////////
    struct details_ctl_factory : public ctl_factory {
      details_ctl_factory() : ctl_factory("details") {}
      virtual ctl *create(element *e) override;
    };

    static details_ctl_factory *_details_ctl_factory = 0;

    struct details_ctl : ctl {
      virtual bool focusable(const element *self) override { return true; }

      virtual CTL_TYPE get_type() override { return CTL_LIST; }

      virtual const string &behavior_name() const override {
        return _expandable_ctl_factory->name;
      }

      virtual void detach(view &v, element *self) override {}

      virtual bool attach(view &v, element *self) override {
        tristate_v open = self->atts.get_bool_v("open");
        if (open.is_defined()) {
          if(open)
            self->state_on(v, S_EXPANDED);
          else
            self->state_on(v, S_COLLAPSED);
        }
        ctl::attach(v, self);
        return true;
      }

      virtual void on_attr_change(view &v, element *self, const name_or_symbol &nm) override {
        if (nm == name_or_symbol(CHARS("open"))) {
          bool open = self->atts.get_bool("open");
          if (open)
            self->state_on(v, S_EXPANDED);
          else
            self->state_on(v, S_COLLAPSED);
        }
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override {
        if (evt.cmd == BUTTON_CLICK && evt.target && evt.target->tag == tag::T_SUMMARY) {
          toggle(v, self);
          return true;
        }
        return false;
      }

      void toggle(view &v, element *self) {
        if(self->state.expanded())
          self->state_on(v, S_COLLAPSED);
        else 
          self->state_on(v, S_EXPANDED);
      }
      
      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }
    };

/////////////////////////

    ctl *expandable_ctl_factory::create(element *) {
      return new expandable_ctl();
    }
    ctl *collapsible_ctl_factory::create(element *) {
      return new collapsible_ctl();
    }
    ctl *details_ctl_factory::create(element *) {
      return new details_ctl();
    }

    void init_lists() {
      ctl_factory::add(_expandable_ctl_factory = new expandable_ctl_factory());
      ctl_factory::add(_collapsible_ctl_factory = new collapsible_ctl_factory());
      ctl_factory::add(_details_ctl_factory = new details_ctl_factory());
    }

  } // namespace behavior
} // namespace html
