#include "html.h"

namespace html {
  namespace behavior {

    struct hyperlink_ctl_factory : public ctl_factory {
      hyperlink_ctl_factory() : ctl_factory("hyperlink") {}
      virtual ctl *create(element *el);
    };

    static hyperlink_ctl_factory *_hyperlink_ctl = 0;

    struct hyperlink_ctl : ctl {
      uint key_pressed;
      hyperlink_ctl() : key_pressed(0) {}

      virtual CTL_TYPE get_type() override { return CTL_HYPERLINK; }
      virtual bool     is_atomic() const override { return false; }

      virtual const string &behavior_name() const {
        return _hyperlink_ctl->name;
      }

      virtual bool focusable(const element *self) {
        return !self->attr_disabled();
      }

      virtual void detach(view &v, element *self) {}

      virtual bool set_value(view &pv, element *self, const value &v) override {
        if (v.is_defined())
          self->set_attr(pv, attr::a_href, v.to_string());
        else
          self->remove_attr(pv, attr::a_href);
        return true;
      }
      virtual bool get_value(view &pv, element *self, value &v) override {
        v = ustring(self->atts.get_url(self->doc_url(), attr::a_href));
        return true;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) {
#ifdef SCITERJS
        switch (evt.cmd_bubbling()) 
#else
        switch (evt.cmd)
#endif
        {
        case MOUSE_DCLICK:
        case MOUSE_DOWN:
          if (evt.is_point_button()) {
            v.refresh(self);
            v.set_focus(self, BY_MOUSE);
            v.set_capture(self);
            do_press(v, self, true);
            return true;
          }
          break;
        case MOUSE_UP:
          if (evt.is_point_button()) {
            v.refresh(self);
            v.set_capture(0);
            if (self->state.pressed() && self->state.hover()) {
              self->state.pressed(false);
              do_press(v, self, false);
              do_click(v, self, self, CLICK_BY_MOUSE);
              return true;
            } else
              self->state.pressed(false);
          }
          break;

        case MOUSE_MOVE:
          if (evt.is_point_button()) {
            bool is_here = self->is_inside(v, evt.pos_view);
            if (self->state.hover() != is_here) {
              if (!is_here) {
                self->state.hover(false);
                self->state_off(v, S_ACTIVE);
              } else {
                self->state.hover(true);
                self->state_on(v, S_ACTIVE);
              }
            }
          }
          break;
        case MOUSE_LEAVE: {
          self->state.pressed(false);
        } break;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == ACTIVATE_CHILD && evt.target == self) {
          event_behavior evt(self, self, HYPERLINK_CLICK, CLICK_SYNTHESIZED);
          v.post_behavior_event(evt);
          return true;
        }
        return false;
      }


      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.alt_state) return false;
        if (evt.target != self)
          return false; // handling only key events trgeted to this element!
        switch (evt.cmd) {
        case KEY_DOWN:

          switch (evt.key_code) {
          case KB_RETURN:
          case KB_SPACE:
            key_pressed = evt.key_code;
            self->state_on(v, S_ACTIVE);
            self->state.pressed(true);
            do_press(v, self, true);
            return true;
          }
          break;
        case KEY_UP:
          if (self->state.pressed() && uint(evt.key_code) == key_pressed)
            switch (evt.key_code) {
            case KB_RETURN:
            case KB_SPACE: {
              self->state_off(v, S_ACTIVE);
              self->state.pressed(false);
              key_pressed = 0;
              do_press(v, self, false);
              do_click(v, self, evt.target, CLICK_BY_KEY);
              return true;
            }
            }
          key_pressed = 0;
          break;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_focus &evt) {
        v.refresh(self);
        return false;
      }

      virtual bool on_measure(view &v, element *self) { return false; }
      virtual bool on_timer(view &v, element *self, uint_ptr timer_id,
                            TIMER_KIND kind) {
        return false; /*stop this timer*/
      }

      virtual void do_press(view &v, element *self, bool on) {
        v.update_element(self);
      }
      virtual bool do_click(view &v, element *self, element *target,
                            int reason) {
        event_behavior evt(target, self, HYPERLINK_CLICK, reason);
        v.post_behavior_event(evt);
        return true;
      }

      virtual bool on_method_call(view &v, element *self, method_params *p) {
        switch (p->method_id) {
        case DO_CLICK:
          // v.set_focus(self,BY_CODE);
          do_click(v, self, self, BY_CODE);
          return true;
        }
        return false;
      }

      //virtual bool ssx_disable_reconcilition(element* self, bool& disable) { return false; }
      virtual element* r13n_container(element* self) override { return self; }

    };

    ctl *hyperlink_ctl_factory::create(element *el) {
      return new hyperlink_ctl();
    }

    void init_hyperlink() {
      ctl_factory::add(_hyperlink_ctl = new hyperlink_ctl_factory());
    }

  } // namespace behavior
} // namespace html
