#include "html.h"

namespace html {
  namespace behavior {

    struct frame_ctl_factory : public ctl_factory {
      frame_ctl_factory() : ctl_factory("frame") {}
      virtual ctl *create(element *el);
    };

    frame_ctl_factory *_frame_ctl_factory = 0;

    struct history_ctl_factory : public ctl_factory {
      history_ctl_factory() : ctl_factory("history") {}
      virtual ctl *create(element *el);
    };

    history_ctl_factory *_history_ctl_factory = 0;

    struct frame_set_ctl_factory : public ctl_factory {
      frame_set_ctl_factory() : ctl_factory("frame-set") {}
      virtual ctl *create(element *el);
    };

    frame_set_ctl_factory *_frame_set_ctl_factory = 0;

    struct form_ctl_factory : public ctl_factory {
      form_ctl_factory() : ctl_factory("form") {}
      virtual ctl *create(element *el);
    };
    form_ctl_factory *_form_ctl_factory = 0;

    struct style_bag_ctl_factory : public ctl_factory {
      style_bag_ctl_factory() : ctl_factory("style-bag") {}
      virtual ctl *create(element *el);
    };
    style_bag_ctl_factory *_style_bag_ctl_factory = 0;

    struct history_item {
      handle<element>  frm;
      handle<document> doc;
      point            pos; // scroll pos
      point            dim; // dimensions of the view
      handle<element>  focus;
    };

    struct history_ctl : ctl {
      typedef ctl super;

      weak_handle<element> self;
      array<history_item>  history;
      int                  curpos;

      history_ctl() : curpos(0) {
        // curpos = curpos;
      }
      virtual ~history_ctl() {
        // curpos = curpos;
      }

      virtual CTL_TYPE      get_type() { return CTL_UNKNOWN; }
      virtual const string &behavior_name() const {
        return _history_ctl_factory->name;
      }

      virtual void
      scan_owned_elements(const function<bool(element *)> &scanner) override {
        FOREACH(i, history) {
          history_item &it = history[i];
          if (!it.doc->parent) scanner(it.doc);
        }
      }

      virtual bool attach(view &v, element *self) override {
        this->self = self;
        return super::attach(v, self);
      }

      virtual void detach(view &v, element *self) override {
        FOREACH(i, history) {
          history_item &it = history[i];
          if (!it.doc->parent) it.doc->stray(v);
          it.doc = 0;
          if (!it.frm->parent) it.frm->stray(v);
          it.frm = 0;
        }
        history.clear();
      }

      void notify(view &v, element *self) {
        event_behavior evt(self, self, HISTORY_STATE_CHANGED, 0);
        v.post_behavior_event(evt);
      }

      bool go_next(view &v, element *self) {
        if (curpos < history.size()) {
          history_item         it = history[curpos];
          frame_restore_params prms;
          prms.dim   = it.dim;
          prms.pos   = it.pos;
          prms.doc   = it.doc;
          prms.focus = it.focus;
          if (it.frm->call_behavior_method(v, &prms)) {
            it.dim          = prms.dim;
            it.pos          = prms.pos;
            it.doc          = prms.doc;
            it.focus        = prms.focus;
            history[curpos] = it;
            ++curpos;
            notify(v, self);
            return true;
          }
        }
        return false;
      }

      bool go_prev(view &v, element *self) {
        if (curpos > 0) {
          history_item         it = history[curpos - 1];
          frame_restore_params prms;
          prms.dim   = it.dim;
          prms.pos   = it.pos;
          prms.doc   = it.doc;
          prms.focus = it.focus;

          if (it.frm->call_behavior_method(v, &prms)) {
            it.dim              = prms.dim;
            it.pos              = prms.pos;
            it.doc              = prms.doc;
            it.focus            = prms.focus;
            history[curpos - 1] = it;
            notify(v, self);
            --curpos;
            return true;
          }
        }
        return false;
      }

      bool go(view &v, element *self, int steps) {
        int pos = limit(curpos + steps, 0, history.last_index());
        if (curpos != pos) {
          history_item         it = history[pos];
          frame_restore_params prms;
          prms.dim = it.dim;
          prms.pos = it.pos;
          prms.doc = it.doc;
          prms.focus = it.focus;

          if (it.frm->call_behavior_method(v, &prms)) {
            it.dim = prms.dim;
            it.pos = prms.pos;
            it.doc = prms.doc;
            it.focus = prms.focus;
            history[curpos - 1] = it;
            notify(v, self);
            curpos = pos;
            return true;
          }
        }
        return false;
      }

      bool push_new(view &v, element *self, event_behavior &evt) {
        history.size(curpos);
        history_item it;
        it.frm = evt.target;
        it.dim = evt.target->dim();
        it.doc = evt.source->doc();
        it.pos = it.doc->scroll_pos();
        if (v.get_focus_element() && v.get_focus_element()->belongs_to(self))
          it.focus = v.get_focus_element();
        history.push(it);
        curpos = history.size();
        notify(v, self);
        return true;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        switch (evt.cmd) {
        case HISTORY_PUSH: return push_new(v, self, evt);
        case HISTORY_DROP:
          if (history.size()) {
            history.pop();
            curpos = history.size();
            notify(v, self);
            return true;
          }
          break;
        case HISTORY_PRIOR:
          return go_prev(v, self);
        case HISTORY_NEXT:
          return go_next(v, self);
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.cmd == KEY_DOWN && evt.is_alt()) switch (evt.key_code) {
          case KB_LEFT: {
            event_behavior tevt(self, self, HISTORY_PRIOR, 0);
            v.post_behavior_event(tevt);
            return true;
          }
          case KB_RIGHT: {
            event_behavior tevt(self, self, HISTORY_NEXT, 0);
            v.post_behavior_event(tevt);
            return true;
          }
          }
        return false;
      }

#ifdef SCITER
      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) {
        chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        METHOD(0, goBack) {
          retval = value(go_prev(v, self));
          return true;
        }
        METHOD(0, goForward) {
          retval = value(go_next(v, self));
          return true;
        }
        METHOD(0, canGoBack) {
          retval = value(curpos > 0);
          return true;
        }
        METHOD(0, canGoForward) {
          retval = value(curpos < history.size());
          return true;
        }
#undef METHOD
        return super::on_x_method_call(v, self, name, argv, argc, retval);
      }
#endif

      bool  api_back() {
        if (!self) return false;
        if (auto pv = self->pview())
          return go_prev(*pv, self);
        return false;
      }
      bool  api_forward() {
        if (!self) return false;
        if (auto pv = self->pview())
          return go_next(*pv, self);
        return false;
      }

      int api_get_length() {
        if (!self) return 0;
        if (auto pv = self->pview())
          return curpos;
        return 0;
      }

      int api_get_forward_length() {
        if (!self) return 0;
        if (auto pv = self->pview())
          return history.size() - curpos;
        return 0;
      }

      bool api_go( int steps) {
        if (!self) return 0;
        if (auto pv = self->pview())
          return go(*pv, self, steps);
        return false;
      }


      SOM_PASSPORT_BEGIN_EX(history, history_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(back, api_back),
          SOM_FUNC_EX(forward, api_forward),
          SOM_FUNC_EX(go, api_go),
        )
        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(length, api_get_length),
          SOM_RO_VIRTUAL_PROP(forwardLength, api_get_forward_length),
        )
      SOM_PASSPORT_END
    };

    CTL_TYPE frame_ctl::get_type() { return CTL_FRAME; }

    bool frame_ctl::focusable(const element *self) {
      return !self->state.disabled();
    }

    const string &frame_ctl::behavior_name() const {
      return _frame_ctl_factory->name;
    }

    bool frame_ctl::attach(view &v, element *self) {
      _self = self;
      document *d = self->doc();
      if (!d) return false;
      if (self->atts.exist(attr::a_src)) {
#ifdef SCITERJS
        // to give componentDidMount chance to initialize it
        v.post(delegate(this, &frame_ctl::load_src_document, &v, self), true);
#else
        load_src_document(&v, self);
#endif
      }
      return true;
    }

    bool frame_ctl::load_src_document(view *pv, element *self) {
      document *d = self->doc();
      if (!d) return false;
      string src = self->atts.get_url(d->uri().src, attr::a_src);
      if (src.length()) {
        handle<pump::request> rq = new pump::request(src, DATA_HTML);
        _awaiting = rq;
        rq->dst = self;
        pv->request_data(rq);
      }
      return true;
    }

    document *frame_ctl::root(view &v, element *self) {
      if (self->is_empty()) return 0;
      element *r = self->first_element();
      if (r && r->is_document()) return static_cast<document *>(r);
      return 0;
    }

    bool frame_ctl::on_data_request(view &v, element *self, pump::request *rq) {
      if (rq->data_type == DATA_HTML && rq->dst == self) _awaiting = rq;
      else if (rq->dst == self->first_element() && rq->url.like("parent:*")) {
        string base = self->doc()->uri().src;
        string ru = html::combine_url(base, rq->url()(7));
        rq->url = ru;
      }
      v.notify_data_request(self, rq);
      return false;
    }

    bool frame_ctl::on_data_arrived(view &v, element *self, pump::request *rq) {
      v.notify_data_arrived(self, rq);
      if (rq->data_type == DATA_HTML && !rq->consumed) {
        if (_awaiting == rq) {
          rq->consumed = true;
          _pdocrq = rq;
          load(v, self, rq->real_url(), rq->data(), rq->data_content_encoding);
        }
      }
      return false;
    }

    bool frame_ctl::load(view &v, element *self, const string &url, bytes html,
                         const string &encoding) {
      bool r             = false;
      auto state_changer = [this, &r, &url, &html, &encoding](view &v, element *pel) -> bool {
        r = this->_load(v, pel, url, html, encoding);
        return r;
      };

      if (v.animations_disabled || self->flags.discard_animation)
        state_changer(v, self);
      else
        self->animated_update(v, state_changer);
      return r;
    }

    bool frame_ctl::unload(view &v, document *pdoc) {
      if (pdoc) {
        v.on_before_unload(pdoc);

        mutator_ctx _mut(pdoc, &v);
        pdoc->state.ready(false);
        v.on_unload(pdoc);
        pdoc->remove(true,&v);
        pdoc->pview(0);
        pdoc = 0;
        return true;
      }
      return false;
    }

    bool frame_ctl::_load(view &v, element *self, const string &url, bytes html,
                          const string &encoding) {
      handle<document> prev_d = root(v, self);
      // assert(!br || br->element_type() == BLOCK_HTML);

      bool focus_was_here = v.get_focus_element() &&
                            v.get_focus_element()->belongs_to(self, true);

      if (!v.ask_unload((document *)prev_d.ptr(), view::UNLOAD_BY_LOAD))
        return true;

      _awaiting = 0;

      if (prev_d) {
        event_behavior evt(prev_d, self, HISTORY_PUSH, 0);
        if (v.send_behavior_event(evt)) {
          prev_d->parent = 0;
          //self->clear(&v);
          self->nodes.clear();
        }
        else
          unload(v, prev_d);
        prev_d = nullptr;
      }

      v.purge_deleted_elements();

      if (html.length == 0) return true;

      istream m(html, url, v.debug_mode());

      handle<document> d =
#if defined(SVG_SUPPORT)
          is_svg_markup(m) ? new svg_document(url) :
#endif
          new document(url);

      if (encoding.is_defined())
        m.set_encoding(encoding);
      else
        m.set_utf8_encoding();

      d->_media_vars_provider = this;
      if(_debug_mode || self->atts.get_bool("debug-mode", false))
        d->debug_mode(true);

      ustring dir = self->atts(attr::a_dir);
      if (dir.length()) d->atts.set(attr::a_dir, dir);

      self->append(d,&v);

      swap(d->pdocrq, _pdocrq);

      v.on_load_start(d);

      parse_html(v, m, d);

      if (d->is_empty()) return false;

      string cs = get_content_style_url(self);
      if (cs.length()) {
        string id(char(d->style_serial_no), 1);
        ++d->style_serial_no;

        handle<pump::request> hrq = new pump::request(cs, DATA_STYLE);
        hrq->rq_id = id;
        hrq->dst   = d.ptr();

        if (v.load_data(hrq, true)) {
          ustring      wdata = u8::cvt(hrq->data());
          style_parser stp(id, wdata, d, cs);
          stp.parse(wchars());
        }
      }

      d->setup_layout(v);
      self->get_style(v);

#ifdef DEBUG
      d->dbg_report("frame load");
#endif // DEBUG

      v.on_load_end(d, !!prev_d);

      uint content_flags = CONTENT_ADDED;
      if (prev_d) content_flags |= CONTENT_REMOVED;

      v.on_content_change(d, content_flags);

      self->check_layout(v);
      size sz = self->dim();
      if (!sz.empty()) {
        d->measure(v, sz);
        v.refresh(self);
      }
      if (d->num_resources_requested == 0) v.on_document_complete(d);

      if (focus_was_here) v.set_focus(d, BY_CODE);

      d->operational = true;

#ifdef _DEBUG

      // d->resolve_styles(v);
      // d->styles().usage_stats();
      image::report_number_of_instances();

#endif
      return true;
    }

    void frame_ctl::on_size_changed(view &v, element *self) {
      if (!self->is_empty()) {
        element *root = self->first_element();
        if (root && root->is_document()) {
          document *td = static_cast<document *>(root);
          td->flags.need_delayed_measurement = td->is_large();
          // td->on_window_size(v);
          // td->drop_minmax_dim();
          td->measure(v, self->dim());
        }
      }
    }

    element *frame_ctl::get_history_root(view &v, element *self) {
      element *t = self;
      while (t) {
        if (t->get_named_behavior(_history_ctl_factory->name)) return t;
        t = t->parent;
      }
      t = self;
      while (t) {
        if (t->get_named_behavior(_frame_ctl_factory->name)) return t;
        t = t->parent;
      }
      return v.doc();
    }

    bool frame_ctl::on_method_call(view &v, element *self,
                                   method_params *params) {
      if (params->method_id == FRAME_RESTORE) {
        handle<document> doc = root(v, self);
        handle<document> new_doc;
        handle<element>  focus;
        point            pos;

        if (doc) {
          pos = doc->scroll_pos();
          if (v.get_focus_element() &&
              v.get_focus_element()->belongs_to(doc, true))
            focus = v.get_focus_element();
        }
        size                  dim = self->dim();
        frame_restore_params *frp = (frame_restore_params *)params;
        if (frp->doc == doc) {
          frp->doc->scroll_pos(v, frp->pos);
          new_doc = frp->doc;
        } else {
          // if( doc )
          //  doc->notify_visual_status_change(v, false);

          //self->clear(&v);
          self->nodes.clear();

          if (frp->doc.ptr()) {
            frp->doc->parent = nullptr;
            frp->doc->owner  = nullptr;
            self->append(frp->doc.ptr(),&v);
            // frp->doc->parent = self;
            if (frp->dim != self->dim()) { frp->doc->measure(v, self->dim()); }
            frp->doc->scroll_pos(v, frp->pos);
            // frp->doc->notify_visual_status_change(v, true);
          }
          new_doc  = frp->doc;
          frp->doc = doc;
          v.refresh(self);
        }
        if (frp->focus && frp->focus->pview())
          v.set_focus(frp->focus, BY_CODE);
        else if (new_doc)
          v.set_focus(new_doc, BY_CODE);
        else
          v.set_focus(self, BY_CODE);
        frp->focus = focus;
        frp->pos   = pos;
        frp->dim   = dim;
        return true;
      }
      return false;
    }

    void frame_ctl::on_attr_change(view &v, element *self, const name_or_symbol &nm)
    {
      if (nm == attr::a_src)
        load_src_document(&v, self);
    }

    bool  frame_ctl::api_load_file(ustring in)
    {
      string                out = combine_url(_self->doc()->uri().src, in);
      handle<pump::request> rq = new pump::request(out, DATA_HTML);
      rq->dst = _self;
      if (auto pv = _self->pview()) {
        pv->request_data(rq);
        return true;
      }
      return false;
    }

    bool  frame_ctl::api_load_html(value what, ustring in_url)
    {
      array<byte>  html;
      string  url = combine_url(_self->doc()->uri().src, url::escape(in_url));
      if (what.is_string()) {
        string in_html = u8::cvt(wchars(what.get(W(""))));
        html = in_html.chars_as_bytes();
      }
      else if (what.is_bytes()) {
        html = what.get_bytes();
      }
      else
        return false;
      if (auto pv = _self->pview())
        return load(*pv, _self, url, html(), string());
      return false;
    }

    bool   frame_ctl::api_load_empty() {
      chars html = CHARS("<html><body></body></html>");
      if (auto pv = _self->pview())
        return load(*pv, _self, "about:blank", to_bytes(html));
      return false;
    }

    value  frame_ctl::api_save_bytes()
    {
      html::ostream_8 os;
      os.data.push(BYTES(UTF8_BOM));
      _self->emit_content(os);
      return value::make_bytes(os.data);
    }

    value  frame_ctl::api_save_file(ustring url)
    {
      html::ostream_8 os;
      os.data.push(BYTES(UTF8_BOM));
      _self->emit_content(os);

      ustring fname = url::file_url_to_path(url());
      FILE *fp = 0;
#ifdef WINDOWS
      _wfopen_s(&fp, fname.c_str(), L"wb");
#else
      fp = fopen(u8::cvt(fname), "wb");
#endif
      if (!fp) {
        return value::make_error(W("file I/O failure"));
      }
      else {
        size_t written = fwrite(os.data.head(), os.data.length(), 1, fp);
        fclose(fp);
        if (written)
          return value(true);
        else
          return value::make_error(W("file I/O failure"));
      }
    }

    value  frame_ctl::api_get_mediaVars() {
      tool::value map = tool::value::make_map();
      for (int n = 0; n < _media_vars.size(); ++n) {
        tool::value k = tool::value(_media_vars.key(n));
        map.set_prop(k, _media_vars.value(n));
      }
      return map;
    }
    bool   frame_ctl::api_set_mediaVars(value map)
    {
      if (map.is_map() || map.is_proxy_of_object()) {
        if (auto pv = _self->pview()) {
          set_media_vars(*pv, _self, map, false, true);
          return true;
        }
      }
      return false;
    }

    ustring frame_ctl::api_get_url()
    {
      helement pel = _self->first_element();
      if (pel && pel->is_document())
        return pel.ptr_of<document>()->uri().src;
      else
        return ustring();
    }
    bool    frame_ctl::api_set_url(ustring url)
    {
      string   in = url;
      helement pel = _self->first_element();
      if (pel && pel->is_document()) {
        pel.ptr_of<document>()->uri(tool::url(in));
        return true;
      }
      else
        return false;
    }


    bool frame_ctl::on_x_method_call(view &v, element *self, const char *name,
                                     const value *argv, size_t argc,
                                     value &retval) {
      chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
      METHOD(1, loadDocument) {
        if (argv[0].is_string()) {
          string                in  = argv[0].get(W(""));
          string                out = combine_url(self->doc()->uri().src, in);
          handle<pump::request> rq  = new pump::request(out, DATA_HTML);
          rq->dst                   = self;
          v.request_data(rq);
          retval = value(true);
        } else if (argv[0].is_bytes()) {
          tool::bytes html = argv[0].get_bytes();
          string      url  = self->doc()->uri().src;
          bool        r    = load(v, self, url, html);
          retval           = value(r);
        } else {
          retval = value::make_error(W("string or bytes expected"));
        }
        return true;
      }
      METHOD(2, loadDocument) {
        if (!argv[1].is_string()) return false;
        bytes   html;
        ustring in_url = argv[1].get(W(""));
        string  url = combine_url(self->doc()->uri().src, url::escape(in_url));
        if (argv[0].is_string()) {
          string in_html = u8::cvt(wchars(argv[0].get(W(""))));
          html           = in_html.chars_as_bytes();
        } else if (argv[0].is_bytes()) {
          html = argv[0].get_bytes();
        } else {
          retval = value::make_error(W("string or bytes expected"));
          return true;
        }

        bool r = load(v, self, url, html, string());
        retval = value(r);
        return true;
      }
      METHOD(0, clearDocument) {
        chars html = CHARS("<html><body></body></html>");
        load(v, self, "about:blank", to_bytes(html));
        return true;
      }

      METHOD(1, saveDocument) {
        if (!argv[0].is_string()) {
          retval = value::make_error(W("path is not a string"));
          return true;
        }

        ustring path = argv[0].get(W(""));

        tool::wchars fname = path();
        if (fname.like(W("file://*"))) fname.prune(7);

        html::ostream_8 os;
        os.data.push(BYTES(UTF8_BOM));
        self->emit_content(os);

        FILE *fp = 0;
#ifdef WINDOWS
        _wfopen_s(&fp, fname.start, L"wb");
#else
        fp = fopen(u8::cvt(fname), "wb");
#endif
        if (!fp) {
          retval = value(false);
          return true;
        } else {
          size_t written = fwrite(os.data.head(), os.data.length(), 1, fp);
          fclose(fp);
          if (written)
            retval = value(true);
          else
            retval = value(false);
          return true;
        }
      }

      METHOD(0, saveDocument) {
        html::ostream_8 os;
        os.data.push(BYTES(UTF8_BOM));
        self->emit_content(os);
        retval = value::make_bytes(os.data);
        return true;
      }

      METHOD(0, mediaVars) {
        tool::value map = tool::value::make_map();
        for (int n = 0; n < _media_vars.size(); ++n) {
          tool::value k = tool::value(_media_vars.key(n));
          map.set_prop(k, _media_vars.value(n));
        }
        retval = map;
        return true;
      }
      METHOD(1, mediaVars) {
        if (argv[0].is_map() || argv[0].is_proxy_of_object())
          set_media_vars(v, self, argv[0], false, true);
        return true;
      }
      METHOD(2, mediaVars) {
        bool reset = argv[1].to_bool();
        if (argv[0].is_map() || argv[0].is_proxy_of_object())
          set_media_vars(v, self, argv[0], reset, true);
        return true;
      }
#undef METHOD
      return super::on_x_method_call(v, self, name, argv, argc, retval);
    }

    void frame_ctl::update_media_vars(
        view &v, element *self,
        bool reset_styles) // accepts comma separated list of vars
    {
      // loop for all <html> nodes (e.g. in frames) including the root of the
      // frame itself.
      element *        doc = root(v, self);
      element_iterator bi(v, doc, is_document_filter);
      element *        b = doc;
      do {
        document *pd = static_cast<document *>(b);
        for (int n = 0; n < pd->media_expressions.size(); ++n) {
          handle<eval::conduit> code = pd->media_expressions.elements()[n];
          eval_media_query(&v, pd, code, code->is_true);
        }
      } while (bi(b));

      if (reset_styles) v.add_to_update(doc, true);
    }

    void frame_ctl::set_media_vars(
        view &v, element *self, value map, bool reset,
        bool reapply_styles) // accepts comma separated list of vars
    {
      if (reset)
        _media_vars.clear();

      assert(map.is_map() || map.is_proxy_of_object());

      auto each_name_value = [&](const value &k, const value &v) -> bool {
        ustring us      = k.to_string();
        _media_vars[us] = v;
        return true; // true to continue enumeration, false - to stop
      };
      map.visit(each_name_value);
      update_media_vars(v, self, reapply_styles);
    }

    bool frame_ctl::on(view &v, element *self, event_behavior &evt) {

      handle<document> current_doc = root(v, self);
      if (!current_doc) return false;

      if (evt.cmd == CONTAINER_CLOSING) {
        unload(v, current_doc);
        return true;
      }
      else if ((evt.cmd != HYPERLINK_CLICK) || !evt.source)
        return false;


      string uri = evt.target->atts.get_url(current_doc->uri().src, attr::a_href);

      string target = get_attr(evt.target, "-target");

      if (uri.is_empty()) return false;
      if (uri == CHARS("#")) return false;

      tool::url u;
      u.parse(uri);
      if (u.anchor_only()) {
        element *anchor = find_first(v, current_doc, ustring::format(W("[id='%S'],[name='%S']"), u.anchor.c_str(), u.anchor.c_str()));
        if (anchor) // found
        {
          event_behavior tevt(current_doc, self, HISTORY_PUSH, 1);
          v.send_behavior_event(tevt);
          if (!v.ensure_visible(anchor, true)) {
            event_behavior tevtn(anchor, self, HISTORY_DROP, 1);
            v.send_behavior_event(tevtn);
          }
          return true;
        }
      }
      if (!target.is_empty()) {
        if (target.like("_*")) return false;

        element *frame = find_first(v, get_history_root(v, self), ustring::format(W("frame[id='%S'],frame[name='%S']"), target.c_str(), target.c_str()));
        if (frame) // frame
        {
          handle<pump::request> prq = new pump::request(uri, DATA_HTML);
          prq->dst                  = frame;
          prq->initiator            = evt.target;
          v.request_data(prq);
          return true;
        }
      }
      if (self->parent) // this is frame
      {
        handle<pump::request> rq = new pump::request(uri, DATA_HTML);
        rq->dst                  = self;
        v.request_data(rq);
        return true;
      }
      return false;
    }

    static bool is_splitter(element *self, element *b) {
      return b->parent == self &&
             (b->tag == tag::T_HR || b->tag == tag::T_SPLITTER);
    }

    struct frame_set_ctl : ctl {
      typedef ctl super;

      element* self = nullptr;
      int  pressed_offset;
      int  previous_value;
      int  tracking_no;
      bool change_first;
      int  prev_dim;

      array<value> used_sizes;

      frame_set_ctl() : ctl(0xFFFFFFFF) {
        pressed_offset = 0;
        previous_value = 0;
        tracking_no    = -1;
        change_first   = false;
        prev_dim       = 0;
      }

      virtual CTL_TYPE get_type() { return CTL_FRAMESET; }

      virtual bool focusable(const element *self) {
        return !self->state.disabled();
      }

      virtual element* r13n_container(element* self) override { return self; }

      virtual const string &behavior_name() const {
        return _frame_set_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) {
        this->self = self;
        ctl::attach(v, self);

        /*if(self->element_type() != BLOCK_DIV &&
           self->element_type() != BLOCK_CELL &&
           self->element_type() != BLOCK_WIDGET &&
           self->element_type() != BLOCK_BODY)
        {
           assert(false);
           return ; // only element containers here!
        }*/

        ustring cols = self->atts.get_ustring("cols");
        ustring rows = self->atts.get_ustring("rows");

        array<helement> pans;
        panels(v, self, pans);

        if (cols.length()) {
          parse_cols(cols, v, self, pans);
          self->drop_minmax_dim();
          for (int n = 0; n < pans.size(); ++n) {
            if (n >= used_sizes.size()) break;
            element* b = pans[n];
            if (used_sizes[n].is_defined()) {
              if (!b->a_style) b->a_style = new style_prop_map();
              b->a_style->set(cssa_width, used_sizes[n]);
            }
          }
        } else if (rows.length()) {
          parse_rows(rows, v, self, pans);
          self->drop_minmax_dim();
          for (int n = 0; n < pans.size(); ++n) {
            if (n >= used_sizes.size()) break;
            element* b = pans[n];
            if (used_sizes[n].is_defined()) {
              if (!b->a_style) b->a_style = new style_prop_map();
              b->a_style->set(cssa_width, used_sizes[n]);
            }
          }
        }
        return true;
      }

      virtual void detach(view &v, element *self) {}

      bool panels(view &v, element *self, array<helement> &ps) {
        for (element *b = self->first_element(); b; b = b->next_element()) {
          if (!b || is_splitter(self, b)) continue;
          if (!b->is_it_visible(v)) continue;
          ps.push(b);
        }
        return ps.size() > 0;
      }

      bool on_splitter(element *self, element *t) {
        if (t == self) return true;
        while (t && (t != self)) {
          if (t->parent == self && is_splitter(self, t)) return true;
          t = t->parent;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) {
        // DEBUG: return false;
        // what kind of splitter do we have?
        bool horizontal = self->get_style(v)->flow == flow_horizontal;

        if (!on_splitter(self, evt.target)) return false;

        if (evt.cmd == MOUSE_UP) {
          if ((evt.target == self) && rect(self->dim()).contains(evt.pos))
            evt.cursor =
                cursor::system(horizontal ? cursor_e_resize : cursor_s_resize);
          v.set_capture(0);
          return false;
        }

        if (evt.cmd == MOUSE_CHECK && evt.target == self) {
          if (rect(self->dim()).contains(evt.pos))
            evt.cursor =
                cursor::system(horizontal ? cursor_e_resize : cursor_s_resize);
          return true;
        }

        if ((evt.cmd == MOUSE_MOVE && !evt.is_point_button()) ||
            evt.cmd == MOUSE_DOWN) {
          if (!rect(self->dim()).contains(evt.pos)) return false;
        }

        if (evt.cmd != MOUSE_MOVE && evt.cmd != MOUSE_DOWN) return false;

        array<helement> pans;
        panels(v, self, pans);

        if (pans.size() == 1) {
          element *b = pans[0];
          if (!b->a_style) b->a_style = new style_prop_map();
          b->a_style->set(cssa_width, 1_fx);
          b->a_style->set(cssa_height, 1_fx);
          return false;
        }

        if (pans.size() == 0) { return false; }

        // mouse moved and pressed

        bool need_update = horizontal ? do_horizontal(v, self, evt, pans)
                                      : do_vertical(v, self, evt, pans);

        if (need_update && evt.cmd == MOUSE_MOVE) {
          // size d = self->dim();
          self->ldata->inner_dim = size();
          self->commit_measure(v);
          v.refresh(self);
          event_behavior tevt(self, self, UI_STATE_CHANGED, 0);
          v.post_behavior_event(tevt, true);
        }
        return true; // it is ours - stop event bubbling
      }

      bool do_horizontal(view &v, element *self, event_mouse &evt,
                         array<helement> &pans) {
        if (self->get_style(v)->direction == direction_rtl)
          return do_horizontal_rtl(v, self, evt, pans);

        if ((evt.target == self) && rect(self->dim()).contains(evt.pos))
          evt.cursor = cursor::system(cursor_e_resize);

        if (!evt.is_point_button()) return false;

        if (evt.cmd == MOUSE_DOWN && evt.is_point_button()) {
          int n = 0;
          for (; n < pans.last_index(); ++n) {
            // rect rc(self->dim);
            if (evt.pos.x < pans[n + 1]->pos().x) {
              tracking_no = n;
              break;
            }
          }
          pressed_offset = evt.pos.x;
          v.set_capture_strict(is_splitter(self, evt.target) ? evt.target.ptr()
                                                             : self);
          return false; // don't need updates
        }

        if (tracking_no < 0 || tracking_no > pans.last_index()) return false;

        // int mn = get_max_element_horizontal(v,self,pans);
        // if( mn < 0 )
        //  return false;

        helement to_change;
        change_first = false;
        evt.cursor   = cursor::system(cursor_e_resize);

        /*if( pans.size() > 2 && tracking_no >= pans.last_index() - 1 )
        {
          if( pans.last_index() == tracking_no )
          {
            tracking_no = pans.last_index() - 1;
            to_change = pans[tracking_no];
            change_first = true;
          }
          else
          {
            tracking_no = pans.last_index();
            to_change = pans[tracking_no];
            change_first = false;
          }
        }
        else*/
        if (pans[tracking_no]->get_style(v)->width.is_spring()) {
          if (tracking_no == pans.last_index()) {
            to_change    = pans[tracking_no - 1];
            change_first = true;
          } else {
            to_change    = pans[tracking_no + 1];
            change_first = false;
          }
        } else {
          to_change    = pans[tracking_no];
          change_first = true;
        }


        int w;

        if (change_first) {
          int delta = evt.pos.x - pressed_offset;
          if (delta > 0) {
            if ((self->min_content_width(v) + delta) >= self->dim().x)
              delta = self->dim().x - self->min_content_width(v);
          }
          if (delta < 0) {
            int minw = to_change->min_defined_width(v);
            if ((to_change->dim().x + delta) < minw)
              delta = -(to_change->dim().x - minw);
          }
          w = to_change->dim().x + delta;
          if (w >= 0)
            pressed_offset += delta;
          else
            return false;
        } else {
          int delta = pressed_offset - evt.pos.x;

          if (delta > 0) {
            if ((self->min_content_width(v) + delta) >= self->dim().x)
              delta = self->dim().x - self->min_content_width(v);
          }
          if (delta < 0) {
            int minw = to_change->min_defined_width(v);
            if ((to_change->dim().x + delta) < minw)
              delta = -(to_change->dim().x - minw);
          }

          w = to_change->dim().x + delta;
          if (w >= 0)
            pressed_offset -= delta;
          else
            return false;
        }

        used_sizes.size(pans.size());

        for (int k = 0; k < pans.size(); ++k) {
          element *b = pans[k];
          if (!b) continue;
          int t = b->dim().x;
          if (!b->a_style) b->a_style = new style_prop_map();

          value tl;

          if (b == to_change)
            b->a_style->set(cssa_width, tl = value::make_length(w, value::ppx));
          else if (b->get_style(v)->width.is_spring())
            ; // b->a_style->width.set_flex(1.0f);
          else
            b->a_style->set(cssa_width, tl = value::make_length(t, value::ppx));
          b->drop_style(&v);
          used_sizes[k] = tl;

          //b->drop_minmax_dim();
          b->request_delayed_measure(v, true);
        }
        //on_size_changed_horizontal(v, self, pans);
        //self->request_delayed_measure(v, true);
        return true; // need update
      }


      bool do_horizontal_rtl(view &v, element *self, event_mouse &evt,
                             array<helement> &pans) {
        if (evt.target == self && rect(self->dim()).contains(evt.pos))
          evt.cursor = cursor::system(cursor_w_resize);

        if (!evt.is_point_button()) return false;

        if (evt.cmd == MOUSE_DOWN && evt.is_point_button()) {
          int n = 0;
          for (; n < pans.last_index(); ++n) {
            // rect rc(self->dim);
            if (evt.pos.x > pans[n + 1]->pos().x) {
              tracking_no = n;
              break;
            }
          }
          pressed_offset = evt.pos.x;
          v.set_capture_strict(is_splitter(self, evt.target) ? evt.target.ptr()
                                                             : self);
          return false; // don't need updates
        }

        if (tracking_no < 0 || tracking_no > pans.last_index()) return false;

        int mn = get_max_element_horizontal(v, self, pans);
        if (mn < 0) return false;

        helement to_change;
        change_first = false;

        evt.cursor = cursor::system(cursor_w_resize);

        if (pans.size() > 2 && tracking_no >= pans.last_index() - 1) {
          if (pans.last_index() == mn) {
            tracking_no  = pans.last_index() - 1;
            to_change    = pans[tracking_no];
            change_first = true;
          } else {
            tracking_no  = pans.last_index();
            to_change    = pans[tracking_no];
            change_first = false;
          }
        } else if (tracking_no == mn) {
          to_change    = pans[tracking_no + 1];
          change_first = false;
        } else {
          to_change    = pans[tracking_no];
          change_first = true;
        }

        int w;

        if (change_first) {
          int delta = pressed_offset - evt.pos.x;
          if (delta > 0) {
            if ((self->min_content_width(v) + delta) >= self->dim().x)
              delta = self->dim().x - self->min_content_width(v);
          }
          if (delta < 0) {
            int minw = to_change->min_defined_width(v);
            if ((to_change->dim().x + delta) < minw)
              delta = -(to_change->dim().x - minw);
          }
          w = to_change->dim().x + delta;
          if (w >= 0)
            pressed_offset -= delta;
          else
            return false;
        } else {
          int delta = evt.pos.x - pressed_offset;

          if (delta > 0) {
            if ((self->min_content_width(v) + delta) >= self->dim().x)
              delta = self->dim().x - self->min_content_width(v);
          }
          if (delta < 0) {
            int minw = to_change->min_defined_width(v);
            if ((to_change->dim().x + delta) < minw)
              delta = -(to_change->dim().x - minw);
          }

          w = to_change->dim().x + delta;
          if (w >= 0)
            pressed_offset += delta;
          else
            return false;
        }

        used_sizes.size(pans.size());

        for (int k = 0; k < pans.size(); ++k) {
          element *b = pans[k];
          if (!b) continue;
          int t = b->dim().x;
          value tl;
          if (!b->a_style) b->a_style = new style_prop_map();
          if (b == to_change)
            b->a_style->set(cssa_width, tl = value::make_length(w,value::ppx));
          else if (k == mn)
            b->a_style->set(cssa_width, 1_fx);
          else
            b->a_style->set(cssa_width, tl = value::make_length(t, value::ppx));
          used_sizes[k] = tl;
          b->request_delayed_measure(v, true);
          // b->request_delayed_measure(v);
        }
        return true; // need update
      }

      bool do_vertical(view &v, element *self, event_mouse &evt,
                       array<helement> &pans) {
        if (evt.target == self && rect(self->dim()).contains(evt.pos))
          evt.cursor = cursor::system(cursor_n_resize);

        if (!evt.is_point_button()) return false;

        if (evt.cmd == MOUSE_DOWN && evt.is_point_button()) {
          bool is_on_splitter = is_splitter(self, evt.target);

          int n       = 0;
          tracking_no = -1;

          if (is_on_splitter) {
            tracking_no = pans.get_index(evt.target->prev_element());
          } else {
            for (; n < pans.last_index(); ++n) {
              if (evt.pos.y < pans[n + 1]->pos().y) {
                tracking_no = n;
                break;
              }
            }
          }
          if (tracking_no < 0 || tracking_no > pans.last_index()) return false;

          pressed_offset = evt.pos.y;
          v.set_capture_strict(is_on_splitter ? evt.target.ptr() : self);
          return false; // don't need updates
        }

        if (tracking_no < 0 || tracking_no > pans.last_index()) return false;

        // int mn = get_max_element_vertical(v,self,pans);
        // if( mn < 0 )
        //  return false;

        helement to_change;
        change_first = false;

        evt.cursor = cursor::system(cursor_n_resize);

        /*if( pans.size() > 2 && tracking_no >= pans.last_index() - 1 )
        {
          if( pans.last_index() == mn )
          {
            tracking_no = pans.last_index() - 1;
            to_change = pans[tracking_no];
            change_first = true;
          }
          else
          {
            tracking_no = pans.last_index();
            to_change = pans[tracking_no];
            change_first = false;
          }
        }
        else*/
        if (pans[tracking_no]->get_style(v)->height.is_spring()) {
          if (tracking_no == pans.last_index()) {
            to_change    = pans[tracking_no - 1];
            change_first = true;
          } else {
            to_change    = pans[tracking_no + 1];
            change_first = false;
          }
        } else {
          to_change    = pans[tracking_no];
          change_first = true;
        }

        int h;

        if (change_first) {
          int delta = evt.pos.y - pressed_offset;

          if (delta > 0) {
            if ((self->min_content_height(v) + delta) >= self->dim().y)
              delta = self->dim().y - self->min_content_height(v);
          }

          if (delta < 0) {
            int minh = to_change->min_defined_height(v);
            if ((to_change->dim().y + delta) < minh)
              delta = -(to_change->dim().y - minh);
          }

          h = to_change->dim().y + delta;
          if (h >= 0)
            pressed_offset += delta;
          else
            return false;
        } else {
          int delta = pressed_offset - evt.pos.y;

          if (delta > 0) {
            if ((self->min_content_height(v) + delta) >= self->dim().y)
              delta = self->dim().y - self->min_content_height(v);
          }

          if (delta < 0) {
            int minh = to_change->min_defined_height(v);
            if ((to_change->dim().y + delta) < minh)
              delta = -(to_change->dim().y - minh);
          }

          h = to_change->dim().y + delta;
          if (h >= 0)
            pressed_offset -= delta;
          else
            return false;
        }

        for (int k = 0; k < pans.size(); ++k) {
          element *b = pans[k];
          if (!b) continue;
          int t = b->dim().y;
          if (!b->a_style) b->a_style = new style_prop_map();
          if (b == to_change)
            b->a_style->set(cssa_height, value::make_length(h, value::ppx));
          else if (b->get_style(v)->height.is_spring())
            ;
          else
            b->a_style->set(cssa_height, value::make_length(t, value::ppx));
          b->drop_style();
          b->request_delayed_measure(v,false);
        }
        return true; // need update
      }

      virtual void on_size_changed(view &v, element *self) override {
        bool horizontal = self->get_style(v)->flow == flow_horizontal;
        array<helement> pans;
        panels(v, self, pans);
        if (horizontal)
          on_size_changed_horizontal(v, self, pans);
        else
          on_size_changed_vertical(v, self, pans);
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == VISUAL_STATUS_CHANGED && evt.target->parent == self) {
          array<helement> pans;
          panels(v, self, pans);
          if (pans.size() == 1) {
            element *b = pans[0];
            if (!b->a_style) b->a_style = new style_prop_map();
            b->a_style->set(cssa_width, 1_fx);
            b->a_style->set(cssa_height, 1_fx);
            b->clear_style();
            prev_dim = 100000;
            on_size_changed(v, self);
          } else {
            v.add_to_update(self, true);
          }
        }
        else if (evt.cmd == CONTENT_CHANGED && evt.target == self && (evt.reason & CONTENT_ADDED)) {
          attach(v, self); // to reapply child styles
        }
        return false;
      }

      int get_max_element_horizontal(view &v, element *self,
                                     array<helement> &pans) {
        if (pans.size() <= 0) return -1;
        int mn        = -1;
        int visible_n = -1;
        int maxwidth  = 0;
        for (int i = 0; i < pans.size(); ++i) {
          int t = pans[i]->dim().x;
          if (!pans[i]->is_it_visible(v))
            continue;
          else
            visible_n = i;
          if (t > maxwidth && t > pans[i]->min_width(v)) {
            mn       = i;
            maxwidth = t;
          }
        }
        return mn >= 0 ? mn : visible_n;
      }

      int get_max_element_vertical(view &v, element *self,
                                   array<helement> &pans) {
        if (pans.size() <= 0) return -1;

        int mn        = -1;
        int visible_n = -1;
        int maxheight = 0;
        for (int i = 0; i < pans.size(); ++i) {
          int t = pans[i]->dim().y;
          if (!pans[i]->is_it_visible(v))
            continue;
          else
            visible_n = i;
          if (t > maxheight && t > pans[i]->max_height(v,self->dim().y)) {
            mn        = i;
            maxheight = t;
          }
        }
        return mn >= 0 ? mn : visible_n;
      }

      void on_size_changed_horizontal(view &v, element *self,
                                      array<helement> &pans) {
        bool expanding = prev_dim <= self->dim().x;

        if (expanding) {
          prev_dim = self->dim().x;
          return;
        }

        size content_dim = self->content_dim();

        if (self->dim().x >= content_dim.x || content_dim.x == 0) // there is a free space
        {
          prev_dim = self->dim().x;
          return;
        }

        // no free space

        for (int k = 0; k < pans.size(); ++k) {
          element *b = pans[k];
          if (!b) continue;
          if (!b->is_it_visible(v)) continue;
          int t = b->dim().x;
          if (!b->a_style) b->a_style = new style_prop_map();
          if (!b->get_style(v)->width.is_spring()) {
            t = (t * self->dim().x) / content_dim.x;
            //b->a_style->width.set_pixels(t);
            b->a_style->set(cssa_width, value::make_length(t, value::ppx));
            b->drop_style(&v);
          }
          b->request_delayed_measure(v,true);
        }
        prev_dim = self->dim().x;
      }
      void on_size_changed_vertical(view &v, element *self,
                                    array<helement> &pans) {
        bool expanding = prev_dim < self->dim().y;

        if (expanding) {
          prev_dim = self->dim().y;
          return;
        }

        size content_dim = self->content_dim();

        if (self->dim().y >= content_dim.y || content_dim.y == 0) // there is a free space
        {
          prev_dim = self->dim().y;
          return;
        }

        for (int k = 0; k < pans.size(); ++k) {
          element *b = pans[k];
          if (!b) continue;
          if (!b->is_it_visible(v)) continue;
          int t = b->dim().y;
          if (!b->a_style) b->a_style = new style_prop_map();
          if (!b->get_style(v)->height.is_spring()) {
            t = (t * self->dim().y) / content_dim.y;
            b->a_style->set(cssa_height, value::make_length(t,value::ppx)); //b->a_style->height.set_pixels(t);
            b->drop_style(&v);
          }
          b->request_delayed_measure(v,false);
        }
        prev_dim = self->dim().y;
      }

      void parse_cols(const ustring &def, view &v, element *self,
                      array<helement> &pans) {
        wtokens tz(def, WCHARS(","));
        wchars  t;
        int     n = 0;
        while (tz.next(t)) {
          if (n >= pans.size()) break;
          size_v sv;
          from_string(sv, trim(t));
          element *b = pans[n];
          if (!b) continue;
          if (!b->a_style) b->a_style = new style_prop_map();
          if (sv.is_percent())
            //b->a_style->width.set_flex(sv.percent() / 100.f);
            b->a_style->set(cssa_width, value::make_length(sv.percent() / 100.f,value::sp));
          else
            //b->a_style->width = sv;
            b->a_style->set(cssa_width, sv);
          b->clear_style();
          ++n;
        }
      }
      void parse_rows(const ustring &def, view &v, element *self,
                      array<helement> &pans) {
        wtokens tz(def, WCHARS(","));
        wchars  t;
        int     n = 0;
        while (tz.next(t)) {
          if (n >= pans.size()) break;
          size_v sv;
          from_string(sv, trim(t));
          // declared_sz.push(sv);
          element *b = pans[n];
          if (!b) continue;
          if (!b->a_style) b->a_style = new style_prop_map();
          if (sv.is_percent())
            //b->a_style->height.set_flex(sv.percent() / 100.f);
            b->a_style->set(cssa_height, value::make_length(sv.percent() / 100.f, value::sp));
          else
            //b->a_style->height = sv;
            b->a_style->set(cssa_height, sv);
          b->clear_style();
          ++n;
        }
      }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) {

        chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        METHOD(0, framesetState) {
          bool horizontal = self->get_style(v)->flow == flow_horizontal;
          array<helement> pans;
          panels(v, self, pans);
          value r = value::make_array((uint)pans.length());
          if (horizontal) {
            for (uint n = 0; n < pans.length(); ++n) {
              size_v w = pans[n]->get_style(v)->width;
              if (!w.is_percent() && !w.is_fixed() && !w.is_spring())
                w = size_v(pans[n]->dim().x);
              r.set_element(n, w);
            }
          } else {
            for (uint n = 0; n < pans.length(); ++n) {
              size_v w = pans[n]->get_style(v)->height;
              if (!w.is_percent() && !w.is_fixed() && !w.is_spring())
                w = size_v(pans[n]->dim().y);
              r.set_element(n, w);
            }
          }
          retval = r;
          return true;
        }
        METHOD(1, framesetState) {
          value r = argv[0];

          bool horizontal = self->get_style(v)->flow == flow_horizontal;
          array<helement> pans;
          panels(v, self, pans);

          if (!r.is_array_like() || r.size() != pans.length()) {
            retval = value(false);
            return true;
          }

          if (horizontal)
            for (uint n = 0; n < pans.length(); ++n) {
              value vl = r.get_element(n);
              if (!vl.is_length()) continue;
              if (!pans[n]->a_style) pans[n]->a_style = new style_prop_map();
              //pans[n]->a_style->width = size_v(vl);
              pans[n]->a_style->set(cssa_width, vl);
              pans[n]->reset_style(v);
              // v.add_to_update(pans[n],CHANGES_DIMENSION);
            }
          else
            for (uint n = 0; n < pans.length(); ++n) {
              value vl = r.get_element(n);
              if (!vl.is_length()) continue;
              if (!pans[n]->a_style) pans[n]->a_style = new style_prop_map();
              //pans[n]->a_style->height = size_v(vl);
              pans[n]->a_style->set(cssa_height, vl);
              pans[n]->reset_style(v);
              // v.add_to_update(pans[n],CHANGES_DIMENSION);
            }
          // self->drop_minmax_dim();
          retval = value(true);
          return true;
        }
#undef METHOD
        return super::on_x_method_call(v, self, name, argv, argc, retval);
      }

      value get_state() {
        if (!self) return value();
        view* pv = self->pview();
        if (!pv) return value();

        bool horizontal = self->get_style(*pv)->flow == flow_horizontal;
        array<helement> pans;
        panels(*pv, self, pans);
        value r = value::make_array((uint)pans.length());
        if (horizontal) {
          for (uint n = 0; n < pans.length(); ++n) {
            size_v w = pans[n]->get_style(*pv)->width;
            if (!w.is_percent() && !w.is_fixed() && !w.is_spring())
              w = size_v(pans[n]->dim().x);
            r.set_element(n, w);
          }
        }
        else {
          for (uint n = 0; n < pans.length(); ++n) {
            size_v w = pans[n]->get_style(*pv)->height;
            if (!w.is_percent() && !w.is_fixed() && !w.is_spring())
              w = size_v(pans[n]->dim().y);
            r.set_element(n, w);
          }
        }
        return r;
      }

      bool set_state(value r)
      {
        if (!self) return false;
        view* pv = self->pview();
        if (!pv) return false;

        bool horizontal = self->get_style(*pv)->flow == flow_horizontal;
        array<helement> pans;
        panels(*pv, self, pans);

        if (!r.is_array_like() || r.size() != pans.length())
          return false;

        if (horizontal)
          for (uint n = 0; n < pans.length(); ++n) {
            value vl = r.get_element(n);
            if (vl.is_string()) {
              size_v sv;
              from_string(sv, vl.get_string());
              vl = sv.to_value();
            }
            if (!vl.is_length()) continue;
            if (!pans[n]->a_style) pans[n]->a_style = new style_prop_map();
            pans[n]->a_style->set(cssa_width, vl);
            pans[n]->reset_style(*pv);
          }
        else
          for (uint n = 0; n < pans.length(); ++n) {
            value vl = r.get_element(n);
            if (vl.is_string()) {
              size_v sv;
              from_string(sv, vl.get_string());
              vl = sv.to_value();
            }
            if (!vl.is_length()) continue;
            if (!pans[n]->a_style) pans[n]->a_style = new style_prop_map();
            pans[n]->a_style->set(cssa_height, vl);
            pans[n]->reset_style(*pv);
          }
        return true;
      }

      SOM_PASSPORT_BEGIN_EX(frameset, frame_set_ctl)
        SOM_PROPS(
          SOM_VIRTUAL_PROP(state, get_state, set_state),
        )
      SOM_PASSPORT_END

    };

    /* the form handler */

    struct form_ctl : ctl {
      typedef ctl super;
      element* self = nullptr;
      tool::value initial_values;
      bool        has_file_ctl;

      form_ctl() : has_file_ctl(false) {}

      virtual CTL_TYPE get_type() { return CTL_FORM; }

      // virtual bool focusable(const element* self) { return
      // !self->attr_disabled(); }

      virtual const string &behavior_name() const {
        return _form_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) {
        this->self = self;
        ctl::attach(v, self);
        get_values(v, self, initial_values, 0);
        return true;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        if (val.is_undefined() || val.is_null()) {
          set_values(v, self, initial_values);
          return true;
        } else if (val.is_map() || val.is_proxy_of_object()) {
          initial_values = val;
          set_values(v, self, val);
          return true;
        }
        return false;
      }

      virtual bool get_value(view &v, element *self, value &val) override {
        get_values(v, self, val, 0);
        return true;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.cmd == KEY_DOWN) switch (evt.get_key_code()) {
          case KB_ESCAPE: {
            element *b = find_first(v, self, WCHARS("[role='cancel-button']"));
            if (b) {
              method_params params;
              params.method_id = DO_CLICK;
              return b->call_behavior_method(v, &params);
            }
          } break;
          case KB_RETURN: {
            element *b = find_first(v, self, WCHARS("[role='default-button']"));
            if (b) {
              method_params params;
              params.method_id = DO_CLICK;
              return b->call_behavior_method(v, &params);
            }
          } break;
          }
        return false;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        switch (evt.cmd) {

        case FORM_RESET: {
          if (evt.data.is_undefined())
            set_values(v, self, initial_values);
          else
            set_values(v, self, evt.data);
          event_behavior evt2(evt.target, self, FORM_VALUE_CHANGED, 0);
          v.post_behavior_event(evt2, true);
        }
          return true;

        case FORM_SUBMIT:
          do_send(v, self, evt.data);
          return true;

        case EDIT_VALUE_CHANGED:
        case SELECT_SELECTION_CHANGED:
        case SELECT_VALUE_CHANGED:
        case BUTTON_STATE_CHANGED:
        case FORM_VALUE_CHANGED:
          if (evt.target != self) {
            event_behavior evt2(evt.target, self, FORM_VALUE_CHANGED, 0);
            v.post_behavior_event(evt2, true);
         // NOTE: does not consume all child changes
          }
          break;

        case BUTTON_CLICK:
          if (evt.target) {
            ustring btn_type = evt.target->atts[attr::a_type];
            if (btn_type == WCHARS("submit"))
              return do_submit(v, self, evt.target);
            else if (btn_type == WCHARS("reset"))
              return do_reset(v, self, evt.target);
          }
          break;
        }
        return false;
      }

      bool do_submit(view &v, element *self, element *btn) {
        event_behavior evt(btn, self, FORM_SUBMIT, 0);
        get_values(v, self, evt.data, btn, true /*skip undefined*/);
        v.send_behavior_event(evt);
        return true;
      }

      bool do_reset(view &v, element *self, element *btn) {
        event_behavior evt(btn, self, FORM_RESET, 0);
        evt.data = initial_values;
        v.send_behavior_event(evt);
        return true;
      }

      void do_send(view &v, element *self, const value &params) {
        html::document *d = self->doc();
        if (!d) return;

        string url = self->atts["action"];
        if (url.length())
          url = html::combine_url(d->uri().src, url);
        else
          url = d->uri().src;

        if (!tool::url(url).is_http_or_https())
          return;

        ustring          target         = self->atts["target"];
        handle<document> self_doc       = self->doc();
        helement         target_element = self_doc->parent;

        if (target.length()) {
          // no such things here
          // WCHARS("_blank")
          // WCHARS("_self")
          // WCHARS("_top")
          if (target == WCHARS("_parent")) {
            if (self_doc->parent)
              target_element = self_doc->parent->doc();
            else
              target_element = 0;
          } else if (target == WCHARS("_top"))
            target_element = 0;
          else {
            ustring  sel = ustring::format(W("frame[name='%s'],frame#%s"),
                                          target.c_str(), target.c_str());
            helement t   = find_first(v, v.doc(), sel, false, true);
            if (t) target_element = t;
          }
        }

        bool post      = self->atts("method") == WCHARS("post");
        bool multipart = self->atts("enctype") == WCHARS("multipart/form-data");

        handle<pump::request> prq = new html::pump::request(url, DATA_HTML);

        if (has_file_ctl || multipart) {
          pump::multipart_composer mc(prq);

          auto oneach = [&](const value &key, const value &val) -> bool {
            if (val.is_undefined()) return true;
            string  name = key.to_string();
            ustring sval = val.to_string();
            if (val.is_string() && val.units() == value::UT_FILE_NAME) {
              tool::mm_file mf;
              if (mf.open(sval)) {
                ustring fdrive, fdir, fname, fext;
                split_path(sval, fdrive, fdir, fname, fext);
                mc.add(name, mf, u8::cvt(fname + fext), guess_mime_type(sval, mf));
              }
            } else {
              mc.add(name, sval);
            }
            return true;
          };
          params.visit(oneach);
        } else {
          auto oneach = [&](const value &key, const value &val) -> bool {
            if (val.is_undefined()) return true;
            pump::param p;
            p.name  = key.to_string();
            p.value = val.to_string();
            prq->rq_params.push(p);
            return true;
          };
          params.visit(oneach);
        }

        // ? critical_section cs(v.guard);
        prq->rq_type = (post || has_file_ctl || multipart) ? RQ_POST : RQ_GET;
        prq->dst     = target_element;
        v.request_data(prq);
      }

      bool is_inside_named(const element *self, const element *b) {
        b = b->parent;
        while (b) {
          if (b == self) return false;
          if (b->attr_name().is_defined()) return true;
          b = b->parent;
        }
        return false;
      }

      void get_values(view &v, element *self, value &all, const element *btn,
                      bool skip_undefined = false) {
        map_value *map = new map_value();
        all            = value::make_map(map);
        // array<helement> named;
        // find_all(v,named,self,WCHARS("[name]"),false);

        has_file_ctl = false;
        // for( int i = 0; i < named.size(); ++i)

        auto eachnamed = [&](element *b) -> bool {
          // element* b = named[i];
          (void)b->get_style(v); // to force behaviors to be created.
          CTL_TYPE ct = b->ctl_type(v);

          value rv;

          ustring name = b->atts(attr::a_name);

          if (name.length() == 0) {
            name = b->atts(attr::a_id);
            if (name.length() == 0) return false;
          }
          if (find_first(v, b, WCHARS("[name]"), false, false)) {
            //if (!b->get_value(v, rv, true))
            if (!v.get_element_value(b, rv, true))
              get_values(v, b, rv, btn, skip_undefined); // that is named sub-group that contains other named elements.
          } else if (is_inside_named(self, b))
            return false; // that is inner element inside some group

          else if (ct == CTL_CHECKBOX) {
            if (!b->get_value(v, rv, true)) return false;
          } else if (ct == CTL_RADIO) {
            if (!b->get_value(v, rv, true)) return false;
          } else if (ct == CTL_BUTTON) {
            if (b == btn)
              rv = value(true); // adding submit button with 'true'
            else
              return false;
          } else if (ct == CTL_FILE) {
            has_file_ctl = true;
            if (!b->get_value(v, rv, true)) return false;
          } else {
            //if (!b->get_value(v, rv,true)) return false;
            if (!v.get_element_value(b, rv, true)) return false;
          }

          if (skip_undefined && rv.is_undefined()) return false;
          value &rrv = map->params[value(name, value::UT_SYMBOL)];
          if (rrv.is_undefined()) rrv = rv;
          return false;
        };

        find_all(v, self, WCHARS("[name]"), eachnamed, false);
      }

      /*element* find_by_name_or_id(element* self, wchars name ) {
        each_element gen(self);
        for( element* el; gen(el); )
        {
          ustring aname = el->attr_name();
          if( aname == name )
            return el;
          if( aname.length() )
            gen.dont_go_inside = true;
          aname = el->attr_id();
          if( aname == name )
            return el;
        }
        return nullptr;
      }
      element* find_ctl(element* self, wchars path) {
         for(wchars part = path.chop('.'); !!part; part = path.chop('.')) {
           element * pel = find_by_name_or_id(self,part);
           if( !pel )
              return 0;
           self = pel;
         }
         return self;
      }*/

#if SCITERJS
      void set_values(view &v, element *self, const value &all) {
        has_file_ctl = false;
        auto oneach  = [&](const value &key, const value &val) -> bool {
          ustring path = key.get(W(""));
          ustring sel  = ustring::format(W("[name='%s']"), path.c_str());

          helement ctl = find_first(v, self, sel, false);
          // helement ctl = find_ctl(self,path);

          if (!ctl) return true;

          ctl->get_style(v); // to force behaviors to be resolved.

          int ctl_type = ctl->ctl_type(v);

          if (find_first(v, ctl, WCHARS("[name]"), false, false)) {
            //if (ctl->set_value(v, val, true))
            if(v.set_element_value(ctl,val,true))
              ;
            else if(val.is_map_alike())
              set_values(v, ctl, val);
            return true;
          } else if (is_inside_named(self, ctl))
              return true; // that is inner element inside some group

          if (ctl_type == CTL_BUTTON) {
            return true; // skip buttons
          }
          else if (ctl_type == CTL_RADIO) {
            ustring sv;
            if (!val.is_undefined()) sv = val.to_string();
            array<helement> all;
            find_all(v, all, self, sel, false);
            FOREACH(i, all) {
              if (all[i]->atts(attr::a_value) == sv)
                all[i]->state_on(v, S_CHECKED);
              else
                all[i]->state_off(v, S_CHECKED);
            }
          }
          else if (ctl_type == CTL_CHECKBOX)
            ctl->set_value(v, val);
          else
            //ctl->set_value(v, val);
            v.set_element_value(ctl, val, true);

          return true;
        };
        all.visit(oneach);
      }
#else
      void set_values(view &v, element *self, const value &all) {
        has_file_ctl = false;
        auto oneach = [&](const value &key, const value &val) -> bool {
          ustring path = key.get(W(""));
          ustring sel = ustring::format(W("[name='%s']"), path.c_str());

          helement ctl = find_first(v, self, sel, false);
          // helement ctl = find_ctl(self,path);

          if (!ctl) return true;

          ctl->get_style(v); // to force behaviors to be resolved.

          int ctl_type = ctl->ctl_type(v);

          if (find_first(v, ctl, WCHARS("[name]"), false, false)) {
            if (ctl->set_value(v, val, true))
              ;
            else if (val.is_map_alike())
              set_values(v, ctl, val);
            return true;
          }
          else if (is_inside_named(self, ctl))
            return true; // that is inner element inside some group

          if (ctl_type == CTL_BUTTON) {
            return true; // skip buttons
          }
          else if (ctl_type == CTL_RADIO) {
            ustring sv;
            if (!val.is_undefined()) sv = val.to_string();

            array<helement> all;
            find_all(v, all, self, sel, false);
            FOREACH(i, all) {
              if (all[i]->atts(attr::a_value) == sv)
                all[i]->state_on(v, S_CHECKED);
              else
                all[i]->state_off(v, S_CHECKED);
            }
          }
          else if (ctl_type == CTL_CHECKBOX)
            ctl->set_value(v, val);
          else
            ctl->set_value(v, val);

          return true;
        };
        all.visit(oneach);
      }
#endif

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) {
        chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        METHOD(0, submit) {
          do_submit(v, self, self);
          return true;
        }
        METHOD(0, reset) {
          do_reset(v, self, self);
          return true;
        }
#undef METHOD
        return super::on_x_method_call(v, self, name, argv, argc, retval);
      }

      virtual element* r13n_container(element* self) { return self; }

      bool api_submit() {
        if (auto pv = self->pview()) {
          do_submit(*pv, self, self);
          return true;
        }
        return false;
      }
      bool api_reset() {
        if (auto pv = self->pview()) {
          do_reset(*pv, self, self);
          return true;
        }
        return false;
      }

      SOM_PASSPORT_BEGIN_EX(form, form_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(reset, api_reset),
          SOM_FUNC_EX(submit, api_submit),
        )
      SOM_PASSPORT_END

    };

    ctl *frame_ctl_factory::create(element *el) { return new frame_ctl(); }

    ctl *history_ctl_factory::create(element *el) { return new history_ctl(); }

    ctl *frame_set_ctl_factory::create(element *el) {
      return new frame_set_ctl();
    }

    ctl *form_ctl_factory::create(element *el) { return new form_ctl(); }

    struct style_bag_ctl : ctl {
      typedef ctl super;
      style_bag_ctl() {}

      virtual CTL_TYPE get_type() { return CTL_UNKNOWN; }

      virtual const string &behavior_name() const {
        return _style_bag_ctl_factory->name;
      }
      virtual bool attach(view &v, element *self) {
        ctl::attach(v, self);
        if (self->pview() == &v) {
          document *pd = self->doc();
          string sid = self->sequential_id();
          if (pd && !sid) { // created outside of normal parsing
            v.refresh(pd);
            pd->load_style(self, self->get_text(v));
            pd->drop_styles(v);
          }
        }
        return true;
      }

      virtual void detach(view &v, element *self) {
        if (self->pview() == &v) {
          document *pd = self->doc();
          string sid = self->sequential_id();
          if (pd && sid) {
            v.refresh(pd);
            pd->styles().remove_styles(self);
            pd->drop_styles(v);
          }
        }
        ctl::detach(v, self);
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == DISABLED_STATUS_CHANGED && evt.target == self) {
          document *pd = self->doc();
          if (!pd) return false;

          pd->styles().remove_styles(self);
          if (!self->state.disabled()) {
            pd->load_style(self, self->get_text(v));
          } else {
            pd->drop_layout();
            pd->measure_inplace(v);
          }
          v.refresh(pd);
          return true;
        }
        return false;
      }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) {
        chars fname = chars_of(name);
#define METHOD(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        METHOD(0, activate) {
          document *pd = self->doc();
          if (!pd) return true;
          pd->styles().remove_styles(self);
          // int lineno = self->atts.get_int(attr::a___lineno,0);
          pd->load_style(self, self->get_text(v));
          pd->drop_styles(v);
          return true;
        }
        METHOD(0, deactivate) {
          document *pd = self->doc();
          if (!pd) return true;
          pd->styles().remove_styles(self);
          pd->drop_styles(v);
          pd->measure_inplace(v);
          v.refresh(pd);
          return true;
        }
#undef METHOD
        return super::on_x_method_call(v, self, name, argv, argc, retval);
      }
    };

    ctl *style_bag_ctl_factory::create(element *el) {
      return new style_bag_ctl();
    }

    void init_frame() {
      ctl_factory::add(_frame_ctl_factory = new frame_ctl_factory());
      ctl_factory::add(_history_ctl_factory = new history_ctl_factory());
      ctl_factory::add(_frame_set_ctl_factory = new frame_set_ctl_factory());
      ctl_factory::add(_form_ctl_factory = new form_ctl_factory());
      ctl_factory::add(_style_bag_ctl_factory = new style_bag_ctl_factory());
    }

  } // namespace behavior

} // namespace html
