#include "html.h"
#include "html-behaviors.h"

namespace html {
  namespace behavior {

    //|
    //| marquee effect
    //|
    struct marquee_ctl_factory : public ctl_factory {
      marquee_ctl_factory() : ctl_factory("marquee") {}
      virtual ctl *create(element *) override;
    };

    static marquee_ctl_factory *_marquee_ctl = 0;

    struct marquee_ctl : public virtual ctl, public virtual animation {
      element *  _this;
      bool       _horizontal; // true - horizontal, false - vertical
      tristate_v _hover_forward;
      bool       _forward;       // true - up/left, false - down/right
      enum_v     _speed;         // normal = 0 | slow = 1 | fast = 2
      enum_v     _hover_speed;   //
      int_v      _play_count;    // undefined - infinite, 0 - no animation.
      int        _play_cycle_no; //
      enum_v     _style;         // scroll = 0 | slide = 1 | alternate = 2
      size       _scroll_pos;

      marquee_ctl() : _play_cycle_no(0) {}

      virtual CTL_TYPE      get_type() { return CTL_UNKNOWN; }
      virtual const string &behavior_name() const { return _marquee_ctl->name; }

      enum STEP_DELAYS {
        DELAY_FAST   = ANIMATION_TIMER_SPAN,
        DELAY_NORMAL = ANIMATION_TIMER_SPAN * 2,
        DELAY_SLOW   = ANIMATION_TIMER_SPAN * 3,
      };

      virtual bool attach(view &v, element *self) override {
        _this = self;
        ctl::attach(v, self);
        fetch_params(v, self);
        return true;
      }

      virtual void detach(view &v, element *self) {
        self->scroll_pos(v, point(0, 0));
        v.remove_animation(self, this);
        _this = 0;
        ctl::detach(v, self);
      }

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }

      enum animation_style { scroll = 0, slide = 1, alternate = 2 };

      virtual bool is_finite(view &, element *) {
        return _play_count.is_undefined();
      }

      virtual void fetch_params(view &v, element *self) {
        ustring mscroll = this->get_attr(self, "-marquee");
        _horizontal     = false;

        if (mscroll == WCHARS("horizontal")) {
          _horizontal = true;
        } else if (mscroll == WCHARS("vertical")) {
          _horizontal = false;
        } else if (self->get_style(v)->flow == flow_horizontal ||
                   self->get_style(v)->flow == flow_h_flow ||
                   self->get_style(v)->overflow_y <= overflow_visible)
          _horizontal = true;

        ustring direction = this->get_attr(self, "-marquee-direction");
        _forward          = true;
        if (direction == WCHARS("reverse")) {
          _forward = false;
        } else // if( direction == L"forward" )
        {
          _forward = true;
        }

        if (!self->a_style) self->a_style = new style_prop_map();
        if (_horizontal)
          self->a_style->set(cssa_overflow_x, overflow_ev(overflow_hidden_scroll));
        else
          self->a_style->set(cssa_overflow_y, overflow_ev(overflow_hidden_scroll));

        ustring speed = this->get_attr(self, "-marquee-speed");
        _speed        = 2;
        if (speed == WCHARS("slow"))
          _speed = 1;
        else if (speed == WCHARS("normal"))
          _speed = 2;
        else if (speed == WCHARS("fast"))
          _speed = 3;

        ustring count = this->get_attr(self, "-marquee-play-count");
        if (count.length() == 0 || count == WCHARS("infinite"))
          _play_count.clear();
        else
          _play_count = this->get_attr(self, "-marquee-play-count", 0);

        ustring style = this->get_attr(self, "-marquee-style");
        if (style == WCHARS("scroll"))
          _style = scroll;
        else if (style == WCHARS("slide"))
          _style = slide;
        else if (style == WCHARS("alternate"))
          _style = alternate;
      }

      bool start_animation() {
        if (!_this) return true;
        if (_this->has_animation(this))
          return true;
        if (view *pv = _this->pview()) 
          pv->add_animation(_this, this);
        return true;
      }

      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) override {
        if (!_this) return false;
        if (!self->has_animation(this)) {
          if (_play_count.is_undefined() || _play_count != 0)
            v.post(delegate(this, &marquee_ctl::start_animation));
        }
        return false;
      }

      bool step_forward(int &pos, int content_size, int view_size) {
        int save_pos = pos;
        ++pos;
        switch (_style) {
        case scroll:
          if (pos > content_size) {
            ++_play_cycle_no;
            pos = -view_size;
          }
          break;
        case slide:
          if (pos > content_size - view_size) {
            ++_play_cycle_no;
            pos = -view_size;
          }
          break;
        case alternate:
          if (pos > content_size - view_size) {
            ++_play_cycle_no;
            _forward = !_forward;
          }
          break;
        }
        if (_hover_forward.is_undefined() &&
            _play_cycle_no >= _play_count.val(100)) {
          pos = save_pos;
          return true;
        }
        return false;
      }
      bool step_backward(int &pos, int content_size, int view_size) {
        int save_pos = pos;
        --pos;
        switch (_style) {
        case scroll:
          if (pos < -view_size) {
            ++_play_cycle_no;
            pos = content_size;
          }
          break;
        case slide:
          if (pos < 0) {
            ++_play_cycle_no;
            pos = content_size;
          }
          break;
        case alternate:
          if (pos < 0) {
            ++_play_cycle_no;
            _forward = !_forward;
          }
          break;
        }
        if (_hover_forward.is_undefined() &&
            _play_cycle_no >= _play_count.val(100)) {
          pos = save_pos;
          return true;
        }
        return false;
      }

      virtual uint start(view &v, element *el, const style *nst,
                         const style *ost) override {
        el->commit_measure(v);
        return ANIMATION_TIMER_SPAN;
      }

      // animation::
      virtual uint step(view &v, element *self, uint current_clock) override {
        point sp   = self->scroll_pos();
        bool  stop = false;

        if (_hover_speed.is_defined() && _hover_speed == 0) return DELAY_SLOW;

        if (_horizontal)
          if (_hover_forward.val(_forward))
            stop =
                step_forward(sp.x, self->max_content_width(v), self->dim().x);
          else
            stop =
                step_backward(sp.x, self->max_content_width(v), self->dim().x);
        else if (_hover_forward.val(_forward))
          stop = step_forward(sp.y, self->max_content_height(v), self->dim().y);
        else
          stop =
              step_backward(sp.y, self->max_content_height(v), self->dim().y);
        if (stop)
          v.remove_animation(self);
        else
          //v.scroll_window(sp, self, SCROLL, false, true);
          //self->set_scroll_pos(v, sp, false, true);
        {
          self->scroll_pos(v, sp, true);
          self->refresh(v);
        }

        switch (_hover_speed.val(_speed)) {
        default:
        case 0: return 0;
        case 1: return DELAY_SLOW;
        case 2: return DELAY_NORMAL;
        case 3: return DELAY_FAST;
        }
      }

      void setup_speed_and_direction(int pos, int view_size) {
        bool forward = true;
        _hover_speed = 0;
        int part     = (8 * pos) / view_size;
        switch (part) {
        case 0:
          forward      = false;
          _hover_speed = 3;
          break;
        case 1:
          forward      = false;
          _hover_speed = 2;
          break;
        case 2:
          forward      = false;
          _hover_speed = 1;
          break;
        case 3:
        case 4:
          forward      = false;
          _hover_speed = 0;
          break;
        case 5:
          forward      = true;
          _hover_speed = 1;
          break;
        case 6:
          forward      = true;
          _hover_speed = 2;
          break;
        case 7:
          forward      = true;
          _hover_speed = 3;
          break;
        }
        // if( bool(_forward) != forward )
        _hover_forward = forward;
      }

      inline bool on(view &v, element *self, event_mouse &evt) {
        switch (evt.cmd) {
        case MOUSE_MOVE:
          if (_horizontal)
            setup_speed_and_direction(evt.pos.x, self->dim().x);
          else
            setup_speed_and_direction(evt.pos.y, self->dim().y);
          //_hover_forward = !_forward;
          break;
        case MOUSE_LEAVE:
        case MOUSE_LEAVE | EVENT_HANDLED:
          _hover_forward.clear();
          _hover_speed.clear();
          break;
        }
        return false;
      }
    };

    ctl *marquee_ctl_factory::create(element *) { return new marquee_ctl(); }

    //|
    //| sticky scroll
    //|

    struct sticky_scroll_ctl_factory : public ctl_factory {
      sticky_scroll_ctl_factory() : ctl_factory("sticky-scroll") {}
      virtual ctl *create(element *) override;
    };

    static sticky_scroll_ctl_factory *_sticky_scroll_ctl_factory = 0;

    struct sticky_scroll_ctl : public ctl {
      element *self;
      helement sticky;
      int      offset;

      sticky_scroll_ctl() : self(0) {}
      virtual CTL_TYPE      get_type() { return CTL_UNKNOWN; }
      virtual const string &behavior_name() const {
        return _sticky_scroll_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        this->self = self;
        ctl::attach(v, self);
        return true;
      }

      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) {
        auto_state<helement> _(v.sticky_scrollable, self);
        v.sticky_scrollable_anchor = nullptr;
        draw_element_content(v, self, sf, pos);
        if (v.sticky_scrollable_anchor) {
          point off = v.sticky_scrollable_anchor->rel_pos(v, self);
          sticky    = v.sticky_scrollable_anchor;
          offset    = off.y;
          // v.sticky_scrollable_anchor->dbg_report("sticky_scrollable_anchor:");
          v.sticky_scrollable_anchor = nullptr;
        }
        return true;
      }

      virtual bool update_scroll_position(view &v, element *self) {

        if (sticky) {

          point sp  = self->scroll_pos();
          int   off = sticky->rel_pos(v, self).y;
          off += sp.y;
          off -= offset;
          sp.y = off;
          if (sp != self->scroll_pos()) {
            handle<element> what = self;
            v.async([sp, &v, what]() -> bool {
              what->set_scroll_pos(v, sp, false, true);
              return true;
            });
          }
          sticky = nullptr;
        }
        return true;
      }

      virtual bool set_scroll_pos(view &v, element *self, point pos,
                                  bool smooth) {
        sticky = nullptr;
        return false;
      }
    };

    ctl *sticky_scroll_ctl_factory::create(element *for_that) {
      return nullptr;
      return new sticky_scroll_ctl();
    }


    struct layer_ctl_factory : public ctl_factory {
      layer_ctl_factory() : ctl_factory("layer") {}
      virtual ctl *create(element *) override;
    };

    static layer_ctl_factory *_layer_ctl_factory = 0;

    struct layer_ctl : public ctl {
      element *self;
      hbitmap  buffer;
      rect     invalid;
      //int      n_animations;
            
      layer_ctl() : self(nullptr) /*, n_animations(0)*/ {}
      virtual CTL_TYPE      get_type() { return CTL_UNKNOWN; }
      virtual const string &behavior_name() const {
        return _layer_ctl_factory->name;
      }

      virtual bool attach(view &v, element *self) override {
        this->self = self;
        ctl::attach(v, self);
        self->flags.layered = 1;
        return true;
      }

      virtual void detach(view& v, element *self) override {
        self->flags.layered = 0;
      }

      virtual void invalidate_surface(element *self, const rect& invalid_rc) override {
        invalid |= invalid_rc;
      }

      virtual bool draw(view &v, graphics *sf, element *self, point pos) override {

        //if(n_animations)
        //  return false; // if layer contains animations we are not buffering this container;

        rect brc = self->border_box(v); 

        point off = self->border_distance(v).s;

        rect drc = brc + pos;
        
        size scaling = v.pixels_scaling();
        size dim = brc.size() * scaling;
        
        bool redraw = false;

        if (!invalid.empty()) {
          redraw = true;
          invalid = rect();
        }

        if (!buffer || (buffer->dim() != dim)) {
          buffer = new bitmap(dim, true, false);
          redraw = true;
        }
        else if(redraw) {
          buffer->clear(argb(0, 0, 0, 0));
        }

        if (redraw) {
        
            handle<graphics> sfi = app()->create_bitmap_bits_graphics(buffer, argb(0, 0, 0, 0));
            if (!sfi) {
              buffer = nullptr;
              return false;
            }
            sfi->offset(-pos + off);
            sfi->set_clip_rc(drc);
            sfi->translate(pos);
            sfi->scale(scaling);
            auto_state<graphics *> _(v.drawing_surface, sfi);

            self->do_draw(v, sfi, point(0,0), true, false /*prevent recursive call*/ );
            self->draw_outlines(v, sfi, point(0,0), true, true);
            v.commit_drawing_buffers();
            //dbg_printf("layer redraw\n");
        }

        sf->draw(buffer, drc);
        return true;
      }
      
      /*inline bool on(view &v, element *self, event_behavior &evt) 
      {
        if (evt.cmd == ANIMATION || evt.cmd == (ANIMATION | EVENT_HANDLED)) {
          if (evt.reason) 
            ++n_animations;
          else 
            --n_animations;
        }
        return false;
      }*/


    };

    ctl *layer_ctl_factory::create(element *for_that) {
      return new layer_ctl();
    }
        
    void init_effects() {
      // ctl_factory::add( _reflection_ctl = new reflection_ctl_factory() );
      // ctl_factory::add( _magnifier_ctl = new magnifier_ctl_factory() );
      ctl_factory::add(_layer_ctl_factory = new layer_ctl_factory() );
      ctl_factory::add(_sticky_scroll_ctl_factory = new sticky_scroll_ctl_factory());
      ctl_factory::add(_marquee_ctl = new marquee_ctl_factory());
    }

  } // namespace behavior
} // namespace html
