#include "html.h"

namespace html {
  namespace behavior {

    datetime_t get_time_zone_shift(view &v, element *self) {
      string s = self->atts.get_string("timezone");
      if (s.length()) 
        return tool::time_zone_shift(s);
      return date_time::local_offset();
    }

    struct date;
    struct time;

    date_time to_date_time(date d);
    date_time to_date_time(date d, time t);
    date_time to_date_time(time t);

    struct date {
      int year;
      int month;
      int day;

      datetime_t shift;

      date(datetime_t tz_shift) : year(0), month(0), day(0), shift(tz_shift) {}
      date(date_time dt, datetime_t tz_shift) : shift(tz_shift)  {
        //year = dt.year();
        //month = dt.month();
        //day = dt.day();
        dt.to_timezone(shift);
        year = dt.year();
        month = dt.month();
        day = dt.day();
      }
      date(const date &dt) : year(dt.year), month(dt.month), day(dt.day), shift(dt.shift) {}

      bool operator==(const date &r) const {
        return year == r.year && month == r.month && day == r.day;
      }
      bool operator!=(const date &r) const {
        return year != r.year || month != r.month || day != r.day;
      }

      bool parse(const wchar *str) {
        string s(str);
        return parse((const char *)s);
      }
      bool parse(const char *str) {
        day = 0;
        if (str && str[0]) {
          if (strcmp(str, "today") == 0 || strcmp(str, "now") == 0) {
            set_today();
            return true;
          }
#ifdef WINDOWS
          int fields = sscanf_s(str, "%d-%d-%d", &year, &month, &day);
#else
          int fields = sscanf(str, "%d-%d-%d", &year, &month, &day);
#endif
          if (fields == 2 && month >= 1 && month <= 12)
            return true;
          else if (fields == 3 && month >= 1 && month <= 12) {
            if (is_valid()) return true;
          }
        }

        return false;
      }

      void set_today() {
        date_time dt = date_time::now(true);
        dt.to_timezone(shift);
        year         = uint(dt.year());
        month        = uint(dt.month());
        day          = uint(dt.day());
      }

      ustring to_string() const {
        return ustring::format(W("%04u-%02u-%02u"), year, month, day);
      }

      ustring print(bool full, const ustring &locale) const {
        date_time dt = to_date_time(*this);
        dt.set_date(year, month, day);
        return dt.default_format(full, date_time::DATE,locale);
      }

      ustring month_name(const ustring &locale) const {
        date_time dt = to_date_time(*this);
        dt.set_date(year, month, 1);
        return dt.locale_format(W("MMMM"), locale);
      }
      bool is_valid() const {
        if (!day || !month || !year) return false;
        date_time dt = to_date_time(*this);
        dt.set_date(year, month, day);
        return dt.day() == day && dt.month() == month && dt.year() == year;
      }
      bool is_empty() const { return !day || !month || !year; }
      void set_empty() { day = month = year = 0; }

      //operator date_time() { return date_time(year, month, day); }
    };

    struct time {
      int hours;
      int minutes;
      int seconds;
      datetime_t shift;

      time(datetime_t tz_shift) : hours(-1), minutes(-1), seconds(-1), shift(tz_shift) {}

      time(date_time dt, datetime_t tz_shift): shift(tz_shift)
      {
        dt.to_timezone(shift);
        hours = dt.hours();
        minutes = dt.minutes();
        seconds = dt.seconds();
      }
      time(const time &dt)
          : hours(dt.hours), minutes(dt.minutes), seconds(dt.seconds), shift(dt.shift) {}

      bool parse(const char *str) {
        if (str && str[0]) {
          if (strcmp(str, "now") == 0) {
            set_now();
            return true;
          }
          seconds = 0;
#ifdef WINDOWS
          int fields = sscanf_s(str, "%d:%d:%d", &hours, &minutes, &seconds);
#else
          int fields = sscanf(str, "%d:%d:%d", &hours, &minutes, &seconds);
#endif
          if (fields == 2 || fields == 3) return true;
        }
        set_now();
        return false;
      }
      bool parse(const wchar *str) { return parse(string(str)); }

      void set_now() {
        date_time dt = date_time::now();
        dt.to_timezone(shift);
        hours        = dt.hours();
        minutes      = dt.minutes();
        seconds      = dt.seconds();
      }

      ustring to_string() const {
        return ustring::format(W("%02u:%02u:%02u"), hours, minutes, seconds);
      }

      bool is_valid() const {
        return hours >= 0 && hours < 24 && minutes >= 0 && minutes <= 59 &&
               seconds >= 0 && seconds <= 59;
      }
      bool is_empty() const { return !is_valid(); }

      double duration_seconds() const {
        if (is_valid()) return seconds + 60 * minutes + 60 * 60 * hours;
        return 0;
      }

      void set_empty() {
        hours = minutes = seconds = -1;
      }

    };

    date_time to_date_time(date d) {
      date_time dt(d.year, d.month, d.day);
      dt.from_timezone(d.shift);
      return dt;
    }
    date_time to_date_time(date d,time t) {
      date_time dt(d.year, d.month, d.day, t.hours,t.minutes,t.seconds);
      dt.from_timezone(d.shift);
      return dt;
    }
    date_time to_date_time(time t) {
      date_time dt(0, 0, 0, t.hours, t.minutes, t.seconds);
      dt.from_timezone(t.shift);
      return dt;
    }


    struct calendar_ctl_factory : ctl_factory {
      calendar_ctl_factory() : ctl_factory("calendar") {}
      virtual ctl *create(element *el);
    };

    static calendar_ctl_factory *_calendar_ctl_factory = 0;

    typedef enum { DAY_VIEW, MONTH_VIEW, YEAR_VIEW, TENYEAR_VIEW } eViewType;

    struct view_parent {
      date current;

      view_parent() : current(0) {}
      virtual void     roll_up(view &v, element *self, int levels = 1)     = 0;
      virtual void     drill_down(view &v, element *self)                  = 0;
      virtual bool     notify_changed(view &v, element *self, uint reason) = 0;
      virtual element *get_target_td(element *self, element *b)            = 0;
    };

    const wchar *get_today_caption() {
#if defined(SCITER)
      const wchar *def       = W("Today");
      static wchar today[64] = {0};
#if defined(WINDOWS) //&& defined(PLATFORM_DESKTOP)
      if (!today[0]) {
#define IDS_TODAY 0x1043
        HMODULE localizedTodayLibrary =
            LoadLibraryEx(TEXT("comctl32.dll"), NULL, LOAD_LIBRARY_AS_DATAFILE);
        if (localizedTodayLibrary) {
          LoadStringW(localizedTodayLibrary, IDS_TODAY, today, items_in(today));
          FreeLibrary(localizedTodayLibrary);
          wchars wc = chars_of(today);
          if (wc.like(L"*:")) today[wc.length - 1] = 0;
        }
      }
#endif
      return today[0] ? today : def;
#else
      return W("");
#endif
    }

    struct time_view : resource {
      view_parent *owner;
      time_view(view_parent *p) : owner(p) {}
      virtual ~time_view() {}

      virtual void get_html(view &v, element *self, const date &current,
                            utf8_ostream &out)                              = 0;
      virtual void do_click(view &v, element *self, element *td,
                            event_mouse &evt, uint reason)                  = 0;
      virtual void do_press(view &v, element *self, element *td,
                            uint reason)                                    = 0;
      virtual void get_header_caption(element* self, utf8_ostream &out, const date &  current)                  = 0;
      virtual void go_to_lastval(view &v, element *self, uint reason = -1)  = 0;
      virtual void go_to_firstval(view &v, element *self, uint reason = -1) = 0;

      virtual void go_next_val(view &v, element *self, int dir,
                               uint reason)     = 0;
      virtual void go_next_capval1(view &v, element *self, int dir,
                                   uint reason) = 0;
      virtual void go_next_capval2(view &v, element *self, int dir,
                                   uint reason) = 0;

      virtual int line_size() = 0;

      void get_header(view &v, element *self, const date &current,
                      utf8_ostream &out) {
        bool is_ltr = self->get_style(v)->direction != direction_rtl;

        out << "<caption>"
            << (is_ltr ? "<div.prev.nav></div>" : "<div.next.nav></div>");
        get_header_caption(self,out, current);
        out << (is_ltr ? "<div.next.nav></div>" : "<div.prev.nav></div>")
            << "</caption>";
      }

      void get_epilog(view &v, element *self, const date &current,
                      utf8_ostream &out) {
        //date_time today = date_time::now();
        date d(owner->current.shift);
        d.set_today();
        date_time dt = to_date_time(d);
        dt.to_timezone(d.shift);
        out << "<footer>"
            << "<label>" << get_today_caption() << ":</label><div.today>"
            << dt.default_format(false, date_time::DATE, self->get_lang()) << "</div>"
            << "</footer>";
      }

      void show_view(view &v, element *self, const date &current) {
        utf8_ostream html;
        helement     hself = self;
        date         dt    = current;
        if (!dt.is_valid()) dt.set_today();
        get_html(v, self, dt, html);
        v.set_element_html(hself, html.utf8(), SE_REPLACE);
      }

      virtual void update_view(view &v, element *self, const date &current) {
        show_view(v, self, current);
      }

      void notify_new_month(view &v, element *self, uint reason) {
        event_behavior evt(self, self, UI_STATE_CHANGED, reason);
        v.send_behavior_event(evt);
      }

      // virtual void preprocess_go_to_day() {};

      virtual void go_to_day(view &v, element *self, const date &dt,
                             uint reason = uint(-1)) {
        ustring  sel = WCHARS("td:current");
        element *pc  = find_first(v, self, sel);
        if (pc) pc->state_off(v, S_CURRENT);

        owner->current = dt;
        // preprocess_go_to_day();

        sel = ustring::format(W("td[value='%04u-%02u-%02u']"),
                              owner->current.year, owner->current.month,
                              owner->current.day);
        pc  = find_first(v, self, sel);
        if (pc) {
          pc->state_on(v, S_CURRENT);
          if (reason != uint(-1)) {
            owner->notify_changed(v, self, reason);
          }
        }
      }
    };

    struct day_view : public time_view {

      uint total_days; // num days in current month

      day_view(view_parent *p) : time_view(p), total_days(0) {}

      virtual void go_next_val(view &v, element *self, int dir, uint reason) {
        go_next_day(v, self, dir, reason);
      }

      virtual void go_next_capval1(view &v, element *self, int dir,
                                   uint reason) {
        go_next_month(v, self, dir, reason);
      }

      virtual void go_next_capval2(view &v, element *self, int dir,
                                   uint reason) {
        go_next_year(v, self, dir, reason);
      }

      void go_next_year(view &v, element *self, int dir, uint reason) {
        owner->current.year += dir < 0 ? -1 : 1;
        update_view(v, self, owner->current);
        go_to_day(v, self, owner->current, reason);
      }

      virtual void go_to_lastval(view &v, element *self, uint reason = -1) {
        date dt = owner->current;
        dt.day  = total_days;
        go_to_day(v, self, dt, reason);
      }

      virtual void go_to_firstval(view &v, element *self, uint reason = -1) {
        date dt = owner->current;
        dt.day  = 1;
        go_to_day(v, self, dt, reason);
      }

      virtual int line_size() { return 7; }

      void go_next_month(view &v, element *self, int dir, uint reason) {
        if (dir < 0) {
          if (--owner->current.month < 1) {
            owner->current.month = 12;
            --owner->current.year;
          }
        } else {
          if (++owner->current.month > 12) {
            owner->current.month = 1;
            ++owner->current.year;
          }
        }
        update_view(v, self, owner->current);
        go_to_day(v, self, owner->current, reason);
      }

      // virtual void preprocess_go_to_day()
      //{
      //    //if(owner->current.day > total_days)
      //    //    owner->current.day = total_days;
      //}

      void go_next_day(view &v, element *self, int delta, uint reason) {

        date_time t = to_date_time(owner->current);
        //t.set_date(owner->current.year, owner->current.month,
        //           owner->current.day);
        t.to_local();
        t.day(t.day() + delta);

        int prev_month = owner->current.month;

        date dt(owner->current.shift);
        dt.day   = t.day();
        dt.month = t.month();
        dt.year  = t.year();

        if (dt.month != prev_month) {
          owner->current = dt;
          update_view(v, self, owner->current);
        }
        go_to_day(v, self, dt, reason);
      }

      void get_header_caption(element* self, utf8_ostream &out, const date &current) {
        out << "<div .month .button month=" << tool::itoa(current.month) << " >"
            << current.month_name(self->get_lang()) << "</div><div .year .button>"
            << tool::itoa(current.year) << "</div>";
      }

      int first_day_of_week(element *self) {
        ustring fdowa;
        if (self->tag == tag::T_WIDGET && self->parent &&
            self->parent->tag == tag::T_POPUP && self->parent->parent &&
            self->parent->parent->tag == tag::T_INPUT)
          self = self->parent->parent;
        self->get_attr("-firstdayofweek", fdowa);

        int fdow = date_time::first_day_of_week(self->get_lang());

        if (fdowa.length()) {
          wchars s = fdowa;
          s        = trim(s);
          int n    = to_int(s);
          if (s.length == 0 && is_between(n, 1, 7)) fdow = n - 1;
        }
        return fdow;
      }

      void get_start_day(view &v, element *self, date_time &dt,
                         const date &current) {
        int fdow = first_day_of_week(self);
        dt.set_date(current.year, current.month, 1);
        for (int i = 0; i < 7 * 5; ++i) {
          if (dt.day_of_week() == fdow) break;
          dt.day(dt.day() - 1);
        }
      }

      virtual void get_html(view &v, element *self, const date &current,
                            utf8_ostream &out) {
        int i;
        int fdow = first_day_of_week(self);

        date d(owner->current.shift);
        d.set_today();

        date_time today = date_time(d.year,d.month,d.day);

        get_header(v, self, current, out);
        out << "<table><tbody>";
        out << "<tr>";
        for (i = 0; i < 7; ++i)
          out << "<th .weekday>"
              << date_time::week_day_name((i + fdow) % 7, 3, self->get_lang()) << "</th>";
        out << "</tr>";

        date_time t;
        get_start_day(v, self, t, current);
        total_days = 0;

        for (int n = 0; n < 6; ++n) {
          out << "<tr>";
          for (i = 0; i < 7; ++i) {
            out << "<td .day";
            if (t.day_of_week() >= 5) out << " .off";
            if (t == today)
              out << " .today";
            else if (t < today)
              out << " .past";

            if (current.year != t.year() || current.month != t.month())
              out << " .other-month";
            else
              total_days = t.day();

            out << " value="
                << string::format("%04u-%02u-%02u", t.year(), t.month(), t.day())
                << " >";
            out << string::format("%d", t.day()) << "</td>";

            t.day(t.day() + 1);
          }
          out << "</tr>";
        }

        out << "</tbody></table>";
        get_epilog(v, self, current, out);
      }

      void update_view(view &v, element *self, const date &current) {
        date_time t;
        get_start_day(v, self, t, current);

        date d(owner->current.shift);
        d.set_today();

        date_time today = date_time(d.year, d.month, d.day);

        ustring   month_name = current.month_name(self->get_lang());
        find_first(v, self, WCHARS("div.button.month"))
            ->set_text(v, month_name);
        ustring year_name = tool::itow(current.year);
        find_first(v, self, WCHARS("div.button.year"))->set_text(v, year_name);

        total_days = 0;

        for (int n = 0; n < 6; ++n) {
          element *row = find_first(
              v, self, ustring::format(W("tr:nth-child(%d)"), n + 2));
          assert(row && (row->n_children() >= 6));
          if (!(row && (row->n_children() >= 6))) return;

          int            i = 0;
          child_iterator cit(v, row);

          for (element *day; cit(day); ++i) {
            day->set_attr("value", string::format("%04u-%02u-%02u", t.year(),
                                                  t.month(), t.day()));
            ustring val = string::format("%d", t.day());
            day->set_text(v, val);

            val = W("day");
            if (t.day_of_week() >= 5) val += W(" off");
            if (t == today)
              val += W(" today");
            else if (t <= today)
              val += W(" past");

            if (current.year != t.year() || current.month != t.month())
              val += W(" other-month");
            else
              total_days = t.day();

            if (day->attr_class() != val) day->set_attr("class", val);
            t.day(t.day() + 1);
          }
        }
        notify_new_month(v, self, 0);
      }

      void do_press(view &v, element *self, element *td,
                    uint reason = uint(-1)) {
        if (td->is_of_class("day")) {
          date dt(owner->current.shift);
          dt.parse(td->atts.get_string(attr::a_value));
          go_to_day(v, self, dt, reason);
        }
      }

      virtual void do_click(view &v, element *self, element *td,
                            event_mouse &evt, uint reason) {
        if (evt.target->is_of_class("prev", "nav"))
          go_next_month(v, self, -1, CLICK_SYNTHESIZED);
        else if (evt.target->is_of_class("next", "nav"))
          go_next_month(v, self, +1, CLICK_SYNTHESIZED);
        else if (td->is_of_class("day")) {
          date dt(owner->current.shift);
          dt.parse(td->atts.get_string(attr::a_value));
          go_to_day(v, self, dt, reason);
        } else if (td->tag == tag::T_CAPTION) {
          if (evt.target->is_of_class("year") ||
              evt.target->parent->is_of_class("year"))
            owner->roll_up(v, self, 2);
          else
            owner->roll_up(v, self, 1);
        }
      }
    };

    struct month_view : public time_view {

      month_view(view_parent *p) : time_view(p) {}

      virtual void go_to_lastval(view &v, element *self, uint reason = -1) {
        element *td = find_first(
            v, self, WCHARS("tr:nth-last-child(1) > td:nth-last-child(1)"));
        assert(td);

        string dts = td->attr_value();
        date   dt(owner->current.shift);
        dt.parse(dts);
        go_to_day(v, self, dt, reason);
      }

      virtual void go_to_firstval(view &v, element *self, uint reason = -1) {
        element *td =
            find_first(v, self, WCHARS("tr:nth-child(2) > td:first-child"));
        assert(td);

        string dts = td->attr_value();
        date   dt(owner->current.shift);
        dt.parse(dts);
        go_to_day(v, self, dt, reason);
      }

      void go_next_year(view &v, element *self, int dir, uint reason) {
        owner->current.year += dir < 0 ? -1 : 1;
        update_view(v, self, owner->current);
        go_to_day(v, self, owner->current, reason);
      }

      void go_next_month(view &v, element *self, int delta, uint reason) {
        date_time t;
        t.set_date(owner->current.year, owner->current.month,
                   owner->current.day);
        int  new_month = owner->current.month + delta;
        bool new_year  = false;
        if (new_month > 12) {
          t.year(t.year() + 1);
          new_month -= 12;
          new_year = true;
        }
        if (new_month < 1) {
          t.year(t.year() - 1);
          new_month += 12;
          new_year = true;
        }
        t.month(new_month);

        date dt(owner->current.shift);
        dt.day   = t.day();
        dt.month = t.month();
        dt.year  = t.year();

        if (new_year) {
          owner->current = dt;
          update_view(v, self, owner->current);
        }
        go_to_day(v, self, dt, reason);
      }

      virtual int line_size() { return 4; }

      virtual void get_html(view &v, element *self, const date &current,
                            utf8_ostream &out) {
        int i;

        date_time today = date_time::now();
        date_time t;
        t.set_date(current.year, 1, 1);

        get_header(v, self, current, out);
        out << "<table><tbody>";
        for (int n = 0; n < 3; ++n) {
          out << "<tr>";
          for (i = 0; i < 4; ++i) {
            out << "<td .month";
            if (t.year() == today.year() && t.month() == today.month())
              out << " .today";

            out << " value="
                << string::format("%04u-%02u-%02u", t.year(), t.month(),
                                  t.day())
                << " >";
            out << t.locale_format(W("MMM"), self->get_lang()) << "</td>";

            t.month(t.month() + 1);
          }
          out << "</tr>";
        }
        out << "</tbody></table>";
        get_epilog(v, self, current, out);
      }

      virtual void do_click(view &v, element *self, element *td,
                            event_mouse &evt, uint reason) {
        if (evt.target->is_of_class("prev", "nav"))
          go_next_year(v, self, -1, CLICK_SYNTHESIZED);
        else if (evt.target->is_of_class("next", "nav"))
          go_next_year(v, self, +1, CLICK_SYNTHESIZED);
        else if (td->is_of_class("month")) {
          date dt(owner->current.shift);
          dt.parse(td->attr_value());
          owner->current = dt;
          go_to_day(v, self, dt);
          owner->drill_down(v, self);
        } else if (td->tag == tag::T_CAPTION) {
          owner->roll_up(v, self);
        }
      }

      virtual void do_press(view &v, element *self, element *td, uint reason) {
        if (td->is_of_class("month")) {
          date dt(owner->current.shift);
          dt.parse(td->attr_value());
          go_to_day(v, self, dt, reason);
        }
      }

      virtual void get_header_caption(element* self, utf8_ostream &out, const date &current) override {
        out << "<div .year .button>" << tool::itoa(current.year) << "</div>";
      }

      virtual void go_next_val(view &v, element *self, int dir, uint reason) {
        go_next_month(v, self, dir, reason);
      }

      virtual void go_next_capval1(view &v, element *self, int dir,
                                   uint reason) {
        go_next_year(v, self, dir, reason);
      }

      virtual void go_next_capval2(view &v, element *self, int dir,
                                   uint reason) {
        go_next_year(v, self, dir, reason);
      }
    };

    template <class T, int t_period> struct year_switcher {
      void go_next(view &v, element *self, int dir, uint reason) {
        T *      pT            = static_cast<T *>(this);
        date &   current       = pT->owner->current;
        date     t             = current;
        element *current_block = find_first(v, self, WCHARS("td:current"));
        bool     bFirst =
            current_block ==
            find_first(v, self, WCHARS("tr:nth-child(2) > td:first-child"));
        bool bLast =
            current_block ==
            find_first(v, self, WCHARS("tr:nth-child(4) > td:nth-child(4)"));
        bool bLeftBottom =
            current_block ==
            find_first(v, self, WCHARS("tr:nth-child(4) > td:first-child"));
        bool bRightTop =
            current_block ==
            find_first(v, self, WCHARS("tr:nth-child(2) > td:nth-child(4)"));

        bool bChangePage = false;

        if (bLast && dir > 0) bChangePage = true;

        if (bFirst && dir < 0) bChangePage = true;

        current.year += dir * t_period;
        // unsigned int first_year = pT->get_first_year(current);

        if ((current.year < get_first_year(t)) ||
            (current.year > get_first_year(t) + 11 * t_period) || bChangePage) {
          // fixup
          if ((dir > 1 && !bFirst) || (dir < -1 && !bLast)) {
            current.year += dir > 0 ? -2 * t_period : 2 * t_period;
          }
          if ((bLeftBottom && dir > 1) || (bRightTop && dir < -1)) {

            date dt = current;
            dt.year += dir * t_period;
            pT->update_view(v, self, dt);
            pT->go_to_day(v, self, current, reason);
            return;
          }

          pT->update_view(v, self, current);
        }
        pT->go_to_day(v, self, current, reason);
      }

      int get_first_year(const date &current) {
        return current.year / (10 * t_period) * (10 * t_period) - 1 * t_period;
      }
    };

    struct year_view : public time_view, public year_switcher<year_view, 1> {

      year_view(view_parent *p) : time_view(p) {}

      virtual int line_size() { return 4; }

      virtual void get_header_caption(element* self, utf8_ostream &out, const date &current) override {
        out << "<div .decade .button>" << tool::itoa(get_first_year(current)) << '-' << tool::itoa(get_first_year(current) + 11) << "</div>";
      }

      void go_next_decade(view &v, element *self, int dir, uint reason) {
        owner->current.year += dir < 0 ? -10 : 10;
        update_view(v, self, owner->current);
        go_to_day(v, self, owner->current, reason);
      }

      virtual void get_html(view &v, element *self, const date &current,
                            utf8_ostream &out) {
        int i;
        int first_year = get_first_year(current);

        date_time today = date_time::now();
        date_time t;
        t.set_date(first_year, current.month, current.day);

        get_header(v, self, current, out);

        out << "<table><tbody>";
        for (int n = 0; n < 3; ++n) {
          out << "<tr>";
          for (i = 0; i < 4; ++i) {
            out << "<td .year";
            if (t.year() == today.year()) out << " .today";
            if (t.year() == first_year || t.year() >= first_year + 11)
              out << " .other-year";

            out << " value="
                << string::format("%04u-%02u-%02u", t.year(), t.month(),
                                  t.day())
                << " >";
            out << tool::itoa(t.year()) << "</td>";

            t.year(t.year() + 1);
          }
          out << "</tr>";
        }

        out << "</tbody></table>";
        get_epilog(v, self, current, out);
      }

      virtual void do_click(view &v, element *self, element *td,
                            event_mouse &evt, uint reason) {
        if (evt.target->is_of_class("prev", "nav"))
          go_next_decade(v, self, -1, CLICK_SYNTHESIZED);
        else if (evt.target->is_of_class("next", "nav"))
          go_next_decade(v, self, +1, CLICK_SYNTHESIZED);
        else if (td->is_of_class("year")) {
          date dt(owner->current.shift);
          dt.parse(td->attr_value());
          owner->current = dt;
          go_to_day(v, self, dt);
          owner->drill_down(v, self);
        } else if (td->tag == tag::T_CAPTION) {
          owner->roll_up(v, self);
        }
      }

      virtual void do_press(view &v, element *self, element *td, uint reason) {
        if (td->is_of_class("year")) {
          date dt(owner->current.shift);
          dt.parse(td->attr_value());
          go_to_day(v, self, dt, reason);
        }
      }

      virtual void go_to_lastval(view &v, element *self, uint reason = -1) {
        date dt = owner->current;
        dt.year = get_first_year(dt) + 10;
        go_to_day(v, self, dt, reason);
      }

      virtual void go_to_firstval(view &v, element *self, uint reason = -1) {
        date dt = owner->current;
        dt.year = get_first_year(dt) + 1;
        go_to_day(v, self, dt, reason);
      }

      virtual void go_next_val(view &v, element *self, int dir, uint reason) {
        go_next(v, self, dir, reason);
      }

      virtual void go_next_capval1(view &v, element *self, int dir,
                                   uint reason) {
        go_next_decade(v, self, dir, reason);
      }

      virtual void go_next_capval2(view &v, element *self, int dir,
                                   uint reason) {
        go_next_decade(v, self, dir, reason);
      }
    };

    struct century_view : public time_view,
                          public year_switcher<century_view, 10> {

      century_view(view_parent *p) : time_view(p) {}

      virtual int line_size() { return 4; }

      virtual void get_header_caption(element* self, utf8_ostream &out, const date &current) override {
        out << "<div .century .button>" << tool::itoa(get_first_year(current))
            << '-' << tool::itoa(get_first_year(current) + 110) << "</div>";
      }

      void go_next_century(view &v, element *self, int dir, uint reason) {
        owner->current.year += dir < 0 ? -100 : 100;
        update_view(v, self, owner->current);
        go_to_day(v, self, owner->current, reason);
      }

      virtual void get_html(view &v, element *self, const date &current,
                            utf8_ostream &out) {
        int i;
        int first_year = get_first_year(current);

        date_time today = date_time::now();
        date_time t;
        t.set_date(first_year, current.month, current.day);

        int year_delta = current.year - current.year / 10 * 10;

        get_header(v, self, current, out);
        out << "<table><tbody>";
        for (int n = 0; n < 3; ++n) {
          out << "<tr>";
          for (i = 0; i < 4; ++i) {
            out << "<td .decade";
            if (t.year() == today.year()) out << " .today";
            if (t.year() == first_year || t.year() >= first_year + 110)
              out << " .other-decade";

            out << " value="
                << string::format("%04u-%02u-%02u", t.year() + year_delta,
                                  t.month(), t.day())
                << " >";
            out << string::format("%d-<br>%d</td>", t.year(), t.year() + 9);

            t.year(t.year() + 10);
          }
          out << "</tr>";
        }
        out << "</tbody></table>";
        get_epilog(v, self, current, out);
      }

      virtual void do_click(view &v, element *self, element *td,
                            event_mouse &evt, uint reason) {
        if (evt.target->is_of_class("prev", "nav"))
          go_next_century(v, self, -1, CLICK_SYNTHESIZED);
        else if (evt.target->is_of_class("next", "nav"))
          go_next_century(v, self, +1, CLICK_SYNTHESIZED);
        else if (td->is_of_class("decade")) {
          date dt(owner->current.shift);
          dt.parse(td->attr_value());
          owner->current = dt;
          go_to_day(v, self, dt);
          owner->drill_down(v, self);
        }
      }

      virtual void do_press(view &v, element *self, element *td, uint reason) {
        if (td->is_of_class("century")) {
          date dt(owner->current.shift);
          dt.parse(td->attr_value());
          go_to_day(v, self, dt, reason);
        }
      }

      virtual void go_to_lastval(view &v, element *self, uint reason = -1) {
        date dt = owner->current;
        dt.year = get_first_year(dt) + 100;
        go_to_day(v, self, dt, reason);
      }

      virtual void go_to_firstval(view &v, element *self, uint reason = -1) {
        date dt = owner->current;
        dt.year = get_first_year(dt) + 10;
        go_to_day(v, self, dt, reason);
      }

      virtual void go_next_val(view &v, element *self, int dir, uint reason) {
        go_next(v, self, dir, reason);
      }

      virtual void go_next_capval1(view &v, element *self, int dir,
                                   uint reason) {
        go_next_century(v, self, dir, reason);
      }

      virtual void go_next_capval2(view &v, element *self, int dir,
                                   uint reason) {
        go_next_century(v, self, dir, reason);
      }
    };

    struct calendar_ctl : ctl, view_parent {
      element* self;
      handle<time_view> views[4];
      handle<time_view> current_view;
      int        current_view_id = 0;
      bool       m_bPressed = false;
      size       intrinsic;

      datetime_t tz = 0;

      calendar_ctl(const element *self) {}

      void init(view& v, element* self) {
        views[0] = new day_view(this);
        views[1] = new month_view(this);
        views[2] = new year_view(this);
        views[3] = new century_view(this);
        if (current_view) {
          current_view = views[current_view_id];
        }
        else {
          string mode = self->atts.get_string("mode");
          init_mode(mode);
        }
        m_bPressed = false;
        element* owner = self;
        if (self->parent && self->parent->parent && (self->parent->tag == tag::T_POPUP)) // popup calendar
          owner = self->parent->parent;

        current.shift = tz = get_time_zone_shift(v, owner);

        get_value(self, current);
        if (current.is_empty()) current.set_today();

        current_view->show_view(v, self, current);
        current_view->go_to_day(v, self, current);

        intrinsic.x = self->min_content_width(v);
        self->layout_width(v, intrinsic.x);
        intrinsic.y = self->min_content_height(v);
      }

      virtual CTL_TYPE get_type() { return CTL_DATE; }

      virtual const string &behavior_name() const {
        return _calendar_ctl_factory->name;
      }

      virtual void on_lang_change(view& pv, element* self) override {
        auto val = current;
        self->clear();
        init(pv, self);
        set_date(pv, self, val);
        pv.add_to_update(self, CHANGES_MODEL);
      }
     

      virtual bool focusable(const element *self) {
        return !self->attr_disabled();
      }

      virtual void detach(view &v, element *self) { self = nullptr; }
      virtual bool attach(view &v, element *self) override {
        this->self = self;
        self->clear(&v);

        init(v, self);

        return true;
      }

      void show_view(view &v, element *self) {
        current_view = views[current_view_id];
        current_view->show_view(v, self, current);
        current_view->go_to_day(v, self, current);
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd != DO_SWITCH_TAB) return false;
        if (evt.reason) // roll up
        {
          current_view_id += uint(evt.reason);
          if (current_view_id > TENYEAR_VIEW) current_view_id = TENYEAR_VIEW;
        } else // drill down
        {
          current_view_id--;
          if (current_view_id < DAY_VIEW) current_view_id = DAY_VIEW;
        }
        show_view(v, self);
        notify_state_changed(v, self, 0);
        return true;
      }
      /*virtual bool on_timer   (view& v, element* self, uint_ptr timer_id,
      TIMER_KIND kind )
      {
          if( timer_id == 1 ) // roll up
          {
              current_view_id++;
              if( current_view_id > TENYEAR_VIEW )
                  current_view_id = TENYEAR_VIEW;
          } else // drill down
          {
              current_view_id--;
              if( current_view_id < DAY_VIEW )
                  current_view_id = DAY_VIEW;
          }
          show_view(self, v);
          return false;
      }*/

      virtual void drill_down(view &v, element *self) {
        // pv->start_timer(self, 50);
        event_behavior evt(self, self, DO_SWITCH_TAB, 0);
        v.post_behavior_event(evt);
      }

      virtual void roll_up(view &v, element *self, int levels = 1) {
        event_behavior evt(self, self, DO_SWITCH_TAB, levels);
        v.post_behavior_event(evt);
        // pv->start_timer(self, 50, 1);
      }

      virtual bool is_empty(const element *self, bool &yes) {
        yes = !current.is_valid();
        return true;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        if (val.is_date()) {
          // date dt = val.get_date();
          date_time t = val.get_date();
          return set_date(v, self, date(t, tz));
        }
        ustring sval = val.get(W(""));
        return set_text(v, self, sval);
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        /*
        element *cd = find_first(v, const_cast<element *>(self), WCHARS("td:current"));
        if (cd) {
          ustring sval = cd->attr_value();
          date dt(tz);
          if (dt.parse(string(sval))) {
            date_time t = to_date_time(dt);
            val = value(t, date_time::DT_HAS_DATE | date_time::DT_UTC);
          } else
            val = value(sval);
        }*/

        date_time t = to_date_time(current);
        val = value(t, date_time::DT_HAS_DATE | date_time::DT_UTC);
        return true;
      }

      virtual bool set_text(view &v, element *self, wchars text) {
        // ustring us(text,length);
        date dt(tz);
        if (dt.parse(string(text))) return set_date(v, self, dt);
        return false;
      }

      bool set_date(view &v, element *self, date dt) {
        current = dt;
        current_view->update_view(v, self, dt);
        current_view->go_to_day(v, self, dt);
        return true;
      }

      virtual bool notify_changed(view &v, element *self, uint reason) {
        {
          event_behavior evt(self, self, SELECT_VALUE_CHANGED, reason);
          v.post_behavior_event(evt);
        }
        return true;
      }

      virtual bool notify_state_changed(view &v, element *self, uint reason) {
        {
          event_behavior evt(self, self, UI_STATE_CHANGED, reason);
          v.post_behavior_event(evt);
        }
        return true;
      }


      element *get_target_td(element *self, element *b) {
        while (b) {
          if (self == b) return 0;
          if (b->tag == tag::T_TD || b->tag == tag::T_TH ||
              b->tag == tag::T_CAPTION)
            return b;
          else
            b = b->parent;
        }
        return 0;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) {
        element *td_parent = get_target_td(self, evt.target);

        switch (evt.cmd) {
        case MOUSE_DOWN: v.set_focus(self, BY_MOUSE); m_bPressed = true;
        case MOUSE_MOVE:
          if (!td_parent) return false;
          if (!evt.is_point_button() || !m_bPressed) return false;
          current_view->do_press(v, self, td_parent, uint(-1));
          return true;
        // case MOUSE_TICK:
        case MOUSE_UP:
          if (!m_bPressed) return false;
        case MOUSE_DCLICK:
          if (!evt.is_point_button()) return false;
          m_bPressed = false;
          if (evt.target->is_of_class("today") && !td_parent) {
            date      today(tz);
            today.set_today();
            current_view->update_view(v, self, today);
            current_view->go_to_day(v, self, today, CLICK_SYNTHESIZED);
          } else {
            if (!td_parent) return false;
            current_view->do_click(v, self, td_parent, evt, CLICK_BY_MOUSE);
          }
          return true;
        case MOUSE_WHEEL:
          if (!td_parent) return false;
          if (td_parent->tag == tag::T_CAPTION) {
            if (evt.target->is_of_class("month"))
              current_view->go_next_capval1(v, self, (int)evt.get_wheel_delta(),
                                            CLICK_SYNTHESIZED);
            else
              current_view->go_next_capval2(v, self, (int)evt.get_wheel_delta(),
                                            CLICK_SYNTHESIZED);
          } else
            current_view->go_next_val(v, self,
                                      evt.get_wheel_delta() < 0
                                          ? +current_view->line_size()
                                          : -current_view->line_size(),
                                      CLICK_SYNTHESIZED);
          return true;
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.cmd != KEY_DOWN) return false;

        switch (evt.key_code) {
        case KB_LEFT:
          current_view->go_next_val(v, self, -1, CLICK_BY_KEY);
          return true;
        case KB_RIGHT:
          current_view->go_next_val(v, self, +1, CLICK_BY_KEY);
          return true;
        case KB_UP:
          current_view->go_next_val(v, self, -current_view->line_size(),
                                    CLICK_BY_KEY);
          return true;
        case KB_DOWN:
          current_view->go_next_val(v, self, +current_view->line_size(),
                                    CLICK_BY_KEY);
          return true;

        case KB_PRIOR:
          if (evt.is_shift())
            current_view->go_next_capval2(v, self, -1, CLICK_BY_KEY);
          else
            current_view->go_next_capval1(v, self, -1, CLICK_BY_KEY);
          return true;
        case KB_NEXT:
          if (evt.is_shift())
            current_view->go_next_capval2(v, self, +1, CLICK_BY_KEY);
          else
            current_view->go_next_capval1(v, self, +1, CLICK_BY_KEY);
          return true;
        case KB_HOME: {
          current_view->go_to_firstval(v, self, CLICK_BY_KEY);
        }
          return true;
        case KB_RETURN: {
          if (current_view_id > 0) {
            drill_down(v, self);
            return true;
          } else
            return false;
        }
        case KB_END: {
          current_view->go_to_lastval(v, self, CLICK_BY_KEY);
        }
          return true;
        }
        return false;
      }

      virtual bool on_focus(view &v, element *self, event_focus &evt) {
        v.refresh(self);
        return true;
      }

      void get_value(element *self, date &dt) {
        string str = self->attr_value();
        dt.parse(str);
      }

      virtual bool max_intrinsic_width(view &v, element *self, int &value) {
        value = intrinsic.x;
        return true;
      }
      virtual bool min_intrinsic_width(view &v, element *self, int &value) {
        value = intrinsic.x;
        return true;
      }
      virtual bool max_intrinsic_height(view &v, element *self, int &value) {
        value = intrinsic.y;
        return true;
      }
      virtual bool min_intrinsic_height(view &v, element *self, int &value) {
        value = intrinsic.y;
        return true;
      }

      string get_view_mode() {
        switch (current_view_id) {
          default:
          case DAY_VIEW: return CHARS("days");
          case MONTH_VIEW: return CHARS("months");
          case YEAR_VIEW: return CHARS("years");
          case TENYEAR_VIEW: return CHARS("century");
        }
      }

      bool init_mode(string v) {
        if (v == CHARS("days")) current_view_id = DAY_VIEW;
        else if (v == CHARS("months")) current_view_id = MONTH_VIEW;
        else if (v == CHARS("years")) current_view_id = YEAR_VIEW;
        else if (v == CHARS("century")) current_view_id = TENYEAR_VIEW;
        else {
          current_view = views[current_view_id = DAY_VIEW];
          return false;
        }
        current_view = views[current_view_id];
        return true;
      }

      bool set_view_mode(string v) {
        if (!init_mode(v))
          return false;
        if (view *pv = self->pview()) {
          show_view(*pv, self);
          return true;
        }
        return false;
      }

      bool step_up(int_v n) {
        if (view* pv = self->pview()) {
          current_view->go_next_val(*pv, self, n.val(1), CLICK_SYNTHESIZED);
          return true;
        }
        return false;
      }
      bool step_down(int_v n) {
        if (view* pv = self->pview()) {
          current_view->go_next_val(*pv, self, n.val(-1), CLICK_SYNTHESIZED);
          return true;
        }
        return false;
      }

      SOM_PASSPORT_BEGIN_EX(calendar, calendar_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(stepUp, step_up),
          SOM_FUNC_EX(stepDown, step_down),
        )
        SOM_PROPS(
          SOM_VIRTUAL_PROP(mode, get_view_mode, set_view_mode),
        )
       SOM_PASSPORT_END
    };

    ctl *calendar_ctl_factory::create(element *el) {
      return new calendar_ctl(el);
    }

    struct date_ctl_factory : ctl_factory {
      date_ctl_factory() : ctl_factory("date") {}
      virtual ctl *create(element *el);
    };

    static date_ctl_factory *_date_ctl_factory = 0;

    struct date_ctl : ctl {
      date current;
      date pre_popup_value;
      datetime_t tz;
      date_time::date_format_order order;

      // bool    popup_shown;
      typedef abutton super;

      date_ctl(): current(0), pre_popup_value(0) {} //:popup_shown(false)

      virtual bool focusable(const element *self) { return true; }

      virtual const string &behavior_name() const {
        return _date_ctl_factory->name;
      }

      virtual CTL_TYPE get_type() { return CTL_DATE; }

      void init(view &v, element *self) {
        if (self->n_children() != 2 ||
          !self->has_child_of_type(tag::T_CAPTION) ||
          !self->has_children_of_type(tag::T_BUTTON)) {
          self->clear(&v);
          element *cap = new element(tag::T_CAPTION);
          self->append(cap);
          cap->state.non_tabable(true);
          cap->get_style(v);
          value m = gen_mask(v, self);
          v.call_behavior_method(cap, "mask", &m, 1, m);
          element *btn = new element(tag::T_BUTTON);
          btn->state.non_tabable(true);
          self->append(btn);
          cap->get_style(v);
          v.add_to_update(self, true);
        }
        else {
          element *cap = get_caption(v, self);
          cap->state.non_tabable(true);
          get_button(v, self)->state.non_tabable(true);
          cap->get_style(v);
          value m = gen_mask(v, self);
          v.call_behavior_method(cap, "mask", &m, 1, m);
        }
      }

      virtual bool attach(view &v, element *self) {

        current.shift = pre_popup_value.shift = tz = get_time_zone_shift(v, self);

        ctl::attach(v, self);
        init(v, self);
        self->state.invalid(true);

        if (get_value_from_attr(self, current)) 
          show_date(v, self, current);
        return true;
      }

      virtual bool is_empty(const element *self, bool &yes) {
        yes = current.is_empty();
        return true;
      }

      virtual void detach(view &v, element *self) { ctl::detach(v, self); }

      virtual void on_lang_change(view& pv, element* self) override {
        auto val = current;
        self->clear();
        init(pv, self);
        show_date(pv, self, val);
        pv.add_to_update(self, CHANGES_MODEL);
      }

      void notify_changed(view &v, element *self, uint reason) {
        get_caption_value(v, self, current);
        event_behavior evt(self, self, SELECT_VALUE_CHANGED, reason);
        v.post_behavior_event(evt);
      }

      bool get_value_from_attr(element *self, date &dt) {
        if (self->atts.exist(attr::a_value)) {
          string str = self->attr_value();
          return dt.parse(str);
        }
        return false;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        date dt(get_time_zone_shift(v,self));
        if (val.is_date()) {
          date_time t = val.get_date();
          dt = date(t, dt.shift);
        }
        set_caption_value(v, self, dt);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        /*date dt(tz);
        if (get_caption_value(v, self, dt)) {
          date_time t = to_date_time(dt);
          val = value(t);
        } else
          val.clear();*/
        val = value(to_date_time(current));
        return true;
      }

      element *get_caption(view &v, const element *self) {
        child_iterator cit(v, const_cast<element *>(self));
        for (element *b; cit(b);) {
          if (b->tag == tag::T_CAPTION) return b;
        }
        return 0;
      }

      element *get_button(view &v, element *self) {
        child_iterator cit(v, const_cast<element *>(self));
        for (element *b; cit(b);) {
          if (b->tag == tag::T_BUTTON) return b;
        }
        return 0;
      }

      void set_caption_value(view &v, element *self, const ustring &val) {
        date dt(tz);
        dt.parse(string(val));
        set_caption_value(v, self, dt);
      }

      void set_caption_value(view &v, element *self, date dt) {
        show_date(v, self, dt);
        /*
        element* cap = get_caption(v,self);
        if( dt.is_valid() )
        {
          show_date(v,self,dt);
        }
        else
        {
          if(cap) cap->set_text(v,wchars());
        }
        current = dt;
        if(cap)
          v.add_to_update(self,false);*/
      }

      void set_caption_value(view &v, element *self) {
        element *cd = find_first(v, self, WCHARS("td:current"));
        if (cd) {
          ustring val = cd->attr_value();
          set_caption_value(v, self, val);
        }
      }

      /*virtual bool child_focusable(const element* self, const element* child)
      {
        if( child == get_calendar(self) ) return false;
        if( child == get_caption(self) ) return false;
        return true;
      }*/

      /*ustring gen_mask(view& v, element* self)
      {
        date_time::date_format_order order; wchar separator;
        date_time::date_format(order,separator);
        ustring mask;
        switch(order)
        {
          default:
          case date_time::DMY:
          case date_time::MDY:
            mask = ustring::format(L"##%c##%c####", separator, separator);
            break;
          case date_time::YMD:
            mask = ustring::format(L"####%c##%c##", separator, separator);
            break;
        }
        return mask;
      }*/

      value gen_mask(view &v, element *self) {

/*      ustring format = self->get_attr("-format");

        if (format.is_defined()) {
          if (format.like(W("*YYYY*MM*DD*"))) order = date_time::YMD;
          else if (format.like(W("*DD*MM*YYYY*"))) order = date_time::DMY;
          else if (format.like(W("*MM*DD*YYYY*"))) order = date_time::MDY;
          else goto DEFAULT;
          format.replace(W("YYYY"), W("0000"));
          format.replace(W("MM"), W("00"));
          format.replace(W("DD"), W("DD"));
        }
      DEFAULT: */
        wchar                        separator[2];
        separator[1] = 0;
        date_time::date_format(order, separator[0], self->get_lang());

        value m = value::make_array(5);

        value f_sep = value(separator);

        value f_day = value::make_map();
        f_day.set_prop("type", value(WCHARS("integer")));
        f_day.set_prop("min", value(1));
        f_day.set_prop("max", value(31));
        f_day.set_prop("width", value(2));
        f_day.set_prop("leading-zero", value(true));
        f_day.set_prop("class", value(WCHARS("day")));

        value f_month = value::make_map();
        f_month.set_prop("type", value(W("integer")));
        f_month.set_prop("min", value(1));
        f_month.set_prop("max", value(12));
        f_month.set_prop("width", value(2));
        f_month.set_prop("leading-zero", value(true));
        f_month.set_prop("class", value(WCHARS("month")));

        value f_year = value::make_map();
        f_year.set_prop("type", value(WCHARS("integer")));
        f_year.set_prop("width", value(4));
        f_year.set_prop("leading-zero", value(true));
        f_year.set_prop("class", value(WCHARS("year")));

        switch (order) {
        default:
        case date_time::DMY:
          m.set_element(0, f_day);
          m.set_element(1, f_sep);
          m.set_element(2, f_month);
          m.set_element(3, f_sep);
          m.set_element(4, f_year);
          break;
        case date_time::MDY:
          m.set_element(0, f_month);
          m.set_element(1, f_sep);
          m.set_element(2, f_day);
          m.set_element(3, f_sep);
          m.set_element(4, f_year);
          break;
        case date_time::YMD:
          m.set_element(0, f_year);
          m.set_element(1, f_sep);
          m.set_element(2, f_month);
          m.set_element(3, f_sep);
          m.set_element(4, f_day);
          break;
        }
        return m;
      }

      void show_date(view &v, element *self, const date &dt) {
        element *cap = get_caption(v, self);
        if (!cap) return;

        current = dt;
        
        if (!dt.is_valid()) {
          self->state.invalid(true);
          cap->set_value(v, value());
          return;
        }

        date_time::date_format_order order;
        wchar                        separator;
        date_time::date_format(order, separator, self->get_lang());

        value varr = value::make_array(3);
        switch (order) {
        case date_time::MDY:
          varr.set_element(0, value(dt.month));
          varr.set_element(1, value(dt.day));
          varr.set_element(2, value(dt.year));
          break;
        case date_time::DMY:
          varr.set_element(0, value(dt.day));
          varr.set_element(1, value(dt.month));
          varr.set_element(2, value(dt.year));
          break;
        case date_time::YMD:
          varr.set_element(0, value(dt.year));
          varr.set_element(1, value(dt.month));
          varr.set_element(2, value(dt.day));
          break;
        default: assert(false); return;
        }
        self->state.invalid(false);
        cap->set_value(v, varr);
      }

      bool get_caption_value(view &v, const element *self, date &dt) {
        element *cap = get_caption(v, self);
        if (!cap) return false;

        ustring text = cap->get_text(v);

        date_time::date_format_order order;
        wchar                        separator;
        date_time::date_format(order, separator, self->get_lang());

        value varr;
        if (!cap->get_value(v, varr)) return false;
        if (!varr.is_array_like()) return false;

        if (varr.size() != 3) return false;

        value t;

        switch (order) {
        case date_time::MDY:
          t = varr.get_element(0);
          if (t.is_undefined()) return false;
          dt.month = t.get(0);
          t        = varr.get_element(1);
          if (t.is_undefined()) return false;
          dt.day = t.get(0);
          t      = varr.get_element(2);
          if (t.is_undefined()) return false;
          dt.year = t.get(0);
          break;
        case date_time::DMY:
          t = varr.get_element(0);
          if (t.is_undefined()) return false;
          dt.day = t.get(0);
          t      = varr.get_element(1);
          if (t.is_undefined()) return false;
          dt.month = t.get(0);
          t        = varr.get_element(2);
          if (t.is_undefined()) return false;
          dt.year = t.get(0);
          break;
        case date_time::YMD:
          t = varr.get_element(0);
          if (t.is_undefined()) return false;
          dt.year = t.get(0);
          t       = varr.get_element(1);
          if (t.is_undefined()) return false;
          dt.month = t.get(0);
          t        = varr.get_element(2);
          if (t.is_undefined()) return false;
          dt.day = t.get(0);
          break;
        default: return false; break;
        }
        current = dt;
        return dt.is_valid();
      }

      virtual bool on(view &v, element *self, event_mouse &evt) {
        if (evt.cmd == MOUSE_DOWN || evt.cmd == (MOUSE_DOWN | EVENT_HANDLED) ||
            evt.cmd == MOUSE_DCLICK ||
            evt.cmd == (MOUSE_DCLICK | EVENT_HANDLED)) {
          element *btn = get_button(v, self);
          if (btn && evt.target->belongs_to(btn, true) &&
              !self->state.owns_popup()) {
            show_popup(v, self);
            return true;
          }
        }
        /*else if(evt.cmd == MOUSE_IDLE && !popup_shown)
        {
          show_popup(v, self);
          return true;
        }*/

        if (evt.cmd == (MOUSE_DOWN | EVENT_SINKING)) {
          element *cap = get_caption(v, self);
          if (cap && evt.target == cap) {
            v.set_focus(self, BY_MOUSE);
            activate_caption(v, self, true, BY_MOUSE);
            return false;
          }
        }

        return false;
      }

      void activate_caption(view &v, element *self, bool on,
                            FOCUS_CAUSE cause) {
        element *cap = get_caption(v, self);
        if (!cap) return;
        cap->state.current(on);
        cap->drop_layout(&v);
        cap->commit_measure(v);
        // v.add_to_update(cap,false);
        event_focus evt(cap, on ? FOCUS_IN : FOCUS_OUT, cause, true);
        cap->on(v, evt);
      }

      virtual bool on(view &v, element *self, event_focus &evt) override {
        if (evt.cmd == FOCUS_IN || evt.cmd == (FOCUS_IN | EVENT_HANDLED)) {
          activate_caption(v, self, true, evt.cause);
        } else if (evt.cmd == FOCUS_OUT ||
                   evt.cmd == (FOCUS_OUT | EVENT_HANDLED)) {
          // if(evt.target && evt.target->belongs_to(self, false))
          //  evt.cancel = true;
          if (evt.target && evt.target->belongs_to(v, self, true))
            activate_caption(v, self, false, evt.cause); // popup
          else {
            close_popup(v, self);
            activate_caption(v, self, false, evt.cause);
          }
        }
        else
          return false;
        return true;
      }

      virtual bool on(view &v, element *self, event_command &evt) override {
        element *cap = get_caption(v, self);
        if (cap) return cap->on(v, evt);
        return false;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override {

        /*if(evt.cmd == POPUP_READY)
         {
           popup_shown = true;
           return false;
         }
         else if(evt.cmd == POPUP_DISMISSED)
         {
           popup_shown = false;
           return false;
         }*/

        element *pup = get_popup(v, self);

        if (evt.cmd == EDIT_VALUE_CHANGED &&
            evt.target == get_caption(v, self)) {
          bool empty = false;
          is_empty(self, empty);
          if (!empty) {
            date dt(tz);
            bool is_valid = get_caption_value(v, self, dt);
            current = dt;
            if (is_valid && self->state.invalid())
              self->state_off(v, S_INVALID);
            else if (!is_valid && !self->state.invalid())
              self->state_on(v, S_INVALID);
          }
          else {
            current.set_empty();
          }
          event_behavior tevt(self, self, SELECT_VALUE_CHANGED, CLICK_BY_KEY);
          v.post_behavior_event(tevt);
          return true;
        } else if (evt.cmd == MASKED_EDIT_INCREMENT && evt.target == get_caption(v, self)) {
          date_time::date_format_order order;
          wchar                        separator;
          date_time::date_format(order, separator, self->get_lang());
          int yi, mi, di;
          switch (order) {
          default:
          case date_time::DMY:
            yi = 2;
            mi = 1;
            di = 0;
            break;
          case date_time::MDY:
            yi = 2;
            mi = 0;
            di = 1;
            break;
          case date_time::YMD:
            yi = 0;
            mi = 1;
            di = 2;
            break;
          }
          date_time now = date_time::now();
          //date_time dt(evt.data.get_element(yi).get(now.year()),
          //             evt.data.get_element(mi).get(now.month()),
          //             evt.data.get_element(di).get(now.day()));

          date_time dt(evt.data.get_element(yi).get(now.year()),
                    evt.data.get_element(mi).get(1),
                    evt.data.get_element(di).get(1));

          int cmd   = (short)hiword(uint(evt.reason));
          int group = loword(uint(evt.reason));
          if (group == yi) {
            switch (cmd) {
            case -2: dt.year(0); break;
            case +2: dt.year(9999); break;
            case -1: dt.year(max(0, int(dt.year()) - 1)); break;
            case +1: dt.year(min(9999, int(dt.year()) + 1)); break;
            }
            evt.data.set_element(yi, value(dt.year()));
          }
          else if (group == mi) {
            switch (cmd) 
            {
              case -2: dt.month(1); break;
              case +2: dt.month(12); break;
              case -1: 
                if (evt.data.get_element(mi).is_undefined())
                  dt.month(1);
                else
                  dt.month(dt.month() - 1); 
                break;
              case +1: 
                if (evt.data.get_element(mi).is_undefined())
                  dt.month(12);
                else
                  dt.month(dt.month() + 1);
                break;
            }
            evt.data.set_element(mi, value(dt.month()));
          }
          else if (group == di) {
            switch (cmd) {
            case -2: dt.day(1); break;
            case +2: dt.day(date_time::days_in_month(dt.year(), dt.month())); break;
            case -1: 
              if (evt.data.get_element(di).is_undefined())
                dt.day(1);
              else 
                dt.day(dt.day() - 1);
              break;
            case +1: 
              if (evt.data.get_element(di).is_undefined())
                dt.day(date_time::days_in_month(dt.year(), dt.month()));
              else
                dt.day(dt.day() + 1);
              break;
            }
            evt.data.set_element(di, value(dt.day()));
          }
          event_behavior tevt(self, self, SELECT_VALUE_CHANGED, CLICK_BY_KEY);
          v.post_behavior_event(tevt);
          return true;
        } else if (evt.cmd == CHANGE && pup && evt.target == pup->first_element()) {
          evt.target = self;
          if (!is_readonly(self)) set_caption_value(v, self);
          if (evt.reason == CLICK_BY_MOUSE) {
            event_behavior tevt(pup, pup, CLOSE_POPUP, 0);
            v.post_behavior_event(tevt);
            v.set_focus(self, BY_CODE);
            activate_caption(v, self, true, BY_CODE);
            // close_popup(v,self);
            // return true;
          }
          event_behavior tevt(self, self, SELECT_VALUE_CHANGED, CLICK_BY_MOUSE);
          v.post_behavior_event(tevt);
          evt.cancel_event();
          return true;
        }
        else if (evt.cmd == CHANGE && evt.target != self) {
          //self = self;
        }


        return false;
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        element *cal = get_calendar(v, self);

        if (cal && evt.cmd == KEY_DOWN) {
          if (evt.key_code == KB_RETURN || evt.key_code == KB_TAB) {
            close_popup(v, self);
            activate_caption(v, self, true, BY_CODE);
            v.set_focus(self, BY_KEY_NEXT);
            return evt.key_code == KB_TAB ? false : true;
          }
          if (evt.key_code == KB_ESCAPE) {
            close_popup(v, self);
            set_caption_value(v, self, pre_popup_value);
            activate_caption(v, self, true, BY_CODE);
            return true;
          }
        }

        if (cal) return cal->on(v, evt);

        if ((evt.cmd == KEY_DOWN) && (evt.key_code == KB_RETURN) && !cal) {
          show_popup(v, self);
          return true;
        }
        if (evt.cmd == KEY_DOWN && (evt.key_code == KB_DOWN) &&
            evt.is_shortcut() && !cal) {
          show_popup(v, self);
          return true;
        }

        element *cap = get_caption(v, self);
        if (cap && cap->on(v, evt)) {
          date dt(tz);
          get_caption_value(v, self, dt);
          // v.refresh(self);
          // self->drop_styles(v);
          current = dt;
          // v.refresh(self);
          return true;
        }
        return false;
      }

      /*virtual bool do_press(view& v, element* self, element* target, bool on,
      bool by_mouse)
      {
        show_popup(v, self);
        return super::do_press(v, self, target, on, by_mouse);
      }*/

      virtual bool do_click(view &v, element *self, element *target,
                            bool by_mouse = false) {
        show_popup(v, self);
        return true;
      }

      element *get_popup(view &v, element *self, bool create = false) {
        child_iterator cit(v, self);
        for (element *b; cit(b);) {
          if (b->tag == tag::T_POPUP) {
            assert(b->first_element());
            // b->drop_layout(&v);
            return b;
          }
        }
        if (create) {
          date dt(tz);
          if (!get_caption_value(v, self, dt)) dt.year = 0;
          element *popup = new element(tag::T_POPUP);
          self->append(popup);
          element *cal = new element(tag::T_WIDGET);
          cal->atts.set(attr::a_type, "calendar");
          if (dt.year) cal->atts.set(attr::a_value, dt.to_string());
          popup->append(cal);
          return popup;
        }
        return 0;
      }

      element *get_calendar(view &v, element *self) {
        element *p = get_popup(v, self);
        if (p && p->state.popup() && p->n_children()) return p->first_element();
        return 0;
      }

      virtual void show_popup(view &v, element *self) {
        element *popup = get_popup(v, self, true);
        get_caption_value(v, self, pre_popup_value);
        activate_caption(v, self, false, BY_CODE);
        if (!popup->state.popup()) {
          helement cal = popup->first_element();
          v.show_popup(popup, self, POPUP_WINDOW, 0x12);
          if (pre_popup_value.is_valid())
            cal->set_value(v, value::make_date(to_date_time(pre_popup_value).time()));
          v.set_focus(popup->first_element(), BY_CODE);
          // WRONG!: popup->subs[0]->state_on(*v,S_CURRENT);
          // calendar already has current element
        }
      }

      virtual void close_popup(view &v, element *self) {
        element *popup = get_popup(v, self, false);
        if (popup) {
          if (popup->state.popup()) v.close_popup(popup, false);
          // popup->remove(true);
        }
      }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) {
        chars fname = chars_of(name);
#define ACTION(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        ACTION(0, now) {
          date dt(tz);
          dt.set_today();
          set_caption_value(v, self, dt);
          return true;
        }
        ACTION(0, today) {
          date dt(tz);
          dt.set_today();
          set_caption_value(v, self, dt);
          return true;
        }
        ACTION(0, showPopup) {
          show_popup(v, self);
          return true;
        }
#undef ACTION
        return ctl::on_x_method_call(v, self, name, argv, argc, retval);
      }
    };

    ctl *date_ctl_factory::create(element *el) { return new date_ctl(); }

    struct time_ctl_factory : ctl_factory {
      time_ctl_factory() : ctl_factory("time") {}
      virtual ctl *create(element *el);
    };

    static time_ctl_factory *_time_ctl_factory = 0;

    struct time_ctl : ctl {
      time        current;
      datetime_t  tz;
      typedef ctl super;

      time_ctl(): current(0) {}

      virtual bool focusable(const element *self) { return true; }

      virtual const string &behavior_name() const {
        return _time_ctl_factory->name;
      }

      virtual CTL_TYPE get_type() { return CTL_TIME; }

      void init(view &v, element *self) {
        child_iterator cid(v, self);
        element *      b = 0;
        if (self->n_children() != 3 || (cid(b), b->tag != tag::T_CAPTION) ||
          (cid(b), b->tag != tag::T_BUTTON) ||
          (cid(b), b->tag != tag::T_BUTTON)) {
          self->clear(&v);
          element *cap = new element(tag::T_CAPTION);
          self->append(cap);
          cap->state.non_tabable(true);
          cap->get_style(v);
          value m = gen_mask(v, self);
          v.call_behavior_method(cap, "mask", &m, 1, m);
          element *b1 = new element(tag::T_BUTTON);
          b1->atts.set(attr::a_class, WCHARS("plus"));
          b1->state.non_tabable(true);
          self->append(b1);
          element *b2 = new element(tag::T_BUTTON);
          b2->atts.set(attr::a_class, WCHARS("minus"));
          b2->state.non_tabable(true);
          self->append(b2);
        }
      }

      virtual bool attach(view &v, element *self) {
        current.shift = tz = get_time_zone_shift(v, self);
        ctl::attach(v, self);
        init(v, self);
        if (get_value_from_attr(self, current))
          show_time(v, self,current); // set_caption_value(v,self,current); cannot be used!
        // self->measure(*v);
        v.add_to_update(self, CHANGES_DIMENSION);
        return true;
      }

      virtual bool is_empty(const element *self, bool &yes) {
        yes = current.is_empty();
        return true;
      }

      virtual void detach(view &v, element *self) { ctl::detach(v, self); }

      virtual void on_lang_change(view& pv, element* self) override {
        auto val = current;
        self->clear();
        auto ln = self->get_lang();
        init(pv, self);
        show_time(pv, self, val);
        pv.add_to_update(self, CHANGES_MODEL);
      }


      void notify_changed(view &v, element *self, uint reason) {
        get_caption_value(v, self, current);
        event_behavior evt(self, self, EDIT_VALUE_CHANGED, reason);
        v.post_behavior_event(evt);
      }

      bool get_value_from_attr(element *self, time &dt) {
        string str = self->attr_value();
        if (str.length()) return dt.parse(str);
        return false;
      }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        time dt(0);
        if (val.is_date()) {
          date_time t = val.get_date();
          dt = time(t, tz);
        }
        set_caption_value(v, self, dt);
        return true;
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        /*time t(tz);
        if (get_caption_value(v, self, t)) {
          // val = value::make_duration();

          bool      has_seconds = !is_date_short(self);
          date_time dt;
          uint      u = date_time::DT_HAS_TIME | date_time::DT_UTC;

          if (has_seconds) u |= date_time::DT_HAS_SECONDS;

          tool::date_time today = tool::date_time::now();
          date d(today,tz);
          dt = to_date_time(d,t);

          val = value(dt, u);
        } else
          val.clear();*/

        uint      u = date_time::DT_HAS_TIME | date_time::DT_UTC;
        bool      has_seconds = !is_date_short(self);
        if (has_seconds) u |= date_time::DT_HAS_SECONDS;
        
        date_time dt;
        tool::date_time today = tool::date_time::now();
        date d(today, tz);
        dt = to_date_time(d, current);
        val = value(dt, u);
        return true;
      }

      element *get_caption(view &v, element *self) {
        child_iterator cit(v, self);
        for (element *b; cit(b);) {
          if (b->tag == tag::T_CAPTION) return b;
        }
        return 0;
      }

      element *get_minus_button(element *self) {
        element *b = self->child(2);
        assert(b && b->tag == tag::T_BUTTON);
        if (!b || b->tag != tag::T_BUTTON) return 0;
        return b;
      }
      element *get_plus_button(element *self) {
        element *b = self->child(1);
        assert(b && b->tag == tag::T_BUTTON);
        if (!b || b->tag != tag::T_BUTTON) return 0;
        return b;
      }

      void set_caption_value(view &v, element *self, const ustring &val) {
        time dt(get_time_zone_shift(v, self));
        if (val.length()) {
          if (!dt.parse(val)) return;
        }
        set_caption_value(v, self, dt);
      }

      void set_caption_value(view &v, element *self, time dt) {
        show_time(v, self, dt);
        current = dt;
        v.refresh(self);
        self->drop_layout(&v);
        self->commit_measure(v);
        v.refresh(self);
      }

      /*void set_caption_value(view& v, element* self)
      {
        element *cd = find_first(self,"td:current");
        if( cd )
        {
          ustring val = cd->attr_value();
          set_caption_value(pv, self, val );
        }
      }*/

      virtual bool child_focusable(const element *self, const element *child) {
        return false;
      }

      bool is_date_short(const element *self) {
        return self->has_attr("-no-seconds");
      }

      value gen_mask(view &v, element *self) {
        date_time::time_format_hours      hours_fmt;
        date_time::time_format_marker_pos marker_pos;
        ustring                           am_text;
        ustring                           pm_text;
        date_time::time_format(hours_fmt, marker_pos, am_text, pm_text,
                               self->get_lang());
        // int am_pm_length = max(am_text.length(),pm_text.length());
        // ustring am_pm = ustring('@',max(am_text.length(),pm_text.length()));

        bool has_seconds = !is_date_short(self);

        value m = value::make_array();

        value fminutes = value::make_map();
        fminutes.set_prop("type", value(CHARS("integer")));
        fminutes.set_prop("class", value(WCHARS("minute")));
        fminutes.set_prop("min", value(0));
        fminutes.set_prop("max", value(59));
        fminutes.set_prop("leading-zero", value(true));
        fminutes.set_prop("width", value(2));

        value fseconds;
        if (has_seconds) {
          fseconds = value::make_map();
          fseconds.set_prop("type", value("integer"));
          fseconds.set_prop("class", value(WCHARS("second")));
          fseconds.set_prop("min", value(0));
          fseconds.set_prop("max", value(59));
          fseconds.set_prop("leading-zero", value(true));
          fseconds.set_prop("width", value(2));
        }

        if (hours_fmt == date_time::H_24) {
          value f24 = value::make_map();
          f24.set_prop("type", value(CHARS("integer")));
          f24.set_prop("class", value(WCHARS("hour")));
          f24.set_prop("min", value(0));
          f24.set_prop("max", value(23));
          f24.set_prop("width", value(2));
          m.set_element(0, f24);
          m.set_element(1, value(WCHARS(":")));
          m.set_element(2, fminutes);
          if (has_seconds) {
            m.set_element(3, value(WCHARS(":")));
            m.set_element(4, fseconds);
          }
        } else if (marker_pos == date_time::MP_BEFORE) {
          // mask = am_pm + L" ##:##:##";
          value am_pm_items = value::make_array(2);
          am_pm_items.set_element(0, value(am_text));
          am_pm_items.set_element(1, value(pm_text));

          value am_pm = value::make_map();
          am_pm.set_prop("type", value("enum"));
          am_pm.set_prop("class", value("ampm"));
          am_pm.set_prop("items", am_pm_items);

          value f12 = value::make_map();
          f12.set_prop("type", value("integer"));
          f12.set_prop("class", value(WCHARS("hour")));
          f12.set_prop("min", value(1));
          f12.set_prop("max", value(12));
          f12.set_prop("width", value(2));
          m.set_element(0, am_pm);
          m.set_element(1, value(WCHARS(" ")));
          m.set_element(2, f12);
          m.set_element(3, value(WCHARS(":")));
          m.set_element(4, fminutes);
          if (has_seconds) {
            m.set_element(5, value(WCHARS(":")));
            m.set_element(6, fseconds);
          }
        } else {
          value am_pm_items = value::make_array(2);
          am_pm_items.set_element(0, value(am_text));
          am_pm_items.set_element(1, value(pm_text));

          value am_pm = value::make_map();
          am_pm.set_prop("type", value("enum"));
          am_pm.set_prop("class", value("ampm"));
          am_pm.set_prop("items", am_pm_items);

          value f12 = value::make_map();
          f12.set_prop("type", value("integer"));
          f12.set_prop("class", value(WCHARS("hour")));
          f12.set_prop("min", value(1));
          f12.set_prop("max", value(12));
          f12.set_prop("width", value(2));
          m.set_element(0, f12);
          m.set_element(1, value(WCHARS(":")));
          m.set_element(2, fminutes);
          if (has_seconds) {
            m.set_element(3, value(WCHARS(":")));
            m.set_element(4, fseconds);
            m.set_element(5, value(WCHARS(" ")));
            m.set_element(6, am_pm);
          } else {
            m.set_element(3, value(WCHARS(" ")));
            m.set_element(4, am_pm);
          }
        }
        return m;
      }

      value time2varr(const element *self, const time &dt) {
        date_time::time_format_hours      hours_fmt;
        date_time::time_format_marker_pos marker_pos;
        ustring                           am_text;
        ustring                           pm_text;
        date_time::time_format(hours_fmt, marker_pos, am_text, pm_text,
                               self->get_lang());

        bool has_seconds = !is_date_short(self);

        if (hours_fmt == date_time::H_24) {
          value varr = value::make_array(has_seconds ? 3 : 2);
          // text = ustring::format(L"%02u:%02u:%02u", dt.hours, dt.minutes,
          // dt.seconds);
          varr.set_element(0, value(dt.hours));
          varr.set_element(1, value(dt.minutes));
          if (has_seconds) varr.set_element(2, value(dt.seconds));
          return varr;
        } else if (marker_pos == date_time::MP_BEFORE) {
          int h = dt.hours % 12;
          if (!h)
            h = 12; // there was no concept of the zero at the time this was
                    // invented. 12 is serving a role of zero in this system.
                    // Pure people, "12:46 am" for them is before "11:05 am",
                    // doh!
          // text = ustring::format(L"%s %02u:%02u:%02u", dt.hours < 12?
          // am_text.c_str():pm_text.c_str(), h, dt.minutes, dt.seconds);
          value varr = value::make_array(has_seconds ? 4 : 3);
          varr.set_element(0, value(dt.hours < 12 ? am_text : pm_text));
          varr.set_element(1, value(h));
          varr.set_element(2, value(dt.minutes));
          if (has_seconds) varr.set_element(3, value(dt.seconds));
          return varr;
        } else if (marker_pos == date_time::MP_AFTER) {
          int h = dt.hours % 12;
          if (!h) h = 12;
          // text = ustring::format(L"%02u:%02u:%02u %s", h, dt.minutes,
          // dt.seconds, dt.hours < 12? am_text.c_str():pm_text.c_str());
          value varr = value::make_array(has_seconds ? 4 : 3);
          varr.set_element(0, value(h));
          varr.set_element(1, value(dt.minutes));
          if (has_seconds) {
            varr.set_element(2, value(dt.seconds));
            varr.set_element(3, value(dt.hours < 12 ? am_text : pm_text));
          } else {
            varr.set_element(2, value(dt.hours < 12 ? am_text : pm_text));
          }

          return varr;
        }
        return value();
      }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override {
        chars fname = chars_of(name);
#define ACTION(ARGC, NAME) if (argc == ARGC && fname == CHARS(#NAME))
        ACTION(0, now) {
          time dt(get_time_zone_shift(v, self));
          dt.set_now();
          set_caption_value(v, self, dt);
          return true;
        }
#undef ACTION
        return ctl::on_x_method_call(v, self, name, argv, argc, retval);
      }

      void show_time(view &v, element *self, const time &dt) {
        element *cap = get_caption(v, self);
        if (!cap) return;

        current = dt;

        if (current.is_valid()) {
          value varr = time2varr(self, current);
          cap->set_value(v, varr);
        } else
          cap->set_value(v, value());
      }

      time varr2time(const element *self, const value &varr, int *hi = 0,
                     int *mi = 0, int *si = 0) {
        time dt(0); 
        if (!varr.is_array_like()) return dt;
        date_time::time_format_hours      hours_fmt;
        date_time::time_format_marker_pos marker_pos;
        ustring                           am_text;
        ustring                           pm_text;
        date_time::time_format(hours_fmt, marker_pos, am_text, pm_text,
                               self->get_lang());

        bool has_seconds = !is_date_short(self);

        if (hours_fmt == date_time::H_24) {
          // n = swscanf(text, L"%02u:%02u:%02u", &dt.hours, &dt.minutes,
          // &dt.seconds);
          if (varr.size() != (has_seconds ? 3u : 2u)) return dt;
          value t = varr.get_element(0);
          if (t.is_undefined()) return dt;
          dt.hours = t.get(0);
          t        = varr.get_element(1);
          if (t.is_undefined()) return dt;
          dt.minutes = t.get(0);
          if (has_seconds) {
            t = varr.get_element(2);
            if (t.is_undefined()) return dt;
            dt.seconds = t.get(0);
          } else
            dt.seconds = 0;

          if (hi) {
            *hi = 0;
            *mi = 1;
            *si = 2;
          }
        } else if (marker_pos == date_time::MP_BEFORE) {
          // n = swscanf(text, L"%s %02u:%02u:%02u", buf, &dt.hours,
          // &dt.minutes, &dt.seconds);
          if (varr.size() != (has_seconds ? 4u : 3u)) return dt;
          value t = varr.get_element(1);
          if (t.is_undefined()) return dt;
          dt.hours = t.get(0);
          t        = varr.get_element(2);
          if (t.is_undefined()) return dt;
          dt.minutes = t.get(0);

          if (has_seconds) {
            t = varr.get_element(3);
            if (t.is_undefined()) return dt;
            dt.seconds = t.get(0);
          } else
            dt.seconds = 0;

          t = varr.get_element(0);
          if (!t.is_string()) return dt;

          if (dt.hours == 12) dt.hours = 0;
          if (t.get(W("")) == pm_text) dt.hours += 12;

          if (hi) {
            *hi = 1;
            *mi = 2;
            *si = 3;
          }
        } else if (marker_pos == date_time::MP_AFTER) {
          // n = swscanf(text, L"%02u:%02u:%02u %s", &dt.hours, &dt.minutes,
          // &dt.seconds, buf);
          if (varr.size() != (has_seconds ? 4u : 3u)) return dt;
          value t = varr.get_element(0);
          if (t.is_undefined()) return dt;
          dt.hours = t.get(0);
          t        = varr.get_element(1);
          if (t.is_undefined()) return dt;
          dt.minutes = t.get(0);

          if (has_seconds) {
            t = varr.get_element(2);
            if (t.is_undefined()) return dt;
            dt.seconds = t.get(0);
            t          = varr.get_element(3);
          } else {
            dt.seconds = 0;
            t          = varr.get_element(2);
          }

          if (!t.is_string()) return dt;

          if (dt.hours == 12) dt.hours = 0;
          if (t.get(W("")) == pm_text) dt.hours += 12;
          if (hi) {
            *hi = 0;
            *mi = 1;
            *si = 2;
          }
        }
        return dt;
      }

      bool get_caption_value(view &v, element *self, time &dt) {
        element *cap = get_caption(v, self);
        if (!cap) return false;

        // ustring text = cap->get_text();
        value varr;
        if (!cap->get_value(v, varr)) return false;
        if (!varr.is_array_like()) return false;

        dt = varr2time(self, varr);
        return dt.is_valid();
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        if (evt.cmd == MOUSE_DOWN || evt.cmd == (MOUSE_DOWN | EVENT_HANDLED) ||
            evt.cmd == MOUSE_DCLICK ||
            evt.cmd == (MOUSE_DCLICK | EVENT_HANDLED)) {
          if (evt.target->belongs_to(self, true)) {
            element *cap = get_caption(v, self);
            v.set_focus(self, BY_MOUSE);
            if (cap && evt.target == cap) {
              activate_caption(v, self, true, BY_MOUSE);
              return true;
            }
          }
        }
        return false;
      }

      virtual bool on(view &v, element *self, event_command &evt) override {
        element *cap = get_caption(v, self);
        if (cap) return cap->on(v, evt);
        return false;
      }

      void activate_caption(view &v, element *self, bool on,
                            FOCUS_CAUSE cause) {
        element *cap = get_caption(v, self);
        if (!cap) return;
        event_focus evt(self, on ? FOCUS_IN : FOCUS_OUT, cause, true);
        cap->state.current(on);
        if (on)
          select_default_group(v, self);
        cap->on(v, evt);
        /*if(on)
        {
          //dbg_printf("caption activated\n");
          v.set_focus(self,cause);
        }*/
      }

      virtual bool on(view &v, element *self, event_focus &evt) {
        if (evt.cmd == FOCUS_IN || evt.cmd == (FOCUS_IN | EVENT_HANDLED))
          activate_caption(v, self, true, evt.cause);
        else if (evt.cmd == FOCUS_OUT ||
          evt.cmd == (FOCUS_OUT | EVENT_HANDLED))
          activate_caption(v, self, false, evt.cause);
        else
          return false;
        return true;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) {

        if ((evt.cmd == (EVENT_SINKING | EDIT_VALUE_CHANGED)) && evt.target == get_caption(v, self)) {
          notify_changed(v, self, uint(evt.reason));
          return true;
        } else if (evt.cmd == MASKED_EDIT_INCREMENT &&
                   evt.target == get_caption(v, self)) {
          int hi = -1, mi = -1, si = -1;

          bool has_seconds = !is_date_short(self);

          time      t = varr2time(self, evt.data, &hi, &mi, &si);
          date_time dt(2000, 1, 1, t.hours, t.minutes, t.seconds);

          int cmd   = (short)hiword(uint(evt.reason));
          int group = loword(uint(evt.reason));
          if (group == hi) switch (cmd) {
            case -2: dt.hours(0); break;
            case +2: dt.hours(23); break;
            case -1: dt.hours(dt.hours() - 1); break;
            case +1: dt.hours(dt.hours() + 1); break;
            }
          else if (group == mi)
            switch (cmd) {
            case -2: dt.minutes(0); break;
            case +2: dt.minutes(59); break;
            case -1: dt.minutes(dt.minutes() - 1); break;
            case +1: dt.minutes(dt.minutes() + 1); break;
            }
          else if (group == si && has_seconds)
            switch (cmd) {
            case -2: dt.seconds(0); break;
            case +2: dt.seconds(59); break;
            case -1: dt.seconds(dt.seconds() - 1); break;
            case +1: dt.seconds(dt.seconds() + 1); break;
            }
          else
            return false;

          t.hours   = dt.hours();
          t.minutes = dt.minutes();
          t.seconds = dt.seconds();
          evt.data  = time2varr(self, t);
          return true;
        }
        if (evt.cmd == BUTTON_PRESS && evt.target == get_minus_button(self)) {
          element * cap = get_caption(v, self);
          event_key ke(cap, KEY_DOWN, KB_DOWN, 0);
          return cap->on(v, ke);
        } else if (evt.cmd == BUTTON_PRESS && evt.target == get_plus_button(self)) {
          element * cap = get_caption(v, self);
          event_key ke(cap, KEY_DOWN, KB_UP, 0);
          return cap->on(v, ke);
        }
        return false;
      }

      void select_default_group(view &v, element *self) {
        ustring defgroup = self->atts.get_ustring("default-group");
        if (defgroup.is_defined()) {
          element* pgc = find_first(v, self, WCHARS("span:current"));
          if (pgc)
            return;
          element* pg = find_first(v, self, ustring::format(W("span[class^='%s']"), defgroup.c_str())());
          if (pg) {
            value gn = value(pg->index());
            v.call_behavior_method(self->first_element(), "selectGroup", &gn, 1, gn);
          }
        }
      }

      virtual bool on(view &v, element *self, event_key &evt) {
        /*
        element* cal = get_calendar(self);

        if(cal && evt.cmd == KEY_DOWN)
        {
          if( evt.key_code == KB_RETURN )
          {
            close_popup(v,self);
            activate_caption(v,self,true, BY_CODE);
            return true;
          }
          if( evt.key_code == KB_ESCAPE )
          {
            close_popup(v,self);
            set_caption_value(v,self,pre_popup_value);
            activate_caption(v,self,true, BY_CODE);
            return true;
          }
        }
   

        if(cal)
          return cal->on(*v,evt);

        if( evt.cmd == KEY_DOWN && (evt.key_code == KB_DOWN || evt.key_code ==
   KB_RETURN) && !cal)
        {
          show_popup(v,self);
          return true;
        }*/

        element *cap = get_caption(v, self);
        if (cap) {
          cap->on(v, evt);
          time dt(get_time_zone_shift(v,self));
          get_caption_value(v, self, dt);
          v.refresh(self);
          self->drop_styles(v);
          current = dt;
          v.refresh(self);
        }
        return false;
      }

      /*virtual bool do_click(view& v, element* self, element* target, bool
      by_mouse = false)
      {
        show_popup(v, self);
        //do_click(view& v, element* self, element* target, bool by_mouse =
      false)
        //return super::do_press(v, self, target, on, by_mouse);
        //return do_click(v, self, target, by_mouse);
        return true;
      }*/
    };

    ctl *time_ctl_factory::create(element *el) { return new time_ctl(); }

    void init_datetime() {
      ctl_factory::add(_calendar_ctl_factory = new calendar_ctl_factory());
      ctl_factory::add(_date_ctl_factory = new date_ctl_factory());
      ctl_factory::add(_time_ctl_factory = new time_ctl_factory());
    }

  } // namespace behavior
} // namespace html
