#ifndef __html_clipboard_h__
#define __html_clipboard_h__

#include "tool/tool.h"
#include "gool/gool.h"
//#include "html-dom.h"

namespace html {

  using namespace tool;
  using namespace gool;

  struct bookmark;
  struct selection_ctx;

  class clipboard {
  public:
    enum clipboard_format {
      cf_undefined = 0,
      cf_text      = 0x01,
      cf_html      = 0x02,
      cf_hyperlink = 0x04,
      cf_json      = 0x08,
      cf_file      = 0x10,
      cf_picture   = 0x20,
      cf_xml       = 0x40,

      format_count = 7
    };

    /*enum CONTENT_PRODUCER_ACTION {
      CPA_DEFAULT,
      CPA_SKIP,
      CPA_REPLACE,
      CPA_PREPEND,
      CPA_APPEND,
    };

    struct content_producer {
      CONTENT_PRODUCER_ACTION on_char(wchar inc, ustring& replc) { return
    CPA_DEFAULT; } CONTENT_PRODUCER_ACTION on_elem_start(element* el, ustring&
    replc) { return CPA_DEFAULT; } CONTENT_PRODUCER_ACTION on_elem_end(element*
    el, ustring& replc) { return CPA_DEFAULT; }
    }; */

    struct item : resource {
      clipboard_format data_type;
      virtual bytes    get_data() const { return bytes(); }
      virtual chars    item_type_name() const = 0;
    };
    // array<item>         items;

    struct html_item : item {
      static clipboard_format item_type() { return cf_html; }
      string val; // html that may contain <!--StartFragment--> and
                  // <!--EndFragment-->
      string url;
      string title;
      string generator;

      html_item() { data_type = cf_html; }

      virtual chars    item_type_name() const { return CHARS("html"); }

      static html_item *from_cf_html(bytes cf_html);
      static html_item *from_cf_html(chars cf_html) {
        return from_cf_html(bytes((const byte *)cf_html.start, cf_html.length));
      }
        //      array<byte> get_cf_html();
#if defined(OSX)
      static html_item *from_pasteboard(void *nspasteboard);
#endif
    };

    struct json_item : item {
	    static clipboard_format item_type() { return cf_json; }
      array<byte> val;
      json_item(bytes json) {
        data_type = cf_json;
        val       = json;
      }
      bytes get_data() const { return val(); }
      virtual chars    item_type_name() const { return CHARS("json"); }
    };

    struct text_item : item {
      //enum { item_type = cf_text };
	    static clipboard_format item_type() { return cf_text; }
      ustring val;

      text_item(const wchar *text) {
        data_type = cf_text;
        val       = text;
      }
      text_item(wchars text) {
        data_type = cf_text;
        val       = text;
      }
      text_item(const ustring &text) {
        data_type = cf_text;
        val       = text;
      }

      mutable string val_bytes;
      bytes          get_data() const {
        if (val_bytes.length() == 0) val_bytes = u8::cvt(val);
        return bytes(val_bytes.chars_as_bytes());
      }

      virtual chars    item_type_name() const { return CHARS("text"); }
    };
    struct link_item : item {
      //enum { item_type = cf_hyperlink };
	  static clipboard_format item_type() { return cf_hyperlink; }
      ustring caption;
      ustring url;
      link_item(ustring caption, ustring url) {
        data_type     = cf_hyperlink;
        this->caption = caption;
        this->url     = url;
      }

      mutable string val_bytes;
      bytes          get_data() const {
        if (val_bytes.length() == 0) {
          val_bytes = u8::cvt(url);
          val_bytes += '\0';
          val_bytes += u8::cvt(caption);
        }
        return val_bytes.chars_as_bytes();
      }
      virtual chars    item_type_name() const { return CHARS("link"); }
    };
    struct file_item : item {
      //enum { item_type = cf_file };
	  static clipboard_format item_type() { return cf_file; }
      array<ustring> filenames;
      file_item(array<ustring> &names) {
        data_type = cf_file;
        filenames.swap(names);
      }

      mutable string val_bytes;
      bytes          get_data() const {
        if (val_bytes.length() == 0)
          for (int n = 0; n < filenames.size(); ++n) {
            if (val_bytes.length()) val_bytes += '\0';
            val_bytes += u8::cvt(filenames[n]);
          }
        return val_bytes.chars_as_bytes();
      }
      virtual chars    item_type_name() const { return CHARS("file"); }
    };
    struct image_item : item {
      //enum { item_type = cf_picture };
  	  static clipboard_format item_type() { return cf_picture; }
      handle<gool::image> image;
      image_item(gool::image *im) {
        data_type = cf_picture;
        image     = im;
      }
      virtual chars    item_type_name() const { return CHARS("picture"); }
    };

    struct data : public resource {
      array<handle<item>> items;

      void add(clipboard::item *pi) {
        for (int i = 0; i < items.size(); ++i)
          if (pi->data_type == items[i]->data_type) {
            items[i] = pi;
            return;
          }
        items.push(pi);
      }
      template <typename IT> IT *get() {
        for (int i = 0; i < items.size(); ++i)
          if (IT::item_type() == items[i]->data_type) return (IT *)items[i].ptr();
        return 0;
      }

      bytes get_data(uint dt) {
        for (int i = 0; i < items.size(); ++i)
          if (dt == uint(items[i]->data_type)) return items[i]->get_data();
        return bytes();
      }

      clipboard::item *get_default() {
        if (items.size()) return items[0];
        return 0;
      }

      uint formats() const {
        uint r = 0;
        for (int i = 0; i < items.size(); ++i)
          r |= items[i]->data_type;
        return r;
      }

      void available_formats(function<bool(clipboard::clipboard_format cf)> cb) {
        for (int i = 0; i < items.size(); ++i) {
          clipboard::clipboard_format cf = items[i]->data_type;
          if (!cb(cf)) break;
        }
      }

      array<string> available_format_names() {
        array<string> out;
        for (int i = 0; i < items.size(); ++i) {
          out.push(items[i]->item_type_name());
        }
        return out;
      }


      /*
      clipboard_format  data_type;
      array<byte>       text;
      array<ustring>    file_list;
      ustring           link_caption;
      ustring           link_url;
      handle<dib>       image;
      data( clipboard_format cf ) : data_type(cf) {} */
    };

    clipboard() {}
    ~clipboard() {}

      // struct callback
      //{
      //  virtual bool has( clipboard_format cf ) = 0; // false to stop
      //  enumeration
      //};

#ifdef WINDOWS
    static CLIPFORMAT CF_HTML();
    static CLIPFORMAT CF_HYPERLINK();
    static CLIPFORMAT CF_FILEDESCRIPTOR();
    static CLIPFORMAT CF_FILECONTENTS();
    static CLIPFORMAT CF_JSON();
#endif

    static int available_formats() {
      int  mask = 0;
      auto cb   = [&mask](clipboard::clipboard_format cf) -> bool {
        mask |= cf;
        return true;
      };
      available_formats(cb);
      return mask;
    }

    static void available_formats(tool::function<bool(clipboard::clipboard_format cf)> cb); // true - continue , false - stop

    static array<string> available_format_names() { return get()->available_format_names(); }

    static void empty();

    static uint get_sequence_num(); // returns sequence number of clipboard data

    static data *get(uint formats = uint(-1));
    static bool  set(clipboard::data *cbdata);

    static bool get(ustring &data);
#ifdef WINDOWS
    static bool get_cf_html(array<byte> &text); // returns raw CF_HTML bytes
#endif
    static bool get_html(array<byte> &text, string &url); // parses CF_HTML
                                                          // bytes and returns
                                                          // pure html and URL
                                                          // if it is there.
    static bool get_text(array<byte> &text);
    static bool get_json(tool::value &json);
    static bool get_link(ustring &url, ustring &caption);
    static bool get_image(handle<gool::image> &img);
    static bool get_files(array<ustring> &filenames);

    // static bool set(const ustring &data);
    static bool set_text(wchars us);

    static bool _set_text(wchars us);
    static void _set_html(chars html);
    static void _set_image(const gool::image *data);
    static void _set_link(wchars url, wchars caption);
    static void _set_files(const array<ustring>& fi);
    // static void set(view& v, visitor& vis);
    static void set(view &v, selection_ctx &sctx);
    static void set(const gool::image *data);

    static void html_cf(view &v, const bookmark &start, const bookmark &end,
                        array<char> &html);
    static void html_cf(view &v, selection_ctx &sctx, array<char> &html);
    static void html_cf(chars html_fragment, chars url, array<char> &html);
    static void text_cf(view &v, selection_ctx &sctx, array<wchar> &text);

    // static void text_cf(view& v, selection_ctx& sctx, array<wchar>& text);
  };

} // namespace html

#endif
