#ifndef __html_behaviors_h__
#define __html_behaviors_h__

#include "html-behavior.h"
#include "html-dom.h"
#include "html-actions-stack.h"
#include "html-view.h"
#include "tool/tool.h"
#include "tool/tl_spell_checker.h"
//#include "html-dom-merge.h"
//#include "sdk-headers.h"

namespace html {
  namespace behavior {

    void shutdown_ctl_image();

    void init(bool start);

    // non-focusable but mouse clickable thing. Generates BUTTON_CLICK
    struct clickable : ctl {
      virtual CTL_TYPE get_type() { return CTL_CLICKABLE; }

      virtual const string &behavior_name() const;

      virtual bool focusable(const element *self) { return false; }

      virtual bool on(view &v, element *self, event_mouse &evt);
      virtual bool on(view &v, element *self, event_focus &evt);

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == ACTIVATE_CHILD && evt.target == self) {
          event_behavior evt(self, self, BUTTON_CLICK, CLICK_SYNTHESIZED);
          v.post_behavior_event(evt);
          return true;
        }
        return false;
      }

      virtual bool on_method_call(view &v, element *self, method_params *p);

      virtual element* r13n_container(element* self) override { return self; }
    };

    struct abutton : ctl // abstract button - root of all other buttons
    {
      int key_pressed;
      abutton() : key_pressed(0) {}
      virtual bool focusable(const element *self) { return true; }

      virtual bool attach(view &v, element *self) override;
      virtual void detach(view &v, element *self) override;

      virtual bool is_command(view &v, element *self) { return false; }

      virtual bool set_value(view &pv, element *self, const value &v) override;
      virtual bool get_value(view &pv, element *self, value &v) override;

      virtual bool on(view &v, element *self, event_mouse &evt);
      virtual bool on(view &v, element *self, event_key &evt);
      virtual bool on(view &v, element *self, event_focus &evt);

      virtual bool on(view &v, element *self, event_behavior &evt) {
        if (evt.cmd == ACTIVATE_CHILD && evt.target == self) {
          event_behavior evt(self, self, BUTTON_CLICK, CLICK_SYNTHESIZED);
          v.post_behavior_event(evt);
          return true;
        }
        return false;
      }

      virtual bool do_press(view &v, element *self, element *target, bool on,
                            bool by_mouse);
      virtual bool do_click(view &v, element *self, element *target,
                            bool by_mouse = false);
      virtual bool on_method_call(view &v, element *self, method_params *p);

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }
    };

    struct windowed : public ctl // child window
                    , public iwindow {
      HWINDOW  hwnd;
      element *self;

      DEFINE_TYPE_ID_DERIVED(windowed,ctl);

      windowed(view &v, HWINDOW hw, element *self, uint subs = uint(-1))
          : ctl(subs) {
        // iwindow::type = CHILD_WINDOW;
        init_type(CHILD_WINDOW);
        hwnd                   = hw;
        self->flags.has_window = 1;
        v.windows.push(this);
        root(self);
      }

      virtual element *root() const override {
        return self;
      } // root element hosted by the window
      virtual void root(element *r) override {
        self = r;
      } // root element hosted by the window

      virtual bool get_window(iwindow *&hw) {
        hw = this;
        return true;
      }

      virtual bool is_child() const { return true; }


      virtual const string &behavior_name() const {
        static string name = " window"; // space left there intentionally
        return name;
      }
      virtual CTL_TYPE get_type() { return CTL_WINDOW; }
      virtual bool     attach(view &v, element *self) {
        assert(hwnd);
        return true;
      }
      virtual void detach(view &v, element *self) {
        root(nullptr);
        self->flags.has_window = 0;
        v.windows.remove_by_value(this);
      }
      virtual bool get_hwnd(HWINDOW &hw) {
        hw = hwnd;
        return true;
      }

      // iwindow:
      virtual HWINDOW get_hwnd() const override { return hwnd; }
      virtual void    set_hwnd(HWINDOW hw) override { hwnd = hw; }
    };

    struct windowed_element : public ctl // child window
    {
      handle<iwindow> pwnd;

      windowed_element() {}

      virtual bool is_child() const { return false; }

      virtual const string &behavior_name() const;

      virtual CTL_TYPE get_type() { return CTL_WINDOW; }
      virtual bool     attach(view &v, element *self) {
        self->flags.has_window = 1;
        auto replacer = [&](view &v, element *self, element *anchor) -> rect {
          return self->border_box(v, element::TO_SCREEN);
        };
        pwnd = v.create_window(self, nullptr, gool::TOOL_WINDOW, replacer,
                               html::ELEMENT_WINDOW_DOM_ATTACHED);

        return true;
      }
      virtual void detach(view &v, element *self) {
        self->flags.has_window = 0;
        if (pwnd) { v.close_popup(self, false); }
      }

      virtual bool draw_foreground(view &v, element *self, graphics *sf,
                                   point pos) override {
        return sf->get_drawing_root() != self;
      }
      virtual bool draw_background(view &v, element *self, graphics *sf,
                                   point pos) override {
        return sf->get_drawing_root() != self;
      }
      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos) override {
        return sf->get_drawing_root() != self;
      }
      virtual bool draw_scrollbars(view &v, element *self, graphics *sf,
                                   point pos) override {
        return sf->get_drawing_root() != self;
      }
    };

    struct htmlarea_ctl : ctl, selection_ctx, editing_ctx 
    {
      element* self;
      uint     caret_opacity;

      //uint_v     last_dbl_click_clock;
      tristate_v mouse_down_on_selection; // was mouse down on selection.

      htmlarea_ctl() : caret_opacity(0) {}
      virtual ~htmlarea_ctl() {
        // self = self;
      }

      virtual CTL_TYPE       get_type() { return CTL_HTMLAREA; }
      virtual selection_ctx *get_selection_ctx() const override { return const_cast<htmlarea_ctl *>(this); }

      virtual const string &behavior_name() const;

      virtual bool focusable(const element *self) override {
        return !self->attr_disabled();
      }

      virtual bool is_readonly(const element *b) const override { return true; }
      virtual bool is_atomic() const { return false; }

      virtual bool attach(view &v, element *self) {
        this->self = self;
        return true;
      }
      virtual void detach(view &v, element *self) {
        this->self = 0;
        anchor = caret = bookmark();
      }

      virtual bool on(view &v, element *self, event_mouse &evt);
      virtual bool on(view &v, element *self, event_key &evt) {
        if (evt.target->belongs_to(self, true)) switch (evt.cmd) {
          case KEY_DOWN: return on_key_down(v, self, evt);
          case KEY_CHAR: return on_char(v, self, evt);
          case KEY_COMP_STRING: return on_ime_comp_string(v, self, evt);
          case KEY_RESULT_STRING: return on_ime_result_string(v, self, evt);
          }
        return false;
      }

      virtual bool on(view &v, element *self, event_focus &evt) override;
      virtual bool on_timer(view &v, element *self, timer_id tid,
                            TIMER_KIND kind) override;
      virtual bool on_method_call(view &v, element *self,
                                  method_params *p) override;
      virtual bool on(view &v, element *self, event_command &evt) override;
      virtual bool on(view &v, element *self, event_behavior &evt) override;

      virtual bool on_key_down(view &v, element *self, event_key &evt);
      virtual bool on_char(view &v, element *self, event_key &evt) {
        return false;
      }
      virtual bool on_ime_comp_string(view &v, element *self, event_key &evt) {
        return false;
      }
      virtual bool on_ime_result_string(view &v, element *self,
                                        event_key &evt) {
        return false;
      }

      // selection_ctx stuff
      virtual element *selection_root() const override { return self; }
      virtual argb     selection_back_color(view &v) override;
      virtual argb     selection_fore_color(view &v) override;
      virtual argb     selection_text_color(view &v) override;
      virtual argb     selection_caret_color(view &v) override;
      virtual bool     draw_content(view &v, element *self, graphics *sf,
                                    point pos) override;
      virtual bool     allow_text_selection_on(text_block *tb);
      virtual bool     draw_caret(view &v, graphics *pg,
                                  const caret_metrics &cm) override;

      // operations:
      virtual bool select_all(view &v);
      virtual bool can_copy(view &v) {
        return has_non_collapsed_selection();
      }

      virtual bool copy(view &v);

      virtual void show_caret(view &v, bool on);
      virtual void refresh_selection(view &v);
      bool         get_bookmark_place(view &v, bookmark bm, rect &rc);

      virtual bool move_caret(view &v, ADVANCE_DIR dir, bool keep_anchor,
                              uint n = 1);

      ENSURE_VISIBLE_MANNER scroll_caret_manner(view &v) const {
        style *cs     = self->get_style(v);
        bool   smooth = cs->smooth_scroll_step(true);
        return smooth ? SCROLL_SMOOTH : SCROLL;
      }

      // editing_ctx
      virtual element *root() override { return self; }
      virtual element *root_at(view &v, bookmark bm) override;
      virtual element *root_at(view &v, element *el) override;
      virtual bool     select(view &v, bookmark c,
                              bookmark a = bookmark()) override;
      virtual bookmark get_caret() override { return caret; }
      virtual bookmark get_anchor() override { return anchor; }

      // utils
      virtual bool ensure_visible(view &v, bookmark c,
                                  ENSURE_VISIBLE_MANNER how);

      virtual bool perform_drag(view &v, element *self, event_mouse &evt);

      virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }
    };

    enum CMD_STATES {
      CMD_AVAILABLE = 0,
      CMD_SELECTED  = 0x01,
      CMD_DISABLED  = 0x02,
    };

    struct tag_shelve {
      array<tag::symbol_t>        tags_to_apply;
      array<array<tag::symbol_t>> tag_lists_to_remove;

      bool apply_contains_one_of(slice<tag::symbol_t> t) {  return tags_to_apply().contains_one_of(t); }
      bool remove_contains_one_of(slice<tag::symbol_t> t) { FOREACH(i, tag_lists_to_remove) { if (tag_lists_to_remove[i]().contains_one_of(t)) return true; } return false; }

      bool apply_contains(tag::symbol_t t) { return apply_contains_one_of(slice<tag::symbol_t>(t)); }
      bool remove_contains(tag::symbol_t t) { return remove_contains_one_of(slice<tag::symbol_t>(t)); }

      bool unshelve_apply(slice<tag::symbol_t> t) {
        while (t)
          if (tags_to_apply.remove_by_value(t++))
            return true;
        return false;
      }
      bool unshelve_remove(slice<tag::symbol_t> t) {
        FOREACH(i, tag_lists_to_remove) {
          if (tag_lists_to_remove[i]().contains_one_of(t)) {
            tag_lists_to_remove.remove(i);
            return true;
          }
        }
        return false;
      }

      void push_apply(tag::symbol_t t) {
        assert(!tags_to_apply().contains(t));
        tags_to_apply.push(t);
      }
      void push_remove(slice<tag::symbol_t> t) {
        assert(!tag_lists_to_remove().contains(t));
        tag_lists_to_remove.push(t);
      }

      bool has_applies() const { return tags_to_apply.size() > 0; }
      bool has_removes() const { return tag_lists_to_remove.size() > 0; }

      bool is_empty() const { return !has_applies() && !has_removes(); }

      void each_apply(function<void(tag::symbol_t)> cb) { FOREACH(i, tags_to_apply) cb(tags_to_apply[i]);  }
      void each_remove(function<void(slice<tag::symbol_t>)> cb) { FOREACH(i, tag_lists_to_remove) cb(tag_lists_to_remove[i]()); }

      void clear() { tags_to_apply.clear(); tag_lists_to_remove.clear(); }

    };

    // ENABLE_BIT_ENUM_OPERATORS(CMD_STATES,uint)


    struct richtext_ctl : htmlarea_ctl,
                          media_vars_provider
    //, animation
    {
      typedef htmlarea_ctl super;

      struct morph_ctx : public morph_options {
        view& v;
        handle<action> op;
        richtext_ctl*  rt;

        morph_ctx(view& v_, richtext_ctl*  rt_, handle<action> op_) : v(v_), rt(rt_), op(op_) { children_only = true; }

        virtual bool change_text(text* node, wchars to_text) override;
        virtual bool remove_node(node* old_node) override;
        virtual bool insert_node(node* el, int index, node* node) override;
        virtual bool replace_node(node* old_node, node* by_node) override;
        virtual bool update_atts(node* of, const html::attribute_bag& from) override;
        virtual bool reposition_node(node* el, int index, node* node) override;
    };

#if defined(FANCY_CARET)
      uint caret_cycle; // how many times the caret blinked
#endif
      media_variables       _media_vars;
      handle<pump::request> _awaiting; // awaiting this request to be delivered
      handle<element>       _body; // is set if the richtext contains full <html><body> document.

      array<handle<action>> stack;
      handle<action> transact_group;

      struct caret_def {
        point offset;
        handle<gool::path> path;
      };

      caret_def _block_start;
      caret_def _block_end;

      caret_def _row_start;
      caret_def _row_end;

      caret_def block_start(view& v);
      caret_def block_end(view& v);

      caret_def row_start(view& v);
      caret_def row_end(view& v);

      int depth;

      uint_ptr modified_point;

      bool input_sequence_active;

      handle<spell_checker> _spell_checker;

      //< bold/italic/underline "registers"
      // on next char typed add/remove these:
      tag_shelve span_shelve;
      //>

      void    pop();
      void    drop_tail();
      action *top() const;

#if 0
      string      pasted_image_url;
      array<byte> pasted_image_data;
      string      pasted_image_mime_type;
#endif

      richtext_ctl()
          : input_sequence_active(false), depth(0), modified_point(0)
#if defined(FANCY_CARET)
            ,
            caret_cycle(0)
#endif
      {
      }
      virtual ~richtext_ctl() {
        depth = 0;
        drop_tail();
      }

      virtual bool attach(view &v, element *self);
      virtual void detach(view &v, element *self) {
        self->state.content_editable(false);
        self->drop_styles(v);
        return super::detach(v, self);
      }

      virtual CTL_TYPE      get_type() override { return CTL_RICHTEXT; }
      virtual const string &behavior_name() const;

      virtual bool wants_keyboard(const element *b) const {
        return !is_readonly(b);
      }

      virtual element* r13n_container(element* self) override {
        if (is_readonly(self))
          return self; /*reconcilliation is enabled*/
        //return super::r13n_container(self);
        return nullptr;
      }

      virtual bool is_richtext() const { return true; }
      virtual bool is_plaintext() const { return false; }

      virtual bool is_readonly(const element *b) const override {
        return ctl::is_readonly(b);
      }
      // virtual bool is_editable(const element* b) const override { return
      // has_selection() && super::is_editable(self); }

      virtual bool is_empty(const element *self, bool &yes) override
      {
        if (self->nodes.size() >= 2) {
          yes = false;
          return true;
        }
        yes = false;
        if (self->nodes.size() == 0) { yes = true; }
        else if (self->nodes.size() == 1)
        {
          // simple element containing one text node
          if (self->nodes[0]->is_text() && self->nodes[0].ptr_of<text>()->is_space()) {
            yes = true;
          }
          else if (self->nodes[0]->is_element() && self->nodes[0].ptr_of<element>()->is_empty()) {
            yes = true;
          }
        }
        return true;
      }

      virtual bool advance(view &v, bookmark &bm, ADVANCE_DIR cmd,
                           point vpt = point());
      virtual bool advance_forward(view &v, bookmark &bm);
      virtual bool advance_backward(view &v, bookmark &bm);
      bool         advance_bookmark(view &v, bookmark &bm, bool forward,
                                    function<bool(bookmark &bm)> on_pos);
      bookmark insert_text_block_placholder(view &v, element *pel, bool after);

      // editing root:
      virtual element *root() override {
        return _body.ptr() ? _body.ptr() : self;
      }

      virtual media_variables &media_vars() { return _media_vars; }

      virtual bool need_ime(element *self) {
        if (is_readonly(self)) return false;
        if (self->atts.exist("noime")) return false;
        return true;
      }

      void drop_undo_stack() {
        depth = 0;
        drop_tail();
      }

      virtual bool select(view &v, bookmark c,
                          bookmark a = bookmark()) override;

      virtual bool set_value(view &v, element *self, const value &val) override;
      virtual bool get_value(view &v, element *self, value &val) override;

      virtual bool set_text(view &v, element *self, wchars val) override;
      virtual bool get_text(view &v, element *self, ustring &val) override;

      virtual bool set_html(view &v, element *self, bytes html) override;

      virtual bool on_char(view &v, element *self, event_key &evt);
      virtual bool on_key_down(view &v, element *self, event_key &evt);
      virtual bool on_ime_comp_string(view &v, element *self, event_key &evt);
      virtual bool on_ime_result_string(view &v, element *self, event_key &evt);

      virtual bool clear_comp_chars(view &v) override;
      bookmark     remove_chars_at(view &v, bookmark start,
                                   bookmark end); // will not affect undo/redo
      bookmark     insert_chars_at(view &v, bookmark pos,
                                   wchars text); // will not affect undo/redo
      bool         insert_ime_range(view &v, wchars s, int_v caret_pos);

      virtual bool get_caret_place(view &v, element *self, rect &rc) override;

      virtual bool on_timer(view &v, element *self, timer_id tid,
                            TIMER_KIND kind) override;
      virtual void show_caret(view &v, bool onOff) override;
      virtual bool on(view &v, element *self, event_focus &evt) override;
      virtual bool on(view &v, element *self, event_command &evt) override;
      virtual bool on(view &v, element *self, event_exchange &evt) override;
      virtual bool on(view &v, element *self, event_behavior &evt) override;
      bool         set_target_to(view &v, bookmark pos);

      virtual argb selection_caret_color(view &v) override;
      virtual argb selection_tail_back_color(view &v) override;
      virtual bool draw_caret(view &v, graphics *pg,
                              const caret_metrics &cm) override;

      element *selection_contains(view &v, slice<tag::symbol_t> list, const attribute_bag &atts = attribute_bag());
      element *selection_contains(view & v, wchars selector /*:root is base parent*/) override;
      bool     selection_contains(view & v, wchars selector /*:root is base parent*/, array<helement>& list) override;

      tristate_v selection_contains_blocks(view &v, slice<tag::symbol_t> list, const attribute_bag &atts = attribute_bag());
      virtual bool selection_each(view & v, function<bool(element *, bool &skip)> visitor) override;
      bool each_element(view &v, bookmark start, bookmark end, function<bool(element *, bool &skip)> visitor);
      bool each_block_element(view &v, bookmark start, bookmark end, function<bool(element *, bool &skip)> visitor, bool ui = false);

      void on_document_status_changed(view &v, bool is_modified_now) {
        self->drop_style(&v);
        event_behavior evt(self, self, CONTENT_MODIFIED, is_modified_now);
        v.post_behavior_event(evt, true);
      }

      bool set_span(view &v, tag::symbol_t t,
                    const attribute_bag &atts = attribute_bag());
      // bool remove_spans(view& v, slice<tag::symbol_t> list, const
      // attribute_bag& atts = attribute_bag());
      bool toggle_span(view &v, slice<tag::symbol_t> list,
                       const attribute_bag &atts = attribute_bag());
      bool toggle_list(view &v, tag::symbol_t list, const attribute_bag &atts,
                       bool on_off);
      bool toggle_pre(view &v, const attribute_bag &atts, bool on_off);

      bool unwrap(view &v, tag::symbol_t t,
                  const attribute_bag &atts = attribute_bag());

      virtual bool delete_forward(view &v);
      virtual bool delete_backward(view &v);

      virtual bool merge_cells(view &v);
      virtual bool delete_rows(view &v);
      virtual bool delete_cols(view &v);
      virtual bool split_cells(view &v);
      virtual bool insert_column(view &v, bool after);
      virtual bool insert_row(view &v, bool after);

      virtual void node_expand(node *n, int at, int nitems);
      virtual void node_shrink(node *n, int at, int nitems);

      virtual bool check_empty(view &v, bookmark& bm);

      // operations
      virtual bool can_cut(view &v) { return is_editable(self) && can_copy(v); }
      virtual bool cut(view &v);
      virtual bool paste(view &v);
      virtual bool paste_text(view &v);
      virtual bool paste(view &v, clipboard::data *what,
                         bookmark where = bookmark());
      virtual bool can_paste(view &v);

      wchars unindents_selector() {
        return WCHARS("blockquote,blockquote>*,li,li>*,dd,dd>*,dt");
      }

      virtual bool can_indent(view &v);
      virtual bool can_unindent(view &v);
      virtual bool do_indent(view &v);
      virtual bool do_unindent(view &v);

      // richtext as a frame
      bool load_html(view &v, element *self, const string &url, bytes html,  const string &encoding = string());
      bool _load_html(view &v, element *self, const string &url, bytes html, const string &encoding);
      bool merge_html(view &v, element *self, const string &url, bytes html, const string &encoding, morph_options* pmr = nullptr);
      bool check_cannonic_document_structure(view& v, element *self, handle<action> op);
      bool check_cannonic_document_structure(view& v, element *self);

      virtual bool on_data_request(view &v, element *self, pump::request *rq);
      virtual bool on_data_arrived(view &v, element *self, pump::request *rq);
      void update_media_vars(view &v, element *self, bool reset_styles = true);
      void set_media_vars(view &v, element *self, value map, bool reset,
                          bool reapply_styles);

#ifndef SCITERJS
      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval);
#endif

      void push(view &v, action *act);

      bool get_modified() const { return (uint_ptr)top() != modified_point; }
      void set_modified(bool on) {
        if (on)
          modified_point = 0;
        else
          modified_point = (uint_ptr)top();
      }
      void set_modified(view &v, bool on) {
        bool was_modified = get_modified();
        set_modified(on);
        bool is_modified = get_modified();
        if (was_modified != is_modified)
          on_document_status_changed(v, is_modified);
      }

      // bool   show_composition() const { return is_composition; } //
      // smartphone input mode.

      bool undo(view &v);
      int  can_redo(view &v) const {
        return stack.size() ? stack.size() - depth : 0;
      }

      bool redo(view &v);
      int  can_undo(view &v) const { return depth; }

      bool has_changes();
      void clear();

      html::behavior::insert_text *top_insert_text();
      remove_char_backward *       top_remove_char_backward();
      remove_char_forward *        top_remove_char_forward();

      bool remove_selection(view &v);
      // bool drop_html( view& v, bookmark where, chars html, bool move );
      // bool drop_text( view& v, bookmark where, wchars text, bool move );

      bool insert_char(view &v, bookmark where, wchar uc, bool final = true);

      bool close_input_sequence();
      bool remove_char_after(view &v, bookmark bm);
      bool remove_char_before(view &v, bookmark bm);
      bool remove_element(view &v, element *el, bool after = true);

      // bool set_hyperlink(span_id_t href_id); // to current selection
      // bool remove_hyperlinks(view& v, bookmark start, bookmark end ); // in
      // current selection  bool check_hyperlink();

      bool unwrap_element(view &v, element *el);

      virtual bool insert_html(view &v, bytes html, bookmark where = bookmark(),
                               const clipboard::html_item *phi = nullptr);
      virtual bool insert_text(view &v, wchars text,
                               bookmark where = bookmark());
      virtual bool insert_image(view &v, himage pimg,
                               bookmark where = bookmark());

      bool apply_span(view &v, tag::symbol_t t, const attribute_bag &atts = attribute_bag());
      bool remove_spans(view &v, slice<tag::symbol_t> tlist,const attribute_bag &atts = attribute_bag());

      bool shelve_apply_span(view &v, tag::symbol_t t);
      bool shelve_remove_spans(view &v, slice<tag::symbol_t> t);

      bool     apply_span(view &v, handle<action> group, bookmark &start, bookmark &end, tag::symbol_t t, const attribute_bag &atts = attribute_bag());
      bool     remove_spans(view &v, handle<action> group, bookmark &start, bookmark &end, slice<tag::symbol_t> tlist, const attribute_bag &atts = attribute_bag());

      bookmark delete_range(view &v, bookmark start, bookmark end, bool forward);
      bookmark delete_range_in(view &v, action *act, bookmark start, bookmark end, bool forward);

      pair<bookmark, bookmark> delete_cells(view &v, block_table_body *table, irange table_rows, irange table_cols, bool merge);
      pair<bookmark, bookmark> insert_cells(view &v, block_table_body *table, int at, bool column);
      pair<bookmark, bookmark> split_cells(view &v, block_table_body *table_body, irange table_rows, irange table_cols);

      bookmark delete_char(view &v, bookmark at, bool forward);

      virtual bool insert_chars(view &v, bookmark start, bookmark end, wchars text);

      bool insert_row(view &v, bookmark at);

      //bool insert_plaintext_chars(view &v, bookmark start, bookmark end, wchars text);

      virtual bool insert_break(view &v, bookmark start,bookmark end); // <li>some|text</li> -> <li>some</li><li>text</li>

      bool insert_soft_break(view &v, bookmark start, bookmark end); // <br>
      bool insert_block_break(
          view &v, bookmark start,
          bookmark end); // <li>some|text</li> -> <li>some<p>text</p></li>
      bool insert_element(view &v, bookmark start, bookmark end, helement el);

      bool apply_list(view &v, bookmark start, bookmark end, tag::symbol_t t,
                      const attribute_bag &atts = attribute_bag());
      bool remove_list(view &v, bookmark &start, bookmark &end, tag::symbol_t t,
                       const attribute_bag &atts = attribute_bag());

      bool apply_pre(view &v, const attribute_bag &atts = attribute_bag());
      bool remove_pre(view &v);
      bool can_pre(view& v);

      bool indent(view &v, bookmark start, bookmark end);
      bool can_indent(view &v, bookmark start, bookmark end);
      bool unindent(view &v, bookmark start, bookmark end);
      bool can_unindent(view &v, bookmark start, bookmark end);

      bool morph_blocks(view &v, bookmark start, bookmark end, tag::symbol_t t,
                        bool exec);
      bool apply_block(view &v, bookmark start, bookmark end, tag::symbol_t t);
      bool remove_block(view &v, bookmark start, bookmark end, tag::symbol_t t);

      virtual bool can_spell_check(view &v);
      virtual bool rq_spell_check(view &v);
      virtual bool spell_check(view &v);


      value api_load(value urlOrBytes, value url);
      bool api_loadEmpty();
      ustring api_get_url();
      bool    api_set_url(ustring url);
      value   api_contentToSource();
      value   api_sourceToContent(ustring source, ustring url,value selStart, value selEnd);
      value   api_contentToBytes();
      value   api_bytesToContent(value source, value url);
      value   api_save(ustring url);
      //value   api_mediaVars();
      //ustring api_get_baseUrl();
      //bool    api_set_baseUrl(ustring url);


      value  api_update(tool::value cb, ustring name);

      SOM_PASSPORT_BEGIN_EX(richtext, richtext_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(load, api_load),
          SOM_FUNC_EX(loadEmpty, api_loadEmpty),
          SOM_FUNC_EX(contentToSource, api_contentToSource),
          SOM_FUNC_EX(sourceToContent, api_sourceToContent),
          SOM_FUNC_EX(contentToBytes, api_contentToBytes),
          SOM_FUNC_EX(bytesToContent, api_bytesToContent),
          SOM_FUNC_EX(save, api_save),
          SOM_FUNC_EX(update, api_update),
        )
        SOM_PROPS(
          SOM_VIRTUAL_PROP(url, api_get_url, api_set_url),
          SOM_RO_VIRTUAL_PROP(isModified, get_modified),
          SOM_RO_VIRTUAL_PROP(body, root),
        )
      SOM_PASSPORT_END


    };

    struct transact_ctx : sciter::om::asset<transact_ctx> {
      handle<view>         pv;
      handle<element>      rt;
      handle<richtext_ctl> rt_ctl;
      handle<range_action> act;

      transact_ctx(element *el, wchars name);

      bool active() const { return act.ptr() != 0; }

      bool         remove_attr(element *on, string name);
      bool         set_attr(element *on, string name, ustring val);
      bool         set_tag(element *on, string tag);
      slice<hnode> insert_html(bookmark &at, bytes html);
      bool         insert_text(bookmark &at, wchars text);

      bookmark     delete_selection();
      bookmark     delete_range(bookmark from, bookmark to);
      bookmark     remove_selection();
      bool         delete_node(node* pn);
      bool         split(bookmark &at, element *until);
      bool         wrap(bookmark &start, bookmark &end, element *into);
      pair<bookmark,bookmark> unwrap(element *that);
      bool         set_text(node *into, wchars text);
      bool         insert_node(bookmark &at, node* pn);

      bool         merge(bytes html, morph_options* pmr = nullptr);

      void commit();
      void rollback();

      array<hnode> api_insert_html(node* hn, int offset, ustring html)
      {
        bookmark bm(hn, offset, false);
        return insert_html(bm, u8::cvt(html).chars_as_bytes());
      }
      tool::value  api_insert_text(node* hn, int offset, ustring text)
      {
        bookmark bm(hn, offset, false);
        insert_text(bm, text);
        return value::make_array({value::make_resource(bm.node), value(bm.linear_pos())});
      }

      tool::value api_delete_selection() {
        bookmark bm = delete_selection();
        return value::make_array({ value::make_resource(bm.node), value(bm.linear_pos()) });
      }

      tool::value api_delete_range(node* hn1, int offset1, node* hn2, int offset2) {
        bookmark bm1(hn1, offset1, false);
        bookmark bm2(hn2, offset2, false);
        bookmark bm = delete_range(bm1, bm2);
        return value::make_array({ value::make_resource(bm.node), value(bm.linear_pos()) });
      }

      tool::value api_remove_selection() {
        bookmark bm = remove_selection();
        return value::make_array({ value::make_resource(bm.node), value(bm.linear_pos()) });
      }

      tool::value api_split(node* hn, int offset, element *until) {
        bookmark bm(hn, offset, false);
        if(!split(bm, until))
          return value::make_error("split failure");
        return value::make_array({ value::make_resource(bm.node), value(bm.linear_pos()) });
      }

      tool::value api_wrap(node* hn1, int offset1, node* hn2, int offset2, element *into)
      {
        bookmark bm1(hn1, offset1, false);
        bookmark bm2(hn2, offset2, false);
        if (!wrap(bm1, bm2, into))
          return value::make_error("wrap failure");
        return value::make_array({ value::make_resource(bm1.node), value(bm1.linear_pos()),value::make_resource(bm2.node), value(bm2.linear_pos()) });
      }

      tool::value api_unwrap(element *that) {
        pair<bookmark, bookmark> p = unwrap(that);
        return value::make_array({ value::make_resource(p.first.node), value(p.first.linear_pos()),value::make_resource(p.second.node), value(p.second.linear_pos()) });
      }

      tool::value api_insert_node(node* hn, int offset, node* pn) {
        bookmark bm(hn, offset, false);
        insert_node(bm, pn);
        return value::make_array({ value::make_resource(bm.node), value(bm.linear_pos()) });
      }

      bool api_exec_command(ustring cmd, tool::value param) {
        return html::exec_command(*pv.ptr(), rt, rt, cmd,param);
      }

      bool api_set_text(node *into, ustring text) {
        return set_text(into, text);
      }

      SOM_PASSPORT_BEGIN_EX(Transaction, transact_ctx)
        SOM_FUNCS(
          SOM_FUNC_EX(removeAttribute, remove_attr),
          SOM_FUNC_EX(setAttribute, set_attr),
          SOM_FUNC_EX(setTag, set_tag),
          SOM_FUNC_EX(setText, api_set_text),
          SOM_FUNC_EX(insertHTML, api_insert_html),
          SOM_FUNC_EX(insertText, api_insert_text),
          SOM_FUNC_EX(insertNode, api_insert_node),
          SOM_FUNC_EX(deleteSelection, api_delete_selection),
          SOM_FUNC_EX(deleteRange, api_delete_range),
          SOM_FUNC_EX(deleteNode, delete_node),
          SOM_FUNC_EX(collapseSelection, api_remove_selection),
          SOM_FUNC_EX(split, api_split),
          SOM_FUNC_EX(wrap, api_wrap),
          SOM_FUNC_EX(unwrap, api_unwrap),
          SOM_FUNC_EX(execCommand, api_exec_command),
        )
//        SOM_PROPS(
//          SOM_VIRTUAL_PROP(url, api_get_url, api_set_url),
//          )
        SOM_PASSPORT_END


    };

    struct plaintext_ctl : public richtext_ctl {
      typedef richtext_ctl super;

      plaintext_ctl() : richtext_ctl() {}
      virtual CTL_TYPE      get_type() override { return CTL_PLAINTEXT; }
      virtual const string &behavior_name() const;

      virtual bool is_richtext() const { return false; }
      virtual bool is_plaintext() const { return true; }

      virtual bool attach(view &v, element *self);

      // int  spaces_per_tab();

      virtual bool set_value(view &v, element *self, const value &val) override;
      virtual bool get_value(view &v, element *self, value &val) override;

      virtual bool paste(view &v) override;
      virtual bool can_paste(view &v) override;
      virtual bool copy(view &v) override;

      virtual bool is_empty() const ;
      virtual bool is_empty(const element *self, bool &yes) override;

      // virtual bool insert_text(view& v, wchars t) override
      //{
      //  return insert_chars(v,anchor,caret,t);
      //}

      virtual bool draw_caret(view &v, graphics *pg,
                              const caret_metrics &cm) override;

      virtual bool insert_break(view &v, bookmark start, bookmark end) override;
      virtual bool insert_chars(view &v, bookmark start, bookmark end, wchars text) override;

      virtual bool can_indent(view &v) { return false; }
      virtual bool can_unindent(view &v) { return false; }
      virtual bool do_indent(view &v) { return false; }
      virtual bool do_unindent(view &v) { return false; }

      bool set_text(view &v, element *self, wchars text, int_v apos = int_v(),
                    int_v cpos = int_v());
      bool get_text(view &v, element *self, array<wchar> &chars);
      bool get_text(view &v, element *self, array<wchar> &chars, int_v &apos,
                    int_v &cpos);

      virtual bool on_data_arrived(view &v, element *self,
                                   pump::request *rq) override;
      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override;

      virtual element *root() override { return self; }
      virtual bool check_empty(view &v, bookmark& bm) override;

      virtual bool can_spell_check(view &v) override {
        return self->atts.get_bool(attr::a_spellcheck, false);
      }

      virtual bool advance_forward(view &v, bookmark &bm) {
        return html::advance(v, root(), bm, ADVANCE_RIGHT);
      }
      virtual bool advance_backward(view &v, bookmark &bm) {
        return html::advance(v, root(), bm, ADVANCE_LEFT);
      }

      virtual bool paste(view &v, clipboard::data *what, bookmark where) override {
        clipboard::text_item * ti = 0;

        if ((ti = what->get<html::clipboard::text_item>()) != nullptr) {
          return insert_text(v, ti->val(), where);
        }
        return false;
      }

      // script accessors:

      bool load(ustring url);
      bool save(ustring url);

      bool appendLine(tool::value stringOrArray);
      bool insertLine(int at, tool::value stringOrArray);
      bool removeLine(int at, int howmany);
      bool selectAll();
      bool selectRange(int line1, int pos1, int line2, int pos2);

      tool::ustring getContent();
      bool          setContent(tool::value stringOrArray);
      int     getLines();
      bool    getLine(int n, tool::ustring& text);
      bool    setLine(int n, tool::ustring text);
      bool    nextLine(int& n, tool::ustring& text);

      tool::array<int> getSelectionStart();
      tool::array<int> getSelectionEnd();
      tool::ustring    getSelectionText();

      SOM_PASSPORT_BEGIN_EX(plaintext, plaintext_ctl)
        SOM_FUNCS(
          SOM_FUNC(selectAll),
          SOM_FUNC(selectRange),
          SOM_FUNC(insertLine),
          SOM_FUNC(removeLine),
          SOM_FUNC(appendLine),
          SOM_FUNC_EX(update, api_update),
        )

        SOM_PROPS(
          SOM_VIRTUAL_PROP(content, getContent, setContent),
          SOM_RO_VIRTUAL_PROP(lines, getLines),
          SOM_RO_VIRTUAL_PROP(selectionStart, getSelectionStart),
          SOM_RO_VIRTUAL_PROP(selectionEnd, getSelectionEnd),
          SOM_RO_VIRTUAL_PROP(selectionText, getSelectionText),
          SOM_RO_VIRTUAL_PROP(isModified, get_modified),
        )
        SOM_ITEM_GET(getLine)
        SOM_ITEM_SET(setLine)
        SOM_ITEM_NEXT(nextLine)

     SOM_PASSPORT_END

    };

    struct textarea_ctl : ctl, selection_ctx {
      typedef ctl super;
      text_block *self;
      // int 			  _caret_pos;
      // int 			  _anchor_pos;
      // int 			  _scroll_pos;
      point _caret_pos; // for advance_up/down
      // int_v       _caret_drag_pos; // >= 0 - drag is in this ctl
      tristate_v _mouse_down_on_selection; // was mouse down on selection.
      tristate_v _double_click;            // was double click;
      //handle<text_block> _novalue_text_block;
      //ustring            _novalue_text;

      enum E_OP {
        INSERT_CHARS,
        REMOVE_CHARS,
      };

      struct e_state : resource {
        array<wchar>    text;
        bookmark        pos;
        bookmark        endpos;
        E_OP            op;
        handle<e_state> initial;
        handle<e_state> next;
      };
      handle<e_state> _to_undo;
      handle<e_state> _to_redo;

      // int               _ime_start;
      // int               _ime_length;

      bool _is_empty;
      bool _is_valid;
      int  _state_seq_no;

      handle<spell_checker> _spell_checker;

      textarea_ctl()
          : self(0), _is_empty(false), _state_seq_no(0), _is_valid(false) {}

      virtual CTL_TYPE get_type() { return CTL_TEXTAREA; }

      virtual const string &behavior_name() const;

      virtual bool focusable(const element *self) {
        return !self->attr_disabled();
      }
      virtual bool need_ime(element *self) {
        if (is_readonly(self)) return false;
        if (self->atts.exist("noime")) return false;
        return true;
      }

      virtual bool wants_keyboard(const element *b) const {
        return !is_readonly(b);
      }

      virtual bool attach(view &v, element *self); //  { this->self = self; }
      virtual void detach(view &v, element *self); //  { this->self = 0; }
      virtual html::text *  text_node() const;
      virtual wchars        chars() const;
      virtual array<wchar> &edit_buffer(view &v, element *self);
      virtual void          update(view &v, element *self);
      virtual void          check_empty(view &v, element *self);

      virtual bool          needs_show_caret(view &v, element *self) {
        if (!is_editable(self)) return false;
        if (self->state.owns_focus()) return true;
        // case when the editor is an editable <caption> inside <input|number>
        if (self->state.current() && self->parent && self->parent->state.owns_focus()) return true;
        return false;
      }

      virtual bool copy_allowed() const { return true; }

      virtual int maxlength() const {
        int n = self->atts.get_int(attr::a_maxlength, 32000);
        if (self->tag == tag::T_CAPTION && self->parent &&
            self->state.synthetic())
          return self->parent->atts.get_int(attr::a_maxlength, n);
        return n;
      }

      // selection_ctx stuff

      virtual element *selection_root() const override { return self; }
      virtual argb     selection_back_color(view &v);
      virtual argb     selection_fore_color(view &v);
      virtual argb     selection_text_color(view &v);
      virtual argb     selection_caret_color(view &v);
      virtual bool     draw_content(view &v, element *self, graphics *sf, point pos);
      virtual bool     allow_text_selection_on(text_block *tb);

      virtual bool get_auto_width(view &v, element *self, int &value);
      virtual bool get_auto_height(view &v, element *self, int &value);
      // bool set_caret_to(view& v, bookmark bm, bool keep_anchor);

      virtual bool set_value(view &pv, element *self, const value &v) override;
      virtual bool get_value(view &pv, element *self, value &v) override;
      //virtual wchars get_text_value(view &v, element *self);

      virtual bool set_text(view &v, element *self, wchars val) override;
      virtual bool get_text(view &v, element *self, ustring &val) override;

      virtual bool is_empty(const element *self, bool &yes) override {
        yes = _is_empty;
        return true;
      }

      virtual bool on(view &v, element *self, event_mouse &evt) override;
      virtual bool on(view &v, element *self, event_key &evt) override;
      virtual bool on(view &v, element *self, event_focus &evt) override;
      virtual bool on_timer(view &v, element *self, timer_id tid,
                            TIMER_KIND kind) override;
      virtual bool on(view &v, element *self, event_behavior &evt) override;
      virtual bool on(view &v, element *self, event_command &evt) override;
      virtual bool on(view &v, element *self, event_exchange &evt) override;

      // virtual bool on_method_call(view& v, element* self, method_params *p);
      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override;

      virtual bool draw_caret(view &v, graphics *pg,
                              const caret_metrics &cm) override;

      bool perform_drag(view &v, element *self, event_mouse &evt);

      virtual bool check_char(wchar c, bool signal);
      virtual bool check_chars(array<wchar> &txt);
      ustring      get_filter_attr();
      virtual void clear(view &v);
      virtual bool clear_comp_chars(view &v);

      // virtual int     find_char_pos(view& v, point pt);
      // virtual int     find_drop_pos(view& v, point pt); // cursor position
      // suitable for drop.

      virtual bool move_caret(view &v, ADVANCE_DIR dir, bool keep_anchor,
                              uint n = 1);
      bool         move_caret_to(view &v, bookmark bm, bool keep_anchor);
      bool         move_caret_to(view &v, point pt, bool keep_anchor);
      bool         move_target_to(view &v, bookmark bm); // D&D target

      void update_caret(view &v);
      bool selection_exists() { return caret != anchor; }
      int  selection_start() {
        return caret < anchor ? (caret.pos + caret.after_it)
                              : (anchor.pos + anchor.after_it);
      }
      int selection_end() {
        return caret > anchor ? (caret.pos + caret.after_it)
                              : (anchor.pos + anchor.after_it);
      }
      wchars selected_text();
      bool   is_on_selection(view &v, point pt);

      // virtual rect    get_caret_place(view& v);
      virtual bool get_caret_place(view &v, element *self, rect &rc) override;
      void         scroll_to_view(view &v);
      void         show_caret(view &v, bool onOff);
      void         refresh_caret(view &v);

      // virtual bool    handle_key_at(view& v, int& pos, uint k, bool isCtl,
      // bool isShift) { return false; }

      virtual bool     insert_chars_at(view &v, bookmark &cp, wchars s,
                                       bool save_state = true);
      virtual bookmark remove_chars_at(view &v, bookmark start, bookmark end,
                                       bool save_state = true);

      virtual bool     insert_char(view &v, wchar c);
      virtual bool     is_valid_char_at(wchar c, int pos) { return true; }
      bool             delete_char(view &v, int dir); // -1, 0(del sel only), +1
      virtual bookmark delete_range(view &v, bookmark start, bookmark end) {
        return remove_chars_at(v, start, end);
      }
      virtual bool insert_range(view &v, wchars s, bookmark at = bookmark(),
                                bool delete_selection = true);
      virtual bool insert_ime_range(view &v, wchars s, int_v caret_pos);
      // virtual bool    delete_ime_range(view& v);
      virtual bool is_multiline() { return true; }

      virtual bool cut(view &v);
      virtual bool can_cut(view &v) {
        return is_editable(self) && selection_exists();
      }
      virtual bool copy(view &v);
      virtual bool can_copy(view &v) { return selection_exists(); }
      virtual bool paste(view &v);
      virtual bool paste(view &v, const ustring &us);
      virtual bool can_paste(view &v) { return is_editable(self); }

      virtual bool select_all(view &v);
      virtual bool undo(view &v);
      virtual bool restore_state(view &v, e_state *ps);
      virtual bool can_undo(view &v);
      virtual bool redo(view &v);
      virtual bool can_redo(view &v);

      virtual bool notify_changed(view &v, uint reason);
      virtual bool notify_changing(view &v, uint reason);
      virtual bool notify_changing(view &v, uint reason, array<wchar> &text);

      // virtual bool    setup_context_menu(view& v, element* self,
      // event_behavior& evt);
      virtual bool select(view &v, int start, int end);
      virtual bool select(view &v, bookmark c, bookmark a) override;

      virtual bool can_spell_check(view &v);
      virtual bool rq_spell_check(view &v);
      virtual bool spell_check(view &v);

      // script access

      int     get_selectionStart();
      int     get_selectionEnd();
      ustring get_selectionText();

      bool    do_selectRange(int_v start, int_v end);
      bool    do_selectAll();

      bool    do_insertText(ustring text);
      bool    do_appendText(ustring text);
      bool    do_removeText();

      SOM_PASSPORT_BEGIN_EX(textarea, textarea_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(selectAll,   do_selectAll),
          SOM_FUNC_EX(selectRange, do_selectRange),
          SOM_FUNC_EX(removeText,  do_removeText),
          SOM_FUNC_EX(insertText,  do_insertText),
          SOM_FUNC_EX(appendText,  do_appendText))

        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(selectionStart, get_selectionStart),
          SOM_RO_VIRTUAL_PROP(selectionEnd, get_selectionEnd),
          SOM_RO_VIRTUAL_PROP(selectionText, get_selectionText)
        )
      SOM_PASSPORT_END


    };

    struct edit_ctl : public textarea_ctl {
      typedef textarea_ctl super;

      edit_ctl() {}

      virtual CTL_TYPE get_type() override { return CTL_EDIT; }

      virtual const string &behavior_name() const override;

      virtual bool attach(view &v, element *self) override;

      virtual bool get_auto_width(view &v, element *self, int &value) override;
      virtual bool get_auto_height(view &v, element *self, int &value) override;

      // virtual bool check_char(wchar c, bool signal ) override  { return true;
      // }  virtual bool check_chars(array<wchar>& txt ) override  { return true;
      // }
      virtual bool is_multiline() override { return false; }

      virtual array<wchar> &edit_buffer(view &v, element *self) override;

      virtual bool can_spell_check(view &v) {
        return self->atts.get_bool(attr::a_spellcheck, false);
      }

      bool is_nullable(element *self) { return self->atts.get_bool("nullable"); }

      virtual bool on(view &v, element *self, event_mouse &evt) override {
        if (evt.cmd == MOUSE_DOWN && evt.is_point_button() && evt.is_on_icon) {
          if (!exec_command(v, self, self, event_command::EDIT_SELECT_ALL())) return false;
          if (!exec_command(v, self, self, event_command::EDIT_DELETE_NEXT())) return false;
          return true;
        }
        return super::on(v, self, evt);
      }

      virtual bool on(view &v, element *self, event_focus &evt) override {
        if (evt.cmd == FOCUS_OUT)
          select(v, -1, -1);
        return super::on(v, self, evt);
      }


      SOM_PASSPORT_BEGIN_EX(edit, edit_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(selectAll, do_selectAll),
          SOM_FUNC_EX(selectRange, do_selectRange),
          SOM_FUNC_EX(removeText, do_removeText),
          SOM_FUNC_EX(insertText, do_insertText),
          SOM_FUNC_EX(appendText, do_appendText))

        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(selectionStart, get_selectionStart),
          SOM_RO_VIRTUAL_PROP(selectionEnd, get_selectionEnd),
          SOM_RO_VIRTUAL_PROP(selectionText, get_selectionText)
        )
      SOM_PASSPORT_END


    };

    struct password_ctl : public edit_ctl {
      typedef edit_ctl super;

      password_ctl() {}

      virtual CTL_TYPE get_type() override { return CTL_PASSWORD; }

      virtual const string &behavior_name() const override;

      virtual bool attach(view &v, element *self) override;

      //virtual bool need_ime(element *self) override { return false; }

      virtual array<wchar> &edit_buffer(view &v, element *self) override;
      virtual void          update(view &v, element *self) override;

      virtual bool cut(view &v) override {
        beep();
        return false;
      }
      virtual bool can_cut(view &v) override { return false; }
      virtual bool copy(view &v) override {
        beep();
        return false;
      }
      virtual bool can_copy(view &v) override { return false; }
      virtual bool copy_allowed() const { return false; }
      wchar        placeholder(element *self);

      virtual bool can_spell_check(view &v) { return false; }

      virtual bool move_caret(view &v, ADVANCE_DIR dir, bool keep_anchor,
                              uint n);
      virtual bool get_value(view &pv, element *self, value &v) override;

      //bool get_text(view & v, element * self, ustring & val) override;
      bool a11y_get_value(view & v, element * self, ustring & sval) override;

      array<wchar> _buffer;
    };

    struct select_ctl : ctl // select (list)
    {
      element* _self = nullptr;
      weak_handle<element> _current_option;
      weak_handle<element> _value_option;
      bool            _in_set_value;

      function<value(const ustring &)> _coerce;

      bool (*is_select_option)(view &v, element *b);

      enum SELECT_TYPE {
        SINGLE,
        MULTIPLE,
        CHECKMARKS,
      };

      enum SET_WHAT {
        SET_ONLY_CURRENT,
        SET_VALUE,
      };

      select_ctl();
      virtual bool focusable(const element *self) { return true; }

      virtual element* r13n_container(element* self) override {
        if (!self->parent) return self;
        if (self->tag == tag::T_POPUP && self->attr_type() == WCHARS("select")) return self->parent;
        if (self->parent->behavior && self->parent->behavior->get_type() == CTL_DD_SELECT) return self->parent;
        return self;
      }

      virtual SELECT_TYPE select_type(const element *self) { return SINGLE; }

      virtual CTL_TYPE      get_type() { return CTL_SELECT_SINGLE; }
      virtual const string &behavior_name() const;

      virtual bool attach(view &v, element *self);
      virtual void detach(view &v, element *self);

      bool is_select_on_popup(view &v, element *self) {
        if (!self->parent) return false;
        if (self->tag == tag::T_POPUP && self->attr_type() == WCHARS("select")) return true;
        if (self->parent->ctl_type(v) == CTL_DD_SELECT) return true;
        return false;
      }

      virtual bool a11y_get_focus(view &v, element *self, helement& pf) override;

      virtual bool get_auto_width(view &v, element *self, int &value);
      virtual bool get_auto_height(view &v, element *self, int &value);

      virtual bool set_value(view &pv, element *self, const value &v) override;
      virtual bool get_value(view &pv, element *self, value &v) override;

      virtual bool on(view &v, element *self, event_mouse &evt);
      virtual bool on(view &v, element *self, event_key &evt);
      virtual bool on(view &v, element *self, event_focus &evt);
      virtual bool on(view &v, element *self, event_behavior &evt);
      virtual bool on(view &v, element *self, event_command &evt);

      bool select_by_char(view &v, element *self, uint key_code);
      virtual bool on_method_call(view &v, element *self, method_params *p);

      // ---
      element *    get_first_option(view &v, element *self);
      element *    get_target_option(view &v, element *self, element *target);
      virtual bool on_option(view &v, element *self, event_mouse &evt,
                             element *opt);
      virtual bool select_option(view &v, element *self, element *opt,
                                 uint alts, SET_WHAT what = SET_VALUE);
      virtual bool select_next_option(view &v, element *self, int cmd,
                                      uint alts, uint reason);
      virtual uint selected_state_flags() { return S_CURRENT | S_CHECKED; }
      virtual bool drop_selection(view &v, element *self,
                                  bool including_anchor = true);
      virtual void notify_change(view &v, element *self, element *opt,
                                 uint reason);
      virtual bool notify_changing(view &v, element *self, element *opt,
                                   uint reason);
      virtual void notify_state_changing(view &v, element *self, element *opt,
                                        uint reason);
      virtual void set_option_state(view &v, element *self, element *opt,
                                    uint64 state, uint64 state_to_clear = 0);
      virtual void clear_option_state(view &v, element *self, element *opt,
                                      uint64 state);

      bool set_value_multiple(view &pv, element *self, const value &v);
      bool get_value_multiple(view &pv, element *self, value &v);

      void attach_multiple(view &v, element *self);
      bool select_option_multiple(view &v, element *self, element *opt,
                                  uint alts, SET_WHAT what = SET_VALUE);
      void fill_selection_multiple(view &v, element *self, element* opt1, element* opt2);

      virtual element *get_options_container(element *self) override { return self; }
      //virtual element* r13n_container(element* self) override { return self; /*reconcilliation is enabled*/ }
      virtual bool a11y_get_value(view &v, element *self, ustring& sval)
      {
        if (_value_option) {
          sval = _value_option->get_text(v);
          return true;
        }
        return false;
      }

      helement api_options() {
        return _self;
      }

      SOM_PASSPORT_BEGIN_EX(select, select_ctl)
        /*SOM_FUNCS(
          SOM_FUNC_EX(showPopup, api_show_popup),
          SOM_FUNC_EX(hidePopup, api_hide_popup)) */
        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(options, api_options),
          )
      SOM_PASSPORT_END


    };

    struct select_multiple_ctl : public select_ctl {
      virtual const string &behavior_name() const;

      virtual CTL_TYPE get_type() { return CTL_SELECT_MULTIPLE; }

      SELECT_TYPE select_type(const element *self) override { return MULTIPLE; }

      virtual bool attach(view &v, element *self) {
        attach_multiple(v, self);
        return true;
      }

      virtual bool select_option(view &v, element *self, element *opt,
                                 uint alts, SET_WHAT what = SET_VALUE) {
        return select_option_multiple(v, self, opt, alts, what);
      }

      virtual uint selected_state_flags() { return S_CURRENT | S_CHECKED; }

      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        return set_value_multiple(v, self, val);
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        return get_value_multiple(v, self, val);
      }
    };
    struct select_checkmarks_ctl : public select_ctl {
      virtual CTL_TYPE get_type() { return CTL_SELECT_MULTIPLE; }
      SELECT_TYPE      select_type(const element *self) { return CHECKMARKS; }
      virtual const string &behavior_name() const;
      virtual bool          attach(view &v, element *self);

      virtual uint selected_state_flags() { return S_CURRENT | S_CHECKED; }

      virtual bool select_option(view &v, element *self, element *opt,
                                 uint alts, SET_WHAT what = SET_VALUE) {
        return select_ctl::select_option(v, self, opt, alts, SET_ONLY_CURRENT);
      }

      virtual void notify_change(view &v, element *self, element *opt,
                                 uint reason) override;

      virtual bool on(view &v, element *self, event_key &evt);
      virtual bool on_option(view &v, element *self, event_mouse &evt,
                             element *opt);
      virtual bool set_value(view &v, element *self,
                             const value &val) override {
        return set_value_multiple(v, self, val);
      }
      virtual bool get_value(view &v, element *self, value &val) override {
        return get_value_multiple(v, self, val);
      }

    };

    struct dd_select_ctl : ctl // dropdown select
    {
      element*                         _self = nullptr;
      handle<element>                  _caption;
      handle<element>                  _button;
      weak_handle<element>             _popup;
      handle<element>                  _value_option;
      handle<element>                  _prepopup_value_option;
      handle<element>                  _confirmed_value_option;
      bool                             _is_nullable;
      uint                             _closed_at = 0;
      //int_v   _width;
      //int_v   _height;
      function<value(const ustring &)> _coerce;

      dd_select_ctl() : _is_nullable(false) {}
      virtual bool focusable(const element *self) {
        return !is_editable(self);
        // return true;
      }

      virtual CTL_TYPE      get_type() { return CTL_DD_SELECT; }
      virtual const string &behavior_name() const;

      virtual bool attach(view &v, element *self) override;
      virtual void detach(view &v, element *self) override;
      virtual bool is_multy() const { return false; }
      virtual bool is_nullable() const { return _is_nullable; }

      virtual void update_model(view &v, element *self) {}

      virtual bool set_value(view &v, element *self, const value &val) override;
      virtual bool get_value(view &v, element *self, value &val) override;

      virtual bool get_auto_width(view &v, element *self, int &value) override;
      virtual bool get_auto_height(view &v, element *self, int &value) override;

      virtual bool on(view &v, element *self, event_mouse &evt) override;
      virtual bool on(view &v, element *self, event_behavior &evt) override;
      virtual bool on(view &v, element *self, event_key &evt) override;
      virtual bool on(view &v, element *self, event_focus &evt) override;
      virtual bool on(view &v, element *self, event_command &evt) override;

      virtual bool on_x_method_call(view &v, element *self, const char *name,
        const value *argv, size_t argc,
        value &retval) override;

      virtual bool is_dropdown_button(view &v, element *self, element *target);

      virtual bool is_empty(const element *self, bool &yes);

      bool owns_popup_list(view &v, element *self);

      virtual bool is_editable(const element *self) const {
        return _caption && _caption->get_named_behavior("edit") != 0;
      }
      virtual bool set_caption(view &v, element *self);
      void         show_popup(view &v, element *self, bool set_focus = true);
      void         close_popup(view &v, element *self, bool set_focus = false);
      void notify_change(view &v, element *self, element *opt, uint reason);

      virtual element *get_options_container(element *self) override {
        return _popup;
      }

      virtual element* r13n_container(element* self) override {
        return _popup ? _popup.ptr() : self; /*reconcilliation is enabled*/
      }

      virtual element* logical_container(element* self) {
        return _popup ? _popup.ptr() : self;
      }

      virtual bool a11y_get_children(element *self, array<hnode>& children) 
      { 
        children.clear();
        if(self->state.owns_popup())
          children.push(_popup.ptr());
        return true; 
      }

      virtual bool a11y_is_atomic(view &v, element *self) { return true; } 

      virtual bool a11y_get_state(view &v, element *self, ui_state& st) { 
        if(!_popup) return false;
        if (self->state.owns_popup())
          st.expanded(true);
        else
          st.collapsed(true);
        return true; 
      }

      virtual bool a11y_get_value(view &v, element *self, ustring& sval)
      {
        if (_caption) {
          sval = _caption->get_text(v);
          //dbg_printf("a11y_get_value %S\n",sval.c_str());
          return true;
        }
        return false;
      }

      bool api_show_popup(tristate_v set_focus) {
        if (!_self) return false;
        view* pv = _self->pview();
        if (!pv) return false;
        show_popup(*pv, _self, set_focus.val(TRUE));
        return true;
      }

      bool api_hide_popup() {
        if (!_self) return false;
        view* pv = _self->pview();
        if (!pv) return false;
        close_popup(*pv, _self);
        return true;
      }

      helement api_options() {
        return _popup;
      }

      SOM_PASSPORT_BEGIN_EX(select, dd_select_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(showPopup, api_show_popup),
          SOM_FUNC_EX(hidePopup, api_hide_popup))
        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(options, api_options),
        )
      SOM_PASSPORT_END


    };

    struct dd_multi_select_ctl : dd_select_ctl // dropdown multi-select
    {
      handle<element> _total;

      typedef dd_select_ctl super;
      dd_multi_select_ctl() {}
      virtual const string &behavior_name() const;

      virtual bool is_multy() const { return true; }

      virtual void update_model(view &v, element *self);
      virtual bool get_auto_width(view &v, element *self, int &value) override;

      virtual bool set_value(view &v, element *self, const value &val) override;
      virtual bool get_value(view &v, element *self, value &val) override;

      virtual bool is_dropdown_button(view &v, element *self,
                                      element *target) override;

      virtual bool is_editable(const element *self) const override {
        return false;
      }
      virtual bool set_caption(view &v, element *self) override;
    };

    struct highlighted_ctl : ctl {
      handle<image> fore;
      weak_handle<element> el;

      highlighted_ctl() : ctl() {}

      virtual const string &behavior_name() const override;
      virtual CTL_TYPE      get_type() override { return CTL_UNKNOWN; }

      virtual bool attach(view &v, element *self) override;
      virtual void detach(view &v, element *self) override;

      virtual bool draw_foreground(view &v, element *self, graphics *sf,
                                   point pos) override;

      virtual bool is_atomic() const override { return false; }

      image *get_fore();
    };

    struct frame_ctl : ctl, media_vars_provider {
      typedef ctl super;

      media_variables       _media_vars; // media vars of the frame
      handle<pump::request> _awaiting;   // awaiting this request to be delivered
      handle<pump::request> _pdocrq;     // document's rq being processed 
      bool                  _debug_mode = false;
      element*              _self = nullptr;

      frame_ctl() : super(uint(-1)) {}

      virtual CTL_TYPE get_type() override;

      virtual bool             focusable(const element *self) override;
      virtual media_variables &media_vars() override { return _media_vars; }

      virtual const string &behavior_name() const override;

      virtual bool attach(view &v, element *self) override;

      bool load_src_document(view *pv, element *self);

      document *root(view &v, element *self);

      virtual void on_attr_change(view &v, element *self, const name_or_symbol &nm) override;

      virtual bool on_data_request(view &v, element *self,
                                   pump::request *rq) override;
      virtual bool on_data_arrived(view &v, element *self,
                                   pump::request *rq) override;
      virtual void on_size_changed(view &v, element *self) override;
      virtual bool on(view &v, element *self, event_behavior &evt) override;

      virtual bool on_method_call(view &v, element *self,
                                  method_params *params) override;
      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval) override;

      bool    api_load_file(ustring url);
      bool    api_load_html(value string_or_bytes, ustring url);
      bool    api_load_empty();
      value   api_save_file(ustring url);
      value   api_save_bytes();
      value   api_get_mediaVars();
      bool    api_set_mediaVars(value map);
      ustring api_get_url();
      bool    api_set_url(ustring u);
      value   api_get_document() {
        if (helement hel = _self->first_element())
          return value::make_resource(hel.ptr());
        return value::null_val();
      }

      bool api_get_debug_mode() { return _debug_mode; }
      bool api_set_debug_mode(bool on) { _debug_mode = on; return on; }

      SOM_PASSPORT_BEGIN_EX(frame, frame_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(loadHtml, api_load_html),
          SOM_FUNC_EX(loadFile, api_load_file),
          SOM_FUNC_EX(loadEmpty, api_load_empty),
          SOM_FUNC_EX(saveFile, api_save_file),
          SOM_FUNC_EX(saveBytes, api_save_bytes),
        )
        SOM_PROPS(
          SOM_VIRTUAL_PROP(mediaVars, api_get_mediaVars, api_set_mediaVars),
          SOM_VIRTUAL_PROP(url, api_get_url, api_set_url),
          SOM_VIRTUAL_PROP(debugMode, api_get_debug_mode, api_set_debug_mode),
          SOM_RO_VIRTUAL_PROP(document, api_get_document)
        )
      SOM_PASSPORT_END

      virtual string get_content_style_url(element *self) {
        string url = get_attr(self, "-content-style");
        if (url.is_undefined()) return url;
        return combine_url(self->doc()->uri().src, url);
      }

      bool load(view &v, element *self, const string &url, bytes html,
                const string &encoding = string());
      bool _load(view &v, element *self, const string &url, bytes html,
                 const string &encoding);

      bool unload(view &v, document *pd);
      

      element *get_history_root(view &v, element *self);

      void update_media_vars(
          view &v, element *self,
          bool reset_styles = true); // accepts comma separated list of vars
      void set_media_vars(
          view &v, element *self, value map, bool reset,
          bool reapply_styles); // accepts comma separated list of vars

      virtual bool set_value(view &v, element *self, const value &val) { 
          
          if (val.is_string()) {
            string                in = val.get(W(""));
            string                out = combine_url(self->doc()->uri().src, in);
            handle<pump::request> rq = new pump::request(out, DATA_HTML);
            rq->dst = self;
            v.request_data(rq);
          }
          else if (val.is_bytes()) {
            tool::bytes html = val.get_bytes();
            string      url = self->doc()->uri().src;
            load(v, self, url, html);
          }
          else {
                         
          }
          return true;
      }
      virtual bool get_value(view &v, element *self, value &val) 
      { 
        document * pcd = root(v,self);
        if (pcd) val = value(pcd->uri().src, value::UT_URL); else val.clear();
        return true;
      }

    };

#if defined(USE_VIDEO)

#define CUSTOM_VIDEO_CTL_NAME "custom-video"

    struct video_ctl : ctl, animation {
      typedef ctl super;

      view *    _view;
      element * _self;
      tool::url _video_url;
      rect  _box; // if CONTAIN - dst box, if COVER - src box
      size  _dim;
      bool  _attached;
      mutex _guard;

      weak_handle<element> coordinator;

      handle<gool::bitmap> frame; // current frame;

      video_ctl() : _attached(false) {}
      virtual ~video_ctl() {}

      virtual CTL_TYPE get_type() { return CTL_IMAGE; }

      // infinite animation
      virtual bool is_finite(view &, element *) { return false; }

      virtual element* r13n_container(element* self) override {
        return self; /*reconcilliation is enabled*/
      }

      bool is_coordinated() const { return !!coordinator; }

      virtual const string &behavior_name() const;

      virtual bool focusable(const element *self) override {
        return false; /*!self->attr_disabled();*/
      }

      virtual void detach(view &v, element *self) override {
        _attached = false;
        if (_view && _self) {
          close_movie();
          stop_animation(_view, _self);
        }
        _view = 0;
        _self = 0;
      }
      virtual bool attach(view &v, element *self) override {
        _view     = &v;
        _self     = self;
        _attached = true;
        subscription = HANDLE_BEHAVIOR_EVENT | HANDLE_DATA_ARRIVED;

        if (self->atts.exist(attr::a_src)) {
          _video_url.parse(self->atts.get_url(self->doc()->uri().src, attr::a_src));
          if (load_movie(_video_url))
            play();
        }
        return true;
      }

      // animation stuff
      virtual uint start(view &v, element *self, const style * /*nst*/,
                         const style * /*ost*/) {
        return step(v, self, 0);
      }
      virtual uint step(view &, element *, uint /*current_clock*/) = 0;

      virtual bool load_movie(const tool::url &url) = 0;
      virtual void close_movie()                    = 0;

      virtual bool is_movie_open() const  = 0;
      virtual bool is_movie_ended() const = 0;

      // Returns the length of the video, in seconds.
      virtual double get_movie_duration() const = 0;

      // Returns the video's natural size, in pixels.
      // If no video is loaded, the size returned will be 0 x 0.
      virtual size get_movie_normal_size() const = 0;

      virtual bool play()             = 0; // starts the video playing.
      virtual void stop()             = 0;
      virtual bool is_playing() const = 0;

      virtual void goto_start() = 0; // Moves the video's position back to the start.
      virtual bool set_position(double seconds) = 0; // Sets the video's position to a given time.
      virtual double get_position() const = 0;

      // Changes the video playback rate.
      // A value of 1 is normal speed, greater values play it proportionately
      // faster, smaller values play it slower.
      void set_speed(float new_speed);

      // Changes the video's playback volume.
      // new_volume - the volume in the range 0 (silent) to 1.0 (full)
      virtual bool   set_volume(double new_volume)   = 0;
      virtual double get_volume() const              = 0;
      virtual bool   set_balance(double new_balance) = 0;
      virtual double get_balance() const             = 0;

      // rect get_target_box(bool& need_clipping );
      virtual bool draw_content(view &v, element *self, graphics *sf,
                                point pos);
      virtual bool draw_foreground(view &v, element *self, graphics *sf,
                                   point pos);

      virtual image *get_fore_image(view &v, element *self);

      virtual image *provide_fore_image() override { return frame; }

      virtual bool on_x_method_call(view &v, element *self, const char *name,
                                    const value *argv, size_t argc,
                                    value &retval);

      void notify_video_initialized(bool onoff) {
        if (!_attached) return;
        event_behavior evt(_self, _self, VIDEO_INITIALIZED, onoff);
        if (_view && _self) _view->post_behavior_event(evt, false);
      }
      void notify_video_started(bool resumed) {
        if (!_attached) return;
        event_behavior evt(_self, _self, VIDEO_STARTED, resumed);
        if (_view && _self) _view->post_behavior_event(evt, false);
      }
      void notify_video_stopped(bool ended) {
        if (!_attached) return;
        event_behavior evt(_self, _self, VIDEO_STOPPED, ended);
        if (_view && _self) _view->post_behavior_event(evt, false);
      }
      void notify_video_frame_ready() { 
        event_behavior evt(_self,_self, UI_STATE_CHANGED, 0); 
        if (_view && _self) _view->post_behavior_event(evt,true);
      }

      virtual void on_start_streaming() {}
      virtual void on_stop_streaming() {}

      bool start_animation(view *pv, element *self) {
        if(!is_coordinated())
           pv->add_animation(self, this);
        return true;
      }
      bool stop_animation(view *pv, element *self) {
        if (!is_coordinated())
          pv->remove_animation(self, this);
        return true;
      }

      virtual bool on(view &v, element *self, event_behavior &evt) override {
        if (evt.cmd == (VIDEO_STARTED | EVENT_SINKING) 
         || evt.cmd == (VIDEO_STARTED | EVENT_SINKING | EVENT_HANDLED)) {
          event_behavior sevt(_self, _self, VIDEO_COORDINATE, 0);
          if (v.send_behavior_event(sevt) && sevt.source) {
            coordinator = evt.source;
          }
        }
        return super::on(v, self, evt);
      }

      bool notify_frame_ready(view& v, element* self) {
        if( !this->is_coordinated() )
          return false;
        event_behavior sevt(coordinator, self , VIDEO_FRAME, 0);
        sevt.is_bubbling = false;
        v.post_behavior_event(sevt);
        return true;

      }

      int get_movie_normal_width() {
        return this->get_movie_normal_size().x;
      }
      int get_movie_normal_height() {
        return this->get_movie_normal_size().y;
      }

      array<int> get_movie_rendering_box() {
        // bool need_clipping = false;
        if (view* pv = _self->pview()) {
          rect box = _self->foreground_image_box(*pv);
          int r[4] = { box.left(), box.top(), box.width(), box.height() };
          return items_of(r);
        }
        return array<int>();
      }

      bool load_movie_(string url) {
        tool::url u(combine_url(_self->doc()->uri().src, url));
        return load_movie(u);
      }

      SOM_PASSPORT_BEGIN_EX(video, video_ctl)
        SOM_FUNCS(
          SOM_FUNC_EX(load, load_movie_),
          SOM_FUNC_EX(unload, close_movie),
          SOM_FUNC_EX(play, play),
          SOM_FUNC_EX(stop, stop),
          SOM_FUNC_EX(rewind, goto_start),
        )
        SOM_PROPS(
          SOM_RO_VIRTUAL_PROP(isPlaying, is_playing),
          SOM_RO_VIRTUAL_PROP(isEnded, is_movie_ended),
          SOM_RO_VIRTUAL_PROP(duration, get_movie_duration),
          SOM_VIRTUAL_PROP(position, get_position, set_position),
          SOM_RO_VIRTUAL_PROP(height, get_movie_normal_height),
          SOM_RO_VIRTUAL_PROP(width, get_movie_normal_width),
          SOM_RO_VIRTUAL_PROP(renderingBox, get_movie_rendering_box),
          SOM_VIRTUAL_PROP(audioVolume, get_volume, set_volume),
          SOM_VIRTUAL_PROP(audioBalance, get_balance, set_balance),
        )
      SOM_PASSPORT_END



    };

#endif

    struct window_frame_ctl : ctl {
      typedef ctl super;

      bool   dragging = false;
      size   dragging_offset;
      uint_v ht_code;
      point  resize_off;

      window_frame_ctl() {}

      virtual CTL_TYPE get_type() { return CTL_UNKNOWN; }

      virtual const string &behavior_name() const {
        static string s = CHARS("window-frame");
        return s;
      }

      // virtual bool focusable(const element* self) { return false; }

      virtual void detach(view &v, element *self) {
        return super::detach(v, self);
      }
      virtual bool attach(view &v, element *self) {
        return super::attach(v, self);
      }

      virtual uint hittest(view &v, element *self, point pos_view);

      virtual size border_band(view &v, element *self);

      virtual bool on(view &v, element *self, event_mouse &evt) override;
      virtual bool on(view &v, element *self, event_behavior &evt) override;

      bool drag_loop(view &v, event_mouse &evt);
      bool size_loop(view &v, event_mouse &evt);

      virtual element* r13n_container(element* self) override { return self; }

    };

    extern void init_buttons();
    extern void init_selects();
    extern void init_menu();
    extern void init_edits();
    extern void init_htmlarea();
    extern void init_frame();
    extern void init_bars();
    extern void init_hyperlink();
    extern void init_lists();
    extern void init_datetime();
    extern void init_table_ctl();
    extern void init_image();
    extern void init_masked_edit();
    extern void init_numeric_edit();
    extern void init_gestures();
    extern void init_scrollbar();
    extern void init_table_ctl();
    extern void init_effects();
    extern void init_richtext();
    extern void init_plaintext();
    extern void init_output();
    extern void init_media();
    extern void init_virtual_list();
#ifdef USE_LOTTIE
    extern void init_lottie();
#endif

    //#if defined(PAGER_SUPPORT)
    extern void init_pager();
    //#endif

#if defined(SCITER) || defined(SCITERJS)
    extern void init_reactor();
#endif


  } // namespace behavior
} // namespace html

#endif
