#ifndef __html_animator_h__
#define __html_animator_h__

#include "tool/tool.h"
#include "html-style.h"
#include "html-primitives.h"

namespace html {
  class view;
  struct element;
  struct style;
  struct iwindow;
      
  struct animation : public virtual  resource {
    animation() : start_clock(0), next_clock(0) {}

    virtual const char *name() { return "unknown"; }

    virtual bool is_finite(view &, element *) { return true; }

    virtual uint event_type() { return ANIMATION; }

    virtual bool changes_styles() const { return false; }

    virtual uint start(view &, element *, const style * /*nst*/,
                       const style * /*ost*/) {
      return 0;
    }
    virtual uint step(view &, element *, uint /*current_clock*/) { return 0; }
    virtual void stop(view &v, element *el) { for (auto& cb : on_end) cb(v, el, this); }
    virtual bool update_style(view &v, element *el, style &st) {
      if (next) return next->update_style(v, el, st);
      return false;
    } // update the style according to animation state

    // virtual bool will_draw() const { return false; }
    virtual bool draw(view & /*v*/, graphics * /*pg*/, element * /*b*/,
                      point /*pos*/) {
      return false;
    }
    virtual bool draw_content(view & /*v*/, graphics * /*pg*/, element * /*b*/,
                              point /*pos*/) {
      return false;
    }

    static void finalize(view &v, element *b);
    static void finalize_finite(view &v, element *b);

    virtual gool::image *provide_fore_image() { return nullptr; }
    virtual gool::image *provide_back_image() { return nullptr; }

    uint              start_clock;
    uint              next_clock;

    typedef function<void(view&, element*, animation*)> callback_t;

    array<callback_t> on_end;
    
    handle<animation> next;
  };
  typedef handle<animation> hanimation;

  struct image_animator : public animation {

    DEFINE_TYPE_ID(image_animator)

    virtual bool        is_finite(view &, element *) { return false; }
    virtual const char *name() { return "image"; }

    virtual bool draw(view &v, graphics *pg, element *b, point pos);

    virtual uint           start(view &v, element *el, const style *nst, const style *ost) = 0;
    virtual void           stop(view &v, element *el);
    handle<animated_image> himg;
  };

  struct fore_image_animator : public image_animator {
    DEFINE_TYPE_ID_DERIVED(fore_image_animator, image_animator)
    virtual uint start(view &v, element *el, const style *nst,
                       const style *ost);
    virtual uint step(view &v, element *el, uint current_clock);
  };
  struct back_image_animator : public image_animator {
    DEFINE_TYPE_ID_DERIVED(back_image_animator, image_animator)
    virtual uint start(view &v, element *el, const style *nst,
                       const style *ost);
    virtual uint step(view &v, element *el, uint current_clock);
  };


  struct css_std_property_animator : public animation 
  {
    DEFINE_TYPE_ID_DERIVED(css_std_property_animator,animation)

    handle<transition_def> transitions;
    uint current_clock = 0;

    bool finished = false;

    uint sct_max = 0; //STYLE_CHANGE_TYPE(0);

    virtual uint event_type() override { 
#ifdef SCITERJS
      return TRANSITION;
#else
      return ANIMATION;
#endif // SCITERJS
     
    }

#ifdef _DEBUG
    //bool dbg_trigger = false;
#endif 
    struct property_ctx {
      uint   property_sym;
      uint   duration, delay;
      uint_v s_duration, s_delay;
      int    current_clock = 0; // when backward goes backward
      bool   finished;

      ease::function     easef;
      value              start, current, end;
      handle<transforms> start_transforms;
      handle<transforms> end_transforms;
      handle<transforms> current_transforms;
      property_ctx() : property_sym(0), duration(0), delay(0), finished(false) {}
      // changes current value using easef
      bool morph(view &v, element *b, uint delta);

      bool end_defined() const { return end.is_defined() || end_transforms.is_defined(); }
      bool start_defined() const { return start.is_defined() || start_transforms.is_defined(); }

    };
    enum _ {
      STEP_DELAY = ANIMATION_TIMER_SPAN, // ms
    };

    hash_table<uint,property_ctx> props;
    handle<morphing_image> back;
    handle<morphing_image> fore;

    handle<style>          initial_style;
    handle<style>          final_style;

    css_std_property_animator(const style* initial, const style* final): initial_style(initial), final_style(final) {}
    virtual ~css_std_property_animator() {}

    virtual bool changes_styles() const { return true; }

    //bool is_same(const style* initial, const style* final) {
    //  return *initial == *initial_style && *final == *final_style;
    //}

    virtual const char *name() { return "style"; }

    virtual bool always_visible() { return false; }
    
    virtual uint start(view &v, element *b, const style *new_style, const style *old_style);

    virtual uint step(view &v, element *b, uint current_clock);
    virtual void stop(view &v, element *b);

            void init_property_ctx(view &v, element *b, const transition_item& ti, property_ctx& ctx, const style *new_style, const style *old_style);

    virtual uint update_targets(view &v, element *b, const style *new_style, const style *old_style);

    virtual bool update_style(view &v, element *b, style &s);

    virtual gool::image *provide_fore_image() override {
      if (fore) return fore;
      return next ? next->provide_fore_image() : nullptr;
    }
    virtual gool::image *provide_back_image() override {
      if (back) return back;
      return next ? next->provide_back_image() : nullptr;
    }


  };

  struct css_std_animate_animator : public animation {
    DEFINE_TYPE_ID_DERIVED(css_std_animation_animator,animation)

    enum {
      STEP_DELAY = ANIMATION_TIMER_SPAN, // ms
    };

    typedef css_std_property_animator::property_ctx property_ctx;

    uint duration = 0; 
    uint delay = 0;
    uint current_clock = 0;
    uint current_frame_start_clock = 0;
    uint current_frame_end_clock = 0;
    uint delay_end_clock = 0;

    uint   current_frame_no = 0;
    uint   frames_count = 0; // iterations_count * frames->size()
    uint   sct_max = 0; //STYLE_CHANGE_TYPE(0);

    handle<keyframes> frames;
    
    hash_table<uint, property_ctx> props;

    css_std_animate_animator() {}
    virtual ~css_std_animate_animator() {
      sct_max = sct_max;
    }

    virtual bool changes_styles() const { return true; }

    virtual const char *name() { return "animate"; }

    virtual bool is_finite(view &, element *) { return false; }

    virtual uint start(view &v, element *b, const style *nst, const style *ost);
    virtual bool next_frame(view &v, element *b);
    virtual uint step(view &v, element *b, uint current_clock);
    virtual void stop(view &v, element *b);

            void apply_props_to_style(view& v, element* b, slice<keyframes::prop_def> props, style& s);

            int  nframes() const { 
              // number of frames = number of edges - 1
              return frames ? frames->edges.size() - 1 : 0;
            }

    virtual bool update_style(view &v, element *b, style &s);

    void init_property_ctx(view &v, element *b, uint p_sym, uint  duration, bool forward, ease::function ease, property_ctx& ctx, const style *new_style, const style *old_style);
  };

  struct script_animator : public animation {
    DEFINE_TYPE_ID_DERIVED(script_animator, animation)

#if defined(SCITER)
    tis::pvalue       cbf;
    tis::pvalue       fin_cbf;
#else
    ease::function    easef = ease::linear;
    function<bool(view&, element*, double)> cbf;
#endif
    //uint              final_clock;
    uint_v duration;
    uint_v frame_duration;

    script_animator() {}
    virtual ~script_animator() {}

    virtual const char *name() { return "script"; }

    double progress(uint current_clock) const {
      if (duration.is_defined())
        return double(current_clock - start_clock) /
               double(duration.val(0));
      return 0;
    }

    virtual bool is_finite(view &, element *) { return duration.is_defined(); }

    virtual uint start(view &v, element *b, const style *nst, const style *ost) override;
    virtual uint step(view &v, element *b, uint current_clock) override;
    virtual void stop(view &v, element *b) override;
  };


  struct scroll_animator : public animation {
    DEFINE_TYPE_ID_DERIVED(scroll_animator,animation)
    // enum _ { MAX_TILES=10 };
    // sizef           step_ms; // step per millisecond - that is scroll speed;
    int   step_no;
    point final_scroll_pos;
    point scroll_pos; // current scroll position
    size  delta;
    sizef speed;   // pixels per millisecond
    float inertia; // inertia factor, larger content - larger intertia.
    uint  prev_clock;
#ifdef USE_TOUCH
    bool  is_touch = false;
#endif
    weak_handle<element> self;

    // size            client_dim;
    // size            content_dim;
    // size            tile_dim;
    // array<hbitmap>  tiles;        // tiles that contain rendered content
    // point           tiles_offset; // offset of first tile in content space

    enum scroll_mode {
      WHEEL, // to target
      INERTIAL,
    };
    scroll_mode mode;

    scroll_animator(view& v, point fpos, scroll_mode smode = WHEEL);
    virtual ~scroll_animator() {}

    // virtual int type() { return SCROLL_ANIMATOR; }
    virtual const char *name() override { return "scroll"; }

    virtual void set_final_pos(view &v, point fp);

    virtual uint start(view &v, element *b, const style *new_style,
                       const style *old_style) override;
    virtual uint step(view &v, element *b, uint current_clock) override;
    uint         step_wheel(view &v, element *b, uint current_clock);
    uint         step_inertial(view &v, element *b, uint current_clock);
    virtual void stop(view &v, element *b) override;
    // virtual bool draw(view& v, graphics* pg, element* b, point pos )
    // override;
    //        bool render_buffer(view& v, element* b, point buffer_pos,bitmap*
    //        tile); void init_buffer(view& v, element* b);

    virtual uint check_stop(view& v, element* b, uint current_clock);

  };

  struct effect_animator : public animation {
    enum STATE {
      FORWARD,
      ROLLBACK_FORWARD,
      BACKWARD,
      ROLLBACK_BACKWARD,
    };

    DEFINE_TYPE_ID_DERIVED(effect_animator,animation)
    virtual const char *name() override { return "blend"; }
    effect_animator()
        : content_only(), state(), easef(), duration(), delay(), initial(),
          final(), s_initial(), s_final(), area_off(), area_dim(),
          start_clock(), clock(), end_clock() {}

    // virtual bool will_draw() const { return initial && final; }
    virtual uint start(view &v, element *el, const style *nst,
                       const style *ost) override;
    virtual uint step(view &v, element *el, uint current_clock) override;
    //virtual void stop(view &v, element *el);
    virtual bool reverse(view &v, element *b, const style *new_style,
                         const style *old_style);
    virtual bool draw(view &v, graphics *pg, element *b, point pos) override {
      return _draw(v, pg, b, pos, false);
    }
    virtual bool draw_content(view &v, graphics *pg, element *b,
                              point pos) override {
      return _draw(v, pg, b, pos, true);
    }
    bool _draw(view &v, graphics *pg, element *b, point pos,
               bool content_layer);

    virtual bool prepare_initial(view &v, element *el);
    virtual bool prepare_final(view &v, element *el);

    void init_full(view &v, element *el, const style *nst, const style *ost,
                   iwindow *pw);
    void init_content(view &v, element *el, const style *nst, const style *ost,
                      iwindow *pw);
    bool draw_blend(view &v, graphics *pg, element *b, point pos,
                    float progress);
    bool draw_blend_atop(view &v, graphics *pg, element *b, point pos,
                         float progress);

    bool draw_slide_top(view &v, graphics *pg, element *b, point pos,
                        float progress);
    bool draw_slide_bottom(view &v, graphics *pg, element *b, point pos,
                           float progress);
    bool draw_slide_left(view &v, graphics *pg, element *b, point pos,
                         float progress);
    bool draw_slide_right(view &v, graphics *pg, element *b, point pos,
                          float progress);

    bool draw_slide_over_top(view &v, graphics *pg, element *b, point pos,
                             float progress);
    bool draw_slide_over_bottom(view &v, graphics *pg, element *b, point pos,
                                float progress);
    bool draw_slide_over_left(view &v, graphics *pg, element *b, point pos,
                              float progress);
    bool draw_slide_over_right(view &v, graphics *pg, element *b, point pos,
                               float progress);

    bool draw_remove_top(view &v, graphics *pg, element *b, point pos,
      float progress);
    bool draw_remove_bottom(view &v, graphics *pg, element *b, point pos,
      float progress);
    bool draw_remove_left(view &v, graphics *pg, element *b, point pos,
      float progress);
    bool draw_remove_right(view &v, graphics *pg, element *b, point pos,
      float progress);

    bool draw_scroll_top(view &v, graphics *pg, element *b, point pos,
                         float progress);
    bool draw_scroll_bottom(view &v, graphics *pg, element *b, point pos,
                            float progress);
    bool draw_scroll_left(view &v, graphics *pg, element *b, point pos,
                          float progress);
    bool draw_scroll_right(view &v, graphics *pg, element *b, point pos,
                           float progress);

    virtual bool is_worker() const { return false; }

    handle<animated_effect> effect_def;
    bool                    content_only; // content area only animation
    int_v                   state;
    ease::function          easef;
    uint                    duration;
    uint                    delay;
    // bool            forward;  // true if from no blend to blend
    handle<bitmap> initial;
    handle<bitmap> final;
    hstyle         s_initial;
    hstyle         s_final;
    point          area_off; // offset from its view position
    size           area_dim;
    uint           start_clock;
    uint           clock;
    uint           end_clock;

#ifdef _DEBUG
    ustring id;
#endif
  };

  typedef function<bool(view &v, element *pel)> state_changer_t;

  struct effect_animator_worker : public effect_animator {
    state_changer_t state_changer;

    effect_animator_worker(const state_changer_t &sc, animated_effect* paeff, effect_animator::STATE st) : state_changer(sc) { 
      effect_def = paeff; 
      state = st;
    }

    virtual bool prepare_initial(view& v, element * /*el*/) override;
    virtual bool prepare_final(view &v, element *el) override;

    virtual bool is_worker() const override { return true; }
    // it is irreversible 
    virtual bool reverse(view &v, element *b, const style *new_style, const style *old_style) { return true; }

  };

} // namespace html

#endif
