#include "html.h"


namespace html {

  bool need_animation(const transition_def *transitions,
                      const style *initial_style, const style *final_style) {
    for (int n = 0; n < transitions->props.size(); ++n) {
      const transition_item& p_def = transitions->props.value(n);
      // const transition_item& p_def = transitions->props.value(n);
      switch (p_def.prop_sym) {
      case cssa_transform: {
        const transforms *start_transforms = initial_style->transforms;
        const transforms *end_transforms   = final_style->transforms;
        if (start_transforms && !end_transforms)
          return true;
        else if (end_transforms && !start_transforms)
          return true;
        else if (end_transforms && start_transforms) {
          if (!end_transforms->is_compatible(start_transforms))
            continue;
          else if (end_transforms->is_identical(start_transforms))
            continue;
          return true;
        } else
          continue; // no transforms defined at start and end

      } break;
      case cssa_background_gradient: {
        if (initial_style->back_gradient && final_style->back_gradient &&
            (*(initial_style->back_gradient.ptr()) !=
             *(final_style->back_gradient.ptr())) &&
            initial_style->back_gradient->is_compatible(
                final_style->back_gradient)) {
          return true;
        } else
          continue;
      } break;
      case cssa_foreground_gradient: {
        if (initial_style->fore_gradient && final_style->fore_gradient &&
            (*(initial_style->fore_gradient.ptr()) !=
             *(final_style->fore_gradient.ptr())) &&
            initial_style->fore_gradient->is_compatible(
                final_style->fore_gradient)) {
          return true;
        } else
          continue;
      } break;

      case cssa_visibility:
      case cssa_display:
        continue; // changes on these alone shall not trigger animations

      default: {
        value start = initial_style->to_value(p_def.prop_sym);
        value end   = final_style->to_value(p_def.prop_sym);
        if (start == end) continue;
        return true;
      } break;
      }
    }
    return false;
  }


  void to_ppx(view &v, value &val, element *b, int attr_sym) {
    if (val.units() == size_v::unit_type::ppx) return;

    size_v sz = val;
    int    px = 0;
    switch (attr_sym) {
    case cssa_height:
      if (b->dim().x && (sz.is_auto() || sz.is_max_intrinsic() || sz.is_min_intrinsic()))
        b->set_width(v, b->dim().x);
      px = pixels(v, b,sz).height();
      break;
    case cssa_foreground_position_top:
    case cssa_foreground_position_bottom:
    case cssa_foreground_height:
      px = pixels(v, b, sz, b->foreground_clip_box(v).size()).height();
      break;
    case cssa_foreground_position_left:
    case cssa_foreground_position_right:
    case cssa_foreground_width:
      px = pixels(v, b, sz, b->foreground_clip_box(v).size()).width();
      break;
    case cssa_background_position_top:
    case cssa_background_position_bottom:
    case cssa_background_height:
      px = pixels(v, b, sz, b->background_clip_box(v).size()).height();
      break;
    case cssa_background_position_left:
    case cssa_background_position_right:
    case cssa_background_width:
      px = pixels(v, b, sz, b->background_clip_box(v).size()).width();
      break;

    case cssa_border_top_right_radius_x:
    case cssa_border_bottom_right_radius_x:
    case cssa_border_bottom_left_radius_x:
    case cssa_border_top_left_radius_x:
      px = pixels(v, b, sz, b->border_box(v).size()).width();
      break;

    case cssa_border_top_right_radius_y:
    case cssa_border_bottom_right_radius_y:
    case cssa_border_bottom_left_radius_y:
    case cssa_border_top_left_radius_y:
      px = pixels(v, b, sz, b->border_box(v).size()).height();
      break;

    default:
      if (!b->is_x_minmax_valid() && (sz.is_auto() || sz.is_max_intrinsic() || sz.is_min_intrinsic())) {
        b->check_layout(v);
        b->calc_intrinsic_widths(v);
      }
      px = pixels(v, b, sz).width();
      break;
    }
    val = value::make_ppx_length(px);
  }

  bool is_linear_gradient(const value &v) {
    return v.is_resource() && v.get_resource()->is_of_type<linear_gradient>();
  }
  bool is_radial_gradient(const value &v) {
    return v.is_resource() && v.get_resource()->is_of_type<radial_gradient>();
  }

  bool is_shadow_def(const value &v) {
    return v.is_resource() && v.get_resource()->is_of_type<shadow_def>();
  }

  bool is_image_ref(const value &v) {
    return v.is_resource() && v.get_resource()->is_of_type<image>();
  }

  bool is_morphing_image_ref(const value &v) {
    return v.is_resource() && v.get_resource()->is_of_type<morphing_image>();
  }

  color_v morph_color(view& v, element* b, color_v c1, color_v c2, real n)
  {
#ifdef DEBUG
    if (c1 == c2)
      n = n;
#endif
    //if (c1 == c2)
    //  return c1;
    return argb::morph(c1.to_argb(), c2.to_argb(), n);
  }
 

  float morph_float(float start, float end, real n) {
    return start + float(n) * (end - start);
  }

  // css std transition

  bool css_std_property_animator::property_ctx::morph(view &v, element *b, uint clock) {

    float n = 0;

    current_clock = clock;

#ifdef _DEBUG
    //string sn = cssa::symbol_name(this->property_sym);
    //dbg_printf("css_std_property_animator::property_ctx::morph progress %s\n", sn.c_str());
#endif

    if (current_clock >= int(delay + duration)) {
      n = 1.0f;
      current_clock = int(delay + duration);
      finished = true;
#ifdef _DEBUG
      //string sn = cssa::symbol_name(this->property_sym);
      //dbg_printf("css_std_property_animator::property_ctx::morph finished 1 %s\n",sn.c_str());
#endif
    }
    else if (current_clock > int(delay)) {
      float progress = float(current_clock - delay) / float(duration);
      n = easef(progress, 0.0, 1.0, 1);
    }

    if (property_sym == cssa_foreground_frame_no ||
      property_sym == cssa_background_frame_no) {
      int r_start = start.get_int();
      int r_end = end.get_int();
      int r_next = int(n * (r_end - r_start)) + r_start;
      current = value(r_next);
    }
    else if (property_sym == cssa_visibility) {
      int r_next = min(start.get_int(), end.get_int()); // as visibility_visible == 0
      current = value(r_next);
    }
    else if (property_sym == cssa_display) {
      int r_next = max(start.get_int(), end.get_int()); // as display_none == 0
      current = value(r_next);
    }
    else if (property_sym == cssa_position) {
      int r_next = max(start.get_int(), end.get_int()); // as display_none == 0
      current = value(r_next);
    }
    else if (start_transforms && end_transforms) {
      current_transforms->morph(v, b, start_transforms, end_transforms, n);
      finished = current_transforms->is_identical(end_transforms);
    }
    else if (start.is_length() || end.is_length()) {
      uint start_u = start.units();
      uint end_u = end.units();
      if (start_u != end_u) {
        if (start_u == size_v::unit_type::sp) {
          end = value::make_spring_length(0);
          end_u = size_v::unit_type::sp;
        }
        else if (end_u == size_v::unit_type::sp) {
          start_u = size_v::unit_type::sp;
          start = value::make_spring_length(0);
        }
        /*else if (start_u == size_v::unit_type::pr) {
          end_u = size_v::unit_type::pr;
          end = value::make_percent_length(0);
        }
        else if (end_u == size_v::unit_type::pr) {
          start_u = size_v::unit_type::pr;
          start = value::make_percent_length(0);
        }*/
        else {
          to_ppx(v, start, b, property_sym);
          to_ppx(v, end, b, property_sym);
          start_u = end_u = size_v::unit_type::ppx;
        }
      }
      
      if (start.is_length_literal() || end.is_length_literal()) {
        value p_current = current;
        current = (n >= 0.99999f) ? end : start;
        return current != p_current;
      }
       

      real  r_start = start.get_double();
      real  r_end = end.get_double();
      real  r_next = n * (r_end - r_start) + r_start;
      value p_current = current;
      current = value::make_length(r_next, start_u);
      return current != p_current;
      // morph length value
    }
    else if (start.is_color() || end.is_color()) {
      current = morph_color(v, b, color_v(start), color_v(end), n).to_value();
      //dbg_printf("clr %x at %f | %x %x %d \n", current.get(0), n, start.get(0),end.get(0), backward);
      return false; // no need to remeasure
    }
    else if (is_linear_gradient(start) && is_linear_gradient(end) &&
      is_linear_gradient(current)) {
      linear_gradient *c = static_cast<linear_gradient *>(current.get_resource());
      linear_gradient *s = static_cast<linear_gradient *>(start.get_resource());
      linear_gradient *e = static_cast<linear_gradient *>(end.get_resource());
      c->morph(v, b, s, e, n);
      return false; // no need to remeasure
    }
    else if (is_radial_gradient(start) && is_radial_gradient(end) &&
      is_radial_gradient(current)) {
      radial_gradient *c =
        static_cast<radial_gradient *>(current.get_resource());
      radial_gradient *s = static_cast<radial_gradient *>(start.get_resource());
      radial_gradient *e = static_cast<radial_gradient *>(end.get_resource());
      c->morph(v, b, s, e, n);
      return false; // no need to remeasure
    }
    else if (is_shadow_def(start) && is_shadow_def(end) && is_shadow_def(current)) {
      shadow_def *c = static_cast<shadow_def *>(current.get_resource());
      shadow_def *s = static_cast<shadow_def *>(start.get_resource());
      shadow_def *e = static_cast<shadow_def *>(end.get_resource());
      c->morph(v, b, s, e, n);
    }
    else if (is_image_ref(start) && is_image_ref(end) &&
      is_morphing_image_ref(current)) {
      gool::morphing_image *c = current.get_resource<gool::morphing_image>();
      gool::image *         s = start.get_resource<gool::image>();
      gool::image *         e = end.get_resource<gool::image>();
      c->morph(s, e, n);
      // current = value(n);
    }

    else if (start.is_int() || end.is_int()) {
      if (start.is_undefined())
        current = end;
      else if (end.is_undefined())
        current = start;
      else if (n > 0)
        current = end;
      else
        current = start;
      return false;
    }
    else if (start.is_enum() || end.is_enum()) {
      if (start.is_undefined())
        current = end;
      else if (end.is_undefined())
        current = start;
      else if (n > 0)
        current = end;
      else
        current = start;
      return false;
    }
    else if (start.is_string() || end.is_string()) {
      value pv = current;
      current = n ? end : start;
      return pv != current;
    }
    else {
      real r_start = start.get_double();
      real r_end = end.get_double();
      real r_next = n * (r_end - r_start) + r_start;
      current = value(r_next);
      return false;
    }

    return false;
    // return value();
  }

  void css_std_property_animator::init_property_ctx(view &v, element *b, const transition_item& p_def, property_ctx& ctx, const style *new_style, const style *old_style)
  {
    ctx.property_sym = p_def.prop_sym;
    ctx.duration = p_def.duration;
    ctx.delay = p_def.delay;
    ctx.easef = p_def.easef;
    ctx.finished = false;
    assert(ctx.easef);

    switch (ctx.property_sym)
    {
    case cssa_transform: {
      ctx.start_transforms = old_style->transforms;
      ctx.end_transforms = new_style->transforms;
      if (ctx.start_transforms && !ctx.end_transforms)
        ctx.end_transforms = ctx.start_transforms->make_initial();
      else if (ctx.end_transforms && !ctx.start_transforms)
        ctx.start_transforms = ctx.end_transforms->make_initial();
      else if (ctx.end_transforms && ctx.start_transforms) {
        if (!ctx.end_transforms->is_compatible(ctx.start_transforms))
          ctx.finished = true;
        else if (ctx.end_transforms->is_identical(ctx.start_transforms))
          ctx.finished = true;
      }
      else {
        ctx.finished = true;
        return;
      }
      ctx.current_transforms = ctx.start_transforms->clone();
    } break;
    case cssa_background_gradient: {
      if (!gradient::is_compatible(old_style->back_gradient, new_style->back_gradient))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->back_gradient);
      ctx.end = value::wrap_resource(new_style->back_gradient);
      ctx.current = value::wrap_resource(old_style->back_gradient->clone());
    } break;
    case cssa_foreground_gradient: {
      if (!gradient::is_compatible(old_style->fore_gradient, new_style->fore_gradient))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->fore_gradient);
      ctx.end = value::wrap_resource(new_style->fore_gradient);
      ctx.current = value::wrap_resource(old_style->fore_gradient->clone());
    } break;
    case cssa_box_shadow: 
    {
      value_handle<shadow_def> old_box_shadow = old_style->box_shadow;
      value_handle<shadow_def> new_box_shadow = new_style->box_shadow;
      if(!old_box_shadow && !new_box_shadow) { ctx.finished = true; return; }
      if (old_box_shadow && !new_box_shadow) { new_box_shadow = old_box_shadow->clone_invisible(); }
      else if (!old_box_shadow && new_box_shadow) { old_box_shadow = new_box_shadow->clone_invisible(); }
      else if (!shadow_def::is_compatible(old_style->box_shadow, new_style->box_shadow))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_box_shadow);
      ctx.end = value::wrap_resource(new_box_shadow);
      ctx.current = value::wrap_resource(old_box_shadow->clone());
    } break;

    case cssa_text_shadow: 
    {
      if (!shadow_def::is_compatible(old_style->text_shadow, new_style->text_shadow))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->text_shadow);
      ctx.end = value::wrap_resource(new_style->text_shadow);
      ctx.current = value::wrap_resource(old_style->text_shadow->clone());
    } break;

    case cssa_background_image: {
      if (!old_style->back_image.id.img() || !new_style->back_image.id.img()) 
      {
        ctx.finished = true;
        return;
      }
      handle<gool::image> s = old_style->back_image.id.img();
      handle<gool::image> e = new_style->back_image.id.img();
      if (s == e) {
        ctx.finished = true;
        return;
      }
      if (s) ctx.start.set_resource(s);
      if (e) ctx.end.set_resource(e);
      this->back = new morphing_image(s, e);
      ctx.current.set_resource(this->back);
    } break;

    case cssa_foreground_image: 
    {
      if (!old_style->fore_image.id.img() || !new_style->fore_image.id.img())
      {
        ctx.finished = true;
        return;
      }
      handle<gool::image> s = old_style->fore_image.id.img();
      handle<gool::image> e = new_style->fore_image.id.img();
      if (s == e) {
        ctx.finished = true;
        return;
      }
      if (s) ctx.start.set_resource(s);
      if (e) ctx.end.set_resource(e);
      this->fore = new morphing_image(s, e);
      ctx.current.set_resource(this->fore);
    } break;

    case cssa_max_width: 
    {
      ctx.start = old_style->max_width.is_undefined() || old_style->max_width.is_auto()
          ? old_style->to_value(cssa_width)
          : old_style->to_value(p_def.prop_sym);
      ctx.end = new_style->max_width.is_undefined() || new_style->max_width.is_auto()
          ? new_style->to_value(cssa_width)
          : new_style->to_value(p_def.prop_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_min_width:
    {
      ctx.start = old_style->min_width.is_undefined() || old_style->min_width.is_auto()
        ? old_style->to_value(cssa_width)
        : old_style->to_value(p_def.prop_sym);
      ctx.end = new_style->min_width.is_undefined() || new_style->min_width.is_auto()
        ? new_style->to_value(cssa_width)
        : new_style->to_value(p_def.prop_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_max_height:
    {
      ctx.start = old_style->max_height.is_undefined() || old_style->max_height.is_auto()
        ? old_style->to_value(cssa_height)
        : old_style->to_value(p_def.prop_sym);
      ctx.end = new_style->max_height.is_undefined() || new_style->max_height.is_auto()
        ? new_style->to_value(cssa_height)
        : new_style->to_value(p_def.prop_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_min_height:
    {
      ctx.start = old_style->min_height.is_undefined() || old_style->min_height.is_auto()
        ? old_style->to_value(cssa_height)
        : old_style->to_value(p_def.prop_sym);
      ctx.end = new_style->min_height.is_undefined() || new_style->min_height.is_auto()
        ? new_style->to_value(cssa_height)
        : new_style->to_value(p_def.prop_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_visibility:
      ctx.start = old_style->visibility.val();
      ctx.end = new_style->visibility.val();
      if (ctx.start == ctx.end) 
      {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    break;

    case cssa_display:
      ctx.start = old_style->display.val();
      ctx.end = new_style->display.val();
      if (ctx.start == ctx.end) 
      {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
      break; 

/*    case cssa_height: 
      if (new_style->height.is_min_intrinsic()) {
        b->measure_inplace(v);
        ctx.end = size_v(b->min_content_height(v), size_v::unit_type::px);
        goto DEFAULT_HANDLING;
      }
      // else fall through
*/

    case cssa_position:
      ctx.start = old_style->position.val();
      ctx.end = new_style->position.val();
      if (ctx.start == ctx.end) 
      {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
      break;


    default: 
      {
        ctx.end = new_style->to_value(p_def.prop_sym);
//DEFAULT_HANDLING:
        ctx.start = old_style->to_value(p_def.prop_sym);
//DEFAULT_CHECK:
        if (ctx.start == ctx.end) {
          ctx.finished = true;
          return;
        }
        ctx.current = ctx.start;
      } break;
    }
  }

  uint css_std_property_animator::start(view &v, element *b, const style *new_style, const style *old_style) {
    super::start(v, b, new_style, old_style);
    // assert(new_style->unique);
    uint system_clock = v.get_animation_ticks();
    current_clock = system_clock;

    transitions = new_style->transitions;
    assert(transitions);

    uint n_active = 0;

    for (int n = 0; n < transitions->props.size(); ++n) {
      const transition_item& ti = transitions->props.value(n);
      bool created = false;
      property_ctx& ctx = props.get_ref(ti.prop_sym, created);
      assert(created);
      init_property_ctx(v, b, ti, ctx, new_style, old_style);
      ++n_active;
    }
    if (n_active == 0) 
      return 0;

    return step(v, b, system_clock);
  }

  uint css_std_property_animator::step(view &v, element *b, uint system_clock) {

#if defined(DELAYED_ANIMATION_END)
    if(finished)
      return 0;
#endif

    current_clock = system_clock;
    uint local_clock = current_clock - start_clock;

    bool rem = false;
    uint cnt = 0;
    FOREACH(n, props) 
    {
      property_ctx &ctx = props(n);
#ifdef DEBUG
      if (ctx.easef.inversed)
        ctx.easef.inversed = ctx.easef.inversed;
#endif
      rem |= ctx.morph(v, b, local_clock);
      if (!ctx.finished) {
        ++cnt;
      }
    }

    if (!b->c_style->animated || !b->c_style->unique) {
      b->c_style = style::create_unique();
      *b->c_style = *b->d_style;
      b->c_style->animated = true;
      b->c_style->unique = true;
    }

    if (b->animator) {
      style p_style = *b->c_style;
      b->animator->update_style(v, b, *b->c_style);
      STYLE_CHANGE_TYPE sct = changes(&p_style, b->c_style);
      sct_max = max(sct_max, uint(sct));
      if (sct) v.add_to_update(b, sct);
      else v.refresh(b);
    }

    finished = finished | (cnt == 0);
#if !defined(DELAYED_ANIMATION_END)
    if (finished)
      return 0;
#endif
    return STEP_DELAY;
  }
  void css_std_property_animator::stop(view &v, element *b) {
    animation::stop(v, b);
    //dbg_printf("css_std_property_animator::stop\n");
    v.add_to_update(b, (STYLE_CHANGE_TYPE)sct_max);
  }

  uint css_std_property_animator::update_targets(view &v, element *b, const style *new_style, const style *old_style)
  {
    hdocument pd = b->doc();
    string    url = pd->uri().src;
    uint      max_time = 0;
    
    if (!transitions)
      return max_time;

    start_clock = v.get_animation_ticks();

    assert(old_style != element::null_style);
      
    for (int n = 0; n < transitions->props.size(); ++n) {
      const transition_item& p_def = transitions->props.value(n);

      bool created = false;
      property_ctx& ctx = props.get_ref(p_def.prop_sym,created);
      if (created)
        init_property_ctx(v, b, p_def, ctx, new_style, old_style);
      else if (!ctx.end_defined() || !ctx.start_defined())
        init_property_ctx(v, b, p_def, ctx, new_style, old_style);

      if (!ctx.easef) continue;

      value nv;

      switch (p_def.prop_sym) {
      case cssa_transform: {
        if (transforms::is_identical(ctx.end_transforms, new_style->transforms)) {
          ;//dbg_printf("cssa_transform identical to end\n");
        }
        else if (transforms::is_identical(ctx.start_transforms,new_style->transforms) && !ctx.finished)
        {
          ctx.start_transforms = ctx.current_transforms->clone();
          ctx.end_transforms = new_style->transforms ? new_style->transforms->clone() : ctx.current_transforms->make_initial();
          if (ctx.current_clock > int(ctx.delay)) {
            ctx.finished = false;
            if (ctx.s_duration.is_undefined()) ctx.s_duration = ctx.duration;
            if (ctx.s_delay.is_undefined()) ctx.s_delay = ctx.delay;
            ctx.duration = ctx.current_clock;
            ctx.delay = 0;
            ctx.current_clock = 0;
            ctx.easef.inversed = true;
          }
          else {
            ctx.finished = true;
          }
        } else if(ctx.current_transforms) {
          if (ctx.s_duration.is_defined()) ctx.duration = ctx.s_duration;
          if (ctx.s_delay.is_defined()) ctx.delay = ctx.s_delay;
          ctx.start_transforms = ctx.current_transforms->clone();
          ctx.end_transforms = new_style->transforms ? new_style->transforms->clone() : ctx.current_transforms->make_initial();
          ctx.finished = false;
          ctx.current_clock = 0;
          ctx.easef.inversed = false;
          //dbg_printf("cssa_transform restart\n");
        } else 
          init_property_ctx(v, b, p_def, ctx, new_style, old_style);
      } break;
      case cssa_background_gradient: {
        if (!gradient::is_compatible(old_style->back_gradient, new_style->back_gradient))
        {
          ctx.finished = true;
          continue;
        }
        nv = value::wrap_resource(new_style->back_gradient);
        goto DEFAULT_HANDLING;
      } break;
      case cssa_foreground_gradient: 
      {
        if (!gradient::is_compatible(old_style->fore_gradient, new_style->fore_gradient))
        {
          ctx.finished = true;
          continue;
        }
        nv = value::wrap_resource(new_style->fore_gradient);
        goto DEFAULT_HANDLING;
      } break;

      case cssa_box_shadow: {
        //if (!shadow_def::is_compatible(old_style->box_shadow, new_style->box_shadow)) 
        if(!new_style->box_shadow)
        {
          ctx.finished = true;
          continue;
        }
        nv = value::wrap_resource(new_style->box_shadow);
        goto DEFAULT_HANDLING;
      } break;

      case cssa_text_shadow: {
        if (!shadow_def::is_compatible(old_style->text_shadow, new_style->text_shadow)) {
          ctx.finished = true;
          continue;
        }
        nv = value::wrap_resource(new_style->text_shadow);
        goto DEFAULT_HANDLING;
      } break;

      case cssa_background_image: 
      {
        this->back = nullptr;
        if (!old_style->back_image.id.img() && !new_style->back_image.id.img())
        {
          ctx.finished = true;
          continue;
        }
        handle<gool::image> s = old_style->back_image.id.img();
        handle<gool::image> e = new_style->back_image.id.img();
        if (s == e) 
        {
          ctx.finished = true;
          continue;
        }
        if (s) ctx.start.set_resource(s);
        if (e) ctx.end.set_resource(e);
        this->back = new morphing_image(s, e);
        ctx.current.set_resource(this->back);
      } break;

      case cssa_foreground_image: {
        this->fore = nullptr;
        if (!old_style->fore_image.id.img() && !new_style->fore_image.id.img())
        {
          ctx.finished = true;
          continue;
        }
        handle<gool::image> s = old_style->fore_image.id.img();
        handle<gool::image> e = new_style->fore_image.id.img();
        if (s == e)
        {
          ctx.finished = true;
          continue;
        }
        if (s) ctx.start.set_resource(s);
        if (e) ctx.end.set_resource(e);
        this->fore = new morphing_image(s, e);
        ctx.current.set_resource(this->fore);
      } break;

      case cssa_max_width:
        nv = new_style->max_width.is_undefined() || new_style->max_width.is_auto()
          ? new_style->to_value(cssa_width)
          : new_style->to_value(p_def.prop_sym);
        goto DEFAULT_HANDLING;
      
      case cssa_min_width:
        nv = new_style->min_width.is_undefined() || new_style->min_width.is_auto()
          ? new_style->to_value(cssa_width)
          : new_style->to_value(p_def.prop_sym);
        goto DEFAULT_HANDLING;

      case cssa_max_height:
        nv = new_style->max_height.is_undefined() || new_style->max_height.is_auto()
          ? new_style->to_value(cssa_height)
          : new_style->to_value(p_def.prop_sym);
        goto DEFAULT_HANDLING;

      case cssa_min_height:
        nv = new_style->min_height.is_undefined() || new_style->min_height.is_auto()
          ? new_style->to_value(cssa_height)
          : new_style->to_value(p_def.prop_sym);
        goto DEFAULT_HANDLING;

      case cssa_visibility:
        nv = new_style->visibility.val();
        goto DEFAULT_HANDLING;

      case cssa_display:
        nv = new_style->display.val();
        goto DEFAULT_HANDLING;

      default: {
          nv = new_style->to_value(p_def.prop_sym);
DEFAULT_HANDLING:
          if (ctx.end == nv) {
          }
          else if (ctx.start == nv && !ctx.finished)
          {
            ctx.start = ctx.current;
            ctx.end = nv;
            if (ctx.current_clock > int(ctx.delay)) {
              ctx.finished = false;
              if (ctx.s_duration.is_undefined()) ctx.s_duration = ctx.duration;
              if (ctx.s_delay.is_undefined()) ctx.s_delay = ctx.delay;
              ctx.duration = ctx.current_clock;
              ctx.delay = 0;
              ctx.current_clock = 0;
              ctx.easef.inversed = true;
            }
            else {
              ctx.finished = true;
            }
          }
          else {
            if (ctx.s_duration.is_defined()) ctx.duration = ctx.s_duration;
            if (ctx.s_delay.is_defined()) ctx.delay = ctx.s_delay;
            ctx.easef.inversed = false;
            ctx.start = ctx.current;
            ctx.end = nv;
            ctx.current_clock = 0;
            ctx.finished = false;
          }
        } break;
      }
      //props.push(ctx);
      max_time = max(max_time, ctx.duration + ctx.delay);
    }
    return max_time;
  }

  bool css_std_property_animator::update_style(view &v, element *b, style &s) {
    // dbg_printf("css_std_property_animator::update_style\n");
    hdocument pd = b->doc();
    string    url = pd->uri().src;
    FOREACH(n, props) {
      property_ctx &ctx = props(n);
      if (ctx.current_transforms)
        s.transforms = ctx.current_transforms;
      else {
        set_attribute_value(element_context(v,pd,b), s, ctx.property_sym, ctx.current);
      }
    }
    super::update_style(v, b, s);
    return true;
  }

  uint css_std_animate_animator::start(view &v, element *b, const style *new_style, const style *old_style)
  {
    if (!new_style->animation_name.length()) return 0;

    do {
      if (new_style->animation_style_bag) {
        frames = new_style->animation_style_bag->get_keyframes(new_style->animation_name);
        if (frames)
          break;
      }
      document* pd = b->doc();
      frames = pd->styles().get_keyframes(new_style->animation_name);
      if (frames)
        break;
      frames = html_app()->stock_styles().get_keyframes(new_style->animation_name);
      if (frames)
        break;
      view::debug_printf(OT_CSS, OS_WARNING, "keyframes: %s not found\n", new_style->animation_name.c_str());
      return 0;
      
    } while (false);

    frames_count = new_style->animation_iteration_count * (frames->edges.size() - 1) + 1;
    duration = new_style->animation_duration;
    delay = new_style->animation_delay;
    
    if (frames_count == 0) return 0;
    // assert(new_style->unique);
    uint system_clock = v.get_animation_ticks();
    current_clock = system_clock;
    delay_end_clock = current_clock + delay;

    super::start(v, b, new_style, old_style);
    return step(v, b, current_clock);

  }

  bool css_std_animate_animator::next_frame(view &v, element *b)
  {
    if (++current_frame_no >= frames_count)
      return false;

    int nedges = frames->edges.size();

    int edge_no = (current_frame_no - 1) % nframes();

#ifdef DEBUG
    if (edge_no == 0)
      edge_no = edge_no;
#endif
    int next_edge_no = edge_no + 1;

    bool forward = true;

    switch (b->d_style->animation_direction) 
    {
      case  animation_direction_normal: 
        break;
      case  animation_direction_reverse: 
        REVERSE:
        forward = false; 
        edge_no = nframes() - edge_no;
        next_edge_no = nframes() - next_edge_no;
        break;
      case  animation_direction_alternate: 
        if ((current_frame_no & 1) == 0) goto REVERSE;
        break;
      case  animation_direction_alternate_reverse:
        if ((current_frame_no & 1) == 1) goto REVERSE;
        break;
    }
    
    const keyframes::edge_def& curr_edge = frames->edges[edge_no];
    const keyframes::edge_def& next_edge = frames->edges[next_edge_no];

    slice<keyframes::prop_def> key_props = curr_edge.key_props();

    uint frame_duration = uint(max(0.0f,(abs(next_edge.key_position - curr_edge.key_position)) * duration));
    auto easef = b->d_style->animation_timing_function.val();
    
    if (current_frame_end_clock) {
      current_frame_end_clock += frame_duration;
      current_frame_start_clock = current_frame_end_clock - frame_duration;
    }
    else {
      current_frame_start_clock = current_clock;
      current_frame_end_clock = current_clock + frame_duration;
    }

    style initial_style = *b->c_style;
    apply_props_to_style(v, b, curr_edge.key_props(), initial_style);
    style final_style = initial_style;
    apply_props_to_style(v, b, next_edge.key_props(), final_style);
   
    for (uint n = 0; n < key_props.length; ++n) {
      uint prop_sym = key_props[n].prop_symbol;
      cssa::each_terminal_prop_of(prop_sym, [&](cssa::symbol_t ts) {
        bool created = false;
        property_ctx& ctx = props.get_ref(ts, created);
        init_property_ctx(v, b, ts, frame_duration, forward, easef, ctx, &final_style, &initial_style);
      });
    }

    return true;
  }

  void css_std_animate_animator::init_property_ctx(view &v, element *b, 
      uint property_sym,
      uint  duration, 
      bool  forward,
      ease::function easef, 
      property_ctx& ctx,
      const style *new_style, const style *old_style)
  {
    ctx.property_sym = property_sym;
    ctx.duration = duration;
    ctx.delay = 0;
    ctx.easef = easef;
    ctx.easef.inversed = !forward;
    ctx.current_clock = 0;
    ctx.finished = false;
    assert(ctx.easef);

    switch (ctx.property_sym)
    {
    case cssa_transform: {
      ctx.start_transforms = old_style->transforms;
      ctx.end_transforms = new_style->transforms;
      if (ctx.start_transforms && !ctx.end_transforms)
        ctx.end_transforms = ctx.start_transforms->make_initial();
      else if (ctx.end_transforms && !ctx.start_transforms)
        ctx.start_transforms = ctx.end_transforms->make_initial();
      else if (ctx.end_transforms && ctx.start_transforms) {
        if (!ctx.end_transforms->is_compatible(ctx.start_transforms))
          ctx.finished = true;
        else if (ctx.end_transforms->is_identical(ctx.start_transforms))
          ctx.finished = true;
      }
      else {
        ctx.finished = true;
        return;
      }
      ctx.current_transforms = ctx.start_transforms->clone();
    } break;
    case cssa_background_gradient: {
      if (!gradient::is_compatible(old_style->back_gradient, new_style->back_gradient))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->back_gradient);
      ctx.end = value::wrap_resource(new_style->back_gradient);
      ctx.current = value::wrap_resource(old_style->back_gradient->clone());
    } break;
    case cssa_foreground_gradient: {
      if (!gradient::is_compatible(old_style->fore_gradient, new_style->fore_gradient))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->fore_gradient);
      ctx.end = value::wrap_resource(new_style->fore_gradient);
      ctx.current = value::wrap_resource(old_style->fore_gradient->clone());
    } break;
    case cssa_box_shadow:
    {
      if (!shadow_def::is_compatible(old_style->box_shadow, new_style->box_shadow))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->box_shadow);
      ctx.end = value::wrap_resource(new_style->box_shadow);
      ctx.current = value::wrap_resource(old_style->box_shadow->clone());
    } break;

    case cssa_text_shadow:
    {
      if (!shadow_def::is_compatible(old_style->text_shadow, new_style->text_shadow))
      {
        ctx.finished = true;
        return;
      }
      ctx.start = value::wrap_resource(old_style->text_shadow);
      ctx.end = value::wrap_resource(new_style->text_shadow);
      ctx.current = value::wrap_resource(old_style->text_shadow->clone());
    } break;


    case cssa_max_width:
    {
      ctx.start = old_style->max_width.is_undefined() || old_style->max_width.is_auto()
        ? old_style->to_value(cssa_width)
        : old_style->to_value(property_sym);
      ctx.end = new_style->max_width.is_undefined() || new_style->max_width.is_auto()
        ? new_style->to_value(cssa_width)
        : new_style->to_value(property_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_min_width:
    {
      ctx.start = old_style->min_width.is_undefined() || old_style->min_width.is_auto()
        ? old_style->to_value(cssa_width)
        : old_style->to_value(property_sym);
      ctx.end = new_style->min_width.is_undefined() || new_style->min_width.is_auto()
        ? new_style->to_value(cssa_width)
        : new_style->to_value(property_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_max_height:
    {
      ctx.start = old_style->max_height.is_undefined() || old_style->max_height.is_auto()
        ? old_style->to_value(cssa_height)
        : old_style->to_value(property_sym);
      ctx.end = new_style->max_height.is_undefined() || new_style->max_height.is_auto()
        ? new_style->to_value(cssa_height)
        : new_style->to_value(property_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_min_height:
    {
      ctx.start = old_style->min_height.is_undefined() || old_style->min_height.is_auto()
        ? old_style->to_value(cssa_height)
        : old_style->to_value(property_sym);
      ctx.end = new_style->min_height.is_undefined() || new_style->min_height.is_auto()
        ? new_style->to_value(cssa_height)
        : new_style->to_value(property_sym);
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;

    case cssa_visibility:
      ctx.start = old_style->visibility.val();
      ctx.end = new_style->visibility.val();
      if (ctx.start == ctx.end)
      {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
      break;

    case cssa_display:
      ctx.start = old_style->display.val();
      ctx.end = new_style->display.val();
      if (ctx.start == ctx.end)
      {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
      break;

      /*    case cssa_height:
      if (new_style->height.is_min_intrinsic()) {
      b->measure_inplace(v);
      ctx.end = size_v(b->min_content_height(v), size_v::unit_type::px);
      goto DEFAULT_HANDLING;
      }
      // else fall through
      */

    default:
    {
      ctx.end = new_style->to_value(property_sym);
      //DEFAULT_HANDLING:
      ctx.start = old_style->to_value(property_sym);
      //DEFAULT_CHECK:
      if (ctx.start == ctx.end) {
        ctx.finished = true;
        return;
      }
      ctx.current = ctx.start;
    } break;
    }
  }

  void css_std_animate_animator::apply_props_to_style(view& v, element* b, slice<keyframes::prop_def> props, style& s)
  {
    document *pd = b->doc();
    string    url = pd->uri().src;
    for(auto& p : props) {
      set_attribute_value(element_context(v,pd,b), s, p.prop_symbol, p.prop_value());
    }
  }

  bool css_std_animate_animator::update_style(view &v, element *b, style &s) {
    // dbg_printf("css_std_animate_animator::update_style\n");

    //if (!is_running)
    //  return false;

    hdocument pd = b->doc();
    string    url = pd->uri().src;
    FOREACH(n, props) {
      property_ctx &ctx = props(n);
      if (ctx.current_transforms)
        s.transforms = ctx.current_transforms;
      else {
        set_attribute_value(element_context(v,pd,b), s, ctx.property_sym, ctx.current);
      }
    }
    super::update_style(v, b, s);
    return true;
  }


  uint css_std_animate_animator::step(view &v, element *b, uint system_clock) {

    uint delta = system_clock - current_clock;
    current_clock = system_clock;

    if (b->d_style->animation_play_state == animation_play_state_paused) {
      //current_frame_end_clock += delta;
      return STEP_DELAY;
    }

    if (delay_end_clock && (current_clock < delay_end_clock)) {
      //current_frame_end_clock += delta;
      return STEP_DELAY;
    }
    else if (delay_end_clock) {
      //delta = delay_end_clock - current_clock;
      //current_frame_start_clock = current_clock;
      //current_frame_end_clock = current_clock + duration;
      delay_end_clock = 0;
    }

    if (current_clock >= current_frame_end_clock) {
      if (!next_frame(v, b))
        return 0;
    }

    bool rem = false;
    //uint cnt = 0;
    FOREACH(n, props)
    {
      property_ctx &ctx = props(n);
      rem |= ctx.morph(v, b, current_clock - current_frame_start_clock);
    }
    if (!b->c_style->animated || !b->c_style->unique) {
      b->c_style = style::create_unique();
      *b->c_style = *b->d_style;
      b->c_style->animated = true;
      b->c_style->unique = true;
    }

    if (b->animator) {
      style p_style = *b->c_style;
      update_style(v, b, *b->c_style);
      STYLE_CHANGE_TYPE sct = changes(&p_style, b->c_style);
      sct_max = max(sct_max, sct);
      if (sct) v.add_to_update(b, sct);
      else v.refresh(b);
    }
    //else
    //  return 0;

    if (delay) {
      //delay_end_clock = current_clock + delay;
      delay = 0;
    }

    return STEP_DELAY;
  }

  void css_std_animate_animator::stop(view &v, element *b) {
    animation::stop(v, b);
    //dbg_printf("css_std_animate_animator::stop\n");
    //is_running = false;
    if (sct_max) {
      b->drop_style(&v);
      v.add_to_update(b, STYLE_CHANGE_TYPE(sct_max));
    }
  }

} // namespace html
