#include "html.h"
#include "html-actions-stack.h"
#include "html-dom-merge.h"

namespace html {
  namespace behavior {

    node *split_at(view &v, editing_ctx *ectx, action *ra, bookmark &bm,
                   handle<element> until, bool after, bool &created,
                   bookmark &other, bool force_creation = false) {
      if (bm.node == until) return nullptr;

      // bm.linearize();
      int idx = bm.linear_pos();
      // int other_idx = other.valid() ? other.linear_pos();
      // dbg_print(bm);

      /*if (bm.node->is_text()) { -- to funky - needs additional flag
        auto isspace = [](wchar c) { return is_space(c) && (c != NBSP_CHAR); };
        wchars chars = bm.node.ptr_of<text>()->chars();
        if (after) {
          for (int n = idx - 1; n > 0; --n) {
            if (!isspace(chars[n]))
              break;
            idx = n;
          }
        }
        else {
          for (int n = idx; n < chars.size(); ++n) {
            idx = n;
            if (!isspace(chars[n]))
              break;
          }
        }
      }*/

      hnode nn = split_node::exec(v, ectx, ra, bm.node, idx, force_creation);
      if (nn) {
        if (bm.node == other.node && other.linear_pos() >= idx) {
          other.node     = nn;
          other.pos      = other.linear_pos() - idx;
          other.after_it = false;
        } else if ((other.node == bm.node->parent) && !after)
          other.pos = other.pos + 1;
        if (after)
          bm = bm.node->this_pos(true);
        else
          bm = nn->this_pos(false);
        created = true;
        if (bm.node == until) return nn;
        return split_at(v, ectx, ra, bm, until, after, created, other);
      }
      if (bm.at_start())
        bm = bm.node->this_pos(false);
      else if (bm.at_end())
        bm = bm.node->this_pos(true);
      else
        bm = bm.node->this_pos(after);
      // dbg_print(bm);
      return split_at(v, ectx, ra, bm, until, after, created, other);
    }

    helement split_element_at(view &v, editing_ctx *ectx, action *ra,
                              bookmark &bm, handle<element> el, bool after,
                              bool &created, bookmark &other) {
      node *pn = split_at(v, ectx, ra, bm, el->parent.ptr(), after, created, other);
      ASSERT(pn && pn->is_element());
      helement newel = static_cast<element *>(pn);
      return newel;
    }

    static void undo_chain(view &v, editing_ctx *ectx, atomic_action *head) {
      if (!head) return;
      head->undo(v, ectx);
      undo_chain(v, ectx, head->prev);
    }
    static void redo_chain(view &v, editing_ctx *ectx, atomic_action *head) {
      if (!head) return;
      redo_chain(v, ectx, head->prev);
      head->redo(v, ectx);
    }

    action::action(editing_ctx *ectx, const ustring &c) : caption(c) {
      caret_before.set(ectx->get_caret(), ectx->root());
    }

    void action::close(editing_ctx *ectx) {
      caret_after.set(ectx->get_caret(), ectx->root());
    }

    void action::undo(view &v, editing_ctx *ectx) {
      undo_chain(v, ectx, chain);
      bookmark c = this->caret_before.get(ectx->root());
      ectx->select(v, c);
    }
    void action::redo(view &v, editing_ctx *ectx) {
      redo_chain(v, ectx, chain);
      bookmark c = this->caret_after.get(ectx->root());
      ectx->select(v, c);
    }

    node *placeholder_action::placeholder() const {
      ASSERT(this->chain);
      return this->chain.ptr_of<insert_node>()->n;
    }

    range_action::range_action(editing_ctx *ectx, const ustring &c)
        : action(ectx, c) {
      anchor_before.set(ectx->get_anchor(), ectx->root());
    }
    void range_action::close(editing_ctx *ectx) {
      super::close(ectx);
      anchor_after.set(ectx->get_anchor(), ectx->root());
    }

    void range_action::undo(view &v, editing_ctx *ectx) {
      undo_chain(v, ectx, chain);
      bookmark start = this->anchor_before.get(ectx->root());
      bookmark end   = this->caret_before.get(ectx->root());
      ectx->select(v, start, end);
    }
    void range_action::redo(view &v, editing_ctx *ectx) {
      redo_chain(v, ectx, chain);
      bookmark start = this->anchor_after.get(ectx->root());
      bookmark end   = this->caret_after.get(ectx->root());
      ectx->select(v, start, end);
    }

    node *split_node::exec(view &v, editing_ctx *ectx, action *group, hnode pn,
                           int pos, bool force) {
      handle<split_node> op = new split_node();
      op->n                 = pn;
      op->pos               = pos;
      node *n               = pn->split(v, pos, force);
      if (n) group->add(op.ptr());
      return n;
    }

    void split_node::undo(view &v, editing_ctx *ectx) {
      n->unsplit(v);
      // v.add_to_update(n->get_element(),CHANGES_MODEL);
    }
    void split_node::redo(view &v, editing_ctx *ectx) {
      n->split(v, pos);
      // v.add_to_update(n->get_element(),CHANGES_MODEL);
    }

    element *
    wrap_nodes::exec(view &v, editing_ctx *ectx, action *group, helement el,
                     int start, int end, tag::symbol_t t,
                     const attribute_bag &atts) // returns new element created
    {
      element *wrapper = new element(t);
      wrapper->atts    = atts;
      return exec(v, ectx, group, el, start, end, wrapper);
    }

    element *wrap_nodes::exec(view &v, editing_ctx *ectx, action *group,
                              helement el, int start, int end,
                              element *wrapper) // returns new element created
    {
      handle<wrap_nodes> op = group->add(new wrap_nodes());
      op->wrapper           = wrapper;
      op->base              = el;
      op->start             = start;
      op->end               = end;
      op->base->drop_layout_tree(&v);
      array<hnode> nodes = op->base->nodes(start, end);
      FOREACH(i, nodes)
      nodes[i]->remove(false);
      for (int n = 0; n < nodes.size(); ++n)
        op->wrapper->append(nodes[n]);
      op->base->insert(start, op->wrapper, &v);
      v.add_to_update(op->wrapper, CHANGES_MODEL);
      v.add_to_update(op->base, CHANGES_MODEL);
      return op->wrapper;
    }

    void wrap_nodes::undo(view &v, editing_ctx *ectx) {
      array<hnode> nodes = wrapper->nodes;
      wrapper->remove(false);
      FOREACH(i, nodes)
      nodes[i]->remove(false);
      base->insert_nodes(start, nodes(), &v);
    }
    void wrap_nodes::redo(view &v, editing_ctx *ectx) {
      array<hnode> nodes = base->nodes(start, end);
      FOREACH(i, nodes)
      nodes[i]->remove(false);
      for (int n = 0; n < nodes.size(); ++n)
        wrapper->append(nodes[n]);
      base->insert(start, wrapper, &v);
      v.add_to_update(wrapper, CHANGES_MODEL);
      v.add_to_update(base, CHANGES_MODEL);
    }

    pair<bookmark, bookmark> unwrap_element::exec(view &v, editing_ctx *ectx,
                                                  action *group, helement el) {
      handle<unwrap_element> op = group->add(new unwrap_element());
      op->pel                   = el->parent;
      op->el                    = el;
      op->pos                   = max(0,el->node_index);
      op->nodes                 = el->nodes;
      op->el->clear();
      el->remove(true);
      op->pel->insert_nodes(op->pos, op->nodes());
      v.add_to_update(op->pel, CHANGES_MODEL);
      return pair<bookmark, bookmark>(op->nodes.first()->start_caret_pos(v),
                                      op->nodes.last()->end_caret_pos(v));
    }
    void unwrap_element::undo(view &v, editing_ctx *ectx) {
      for (int n = 0; n < nodes.size(); ++n)
        pel->nodes[pos]->remove(true);
      pel->insert(pos, el);
      el->insert_nodes(0, nodes());
      v.add_to_update(pel, CHANGES_MODEL);
    }
    void unwrap_element::redo(view &v, editing_ctx *ectx) {
      el->clear();
      el->remove(true);
      pel->insert_nodes(pos, nodes());
      v.add_to_update(pel, CHANGES_MODEL);
    }

    bool morph_element::exec(view &v, editing_ctx *ectx, action *group,
                             helement pel, tag::symbol_t element_tag) {
      handle<morph_element> op = group->add(new morph_element());
      op->el                   = pel;
      op->old_tag              = pel->tag;
      pel->tag = op->new_tag = element_tag;

      v.refresh(pel);
      pel->drop_styles(v);

      if (element_tag == tag::T_P) {
        helement t = pel->last_element();
        while (t) {
          helement      prev = t->prev_element();
          tag::TAG_TYPE tt   = tag::type(t->tag);

          if (tt != tag::INLINE_BLOCK_TAG && tt != tag::INLINE_TAG &&
              tt != tag::INFO_TAG) {
            delete_node::exec(v, ectx, group, t.ptr());
            insert_node::exec(v, ectx, group, pel->parent, pel->node_index + 1,t);
          }
          t = prev;
        }
      }

      event_behavior evt(pel, pel, CONTENT_CHANGED, ATTRIBUTES_CHANGED);
      v.post_behavior_event(evt); // notify the view that its content has been changed.
      v.add_to_update(pel, CHANGES_MODEL);
      return true;
    }

    void morph_element::undo(view &v, editing_ctx *ectx) {
      el->tag = old_tag;
      event_behavior evt(el, el, CONTENT_CHANGED, ATTRIBUTES_CHANGED);
      v.post_behavior_event(evt); // notify the view that its content has been changed.
      v.add_to_update(el, CHANGES_MODEL);
    }
    void morph_element::redo(view &v, editing_ctx *ectx) {
      el->tag = new_tag;
      event_behavior evt(el, el, CONTENT_CHANGED, ATTRIBUTES_CHANGED);
      v.post_behavior_event(evt); // notify the view that its content has been changed.
      v.add_to_update(el, CHANGES_MODEL);
    }

    bool change_attr::set(view &v, editing_ctx *ectx, action *group,
                          helement pel, const string &name,
                          const ustring &val) {
      handle<change_attr> op = group->add(new change_attr());
      op->el                 = pel;
      op->name               = name;
      op->old_val_exist      = op->el->atts.get(name, op->old_val);
      op->new_val_exist      = true;
      op->el->atts.set(name, op->new_val = val);
      v.add_to_update(op->el, CHANGES_MODEL);
      return true;
    }
    bool change_attr::del(view &v, editing_ctx *ectx, action *group,
                          helement pel, const string &name) {
      handle<change_attr> op = group->add(new change_attr());
      op->el                 = pel;
      op->name               = name;
      op->old_val_exist      = op->el->atts.get(name, op->old_val);
      op->new_val_exist      = false;
      op->el->atts.remove(name);
      v.add_to_update(op->el, CHANGES_MODEL);
      return true;
    }

    void change_attr::undo(view &v, editing_ctx *ectx) {
      if (old_val_exist)
        el->atts.set(name, old_val);
      else
        el->atts.remove(name);
      v.add_to_update(el, CHANGES_MODEL);
    }
    void change_attr::redo(view &v, editing_ctx *ectx) {
      if (new_val_exist)
        el->atts.set(name, new_val);
      else
        el->atts.remove(name);
      v.add_to_update(el, CHANGES_MODEL);
    }

    void attributes_changed::record(view &v, editing_ctx *ectx, action *group,
                                    helement pel, const attribute_bag &oa,
                                    const attribute_bag &na) {
      handle<attributes_changed> op = group->add(new attributes_changed());
      op->el                        = pel;
      op->old_atts                  = oa;
      op->new_atts                  = na;
      op->el->atts                  = na;
      v.add_to_update(op->el, CHANGES_MODEL);
    }

    void attributes_changed::undo(view &v, editing_ctx *ectx) {
      el->atts = old_atts;
      v.add_to_update(el, CHANGES_MODEL);
    }
    void attributes_changed::redo(view &v, editing_ctx *ectx) {
      el->atts = new_atts;
      v.add_to_update(el, CHANGES_MODEL);
    }

    void node_deleted::record(view &v, editing_ctx *ectx, action *group,
                              node *pn, int index) {
      ASSERT(pn && pn->parent);
      handle<node_deleted> op = group->add(new node_deleted());
      op->parent              = pn->parent;
      op->n                   = pn;
      op->pos                 = index;
      v.add_to_update(op->parent, CHANGES_MODEL);
    }

    void node_deleted::undo(view &v, editing_ctx *ectx) {
      parent->insert(pos, n, &v);
      v.add_to_update(parent, CHANGES_MODEL);
    }

    void node_deleted::redo(view &v, editing_ctx *ectx) {
      n->remove(true,&v);
      v.add_to_update(parent, CHANGES_MODEL);
    }

    void node_added::record(view &v, editing_ctx *ectx, action *group, node *pn,
                            int index) {
      ASSERT(pn && pn->parent);
      handle<node_added> op = group->add(new node_added());
      op->parent            = pn->parent;
      op->n                 = pn;
      op->pos               = index;
      v.add_to_update(op->parent, CHANGES_MODEL);
    }

    void node_added::undo(view &v, editing_ctx *ectx) {
      n->remove(true,&v);
      v.add_to_update(parent, CHANGES_MODEL);
    }

    void node_added::redo(view &v, editing_ctx *ectx) {
      parent->insert(pos, n, &v);
      v.add_to_update(parent, CHANGES_MODEL);
    }

    void node_replaced::record(view &v, editing_ctx *ectx, action *group,
                               node *on, node *nn, int index) {
      ASSERT(nn && nn->parent);
      handle<node_replaced> op = group->add(new node_replaced());
      op->parent               = nn->parent;
      op->on                   = on;
      op->nn                   = nn;
      op->pos                  = index;
      v.add_to_update(op->parent, CHANGES_MODEL);
    }

    void node_replaced::undo(view &v, editing_ctx *ectx) {
      ASSERT(nn && parent);
      nn->remove(true);
      parent->insert(pos, on);
      v.add_to_update(parent, CHANGES_MODEL);
    }

    void node_replaced::redo(view &v, editing_ctx *ectx) {
      on->remove(true);
      parent->insert(pos, nn);
      v.add_to_update(parent, CHANGES_MODEL);
    }

    static text *next_text_node(node *n) {
      node *t = n->next_node();
      return (t && t->is_text()) ? t->cast<text>() : 0;
    }
    static text *prev_text_node(node *n) {
      node *t = n->prev_node();
      return (t && t->is_text()) ? t->cast<text>() : 0;
    }

    bool is_natural_text_pos(bookmark pos) {
      if (!pos.node->is_text()) return false;
      // if(!pos.at_char_pos()) return false;
      if (!pos.node->parent) return false;
      node *pn = pos.node;
      if (!pn->parent->nodes.is_valid_index(pn->node_index)) return false;
      if (pn->parent->nodes[pn->node_index] != pn) return false;
      return true;
    }

    bool insert_text::exec(view &v, editing_ctx *ectx, action *group,
                           bookmark &pos, wchars chars) {
      if (!is_natural_text_pos(pos)) {
        if (pos.after_it) {
          handle<text> target = next_text_node(pos.node);
          if (target)
            pos = bookmark(target, 0, false);
          else // we need to create one
          {
            target = new text(chars);
            insert_node::exec(v, ectx, group, pos.node->parent,
                              pos.node->node_index + 1, target);
            pos = target->end_caret_pos(v);
            return true;
          }
        } else {
          handle<text> target = prev_text_node(pos.node);
          if (target)
            pos = bookmark(target, target->chars.last_index(), true);
          else // we need to create one
          {
            target     = new text(chars);
            element *p = pos.node->parent;
            ASSERT(p);
            // element *root = ectx->root();
            int idx = pos.node->node_index;
            if (p->nodes.size() == 1 && pos.node->is_element() &&
                pos.node.ptr_of<element>()->tag == tag::T_BR)
              delete_node::exec(v, ectx, group, pos.node);
            /*else if(p == root)
            {
              tag::symbol_t etag = tag::T_P;
              if(pos.node->is_element()) etag = pos.node.ptr_of<element>()->tag;
              element* nel = new element(etag);
              nel->append(target);
              insert_node::exec(v, ectx, group, p, idx, nel );
              pos = target->end_caret_pos(v);
              return true;
            }*/

            insert_node::exec(v, ectx, group, p, idx, target);
            pos = target->end_caret_pos(v);
            return true;
          }
        }
      }

      handle<insert_text> op = group->add(new insert_text());
      ASSERT(pos.valid());
      ASSERT(pos.node->is_text());

      op->n   = pos.node.ptr_of<text>();
      op->pos = pos.linear_pos();

      // pos.dbg_report("ins char");

      bool r = op->append(v, ectx, group, pos, chars);
      ASSERT(r);
      r = r;

      // pos.pos = pos.pos + op->chars.size() + 1;
      return true;
    }

    static bool needs_nbsp_at_position(wchars text, int pos) {
      if (pos <= 0 || pos >= text.size()) return true;
      if (is_collapsible_space(text[pos - 1])) return true;
      if (is_collapsible_space(text[pos])) return true;
      return false;
    }

    static bool dont_need_nbsp_at_position(wchars text, int pos) {
      //if (pos <= 0 || pos >= text.size()) return false;
      if (pos == 0 && pos < text.last_index() && text[pos] == NBSP_CHAR) {
        if (!is_space(text[pos + 1])) return true;
      }
      if (pos == text.last_index() && pos > 0 && text[pos] == NBSP_CHAR) {
        if (!is_space(text[pos - 1])) return true;
      }
      if (pos > 0 && pos < text.last_index()) {
        if (!is_space(text[pos - 1]) && !is_space(text[pos + 1]) && text[pos] == NBSP_CHAR) return true;
      }
      return false;
    }


    static void update(view& v, node* n) {
      helement nb = n->nearest_block(v);
      nb->try_update_inplace(v);
    }

    bool insert_text::append(view &v, editing_ctx *ectx, action *group, bookmark &bm,
                             wchars chars) {
      ASSERT(bm.valid());
      ASSERT(bm.node->is_text());

      if (bm.node != n) return false;

      helement p = n->parent;

      int pos = bm.linear_pos();

      // pos.linearize();
      if (pos != this->pos + this->chars.size())
        return false; // cannot continue this op

      // int nc = n->chars.size();

      bool is_plaintext = bm.node->get_style(v)->preserve_ws();

      int expand_at     = pos;
      int expand_length = chars.size();

      //if (is_plaintext)
      while (!!chars) {
        wchar c = *chars;
        this->chars.push(c);
        this->n->chars.insert(pos, c);
        ++chars;
        ++pos;
      }
#if 0
      else {
        int initial_pos = pos;
        while (!!chars) {
          wchar c = *chars;
          if (c == ' ') {
            if (needs_nbsp_at_position(n->chars(), pos))
              c = NBSP_CHAR;
          }
          else if (pos > 0 && n->chars[pos - 1] == NBSP_CHAR)
            n->chars[pos - 1] = ' ';
          this->chars.push(c);
          this->n->chars.insert(pos, c);
          ++chars;
          ++pos;
        }
        if (dont_need_nbsp_at_position(this->n->chars(), pos)) {
          this->n->chars[pos] = ' ';
          this->chars.push(' ');
          //++pos;
        }
        if (dont_need_nbsp_at_position(this->n->chars(), initial_pos - 1)) {
          this->n->chars[initial_pos - 1] = ' ';
          this->chars.insert(0, ' ');
        }
      }
#endif 

      if(!is_plaintext)
        nbspify_text::exec(v, ectx, group, bm.node.ptr_of<text>());

      ectx->node_expand(n, expand_at, expand_length);
      ASSERT(n->parent);
#if 0
      n->parent->drop_layout(&v);
      if (!n->parent) // synthetic node
        p->append(n, &v);
      else
        v.add_to_update(n->parent, CHANGES_MODEL);
#else
      update(v, n);
#endif
      bm.pos      = pos - 1;
      bm.after_it = true;

      return true;
    }

    void insert_text::undo(view &v, editing_ctx *ectx) {
      bookmark to = bookmark(n, pos - 1, true);
      n->chars.remove(pos, chars.size());
      if (n->chars.size() == 0) {
        n->chars.push(' ');
        n->chars.size(0);
        to.pos      = 0;
        to.after_it = false;
      }
      ectx->node_shrink(n, pos, chars.size());
      ASSERT(n->parent);
      n->parent->drop_layout(&v);
      v.add_to_update(n->parent, CHANGES_MODEL);
      v.commit_update();
      ectx->select(v, to);
    }

    void insert_text::redo(view &v, editing_ctx *ectx) {
      n->chars.insert(pos, chars());
      ectx->node_expand(n, pos, chars.size());
      ASSERT(n->parent);
      n->parent->drop_layout(&v);
      v.add_to_update(n->parent, CHANGES_MODEL);
      v.commit_update();
      ectx->select(v, bookmark(n, pos + chars.size() - 1, true));
    }

    bool nbspify_text::exec(view &v, editing_ctx *ectx, action *group, text* pt) {

      element* tbe = pt->nearest_text_box();

      bool is_last_node = tbe && pt == tbe->last_node();
      bool is_first_node = tbe && pt == tbe->first_node();

      array<wchar> chars = pt->chars();
      wchar pc = chars[0];
      uint  changes = 0;
      // add nbsps
      for (int n = 1; n < chars.size() - 1; ++n) {
        wchar c = chars[n];
        if (is_collapsible_space(c) && is_collapsible_space(pc)) {
          chars[n] = NBSP_CHAR;
          ++changes;
        }
        pc = chars[n];
      }

      if (is_collapsible_space(chars.last()) && is_last_node) {
        chars.last() = NBSP_CHAR;
        ++changes;
      }
      
      if (is_collapsible_space(chars.first()) && is_first_node) {
        chars.first() = NBSP_CHAR;
        ++changes;
      }

      // remove nbsps 
      for (int n = 1; n < chars.size() - 1; ++n) {
        wchar c = chars[n];
        if (c == NBSP_CHAR && !is_collapsible_space(chars[n-1])) {
          chars[n] = ' ';
          ++changes;
        }
      }
      if (!changes) {
        if (chars.last() == NBSP_CHAR && !is_last_node) {
          chars.last() = ' ';
          ++changes;
        }
        if (chars.first() == NBSP_CHAR && !is_first_node) {
          chars.first() = ' ';
          ++changes;
        }
      }

      if (changes)
        return super::exec(v, ectx, group, pt, chars());
      return true;
    }

    bool replace_text::exec(view &v, editing_ctx *ectx, action *group, text* pt, wchars chars) {

      handle<replace_text> op = group->add(new replace_text());
      ASSERT(pt);
      ASSERT(pt->is_text());

      op->n = pt;
      op->chars = pt->chars;
      pt->chars = chars;

      pt->parent->drop_layout(&v);
      v.add_to_update(pt->parent, CHANGES_MODEL);

      return true;
    }

    void replace_text::undo(view &v, editing_ctx *ectx) {
      n->chars.swap(chars);
      ASSERT(n->parent);
      n->parent->drop_layout(&v);
      v.add_to_update(n->parent, CHANGES_MODEL);
      v.commit_update();
    }

    void replace_text::redo(view &v, editing_ctx *ectx) {
      n->chars.swap(chars);
      ASSERT(n->parent);
      n->parent->drop_layout(&v);
      v.add_to_update(n->parent, CHANGES_MODEL);
      v.commit_update();
    }

    bool insert_node::exec(view &v, editing_ctx *ectx, action *group,
                           element *p, int idx, node *n) {
      handle<insert_node> op = group->add(new insert_node());
      op->parent             = p;
      op->pos                = idx;
      op->n                  = n;
      p->insert(idx, n, &v);
      ectx->node_expand(p, idx, 1);
      return true;
    }
    void insert_node::undo(view &v, editing_ctx *ectx) {
      // helement p = parent.get(ectx
      ectx->node_shrink(parent, pos, 1);
      n->remove(true, &v);
    }
    void insert_node::redo(view &v, editing_ctx *ectx) {
      parent->insert(pos, n, &v);
      ectx->node_expand(parent, pos, 1);
    }

    insert_nodes *insert_nodes::exec(view &v, editing_ctx *ectx, action *group,
                                     element *p, int idx, slice<hnode> n) {
      handle<insert_nodes> op = group->add(new insert_nodes());
      op->parent              = p;
      op->pos                 = idx;
      op->nodes               = n;
      p->insert_nodes(idx, n, &v);
      ectx->node_expand(p, idx, n.size());
      return op;
    }

    insert_nodes *insert_nodes::exec_postfactum(view &v, editing_ctx *ectx,
                                                action *group, element *p,
                                                int idx, slice<hnode> n) {
      handle<insert_nodes> op = group->add(new insert_nodes());
      op->parent              = p;
      op->pos                 = idx;
      op->nodes               = n;
      ectx->node_expand(p, idx, n.size());
      return op;
    }

    void insert_nodes::undo(view &v, editing_ctx *ectx) {
      // helement p = parent.get(ectx
      ectx->node_shrink(parent, pos, nodes.size());
      for (int i = nodes.last_index(); i >= 0; --i)
        nodes[i]->remove(true, &v);
    }
    void insert_nodes::redo(view &v, editing_ctx *ectx) {
      parent->insert_nodes(pos, nodes(), &v);
      ectx->node_expand(parent, pos, 1);
    }

    //////////

    bookmark delete_text_range::exec(view &v, editing_ctx *ectx, action *group,
                                     text *te, int start, int end) {
      if (end < 0) end = te->chars.size();

      wchars r = te->chars(start, end);
      if (!r) // don't need to remove anything
        return te->end_caret_pos(v);

      handle<delete_text_range> op = group->add(new delete_text_range());
      op->n                        = te;
      op->chars                    = r;
      op->start                    = start;
      op->end                      = end;
      te->chars.remove(start, end - start);
      ectx->node_shrink(te, start, end - start);

      bool is_plaintext = te->get_style(v)->preserve_ws();

      /*if (!is_plaintext) {
        int pos = limit<int>(start, 0, te->chars.last_index());
        if (needs_nbsp_at_position(te->chars(), pos)) {
          te->chars[pos] = NBSP_CHAR;
          op->nbsp_replacement_pos = pos;
        }
      }*/

      if (!is_plaintext)
        nbspify_text::exec(v, ectx, group, te);


      v.add_to_update(te->parent, CHANGES_MODEL);
      if (start < te->chars().size())
        return bookmark(te, start, false);
      else
        return bookmark(te, te->chars.last_index(), true);
    }
    void delete_text_range::undo(view &v, editing_ctx *ectx) {
      if (nbsp_replacement_pos.is_defined())
        n->chars[nbsp_replacement_pos] = ' ';
      n->chars.insert(start, chars());
      ectx->node_expand(n, start, chars.size());
      v.add_to_update(n->parent, CHANGES_MODEL);
    }
    void delete_text_range::redo(view &v, editing_ctx *ectx) {
      n->chars.remove(start, end - start);
      if (nbsp_replacement_pos.is_defined())
        n->chars[nbsp_replacement_pos] = NBSP_CHAR;
      ectx->node_shrink(n, start, end - start);
      v.add_to_update(n->parent, CHANGES_MODEL);
    }

    bookmark delete_nodes_range::exec(view &v, editing_ctx *ectx, action *group,
                                      element *pel, int start, int end) {
      slice<hnode> r = pel->nodes(start, end);
      ASSERT(r);
      ASSERT(pel);
      handle<delete_nodes_range> op = new delete_nodes_range();
      group->add(op.ptr());
      start = max(0, start);
      op->el    = pel;
      op->nodes = r;
      op->start = start;
      op->end   = end;
      // pel->nodes.remove(start,end - start);
      for (int n = start; n < end; ++n)
        op->el->nodes[start]->remove(true, &v);

      ectx->node_shrink(pel, start, end - start);

      return bookmark(pel, start, false);
    }
    void delete_nodes_range::undo(view &v, editing_ctx *ectx) {
      el->insert_nodes(start, nodes(), &v);
      ectx->node_expand(el, start, end - start);
    }
    void delete_nodes_range::redo(view &v, editing_ctx *ectx) {
      for (int n = start; n < end; ++n)
        el->nodes[start]->remove(true, &v);
      ectx->node_shrink(el, start, end - start);
    }

    bookmark delete_node::exec(view &v, editing_ctx *ectx, action *group,
                               node *pn) {
      ASSERT(pn->parent);
      handle<delete_node> op = group->add(new delete_node());
      // ASSERT(pn->parent);
      hnode prev = pn->prev_node();
      hnode next = pn->next_node();
      op->parent = pn->parent;
      op->n      = pn;
      op->pos    = pn->node_index;
      op->n->remove(true, &v);
      ectx->node_shrink(op->n, op->pos, 1);
      if (op->pos > 0 && op->pos < op->parent->nodes.last_index()) {
        hnode l = op->parent->nodes[op->pos - 1];
        hnode r = op->parent->nodes[op->pos];
        if (l->is_text() && r->is_text()) {
          op->glued_at = l->unsplit(v);
          return bookmark(l, op->glued_at, false);
        }
      } else if (op->pos != 0 && op->pos == op->parent->nodes.size()) {
        hnode r = op->parent->nodes[op->pos - 1];
        if (r->is_text()) return r->end_caret_pos(v);
      } else if (op->pos == 0 && op->parent->nodes.size()) {
        hnode l = op->parent->nodes[0];
        if (l->is_text()) return l->start_caret_pos(v);
      }
      /*if(prev)
        return prev->end_pos();
      else if(next)
        return next->start_pos();*/
      return bookmark(op->parent, op->pos, false);
    }
    void delete_node::undo(view &v, editing_ctx *ectx) {
      if (glued_at.is_defined()) {
        hnode l = parent->nodes[pos - 1];
        l->split(v, glued_at);
      }
      parent->insert(pos, n, &v);
      ectx->node_expand(n, pos, 1);
    }
    void delete_node::redo(view &v, editing_ctx *ectx) {
      n->remove(true, &v);
      ectx->node_shrink(n, pos, 1);
      if (glued_at.is_defined()) parent->nodes[pos]->unsplit(v);
    }

    bool is_empty_sequence(slice<hnode> nodes) {

      if (nodes.size() == 0) return true;
      for (int n = 0; n < nodes.size(); ++n) {
        node *pn = nodes[n];
        if (pn->is_comment())
          continue;
        else if (pn->is_text()) {
          if (trim(pn->cast<text>()->chars()).length != 0) return false;
        } else if (pn->is_element()) {
          if (pn->cast<element>()->tag == tag::T_BR) continue;
          return false;
        }
      }
      return true;
    }

    bool is_empty_element(element *el) {
      return is_empty_sequence(el->nodes());
    }

    bool is_empty_node(node *pn) {
      if (pn->is_comment())
        return true;
      else if (pn->is_text()) {
        if (trim(pn->cast<text>()->chars()).length == 0) return true;
      } else if (pn->is_element()) {
        if (pn->cast<element>()->tag == tag::T_BR) return true;
      }
      return false;
    }

    // bool is_last_text_node_in_block(text* pt) {
    //  if( !pt->parent->is_of_type<text_block>() ) return false;
    //}

    /*
    bool  normalize_text::exec(view& v, editing_ctx *ectx, action* group, text*
    pt, int pos)
    {
        int start = pos;
        int end = pos;

        for(; end < pt->chars.size(); ++end )
        {
          if( !is_space(pt->chars[end]) )
            break;
        }
        for(--pos; pos >= 0 ; --pos )
        {
          if( !is_space(pt->chars[pos]) )
            break;
          start = pos;
        }

        if( start == end )
          return false;

        handle<normalize_text> op = group->add(new normalize_text());
        op->n = pt;
        op->pos = start;
        op->chars = pt->chars(start, end);
        for( int i = start; i < end; ++i )
          pt->chars[i] = NBSP_CHAR;
        v.add_to_update(pt->parent,CHANGES_MODEL);
    }

    void  normalize_text::undo(view& v, editing_ctx *ectx)
    {
        int t = pos;
        for( int i = 0; i < chars.size(); ++i, ++t )
          n->chars[t] = chars[i];
        v.add_to_update(n->parent,CHANGES_MODEL);
    }
    void  normalize_text::redo(view& v, editing_ctx *ectx)
    {
        int t = pos;
        for( int i = 0; i < chars.size(); ++i, ++t )
          n->chars[t] = NBSP_CHAR;
        v.add_to_update(n->parent,CHANGES_MODEL);
    } */

    bookmark last_char_removed(view &v, editing_ctx *ectx, action *group, text *pt, bool backward = false) {
      hnode    n = pt;
      helement p;
      int      idx = 0;
      bookmark bm;
      while (n != ectx->root()) {
        p   = n->parent;
        ASSERT(p);
        idx = n->node_index;
        if (backward) {
          if (node* pn = n->prev_node())
            bm = pn->end_pos();
          else
            bm = p->start_pos();
        }
        else {
          if (node* pn = n->next_node())
            bm = pn->start_pos();
          else
            bm = p->start_pos();
        }
        delete_node::exec(v, ectx, group, n);
        if (!is_empty_element(p)) break;
        if (p->is_block_element(v)) {
          p->append(new text(WCHARS("")));
          p->check_layout(v);
          return p->start_caret_pos(v);
        }
        if( p->is_floats_container(v) ) // BFC
          return p->start_caret_pos(v);
        n = p;
      }
      if(bm.node->is_text())
        nbspify_text::exec(v, ectx, group, bm.node.ptr_of<text>());
      return bm;
    }

    bookmark remove_char_forward::exec(view &v, editing_ctx *ectx,
                                       action *group, text *pt, int start,
                                       int end) {
      if (start < 0 || start >= end || end > pt->chars.size())
        return bookmark();

      /*if( start == 0 && end >= pt->chars.size() )
      {
          hnode n = pt;
          helement p;
          int idx = start;
          bookmark bm;
          while(n != ectx->root()) {
            p = n->parent;
            idx = n->node_index;
            bm = delete_node::exec(v,ectx,group,n);
            if(!p || !is_empty_element(p))
              break;
            if(p->is_block_element(v)) {
              p->check_layout(v);
              return p->start_caret_pos(v);
            }
            n = p;
          }
          return bm;
      }*/

      if (start == 0 && end >= pt->chars.size())
        return last_char_removed(v, ectx, group, pt);

      handle<remove_char_forward> op = group->add(new remove_char_forward());

      op->nbsp_injection = false;

      if (pt->get_style(v)->collapse_ws())
        for (; end < pt->chars.size(); ++end) {
          if (!is_space(pt->chars[end])) break;
          op->nbsp_injection = true;
        }

      op->n     = pt;
      op->chars = pt->chars(start, end);
      pt->chars.remove(start, end - start);
      op->pos = start;
      if (op->nbsp_injection) pt->chars.insert(op->pos, NBSP_CHAR);

      ectx->node_shrink(op->n, start, end - start);
      //v.add_to_update(pt->parent, CHANGES_MODEL);
      update(v, op->n);
      if (start >= op->n->chars.size()) return op->n->end_caret_pos(v);
      return bookmark(op->n, start, false);
    }
    bool remove_char_forward::append(view &v, editing_ctx *ectx,
                                         action *group, text *pt, int start,
                                         int end, bookmark& bm) {
      if (start < 0 || start >= end || end > pt->chars.size()) 
        return false;
      if (n != pt || pos != start) 
        return false;

      if (start == 0 && end >= pt->chars.size())
        //return last_char_removed(v, ectx, group, pt);
        return false;

      ASSERT(!nbsp_injection);

      if (pt->get_style(v)->collapse_ws())
        for (; end < pt->chars.size(); ++end) {
          if (!is_space(pt->chars[end])) break;
          nbsp_injection = true;
        }
      chars.push(pt->chars(start, end));
      pt->chars.remove(start, end - start);
      if (nbsp_injection) pt->chars.insert(pos, NBSP_CHAR);
      ectx->node_shrink(n, start, end - start);
      //v.add_to_update(pt->parent, CHANGES_MODEL);
      update(v, n);
      if (start >= n->chars.size()) 
        bm = n->end_caret_pos(v);
      else
        bm = bookmark(n, start, false);
      return true;
    }
    void remove_char_forward::undo(view &v, editing_ctx *ectx) {
      if (nbsp_injection) n->chars.remove(pos);
      n->chars.insert(pos, chars());
      ectx->node_expand(n, pos, chars.size());
      v.add_to_update(n->parent, CHANGES_MODEL);
    }
    void remove_char_forward::redo(view &v, editing_ctx *ectx) {
      n->chars.remove(pos, chars.size());
      if (nbsp_injection) n->chars.insert(pos, NBSP_CHAR);
      ectx->node_shrink(n, pos, chars.size());
      v.add_to_update(n->parent, CHANGES_MODEL);
    }

    bookmark remove_char_backward::exec(view &v, editing_ctx *ectx,
                                        action *group, text *pt, int start,
                                        int end) {
      if (start < 0 || start >= end || end > pt->chars.size())
        return bookmark();

      if (start == 0 && end >= pt->chars.size())
        return last_char_removed(v, ectx, group, pt, true);

      handle<remove_char_backward> op = group->add(new remove_char_backward());

      if (pt->get_style(v)->collapse_ws())
        for (int pos = start - 1; pos >= 0; --pos) {
          if (!is_space(pt->chars[pos])) break;
          start              = pos;
          op->nbsp_injection = true;
        }

      op->n     = pt;
      op->chars = pt->chars(start, end);
      pt->chars.remove(start, end - start);
      op->pos = start;
      if (op->nbsp_injection) pt->chars.insert(op->pos, NBSP_CHAR);

      ectx->node_shrink(op->n, start, end - start);

      //v.add_to_update(pt->parent, CHANGES_MODEL);
      update(v, op->n);

      if (start >= op->n->chars.size()) return op->n->end_caret_pos(v);
      return bookmark(op->n, start, op->nbsp_injection);
    }

    bool remove_char_backward::append(view &v, editing_ctx *ectx,
                                      action *group, text *pt, int start, int end, bookmark& bm) {
      if (start < 0 || start >= end || end > pt->chars.size())
        return false;

      if (n != pt || pos != end) 
        return false;

      if (start == 0 && end >= pt->chars.size()) {
        //bm = last_char_removed(v, ectx, group, pt, true);
        return false;
      }

      if (pt->get_style(v)->collapse_ws())
        for (int t = start - 1; t >= 0; --t) {
          if (!is_space(pt->chars[t])) break;
          start          = t;
          nbsp_injection = true;
        }

      pos = start;
      chars.insert(0, pt->chars(start, end));
      pt->chars.remove(start, end - start);
      if (nbsp_injection) pt->chars.insert(pos, NBSP_CHAR);
      ectx->node_shrink(n, start, end - start);
      //v.add_to_update(pt->parent, CHANGES_MODEL);
      update(v, n);

      if (start >= n->chars.size()) 
        bm = n->end_caret_pos(v);
      else
        bm = bookmark(n, start, nbsp_injection);
      return true;
    }
    void remove_char_backward::undo(view &v, editing_ctx *ectx) {
      if (nbsp_injection) n->chars.remove(pos);
      n->chars.insert(pos, chars());
      ectx->node_expand(n, pos, chars.size());
      v.add_to_update(n->parent, CHANGES_MODEL);
    }
    void remove_char_backward::redo(view &v, editing_ctx *ectx) {
      n->chars.remove(pos, chars.size());
      if (nbsp_injection) n->chars.insert(pos, NBSP_CHAR);
      ectx->node_shrink(n, pos, chars.size());
      v.add_to_update(n->parent, CHANGES_MODEL);
    }

    slice<hnode> insert_html_at(view &v, editing_ctx *ectx, action *group,
                                bookmark &at, const string &source_url,
                                bytes html) {
      ASSERT(at.node->is_element());

      istream m(html, source_url, false);
      m.set_utf8_encoding();

      helement target            = at.node.ptr_of<element>();
      int      node_count_before = target->nodes.size();
      html::insert_html(v, m, target->doc(), target, SE_INSERT,
                        at.linear_pos());
      target               = at.node.ptr_of<element>();
      int node_count_after = target->nodes.size();
      int n_nodes_inserted = node_count_after - node_count_before;
      if (n_nodes_inserted <= 0) return slice<hnode>();
      slice<hnode> nodes_inserted =
          target->nodes(at.linear_pos(), at.linear_pos() + n_nodes_inserted);
      if (nodes_inserted.length == 0) return slice<hnode>();

      insert_nodes *op = insert_nodes::exec_postfactum(
          v, ectx, group, target, at.linear_pos(), nodes_inserted);

      at = op->nodes.last()->end_caret_pos(v);
      return op->nodes();
    }
    
    transact_ctx::transact_ctx(element *el, wchars name) {
      rt = el;
      pv = el->pview();
      if (!pv) return;
      handle<ctl> p = el->get_behavior(html::CTL_RICHTEXT);
      if (!p) p = el->get_behavior(html::CTL_PLAINTEXT);

      if (!p) return;

      rt_ctl = p.ptr_of<richtext_ctl>();
      if (!rt_ctl) return;
      act                    = new html::behavior::range_action(rt_ctl, name);
      rt_ctl->transact_group = act;
    }

    void transact_ctx::commit() {
      if (!act) return;
      rt_ctl->push(*pv, act);
      pv->commit_update();
      act                    = nullptr;
      rt_ctl->transact_group = nullptr;
    }

    void transact_ctx::rollback() {
      if (!act) return;
      act->undo(*pv, rt_ctl);
      act                    = 0;
      rt_ctl->transact_group = nullptr;
    }


    bool transact_ctx::merge(bytes html, morph_options* pmr)
    {
      string url;
      if (rt->first_element())
        url = rt->first_element()->doc()->uri().src;
      return rt_ctl->merge_html(*pv, rt, url, html, "utf-8", pmr);
    }

    bool transact_ctx::remove_attr(element *on, string name) {
      return change_attr::del(*pv, rt_ctl, act, on, name);
    }
    bool transact_ctx::set_attr(element *on, string name, ustring val) {
      return change_attr::set(*pv, rt_ctl, act, on, name, val);
    }
    bool transact_ctx::set_tag(element *on, string tn) {
      return morph_element::exec(*pv, rt_ctl, act, on, tag::symbol(tn));
    }

    bookmark transact_ctx::remove_selection() {
      // return morph_element::exec(*pv,rt_ctl,act,on,tag::symbol(tn));
      bookmark bm = rt_ctl->caret;
      rt_ctl->show_caret(*pv, false);
      return bm;
    }
    bookmark transact_ctx::delete_selection() {
      return rt_ctl->delete_range_in(*pv, act, rt_ctl->anchor, rt_ctl->caret, true);
    }

    bookmark transact_ctx::delete_range(bookmark start, bookmark end) {
      rt_ctl->show_caret(*pv, false);
      if (start != end) {
        if (start > end) swap(start, end);
        start = end = remove_range(*pv, rt_ctl, act, start, end, true, true);
      }
      return start;
    }

    bool transact_ctx::delete_node(node* pn) {
      rt_ctl->show_caret(*pv, false);
      remove_range(*pv, rt_ctl, act, pn->start_pos(), pn->end_pos(), true,true);
      return true;
    }

    bool transact_ctx::split(bookmark &bm, element *until) {
      if (!bm.valid()) return false;
      bool     created = false;
      bookmark dummy;
      split_at(*pv, rt_ctl, act, bm, until, true, created, dummy);
      // rt_ctl->select(*pv, bm);
      return true;
      // split_at
    }

    bool transact_ctx::insert_node(bookmark &bm, node* pn) {
      if (!bm.valid()) return false;
      if (!bm.node->is_element()) return false;
      return insert_node::exec(*pv, rt_ctl, act, bm.node.ptr_of<element>(), bm.linear_pos(), pn);
    }

    bool transact_ctx::wrap(bookmark &start, bookmark &end, element *into) {
      bool r = html::behavior::wrap_into(*pv, rt_ctl, act, start, end, into);
      // rt_ctl->select(*pv, end,start);
      return r;
    }

    pair<bookmark,bookmark> transact_ctx::unwrap(element *that) {
      return unwrap_element::exec(*pv, rt_ctl, act, that);
    }

    slice<hnode> transact_ctx::insert_html(bookmark &at, bytes html) {

      if (at.valid())
        remove_selection();
      else {
        at = delete_selection();
        if (!at.valid()) return slice<hnode>();
      }

      helement el;
      int      pos;

      if (at.node->is_element()) {
        if (at.node.ptr_of<element>()->is_atomic_box()) {
          el  = at.node.ptr_of<element>();
          pos = at.linear_pos();
          at  = el->this_pos(pos != 0);
        }
      } else {
        el  = at.node->parent;
        pos = 0;
        if (at.at_start())
          pos = at.node->node_index;
        else if (at.at_end())
          pos = at.node->node_index + 1;
        else {
          split_node::exec(*pv, rt_ctl, act, at.node, at.linear_pos());
          pos = at.node->node_index + 1;
        }
        at = bookmark(el, pos, false);
      }
      slice<hnode> nodes = insert_html_at(*pv, rt_ctl, act, at, rt_ctl->root()->doc_url(), html);
      return nodes;
    }

    bool transact_ctx::insert_text(bookmark &at, wchars text) {
      rt_ctl->clear_comp_chars(*pv);
      return insert_text::exec(*pv, rt_ctl, act, at, text);
    }
    bool transact_ctx::set_text(node *pn, wchars chars) {
      rt_ctl->clear_comp_chars(*pv);

      if (pn->is_element()) {
        element* el = pn->cast<element>();
        delete_nodes_range::exec(*pv, rt_ctl, act, el, 0, el->nodes.size());
        if (chars.length)
          insert_node::exec(*pv, rt_ctl, act, el, 0, new html::text(chars));
      }
      else if (pn->is_text()) {
        text* pt = pn->cast<text>();
        replace_text::exec(*pv, rt_ctl, act, pt, chars);
      }
      else
        return false;
      return true;
    }


  } // namespace behavior
} // namespace html
