//|
//|
//| Copyright (c) 2001-2010
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| abstract theme manager
//|
//|

#ifndef __gool_theme_h__
#define __gool_theme_h__

#include "gool-geometry.h"
#include "gool-image.h"

#ifdef THEMES_SUPPORT

enum SHAPE_EXPANSION_MODE {
  SHAPE_NOT_EXPANDABLE = 0,
  SHAPE_HORIZONTAL     = 1, // stretch vertically but tile horizontally
  SHAPE_VERTICAL       = 2, // stretch horizontally but tile vertically
  SHAPE_EXPAND         = 3, // tile horizontally and vertically
  SHAPE_STRETCH_BOTH   = 4  // stretch horizontally and vertically
};

namespace gool {
  using namespace tool;

  // class theme_image;

  enum THEME_GROUPS {
    THEME_GROUP_BUTTON, // 0
    THEME_GROUP_EDIT,
    THEME_GROUP_SCROLLBAR,
    THEME_GROUP_TREEVIEW,
    THEME_GROUP_COMBOBOX,
    THEME_GROUP_TABS,
    THEME_GROUP_SPIN,
    THEME_GROUP_PROGRESS,
    THEME_GROUP_TRACKBAR,
    THEME_GROUP_REBAR,
    THEME_GROUP_TOOLBAR,
    THEME_GROUP_HEADER,   // 11
    THEME_GROUP_TOOLTIP,  // 12
    THEME_GROUP_LISTVIEW, // 13
    THEME_GROUPS_SIZE,
  };

  class theme : public resource {
    array<handle<image>> cache;

  public:
    enum CURRENT_OP { CURRENT = 0, RESET, DESTROY, PEEK };

    theme() {}
    virtual ~theme() {}
    // size   get_image_size(const string& name) = 0;
    static theme *current(CURRENT_OP op = CURRENT);
    static void   on_theme_changed() {
      critical_section _(lock());
      current(theme::RESET);
    }

    static mutex &lock() {
      static mutex _lock;
      return _lock;
    }

    static void shutdown() { current(theme::DESTROY); }

    enum scrollbar_part {
      SCROLLBAR_BASE = 0,
      SCROLLBAR_PLUS,
      SCROLLBAR_MINUS,
      SCROLLBAR_SLIDER,
      SCROLLBAR_PAGE_MINUS,
      SCROLLBAR_PAGE_PLUS,
      SCROLLBAR_CORNER, // sometimes it is known as a "gripper"
    };

    virtual void draw_h_scrollbar(graphics *sf, uint pt, uint st, rect rc) = 0;
    virtual void draw_v_scrollbar(graphics *sf, uint pt, uint st, rect rc) = 0;

    // metrics
    virtual int scrollbar_width();
    virtual int scrollbar_height();
    virtual int small_icon_width();
    virtual int small_icon_height();
    virtual int border_width();
    virtual int border_3d_width();

    virtual bool is_classic_theme() const {
      return false;
    } // if true - theme emulation
    virtual bool is_ux_theme() const {
      return !is_classic_theme();
    } // if true - uxtheme

    // image* get_image(const string& name);
    image *get_image(chars name);
    uint   get_image_id(chars name);

    image *get_image(uint id);

    virtual image *get_image_impl(uint id) = 0;

    static bool processing; // true if inside theme call
  };

  class theme_image : public image {
    image *get_image_impl() const {
      return theme::current()->get_image_impl(image_id);
    }

  public:
    uint image_id;

    theme_image(uint ni) : image_id(ni) {}
    virtual ~theme_image() {}

    virtual handle<bitmap> get_bitmap(graphics *sf, size sz) override;

    virtual bool is_valid() const override { return true; }

    virtual void draw(graphics *gfx, rectf dst, rect src, byte opacity) override;
    virtual void draw(graphics *gfx, rect dst, rect src, byte opacity) override;

    virtual size dim(const rect &for_dst) const override {
      return for_dst.size();
    }
    virtual size dim() const override;
    virtual bool is_transparent() const override;

    virtual void drop_cache() override;
  };

  inline image *theme::get_image(uint id) {
    critical_section cs(gool::lock);
    if (id >= cache.length())
      cache.length(id + 1);
    else if (cache[id])
      return cache[id];
    return (cache[id] = new theme_image(id));
  }

  inline handle<bitmap> theme_image::get_bitmap(graphics *sf, size sz) {
    handle<image> pim = get_image_impl();
    return pim ? pim->get_bitmap(sf, sz) : 0;
  }

  inline void theme_image::draw(graphics *gfx, rectf dst, rect src,
    byte opacity) {
    handle<image> pim = get_image_impl();
    if (pim) pim->draw(gfx, dst, src, opacity);
  }

  inline void theme_image::draw(graphics *gfx, rect dst, rect src,
    byte opacity) {
    handle<image> pim = get_image_impl();
    if (pim) pim->draw(gfx, dst, src, opacity);
  }


  inline size theme_image::dim() const {
    image *pim = get_image_impl();
    return pim ? pim->dim() : size();
  }
  inline bool theme_image::is_transparent() const {
    image *pim = get_image_impl();
    return pim ? pim->is_transparent() : false;
  }
  inline void theme_image::drop_cache() {
    image *pim = get_image_impl();
    if (pim) pim->drop_cache();
  }

} // namespace gool
#endif
#endif
