//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| DIB I/O - read/write JPEG,PNG, GIF
//|
//|

#include "gool.h"

extern int check_png_version();

#ifdef PNG_SUPPORT
#ifndef LINUX
#include "external/zlib/zlib.h"
#include "external/zlib/zlib.h"
#endif
#include "external/png/png.h"
#include "external/png/pngstruct.h"
#include "external/png/pnginfo.h"
#endif

#include "gool-exif.h"

/*namespace gool
{
  struct wic_reader: image_reader
  {
    wic_reader(bytes data): image_reader(data) {}
    virtual image* read();
    //virtual size_t write_impl(image* img, int compression_level);
    //byte* read_packed_dib();
  };
}*/

#if defined(GIF_SUPPORT)
extern bool gif_decode(const unsigned char *gifdata, size_t gifdata_length,
                       unsigned char *&argb, bool &hasalpha, int &width,
                       int &height);
#endif

namespace gool {
  // using namespace pnglib;

  const char *mime_type_of(image::PACKAGING ip) {
    switch (ip) {
    case image::PNG: return "image/png";
    case image::JPG: return "image/jpeg";
    case image::GIF: return "image/gif";
    case image::BMP: return "image/bmp";
    case image::SVG: return "image/svg";
    case image::WEBP: return "image/webp";
    case image::ICO:
      return "image/vnd.microsoft.icon"; // IANA tells us
    }
    return 0;
  }

  const char *file_ext_of(image::PACKAGING ip) {
    switch (ip) {
    case image::PNG: return ".png";
    case image::JPG: return ".jpg";
    case image::GIF: return ".gif";
    case image::BMP: return ".bmp";
    case image::SVG: return ".svg";
    case image::ICO: return ".ico";
    case image::WEBP: return ".webp";
    }
    return 0;
  }

    /*inline void swap_rb(argb *bits, dword count)
    {
      while(count--)
      {
        tool::swap(bits->blue,bits->red);
        bits++;
      }
    }
    inline void swap_rb(rgb *bits, dword count)
    {
      while(count--)
      {
        tool::swap(bits->blue,bits->red);
        bits++;
      }
    }

    inline void rgba2bgr(const argb *in, rgb* out, dword count)
    {
      while(count--)
      {
        out->red = in->blue;
        out->green = in->green;
        out->blue = in->red;
        ++out; ++in;
      }
    }
    */

#ifdef PNG_SUPPORT

#ifndef MONOLITHIC
  #pragma comment(lib, "png")
  #pragma comment(lib, "zlib")
#endif

  struct png_reader : public image_reader {
    png_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    virtual handle<image> read(html::document *pd) override;
    // virtual size_t write_impl(image* img, int compression_level);
  };

  struct png_writer : public image_writer {
    png_writer(tool::array<byte> &buf) : image_writer(buf) {}
    virtual size_t write(image *img, int compression_level);
  };

  static void png_read_data(png_structp png_ptr, png_bytep data,
                            png_size_t length) {
    png_reader *pio = (png_reader *)png_get_io_ptr(png_ptr);
    if (!pio->read_bytes(data, length)) png_error(png_ptr, "PNG read error");
  }

  static void png_write_data(png_structp png_ptr, png_bytep data,
                             png_size_t length) {
    png_writer *pio = (png_writer *)png_get_io_ptr(png_ptr);
    pio->write_bytes(data, length);
  }

  static void png_flush_data(png_structp png_ptr) {
    ; // does nothing
  }
#pragma warning(push)
#pragma warning(disable : 4611) // warning C4611: interaction between '_setjmp'
                                // and C++ object destruction is non-portable

#define pnglib_check_sig(sig, n) !png_sig_cmp((sig), 0, (n))

  handle<image> png_reader::read(html::document *pd) {
    if (data.length <= 8) return nullptr;

    if (!pnglib_check_sig(const_cast<byte *>(data.start), 8)) return nullptr;

    png_structp png_ptr =
        png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
    if (!png_ptr) // out of memory?
      return nullptr;
    png_infop info_ptr = png_create_info_struct(png_ptr);
    if (!info_ptr) { // out of memory?
      png_destroy_read_struct(&png_ptr, NULL, NULL);
      return nullptr;
    }
    // setup i/o routine
    png_set_read_fn(png_ptr, this, png_read_data);

    handle<image> img;

    // init jmpbuf
    if (setjmp(png_jmpbuf(png_ptr))) { // error occurred
      png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
      return nullptr;
    }
    png_read_info(png_ptr, info_ptr);
    png_set_bgr(png_ptr);
    // png_read_png (png_ptr,info_ptr, PNG_TRANSFORM_BGR, NULL);
    png_uint_32 width, height;
    int         bit_depth, color_type, interlace_type;
    png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type,
                 &interlace_type, NULL, NULL);

    // if (interlace_type!=PNG_INTERLACE_NONE)
    //{
    //  png_destroy_read_struct(&png_ptr,&info_ptr,NULL);
    //  return nullptr;
    //}
    // configure transformations, we always want RGB data in the end
    if (color_type == PNG_COLOR_TYPE_GRAY ||
        color_type == PNG_COLOR_TYPE_GRAY_ALPHA)
      png_set_gray_to_rgb(png_ptr);
    if (color_type == PNG_COLOR_TYPE_PALETTE) png_set_palette_to_rgb(png_ptr);
    if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8)
      png_set_expand_gray_1_2_4_to_8(png_ptr);
    if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
      png_set_tRNS_to_alpha(png_ptr);
    if (bit_depth == 16) png_set_strip_16(png_ptr);
    if (bit_depth < 8) png_set_packing(png_ptr);

    //#ifdef PNG_FLOATING_POINT_SUPPORTED
    // png_color_16  my_background={0,0,0,0,0};
    // png_color_16p dib_background;
    // if (png_get_bKGD(png_ptr, info_ptr, &dib_background))
    //  png_set_background(png_ptr, dib_background,
    //   PNG_BACKGROUND_GAMMA_FILE, 1, 1.0);
    // else
    //  png_set_background(png_ptr, &my_background,
    //   PNG_BACKGROUND_GAMMA_SCREEN, 0, 1.0);

    double screen_gamma = 2.2; // typical value
    double gamma;
    if (png_get_gAMA(png_ptr, info_ptr, &gamma))
      png_set_gamma(png_ptr, screen_gamma, gamma);
    else
      png_set_gamma(png_ptr, screen_gamma, 0.45455);
    //#endif

    // update info after applying transformations
    png_read_update_info(png_ptr, info_ptr);
#ifdef HAS_APNG
    // aPNG ++
    png_uint_32 num_frames = 0;
    png_uint_32 num_plays  = 0;
    png_uint_32 is_animated =
        png_get_acTL(png_ptr, info_ptr, &num_frames, &num_plays);
// aPNG --
#endif

    size_t rowBytes = png_get_rowbytes(png_ptr, info_ptr);
    size   dim(width, height);
    uint   bpp = uint(rowBytes / width);
#ifdef HAS_APNG
    if (is_animated) {
      animated_image *pani = new animated_image(dim);
      pani->iterations(num_plays);
      img                                       = pani;
      animated_image::framemode prev_dispose_op = animated_image::nothing;
      for (uint n = 0; n < num_frames; ++n) {
        png_read_frame_head(png_ptr, info_ptr);
        handle<bitmap> frame;

        point frame_pos =
            point(info_ptr->next_frame_x_offset, info_ptr->next_frame_y_offset);
        size frame_dim =
            size(info_ptr->next_frame_width, info_ptr->next_frame_height);
        uint frame_duration = 1; // milliseconds

        if (frame_dim.empty()) {
          // return nullptr; // bad aPNG frame
          frame_dim = dim;
        }

        if (info_ptr->next_frame_delay_num) {
          frame_duration = info_ptr->next_frame_delay_num * 1000;
          if (info_ptr->next_frame_delay_den == 0)
            frame_duration /= 100;
          else
            frame_duration /= info_ptr->next_frame_delay_den;
          if (!frame_duration) frame_duration = 1;
        }

        // int   rowBytes = png_get_rowbytes(png_ptr, info_ptr);
        // int   bpp = rowBytes / frame_dim.x;

        animated_image::framemode dispose_op;
        switch (info_ptr->next_frame_dispose_op) {
        default:
        case 0 /*APNG_DISPOSE_OP_NONE*/:
          dispose_op = animated_image::nodispose;
          break;
        case 1 /*APNG_DISPOSE_OP_BACKGROUND*/:
          dispose_op = animated_image::clear;
          break;
        case 2 /*APNG_DISPOSE_OP_PREVIOUS*/:
          dispose_op = animated_image::restore;
          break;
        }

        animated_image::blendmode blend_op;

        switch (info_ptr->next_frame_blend_op) {
        default:
        case 0 /* APNG_BLEND_OP_SOURCE */:
          blend_op = animated_image::copy;
          break;
        case 1 /* APNG_BLEND_OP_OVER */:
          blend_op = animated_image::blend;
          break;
        }

        switch (bpp) {
        case 4: {
          frame = new bitmap(frame_dim, true);
          array<png_byte *> rowPtrs;
          rowPtrs.size(frame_dim.y);
          for (int i = 0; i < frame_dim.y; i++)
            rowPtrs[i] = (byte *)(*frame)[i].start;
          png_read_image(png_ptr, &rowPtrs[0]);
          frame->premultiply();
        } break;
        case 3: {
          frame = new bitmap(frame_dim, false);
          array<rgb> buffer(frame_dim.x);
          for (int i = 0; i < dim.y; i++) {
            png_read_row(png_ptr, (png_bytep)&buffer[0], NULL);
            rgb * src = &buffer[0];
            argb *dst = const_cast<argb *>((*frame)[i].start);
            for (int c = 0; c < frame_dim.x; ++c)
              *dst++ = argb(*src++);
          }
        } break;
        default: assert(0);
        }
        if (frame)
          pani->add(frame, frame_duration, prev_dispose_op, frame_pos,
                    blend_op);
        prev_dispose_op = dispose_op;
      }
    } else
#endif // HAS_APNG
    // normal png
    {
      handle<bitmap> bmp;
      switch (bpp) {
      case 4: {
        img = bmp = new bitmap(dim, true);
        array<png_byte *> rowPtrs;
        rowPtrs.size(dim.y);
        for (int i = 0; i < dim.y; i++)
          rowPtrs[i] = (byte *)(*bmp)[i].start;
        png_read_image(png_ptr, &rowPtrs[0]);
        bmp->premultiply();
      } break;
      case 3: {
        img = bmp = new bitmap(dim, false);
        if (interlace_type != PNG_INTERLACE_NONE) {
          array<rgb> buffer(dim.x*dim.y);
          array<png_byte *> rowPtrs;
          rowPtrs.size(dim.y);
          for (int i = 0; i < dim.y; ++i)
            rowPtrs[i] = (byte *)&buffer[i*dim.x];
          png_read_image(png_ptr, &rowPtrs[0]);
          auto dst = bmp->_pixels.target();
          auto src = buffer();
          for (int n = 0; n < buffer.size(); ++n)
            dst[n] = argb(src[n]);
        }
        else {
          array<rgb> buffer(dim.x);
          for (int i = 0; i < dim.y; i++) {
            png_read_row(png_ptr, (png_bytep)&buffer[0], NULL);
            rgb * src = &buffer[0];
            argb *dst = bmp->target_row(i).start;
            for (int c = 0; c < dim.x; ++c)
              *dst++ = argb(*src++);
          }
        }
      } break;
      default: assert(0);
      }
    }
    png_read_end(png_ptr, info_ptr);
    png_destroy_read_struct(&png_ptr, &info_ptr, NULL);

    return img;
  }

#pragma warning(pop)

  size_t png_writer::write(image *img, int compression_level) {
    if (!img->is_valid()) return false;
    handle<bitmap> pdib = img->get_bitmap(0, size());

    png_structp png     = 0;
    png_infop   pngInfo = 0;

    size dim(pdib->dim());

    if (!pdib->is_valid()) return false;

    png = png_create_write_struct(PNG_LIBPNG_VER_STRING, 0, 0, 0);
    if (!png) { return false; }

    pngInfo = png_create_info_struct(png);
    if (!pngInfo) { png_destroy_write_struct(&png, &pngInfo); }

    // if (setjmp(png->jmpbuf))
    //{
    //  png_destroy_write_struct(&png, &pngInfo);
    //  return false;
    //}

    // setup i/o routine
    png_set_write_fn(png, this, png_write_data, png_flush_data);

    // dword fmt = pdib->is_transparent()? PNG_COLOR_TYPE_RGB_ALPHA :
    // PNG_COLOR_TYPE_RGB;
    dword fmt = PNG_COLOR_TYPE_RGB_ALPHA;

    png_set_IHDR(png, pngInfo, dim.x, dim.y, 8, fmt, PNG_INTERLACE_NONE,
                 PNG_COMPRESSION_TYPE_DEFAULT, PNG_FILTER_TYPE_DEFAULT);

    png_write_info(png, pngInfo);
    png_set_bgr(png);
    // png_write_png (png, pngInfo, PNG_TRANSFORM_BGR, NULL);

    array<argb> pixmap;
    int nrows = pdib->dim().y;

    for (int n = 0; n < nrows; ++n)
      pixmap.push( (*pdib)[n] );

    for (uint n = 0; n < pixmap.length(); ++n)
      pixmap[n] = pixmap[n].demultiply();

    array<png_byte *> rowPtrs;
    rowPtrs.size(dim.y);
    for (int i = 0; i < dim.y; i++)
      rowPtrs[i] = //(byte *)(*pdib)[i].start;
          (byte *)&pixmap[i * dim.x];
    png_write_image(png, &rowPtrs[0]);

    png_write_end(png, pngInfo);

    png_destroy_write_struct(&png, &pngInfo);

    return out.size();
  }
#endif // PNG_SUPPORT

#ifdef JPG_SUPPORT
#include "external/jpeg/jpeglib.h"

#ifndef MONOLITHIC
  #pragma comment(lib, "jpeg")
#endif

  struct jpg_reader : public image_reader {
    jpg_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    virtual handle<image> read(html::document *pd);
    // virtual size_t write_impl(image* img, int compression_level);
  };

  typedef std::exception jpg_reader_err;

  METHODDEF(void) __my_error_exit(j_common_ptr cinfo) {
    throw jpg_reader_err();
  }

  handle<image> jpg_reader::read(html::document *pd) {

    // load a JPEG compressed image

    if (data[0] != 0xFF || data[1] != 0xD8)
      return handle<image>(); // not a JPEG signature

    struct jpeg_decompress_struct cinfo;
    struct jpeg_error_mgr         __jerr;

    // int    i, j, k, offset;

    handle<image> bmp;

    JSAMPROW row_pointer[1];
    // bool isgray;

    /* Step 1: allocate and initialize JPEG decompression object */
    cinfo.err         = jpeg_std_error(&__jerr);
    __jerr.error_exit = __my_error_exit;

    /* Now we can initialize the JPEG decompression object. */
    jpeg_create_decompress(&cinfo);

    try {
      /* Step 2: specify data source (eg, a file) */

      jpeg_mem_src(&cinfo, (byte *)data.start, data.size());

      /* Step 3: read file parameters with jpeg_read_header() */

      (void)jpeg_read_header(&cinfo, (unsigned char)TRUE);

      size dim(cinfo.image_width, cinfo.image_height);

      cinfo.out_color_space = JCS_RGB;

      /*if(cinfo.num_components == 3 && cinfo.out_color_space == JCS_RGB)
      {
          isgray = false;
      }
      /*else if(cinfo.num_components == 1 && cinfo.out_color_space ==
      JCS_GRAYSCALE)
      {
          isgray = true;
      }
      else
      {
        jpeg_destroy_decompress(&cinfo);
        return 0;
      }*/

      bmp = new bitmap(dim, false);

      /* Step 4: set parameters for decompression */

      /* Step 5: Start decompressor */

      (void)jpeg_start_decompress(&cinfo);

      /* Step 6: while (scan lines remain to be read) */
      /*           jpeg_read_scanlines(...); */

      /*if(isgray) {
          array<byte> row(cinfo.image_width);
          row_pointer[0] = &row[0];
          while (cinfo.output_scanline < cinfo.output_height)
          {
            int sl = cinfo.output_scanline;
            (void) jpeg_read_scanlines(&cinfo, row_pointer , 1);
            //if(cinfo.output_scanline >= cinfo.output_height) break;
            plain_pixel_type* pp = (plain_pixel_type*)hd->row_ptr(sl);
            for(int i = 0; i < int(cinfo.image_width); i++, pp++)
            {
              PixelProxy<plain_pixel_type> proxy(pp);
              proxy->blue = proxy->green = proxy->red = row[i];
            }
          }
      }
      else */
      {
        array<rgb> row(cinfo.image_width);
        while (cinfo.output_scanline < cinfo.output_height) {
          row_pointer[0] = (byte *)&row[0];

          argb *dst =
              bmp.ptr_of<bitmap>()->target_row(cinfo.output_scanline).start;

          // plain_pixel_type *p = (plain_pixel_type
          // *)hd->row_ptr(cinfo.output_scanline);
          (void)jpeg_read_scanlines(&cinfo, row_pointer, 1);
          rgb *prow = &row[0];
          for (unsigned i = 0; i < cinfo.image_width; i++, dst++, prow++)
            dst->set(prow->blue, prow->green, prow->red, 255);
        }
      }

      /* Step 7: Finish decompression */
      (void)jpeg_finish_decompress(&cinfo);

    } catch (jpg_reader_err) { bmp = nullptr; }

    /* Step 8: Release JPEG decompression object */
    jpeg_destroy_decompress(&cinfo);

    return bmp.ptr();
  }

  typedef struct my_jpeg_out_mgr {
    struct jpeg_destination_mgr pub;
    unsigned char               buffer[256];
    array<byte> *               pout;
  } my_jpeg_out_mgr;

  /* this routine is called to initiate the jpeg writing process */

  void my_init_destination(j_compress_ptr cinfo) {
    struct my_jpeg_out_mgr *dest = (struct my_jpeg_out_mgr *)cinfo->dest;
    /* the storage here is staticly allocated */
    dest->pub.next_output_byte = dest->buffer;
    memset(dest->buffer, 0, 256);
    dest->pub.free_in_buffer = 256;
  }

  /* this routine is called each time the buffer gets full */

  boolean my_empty_output_buffer(j_compress_ptr cinfo) {
    struct my_jpeg_out_mgr *dest = (struct my_jpeg_out_mgr *)cinfo->dest;
    dest->pout->push(dest->buffer, 256);
    memset(dest->buffer, 0, 256);
    dest->pub.next_output_byte = dest->buffer;
    dest->pub.free_in_buffer   = 256;
    return TRUE;
  }

  /* and this routine gets called when the writing process is finished */

  void my_term_destination(j_compress_ptr cinfo) {
    struct my_jpeg_out_mgr *dest = (struct my_jpeg_out_mgr *)cinfo->dest;
    int                     n    = 256 - int(dest->pub.free_in_buffer);
    dest->pout->push(dest->buffer, n);
  }

  struct jpg_writer : public image_writer {
    jpg_writer(tool::array<byte> &buf) : image_writer(buf) {}
    virtual size_t write(image *img, int compression_level);
  };

  void rgba2bgr(slice<argb> in, tslice<rgb> out) {
    while (in.length) {
      argb c = in++;
      c      = c.demultiply();
      rgb d;
      d.red   = c.blue;
      d.green = c.green;
      d.blue  = c.red;
      out     = out.copy(d);
    }
  }

  size_t jpg_writer::write(image *img, int quality) {
    if (!img->is_valid()) return false;
    handle<bitmap> pdib = img->get_bitmap(0, size());

    size dim = pdib->dimension();

    array<rgb> rowbuf;
    rowbuf.size(dim.x);

    struct my_jpeg_out_mgr      dest;
    struct jpeg_compress_struct cinfo;
    struct jpeg_error_mgr       jerr;
    JSAMPROW                    row_pointer[1];

    cinfo.err = jpeg_std_error(&jerr);
    jpeg_create_compress(&cinfo);
    cinfo.dest = (struct jpeg_destination_mgr *)&dest;

    dest.pout = &out;

    dest.pub.init_destination    = my_init_destination;
    dest.pub.empty_output_buffer = my_empty_output_buffer;
    dest.pub.term_destination    = my_term_destination;

    cinfo.image_width      = dim.x;
    cinfo.image_height     = dim.y;
    cinfo.input_components = 3;
    cinfo.in_color_space   = JCS_RGB;
    jpeg_set_defaults(&cinfo);
    jpeg_set_quality(&cinfo, quality, true);
    jpeg_start_compress(&cinfo, true);

    while (cinfo.next_scanline < cinfo.image_height) {
      rgba2bgr(pdib->operator[](cinfo.next_scanline), rowbuf.target());
      row_pointer[0] = (byte *)rowbuf.head();
      jpeg_write_scanlines(&cinfo, row_pointer, 1);
    }

    jpeg_finish_compress(&cinfo);
    jpeg_destroy_compress(&cinfo);

    return out.size();
  }

#endif // JPG_SUPPORT

#ifdef GIF_SUPPORT

  struct gif_reader : public image_reader {
    gif_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    virtual handle<image> read(html::document *pd) override;
  };

  bool decode_gif(const byte *data, size_t data_length,
                  handle<gool::image> &pimg /*, bool all*/);

  handle<image> gif_reader::read(html::document *pd) {
    handle<image> pimg;

    if (!decode_gif(data.start, data.length, pimg)) {
      assert(pimg == 0);
      return nullptr;
    }

    handle<animated_image> pani = pimg.ptr_of<animated_image>();
    if (pani->n_frames() == 1)
      return pani->frame_image(0);

    return pimg;
  }

#endif // GIF_SUPPORT

#ifdef WIC_SUPPORT

  struct wic_reader : public image_reader {
    wic_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    virtual handle<image> read(html::document *pd) override;
    // virtual size_t write_impl(image* img, int compression_level);
  };

  handle<image> wic_reader::read(html::document *pd) {
    HRESULT hr;

    auto wic_factory = app()->wic_factory();
    if (!wic_factory) return nullptr;

    d2d::asset<IWICStream> stream;
    hr = wic_factory->CreateStream(stream.target());
    if (FAILED(hr)) return nullptr;
    hr = stream->InitializeFromMemory((BYTE *)data.start, DWORD(data.length));
    if (FAILED(hr)) return nullptr;
    d2d::asset<IWICBitmapDecoder> decoder;
    hr = wic_factory->CreateDecoderFromStream(
        stream, 0, WICDecodeMetadataCacheOnDemand, decoder.target());
    if (FAILED(hr)) return nullptr;

    d2d::asset<IWICBitmapFrameDecode> frame;
    hr = decoder->GetFrame(0, frame.target());
    if (FAILED(hr)) return nullptr;

    d2d::asset<IWICFormatConverter> converter;

    hr = wic_factory->CreateFormatConverter(converter.target());
    if (FAILED(hr)) return nullptr;

    hr = converter->Initialize(
        frame,                         // Input bitmap to convert
        GUID_WICPixelFormat32bppPBGRA, // Destination pixel format
        WICBitmapDitherTypeNone,       // Specified dither pattern
        NULL,                          // Specify a particular palette
        0.f,                           // Alpha threshold
        WICBitmapPaletteTypeCustom);   // Palette translation type
    if (FAILED(hr)) return nullptr;

    UINT x = 0, y = 0;
    hr = converter->GetSize(&x, &y);
    if (FAILED(hr)) return nullptr;
    if( x > 24000 || y > 24000 ) return nullptr;

    // hr = m_pRT->CreateBitmapFromWicBitmap(m_pConvertedSourceBitmap, NULL,
    // &m_pD2DBitmap);
    handle<image> bmp = new bitmap(size(x, y), true);
    hr = converter->CopyPixels(0, x * sizeof(gool::argb),
                      (UINT)bmp.ptr_of<bitmap>()->_pixels.length() * sizeof(gool::argb),
                      (BYTE *)bmp.ptr_of<bitmap>()->_pixels.head());
    if (FAILED(hr)) return nullptr;
    return bmp;
  }

#endif

#ifdef WEBP_SUPPORT

  #include "external/webp/src/webp/decode.h"
  #include "external/webp/src/webp/encode.h"

#ifndef MONOLITHIC
  #pragma comment(lib, "webp")
#endif

  struct webp_reader : public image_reader {
    webp_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    virtual handle<image> read(html::document *pd) override;
  };

  struct webp_writer : public image_writer {
    webp_writer(tool::array<byte> &buf) : image_writer(buf) {}
    virtual size_t write(image *img, int compression_level);
  };

    
  handle<image> webp_reader::read(html::document *pd) {
    handle<image> pimg;

    WebPBitstreamFeatures features = { 0 };
    VP8StatusCode status = WebPGetFeatures(data.start, data.length, &features);

    if (status != VP8_STATUS_OK)
      return pimg;

    gool::size dim = { features.width,features.height };

    pimg = new gool::bitmap(dim, !!features.has_alpha, true);

    auto td = pimg.ptr_of<bitmap>()->pixels_as_target_bytes();

    WebPDecodeBGRAInto(data.start, data.length, td.start, td.length, pimg.ptr_of<bitmap>()->stride());

#ifdef USE_CGX
    if (mirrored_bitmaps()) 
      pimg.ptr_of<bitmap>()->top_to_bottom_inplace();
      // make it bottom to top
#endif

    if(features.has_alpha)
      pimg.ptr_of<bitmap>()->premultiply();
    
    return pimg;
  }

  size_t webp_writer::write(image *img, int compression_level)
  {
    if (!img->is_bitmap())
      return 0;
    bitmap* pbmp = static_cast<bitmap*>(img);
    byte *pbuf = nullptr;
    size_t nbytes = compression_level == 0 ?
      WebPEncodeLosslessBGRA(pbmp->pixels_as_bytes().start, pbmp->dim().x, pbmp->dim().y, pbmp->stride(), &pbuf) :
      WebPEncodeBGRA(pbmp->pixels_as_bytes().start, pbmp->dim().x, pbmp->dim().y, pbmp->stride(), float(limit(compression_level,0,100)), &pbuf);
    if (pbuf) {
      this->write_bytes(pbuf, nbytes);
      WebPFree(pbuf);
    }
    return nbytes;
  }

#endif

  size_t image::save(array<byte> &out, image::PACKAGING type,
                     int compression_level) {
    switch (type) {
#if defined(PNG_SUPPORT)
    case PNG: {
      png_writer pio(out);
      return pio.write(this, compression_level);
    } break;
#endif
#if defined(JPG_SUPPORT)
    case JPG: {
      jpg_writer jio(out);
      return jio.write(this, compression_level);
    } break;
#endif
#if defined(WEBP_SUPPORT)
    case WEBP: {
      webp_writer jio(out);
      return jio.write(this, compression_level);
    } break;
#endif
    default: assert(0); break;
    }
    return 0;
  }

  array<char> image::get_data_url() {
    bytes bytes = get_data();
    if (!bytes || packaging == UNKNOWN) return array<char>();
    // data:image/gif;base64,...
    array<char> out = CHARS("data:");
    out.push(chars_of(mime_type_of(packaging)));
    out.push(CHARS(";base64,"));
    base64_encode(bytes, out);
    return out;
  }

  bytes bitmap::get_data() {
    if (data.size() && packaging != UNKNOWN) return super::get_data();
#ifdef PNG_SUPPORT
    png_writer pnw(data);
    pnw.write(this, 0);
    packaging = PNG;
#endif
    return super::get_data();
  }

  handle<image> image::create(const array<byte> &data, const string &image_url,
                              html::document *pd) {
    handle<image> pimg = create(data(), image_url, pd);
    if (pimg) pimg->data.attach_data(data);
    return pimg;
  }

  struct bgra_reader : public image_reader {
    bgra_reader(bytes dt, const string &url) : image_reader(dt, url) {}
    bool read(uint &ui) {
      if (data.length < 4) return false;
      ui = 0;
      for (uint n = 0; n < 4; ++n) {
        ui <<= 8;
        ui |= data++;
      }
      return true;
    }
    virtual handle<image> read(html::document *) override {
      if (data++ != 'B') return 0;
      if (data++ != 'G') return 0;
      if (data++ != 'R') return 0;
      if (data++ != 'A') return 0;
      uint w = 0, h = 0;
      if (!read(w) || !read(h)) return 0;
      if (data.length != w * h * sizeof(argb)) return 0;
      bitmap *    pb = new bitmap(size(w, h), true);
      slice<argb> bits =
          slice<argb>((argb *)data.start, data.length / sizeof(argb));
      pb->set_bits(bits);
      pb->premultiply();
      return pb;
    }
  };

  handle<image> image::create(bytes data, const string &image_url,
                              html::document *indoc) {
    if (data.length <= 4) // it should have at least signature.
      return 0;

    handle<image> pimg;
    PACKAGING     type = UNKNOWN;

    {
      bgra_reader bgra(data, image_url);
      pimg = bgra.read(indoc);
      if (pimg) {
        type = image::BMP;
        goto SUCC;
      }
    }

    {
#if defined(GIF_SUPPORT)
      gif_reader gif(data, image_url);
      pimg = gif.read(indoc);
      if (pimg) {
        type = image::GIF;
        goto SUCC;
      }
#endif

#if defined(BMP_SUPPORT)
      gool::dib_io bmp(data.start, data.length);
      pimg = bmp.read(pv);
      if (pimg) {
        type = gool::dib::BMP;
        goto SUCC;
      }
#endif

#if defined(PNG_SUPPORT)
      png_reader png(data, image_url);
      pimg = png.read(indoc);
      if (pimg) {
        type = PNG;
        goto SUCC;
      }
#endif

#if defined(SVG_SUPPORT)
      svg_reader svg(data, image_url);
      pimg = svg.read(indoc);
      if (pimg) {
        type = SVG;
        goto SUCC;
      }
#endif

#if defined(WEBP_SUPPORT)
      webp_reader webp(data, image_url);
      pimg = webp.read(indoc);
      if (pimg) {
        type = WEBP;
        goto SUCC;
      }
#endif

#if defined(JPG_SUPPORT) && !defined(WIC_SUPPORT)
      jpg_reader jpg(data, image_url);
      pimg = jpg.read(indoc);
      if (pimg) {
        type = JPG;
        goto SUCC;
      }
#endif


#if defined(WIC_SUPPORT)
      wic_reader wic(data, image_url);
      pimg = wic.read(indoc);
      if (pimg) {
        type = UNKNOWN;
        goto SUCC;
      }
#endif

#if defined(ICO_SUPPORT)
      {
        size minsz, maxsz;
        if (ms_icon::parse_data(data, minsz, maxsz)) {
          icon *ico     = new icon();
          ico->raw_data = data;
          type          = gool::dib::ICO;
          ico->min_size = minsz;
          ico->max_size = maxsz;
          pimg          = ico;
          goto SUCC;
        }
      }
#endif


    }
    return 0;
  SUCC:
    if (pimg) {

      gool::exif exif;
      int orientation;
      if (exif.get_orientation(data, orientation)) {
        switch (orientation) { // Image orientation, start of data corresponds to
        case 0: //unspecified in EXIF data
          break;
        case 1: break;
        case 2: pimg.ptr_of<bitmap>()->flip_x(); break;
        case 3: pimg.ptr_of<bitmap>()->flip_y(); pimg.ptr_of<bitmap>()->flip_x(); break;
        case 4: pimg.ptr_of<bitmap>()->flip_y(); break;
        case 5: pimg.ptr_of<bitmap>()->rotate_90(); break;
        case 6: pimg.ptr_of<bitmap>()->rotate_90(); pimg.ptr_of<bitmap>()->flip_x(); break;
        case 7: pimg.ptr_of<bitmap>()->rotate_90(); pimg.ptr_of<bitmap>()->flip_x(); pimg.ptr_of<bitmap>()->flip_y(); break;
        case 8: pimg.ptr_of<bitmap>()->rotate_90(); pimg.ptr_of<bitmap>()->flip_y(); break;
          break;
        }
      }

      pimg->url       = image_url;
      pimg->packaging = type;
    }
    return pimg;
  }

} // namespace gool
