#include "gool.h"
#include <math.h>

tool::bytes get_resource(const wchar *path);

#define F_PI 3.14159265f
#define F_2PI 6.28318531f
#define D_PI 3.14159265358979323846


namespace gool {

  GRAPHICS_CAPS application::requested_gfx_layer =
#if defined(WINDOWS)
#if defined(USE_SKIA)
    SKIA_OPENGL_GRAPHICS;
#else
    tool::environment::get_os_version() >= tool::environment::WIN_7_1
        ? HARDWARE_GRAPHICS
        : (tool::environment::get_os_version() >= tool::environment::WIN_7
               ? WARP_GRAPHICS
               : SOFTWARE_GRAPHICS);
    //WARP_GRAPHICS;
#endif
#elif defined(OSX)
#if defined(USE_SKIA) //&& 0
      SKIA_OPENGL_GRAPHICS;
#else
      SOFTWARE_GRAPHICS;
      //SKIA_GRAPHICS;
#endif
#else
#if defined(USE_SKIA) && 0 // remove 0 to enable SKIA by default
      // SKIA_GRAPHICS;
      SKIA_OPENGL_GRAPHICS;
#else
      SOFTWARE_GRAPHICS;
#endif
#endif

  font *application::get_system_font() {
    if (!_system_font) {
      ustring name;
      int     size;
      uint    weight;
      bool    italic;
      get_system_font(name, size, weight, italic);
      _system_font = create_font(name, float(size), weight, italic);
    }
    return _system_font;
  }

  locked::counter graphics::_uid_counter;

  mutex application::guard;

  tool::bytes application::get_resource(const wchar *path) {
    return ::get_resource(path);
  }

  application::application() /*: font_family_range_map(nullptr)*/ {
    bitmap_list_head = new bitmap(size(0, 0), false, false);
  }

  void application::clear_bitmap_cache() {
    bitmap *head = bitmap_list_head;
    for (bitmap *t = head->next(); t != head; t = t->next())
      t->drop_cache();
  }

#ifdef THEMES_SUPPORT
  bool theme::processing = false; // true if inside theme call
#endif

  resolution_provider* graphics::get_resolution_provider() const
  {
    return pview ? pview : &resolution_provider::desktop();
  }

  void graphics::free_assets() {
    image_link *pl = _image_links;
    while (pl) {
      image_link *plt = pl;
      pl->detach_from_image();
      pl = pl->next_in_graphics;
      delete plt;
    }
    _image_links = nullptr;
  }

  /**
   *  expandable image sections
   *
   *    wLeft           wRight
   *
   *   +---+-----------+---+
   *   | 0 |    1      | 2 | hTop
   *   |   |           |   |
   *   +---+-----------+---+
   *   |   |           |   |
   *   | 3 |    4      | 5 |
   *   |   |           |   |
   *   +---+-----------+---+
   *   | 6 |    7      | 8 | hBottom
   *   |   |           |   |
   *   +---+-----------+---+
   *
   */

  void split_sections(const rect &r, const rect &margins, rect *nine_rects) {
    int wl = margins.left();
    int wr = margins.right();
    int ht = margins.top();
    int hb = margins.bottom();

    int width  = r.width();
    int height = r.height();

    // constraints
    if (width < (wl + wr)) {
      if (wl == 0)
        wr = width;
      else if (wr == 0)
        wl = width;
      else
        wr = width - wl;
      if (wr < 0) {
        wr = 0;
        wl = width;
      }
    }
    if (height < (ht + hb)) {
      if (ht == 0)
        hb = height;
      else if (hb == 0)
        ht = height;
      else
        hb = height - ht;
      if (hb < 0) {
        hb = 0;
        ht = height;
      }
    }

    width -= (wl + wr);  // interior width
    height -= (ht + hb); // interior height

    int right_x  = wl + width;  // right org
    int bottom_y = ht + height; // bottom org

    // top band
    nine_rects[0] = rect(r.s + point(0, 0), size(wl, ht));
    nine_rects[1] = rect(r.s + point(wl, 0), size(width, ht));
    nine_rects[2] = rect(r.s + point(right_x, 0), size(wr, ht));

    // middle band
    nine_rects[3] = rect(r.s + point(0, ht), size(wl, height));
    nine_rects[4] = rect(r.s + point(wl, ht), size(width, height));
    nine_rects[5] = rect(r.s + point(right_x, ht), size(wr, height));

    // bottom band
    nine_rects[6] = rect(r.s + point(0, bottom_y), size(wl, hb));
    nine_rects[7] = rect(r.s + point(wl, bottom_y), size(width, hb));
    nine_rects[8] = rect(r.s + point(right_x, bottom_y), size(wr, hb));
  }

  static point offset_h(const rect &dst, const rect &src) {
    if (dst.width() > src.width()) return point(0, 0);
    return point((dst.width() - src.width()) / 2, 0);
  }

  static point offset_v(const rect &dst, const rect &src) {
    if (dst.height() > src.height()) return point(0, 0);
    return point(0, (dst.height() - src.height()) / 2);
  }

  void graphics::do_expand(image *img, const rect &dst, const SECTION_DEFS &sds,
                           rect area) {
    rect r = area.empty() ? rect(img->dim()) : area;
    rect src_sections[9];
    split_sections(r, sds.margins, src_sections);

    rect dst_sections[9];
    split_sections(dst, sds.margins, dst_sections);

    bool under_transform = is_under_transform();

    int n;

    // drawing first middle section
    n = 4;
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      rect r = dst_sections[n];
      if (under_transform) r >>= 1;
      if (sds.center == SRM_STRETCH)
        draw(img, r, src_sections[n]);
      else
        fill(img, r, src_sections[n]);
    }
    n = 1; // top bar
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      rect r = dst_sections[n];
      if (under_transform) r >>= gool::size(1, 0);
      if (sds.top == SRM_STRETCH)
        draw(img, r, src_sections[n]);
      else
        fill(img, r, src_sections[n], offset_h(r, src_sections[n]));
    }
    n = 3; // left bar
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      rect r = dst_sections[n];
      if (under_transform) r >>= gool::size(0, 1);
      if (sds.left == SRM_STRETCH)
        draw(img, r, src_sections[n]);
      else
        fill(img, r, src_sections[n], offset_v(r, src_sections[n]));
    }
    n = 5; // right bar
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      rect r = dst_sections[n];
      if (under_transform) r >>= gool::size(0, 1);
      if (sds.left == SRM_STRETCH)
        draw(img, r, src_sections[n]);
      else
        fill(img, r, src_sections[n], offset_v(r, src_sections[n]));
    }
    n = 7; // bottom bar
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      rect r = dst_sections[n];
      if (under_transform) r >>= gool::size(1, 0);
      if (sds.top == SRM_STRETCH)
        draw(img, r, src_sections[n]);
      else
        fill(img, r, src_sections[n], offset_h(r, src_sections[n]));
    }

    // corners:

    // top left
    n = 0;
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      draw(img, dst_sections[n], src_sections[n]);
    }
    // top right
    n = 2;
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      draw(img, dst_sections[n], src_sections[n]);
    }
    // bottom band
    n = 6;
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      draw(img, dst_sections[n], src_sections[n]);
    }
    n = 8;
    if (!src_sections[n].empty() &&
        !dst_sections[n].empty() /*&& dest.is_dirty(dst_sections[n])*/) {
      draw(img, dst_sections[n], src_sections[n]);
    }
  }

  void graphics::fill(gool::image *img, const rect &dst, const rect &src,
                      point offset, size dstcell) {
    size sz = src.size();
    if (sz.empty()) return;
    if (!dstcell.empty()) sz = dstcell;

    auto ren = image_renderer(img);

    clipper _(this, dst, true);

    argb clr;
    if ((sz.x <= 8 || sz.y <= 8) && img->is_solid_color(src, clr))
      return fill(clr.demultiply(), dst);

    if (offset.x > 0) offset.x = (offset.x % sz.x) - sz.x;
    if (offset.y > 0) offset.y = (offset.y % sz.y) - sz.y;

    for (int y = dst.s.y + offset.y; y < dst.e.y; y += sz.y)
      for (int x = dst.s.x + offset.x; x < dst.e.x; x += sz.x) {
        rect r_dst(point(x, y), sz);
        ren(r_dst, src);
      }
  }

  void graphics::draw_rectangle(pointf org, sizef dim, sizef rtl, sizef rtr,
                                sizef rbr, sizef rbl, bool stroke, bool fill) {
#if 0
      rounded_rect rr;
      rr.rect(float(r.origin.x), float(r.origin.y), float(r.corner.x), float(r.corner.y));
      rr.radius(float(rtl.x),float(rtl.y),float(rtr.x),float(rtr.y), float(rbr.x),float(rbr.y),float(rbl.x),float(rbl.y));
      rr.normalize_radius();

      gool::polygonf fig;
      gool::produce_path(rr,fig);
      handle<gool::path> p = app()->create_path();
      p->set(fig);
#else
    gool::normailize_round_rect(dim, rtl, rtr, rbr, rbl);
    handle<gool::path> p = create_path();
    p->add_round_rect(org, dim, rtl, rtr, rbr, rbl);
#endif
    draw_path(p,stroke,fill);
  }

  void graphics::draw_line(pointf start, pointf end, line_end_style start_style,
                           line_end_style end_style) {
    if (!end_style.type && !start_style.type) return draw_line(start, end);
    if (start == end) return draw_line(start, end);

    // const float end_h = end_style? end_style->dim.y: 0;
    // const float end_w = end_style? end_style->dim.x: 0;
    // const float r = w * 0.7f;
    // const float start_h = start_style? start_style->dim.y: 0;
    // const float start_w = start_style? start_style->dim.x: 0;

    linef l(start, end);

    pointf v1 = l.unit(); // unit vector

    switch (end_style.type) {
    case LET_ARROW: {
      pointf o          = end;
      end               = end - end_style.dim.y * v1;
      pointf         n  = pointf(-v1.y, v1.x); // normal to the vector
      pointf         a1 = end + end_style.dim.x * n;
      pointf         a2 = end - end_style.dim.x * n;
      gool::polygonf p;
      p.add(a1);
      p.add(o);
      p.add(a2);
      draw_path(p);
    } break;
    case LET_CIRCLE: {
      pointf o = end;
      end      = end - end_style.dim.y / 2 * v1;
      draw_ellipse(o, end_style.dim);
    } break;
    default: break;
    }

    switch (start_style.type) {
    case LET_ARROW: {
      pointf o          = start;
      start             = start + start_style.dim.y * v1;
      pointf         n  = pointf(-v1.y, v1.x); // normal to the vector
      pointf         a1 = o + start_style.dim.x * n;
      pointf         a2 = o - start_style.dim.x * n;
      gool::polygonf p;
      p.add(a1);
      p.add(o);
      p.add(a2);
      draw_path(p);
    } break;
    case LET_CIRCLE: {
      pointf o = start;
      start    = start + start_style.dim.y / 2 * v1;
      draw_ellipse(o, start_style.dim);
    } break;
    default: break;
    }
    // and the line
    draw_line(start, end);
  }

  void graphics::draw_star(pointf c, sizef r1, sizef r2, float start_angle,
                           int num_rays, bool stroke, bool fill) {
    gool::polygonf p;
    float          da = pi / float(num_rays);
    float          a  = start_angle;
    int            i;
    for (i = 0; i < num_rays; i++) {
      pointf pt;
      pt.x = cosf(a) * r2.x + c.x;
      pt.y = sinf(a) * r2.y + c.y;
      p.add(pt);
      a += da;
      pt.x = cosf(a) * r1.x + c.x;
      pt.y = sinf(a) * r1.y + c.y;
      p.add(pt);
      a += da;
    }
    draw_path(p,stroke,fill);
  }

  void graphics::draw_arc(pointf center, sizef radius, float angle_start, float angle_sweep, bool stroke, bool fill)
  {
    handle<gool::path> p = create_path();
    p->arc(center, radius, angle_start, angle_sweep);
    draw_path(p,fill,stroke);
  }


  

  void path::circ_arc_to(pointf fp1, pointf fp2, float radius, bool rel) {
    if (is_empty()) start(pointf(0, 0), true);

#if 0
    pointd p0 = lastp();
    pointd p1 = fp1;
    pointd p2 = fp2;
    if (rel) {
      p1 += p0;
      p2 += p0;
    }

    if ((p0 == p1) || (p1 == p2) || radius == 0) {
      line_to(p1);
      return;
    }

    // parallel ?
    double dir = (p2.x - p1.x) * (p0.y - p1.y) + (p2.y - p1.y) * (p1.x - p0.x);
    if (dir == 0) {
      line_to(p1);
      return;
    }

    double a2   = (p0.x - p1.x) * (p0.x - p1.x) + (p0.y - p1.y) * (p0.y - p1.y);
    double b2   = (p1.x - p2.x) * (p1.x - p2.x) + (p1.y - p2.y) * (p1.y - p2.y);
    double c2   = (p0.x - p2.x) * (p0.x - p2.x) + (p0.y - p2.y) * (p0.y - p2.y);
    double cosx = (a2 + b2 - c2) / (2 * sqrt(a2 * b2));
    double sinx = sqrt(1 - cosx * cosx);
    double d    = radius / ((1 - cosx) / sinx);

    pointd an = (p1 - p0) / sqrt(a2);
    pointd bn = (p1 - p2) / sqrt(b2);

    pointd p3 = p1 - (an * d);
    pointd p4 = p1 - (bn * d);

    bool clockwise = (dir > 0);

    pointd c;
    c.x = p3.x + an.y * radius * (clockwise ? -1 : 1);
    c.y = p3.y - an.x * radius * (clockwise ? -1 : 1);

    float angle1 = (float)atan2((p3.y - c.y), (p3.x - c.x));
    float angle2 = (float)atan2((p4.y - c.y), (p4.x - c.x));

    line_to(p3);

    //???? lastp(p4); // NOTE: it shall be set before next call, d2d uses it

    auto angle_start = angle1;
    auto angle_swipe = 0.0f;

    if (angle1 < 0) angle1 = 2 * pi + angle1;
    if (angle2 < 0) angle2 = 2 * pi + angle2;

    if (!clockwise) {
      if (angle1 > angle2)
        angle_swipe = -(angle1 - angle2);
      else
        angle_swipe = -(2 * pi + angle1 - angle2);
    } else {
      if (angle1 > angle2)
        angle_swipe = 2 * pi + angle2 - angle1;
      else
        angle_swipe = angle2 - angle1;
    }

    add_arc(pointf(c), radius, angle_start, angle_swipe);
#else

    pointd p0 = lastp();
    pointd p1 = fp1;
    pointd p2 = fp2;
    if (rel) {
      p1 += p0;
      p2 += p0;
    }

    if ((p1.x == p0.x && p1.y == p0.y) || (p1.x == p2.x && p1.y == p2.y) || radius == 0.f) {
      line_to(pointf(p1));
      return;
    }

    pointd  p1p0((p0.x - p1.x), (p0.y - p1.y));
    pointd  p1p2((p2.x - p1.x), (p2.y - p1.y));
    double p1p0_length = sqrt(p1p0.x * p1p0.x + p1p0.y * p1p0.y);
    double p1p2_length = sqrt(p1p2.x * p1p2.x + p1p2.y * p1p2.y);

    double cos_phi = (p1p0.x * p1p2.x + p1p0.y * p1p2.y) / (p1p0_length * p1p2_length);

    if (cos_phi == -1) {
       // all points on a line logic
       line_to(pointf(p1));
       return;
    }
    if (cos_phi == 1) {
      // add infinite far away point
      unsigned int max_length = 65535;
      double factor_max = max_length / p1p0_length;
      pointd ep((p0.x + factor_max * p1p0.x), (p0.y + factor_max * p1p0.y));
      line_to(ep);
      return;
    }

    double tangent = radius / tan(acos(cos_phi) / 2);
    double factor_p1p0 = tangent / p1p0_length;
    pointd t_p1p0((p1.x + factor_p1p0 * p1p0.x), (p1.y + factor_p1p0 * p1p0.y));

    pointd orth_p1p0(p1p0.y, -p1p0.x);
    double  orth_p1p0_length = sqrt(orth_p1p0.x * orth_p1p0.x + orth_p1p0.y * orth_p1p0.y);
    double  factor_ra = radius / orth_p1p0_length;

    // angle between orth_p1p0 and p1p2 to get the right vector orthographic to p1p0
    double cos_alpha = (orth_p1p0.x * p1p2.x + orth_p1p0.y * p1p2.y) / (orth_p1p0_length * p1p2_length);
    if (cos_alpha < 0.0)
      orth_p1p0 = pointd(-orth_p1p0.x, -orth_p1p0.y);

    pointd p((t_p1p0.x + factor_ra * orth_p1p0.x), (t_p1p0.y + factor_ra * orth_p1p0.y));

    // calculate angles for add_arc
    orth_p1p0 = pointd(-orth_p1p0.x, -orth_p1p0.y);
    double sa = acos(orth_p1p0.x / orth_p1p0_length);

    if (orth_p1p0.y < 0.f)
        sa = 2 * D_PI - sa;

    // anticlockwise logic
    bool anticlockwise = false;

    double  factor_p1p2 = tangent / p1p2_length;
    pointd t_p1p2((p1.x + factor_p1p2 * p1p2.x), (p1.y + factor_p1p2 * p1p2.y));
    pointd orth_p1p2((t_p1p2.x - p.x), (t_p1p2.y - p.y));
    double  orth_p1p2_length = sqrt(orth_p1p2.x * orth_p1p2.x + orth_p1p2.y * orth_p1p2.y);
    double  ea = acos(orth_p1p2.x / orth_p1p2_length);
    
    if (orth_p1p2.y < 0)
        ea = 2 * D_PI - ea;
    if ((sa > ea) && ((sa - ea) < D_PI))
        anticlockwise = true;
    if ((sa < ea) && ((ea - sa) > D_PI))
        anticlockwise = true;

    line_to(t_p1p0);

    float angle_start = float(sa);
    float angle_swipe = 0.0;

    if (sa < 0) sa = 2 * D_PI + sa;
    if (ea < 0) ea = 2 * D_PI + ea;

    if (anticlockwise) {
      if (sa > ea)
        angle_swipe = float(-(sa - ea));
      else
        angle_swipe = float(-(2 * D_PI + sa - ea));
    }
    else {
      if (sa > ea)
        angle_swipe = float(2 * D_PI + ea - sa);
      else
        angle_swipe = float(ea - sa);
    }

    add_arc(p, radius, angle_start, angle_swipe);
#endif
  }
  
  void path::arc(pointf c, sizef r, float start, float swipe, bool rel) {
    //if (is_empty()) start(pointf(0, 0), true);

    if (rel)
      c += lastp();

    pointf first;
    first.x = c.x + r.x * cosf(start);
    first.y = c.y + r.y * sinf(start);

    if (is_open())
      line_to(first);
    else
      move_to(first);

    int dir = swipe > 0 ? 1 : -1;
    //swipe *= dir;
    int n = int(abs(swipe) / (F_PI / 2));
    if(swipe > 0)
      swipe -= (F_PI / 2) * n;
    else 
      swipe += (F_PI / 2) * n;

    for (int i = 0; i < min(4,n); ++i, start += dir * (F_PI / 2))
      add_arc(c, r, start, dir * (F_PI / 2));
    add_arc(c, r, start, swipe);
  }


#if 1

  inline float sqr(float x) { return x * x; }
  inline float vmag(float x, float y) { return sqrtf(x * x + y * y); }
  inline float vecrat(float ux, float uy, float vx, float vy) {
    return (ux * vx + uy * vy) / (vmag(ux, uy) * vmag(vx, vy));
  }
  inline float vecang(float ux, float uy, float vx, float vy) {
    float r = vecrat(ux, uy, vx, vy);
    if (r < -1.0f) r = -1.0f;
    if (r > 1.0f) r = 1.0f;
    return ((ux * vy < uy * vx) ? -1.0f : 1.0f) * acosf(r);
  }
  inline void xform_point(float *dx, float *dy, float x, float y,
                          const float *t) {
    *dx = x * t[0] + y * t[2] + t[4];
    *dy = x * t[1] + y * t[3] + t[5];
  }

  inline void xform_vec(float *dx, float *dy, float x, float y,
                        const float *t) {
    *dx = x * t[0] + y * t[2];
    *dy = x * t[1] + y * t[3];
  }

  inline bool is_halph_circle(pointf from, pointf to, sizef r) {
    float dx = fabs(to.x - from.x);
    float dy = fabs(to.y - from.y);
    if (dx < 0.000001 && 2 * r.y == dy) return true;
    if (dy < 0.000001 && 2 * r.x == dx) return true;
    return false;
  }

  void path::arc_to(pointf to, sizef r, float rotation_angle, bool large_arc,
                    bool clockwise, bool rel_to) {
    pointf from = lastp();
    if (rel_to) { to += from; }

    float rx, ry;
    float x1, y1, x2, y2, cx, cy, dx, dy, d;
    float x1p, y1p, cxp, cyp, s, sa, sb;
    float ux, uy, vx, vy, a1, da;
    float x, y, tanx, tany, a, px = 0, py = 0, ptanx = 0, ptany = 0, t[6];
    float sinrx, cosrx;
    int   i, ndivs;
    float hda, kappa;

    // hack needed to fix algoritm below
#pragma TODO("check why I need this ...")
    if (is_halph_circle(from, to, r) && !clockwise) large_arc = true;

    rx             = fabsf(r.x);     // y radius
    ry             = fabsf(r.y);     // x radius
    rotation_angle = rotation_angle; // args[2] / 180.0f * M_PI;
                                     // // x rotation engle
    x1 = from.x; // start point
    y1 = from.y;
    x2 = to.x;
    y2 = to.y;

    dx = x1 - x2;
    dy = y1 - y2;
    d  = sqrtf(dx * dx + dy * dy);
    if (d < 1e-6f || rx < 1e-6f || ry < 1e-6f) {
      // the arc degenerates to a line
      line_to(pointf(x2, y2));
      return;
    }

    sinrx = sinf(rotation_angle);
    cosrx = cosf(rotation_angle);

    // Convert to center point parameterization.
    // http://www.w3.org/TR/SVG11/implnote.html#ArcImplementationNotes
    // 1) Compute x1', y1'
    x1p = cosrx * dx / 2.0f + sinrx * dy / 2.0f;
    y1p = -sinrx * dx / 2.0f + cosrx * dy / 2.0f;
    d   = sqr(x1p) / sqr(rx) + sqr(y1p) / sqr(ry);
    if (d > 1) {
      d = sqrtf(d);
      rx *= d;
      ry *= d;
    }
    // 2) Compute cx', cy'
    s  = 0.0f;
    sa = sqr(rx) * sqr(ry) - sqr(rx) * sqr(y1p) - sqr(ry) * sqr(x1p);
    sb = sqr(rx) * sqr(y1p) + sqr(ry) * sqr(x1p);
    if (sa < 0.0f) sa = 0.0f;
    if (sb > 0.0f) s = sqrtf(sa / sb);
    if (large_arc == clockwise) s = -s;
    cxp = s * rx * y1p / ry;
    cyp = s * -ry * x1p / rx;

    // 3) Compute cx,cy from cx',cy'
    cx = (x1 + x2) / 2.0f + cosrx * cxp - sinrx * cyp;
    cy = (y1 + y2) / 2.0f + sinrx * cxp + cosrx * cyp;

    // 4) Calculate theta1, and delta theta.
    ux = (x1p - cxp) / rx;
    uy = (y1p - cyp) / ry;
    vx = (-x1p - cxp) / rx;
    vy = (-y1p - cyp) / ry;
    a1 = vecang(1.0f, 0.0f, ux, uy); // Initial angle
    da = vecang(ux, uy, vx, vy);     // Delta angle

    // if (vecrat(ux,uy,vx,vy) <= -1.0f) da = F_PI;
    // if (vecrat(ux,uy,vx,vy) >= 1.0f) da = 0;

    if (large_arc) {
      // Choose large arc
      if (da > 0.0f)
        da = da - F_2PI;
      else
        da = F_2PI + da;
    }

    // Approximate the arc using cubic spline segments.
    t[0] = cosrx;
    t[1] = sinrx;
    t[2] = -sinrx;
    t[3] = cosrx;
    t[4] = cx;
    t[5] = cy;

    // Split arc into max 90 degree segments.
    // The loop assumes an iteration per end point (including start and end),
    // this +1.
    ndivs = (int)(fabsf(da) / (F_PI / 2) + 1.0f);
    hda   = (da / (float)ndivs) / 2;
    kappa = fabsf(4.0f / 3.0f * (1.0f - cosf(hda)) / sinf(hda));
    if (da < 0.0f) kappa = -kappa;

    for (i = 0; i <= ndivs; i++) {
      a  = a1 + da * (i / (float)ndivs);
      dx = cosf(a);
      dy = sinf(a);
      xform_point(&x, &y, dx * rx, dy * ry, t);                      // position
      xform_vec(&tanx, &tany, -dy * rx * kappa, dx * ry * kappa, t); // tangent
      if (i > 0)
        cubic_to(pointf(x, y), pointf(px + ptanx, py + ptany),
                 pointf(x - tanx, y - tany));
      px    = x;
      py    = y;
      ptanx = tanx;
      ptany = tany;
    }
    // line_to(pointf(x2,y2));
  }

#else
  void path::arc_to(pointf to, sizef r, float rotation_angle, bool large_arc,
                    bool clockwise, bool rel_to) {
    // assert( opened );
    pointf from = lastp();

    if (rel_to) { to += from; }

    // assert( lastp.x || lastp.y );

    // lastp

    // assert( open ); open = open;

    const float epsilon = 1e-8f;

    r.x = abs(r.x);
    r.y = abs(r.y);

    // Ensure radii are valid
    //-------------------------
    if (r.x < epsilon || r.y < epsilon) {
      line_to(to);
      return;
    }

    if (distance(from, to) < epsilon) {
      // If the endpoints (x, y) and (x0, y0) are identical, then this
      // is equivalent to omitting the elliptical arc segment entirely.
      return;
    }
    bezier_arc_svg arc;

    if (arc.calc(from.x, from.y, r.x, r.y, rotation_angle, large_arc, clockwise,
                 to.x, to.y)) {
      append_path(arc, this);
      // line_to(to);
    } else {
      line_to(to);
    }
  }
#endif

} // namespace gool
