//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| GIF decoder
//|
//|

#include "config.h"
#include "gool.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "tool/tool.h"

#if defined(GIF_SUPPORT)

namespace gool {

  using namespace tool;

#define GIFterminator ';'
#define GIFextension '!'
#define GIFimage ','

#define GIFcomment 0xfe
#define GIFapplication 0xff
#define GIFplaintext 0x01
#define GIFgraphicctl 0xf9

#define MAXCMSIZE 256

  typedef struct GifColorTag {
    byte r, g, b;
  } GifColor;

  typedef struct {
    unsigned int Width;
    unsigned int Height;
    GifColor     ColorMap[MAXCMSIZE];
    unsigned int ColorMap_present;
    unsigned int BitPixel;
    unsigned int ColorResolution;
    int          Background;
    unsigned int AspectRatio;
  } gif_scr;

  /*******************************************************************************/

  struct mem_stream {
    const byte *ptr;
    const byte *end;
    mem_stream(const byte *data, size_t length)
        : ptr(data), end(data + length) {}

    bool read(byte *data, size_t length) {
      if ((ptr + length) > end) {
        ptr = end;
        return false;
      }
      if (length == 1)
        *data = *ptr++;
      else {
        tool::target(data, length).copy(ptr);
        ptr += length;
      }
      return true;
    }
    mem_stream &operator>>(byte &b) {
      if (!read(&b, 1)) b = 0;
      return *this;
    }

    bool eof() { return ptr >= end; }
  };

#define CM_RED 0
#define CM_GREEN 1
#define CM_BLUE 2

#define MAX_LWZ_BITS 12

#define INTERLACE 0x40
#define GLOBALCOLORMAP 0x80
#define LOCALCOLORMAP 0x80
#define BitSet(byte, bit) (((byte) & (bit)) == (bit))

#define LM_to_uint(a, b) (((b) << 8) | (a))

  // struct gif_scr GifScreen;

  typedef struct {
    int   transparent;
    int   delayTime;
    int   inputFlag;
    int   disposal;
    point offset;
  } gif89struct;

  static int  ReadColorMap(mem_stream &ms, int number,
                           GifColor buffer[MAXCMSIZE]);
  static int  DoExtension(gif89struct &Gif89, mem_stream &ms, int label);
  static bool DoAppExtension(int &iterations, mem_stream &ms);
  static int  GetDataBlock(mem_stream &ms, byte *buf, int *ZeroDataBlock);
  // static int ReadImage(gool::dib24& des, mem_stream& ms, int interlace, int
  // *ZeroDataBlock);
  static bool          read_indexes(tool::array<byte> &indexes, mem_stream &ms,
                                    int colors, int width, int height, bool interlace,
                                    int *ZeroDataBlock);
  static gool::bitmap *readFrame(mem_stream &ms, gif_scr &GifScreen,
                                 gif89struct &Gif89);

  bool decode_gif(const byte *data, size_t data_length,
                  handle<gool::image> &pimg /*, bool all*/) {
    handle<gool::bitmap>         pbmp;
    handle<gool::animated_image> pani;
    // uint  duration = 1;
    // bool  restore = false;
    // point offset;

    assert(pimg == 0);

    mem_stream ms(data, data_length);

    byte buf[16];
    byte c;
    // GifColor      localColorMap[MAXCMSIZE];
    // int           useGlobalColormap;
    char         gif_version[4];
    unsigned int i;
    gif_scr      GifScreen;
    gif89struct  Gif89, Gif89prev;
    int          imagecount = 0;
    int          iterations = -1;

    if (!ms.read(buf, 6)) { return false; }

    if (strncmp((char *)buf, "GIF", 3) != 0) { return false; }

    strncpy_s(gif_version, (char *)buf + 3, 3);
    gif_version[3] = '\0';

    if ((strcmp(gif_version, "87a") != 0) &&
        (strcmp(gif_version, "89a") != 0)) {

      return false;
    }

    if (!ms.read(buf, 7)) { return false; }

    GifScreen.Width            = LM_to_uint(buf[0], buf[1]);
    GifScreen.Height           = LM_to_uint(buf[2], buf[3]);
    GifScreen.BitPixel         = 2 << (buf[4] & 0x07);
    GifScreen.ColorResolution  = (((buf[4] & 0x70) >> 3) + 1);
    GifScreen.ColorMap_present = BitSet(buf[4], GLOBALCOLORMAP);

    if (GifScreen.ColorMap_present) { /* Global Colormap */
      if (ReadColorMap(ms, GifScreen.BitPixel, GifScreen.ColorMap)) {

        return false;
      }
    } else {
      /* the GIF spec says that if neither global nor local
       * color maps are present, the decoder should use a system
       * default map, which should have black and white as the
       * first two colors. So we use black, white, red, green, blue,
       * yellow, purple and cyan.
       * I don't think missing color tables are a common case,
       * at least it's not handled by most GIF readers.
       */

      static int colors[] = {0, 7, 1, 2, 4, 3, 5, 6};

      for (i = 0; i < MAXCMSIZE - 8; i++) {
        GifScreen.ColorMap[i].r =
            (byte)((colors[i & 7] & 1) ? ((255 - i) & 0xf8) : 0);
        GifScreen.ColorMap[i].g =
            (byte)((colors[i & 7] & 2) ? ((255 - i) & 0xf8) : 0);
        GifScreen.ColorMap[i].b =
            (byte)((colors[i & 7] & 4) ? ((255 - i) & 0xf8) : 0);
      }
      for (i = MAXCMSIZE - 8; i < MAXCMSIZE; i++) {
        GifScreen.ColorMap[i].r = (byte)((colors[i & 7] & 1) ? 4 : 0);
        GifScreen.ColorMap[i].g = (byte)((colors[i & 7] & 2) ? 4 : 0);
        GifScreen.ColorMap[i].b = (byte)((colors[i & 7] & 4) ? 4 : 0);
      }
    }

    if (GifScreen.ColorMap_present) {
      GifScreen.Background = buf[5];
    } else {
      GifScreen.Background = -1; /* background unspecified */
    }

    GifScreen.AspectRatio = buf[6];

    /*
     * Initialize GIF89 extensions
     */
    Gif89.transparent = -1;
    Gif89.delayTime   = -1;
    Gif89.inputFlag   = -1;
    Gif89.disposal    = 0;
    Gif89prev         = Gif89;

    while (1) {
      // if(imagecount > 0) break; // do not load all sub images
      if (!ms.read(&c, 1)) { return (bool)(imagecount > 0); }

      switch (c) {
      case GIFterminator: goto FINISH;
      case GIFextension:
        if (!ms.read(&c, 1)) { goto FINISH; }
        if (c == GIFapplication) {
          DoAppExtension(iterations, ms);
        } else if (DoExtension(Gif89, ms, c) != 0) {
          goto FINISH;
        }
        // duration = Gif89.delayTime? Gif89.delayTime: 4;
        // restore = Gif89.disposal > 1;
        // offset = Gif89.offset;

        continue;
      case GIFimage: {
        pbmp = readFrame(ms, GifScreen, Gif89);
        if (!pbmp || !pbmp->is_valid()) {
          pani = nullptr;
          pimg = nullptr;
          return false;
        }

        // if(Gif89.delayTime != 300)
        //  Gif89.delayTime = Gif89.delayTime;

        if (!pani) {
          pani = new animated_image(size(GifScreen.Width, GifScreen.Height));
          pani->add(pbmp, 10 * (Gif89.delayTime ? Gif89.delayTime : 10), 0,
                    Gif89.offset);
          pimg = pani;
        } else {
          pani->add(pbmp, 10 * (Gif89.delayTime ? Gif89.delayTime : 10),
                    Gif89prev.disposal, Gif89.offset);
        }
        Gif89prev         = Gif89;
        Gif89.transparent = -1;
        Gif89.delayTime   = -1;
        Gif89.inputFlag   = -1;
        Gif89.disposal    = 0;
        // return true;
        ++imagecount;
        continue;
      }
      default: goto FINISH;
      }
    }

  FINISH:
    if (pani && iterations >= 0) pani->iterations(iterations);

    return (imagecount != 0);
  }

  static gool::bitmap *readFrame(mem_stream &ms, gif_scr &GifScreen,
                                 gif89struct &Gif89) {
    gool::bitmap *pbmp = 0;
    GifColor      localColorMap[MAXCMSIZE];
    int           useGlobalColormap;
    int           ZeroDataBlock = 0;
    byte          buf[16];

    int bitPixel;
    int w, h;

    if (!ms.read(buf, 9)) { return 0; }
    useGlobalColormap = !BitSet(buf[8], LOCALCOLORMAP);
    bitPixel =
        useGlobalColormap ? GifScreen.BitPixel : (1 << ((buf[8] & 0x07) + 1));
    Gif89.offset.x = LM_to_uint(buf[0], buf[1]);
    Gif89.offset.y = LM_to_uint(buf[2], buf[3]);
    w              = LM_to_uint(buf[4], buf[5]);
    h              = LM_to_uint(buf[6], buf[7]);

    if (Gif89.offset.x > (int)GifScreen.Width)
      Gif89.offset.x = (int)GifScreen.Width;
    if (Gif89.offset.y > (int)GifScreen.Height)
      Gif89.offset.y = (int)GifScreen.Height;

    if (Gif89.offset.x + w > (int)GifScreen.Width)
      w = GifScreen.Width - Gif89.offset.x;
    if (Gif89.offset.y + h > (int)GifScreen.Height)
      h = GifScreen.Height - Gif89.offset.y;
    if (w < 0) w = 0;
    if (h < 0) h = 0;

    if (!w || !h) return 0; // wrong image format

    tool::array<byte> indexes;

    if (!useGlobalColormap) {
      if (ReadColorMap(ms, bitPixel, localColorMap)) return 0;
    }

    if (read_indexes(indexes, ms, bitPixel, w, h, BitSet(buf[8], INTERLACE),
                     &ZeroDataBlock)) {

      size sz(w, h);

      GifColor *cmap = useGlobalColormap ? GifScreen.ColorMap : localColorMap;

      if (Gif89.transparent >= 0) {
        pbmp = new bitmap(sz, true, false);
        if (!pbmp) return 0;
        argb trans;
        trans.alfa  = 0;
        trans.blue  = 0;
        trans.green = 0;
        trans.red   = 0;

        /*if(GifScreen.Background >= 0)
        {
          trans.red = cmap[GifScreen.Background].r;
          trans.green = cmap[GifScreen.Background].g;
          trans.blue = cmap[GifScreen.Background].b;
          trans.alfa = 0xff;
        }*/

        byte *idxs = &indexes[0];

        array<argb> pm(w * h);

        argb *p = pm.begin();
        if (!p) {
          delete pbmp;
          return 0;
        }

        int n = 0;
        for (int r = 0; r < h; r++) {
          // p = pbmp->operator()(r);
          for (int c = 0; c < w; c++) {
            byte idx = idxs[n++];
            if (idx == Gif89.transparent)
              *p = trans;
            else {
              GifColor *pc = &cmap[idx];
              p->alfa      = 255;
              p->red       = pc->r;
              p->blue      = pc->b;
              p->green     = pc->g;
            }
            p++;
          }
        }
        pbmp->set_bits(pm());
        // pbmp->premultiply(); // don't needed in case alfa is only 0 or 255
      } else {
        pbmp = new bitmap(sz, false);
        if (!pbmp) return 0;
        argb *p = (*pbmp)(0);
        if (!p) {
          delete pbmp;
          return 0;
        }
        byte *idxs = &indexes[0];
        // int rl = pbmp->row_byte_length();
        int n = 0;
        for (int r = 0; r < h; ++r) {
          p = (*pbmp)(r);
          for (int c = 0; c < w; c++) {
            byte      idx = idxs[n++];
            GifColor *pc  = &cmap[idx];
            p->red        = pc->r;
            p->blue       = pc->b;
            p->green      = pc->g;
            p->alfa       = 255;
            p++;
          }
        }
      }
    }
    return pbmp;
  }

  static int ReadColorMap(mem_stream &ms, int number,
                          GifColor colors[MAXCMSIZE]) {
    int  i;
    byte rgb[3];

    for (i = 0; i < number; ++i) {
      if (!ms.read(rgb, sizeof(rgb))) {
        //fprintf(stderr, "bad colormap\n");
        return (1);
      }

      colors[i].r = rgb[0];
      colors[i].g = rgb[1];
      colors[i].b = rgb[2];
    }

    for (i = number; i < MAXCMSIZE; ++i) {
      colors[i].r = 0;
      colors[i].g = 0;
      colors[i].b = 0;
    }

    return 0;
  }

  static bool DoAppExtension(int &iterations, mem_stream &ms) {
    byte buf[256];
    int  size;
    int  ZeroDataBlock = 0;
    bool gotit         = false;

    /*
    byte   1       : 33 (hex 0x21) GIF Extension code
    byte   2       : 255 (hex 0xFF) Application Extension Label
    byte   3       : 11 (hex (0x0B) Length of Application Block
                     (eleven bytes of data to follow)
    bytes  4 to 11 : "NETSCAPE"
    bytes 12 to 14 : "2.0"
    byte  15       : 3 (hex 0x03) Length of Data Sub-Block
                     (three bytes of data to follow)
    byte  16       : 1 (hex 0x01)
    bytes 17 to 18 : 0 to 65535, an unsigned integer in
                     lo-hi byte format. This indicate the
                     number of iterations the loop should
                     be executed.
    bytes 19       : 0 (hex 0x00) a Data Sub-block Terminator.
    */
    do {
      size = GetDataBlock(ms, buf, &ZeroDataBlock);
      if (size != 11) break;
      if (memcmp("NETSCAPE2.0", buf, 11) != 0) break;
      size = GetDataBlock(ms, buf, &ZeroDataBlock);
      if (size != 3) break;
      if (buf[0] != 1) break;

      iterations = LM_to_uint(buf[1], buf[2]);
      gotit      = true;

      break;

    } while (0);

    while ((size = GetDataBlock(ms, (byte *)buf, &ZeroDataBlock)) > 0)
      ;

    return gotit;
  }

  static int DoExtension(gif89struct &Gif89, mem_stream &ms, int label) {
    byte        buf[256];
    const char *str;
    int         size;
    int         ZeroDataBlock = 0;

    switch (label) {
    case GIFplaintext: /* Plain Text Extension */
      str = "Plain Text Extension";
      /*
       * reset GIF89 extensions after Plain Text
       */
      Gif89.transparent = -1;
      Gif89.delayTime   = -1;
      Gif89.inputFlag   = -1;
      Gif89.disposal    = 0;

      while (GetDataBlock(ms, (byte *)buf, &ZeroDataBlock) > 0)
        continue;
      return (0);

    case GIFapplication: /* Application Extension */
      str = "Application Extension";
      break;

    case GIFcomment: /* Comment Extension */
      while ((size = GetDataBlock(ms, (byte *)buf, &ZeroDataBlock)) > 0)
        ;
      return 0;

    case GIFgraphicctl: /* Graphic Control Extension */
      size            = GetDataBlock(ms, (byte *)buf, &ZeroDataBlock);
      Gif89.disposal  = (buf[0] >> 2) & 0x7;
      Gif89.inputFlag = (buf[0] >> 1) & 0x1;
      Gif89.delayTime = LM_to_uint(buf[1], buf[2]);
      if ((buf[0] & 0x1) != 0) Gif89.transparent = (int)((byte)buf[3]);

      if (Gif89.disposal == 0 && Gif89.inputFlag == 0 && Gif89.delayTime == 0) {

        /* do not keep GCEs only indicating transparency */

        while (GetDataBlock(ms, (byte *)buf, &ZeroDataBlock) > 0)
          ;

        return (0);
      } else
        break;
    default:
      while (GetDataBlock(ms, (byte *)buf, &ZeroDataBlock) > 0)
        ;
      return (1);
    }

    while ((size = GetDataBlock(ms, (byte *)buf, &ZeroDataBlock)) > 0)
      ;

    return (0);
  }

  /*static int ReadFrame(mem_stream& ms, gif_scr& GifScreen)
  {

  }*/

  static int GetDataBlock(mem_stream &ms, byte *buf, int *ZeroDataBlock) {
    byte count;

    count = 0;
    if (!ms.read(&count, 1)) {
      //fprintf(stderr, "error in getting DataBlock size\n");
      return -1;
    }

    *ZeroDataBlock = (int)(count == 0);

    if ((count != 0) && (!ms.read(buf, count))) {
      //fprintf(stderr, "error in reading DataBlock\n");
      return -1;
    }

    return ((int)count);
  }

  typedef struct {
    /*
    **  Pulled out of nextCode
    */
    int curbit, lastbit, get_done, last_byte;
    int return_clear;
    /*
    **  Out of nextLWZ
    */
    int  stack[(1 << (MAX_LWZ_BITS)) * 2], *sp;
    int  code_size, set_code_size;
    int  max_code, max_code_size;
    int  clear_code, end_code;
    byte buf[280];
    int  table[2][(1 << MAX_LWZ_BITS)];
    int  firstcode, oldcode;

  } LZWStack;

  static LZWStack *initLWZ(LZWStack *s, int input_code_size) {
    if (s == NULL) s = (LZWStack *)malloc(sizeof(LZWStack));
    if (!s) return NULL;
    s->set_code_size = input_code_size;
    s->code_size     = s->set_code_size + 1;
    s->clear_code    = 1 << s->set_code_size;
    s->end_code      = s->clear_code + 1;
    s->max_code_size = 2 * s->clear_code;
    s->max_code      = s->clear_code + 2;

    s->curbit = s->lastbit = 0;
    s->last_byte           = 2;
    s->get_done            = false;

    s->return_clear = true;

    s->sp = s->stack;
    memset(s->buf, 0, 280);
    memset(s->table, 0, sizeof(s->table));
    s->firstcode = 0;
    s->oldcode   = 0;
    return s;
  }

  static int nextCode(LZWStack *s, mem_stream &ms, int *ZeroDataBlock) {
    static int maskTbl[16] = {
        0x0000, 0x0001, 0x0003, 0x0007, 0x000f, 0x001f, 0x003f, 0x007f,
        0x00ff, 0x01ff, 0x03ff, 0x07ff, 0x0fff, 0x1fff, 0x3fff, 0x7fff,
    };
    int  i, j, end;
    long ret;

    if (s->return_clear) {
      s->return_clear = false;
      return s->clear_code;
    }

    end = s->curbit + s->code_size;

    if (end >= s->lastbit) {
      int count;

      if (s->get_done) { return -1; }
      s->buf[0] = s->buf[s->last_byte - 2];
      s->buf[1] = s->buf[s->last_byte - 1];

      if ((count = GetDataBlock(ms, &s->buf[2], ZeroDataBlock)) == 0)
        s->get_done = true;

      if (count < 0) return -1;

      s->last_byte = 2 + count;
      s->curbit    = (s->curbit - s->lastbit) + 16;
      s->lastbit   = (2 + count) * 8;

      end = s->curbit + s->code_size;
    }

    j = end / 8;
    i = s->curbit / 8;

    if (i == j)
      ret = (long)s->buf[i];
    else if (i + 1 == j)
      ret = (long)s->buf[i] | ((long)s->buf[i + 1] << 8);
    else
      ret = (long)s->buf[i] | ((long)s->buf[i + 1] << 8) |
            ((long)s->buf[i + 2] << 16);

    ret = (ret >> (s->curbit % 8)) & maskTbl[s->code_size];

    s->curbit += s->code_size;

    return (int)ret;
  }

#define readLWZ(s, ms, z) ((s->sp > s->stack) ? *--(s->sp) : nextLWZ(s, ms, z))

  static int nextLWZ(LZWStack *s, mem_stream &ms, int *ZeroDataBlock) {
    int code, incode;
    int i;

    while ((code = nextCode(s, ms, ZeroDataBlock)) >= 0) {
      if (code == s->clear_code) {

        /* corrupt GIFs can make this happen */
        if (s->clear_code >= (1 << MAX_LWZ_BITS)) { return -2; }

        for (i = 0; i < s->clear_code; ++i) {
          s->table[0][i] = 0;
          s->table[1][i] = i;
        }
        for (; i < (1 << MAX_LWZ_BITS); ++i)
          s->table[0][i] = s->table[1][i] = 0;

        s->code_size     = s->set_code_size + 1;
        s->max_code_size = 2 * s->clear_code;
        s->max_code      = s->clear_code + 2;
        s->sp            = s->stack;
        do {
          s->firstcode = s->oldcode = nextCode(s, ms, ZeroDataBlock);
        } while (s->firstcode == s->clear_code);

        return s->firstcode;
      }
      if (code == s->end_code) {
        int  count;
        byte buf[260];

        if (*ZeroDataBlock) return -2;

        while ((count = GetDataBlock(ms, buf, ZeroDataBlock)) > 0)
          ;

        return -2;
      }

      incode = code;

      if (code >= s->max_code) {
        *(s->sp)++ = s->firstcode;
        code       = s->oldcode;
      }

      while (code >= s->clear_code) {
        *(s->sp)++ = s->table[1][code];
        if (code == s->table[0][code]) { return (code); }
        if (((char *)s->sp - (char *)s->stack) >= sizeof(s->stack)) {
          return (code);
        }
        code = s->table[0][code];
      }

      *(s->sp)++ = s->firstcode = s->table[1][code];

      if ((code = s->max_code) < (1 << MAX_LWZ_BITS)) {
        s->table[0][code] = s->oldcode;
        s->table[1][code] = s->firstcode;
        ++s->max_code;
        if ((s->max_code >= s->max_code_size) &&
            (s->max_code_size < (1 << MAX_LWZ_BITS))) {
          s->max_code_size *= 2;
          ++s->code_size;
        }
      }

      s->oldcode = incode;

      if (s->sp > s->stack) return *--(s->sp);
    }
    return code;
  }

  static bool read_indexes(tool::array<byte> &indexes, mem_stream &ms,
                           int colors, int width, int height, bool interlace,
                           int *ZeroDataBlock) {

    indexes.size(width * height);

    byte *dp, c;
    int   v;
    int   xpos, ypos, ycnt, pass;

    LZWStack *s;

    /*
    **  Initialize the compression routines
    */
    if (!ms.read(&c, 1)) { return (1); }

    s = initLWZ(NULL, c);
    if (!s) return false;

    for (ycnt = 0, ypos = 0, pass = 0; ycnt < height; ycnt++) {
      if (ypos < height) {
        dp = &indexes[ypos * width];
        for (xpos = 0; xpos < width; xpos++) {
          v = readLWZ(s, ms, ZeroDataBlock);
          if (v < 0 || v >= colors) {
            free(s);
            return false;
          }
          *dp++ = (byte)v;
        }
      }
      if (interlace) {
        switch (pass) {
        case 0:
        case 1: ypos += 8; break;
        case 2: ypos += 4; break;
        case 3: ypos += 2; break;
        }

        if (ypos >= height) {
          ++pass;
          switch (pass) {
          case 1: ypos = 4; break;
          case 2: ypos = 2; break;
          case 3:
          default: ypos = 1; break;
          }
        }
      } else {
        ++ypos;
      }
    }

    while (readLWZ(s, ms, ZeroDataBlock) >= 0)
      continue;

    free(s);

    return true;
  }

}; // namespace gool

#endif
