//|
//|
//| Copyright (c) 2001-2005
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| geometry classes: point, rect, etc.
//|
//|

#ifndef __GL_GEOMETRY_H
#define __GL_GEOMETRY_H

#include "tool/tool.h"
//#include <math.h>
#include <cmath>

//#ifdef USE_D2D
//#include <d2d1.h>
//#endif

namespace gool {

  namespace geom {

    using namespace tool;

#pragma warning(disable : 4786) // long names

    template <typename C> class size_t;
    template <typename C> class rect_t;

    template <typename C> class point_t {
    public:
      C x, y;

      point_t() : x(), y() {}
      point_t(C xx, C yy) : x(xx), y(yy) {}
      point_t(const point_t &other) : x(other.x), y(other.y) {}
      point_t(const size_t<C> &a);

      template <typename Y> point_t(const point_t<Y> &a) {
        x = C(a.x);
        y = C(a.y);
      }

      point_t &operator=(const point_t &a) {
        x = a.x;
        y = a.y;
        return *this;
      }

#ifdef WIN32
      point_t(const LPPOINT pp) {
        x = pp->x;
        y = pp->y;
      }
      point_t(const POINT &p) {
        x = p.x;
        y = p.y;
      }
#endif

#ifdef GRAPHAPP
      point_t(const Point &p) : x(p.x), y(p.y) {}
      operator Point() {
        Point p;
        p.x = x;
        p.y = y;
        return p;
      }
#endif

      point_t &operator++() {
        ++x;
        ++y;
        return *this;
      }
      point_t &operator--() {
        --x;
        --y;
        return *this;
      }
      point_t operator++(int) {
        point_t t(x++, y++);
        return t;
      }
      point_t operator--(int) {
        point_t t(x--, y--);
        return t;
      }

      point_t &operator+=(C offset) {
        x += offset;
        y += offset;
        return *this;
      }
      point_t &operator-=(C offset) {
        x -= offset;
        y -= offset;
        return *this;
      }

      template<typename PT>
      point_t &operator+=(const PT &offset) {
        x += offset.x;
        y += offset.y;
        return *this;
      }
      template<typename PT>
      point_t &operator-=(const PT &offset) {
        x -= offset.x;
        y -= offset.y;
        return *this;
      }

      point_t &operator*=(C offset) {
        x *= offset;
        y *= offset;
        return *this;
      }
      point_t &operator/=(C offset) {
        x /= offset;
        y /= offset;
        return *this;
      }

      point_t &operator*=(size_t<C> mul) {
        x *= mul.x;
        y *= mul.y;
        return *this;
      }
      point_t &operator/=(size_t<C> mul) {
        x /= mul.x;
        y /= mul.y;
        return *this;
      }
      point_t &operator*=(point_t<C> mul) {
        x *= mul.x;
        y *= mul.y;
        return *this;
      }
      point_t &operator/=(point_t<C> mul) {
        x /= mul.x;
        y /= mul.y;
        return *this;
      }

      point_t operator-() const { return point_t<C>(-x, -y); }

      point_t &operator&=(const rect_t<C> &bounds);

      point_t operator&(const rect_t<C> &bounds) const;

      C dot(point_t p) const { return x * p.x + y * p.y; }
    };

    template <typename C>
    bool operator==(const point_t<C> &a, const point_t<C> &b) {
      return (a.x == b.x) && (a.y == b.y);
    }
    template <typename C>
    bool operator!=(const point_t<C> &a, const point_t<C> &b) {
      return !(a == b);
    }

    template <typename C>
    inline point_t<C> operator+(const point_t<C> &a, C b) {
      return point_t<C>(a.x + b, a.y + b);
    }
    template <typename C>
    inline point_t<C> operator+(C b, const point_t<C> &a) {
      return point_t<C>(a.x + b, a.y + b);
    }
    template <typename C, typename C1>
    inline point_t<C> operator+(const point_t<C> &a, const point_t<C1> &b) {
      return point_t<C>(a.x + b.x, a.y + b.y);
    }
    template <typename C>
    inline point_t<C> operator-(const point_t<C> &a, C b) {
      return point_t<C>(a.x - b, a.y - b);
    }
    template <typename C>
    inline point_t<C> operator-(C b, const point_t<C> &a) {
      return point_t<C>(a.x - b, a.y - b);
    }
    template <typename C, typename C1>
    inline point_t<C> operator-(const point_t<C> &a, const point_t<C1> &b) {
      return point_t<C>(a.x - b.x, a.y - b.y);
    }
    // scalar multiplication of two vectors:
    template <typename C>
    inline C operator*(const point_t<C> a, const point_t<C> &b) {
      return a.x * b.x + a.y * b.y;
    }
    // multiplication of vector by scalar:
    template <typename C>
    inline point_t<C> operator*(C b, const point_t<C> &a) {
      return point_t<C>(a.x * b, a.y * b);
    }
    template <typename C>
    inline point_t<C> operator*(const point_t<C> &a, C b) {
      return point_t<C>(a.x * b, a.y * b);
    }

    template <typename C>
    inline point_t<C> operator/(const point_t<C> &a, C b) {
      return point_t<C>(a.x / b, a.y / b);
    }
    template <typename C>
    inline point_t<C> operator/(const point_t<C> &a, const point_t<C> &b) {
      return point_t<C>(a.x / b.x, a.y / b.y);
    }

    template <typename C> class size_t {
    public:
      C x, y;

      size_t() : x(), y() {}
      size_t(C xx, C yy) {
        x = xx;
        y = yy;
      }
      size_t(const size_t<C> &a) {
        x = a.x;
        y = a.y;
      }
      size_t(C xy) {
        x = xy;
        y = xy;
      }
      size_t(const point_t<C> &a) {
        x = a.x;
        y = a.y;
      }
      template <typename Y> size_t(const size_t<Y> &a) {
        x = C(a.x);
        y = C(a.y);
      }

#ifdef WIN32
      size_t(const LPSIZE pp) {
        x = pp->cx;
        y = pp->cy;
      }
      size_t(const SIZE &p) {
        x = p.cx;
        y = p.cy;
      }
#endif
      size_t &operator=(const size_t<C> &a) {
        x = a.x;
        y = a.y;
        return *this;
      }
      size_t &operator=(const point_t<C> &a) {
        x = a.x;
        y = a.y;
        return *this;
      }

      size_t operator-() const {
        size_t v;
        v.x = -x;
        v.y = -y;
        return v;
      }

      size_t &operator++() {
        x++;
        y++;
        return *this;
      }
      size_t &operator--() {
        x--;
        y--;
        return *this;
      }
      size_t operator++(int) {
        size_t t(*this);
        x++;
        y++;
        return t;
      }
      size_t operator--(int) {
        size_t t(*this);
        x--;
        y--;
        return t;
      }

      size_t &operator+=(C offset) {
        x += offset;
        y += offset;
        return *this;
      }
      size_t &operator-=(C offset) {
        x -= offset;
        y -= offset;
        return *this;
      }
      size_t &operator+=(const size_t &offset) {
        x += offset.x;
        y += offset.y;
        return *this;
      }
      size_t &operator-=(const size_t &offset) {
        x -= offset.x;
        y -= offset.y;
        return *this;
      }

      size_t &operator*=(C mul) {
        x *= mul;
        y *= mul;
        return *this;
      }
      size_t &operator/=(C div) {
        x /= div;
        y /= div;
        return *this;
      }

      size_t &operator*=(const size_t & mul) {
        x *= mul.x;
        y *= mul.y;
        return *this;
      }
      size_t &operator/=(const size_t & div) {
        x /= div.x;
        y /= div.y;
        return *this;
      }

      size_t normalize() const { return size_t(max(0, x), max(0, y)); }
      bool   empty() const { return x <= 0 || y <= 0; }

      float length() const { return std::sqrt(x * x + y * y); }

      //#ifdef USE_D2D
      //  operator D2D_SIZE_U() const { return *(D2D_SIZE_U*)this; }
      //#endif
    };

    template <typename C>
    bool operator==(const size_t<C> &a, const size_t<C> &b) {
      return (a.x == b.x) && (a.y == b.y);
    }
    template <typename C>
    bool operator!=(const size_t<C> &a, const size_t<C> &b) {
      return (a.x != b.x) || (a.y != b.y);
    }

    template <typename C>
    bool operator>(const size_t<C> &a, const size_t<C> &b) {
      return (a.x > b.x) || (a.y > b.y);
    }
    template <typename C>
    bool operator>=(const size_t<C> &a, const size_t<C> &b) {
      return (a.x >= b.x) || (a.y >= b.y);
    }
    template <typename C>
    bool operator<(const size_t<C> &a, const size_t<C> &b) {
      return (a.x < b.x) || (a.y < b.y);
    }
    template <typename C>
    bool operator<=(const size_t<C> &a, const size_t<C> &b) {
      return (a.x <= b.x) || (a.y <= b.y);
    }

    template <typename C>
    inline point_t<C> operator+(const point_t<C> &a, const size_t<C> &b) {
      return point_t<C>(a.x + b.x, a.y + b.y);
    }
    template <typename C>
    inline point_t<C> operator+(const size_t<C> &a, const point_t<C> &b) {
      return point_t<C>(a.x + b.x, a.y + b.y);
    }
    template <typename C>
    inline point_t<C> operator-(const point_t<C> &a, const size_t<C> &b) {
      return point_t<C>(a.x - b.x, a.y - b.y);
    }

    template <typename C>
    inline size_t<C> operator-(const size_t<C> &a, const size_t<C> &b) {
      return size_t<C>(a.x - b.x, a.y - b.y);
    }
    template <typename C>
    inline size_t<C> operator+(const size_t<C> &a, const size_t<C> &b) {
      return size_t<C>(a.x + b.x, a.y + b.y);
    }

    template <typename C> inline size_t<C> operator*(const size_t<C> &a, C b) {
      return size_t<C>(a.x * b, a.y * b);
    }
    template <typename C> inline size_t<C> operator*(C b, const size_t<C> &a) {
      return size_t<C>(a.x * b, a.y * b);
    }
    template <typename C> inline size_t<C> operator/(const size_t<C> &a, C b) {
      return size_t<C>(a.x / b, a.y / b);
    }

    template <typename C>
    inline size_t<C> operator*(const size_t<C> &a, const size_t<C> &b) {
      return size_t<C>(a.x * b.x, a.y * b.y);
    }
    template <typename C>
    inline size_t<C> operator/(const size_t<C> &a, const size_t<C> &b) {
      return size_t<C>(a.x / b.x, a.y / b.y);
    }

    template <typename C> inline point_t<C>::point_t(const size_t<C> &a) {
      x = a.x;
      y = a.y;
    }
        
    template <typename C> class range_t { //[s,e) range
    public:
      C s;   // start, included
      C e;   // end, not included

      range_t() : s(0), e(0) {}
      range_t(C ll, C hh) : s(ll), e(hh) {}

      bool empty() const { return (s >= e); }
      C    length() const { return (s < e) ? e - s : 0; }
      C    middle() const { return (s < e) ? ((s + e) / 2) : s; }
      void clear() { s = 0; e = 0; }
      C pointOf(C m, C d) const { return (s < e) ? (s + (e * m) / d) : s; }

      static range_t make(C start, C length) { return range_t(start, start + length); }

      inline range_t &operator-=(C offset) {
        s -= offset;
        e -= offset;
        return *this;
      }
      inline range_t &operator+=(C offset) {
        s += offset;
        e += offset;
        return *this;
      }
      inline range_t operator+(C offset) const {
        return range_t(s + offset, e + offset);
      }
      inline range_t operator-(C offset) const {
        return range_t(s - offset, e - offset);
      }
      inline range_t &operator&=(range_t<C> a) {
        s = max(s, a.s);
        e = min(e, a.e);
        return *this;
      }
      inline range_t &operator|=(range_t<C> b) {
        if (empty())
          *this = b;
        else if (!b.empty()) {
          s = min(s, b.s);
          e = max(e, b.e);
        }
        return *this;
      }

      bool covers(range_t other) const {
        assert(!other.empty());
        return (other & *this) == other;
      }
      bool contains(C v) const { return (v >= s) && (v <= e); }
      bool overlaps_with(range_t other) const {
        return (max(s, other.s)) <= (min(e, other.e));
      }
    };

    /*
      //range is not empty - true
      operator bool() const         { return (s <= e); }
      //range is empty - true
      bool operator!() const        { return (s < e); }
    */

    template <typename C>
    inline bool operator==(const range_t<C> &a, const range_t<C> &b) {
      return (a.s == b.s) && (a.e == b.e);
    }
    template <typename C>
    inline bool operator!=(const range_t<C> &a, const range_t<C> &b) {
      return (a.s != b.s) || (a.e != b.e);
    }

    template <typename C> 
    inline bool operator && (C a,range_t<C> b)
      { return (a >= b.s) && (a < b.e); } 
    
    template <typename C> 
    inline bool operator && (range_t<C> a,C b)               
      { return (b >= a.s) && (b < a.e); } 
    
    template <typename C>
      inline range_t<C> operator&(range_t<C> a, range_t<C> b) {
        return range_t<C>(max(a.s, b.s), min(a.e, b.e));
      }

    template <typename C>
      inline range_t<C> operator|(range_t<C> a, range_t<C> b) {
        return range_t<C>(min(a.s, b.s), max(a.e, b.e));
      }

    template <typename C> 
      inline bool operator&&(range_t<C> a, range_t<C> b) {
        return !(a & b).empty();
      }


    template <typename C> class rect_t {
    public:
      point_t<C> s; // start, a.k.a. origin, included
      point_t<C> e; // end, a.k.a. corner opposite to start, not included ?

      rect_t() {}
      rect_t(const rect_t &r) : s(r.s), e(r.e) {}

      template <typename Y>
      rect_t(const rect_t<Y> &r) : s(r.s), e(r.e) {}

      rect_t(C origin_x, C origin_y, C corner_x, C corner_y) : s(origin_x, origin_y), e(corner_x, corner_y) {}
      rect_t(const point_t<C> &o, const point_t<C> &c) : s(o), e(c) {}
      rect_t(const range_t<C> &x, const range_t<C> &y) : s(x.s, y.s), e(x.e, y.e) {}

#ifdef WIN32
      rect_t(const LPRECT pr) : s(pr->left, pr->top), e(pr->right, pr->bottom) {}
      rect_t(const RECT &r) : s(r.left, r.top), e(r.right, r.bottom) {}
#endif
      rect_t(const point_t<C> &o, const size_t<C> &s) : s(o), e(o + s) {}
      rect_t(const size_t<C> &s) : e(s) { }

      static rect_t make_xywh(C x, C y, C w, C h) {
        return make_ltrb(x, y, x + w, y + h);
      }
      static rect_t make_ltrb(C x1, C y1, C x2, C y2) {
        rect_t t; 
        t.s.x = x1;
        t.s.y = y1;
        t.e.x = x2;
        t.e.y = y2;
        return t;
      }


      range_t<C> x() const { return range_t<C>(s.x, e.x); }
      range_t<C> y() const { return range_t<C>(s.y, e.y); }

      C left() const { return s.x; }
      C right() const { return e.x; }
      C top() const { return s.y; }
      C bottom() const { return e.y; }

      C width() const { return e.x - s.x; }
      C height() const { return e.y - s.y; }

      C area() const { return width() * height(); }

      // in- or de- flate rect_t
      rect_t &operator<<=(C a) {
        s += a;
        e -= a;
        return *this;
      }
      rect_t &operator>>=(C a) {
        s -= a;
        e += a;
        return *this;
      }
      rect_t &operator>>=(const size_t<C> &a) {
        s -= a;
        e += a;
        return *this;
      }
      rect_t &operator<<=(const size_t<C> &a) {
        s += a;
        e -= a;
        return *this;
      }

      template <typename CB>
      rect_t &operator>>=(const rect_t<CB> &a) {
        s -= a.s;
        e += a.e;
        return *this;
      }

      template <typename CB>
      rect_t &operator<<=(const rect_t<CB> &a) {
        s += a.s;
        e -= a.e;
        return *this;
      }

      rect_t &operator++() {
        ++e;
        ++s;
        return *this;
      }
      rect_t &operator--() {
        --e;
        --s;
        return *this;
      }
      rect_t &operator+=(C offset) {
        s += offset;
        e += offset;
        return *this;
      }
      rect_t &operator-=(C offset) {
        s -= offset;
        e -= offset;
        return *this;
      }
      rect_t &operator+=(point_t<C> offset) {
        s += offset;
        e += offset;
        return *this;
      }
      rect_t &operator-=(point_t<C> offset) {
        s -= offset;
        e -= offset;
        return *this;
      }

      rect_t &operator*=(C offset) {
        s *= offset;
        e *= offset;
        return *this;
      }
      rect_t &operator/=(C offset) {
        s /= offset;
        e /= offset;
        return *this;
      }
      rect_t &operator*=(point_t<C> mul) {
        s *= mul;
        e *= mul;
        return *this;
      }
      rect_t &operator*=(size_t<C> mul) {
        s *= mul;
        e *= mul;
        return *this;
      }
      rect_t &operator/=(point_t<C> mul) {
        s /= mul;
        e /= mul;
        return *this;
      }
      rect_t &operator/=(size_t<C> mul) {
        s /= mul;
        e /= mul;
        return *this;
      }

      rect_t operator+(point_t<C> offset) const {
        rect_t<C> r(*this);
        r.s += offset;
        r.e += offset;
        return r;
      }
      rect_t operator-(point_t<C> offset) const {
        rect_t<C> r(*this);
        r.s -= offset;
        r.e -= offset;
        return r;
      }
      rect_t &operator+=(point_t<C> offset) const {
        s += offset;
        e += offset;
        return *this;
      }
      rect_t &operator-=(point_t<C> offset) const {
        s -= offset;
        e -= offset;
        return *this;
      }

      rect_t &operator&=(rect_t<C> border) {
        *this = rect_t<C>(x() & border.x(), y() & border.y());
        return *this;
      }
      rect_t &operator|=(rect_t<C> border) {
        if (empty())
          *this = border;
        else if (!border.empty())
          *this = rect_t<C>(x() | border.x(), y() | border.y());
        return *this;
      }
      rect_t &operator|=(point_t<C> pt) {
        if (empty())
          *this = rect_t(pt.x, pt.y, pt.x, pt.y);
        else
          *this = rect_t<C>(x() | range_t<C>(pt.x, pt.x),
                            y() | range_t<C>(pt.y, pt.y));
        return *this;
      }

      size_t<C> size() const { return size_t<C>(width(), height()); }
      size_t<C> dimension() const { return size_t<C>(width(), height()); }

      point_t<C> start() const { return s; }
      point_t<C> end() const { return e; }

      bool empty() const { return (x().empty() || y().empty()); }

      void set(const point_t<C> &o, const size_t<C> &s) {
        s   = o;
        e.x = o.x + s.x;
        e.y = o.y + s.y;
      }
      void set_empty() {
        s.x = s.y = 0;
        e.x = e.y = 0;
      }

      // returns point of rectangle. for WHICH values see numbers on num keypad.
      point_t<C> pointOf(int which) const;
      // sets point of rectangle. for WHICH values see numbers on num keypad.
      void pointOf(int which, const point_t<C> &value);

      void move(point_t<C> pos) {
        e.x = pos.x + e.x - s.x;
        e.y = pos.y + e.y - s.y;
        s   = pos;
      }

      static rect_t<C> normalized(const point_t<C> &o, const size_t<C> &s) {
        rect_t<C> r;
        if (s.x < 0) {
          r.e.x = o.x;
          r.s.x = o.x + s.x;
        } else {
          r.s.x = o.x;
          r.e.x = o.x + s.x;
        }
        if (s.y < 0) {
          r.e.y = o.y;
          r.s.y = o.y + s.y;
        } else {
          r.s.y = o.y;
          r.e.y = o.y + s.y;
        }
        return r;
      }

      bool covered_by(rect_t<C> outline) {
        if (s.x < outline.s.x) return false;
        if (s.y < outline.s.y) return false;
        if (e.x > outline.e.x) return false;
        if (e.y > outline.e.y) return false;
        return true;
      }

      void inscribe(rect_t<C> border) {
        point_t<C> pos(s);
        size_t<C>  sz  = size();
        size_t<C>  bsz = border.size();
        if (sz.x > bsz.x) e.x = s.x + bsz.x;
        if (sz.y > bsz.y) e.y = s.y + bsz.y;

        if (s.x < border.s.x)
          pos.x = border.s.x;
        else if (e.x > border.e.x)
          pos.x = border.e.x - (e.x - s.x);
        if (s.y < border.s.y)
          pos.y = border.s.y;
        else if (e.y > border.e.y)
          pos.y = border.e.y - (e.y - s.y);
        move(pos);
      }

      inline bool overlaps_with(const rect_t<C> &b) const {
        return x().overlaps_with(b.x()) && y().overlaps_with(b.y());
      }
      inline bool contains(point_t<C> &pt) const {
        return x().contains(pt.x) && y().contains(pt.y);
      }
    };

    template <typename C> inline bool operator==(rect_t<C> a, rect_t<C> b) {
      return (a.s == b.s) && (a.e == b.e);
    }
    template <typename C> inline bool operator!=(rect_t<C> a, rect_t<C> b) {
      return !(a == b);
    }

    // template <typename C> inline bool operator<(rect_t<C> a,rect_t<C> b)    {
    // size_t as(a.size()); size_t bs(b.size()); return as.x < bs.x && as.y <
    // bs.y); }  template <typename C> inline bool operator<=(rect_t<C>
    // a,rect_t<C> b)   { size_t as(a.size()); size_t bs(b.size()); return as.x
    // <= bs.x && as.y <= bs.y); }  template <typename C> inline bool
    // operator>(rect_t<C> a,rect_t<C> b)    { return !(a <= b); }  template
    // <typename C> inline bool operator>=(rect_t<C> a,rect_t<C> b)   { return
    // !(a < b); }

    // is point inside?
    // template <typename C> inline bool operator&&(point_t<C> a,rect_t<C> b)
    // { return (a.x && b.x()) && (a.y && b.y()); }  template <typename C> inline
    // bool operator&&(rect_t<C> a,point_t<C> b)    { return (a.x() && b.x) &&
    // (a.y() && b.y); }

    // do these rects have a non empty intersection?
    // template <typename C> inline bool operator&&(rect_t<C> a,rect_t<C> b)   {
    // return (a.x() && b.x()) && (a.y() && b.y()); }

    // intersection
    template <typename C> inline rect_t<C> operator&(rect_t<C> a, rect_t<C> b) {
      return rect_t<C>(a.x() & b.x(), a.y() & b.y());
    }
    // union
    template <typename C> inline rect_t<C> operator|(rect_t<C> a, rect_t<C> b) {
      return rect_t<C>(a.x() | b.x(), a.y() | b.y());
    }

    template <typename C> inline rect_t<C> operator<<(const rect_t<C> &a, C b) {
      rect_t<C> t(a);
      t <<= b;
      return t;
    }
    template <typename C> inline rect_t<C> operator>>(const rect_t<C> &a, C b) {
      rect_t<C> t(a);
      t >>= b;
      return t;
    }
    template <typename CA, typename CB>
    inline rect_t<CA> operator<<(const rect_t<CA> &a, const rect_t<CB> &b) {
      rect_t<CA> t(a);
      t <<= b;
      return t;
    }
    template <typename CA,typename CB>
    inline rect_t<CA> operator>>(const rect_t<CA> &a, const rect_t<CB> &b) {
      rect_t<CA> t(a);
      t >>= b;
      return t;
    }

    template <typename C>
    inline rect_t<C> operator<<(const rect_t<C> &a, size_t<C> b) {
      rect_t<C> t(a);
      t <<= b;
      return t;
    }
    template <typename C>
    inline rect_t<C> operator>>(const rect_t<C> &a, size_t<C> b) {
      rect_t<C> t(a);
      t >>= b;
      return t;
    }

    template <typename C>
    inline point_t<C> &point_t<C>::operator&=(const rect_t<C> &bounds) {
      if (x < bounds.s.x) x = bounds.s.x;
      if (y < bounds.s.y) y = bounds.s.y;
      if (x > bounds.e.x) x = bounds.e.x;
      if (y > bounds.e.y) y = bounds.e.y;
      return *this;
    }

    template <typename C>
    inline point_t<C> point_t<C>::operator&(const rect_t<C> &bounds) const {
      point_t t = *this;
      t &= bounds;
      return t;
    }

    // see mumpad number keys for meaning of which parameter
    template <typename C>
    inline point_t<C> rect_t<C>::pointOf(int which) const {
      switch (which) {
      case 7: return point_t<C>(s);
      case 3: return point_t<C>(e);
      case 1: return point_t<C>(s.x, e.y);
      case 9: return point_t<C>(e.x, s.y);
      case 8: return point_t<C>((e.x + s.x) / 2, s.y);
      case 2: return point_t<C>((e.x + s.x) / 2, e.y);
      case 4: return point_t<C>(s.x, (e.y + s.y) / 2);
      case 6: return point_t<C>(e.x, (e.y + s.y) / 2);
      case 5: return point_t<C>((e.x + s.x) / 2, (e.y + s.y) / 2);
      }
      return point_t<C>();
    }

    // move the rectangle. for WHICH values see numbers on num keypad
    // and comments inside.

    template <typename C>
    inline void rect_t<C>::pointOf(int which, const point_t<C> &v) {
      point_t<C> p;

      switch (which) {
      case 7:
        p = v;
        break; // this top-left == v
      case 3:
        p = v - dimension();
        break; // this bottom-right == v
      case 1:
        p = point_t<C>(v.x, v.y - height());
        break; // this bottom-left == v
      case 9:
        p = point_t<C>(v.x - width(), v.y);
        break; // this top-right == v
      case 8:
        p = point_t<C>(v.x - width() / 2, v.y);
        break; // this top-middle == v
      case 2:
        p = point_t<C>(v.x - width() / 2, v.y - height());
        break; // this bottom-middle == v
      case 4:
        p = point_t<C>(v.x, v.y - height() / 2);
        break; // this center-left == v
      case 6:
        p = point_t<C>(v.x - width(), v.y - height() / 2);
        break; // this center-right == v
      case 5:
        p = point_t<C>(v.x - width() / 2, v.y - height() / 2);
        break; // this center-middle == v
      default: assert(false); return;
      }
      move(p);
      /*

        switch(which) {
          case 7:
            s = v; break;
          case 3:
            e = v; break;
          case 1:
            s.x = v.x; e.y = v.y; break;
          case 9:
            e.x = v.x; s.y = v.y; break;
          case 8:
            s.y = v.y; break;
          case 2:
            e.y = v.y; break;
          case 4:
            s.x = v.x; break;
          case 6:
            e.x = v.x; break;
          case 5:
            break; // NOTHING!
        }
        */
    }

    template <typename C> class line_t {
    public:
      point_t<C> s, e;

      line_t(const point_t<C> &a, const point_t<C> &b) : s(a), e(b) { ; }

      range_t<C> x() const { return range_t<C>(s.x, e.x); }
      range_t<C> y() const { return range_t<C>(s.y, e.y); }

      point_t<C> unit() const; // vector
    };

    template <typename C> class polygon_t : public tool::array<point_t<C>> {
    public:
      typedef tool::array<point_t<C>> super;
      void add(const point_t<C> &pt) { super::push(pt); }
      void set(const rect_t<C> &rc) {
        super::push(rc.pointOf(7));
        super::push(rc.pointOf(9));
        super::push(rc.pointOf(3));
        super::push(rc.pointOf(1));
      }
      bool is_inside(const point_t<C> &pt) const;
    };

    // is point inside?
    template <typename C>
    inline bool operator&&(const point_t<C> a, const polygon_t<C> b) {
      return b.is_inside(a);
    }
    template <typename C>
    inline bool operator&&(const polygon_t<C> a, const point_t<C> b) {
      return a.is_inside(b);
    }

    template <typename C>
    inline bool polygon_t<C>::is_inside(const point_t<C> &pt) const {
      int hits    = 0;
      C   y_save  = 0;
      int i       = 0;
      int npoints = super::size();
      while (i < npoints && (*this)[i].y == pt.y) {
        i++;
      }

      for (int n = 0; n < npoints; n++) {
        int j  = (i + 1) % npoints;
        C   x1 = (*this)[i].x, y1 = (*this)[i].y;
        C   x2 = (*this)[j].x, y2 = (*this)[j].y;

        C dx = x2 - x1;
        C dy = y2 - y1;

        if (dy != 0) {
          C rx = pt.x - x1;
          C ry = pt.y - y1;

          if (y2 == pt.y && x2 >= pt.x) { y_save = y1; }
          if (y1 == pt.y && x1 >= pt.x) {
            if ((y_save > pt.y) != (y2 > pt.y)) { hits--; }
          }
          real s = ry / dy;
          if (s >= 0.0 && s <= 1.0 && (s * dx) >= rx) { hits++; }
        }
        i = j;
      }
      return (hits % 2) != 0;
    }

    /* lines_intersect:
     * http://tog.acm.org/resources/GraphicsGems/gemsii/xlines.c */

    enum INTERSECT_AT_RESULT {
      DONT_INTERSECT = 0,
      DO_INTERSECT   = 1,
      COLLINEAR      = 2
    };

    template <typename T> inline bool same_signs(T a, T b) {
      return (a < 0) == (b < 0);
    }

    template <typename TC>
    INTERSECT_AT_RESULT
    intersects_at(point_t<TC> p1, // First line segment
                  point_t<TC> p2,

                  point_t<TC> p3, // Second line segment
                  point_t<TC> p4,

                  point_t<TC> &pout // Output value: point of intersection
    ) {
      TC a1, a2, b1, b2, c1, c2; // Coefficients of line eqns.
      TC r1, r2, r3, r4;         // 'Sign' values
      TC denom, offset, num;     // Intermediate values

      /* Compute a1, b1, c1, where line joining points 1 and 2
       * is "a1 x  +  b1 y  +  c1  =  0".
       */

      a1 = p2.y - p1.y;
      b1 = p1.x - p2.x;
      c1 = p2.x * p1.y - p1.x * p2.y;

      /* Compute r3 and r4.
       */

      r3 = a1 * p3.x + b1 * p3.y + c1;
      r4 = a1 * p4.x + b1 * p4.y + c1;

      /* Check signs of r3 and r4.  If both point 3 and point 4 lie on
       * same side of line 1, the line segments do not intersect.
       */

      if (r3 != 0 && r4 != 0 && same_signs(r3, r4)) return DONT_INTERSECT;

      /* Compute a2, b2, c2 */

      a2 = p4.y - p3.y;
      b2 = p3.x - p4.x;
      c2 = p4.x * p3.y - p3.x * p4.y;

      /* Compute r1 and r2 */

      r1 = a2 * p1.x + b2 * p1.y + c2;
      r2 = a2 * p2.x + b2 * p2.y + c2;

      /* Check signs of r1 and r2.  If both point 1 and point 2 lie
       * on same side of second line segment, the line segments do
       * not intersect.
       */

      if (r1 != 0 && r2 != 0 && same_signs(r1, r2)) return (DONT_INTERSECT);

      /* Line segments intersect: compute intersection point.
       */

      denom = a1 * b2 - a2 * b1;
      if (denom == 0) return (COLLINEAR);
      offset = denom < 0 ? -denom / 2 : denom / 2;

      /* The denom/2 is to get rounding instead of truncating.  It
       * is added or subtracted to the numerator, depending upon the
       * sign of the numerator.
       */

      num    = b1 * c2 - b2 * c1;
      pout.x = (num < 0 ? num - offset : num + offset) / denom;

      num    = a2 * c1 - a1 * c2;
      pout.y = (num < 0 ? num - offset : num + offset) / denom;

      return DO_INTERSECT;
    } // intersect_at

    template <typename TC>
    INTERSECT_AT_RESULT
    intersects_at(line_t<TC>   l1, // First line segment
                  line_t<TC>   l2,
                  point_t<TC> &pout // Output value: point of intersection
    ) {
      return intersects_at(l1.origin, l1.end, l2.origin, l2.end, pout);
    }

  } // namespace geom

  // integer geometry
  typedef geom::point_t<int>   point;
  typedef geom::size_t<int>    size;
  typedef geom::range_t<int>   range;
  typedef geom::rect_t<int>    rect;
  typedef geom::line_t<int>    line;
  typedef geom::polygon_t<int> polygon;

  typedef geom::point_t<int16>   point16;
  typedef geom::size_t<int16>    size16;
  typedef geom::range_t<int16>   range16;
  typedef geom::rect_t<int16>    rect16;
  typedef geom::line_t<int16>    line16;
  typedef geom::polygon_t<int16> polygon16;

  typedef geom::point_t<float>   pointf;
  typedef geom::size_t<float>    sizef;
  typedef geom::range_t<float>   rangef;
  typedef geom::rect_t<float>    rectf;
  typedef geom::line_t<float>    linef;
  typedef geom::polygon_t<float> polygonf;

  typedef geom::point_t<double>   pointd;
  typedef geom::size_t<double>    sized;
  typedef geom::range_t<double>   ranged;
  typedef geom::rect_t<double>    rectd;
  typedef geom::line_t<double>    lined;
  typedef geom::polygon_t<double> polygond;

  typedef geom::point_t<real>   pointr;
  typedef geom::size_t<real>    sizer;
  typedef geom::range_t<real>   ranger;
  typedef geom::rect_t<real>    rectr;
  typedef geom::line_t<real>    liner;
  typedef geom::polygon_t<real> polygonr;

  template <typename C>
  inline C distance2(geom::point_t<C> p1, geom::point_t<C> p2) {
    C x = p2.x - p1.x;
    C y = p2.y - p1.y;
    return x * x + y * y;
  }

  inline int distance(point p1, point p2) {
    int x = p2.x - p1.x;
    int y = p2.y - p1.y;
    return (int)isqrt(x * x + y * y);
  }
  inline int16 distance(point16 p1, point16 p2) {
    int x = p2.x - p1.x;
    int y = p2.y - p1.y;
    return (int16)isqrt(x * x + y * y);
  }
  inline float distance(pointf p1, pointf p2) {
    float x = p2.x - p1.x;
    float y = p2.y - p1.y;
    return (float)sqrt(x * x + y * y);
  }
  inline double distance(pointd p1, pointd p2) {
    double x = p2.x - p1.x;
    double y = p2.y - p1.y;
    return (double)sqrt(x * x + y * y);
  }

  template <typename C> // unit vector - vector having length 1 along the line.
  inline geom::point_t<C> geom::line_t<C>::unit() const {
    range_t<C> px = x();
    range_t<C> py = y();
    C          z  = distance(s, e);
    return geom::point_t<C>((px.e - px.s) / z, (py.e - py.s) / z);
  }

  // return minimum distance between line segment vw and point p, and the
  // closest point on the line segment, q
  template <typename C>
  inline C distance(geom::point_t<C> v, geom::point_t<C> w, geom::point_t<C> p,
                    geom::point_t<C> &q) {

    if (v == w) // v == w case
    {
      q = v;
      return distance(v, p);
    }

    float distSq = (float)distance2(v, w);

    // consider the line extending the segment, parameterized as v + t (w - v)
    // we find projection of point p onto the line
    // it falls where t = [(p-v) . (w-v)] / |w-v|^2

    float t = (p - v).dot(w - v) / distSq;
    if (t < 0) {
      // beyond the v end of the segment
      q = v;
      return distance(v, p);
    } else if (t > 1.0f) {
      // beyond the w end of the segment
      q = w;
      return distance(w, p);
    }
    // projection falls on the segment
    geom::point_t<C> projection = v + ((w - v) * C(t * 1000)) / 1000;
    q                           = projection;

    return distance(p, projection);
  }

  // return minimum distance between line segment vw and point p
  template <typename C>
  inline C distance(geom::point_t<C> v, geom::point_t<C> w,
                    geom::point_t<C> p) {
    geom::point_t<C> q;
    return distance(v, w, p, q);
  }

  // minimum distance between rect rc and point p
  template <typename C>
  inline C distance(geom::rect_t<C> rc, geom::point_t<C> p) {
    if (rc.contains(p)) return 0;
    return min(distance(rc.pointOf(7), rc.pointOf(9), p),
               distance(rc.pointOf(9), rc.pointOf(3), p),
               distance(rc.pointOf(3), rc.pointOf(1), p),
               distance(rc.pointOf(1), rc.pointOf(7), p));
  }

#if defined(SCAPI) || defined(WINDOWS)
#define PPOINT(p) ((LPPOINT)&p)
#define PRECT(r) ((LPRECT)&r)
#define PSIZE(s) ((LPSIZE)&s)
  inline POINT toPOINT(const point &pt) { return *PPOINT(pt); }
  inline RECT  toRECT(const rect &rc) {
    RECT r;
    r.left   = rc.left();
    r.top    = rc.top();
    r.right  = rc.right();
    r.bottom = rc.bottom();
    return r;
  }

  inline rect fromRECT(const RECT &r) {
    return rect::make_ltrb(r.left, r.top, r.right, r.bottom);
  }
#endif

} // namespace gool

#endif
