//|
//|
//| Copyright (c) 2001-2010
//| Andrew Fedoniouk - andrew@terrainformatica.com
//|
//| graphics primitives
//|
//|

#ifndef __gool_figures_h__
#define __gool_figures_h__

#include "tool/tool.h"
#include "gool-types.h"
#include "gool-geometry.h"
#include "gool-affine.h"

// various polygon generators

namespace gool {
  using namespace tool;

  // borrowed from AGG

  //---------------------------------------------------------path_commands_e
  enum path_commands_e {
    path_cmd_stop     = 0,    //----path_cmd_stop
    path_cmd_move_to  = 1,    //----path_cmd_move_to
    path_cmd_line_to  = 2,    //----path_cmd_line_to
    path_cmd_curve3   = 3,    //----path_cmd_curve3
    path_cmd_curve4   = 4,    //----path_cmd_curve4
    path_cmd_curveN   = 5,    //----path_cmd_curveN
    path_cmd_catrom   = 6,    //----path_cmd_catrom
    path_cmd_ubspline = 7,    //----path_cmd_ubspline
    path_cmd_end_poly = 0x0F, //----path_cmd_end_poly
    path_cmd_mask     = 0x0F  //----path_cmd_mask
  };

  //------------------------------------------------------------path_flags_e
  enum path_flags_e {
    path_flags_none  = 0,    //----path_flags_none
    path_flags_ccw   = 0x10, //----path_flags_ccw
    path_flags_cw    = 0x20, //----path_flags_cw
    path_flags_close = 0x40, //----path_flags_close
    path_flags_mask  = 0xF0  //----path_flags_mask
  };

  inline bool is_vertex(unsigned c) {
    return c >= path_cmd_move_to && c < path_cmd_end_poly;
  }
  inline bool is_drawing(unsigned c) {
    return c >= path_cmd_line_to && c < path_cmd_end_poly;
  }
  inline bool is_stop(unsigned c) { return c == path_cmd_stop; }
  inline bool is_move_to(unsigned c) { return c == path_cmd_move_to; }
  inline bool is_line_to(unsigned c) { return c == path_cmd_line_to; }
  inline bool is_curve(unsigned c) {
    return c == path_cmd_curve3 || c == path_cmd_curve4;
  }
  inline bool is_curve3(unsigned c) { return c == path_cmd_curve3; }
  inline bool is_curve4(unsigned c) { return c == path_cmd_curve4; }
  inline bool is_end_poly(unsigned c) {
    return (c & path_cmd_mask) == path_cmd_end_poly;
  }
  inline bool is_close(unsigned c) {
    return (c & ~(path_flags_cw | path_flags_ccw)) ==
           (path_cmd_end_poly | path_flags_close);
  }
  inline bool is_next_poly(unsigned c) {
    return is_stop(c) || is_move_to(c) || is_end_poly(c);
  }
  inline bool is_cw(unsigned c) { return (c & path_flags_cw) != 0; }
  inline bool is_ccw(unsigned c) { return (c & path_flags_ccw) != 0; }
  inline bool is_oriented(unsigned c) {
    return (c & (path_flags_cw | path_flags_ccw)) != 0;
  }
  inline bool is_closed(unsigned c) { return (c & path_flags_close) != 0; }

  inline unsigned get_close_flag(unsigned c) { return c & path_flags_close; }
  inline unsigned clear_orientation(unsigned c) {
    return c & ~(path_flags_cw | path_flags_ccw);
  }
  inline unsigned get_orientation(unsigned c) {
    return c & (path_flags_cw | path_flags_ccw);
  }
  inline unsigned set_orientation(unsigned c, unsigned o) {
    return clear_orientation(c) | o;
  }

  template <class Vertex_Source>
  inline void produce_path(Vertex_Source &vs, polygonf &pg) {
    pointf pt;
    vs.rewind();
    unsigned cmd = vs.vertex(&pt.x, &pt.y);
    assert(is_move_to(cmd));
    pg.add(pt);
    while (!is_stop(cmd = vs.vertex(&pt.x, &pt.y))) {
      if (is_vertex(cmd))
        pg.add(pt);
      else {
        // assert(false);
        break;
      }
    }
  }

  //--------------------------------------------------------------------
  // append path. The path is joined with the existing one, that is,
  // it behaves as if the pen of a plotter was always down (drawing)
  template <class VertexSource, class Path>
  void append_path(VertexSource &vs, Path *dst) {
    const float vertex_dist_epsilon = 0.00000001f;

    pointf buffer[4];
    uint   buffer_cnt = 0;
    uint   cnt        = 0;

    auto last_vertex = [&](pointf &vx) -> uint {
      return dst->last(vx) ? path_cmd_line_to : path_cmd_stop;
    };

    auto add_vertex = [&](pointf vx, uint cmd) {
      switch (cmd) {
      case path_cmd_stop: assert(false); break;
      case path_cmd_move_to:
        if (cnt++ == 0)
          dst->line_to(vx);
        else
          dst->move_to(vx);
        buffer_cnt = 0;
        break;
      case path_cmd_line_to:
        dst->line_to(vx);
        buffer_cnt = 0;
        break;
      case path_cmd_curve3:
        buffer[buffer_cnt] = vx;
        if (++buffer_cnt == 2) {
          dst->quadratic_to(buffer[1], buffer[0]);
          buffer_cnt = 0;
        }
        break;
      case path_cmd_curve4:
        buffer[buffer_cnt] = vx;
        if (++buffer_cnt == 3) {
          dst->cubic_to(buffer[2], buffer[0], buffer[1]);
          buffer_cnt = 0;
        }
        break;
      default:
        assert(false);
        break;
        // case path_cmd_curve3:
      }
    };

    pointf   p;
    unsigned cmd;
    vs.rewind(0);
    cmd = vs.vertex(&p.x, &p.y);
    if (!is_stop(cmd)) {
      if (is_vertex(cmd)) {
        pointf   p0;
        unsigned cmd0 = last_vertex(p0);
        if (is_vertex(cmd0)) {
          if (distance(p, p0) > vertex_dist_epsilon) {
            if (is_move_to(cmd)) cmd = path_cmd_line_to;
            add_vertex(p, cmd);
          }
        } else {
          if (is_stop(cmd0)) {
            cmd = path_cmd_move_to;
          } else {
            if (is_move_to(cmd)) cmd = path_cmd_line_to;
          }
          add_vertex(p, cmd);
        }
      }
      while (!is_stop(cmd = vs.vertex(&p.x, &p.y))) {
        add_vertex(p, is_move_to(cmd) ? unsigned(path_cmd_line_to) : cmd);
      }
      buffer_cnt = buffer_cnt;
    }
  }

  class arc {
  public:
    arc() : _scale(1.0f), _initialized(false) {}
    arc(float x, float y, float rx, float ry, float a1, float a2,
        bool ccw = true);

    void init(float x, float y, float rx, float ry, float a1, float a2,
              bool ccw = true);

    void  approximation_scale(float s);
    float approximation_scale() const { return _scale; }

    void     rewind();
    unsigned vertex(float *x, float *y);

  private:
    void normalize(float a1, float a2, bool ccw);

    float    _x;
    float    _y;
    float    _rx;
    float    _ry;
    float    _angle;
    float    _start;
    float    _end;
    float    _scale;
    float    _da;
    bool     _ccw;
    bool     _initialized;
    unsigned _path_cmd;
  };

  class plain_rect {
  public:
    plain_rect() {}
    plain_rect(float x1, float y1, float x2, float y2);

    void rect(float x1, float y1, float x2, float y2);

    void     rewind();
    unsigned vertex(float *x, float *y);

  private:
    float    _x1;
    float    _y1;
    float    _x2;
    float    _y2;
    unsigned _status;
  };

  class rounded_rect {
  public:
    rounded_rect() {}
    rounded_rect(float x1, float y1, float x2, float y2, float r);

    void rect(float x1, float y1, float x2, float y2);
    void radius(float r);
    void radius(float rx, float ry);
    void radius(float rx_bottom, float ry_bottom, float rx_top, float ry_top);
    void radius(float rx1, float ry1, float rx2, float ry2, float rx3,
                float ry3, float rx4, float ry4);
    void normalize_radius();

    void  approximation_scale(float s) { _arc.approximation_scale(s); }
    float approximation_scale() const { return _arc.approximation_scale(); }

    void     rewind();
    unsigned vertex(float *x, float *y);

  private:
    float    _x1;
    float    _y1;
    float    _x2;
    float    _y2;
    float    _rx1;
    float    _ry1;
    float    _rx2;
    float    _ry2;
    float    _rx3;
    float    _ry3;
    float    _rx4;
    float    _ry4;
    unsigned _status;
    arc      _arc;
  };

  //-----------------------------------------------------------------------
  void arc_to_bezier(float cx, float cy, float rx, float ry, float start_angle,
                     float sweep_angle, float *curve);

  // arc approximation by bezier curve

  class bezier_arc {
  public:
    //--------------------------------------------------------------------
    bezier_arc() : m_vertex(26), m_num_vertices(0), m_cmd(path_cmd_line_to) {}
    bezier_arc(float x, float y, float rx, float ry, float start_angle,
               float sweep_angle) {
      init(x, y, rx, ry, start_angle, sweep_angle);
    }

    //--------------------------------------------------------------------
    void init(float x, float y, float rx, float ry, float start_angle,
              float sweep_angle);

    //--------------------------------------------------------------------
    void rewind(unsigned) { m_vertex = 0; }

    //--------------------------------------------------------------------
    unsigned vertex(float *x, float *y) {
      if (m_vertex >= m_num_vertices) return path_cmd_stop;
      *x = m_vertices[m_vertex];
      *y = m_vertices[m_vertex + 1];
      m_vertex += 2;
      return (m_vertex == 2) ? path_cmd_move_to : m_cmd;
    }

    // Supplemantary functions. num_vertices() actually returns doubled
    // number of vertices. That is, for 1 vertex it returns 2.
    //--------------------------------------------------------------------
    unsigned     num_vertices() const { return m_num_vertices; }
    const float *vertices() const { return m_vertices; }
    float *      vertices() { return m_vertices; }

  private:
    unsigned m_vertex;
    unsigned m_num_vertices;
    float    m_vertices[26];
    unsigned m_cmd;
  };

  //==========================================================bezier_arc_svg
  // Compute an SVG-style bezier arc.
  //
  // Computes an elliptical arc from (x1, y1) to (x2, y2). The size and
  // orientation of the ellipse are defined by two radii (rx, ry)
  // and an x-axis-rotation, which indicates how the ellipse as a whole
  // is rotated relative to the current coordinate system. The center
  // (cx, cy) of the ellipse is calculated automatically to satisfy the
  // constraints imposed by the other parameters.
  // large-arc-flag and sweep-flag contribute to the automatic calculations
  // and help determine how the arc is drawn.

  class bezier_arc_svg {
  public:
    //--------------------------------------------------------------------
    bezier_arc_svg() : m_arc(), m_radii_ok(false) {}

    bool calc(float x1, float y1, float rx, float ry, float angle,
              bool large_arc_flag, bool sweep_flag, float x2, float y2);

    //--------------------------------------------------------------------
    bool radii_ok() const { return m_radii_ok; }

    //--------------------------------------------------------------------
    void rewind(unsigned) { m_arc.rewind(0); }

    //--------------------------------------------------------------------
    unsigned vertex(float *x, float *y) { return m_arc.vertex(x, y); }

    // Supplemantary functions. num_vertices() actually returns doubled
    // number of vertices. That is, for 1 vertex it returns 2.
    //--------------------------------------------------------------------
    unsigned     num_vertices() const { return m_arc.num_vertices(); }
    const float *vertices() const { return m_arc.vertices(); }
    float *      vertices() { return m_arc.vertices(); }

  private:
    bezier_arc m_arc;
    bool       m_radii_ok;
  };

  class rect_placement {
  public:
    rect_placement(int placement_flags) : flags(placement_flags) {}
    rect_placement() : flags(centred) {}

    rect_placement(const rect_placement &other) : flags(other.flags) {}
    rect_placement &operator=(const rect_placement &other) {
      flags = other.flags;
      return *this;
    }
    bool operator==(const rect_placement &other) const {
      return flags == other.flags;
    }
    bool operator!=(const rect_placement &other) const {
      return flags != other.flags;
    }

    // Flag values that can be combined and used in the constructor.
    enum FLAGS {
      //==============================================================================
      /** Indicates that the source rectangle's left edge should be aligned with
         the left edge of the target rectangle. */
      xLeft = 1,

      /** Indicates that the source rectangle's right edge should be aligned
         with the right edge of the target rectangle. */
      xRight = 2,

      /** Indicates that the source should be placed in the centre between the
         left and right sides of the available space. */
      xMid = 4,

      //==============================================================================
      /** Indicates that the source's top edge should be aligned with the top
         edge of the destination rectangle. */
      yTop = 8,

      /** Indicates that the source's bottom edge should be aligned with the
         bottom edge of the destination rectangle. */
      yBottom = 16,

      /** Indicates that the source should be placed in the centre between the
         top and bottom sides of the available space. */
      yMid = 32,

      //==============================================================================
      /** If this flag is set, then the source rectangle will be resized to
         completely fill the destination rectangle, and all other flags are
         ignored.
      */
      stretchToFit = 64,

      //==============================================================================
      /** If this flag is set, then the source rectangle will be resized so that
         it is the minimum size to completely fill the destination rectangle,
         without changing its aspect ratio. This means that some of the source
         rectangle may fall outside the destination.

          If this flag is not set, the source will be given the maximum size at
         which none of it falls outside the destination rectangle.
      */
      fillDestination = 128,

      /** Indicates that the source rectangle can be reduced in size if
         required, but should never be made larger than its original size.
      */
      onlyReduceInSize = 256,

      /** Indicates that the source rectangle can be enlarged if required, but
         should never be made smaller than its original size.
      */
      onlyIncreaseInSize = 512,

      /** Indicates that the source rectangle's size should be left unchanged.
       */
      doNotResize = (onlyIncreaseInSize | onlyReduceInSize),

      //==============================================================================
      /** A shorthand value that is equivalent to (xMid | yMid). */
      centred = 4 + 32
    };

    int get_flags() const { return flags; }

    bool test_flags(int flagsToTest) const {
      return (flags & flagsToTest) != 0;
    }

    //==============================================================================
    /** Adjusts the position and size of a rectangle to fit it into a space.

        The source rectangle coordinates will be adjusted so that they fit into
        the destination rectangle based on this object's flags.
    */
    template <typename REAL>
    void apply_to(REAL &x, REAL &y, REAL &w, REAL &h, REAL dx, REAL dy, REAL dw,
                  REAL dh) const {

      if (w == 0 || h == 0) return;

      if ((flags & stretchToFit) != 0) {
        x = dx;
        y = dy;
        w = dw;
        h = dh;
      } else {
        REAL scale = (flags & fillDestination) != 0 ? max(dw / w, dh / h)
                                                    : min(dw / w, dh / h);

        if ((flags & onlyReduceInSize) != 0) scale = min(scale, 1.0);

        if ((flags & onlyIncreaseInSize) != 0) scale = max(scale, 1.0);

        w *= scale;
        h *= scale;

        if ((flags & xLeft) != 0)
          x = dx;
        else if ((flags & xRight) != 0)
          x = dx + dw - w;
        else
          x = dx + (dw - w) / 2;

        if ((flags & yTop) != 0)
          y = dy;
        else if ((flags & yBottom) != 0)
          y = dy + dh - h;
        else
          y = dy + (dh - h) / 2;
      }
    }

    /** Returns the rectangle that should be used to fit the given source
       rectangle into the destination rectangle using the current flags.
    */
    template <typename REAL>
    geom::rect_t<REAL> applied_to(const geom::rect_t<REAL> &source,
                                  const geom::rect_t<REAL> &destination) const {
      REAL x = source.left(), y = source.top(), w = source.width(),
           h = source.height();
      apply_to(x, y, w, h, destination.left(), destination.top(),
               destination.width(), destination.height());
      return geom::rect_t<REAL>(x, y, x + w - 1, y + h - 1);
    }

    /** Returns the transform that should be applied to these source coordinates
       to fit them into the destination rectangle using the current flags.
    */
    template <typename REAL>
    geom::trans_affine<REAL>
    get_transform_to_fit(const geom::rect_t<REAL> &source,
                         const geom::rect_t<REAL> &destination) const {
      if (source.empty()) return geom::trans_affine<REAL>();

      REAL newX = destination.left();
      REAL newY = destination.top();

      REAL scaleX = destination.width() / source.width();
      REAL scaleY = destination.height() / source.height();

      if ((flags & stretchToFit) == 0) {
        scaleX = (flags & fillDestination) != 0 ? max(scaleX, scaleY)
                                                : min(scaleX, scaleY);

        if ((flags & onlyReduceInSize) != 0) scaleX = min(scaleX, 1.0f);

        if ((flags & onlyIncreaseInSize) != 0) scaleX = max(scaleX, 1.0f);

        scaleY = scaleX;

        if ((flags & xRight) != 0)
          newX += destination.width() - source.width() * scaleX; // right
        else if ((flags & xLeft) == 0)
          newX += (destination.width() - source.width() * scaleX) /
                  REAL(2.0); // centre

        if ((flags & yBottom) != 0)
          newY += destination.height() - source.height() * scaleX; // bottom
        else if ((flags & yTop) == 0)
          newY += (destination.height() - source.height() * scaleX) /
                  REAL(2.0f); // centre
      }
      geom::trans_affine<REAL> mx =
          geom::trans_affine_translation<REAL>(-source.left(), -source.top());
      mx *= geom::trans_affine_scaling<REAL>(scaleX, scaleY);
      mx *= geom::trans_affine_translation<REAL>(newX, newY);
      return mx;
    }

  private:
    int flags;
  };

} // namespace gool

#endif
