#include "gool.h"

namespace gool {

  const float pi = 3.14159265358979323846f;

  arc::arc(float x, float y, float rx, float ry, float a1, float a2, bool ccw)
      : _x(x), _y(y), _rx(rx), _ry(ry), _scale(2.0f) {
    rewind();
    normalize(a1, a2, ccw);
  }

  void arc::init(float x, float y, float rx, float ry, float a1, float a2,
                 bool ccw) {
    _x  = x;
    _y  = y;
    _rx = rx;
    _ry = ry;
    normalize(a1, a2, ccw);
  }

  void arc::approximation_scale(float s) {
    _scale = s;
    if (_initialized) { normalize(_start, _end, _ccw); }
  }

  void arc::rewind() {
    _path_cmd = path_cmd_move_to;
    _angle    = _start;
  }

  //------------------------------------------------------------------------
  unsigned arc::vertex(float *x, float *y) {
    if (is_stop(_path_cmd)) return path_cmd_stop;
    if ((_angle < _end - _da / 4) != _ccw) {
      *x        = _x + cosf(_end) * _rx;
      *y        = _y + sinf(_end) * _ry;
      _path_cmd = path_cmd_stop;
      return path_cmd_line_to;
    }

    *x = _x + cosf(_angle) * _rx;
    *y = _y + sinf(_angle) * _ry;

    _angle += _da;

    unsigned pf = _path_cmd;
    _path_cmd   = path_cmd_line_to;
    return pf;
  }

  //------------------------------------------------------------------------
  void arc::normalize(float a1, float a2, bool ccw) {
    float ra = (fabs(_rx) + fabs(_ry)) / 2;
    _da      = (float)acos(ra / (ra + 0.125 / _scale)) * 2;
    if (ccw) {
      while (a2 < a1)
        a2 += pi * 2.0f;
    } else {
      while (a1 < a2)
        a1 += pi * 2.0f;
      _da = -_da;
    }
    _ccw         = ccw;
    _start       = a1;
    _end         = a2;
    _initialized = true;
  }

  // plain_rect

  //------------------------------------------------------------------------
  plain_rect::plain_rect(float x1, float y1, float x2, float y2)
      : _x1(x1), _y1(y1), _x2(x2), _y2(y2) {
    if (x1 > x2) {
      _x1 = x2;
      _x2 = x1;
    }
    if (y1 > y2) {
      _y1 = y2;
      _y2 = y1;
    }
  }

  //--------------------------------------------------------------------
  void plain_rect::rect(float x1, float y1, float x2, float y2) {
    _x1 = x1;
    _y1 = y1;
    _x2 = x2;
    _y2 = y2;
    if (x1 > x2) {
      _x1 = x2;
      _x2 = x1;
    }
    if (y1 > y2) {
      _y1 = y2;
      _y2 = y1;
    }
  }

  //--------------------------------------------------------------------
  void plain_rect::rewind() { _status = 0; }

  //--------------------------------------------------------------------
  unsigned plain_rect::vertex(float *x, float *y) {
    switch (_status) {
    case 0:
      *x = _x1;
      *y = _y1;
      _status++;
      return path_cmd_move_to;
    case 1:
      *x = _x2;
      *y = _y1;
      _status++;
      return path_cmd_line_to;
    case 2:
      *x = _x2;
      *y = _y2;
      _status++;
      return path_cmd_line_to;
    case 3:
      *x = _x1;
      *y = _y2;
      _status++;
      return path_cmd_line_to;
    case 4:
      *x = _x1;
      *y = _y1;
      _status++;
      return path_cmd_line_to;
    case 5:
    default: _status++; return path_cmd_end_poly | path_flags_close;
    }
  }

  // rounded_rect

  //------------------------------------------------------------------------
  rounded_rect::rounded_rect(float x1, float y1, float x2, float y2, float r)
      : _x1(x1), _y1(y1), _x2(x2), _y2(y2), _rx1(r), _ry1(r), _rx2(r), _ry2(r),
        _rx3(r), _ry3(r), _rx4(r), _ry4(r) {
    if (x1 > x2) {
      _x1 = x2;
      _x2 = x1;
    }
    if (y1 > y2) {
      _y1 = y2;
      _y2 = y1;
    }
  }

  //--------------------------------------------------------------------
  void rounded_rect::rect(float x1, float y1, float x2, float y2) {
    _x1 = x1;
    _y1 = y1;
    _x2 = x2;
    _y2 = y2;
    if (x1 > x2) {
      _x1 = x2;
      _x2 = x1;
    }
    if (y1 > y2) {
      _y1 = y2;
      _y2 = y1;
    }
  }

  //--------------------------------------------------------------------
  void rounded_rect::radius(float r) {
    _rx1 = _ry1 = _rx2 = _ry2 = _rx3 = _ry3 = _rx4 = _ry4 = r;
  }

  //--------------------------------------------------------------------
  void rounded_rect::radius(float rx, float ry) {
    _rx1 = _rx2 = _rx3 = _rx4 = rx;
    _ry1 = _ry2 = _ry3 = _ry4 = ry;
  }

  //--------------------------------------------------------------------
  void rounded_rect::radius(float rx_bottom, float ry_bottom, float rx_top,
                            float ry_top) {
    _rx1 = _rx2 = rx_bottom;
    _rx3 = _rx4 = rx_top;
    _ry1 = _ry2 = ry_bottom;
    _ry3 = _ry4 = ry_top;
  }

  // inline float max( float r1, float r2 ) { return r1>r2?r1:r2; }
  //--------------------------------------------------------------------
  void rounded_rect::radius(float rx1, float ry1, float rx2, float ry2,
                            float rx3, float ry3, float rx4, float ry4) {
    _rx1 = max(0.0f, rx1);
    _ry1 = max(0.0f, ry1);
    _rx2 = max(0.0f, rx2);
    _ry2 = max(0.0f, ry2);
    _rx3 = max(0.0f, rx3);
    _ry3 = max(0.0f, ry3);
    _rx4 = max(0.0f, rx4);
    _ry4 = max(0.0f, ry4);
  }

  //--------------------------------------------------------------------
  void rounded_rect::normalize_radius() {
    float dx = fabs(_x2 - _x1);
    float dy = fabs(_y2 - _y1);

    float k = 1.0f;
    float t = 0;
    if (_rx1 + _rx2) {
      t = dx / (_rx1 + _rx2);
      if (t < k) k = t;
    }
    if (_rx3 + _rx4) {
      t = dx / (_rx3 + _rx4);
      if (t < k) k = t;
    }
    if (_ry1 + _ry4) {
      t = dy / (_ry1 + _ry4);
      if (t < k) k = t;
    }
    if (_ry2 + _ry3) {
      t = dy / (_ry2 + _ry3);
      if (t < k) k = t;
    }

    if (k < 1.0f) {
      _rx1 *= k;
      _ry1 *= k;
      _rx2 *= k;
      _ry2 *= k;
      _rx3 *= k;
      _ry3 *= k;
      _rx4 *= k;
      _ry4 *= k;
    }
  }

  //--------------------------------------------------------------------
  void rounded_rect::rewind() { _status = 0; }

  //--------------------------------------------------------------------
  unsigned rounded_rect::vertex(float *x, float *y) {
    unsigned cmd = path_cmd_stop;
    switch (_status) {
    case 0:
      _arc.init(_x1 + _rx1, _y1 + _ry1, _rx1, _ry1, pi, pi + pi * 0.5f);
      _arc.rewind();
      _status++;

    case 1:
      cmd = _arc.vertex(x, y);
      if (is_stop(cmd))
        _status++;
      else
        return cmd;

    case 2:
      _arc.init(_x2 - _rx2, _y1 + _ry2, _rx2, _ry2, pi + pi * 0.5f, 0.0f);
      _arc.rewind();
      _status++;

    case 3:
      cmd = _arc.vertex(x, y);
      if (is_stop(cmd))
        _status++;
      else
        return path_cmd_line_to;

    case 4:
      _arc.init(_x2 - _rx3, _y2 - _ry3, _rx3, _ry3, 0.0f, pi * 0.5f);
      _arc.rewind();
      _status++;

    case 5:
      cmd = _arc.vertex(x, y);
      if (is_stop(cmd))
        _status++;
      else
        return path_cmd_line_to;

    case 6:
      _arc.init(_x1 + _rx4, _y2 - _ry4, _rx4, _ry4, pi * 0.5f, pi);
      _arc.rewind();
      _status++;

    case 7:
      cmd = _arc.vertex(x, y);
      if (is_stop(cmd))
        _status++;
      else
        return path_cmd_line_to;

    case 8:
      cmd = path_cmd_end_poly | path_flags_close | path_flags_ccw;
      _status++;
      break;
    }
    return cmd;
  }

  // borrowed from AGG

  // This epsilon is used to prevent us from adding degenerate curves
  // (converging to a single point).
  // The value isn't very critical. Function arc_to_bezier() has a limit
  // of the sweep_angle. If FABS(sweep_angle) exceeds pi/2 the curve
  // becomes inaccurate. But slight exceeding is quite appropriate.
  //-------------------------------------------------bezier_arc_angle_epsilon
  const float bezier_arc_angle_epsilon = 0.01f;

  //------------------------------------------------------------arc_to_bezier
  void arc_to_bezier(float cx, float cy, float rx, float ry, float start_angle,
                     float sweep_angle, float *curve) {
    float x0 = cosf(sweep_angle / 2.0f);
    float y0 = sinf(sweep_angle / 2.0f);
    float tx = (1.0f - x0) * 4.0f / 3.0f;
    float ty = y0 - tx * x0 / y0;
    float px[4];
    float py[4];
    px[0] = x0;
    py[0] = -y0;
    px[1] = x0 + tx;
    py[1] = -ty;
    px[2] = x0 + tx;
    py[2] = ty;
    px[3] = x0;
    py[3] = y0;

    float sn = sinf(start_angle + sweep_angle / 2.0f);
    float cs = cosf(start_angle + sweep_angle / 2.0f);

    unsigned i;
    for (i = 0; i < 4; i++) {
      curve[i * 2]     = cx + rx * (px[i] * cs - py[i] * sn);
      curve[i * 2 + 1] = cy + ry * (px[i] * sn + py[i] * cs);
    }
  }

  //------------------------------------------------------------------------
  void bezier_arc::init(float x, float y, float rx, float ry, float start_angle,
                        float sweep_angle) {
    start_angle = fmodf(start_angle, 2.0f * pi);
    if (sweep_angle >= 2.0f * pi) sweep_angle = 2.0f * pi;
    if (sweep_angle <= -2.0f * pi) sweep_angle = -2.0f * pi;

    if (abs(sweep_angle) < 1e-8f) {
      m_num_vertices = 4;
      m_cmd          = path_cmd_line_to;
      m_vertices[0]  = x + rx * (float)cos(start_angle);
      m_vertices[1]  = y + ry * (float)sin(start_angle);
      m_vertices[2]  = x + rx * (float)cos(start_angle + sweep_angle);
      m_vertices[3]  = y + ry * (float)sin(start_angle + sweep_angle);
      return;
    }

    float total_sweep = 0.0f;
    float local_sweep = 0.0f;
    float prev_sweep;
    m_num_vertices = 2;
    m_cmd          = path_cmd_curve4;
    bool done      = false;
    do {
      if (sweep_angle < 0.0f) {
        prev_sweep  = total_sweep;
        local_sweep = -pi * 0.5f;
        total_sweep -= pi * 0.5f;
        if (total_sweep <= sweep_angle + bezier_arc_angle_epsilon) {
          local_sweep = sweep_angle - prev_sweep;
          done        = true;
        }
      } else {
        prev_sweep  = total_sweep;
        local_sweep = pi * 0.5f;
        total_sweep += pi * 0.5f;
        if (total_sweep >= sweep_angle - bezier_arc_angle_epsilon) {
          local_sweep = sweep_angle - prev_sweep;
          done        = true;
        }
      }

      arc_to_bezier(x, y, rx, ry, start_angle, local_sweep,
                    m_vertices + m_num_vertices - 2);

      m_num_vertices += 6;
      start_angle += local_sweep;
    } while (!done && m_num_vertices < 26);
  }

  //--------------------------------------------------------------------
  bool bezier_arc_svg::calc(float xs, float ys, float rx, float ry, float angle,
                            bool large_arc_flag, bool sweep_flag, float xe,
                            float ye) {
    m_radii_ok = true;

    // assert(xs != 0 && ys != 0);

    if (rx < 0.0f) rx = -rx;
    if (ry < 0.0f) ry = -ry;

    // Calculate the middle point between
    // the current and the final points
    //------------------------
    float dx2 = (xs - xe) / 2.0f;
    float dy2 = (ys - ye) / 2.0f;

    float cos_a = cosf(angle);
    float sin_a = sinf(angle);

    // Calculate (x1, y1)
    //------------------------
    float x1 = cos_a * dx2 + sin_a * dy2;
    float y1 = -sin_a * dx2 + cos_a * dy2;

    // Ensure radii are large enough
    //------------------------
    float prx = rx * rx;
    float pry = ry * ry;
    float px1 = x1 * x1;
    float py1 = y1 * y1;

    // Check that radii are large enough
    //------------------------
    float radii_check = px1 / prx + py1 / pry;
    if (radii_check > 1.0f) {
      rx  = sqrtf(radii_check) * rx;
      ry  = sqrtf(radii_check) * ry;
      prx = rx * rx;
      pry = ry * ry;
      if (radii_check > 10.0f) m_radii_ok = false;
    }

    // Calculate (cx1, cy1)
    //------------------------
    float sign = (large_arc_flag == sweep_flag) ? -1.0f : 1.0f;
    float sq   = (prx * pry - prx * py1 - pry * px1) / (prx * py1 + pry * px1);
    float coef = sign * sqrtf((sq < 0) ? 0 : sq);
    float cx1  = coef * ((rx * y1) / ry);
    float cy1  = coef * -((ry * x1) / rx);

    //
    // Calculate (cx, cy) from (cx1, cy1)
    //------------------------
    float sx2 = (xs + xe) / 2.0f;
    float sy2 = (ys + ye) / 2.0f;
    float cx  = sx2 + (cos_a * cx1 - sin_a * cy1);
    float cy  = sy2 + (sin_a * cx1 + cos_a * cy1);

    // Calculate the start_angle (angle1) and the sweep_angle (dangle)
    //------------------------
    float ux = (x1 - cx1) / rx;
    float uy = (y1 - cy1) / ry;
    float vx = (-x1 - cx1) / rx;
    float vy = (-y1 - cy1) / ry;
    float p, n;

    // Calculate the angle start
    //------------------------
    n       = sqrtf(ux * ux + uy * uy);
    p       = ux; // (1 * ux) + (0 * uy)
    sign    = (uy < 0) ? -1.0f : 1.0f;
    float v = p / n;
    if (v < -1.0f) v = -1.0f;
    if (v > 1.0f) v = 1.0f;
    float start_angle = sign * (float)acos(v);

    // Calculate the sweep angle
    //------------------------
    n    = sqrtf((ux * ux + uy * uy) * (vx * vx + vy * vy));
    p    = ux * vx + uy * vy;
    sign = (ux * vy - uy * vx < 0) ? -1.0f : 1.0f;
    v    = p / n;
    if (v < -1.0f) v = -1.0f;
    if (v > 1.0f) v = 1.0f;
    float sweep_angle = sign * acosf(v);
    if (!sweep_flag && sweep_angle > 0) {
      sweep_angle -= pi * 2.0f;
    } else if (sweep_flag && sweep_angle < 0) {
      sweep_angle += pi * 2.0f;
    }

    // We can now build and transform the resulting arc
    //------------------------
    m_arc.init(0.0f, 0.0f, rx, ry, start_angle, sweep_angle);
    affine_mtx_f mtx = affine_rotation_f(angle);
    mtx *= affine_translation_f(cx, cy);

    for (unsigned i = 2; i < m_arc.num_vertices() - 2; i += 2) {
      mtx.transform(m_arc.vertices() + i, m_arc.vertices() + i + 1);
    }

    // We must make sure that the starting and ending points
    // exactly coincide with the initial (x0,y0) and (x2,y2)
    m_arc.vertices()[0] = xs;
    m_arc.vertices()[1] = ys;
    if (m_arc.num_vertices() > 2) {
      m_arc.vertices()[m_arc.num_vertices() - 2] = xe;
      m_arc.vertices()[m_arc.num_vertices() - 1] = ye;
    }
    return m_radii_ok;
  }

} // namespace gool