
#include "gdi+.h"

#if defined(USE_GDI)

#include "gdi+graphics.h"
#include "gdi+application.h"

namespace gdi {

  bool font::has_glyph_for(uint ucodepoint) const {
    font::char_range  r = {wchar(ucodepoint), 1};
    font::char_range *pfound =
        std::lower_bound(_ranges.begin(), _ranges.end(), r);
    return pfound && pfound < _ranges.end() &&
           ucodepoint >= uint(pfound->first) &&
           ucodepoint < (uint(pfound->first) + pfound->length);
  }
  uint font::glyph_index(uint ucodepoint) const {
    // assert(false);

    screen_dc sdc;
    sdc.select(hfont);

    wchar cp[2]  = {wchar(ucodepoint), 0};
    WORD  gi[10] = {0};

    GetGlyphIndices(sdc, cp, 1, gi, 10);

    return gi[0];
  }

  Gdiplus::Bitmap *graphics::GdiBitmap(bitmap *bmp) {
    if (bmp->gdi_bmp) {
      if (bmp->_pixels.size() && (bmp->_used_generation != bmp->_generation)) {

        bmp->_used_generation = bmp->_generation;

        Rect rect(0, 0, bmp->dim().x, bmp->dim().y);

        BitmapData bmpdata;
        bmp->gdi_bmp->LockBits(&rect, ImageLockModeRead, PixelFormat32bppRGB,
                               &bmpdata);
        memcpy(bmpdata.Scan0, bmp->_pixels.cbegin(),
               bmp->_pixels.size() * sizeof(bmp->_pixels[0]));
        bmp->gdi_bmp->UnlockBits(&bmpdata);
      }

      return bmp->gdi_bmp;
    }

    size        dim    = bmp->dim();
    slice<argb> pixels = bmp->get_bits();
    int         stride = bmp->stride();

    if (pixels.length)
      bmp->gdi_bmp =
          bmp->_has_alpha_bits
              ? new Bitmap(dim.x, dim.y, stride, PixelFormat32bppPARGB,
                           (BYTE *)pixels.start)
              : new Bitmap(dim.x, dim.y, stride, PixelFormat32bppRGB,
                           (BYTE *)pixels.start);
    else
      bmp->gdi_bmp = new Bitmap(dim.x, dim.y, PixelFormat32bppPARGB);

    return bmp->gdi_bmp;
  }

  void graphics::fill(argb c, const rect &dst) {
    SolidBrush br(g2p(c));
    _target->FillRectangle(&br, g2p(dst));
  }

  void graphics::fill(argb c, const rect &dst, const size &radius) {
    SolidBrush br(g2p(c));
    //_target->FillRectangle(&br,g2p(dst));

    size r = radius;
    if (r.x > dst.width()) r.x = dst.width();
    if (r.y > dst.height()) r.y = dst.height();

    GraphicsPath path;
    path.AddArc(dst.right() - r.x, dst.top(), r.x, r.y, 270, 90);
    path.AddArc(dst.right() - r.x, dst.bottom() - r.y, r.x, r.y, 0, 90);
    path.AddArc(dst.left(), dst.bottom() - r.y, r.x, r.y, 90, 90);
    path.AddArc(dst.left(), dst.top(), r.x, r.y, 180, 90);
    path.AddLine(dst.left() + r.x / 2, dst.top(), dst.right() - r.x / 2,
                 dst.top());

    aa_mode _(this);
    _target->FillPath(&br, &path);
  }

  void graphics::fill(argb c, const gool::path *dst) {
    SolidBrush br(g2p(c));
    // dst->seal();
    aa_mode _(this);
    _target->FillPath(&br, static_cast<const gdi::path *>(dst));
  }

  Brush *make_Brush(const linear_brush &lb, Graphics *forgfx) {
    auto stops = lb.stops();

    pointf start = lb.start;
    pointf end   = lb.end;

    // this "beauty" code is just to overcome that strange way of doing
    // gradients in GDI+
    Rect clip;
    forgfx->GetClipBounds(&clip);
    float clipf     = (float)max(clip.Width, clip.Height);
    float areaf     = max(abs(start.x - end.x), abs(start.y - end.y));
    float expansion = clipf / areaf;
    if (expansion < 1.0f) expansion = 1.0f;

    pointf d = end - start; // we increase gradient vector length 10 times in
                            // both directions to simulate borderless gradients
                            // in GDI+
    start -= d * expansion;
    end += d * expansion;

    LinearGradientBrush *pb =
        new LinearGradientBrush(g2p(start), g2p(end), g2p(stops.first().color),
                                g2p(stops.last().color));

    pb->MultiplyTransform(g2p(lb.transform()));

    tool::array<Color> colors(stops.length + 2);
    tool::array<float> positions(stops.length + 2);

    colors[0]    = g2p(stops[0].color);
    positions[0] = 0;

    for (uint n = 1; n <= stops.length; ++n) {
      colors[n]    = g2p(stops[n - 1].color);
      positions[n] = expansion / (2 * expansion + 1.0f) +
                     stops[n - 1].position / (2 * expansion + 1.0f);
    }
    colors.last()    = g2p(stops.last().color);
    positions.last() = 1.0f;

    // for(uint n = 0; n < stops.length; ++n) { colors[n] =
    // g2p(argb(xcolor(stops[n].color))); positions[n] = stops[n].position; }
    Status st = pb->SetInterpolationColors(colors.head(), positions.head(),
                                           INT(stops.length + 2));
    assert(st == Ok);
    st = st;
    pb->SetWrapMode(WrapModeClamp);
    return pb;
  }

  Brush *make_Brush(const radial_brush &lb, Graphics *forgfx) {
    rectf elr(lb.center, lb.center);

    // this "beauty" code is just to overcome that strange way of doing
    // gradients in GDI+
    Rect clip;
    forgfx->GetClipBounds(&clip);
    float clipf     = (float)max(clip.Width, clip.Height);
    float radiusf   = max(lb.radius.x, lb.radius.y);
    float expansion = clipf / radiusf;
    if (expansion < 1.0f) expansion = 1.0f;

    elr >>= lb.radius * expansion;

    GraphicsPath path;
    path.AddEllipse(g2p(elr));

    PathGradientBrush *pb = new PathGradientBrush(&path);

    auto stops = lb.stops();

    tool::array<Color> colors(stops.length + 1);
    tool::array<float> positions(stops.length + 1);

    uint d = uint(stops.length);
    for (uint n = 0; n < stops.length; ++n, --d) {
      colors[d] = g2p(argb(stops[n].color));
      positions[d] =
          (1.00f - stops[n].position) / expansion + (1.0f - 1.0f / expansion);
      // dbg_printf("%d %f\n",d, double(positions[d]));
    }
    positions[0] = 0.0f;
    colors[0]    = g2p(argb(stops.last().color));

    Status st = pb->SetInterpolationColors(colors.head(), positions.head(),
                                           INT(stops.length + 1));
    pb->SetWrapMode(WrapModeClamp);
    pb->SetCenterPoint(g2p(lb.center));
    pb->SetTransform(g2p(lb.transform()));
    assert(st == Ok);
    st = st;
    return pb;
  }

  void graphics::fill(const gool::brush *pgb, const rect &dst) {
    handle<Brush> brush;
    switch (pgb->type()) {
    case brush::LINEAR:
      brush = make_Brush(*static_cast<const linear_brush *>(pgb), _target);
      // fill_linear_gradient( *static_cast<const linear_brush*>(pgb), rc);
      break;
    case brush::RADIAL:
      brush = make_Brush(*static_cast<const radial_brush *>(pgb), _target);
      // fill_radial_gradient( *static_cast<const radial_brush*>(pgb), rc);
      break;
    default: assert(false); return;
    }
    _target->FillRectangle(brush, g2p(dst));
  }

  /*  void graphics::fill( gool::image* img, const rect& dst, const rect&
    src,point offset)
    {
        size sz = src.size();
        if( sz.empty() )
          return;

        argb clr;
        if( (sz.x <= 8 || sz.y <= 8) && img->is_solid_color(src,clr) )
          return fill(clr,dst);

        for(int y = dst.s.y + offset.y; y <= dst.e.y; y += sz.y)
          for(int x = dst.s.x + offset.x; x <= dst.e.x; x += sz.x)
          {
            rect r_dst( point(x, y), sz);
            rect r_target = r_dst & dst;
            if( r_target == r_dst )
              do_draw(img,r_target,src);
            else
            {
              rect r_src( src );
              r_src.s.x += r_target.s.x - r_dst.s.x;
              r_src.s.y += r_target.s.y - r_dst.s.y;
              r_src.e.x -= r_dst.e.x - r_target.e.x;
              r_src.e.y -= r_dst.e.y - r_target.e.y;
              if( !r_src.empty() )
                do_draw(img,r_target,r_src);
            }
          }
    } */

  void graphics::draw(const gool::path *dst, argb c, float width) {
    Pen pn(g2p(c), width);
    // dst->seal();
    aa_mode _(this);
    _target->DrawPath(&pn, static_cast<const gdi::path *>(dst));
  }

  void graphics::draw_line_v(const rect &rc, argb c, int stroke, int step,
                             int_v off) {
    if (c.alfa == 0) return;

    Rect        r(rc.left(), rc.top(), rc.width(), stroke);
    array<Rect> rectangles;

    rectangles.push(r);

    int space     = rc.height() - step - stroke + 1;
    int num_steps = space / step;
    for (int ly = rc.s.y + step; num_steps > 0;) {
      r.Y = ly;
      rectangles.push(r);
      int delta = space / num_steps;
      ly += delta;
      space -= delta;
      --num_steps;
    }
    // last one
    r.Y = rc.e.y - stroke + 1;
    rectangles.push(r);

    SolidBrush br(g2p(c));
    target()->FillRectangles(&br, rectangles.cbegin(), rectangles.size());

    // int s1 = stroke;

    // float w = float(rc.width());
    // float w2 = w / 2.0f;
    // float h2 = rc.width() == 1? 0: w / 2.0f;

    // pointf pts(rc.left() + w2,rc.top() + h2);
    // pointf pte(rc.left() + w2,rc.top() + s1 - h2);

    // Pen pn(g2p(c),w);

    ////SolidBrush br(g2p(c));

    ////_target->FillRectangle(&br, pts.x, pts.y, pte.x - pts.x, pte.y - pts.y);
    //_target->DrawLine(&pn, pts.x, pts.y, pte.x, pte.y);
    //
    // int space = rc.height() - step - stroke + 1;
    // int num_steps = space / step;
    // for(int ly = rc.s.y + step; num_steps > 0;)
    //{
    //  pts.y = ly + h2;
    //  pte.y = ly + s1 - h2;
    //  _target->DrawLine(&pn, pts.x, pts.y, pte.x, pte.y);
    //  int delta = space / num_steps;
    //  ly += delta;
    //  space -= delta;
    //  --num_steps;
    //}
    // pte.y = rc.e.y + 1 - h2;
    // pts.y = rc.e.y + 1 - s1 + h2;
    //_target->DrawLine(&pn, pts.x, pts.y, pte.x, pte.y);
  }

  void graphics::draw_line_h(const rect &rc, argb c, int stroke, int step,
                             int_v off) {
    if (c.alfa == 0) return;

    Rect        r(rc.left(), rc.top(), stroke, rc.height());
    array<Rect> rectangles;

    rectangles.push(r);

    int space     = rc.width() - step - stroke + 1;
    int num_steps = space / step;
    for (int lx = rc.s.x + step; num_steps > 0;) {
      r.X = lx;
      rectangles.push(r);
      int delta = space / num_steps;
      lx += delta;
      space -= delta;
      --num_steps;
    }
    r.X = rc.e.x + 1 - stroke;
    rectangles.push(r);

    SolidBrush br(g2p(c));
    target()->FillRectangles(&br, rectangles.cbegin(), rectangles.size());

    /*int s1 = stroke;

    float w = float(rc.height());
    float h2 = w / 2.0f;
    float w2 = rc.height() == 1? 0: w / 2.0f;

    pointf pts(rc.left() + w2,rc.top() + h2);
    pointf pte(rc.left() + s1 - w2,rc.top() + h2);
       

    Pen pn(g2p(c),w);

    _target->DrawLine(&pn, pts.x, pts.y, pte.x, pte.y);
       

    int space = rc.width() - step - stroke + 1;
    int num_steps = space / step;
    for(int lx = rc.s.x + step; num_steps > 0;)
    {
      pts.x = lx + w2;
      pte.x = lx + s1 - w2;
      _target->DrawLine(&pn, pts.x, pts.y, pte.x, pte.y);
      int delta = space / num_steps;
      lx += delta;
      space -= delta;
      --num_steps;
    }
    pte.x = rc.e.x + 1 - w2;
    pts.x = rc.e.x + 1 - s1 + w2;
    _target->DrawLine(&pn, pts.x, pts.y, pte.x, pte.y);*/
  }

  void graphics::draw_line(pointf p1, pointf p2, argb c, int width) {
    Pen pn(g2p(c), float(width));
    _target->DrawLine(&pn, p1.x, p1.y, p2.x, p2.y);
  }

  bool graphics::push_clip(const rect &r, clipper *cl) {
    // GraphicsContainer ctid = _target->BeginContainer();
    push_state();
    Status s = _target->SetClip(g2p(r), CombineModeIntersect);
    // cl->container_id = ctid;
    super::push_clip(r, cl);
    // setup_target();
    // cl->
    return s == Ok;
  }
  void graphics::pop_clip() {
    // if(_clipper)
    //  _target->EndContainer(_clipper->container_id);
    setup_target();
    super::pop_clip();
    pop_state();
  }

  gool::application *graphics::app() const { return gdi_app(); }

  point graphics::offset(point noff) {
    point off = _offset;
    _offset   = noff;
    _clip_rc -= noff;
    // render_target()->SetTransform(D2D1::Matrix3x2F::Translation(float(_offset.x),float(_offset.y)));
    _target->TranslateTransform(float(noff.x), float(noff.y));
    return off;
  }

  uint graphics::push_state() {
    auto pa = _target->GetSmoothingMode();
    if (pa != SmoothingModeHighQuality) pa = pa;

    uint lv             = uint(_registers_stack.length());
    _registers.state_id = _target->Save();
    _registers_stack.push(_registers);
    return lv;
  }
  bool graphics::pop_state() {
    if (!_registers_stack.size()) return false;
    _registers = _registers_stack.pop();
    _target->Restore(_registers.state_id);
    // auto pa = _target->GetSmoothingMode();
    // assert(pa == SmoothingModeHighQuality);

    return true;
  }
  void graphics::restore_state(uint level) {
    while (_registers_stack.length() && _registers_stack.length() > level)
      pop_state();
  }

  void graphics::push_layer(const gool::rect &clip, byte opacity,
                            function<bool(filter_graph_builder *)> filters) {
    // super::push_layer(clip,opacity,filters);

    layer_t ld;

    if (opacity == 255) {
      ld.container_id       = _target->BeginContainer();
      ld.layer_bitmap       = 0;
      ld.layer_original_gfx = 0;
    } else {
      ld.layer_bitmap =
          new Bitmap(clip.width(), clip.height(), PixelFormat32bppPARGB);
      ld.layer_original_gfx = _target;
      ld.layer_origin       = clip.s;
      ld.layer_opacity      = opacity;
      _target               = new Graphics(ld.layer_bitmap);
      _target->Clear(Color::Transparent);
      ld.container_id = _target->BeginContainer();
      _target->TranslateTransform(float(-ld.layer_origin.x),
                                  float(-ld.layer_origin.y));
      _target->SetClip(g2p(clip));
    }
    _layers.push(ld);
    _registers_stack.push(_registers);
    setup_target();
  }
  void graphics::push_layer(const gool::path *clip, byte opacity) {
    // super::push_layer(clip,opacity);
    const gdi::path *gpath = static_cast<const gdi::path *>(clip);
    RectF            bounds;
    Gdiplus::Status  st = gpath->GetBounds(&bounds);
    assert(st == Gdiplus::Ok);
    st = st;
    gool::rect box(point((int)floorf(bounds.X), (int)floorf(bounds.Y)),
                   size((int)ceilf(bounds.Width), (int)ceilf(bounds.Height)));
    // push_layer(box, opacity);
    // st = _target->SetClip(gpath/*,CombineModeIntersect*/);

    layer_t ld;

    ld.layer_bitmap =
        new Bitmap(box.width(), box.height(), PixelFormat32bppPARGB);
    ld.layer_original_gfx = _target;
    ld.layer_origin       = box.s;
    ld.layer_opacity      = opacity;
    _target               = new Graphics(ld.layer_bitmap);
    _target->Clear(Color::Transparent);
    ld.container_id = _target->BeginContainer();
    _target->TranslateTransform(float(-ld.layer_origin.x),
                                float(-ld.layer_origin.y));

    ld.layer_clip = gpath->Clone();
    //_registers.layer_clip->SetFillMode();
    RectF cr;
    ld.layer_clip->GetBounds(&cr);
    _target->SetClip(cr);
    ld.layer_clip->AddRectangle(g2p(box >> 1));

    _layers.push(ld);
    _registers_stack.push(_registers);
    setup_target();
  }

  void graphics::push_layer(const gool::bitmap *src_clip, bool draw1a,
                            byte opacity) {

    layer_t ld;

    ld.layer_bitmap =
        new Bitmap(src_clip->dim().x, src_clip->dim().y, PixelFormat32bppPARGB);
    // this->GdiBitmap(const_cast<gool::bitmap*>(src_clip));
    ld.layer_original_gfx = _target;
    ld.layer_origin       = point(0, 0);
    ld.layer_opacity      = opacity;
    ld.layer_draw1a       = draw1a;
    ld.layer_mask         = src_clip;
    _target               = new Graphics(ld.layer_bitmap);
    _target->Clear(Color::Transparent);
    ld.container_id = _target->BeginContainer();
    _target->TranslateTransform(float(-ld.layer_origin.x),
                                float(-ld.layer_origin.y));
    _target->SetClip(g2p(gool::rect(src_clip->dim())));

    _layers.push(ld);
    _registers_stack.push(_registers);
    setup_target();
  }

  void graphics::pop_layer() {
    if (!_layers.size()) return;

    _registers = _registers_stack.pop();

    layer_t ld = _layers.pop();

    if (ld.layer_bitmap) {
      //_target->Flush();

      if (ld.layer_mask) {
        BitmapData bmpdata1 = {0};
        Rect       rect(0, 0, ld.layer_bitmap->GetWidth(),
                  ld.layer_bitmap->GetHeight());
        ld.layer_bitmap->LockBits(&rect, ImageLockModeWrite,
                                  PixelFormat32bppPARGB, &bmpdata1);
        slice<argb> bmp_bits((argb *)bmpdata1.Scan0,
                             bmpdata1.Width * bmpdata1.Height);

        argb *dst = const_cast<argb *>(bmp_bits.start);

        argb zero(0, 0, 0, 0);

        slice<argb> mask_bits = ld.layer_mask->_pixels();

        // do sweeping of the bitmap by mask pixels

        if (!ld.layer_draw1a)
          for (uint n = 0; n < mask_bits.length; ++n) {
            uint transparency = mask_bits[n].alfa;
            if (transparency == 255) {
              dst[n] = zero;
              continue;
            } else if (transparency == 0)
              continue;
            argb pix = dst[n].demultiply();
            pix.alfa = argb::channel_t((pix.alfa * (255 - transparency)) >> 8);
            dst[n]   = pix.premultiply();
          }
        else
          for (uint n = 0; n < mask_bits.length; ++n) {
            uint transparency = 255 - mask_bits[n].alfa;
            if (transparency == 255) {
              dst[n] = zero;
              continue;
            } else if (transparency == 0)
              continue;
            argb pix = dst[n].demultiply();
            pix.alfa = argb::channel_t((pix.alfa * (255 - transparency)) >> 8);
            dst[n]   = pix.premultiply();
          }

        ld.layer_bitmap->UnlockBits(&bmpdata1);
      }

      if (ld.layer_clip) {

      // These lines do not work as there is no CompositingModeSourceSweep or
      // the like

      // Status st = _target->SetCompositingMode(CompositingModeSourceOver);
      // assert(st == Ok);
      // st = _target->SetCompositingQuality(CompositingQualityHighQuality);
      // assert(st == Ok);
      // st = _target->SetSmoothingMode(SmoothingModeAntiAlias);
      // assert(st == Ok);
      // SolidBrush tb(Color(255,0,255,0));
      //_target->FillPath(&tb,_registers.layer_clip);

#pragma TODO("consider caching of the mask bitmap here:")

        // prepare sweep mask

        size sz(ld.layer_bitmap->GetWidth(), ld.layer_bitmap->GetHeight());

        Bitmap mask_bmp(sz.x, sz.y, PixelFormat32bppRGB);

        if (sz.x > 0 && sz.y > 0) {
          Graphics mask_gfx(&mask_bmp);

          Status st = mask_gfx.SetSmoothingMode(SmoothingModeAntiAlias);
          assert(st == Ok);
          st = st;

          mask_gfx.TranslateTransform(float(-ld.layer_origin.x - 0.5f),
                                      float(-ld.layer_origin.y - 0.5f));
          SolidBrush tb(Color(255, 255, 255));
          mask_gfx.FillPath(&tb, ld.layer_clip);
        }

        Rect rect(0, 0, sz.x, sz.y);

        BitmapData bmpdata = {0};
        mask_bmp.LockBits(&rect, ImageLockModeRead, PixelFormat32bppRGB,
                          &bmpdata);
        slice<argb> mask_bits((argb *)bmpdata.Scan0,
                              bmpdata.Width * bmpdata.Height);

        BitmapData bmpdata1 = {0};
        ld.layer_bitmap->LockBits(&rect, ImageLockModeWrite,
                                  PixelFormat32bppPARGB, &bmpdata1);
        slice<argb> bmp_bits((argb *)bmpdata1.Scan0,
                             bmpdata1.Width * bmpdata1.Height);

        argb *dst = const_cast<argb *>(bmp_bits.start);

        argb zero(0, 0, 0, 0);

        // do sweeping of the bitmap by mask pixels
        if (!!bmp_bits && !!mask_bits)
          for (uint n = 0; n < mask_bits.length; ++n) {
            uint transparency = mask_bits[n].red;
            if (transparency == 255) {
              dst[n] = zero;
              continue;
            } else if (transparency == 0)
              continue;
            argb pix = dst[n].demultiply();
            pix.alfa = argb::channel_t((pix.alfa * (255 - transparency)) >> 8);
            dst[n]   = pix.premultiply();
          }
        mask_bmp.UnlockBits(&bmpdata);
        ld.layer_bitmap->UnlockBits(&bmpdata1);
      }

      _target->EndContainer(ld.container_id);

      _target = ld.layer_original_gfx;

      float alpha = float(ld.layer_opacity) / 255.0f;

      ColorMatrix color_matrix = {1.0f, 0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 1.0f,
                                  0.0f, 0.0f, 0.0f, 0.0f, 0.0f,  1.0f, 0.0f,
                                  0.0f, 0.0f, 0.0f, 0.0f, alpha, 0.0f, 0.0f,
                                  0.0f, 0.0f, 0.0f, 1.0f};

      Gdiplus::RectF rdst;
      rdst.X      = float(ld.layer_origin.x);
      rdst.Y      = float(ld.layer_origin.y);
      rdst.Width  = float(ld.layer_bitmap->GetWidth());
      rdst.Height = float(ld.layer_bitmap->GetHeight());

      ImageAttributes image_attributes;
      image_attributes.SetColorMatrix(&color_matrix, ColorMatrixFlagsDefault,
                                      ColorAdjustTypeBitmap);
      Gdiplus::Status st =
          _target->DrawImage(ld.layer_bitmap, rdst, 0, 0, rdst.Width,
                             rdst.Height, UnitPixel, &image_attributes);
      // assert( st == Ok);
      st = st;

    } else {
      _target->EndContainer(ld.container_id);
    }
    setup_target();
  }

  pointf graphics::world_to_screen(pointf pt) const {
    PointF p(pt.x, pt.y);
    Status st = _target->TransformPoints(CoordinateSpaceDevice,
                                         CoordinateSpaceWorld, &p, 1);
    assert(st == Ok);
    st = st;
    return pointf(p.X, p.Y);
  }

  pointf graphics::screen_to_world(pointf pt) const {
    PointF p(pt.x, pt.y);
    Status st = _target->TransformPoints(CoordinateSpaceWorld,
                                         CoordinateSpaceDevice, &p, 1);
    assert(st == Ok);
    st = st;
    return pointf(p.X, p.Y);
  }

  void graphics::transform(const affine_mtx_f &m) {
    handle<gdi::Matrix> pm = g2p(m);
    _target->MultiplyTransform(pm);
  }

  void graphics::reset_transform() {
    _target->ResetTransform();
  }
  
  affine_mtx_f graphics::transform() const {
    gdi::Matrix pm;
    _target->GetTransform(&pm);
    return p2g(&pm);
  }

  void graphics::translate(pointf pt) {
    _target->TranslateTransform(pt.x, pt.y);
  }
  void graphics::rotate(float radians, pointf center) {
    gdi::Matrix mtx;
    mtx.RotateAt(radians * 57.2957795f, g2p(center));
    _target->MultiplyTransform(&mtx);
  }
  void graphics::scale(sizef sz, pointf center) {
    gdi::Matrix mtx;
    mtx.Scale(sz.x, sz.y);
    _target->MultiplyTransform(&mtx);
  }
  void graphics::skew(sizef sz, pointf center) {
    gdi::Matrix mtx;
    mtx.Shear(sz.x, sz.y);
    _target->MultiplyTransform(&mtx);
  }

  void graphics::draw_line(pointf start, pointf end) {
    if (_registers.stroke_drawable())
      _target->DrawLine(_registers.stroke_pen(), start.x, start.y, end.x,
                        end.y);
  }

  void graphics::draw_line(pointf s, pointf end, line_end_style sorigin,
                           line_end_style send) {
    return super::draw_line(s, end, sorigin, send);
  }

  void graphics::set_fill(argb c) // set fill brush
  {
    if (c.is_no_color())
      _registers.fill_brush = 0;
    else
      _registers.fill_brush = new SolidBrush(g2p(c));
  }

  void graphics::set_fill(const linear_brush &lb) // set fill brush
  {
    _registers.fill_brush = make_Brush(lb, _target);
  }

  void graphics::set_fill(const radial_brush &rb) {
    _registers.fill_brush = make_Brush(rb, _target);
  }

  void graphics::set_fill(const image *ib) // set fill brush
  {
    handle<bitmap> bmp = const_cast<image *>(ib)->get_bitmap(this, ib->dim());
    _registers.fill_brush = new TextureBrush(GdiBitmap(bmp));
  }

  void graphics::set_stroke(argb c) {
    if (c.is_no_color())
      _registers.stroke_brush = 0;
    else
      _registers.stroke_brush = new SolidBrush(g2p(c));
    _registers._stroke_pen = nullptr;
  }
  void graphics::set_stroke(const linear_brush &lb) {
    _registers.stroke_brush = make_Brush(lb, _target);
    _registers._stroke_pen  = nullptr;
  }
  void graphics::set_stroke(const radial_brush &rb) {
    _registers.stroke_brush = make_Brush(rb, _target);
    _registers._stroke_pen  = nullptr;
  }

  void graphics::draw_ellipse(pointf center, sizef radius, bool stroke, bool fill) {
    if (!_registers.drawable()) return;

    rectf elr(center, center);
    elr >>= radius;

    if (fill && _registers.fill_drawable())
      this->_target->FillEllipse(_registers.fill_brush, g2p(elr));
    if (stroke && _registers.stroke_drawable())
      this->_target->DrawEllipse(_registers.stroke_pen(), g2p(elr));
  }

  void graphics::draw_arc(pointf center, sizef radius, float angle_start,
                          float angle_sweep, bool stroke, bool fill) {
    if (!_registers.drawable()) return;

    rectf elr(center, center);
    elr >>= radius;

    if (fill && _registers.fill_drawable()) {
      _target->FillPie(_registers.fill_brush, g2p(elr),
                       angle_start * 57.2957795f, angle_sweep * 57.2957795f);
      if (stroke && _registers.stroke_drawable())
        _target->DrawPie(_registers.stroke_pen(), g2p(elr),
                         angle_start * 57.2957795f, angle_sweep * 57.2957795f);
    } else if (stroke && _registers.stroke_drawable()) {
      _target->DrawArc(_registers.stroke_pen(), g2p(elr),
                       angle_start * 57.2957795f, angle_sweep * 57.2957795f);
    }
  }

  void graphics::draw_rectangle(pointf org, sizef dim, bool stroke, bool fill) {
    if (!_registers.drawable()) return;

    org += pointf(0.5f, 0.5f);

    if (fill && _registers.fill_drawable())
      _target->FillRectangle(_registers.fill_brush, org.x, org.y, dim.x, dim.y);
    if (stroke && _registers.stroke_drawable())
      _target->DrawRectangle(_registers.stroke_pen(), org.x, org.y, dim.x,
                             dim.y);
  }

  /*void graphics::draw_rectangle(rectf elr)
  {
    if( !_registers.drawable() ) return;
    elr += pointf(0.5f,0.5f);
    if( _registers.fill_drawable() )
      _target->FillRectangle(_registers.fill_brush,g2p(elr));
    if( _registers.stroke_drawable() )
      _target->DrawRectangle(_registers.stroke_pen(),g2p(elr));
  }*/

  void graphics::draw_path(const gool::path *dst, bool stroke, bool fill) {
    if (!_registers.drawable()) return;
    if (_registers.fill_drawable() && fill)
      _target->FillPath(_registers.fill_brush,
                        static_cast<const gdi::path *>(dst));
    if (_registers.stroke_drawable() && stroke)
      _target->DrawPath(_registers.stroke_pen(),
                        static_cast<const gdi::path *>(dst));
  }
  void graphics::draw_path(const gool::polygonf &p, bool stroke, bool fill) {
    if (!_registers.drawable()) return;
    gdi::path path;
    path.set(p, true);
    draw_path(&path);
  }

  void path::reset() { this->Reset(); }

  bool path::is_inside(pointf pt) {
    seal();
    return !!this->IsVisible(pt.x, pt.y);
  }

  void path::set(const polygonf &vertices, bool fill_even_odd) {
    slice<pointf> points = vertices();

    if (points.length == 0) return;

    this->StartFigure();

    pointf p = points++;
    // ps->BeginFigure( pt, D2D1_FIGURE_BEGIN_FILLED );
    while (!!points) {
      pointf n = points++;
      this->AddLine(p.x, p.y, n.x, n.y);
      p = n;
    }
    this->CloseFigure();
    this->SetFillMode(fill_even_odd ? FillModeAlternate : FillModeWinding);
  }

  void path::set(const polygonf &v1, const polygonf &v2) {
    slice<pointf> points = v1();

    if (points.length == 0) return;

    this->StartFigure();
    pointf p = points++;
    // ps->BeginFigure( pt, D2D1_FIGURE_BEGIN_FILLED );
    while (!!points) {
      pointf n = points++;
      this->AddLine(p.x, p.y, n.x, n.y);
      p = n;
    }
    this->CloseFigure();

    points = v2();
    this->StartFigure();
    p = points++;
    while (!!points) {
      pointf n = points++;
      this->AddLine(p.x, p.y, n.x, n.y);
      p = n;
    }
    this->CloseFigure();

    this->SetFillMode(FillModeAlternate);
  }

  /*void path::sink_t::reset()
  {
    ps->Close();
    this->opened = false;
  }*/

  void path::start(pointf start, bool filled) {
    if (_opened) { this->CloseFigure(); }
    this->_sealed = false;
    this->_opened = true;
    this->filled  = filled;

    _firstp = _lastp = start;

    this->StartFigure();

    // ps->BeginFigure( d2d::g2f(firstp = _lastp = start), filled?
    // D2D1_FIGURE_BEGIN_FILLED:D2D1_FIGURE_BEGIN_HOLLOW );
  }
  void path::move_to(pointf pt, bool rel) {
    // if( opened )
    //{
    // ps->EndFigure( D2D1_FIGURE_END_OPEN );
    //  opened = false;
    //}
    if (rel)
      _lastp += pt;
    else
      _lastp = pt;
    if (!_opened) start(_lastp, true);
  }
  void path::line_to(pointf pt, bool rel) {
    _sealed = false;
    if (is_empty()) start(_lastp, true);
    if (rel) pt += _lastp;
    this->AddLine(_lastp.x, _lastp.y, pt.x, pt.y);
    _lastp = pt;
  }

  void path::close() {
    if (!is_empty()) {
      // ps->SetFillMode(D2D1_FILL_MODE_ALTERNATE);
      _sealed = false;
      // ps->EndFigure( D2D1_FIGURE_END_CLOSED );
      this->CloseFigure();
      _opened = false;
    }
  }

  void path::seal() {
    if (!_sealed) {
      if (!is_empty()) {
        // ps->EndFigure( D2D1_FIGURE_END_OPEN );
        _opened = false;
      }
      // ps->Close();
    }
  }

  rectf path::bounds() const {
    const_cast<path *>(this)->seal();
    RectF b;
    this->GetBounds(&b);
    return rectf(b.X, b.Y, b.X + b.Width, b.Y + b.Height);
  }

  void path::add_arc(pointf c, sizef r, float angle_start, float angle_swipe) {
    rectf rc(c, c);
    rc >>= r;

    this->AddArc(rc.left(), rc.top(), rc.width() - 1, rc.height() - 1,
                 angle_start * 57.2957795f, angle_swipe * 57.2957795f);
  }

  void path::quadratic_to(pointf pt, pointf cp, bool rel) {
    if (is_empty()) start(_lastp, true);
    _sealed = false;
    if (rel) {
      pt += _lastp;
      cp += _lastp;
    }

    // quadratic-bezier to cubic-bezier :
    pointf cp1 = _lastp + (cp - _lastp) * 2.0f / 3.0f;
    pointf cp2 = pt + (cp - pt) * 2.0f / 3.0f;

    this->AddBezier(g2p(_lastp), g2p(cp1), g2p(cp2), g2p(pt));

    _lastp = pt;
  }

  void path::cubic_to(pointf pt, pointf cp1, pointf cp2, bool rel) {
    // this->line_to(pt);
    // return;
    if (is_empty()) start(_lastp, true);

    _sealed = false;
    if (rel) {
      pt += _lastp;
      cp1 += _lastp;
      cp2 += _lastp;
    }

    this->AddBezier(g2p(_lastp), g2p(cp1), g2p(cp2), g2p(pt));
    _lastp = pt;

    // D2D1_BEZIER_SEGMENT bs;
    // bs.point1 = d2d::g2f(cp1);
    // bs.point2 = d2d::g2f(cp2);
    // bs.point3 = d2d::g2f(_lastp = pt);
    // ps->AddBezier(bs);
  }

  void path::set_even_odd(bool even_odd) {
    // ps->SetFillMode(even_odd?D2D1_FILL_MODE_ALTERNATE:D2D1_FILL_MODE_WINDING);
    this->SetFillMode(even_odd ? FillModeAlternate : FillModeWinding);
  }

  image_renderer_f graphics::image_renderer(image *img) {

    handle<bitmap> bmp = img->get_bitmap(this, img->dimension());
    if (!bmp) return super::image_renderer(img);

    handle<Graphics> target = _target;
    handle<Bitmap>   tbmp   = GdiBitmap(bmp);

    return [=](rect dst, rect src) {

      RectF rdst = g2p(rectf(dst));
      RectF rsrc = g2p(rectf(src));

      // dirty hack "fixing" border effects in weird GDI+ image stretching :(
      if (rdst.Width > rsrc.Width) rsrc.Width -= 1.0f;
      if (rdst.Height > rsrc.Height) rsrc.Height -= 1.0f;

      assert(rdst.Width > 0 && rdst.Height > 0);
      assert(rsrc.Width > 0 && rsrc.Height > 0);

      Status st = target->DrawImage(tbmp, rdst, rsrc.X, rsrc.Y, rsrc.Width,
                                    rsrc.Height, UnitPixel);
      // assert(st == Ok);
      st = st;

    };
  }

  void graphics::do_draw(image *img, const rectf &dst, const rect &_src,
                         byte opacity) {
    handle<bitmap> bitmap = img->get_bitmap(this, dst.size());

    rect src = _src & rect(img->dim());

    RectF rdst = g2p(dst);
    RectF rsrc = g2p(rectf(src));

    // dirty hack "fixing" border effects in weird GDI+ image stretching :(
    // does not work for <img>: if( rdst.Width > rsrc.Width ) rsrc.Width
    // -= 1.0f; does not work for <img>: if( rdst.Height > rsrc.Height )
    // rsrc.Height -= 1.0f;

    Status st;
    // assert(opacity == 255);
    if (opacity < 255) {
      float alpha = float(opacity) / 255.0f;

      ColorMatrix color_matrix = {1.0f, 0.0f, 0.0f, 0.0f, 0.0f,  0.0f, 1.0f,
                                  0.0f, 0.0f, 0.0f, 0.0f, 0.0f,  1.0f, 0.0f,
                                  0.0f, 0.0f, 0.0f, 0.0f, alpha, 0.0f, 0.0f,
                                  0.0f, 0.0f, 0.0f, 1.0f};

      ImageAttributes image_attributes;
      image_attributes.SetColorMatrix(&color_matrix, ColorMatrixFlagsDefault,
                                      ColorAdjustTypeBitmap);
      st = _target->DrawImage(this->GdiBitmap(bitmap), rdst, rsrc.X, rsrc.Y,
                              rsrc.Width, rsrc.Height, UnitPixel,
                              &image_attributes);

    } else
      st = _target->DrawImage(this->GdiBitmap(bitmap), rdst, rsrc.X, rsrc.Y,
                              rsrc.Width, rsrc.Height, UnitPixel);

    // assert(st == Ok);
  }

  graphics::graphics(gool::bitmap *bmp, argb initc, bool high_quality): super(!initc.is_opaque())  {
    _prefer_quality = high_quality;
    Bitmap *pbmp    = this->GdiBitmap(bmp);
    _target         = new Graphics(pbmp);
    if (initc != argb::undefined()) _target->Clear(g2p(initc));
    // Status st = _target->SetSmoothingMode(SmoothingModeDefault);
    // assert(st == Ok);
    set_clip_rc(rect(bmp->dim()));
    setup_target();
  }

  void graphics::setup_target() {
    _target->SetPageUnit(UnitPixel);
    //_target->SetPageScale(1.0);
    if (_prefer_quality) {
      _target->SetCompositingQuality(CompositingQualityDefault);
      _target->SetPixelOffsetMode(
          PixelOffsetModeHighQuality); // PixelOffsetModeHalf
      _target->SetSmoothingMode(
          SmoothingModeHighQuality); // SmoothingModeHighQuality
      _target->SetInterpolationMode(
          InterpolationModeHighQuality); // InterpolationModeHighQuality
      _target->SetTextRenderingHint(TextRenderingHintAntiAlias);
    } else {
      _target->SetCompositingQuality(CompositingQualityHighSpeed);
      _target->SetPixelOffsetMode(
          PixelOffsetModeHighSpeed); // PixelOffsetModeHalf
      _target->SetSmoothingMode(
          SmoothingModeHighSpeed /*SmoothingModeHighQuality*/); // SmoothingModeHighQuality
      _target->SetInterpolationMode(
          InterpolationModeBilinear); // InterpolationModeHighQuality
      _target->SetTextRenderingHint(TextRenderingHintSystemDefault);
    }
  }

  bool graphics::set_antialiasing(bool onoff) {
    auto aa = _target->GetSmoothingMode();
    bool r  = aa >= SmoothingModeAntiAlias;
    _target->SetSmoothingMode(onoff ? SmoothingModeAntiAlias
                                    : SmoothingModeNone);
    return r;
  }

  graphics::graphics(HWND hwnd): super(false) {
    _target = new Graphics(hwnd);
    setup_target();
  }
  graphics::graphics(HDC hdc) : super(false) {
    _target = new Graphics(hdc);
    setup_target();
  }

  graphics::graphics(Gdiplus::Bitmap *pbm, argb initc, bool high_quality): super( !initc.is_opaque() ) {
    _prefer_quality = high_quality;
    _target         = new Graphics(pbm);
    _target->Clear(g2p(initc));
    size dim(pbm->GetWidth(), pbm->GetHeight());
    set_clip_rc(rect(dim));
    setup_target();
  }

  gool::path *gdi_create_path() { return new gdi::path(); }

  gool::graphics *gdi_create_bitmap_graphics(gool::graphics *proto,
                                             gool::bitmap *  bmp,
                                             gool::argb      initc) {
    return new graphics(bmp, initc);
  }

  bitmap_bits_graphics::~bitmap_bits_graphics() {
    Bitmap *pbmp = this->GdiBitmap(pbitmap);
    uint    format =
        pbitmap->_has_alpha_bits ? PixelFormat32bppPARGB : PixelFormat32bppRGB;
    Rect       rect(0, 0, pbitmap->dim().x, pbitmap->dim().y);
    BitmapData bmpdata;
    pbmp->LockBits(&rect, ImageLockModeRead, format, &bmpdata);
    slice<argb> bits((argb *)bmpdata.Scan0, bmpdata.Width * bmpdata.Height);
    pbitmap->set_bits(bits);
    pbmp->UnlockBits(&bmpdata);
  }

  gool::graphics *gdi_create_bitmap_bits_graphics(gool::bitmap *bmp,
                                                  gool::argb    initc,
                                                  bool          high_quality) {
    return new bitmap_bits_graphics(bmp, initc, high_quality);
  }

#if 0
  gool::text_layout* printing_application::create_text_layout(tool::wchars text, const gool::text_format& tf)
  {
    return new gdi::text_layout(resolution_provider::desktop(), text,tf,this);
  }
#endif

  Pen *graphics::state_registers::stroke_pen() {
    if (!_stroke_pen) {
      if (this->stroke_width > 0.0f && stroke_brush != 0) {
        _stroke_pen = new Pen(stroke_brush, stroke_width);
        switch (cap_style) {
        case CAP_BUTT:
          _stroke_pen->SetEndCap(LineCapFlat);
          _stroke_pen->SetStartCap(LineCapFlat);
          break;
        case CAP_SQUARE:
          _stroke_pen->SetEndCap(LineCapSquare);
          _stroke_pen->SetStartCap(LineCapSquare);
          break;
        case CAP_ROUND:
          _stroke_pen->SetEndCap(LineCapRound);
          _stroke_pen->SetStartCap(LineCapRound);
          break;
        case CAP_TRIANGLE:
          _stroke_pen->SetEndCap(LineCapTriangle);
          _stroke_pen->SetStartCap(LineCapTriangle);
          break;
        }
        switch (line_join) {
        case JOIN_MITER: _stroke_pen->SetLineJoin(LineJoinMiter); break;
        case JOIN_BEVEL: _stroke_pen->SetLineJoin(LineJoinBevel); break;
        case JOIN_ROUND: _stroke_pen->SetLineJoin(LineJoinRound); break;
        case JOIN_MITER_OR_BEVEL:
          _stroke_pen->SetLineJoin(LineJoinMiterClipped);
          break;
        }
        _stroke_pen->SetMiterLimit(1.0);

        switch (dash_style) {
        case DASH_STYLE_SOLID: _stroke_pen->SetDashStyle(DashStyleSolid); break;
        case DASH_STYLE_DASH: _stroke_pen->SetDashStyle(DashStyleDash); break;
        case DASH_STYLE_DOT: _stroke_pen->SetDashStyle(DashStyleDot); break;
        case DASH_STYLE_DASH_DOT:
          _stroke_pen->SetDashStyle(DashStyleDashDot);
          break;
        case DASH_STYLE_DASH_DOT_DOT:
          _stroke_pen->SetDashStyle(DashStyleDashDotDot);
          break;
        case DASH_STYLE_CUSTOM:
          _stroke_pen->SetDashPattern(custom_dash_style.cbegin(),
                                      uint32(custom_dash_style.length()));
          _stroke_pen->SetDashOffset(dash_offset / stroke_width);
          break;
        }
      }
    }
    return _stroke_pen;
  }

} // namespace gdi

#endif
