#ifndef __d2d_view_h__
#define __d2d_view_h__

#include "tool/tool.h"
#include "gool/gool.h"
#include "html/html-layout.h"
#include "html/html-view-print.h"
#include "d2d-objects.h"
#include "d2d-primitives.h"
#include "d2d-graphics.h"
#include "d2d-window-graphics.h"
#include "win/win-view.h"
#include "win/win-frame.h"

EXTERN_DLOADV(_DwmFlush, DwmFlush, dwmapi.dll,
  HRESULT(WINAPI *)());

namespace d2d {
  using namespace tool;
  using namespace gool;

  class application;

  class window_graphics : public graphics {
    friend class application;
    friend class window;
    typedef graphics super;
  protected:
    window_graphics(bool transparent) : super(transparent), _graphics_caps() {}

  public:
    static window_graphics *create(html::iwindow *pw, bool is_transparent,
      GRAPHICS_CAPS caps);

    virtual ~window_graphics() { }

    virtual void resize(gool::size nsz) {
      if (!render_target())
        return;

      com::asset<ID2D1HwndRenderTarget> hwnd_rt;
      render_target()->QueryInterface<ID2D1HwndRenderTarget>(hwnd_rt.target());

      if (hwnd_rt && !nsz.empty() && nsz != _sz) {
        hwnd_rt->Resize(g2u(nsz));
        _sz = nsz;
      }
      set_clip_rc(nsz);
    }

    virtual GRAPHICS_CAPS graphics_caps() const { return _graphics_caps; }

  protected:
    GRAPHICS_CAPS                _graphics_caps;
    gool::size                   _sz;
  };

#if defined(USE_D2D_PLUS)
  class window_graphics_dxgi : public graphics {
    friend class application;
    friend class window;
    typedef graphics super;
  protected:
    window_graphics_dxgi(html::iwindow *pw, bool transparent, GRAPHICS_CAPS caps,
      bool for_layered)
      : super(transparent), _graphics_caps(), _window(pw),
      _requested_caps(caps), _for_layered(for_layered) {}

  public:
    static bool                  is_feasible();
    static window_graphics_dxgi *create(html::iwindow *pw, bool is_transparent,
      GRAPHICS_CAPS caps, bool for_layered);

    virtual ~window_graphics_dxgi() {
      //_render_hwnd_target = 0;
      release_device();
    }

    virtual void resize(gool::size nsz);

    virtual GRAPHICS_CAPS graphics_caps() const { return _graphics_caps; }

    virtual void end_drawing() override;

  protected:
    void release_device();
    virtual bool create_device_swap_chain_bitmap();
    virtual bool create_device();

    html::iwindow*               _window;
    com::asset<IDXGISwapChain1> _swap_chain;
    com::asset<ID2D1Bitmap1>    _bitmap;
    GRAPHICS_CAPS               _graphics_caps;
    GRAPHICS_CAPS               _requested_caps;
    bool                        _for_layered;
  };
#endif

  class dx_window_graphics : public graphics {
    friend class application;
    friend class window;
    typedef graphics super;
  protected:
    dx_window_graphics(bool transparent) : super(transparent), _graphics_caps() {}

  public:
    static dx_window_graphics *create(html::iwindow * pw,
      IDXGISwapChain *pSwapChain);

    virtual ~dx_window_graphics() {}

    virtual GRAPHICS_CAPS graphics_caps() const { return _graphics_caps; }

  protected:
    com::asset<ID2D1RenderTarget> _target;
    GRAPHICS_CAPS            _graphics_caps;
  };


#if defined(USE_DX_ON_LAYERED)
  class layered_window_graphics_dx : public window_graphics {
    typedef window_graphics super;
#ifdef USE_D2D_PLUS
    com::asset<ID3D11Device>       _device;
    com::asset<ID3D11Texture2D>    _texture;
#else
    com::asset<ID3D10Device1>       _device;
    com::asset<ID3D10Texture2D>     _texture;
#endif
    com::asset<IDXGISurface>      _surface;
    layered_window_graphics_dx() : window_graphics(true) { _graphics_caps = HARDWARE_GRAPHICS; }

  public:
    static layered_window_graphics_dx *create(html::iwindow *pw,
      GRAPHICS_CAPS  caps);


    virtual void resize(gool::size nsz) override;
  };

#endif

  class layered_window_graphics_hdc : public window_graphics {
    typedef window_graphics    super;
    handle<dib32>              _bitmap;
    D2D1_RENDER_TARGET_TYPE    _render_target_type;

    layered_window_graphics_hdc(GRAPHICS_CAPS caps) : super(true) {
      if (caps == HARDWARE_GRAPHICS)
        _render_target_type = D2D1_RENDER_TARGET_TYPE_DEFAULT;
      else
        _render_target_type = D2D1_RENDER_TARGET_TYPE_SOFTWARE;
    }

  public:
    virtual GRAPHICS_CAPS graphics_caps() const {
      return _render_target_type == D2D1_RENDER_TARGET_TYPE_HARDWARE
        ? HARDWARE_GRAPHICS
        : WARP_GRAPHICS;
    }

    static layered_window_graphics_hdc *create(html::iwindow *pw,
      GRAPHICS_CAPS  caps);

    virtual void resize(gool::size nsz) override;
  };

  class window : public mswin::window {
    typedef mswin::window super;

  public:
    window(const window_params& params);

    virtual gool::graphics *surface() override;

    virtual GRAPHICS_CAPS graphics_caps() const override;
    ustring               graphics_backend() const override;

    static LPCWSTR        class_name_suffix() { return W(""); }

    virtual bool needs_full_window_refresh() const { return true; }

    // events
    // virtual void attached();
    virtual void on_size(size sz) override;

    virtual bool render(void *hdc, rect paint_rc) override;
    virtual bool print(void *hdc, rect paint_rc) override;

    // virtual void render(html::element* el = 0) override;
    virtual bool render_on(gool::graphics *gfx,
                           rect            paint_rc = rect()) override;
    virtual bool render_layered(void *dc, rect client_rc) override;
    virtual bool render_element(element *p, bool front_layer) override;
    virtual bool render_element_on(element *p, gool::graphics *gfx) override;

    // virtual void test_gfx_paint( point pos ); // for testing purposes
    virtual void clear_surface(ID2D1RenderTarget* rt);

    virtual bool get_transparency() const override { return !!_is_transparent; }
    virtual void set_transparency(bool on_off) override {
      _is_transparent = on_off;
      _gfx            = 0;
      notify_media_change();
    }

    virtual bool set_blurbehind(BLUR_BEHIND on) {
      if (!super::set_blurbehind(on))
        return false;
      if (_DwmSetWindowAttribute) {
        auto exstyle = ::GetWindowLong(get_hwnd(), GWL_EXSTYLE);
        if (on) {
          ::SetWindowLong(get_hwnd(), GWL_EXSTYLE, exstyle | WS_EX_NOREDIRECTIONBITMAP);
          //BOOL on = TRUE;
          //_DwmSetWindowAttribute()(get_hwnd(), DWMWA_HAS_ICONIC_BITMAP, &on, sizeof(on));
        }
        else {
          ::SetWindowLong(get_hwnd(), GWL_EXSTYLE, exstyle & ~WS_EX_NOREDIRECTIONBITMAP);
          //BOOL on = FALSE;
          //_DwmSetWindowAttribute()(get_hwnd(), DWMWA_HAS_ICONIC_BITMAP, &on, sizeof(on));
        }
      }

      _gfx = nullptr;
      return true;
    }

    virtual bool is_direct() const override { 
      //if (!dcomp_device()) return false;
      return 0 != (::GetWindowLong(get_hwnd(), GWL_EXSTYLE) & WS_EX_NOREDIRECTIONBITMAP);
    }

    virtual void set_layered(bool onoff) override {
      super::set_layered(onoff);
      _gfx = 0;
      notify_media_change();
    }

    virtual bool set_frame_type(FRAME_TYPE on) override {
      _gfx = 0;
      return super::set_frame_type(on);
    }

    virtual mswin::popup *create_popup() override;
    //virtual gool::text_layout * create_text_layout(tool::wchars text, const gool::text_format &tf) override;

    virtual void
    setup_text_flow(html::element *elem, html::tflow::text_flow &tf,
                    tool::slice<tool::handle<html::node>> nodes) override;
    // virtual void draw_glyph_runs( html::element* elem, const
    // html::tflow::text_flow& tf, gool::graphics* gfx, gool::point at,
    // html::tflow::sel_context* sctx) override;
    virtual void draw_glyph_run(gool::graphics *              gfx,
                                const html::tflow::text_flow &tf,
                                const html::tflow::glyph_run &gr,
                                gool::pointf at, argb color,
                                const style *run_style) override;

    virtual void reset_surface() override {
      if (doc()) doc()->drop_caches();
      _gfx = 0;
      refresh();
    }

    virtual bool on_activate(ACTIVATE_MODE am) override {
      bool r = super::on_activate(am);
      if (_gfx)
      {
        _gfx->on_window_active_state_change(am != ACTIVATE_MODE::INACTIVE);
        render(nullptr, rect(client_dim()));
      }
      return r;
    }

    virtual void   handle_position_change() override {
      if (_gfx)
        _gfx->on_window_position_change();
    }

    virtual void update() override {
      super::update();
      if (_DwmFlush)
        _DwmFlush()();
    }


  protected:
    // sizef                 _dpi_scale; // how much to scale a design that
    // assumes 96-DPI pixels
    handle<graphics> _gfx;
    tristate_v       _is_transparent;
    GRAPHICS_CAPS    _used_graphics;
    tristate_v       _done_drawing;
  };


  class dx_window : public window {
    typedef window super;

  public:
    dx_window(const window_params& params,IDXGISwapChain *pSwapChain) : super(params), _swap_chain(pSwapChain) {}

    virtual void do_paint() override {
      /* does nothing, render() used instead */
      PAINTSTRUCT ps;
      HDC         hdc;
      hdc = BeginPaint(get_hwnd(), &ps);
      EndPaint(get_hwnd(), &ps);
    }
    virtual void            on_size(size sz) override;
    virtual gool::graphics *surface() override;

    virtual void update() override {
      mswin::window::update();
    }

  protected:
    com::asset<IDXGISwapChain> _swap_chain;
  };


  class popup : public mswin::popup {
    friend class window;
    typedef mswin::popup super;

  protected:
    handle<d2d::graphics> _gfx;

    popup() {}
    virtual ~popup() { _gfx = 0; }

  public:
    // iwindow stuff:
    virtual graphics *surface() override;

    virtual bool render_area(void *hdc, rect rc);
    virtual bool render(void *hdc, rect rc) override;
    virtual bool print(void *hdc, rect rc) override;
    virtual bool render_layered(void *dc, rect client_rc) override;
  };

  inline mswin::popup *window::create_popup() { return new popup(); }

  typedef mswin::frame<window>  frame;
  typedef mswin::dialog<window> dialog;

  class print_view : public html::print_view {
  protected:
    friend class application;
    typedef html::print_view super;
    print_view(const window_params& params): super(params) {}

  public:
    virtual void
                 setup_text_flow(html::element *elem, html::tflow::text_flow &tf,
                                 tool::slice<tool::handle<html::node>> nodes) override;
    virtual void draw_glyph_run(gool::graphics *              gfx,
                                const html::tflow::text_flow &tf,
                                const html::tflow::glyph_run &gr,
                                gool::pointf at, argb color,
                                const html::style *run_style) override;
    virtual void draw_glyph_run_back(gool::graphics *              gfx,
                                     const html::tflow::text_flow &tf,
                                     const html::tflow::glyph_run &gr,
                                     gool::rectf rc, argb back_color) override;

    //virtual html::text_layout *create_text_layout(wchars text, const text_format &tf) override;

    //#endif
  };

} // namespace d2d

#endif