#include "d2d.h"
#include "html/html.h"
#include "win/win-layered.h"

DLOADV(_DwmFlush, DwmFlush, dwmapi.dll,
  HRESULT(WINAPI *)());

namespace d2d {
  using namespace mswin;

  window::window(const window_params& params): super(params) {
    _used_graphics = app->graphics_caps();
    // init_media_vars();
  }


#if TRUE // must use window_graphics_dxgi in order to support CSS filters 
  gool::graphics* window::surface()
  {
    assert(get_hwnd());

    if( !_gfx ) 
    {
      if(is_layered()) {
#if defined(USE_DX_ON_LAYERED)
        if( _used_graphics >= WARP_GRAPHICS )
           _gfx = layered_window_graphics_dx::create(this,_used_graphics);
        if(!_gfx)
#endif
          _gfx = layered_window_graphics_hdc::create(this,_used_graphics);
      }
      else {
#if defined(USE_D2D_PLUS)
        if (_used_graphics == HARDWARE_GRAPHICS && !is_child() && environment::get_os_version() >= environment::WIN_10 )
        {
          if (get_blurbehind())
            _gfx = window_bb_graphics::create(this, !!_is_transparent, HARDWARE_GRAPHICS, false);
          //if(!_gfx) - don't need _dxgi here, window_graphics will work just fine
          //  _gfx = window_graphics_dxgi::create(this, !!_is_transparent, _used_graphics, false);
        }
        if(!_gfx)
#endif
          _gfx = window_graphics::create(this,!!_is_transparent, _used_graphics);
      }

      if(!_gfx)
      {
        _gfx = window_graphics::create(this, !!_is_transparent, WARP_GRAPHICS);
        _used_graphics = WARP_GRAPHICS;
        assert(_gfx);
      }
      else 
        _used_graphics = _gfx->graphics_caps();
    }
    return _gfx;
  }
#else
  gool::graphics *window::surface() {
    assert(get_hwnd());

    if (!_gfx) {
      if (is_layered()) {
#if defined(USE_DX_ON_LAYERED)
        if (_used_graphics >= HARDWARE_GRAPHICS)
          _gfx = layered_window_graphics_dx::create(this, _used_graphics);
        if (!_gfx)
#endif
          _gfx = layered_window_graphics_hdc::create(this, _used_graphics);
      } else
        _gfx = window_graphics::create(this, !!_is_transparent, _used_graphics);
    }

    if (!_gfx) {
      _gfx = window_graphics::create(this, !!_is_transparent, WARP_GRAPHICS);
      _used_graphics = WARP_GRAPHICS;
      assert(_gfx);
    } else
      _used_graphics = _gfx->graphics_caps();

    return _gfx;
  }

#endif

  void window::on_size(size sz) {
    //wrong: if (dim == sz)
    //  return; breaks Notes ?
    gool::graphics *pg = surface();
    if (pg) {
      pg->resize(sz);
      if (!_scrollbar_handling) super::on_size(sz);
    } else
      super::on_size(sz);
    update();
  }

  // request parent window to draw area underneath the window.
  void request_window_background(HWND hwnd, HDC hdc) {
    HWND hwndParent = GetParent(hwnd);
    if (::IsWindow(hwndParent)) {
      POINT pt = {0};
      MapWindowPoints(hwnd, hwndParent, &pt, 1);

      RECT rc;
      GetClipBox(hdc, &rc);

      int saved = SaveDC(hdc);

      if (::GetWindowLong(hwndParent, GWL_EXSTYLE) & WS_EX_LAYOUTRTL) {
        RECT wrc;
        ::GetWindowRect(hwnd, &wrc);
        POINT ptr;
        ptr.x = -(pt.x - (wrc.right - wrc.left)) - rc.left;
        ptr.y = -pt.y - rc.top;
        ::SetViewportOrgEx(hdc, ptr.x, ptr.y, &pt);
        SetLayout(hdc, LAYOUT_RTL);
      } else
        ::SetViewportOrgEx(hdc, -pt.x - rc.left, -pt.y - rc.top, &pt);

      LRESULT lr = ::SendMessage(hwndParent, WM_ERASEBKGND, WPARAM(hdc), 0);
      if (!lr)
        ::SendMessage(hwndParent, WM_PRINTCLIENT, WPARAM(hdc), PRF_CLIENT);

      RestoreDC(hdc, saved);
    }
  }

  void window::clear_surface(ID2D1RenderTarget* rt) 
  {
    html::BLUR_BEHIND bb = get_blurbehind();
    gool::argb back;
    switch (bb) {
      case html::BLUR_BEHIND_ULTRA_DARK:  back = gool::argb(0, 0, 0, 179); break;
      case html::BLUR_BEHIND_DARK:        back = gool::argb(16, 16, 16, 179); break;
      case html::BLUR_BEHIND_LIGHT:       back = gool::argb(230, 230, 230, 179); break;
      case html::BLUR_BEHIND_ULTRA_LIGHT: back = gool::argb(255, 255, 255, 179); break;
      default: back = gool::argb(255, 255, 255, 0); break;
    }
    if (!is_active())
      back.alfa = 255;
    rt->Clear(g2f(back));
  }

  bool window::render(void *hdc, rect paint_rc) {
    
    if (is_painting) return false;
    if (!is_window_visible()) {
      //reset_surface();
      return false;
    }

    for (int n = 0; n < 3; ++n) {
      d2d::graphics *gfx = static_cast<d2d::graphics *>(surface());

      if (!gfx) return false;

#ifdef _DEBUG
      if (is_layered())
        n = n;
#endif 

      auto_state<bool>             _(is_painting, true);
      auto_state<gool::graphics *> _2(drawing_surface, gfx);
      handle<gool::graphics>       _3(gfx);

      com::asset<ID2D1RenderTarget> rt = gfx->render_target();
      if (!rt) return false;

      if (paint_rc.empty() /*|| (gfx->graphics_caps() == HARDWARE_GRAPHICS)*/)
        gfx->set_clip_rc(paint_rc = dimension());
      else
        gfx->set_clip_rc(paint_rc);

      if (paint_rc.empty())
        return true;

      gfx->begin_drawing();

      rt->BeginDraw();
      // paint_rc >>= 3;
      D2D1_RECT_F clip = g2f(paint_rc);
      rt->PushAxisAlignedClip(clip, D2D1_ANTIALIAS_MODE_ALIASED);

      clear_surface(rt);

      if (_is_child && doc() && doc()->get_style(*this)->is_transparent()) {
        if (!_background_bitmap ||
            _background_bitmap->dimension() != this->dim) {
          dib32 dib(this->dim);
          request_window_background(get_hwnd(), dib.DC());
          _background_bitmap = new bitmap(&dib);
        }
        gfx->draw(_background_bitmap, rect(this->dim));
      }

      paint();
      rt->PopAxisAlignedClip();

#ifdef DEBUG_DRAW_PIPELINE
      D2D1_TAG t1 = 0;
      HRESULT  hr = rt->EndDraw(&t1);
#else
      HRESULT hr = rt->EndDraw();
#endif

      if (FAILED(hr)) {
        if (hr == D2DERR_RECREATE_TARGET) {
          this->_gfx = 0;
          paint_rc   = rect(0, 0, -1, -1);
          if (doc()) doc()->drop_caches();
          continue; // another attempt...
        }
        if (_used_graphics == HARDWARE_GRAPHICS) {
          _used_graphics = WARP_GRAPHICS;
          this->_gfx     = 0;
          paint_rc       = rect(0, 0, -1, -1);
          if (doc()) doc()->drop_caches();
          continue; // another attempt...
        }

#ifdef DEBUG_DRAW_PIPELINE
        if (t1) {
          element *t = (element *)t1;
          error_report("Drawing failure on DOM element:", t);
        }
#endif
        assert(false);
        _gfx = 0;
        if (doc()) doc()->drop_caches();
      } else 
        gfx->end_drawing();
      break;
    }
    return true;
  }

  GRAPHICS_CAPS window::graphics_caps() const { return this->_used_graphics; }

  ustring window::graphics_backend() const {
    switch (this->graphics_caps()) {
    case WARP_GRAPHICS: return ustring(W("direct2d-warp"));
    case HARDWARE_GRAPHICS: return ustring(W("direct2d"));
    default: return ustring(W("{unknown}"));
    }
  }

  //#ifdef _DEBUG
  //  static uint prev_tick = 0;
  //#endif

  bool window::render_layered(void *dc, rect client_rc) {

    if (!is_window_visible()) {
      reset_surface();
      return false;
    }

    super::render_layered(dc,client_rc);
    
    bool drawn_nothing = false;

    size client_dim = this->client_dim();
    if (client_dim.empty()) return false;

    if (!doc()) return false;

    if (dim != client_dim) html::view::on_size(client_dim);
    commit_update();

    for (int n = 0; n < 3; ++n) {

      d2d::graphics *gfx = static_cast<d2d::graphics *>(surface());

      if (!gfx) goto CRITICAL_ERROR;

      auto_state<bool> _(is_painting, true);

      com::asset<ID2D1RenderTarget> rt = gfx->render_target();
      if (!rt) return false;

      rt->BeginDraw();

      auto_state<gool::graphics *> _1(drawing_surface, _gfx.ptr());

      rect paint_rc = client_dim; // invalid; - layered gets redrawn in full.

      gfx->set_clip_rc(paint_rc);

      if (!doc()->is_layout_valid()) {
        doc()->commit_measure(*this);
        paint_rc = client_dim;
      }

      D2D1_RECT_F clip = g2f(paint_rc);
      rt->PushAxisAlignedClip(clip, D2D1_ANTIALIAS_MODE_ALIASED);
      rt->Clear(D2D1::ColorF(0x000000, !_done_drawing ? 0.01f : 0.0f));
      paint();
      rt->PopAxisAlignedClip();

      bool first_draw = !_done_drawing;

      do {
        com::asset<ID2D1GdiInteropRenderTarget> gdi_rt;
        HRESULT hr = rt->QueryInterface<ID2D1GdiInteropRenderTarget>(gdi_rt.target());

        if (FAILED(hr)) {
          assert(false); // not ID2D1GdiInteropRenderTarget compatible?
          goto CRITICAL_ERROR;
        }

        layered_window_ctx lwctx(client_dim);
        HDC                hdc = 0;
        hr = gdi_rt->GetDC(D2D1_DC_INITIALIZE_MODE_COPY, &hdc);

        if (hr == D2DERR_TARGET_NOT_GDI_COMPATIBLE) {
          assert(false); // why it is not compatible?
          break;
        } else if (FAILED(hr)) {
          assert(false); // why compatible?
          break;
        } else {
          if (!_done_drawing) { // WHAT A DIRTY HACK IS IT! ATI, SHAME ON YOU!
            // some cards report S_OK on drawings but actually produce nothing
            // ...
            size  probe_size(min(10, client_dim.x), min(10, client_dim.y));
            dib32 probe(probe_size);
            drawn_nothing = true;
            if (!probe.is_valid())
              break;
            BitBlt(probe.DC(), 0, 0, probe_size.x, probe_size.y, hdc, 0, 0, SRCCOPY);
            auto pixels = probe.pixels();
            gool::argb noc(0, 0, 0, 0);
            for (uint n = 0; n < pixels.length; ++n) {
              gool::argb col = pixels[n];
              if (col != noc) {
                drawn_nothing = false;
                refresh();
                break;
              }
            }
          }
          lwctx.update(get_hwnd(), hdc /*,&paint_rc*/);
          hr = gdi_rt->ReleaseDC(0);
        }
      } while (false);

      HRESULT hr = rt->EndDraw();

      if (FAILED(hr) || drawn_nothing) {
        if (doc()) doc()->drop_caches();

        if (hr == D2DERR_RECREATE_TARGET) {
          _gfx    = 0;
          invalid = rect();
          continue; // another attempt...
        }
        // assert(false);
        _gfx = 0;
        switch (this->_used_graphics) {
        case HARDWARE_GRAPHICS: _used_graphics = WARP_GRAPHICS; continue;
        case WARP_GRAPHICS: _used_graphics = SOFTWARE_GRAPHICS; continue;
        case SOFTWARE_GRAPHICS: _used_graphics = UNKNOWN_GRAPHICS; break;
        }
      } else if (!first_draw)
        invalid = rect();

      break;
    }

    _done_drawing = true;

    if (drawn_nothing) {
    CRITICAL_ERROR:
      this->on_graphics_critical_failure();
    }

    return true;
  }


  bool window::render_element(element *el, bool front_layer) {
    size client_dim = this->client_dim();
    if (client_dim.empty()) return false;

    if (el->pview() != this) return false;

    handle<document> pdoc = doc();
    if (!pdoc) return false;

    if (!front_layer) {
      if (has_animations())
        on_animation_tick();
      check_timers_overdue_in_all_views();
    }

    for (int n = 0; n < 3; ++n) {

      d2d::graphics *gfx = static_cast<d2d::graphics *>(surface());

      if (!gfx) goto CRITICAL_ERROR;

      auto_state<bool> _(is_painting, true);

      com::asset<ID2D1RenderTarget> rt = gfx->render_target();
      if (!rt) return false;

      rt->BeginDraw();

      auto_state<gool::graphics *> _1(drawing_surface, _gfx.ptr());

      rect paint_rc = client_dim; // invalid; - layered gets redrawn in full.

      gfx->set_clip_rc(paint_rc);

      if (!pdoc->is_layout_valid()) {
        pdoc->commit_measure(*this);
        paint_rc = client_dim;
      }

      D2D1_RECT_F clip = g2f(paint_rc);
      rt->PushAxisAlignedClip(clip, D2D1_ANTIALIAS_MODE_ALIASED);
      if (!el || !front_layer)
        rt->Clear(D2D1::ColorF(0x000000, !_done_drawing ? 0.01f : 0.0f));
      paint(el, front_layer);
      rt->PopAxisAlignedClip();

      HRESULT hr = rt->EndDraw();

      if (FAILED(hr)) {
        pdoc->drop_caches();

        if (hr == D2DERR_RECREATE_TARGET) {
          _gfx    = 0;
          invalid = rect();
          continue; // another attempt...
        }
        // assert(false);
        _gfx = 0;
        switch (this->_used_graphics) {
        case HARDWARE_GRAPHICS: _used_graphics = WARP_GRAPHICS; continue;
        case WARP_GRAPHICS: _used_graphics = SOFTWARE_GRAPHICS; continue;
        case SOFTWARE_GRAPHICS: _used_graphics = UNKNOWN_GRAPHICS; break;
        }
      }

      invalid = rect();

      break;
    }

    _done_drawing = true;

    return true;

  CRITICAL_ERROR:
    this->on_graphics_critical_failure();
    return false;
  }

  bool window::render_element_on(element *el, gool::graphics *pg) {
    handle<element>       holder = el;
    handle<d2d::graphics> gfx    = static_cast<d2d::graphics *>(pg);
    html::view *          pv     = el->pview();
    if (pv != this) return false;

    auto_state<bool> _(is_painting, true);

    com::asset<ID2D1RenderTarget> rt = gfx->render_target();

    if (!rt) return false;

    gfx->begin_drawing();

    rt->BeginDraw();

    rt->SetTransform(D2D1::Matrix3x2F::Identity());

    if (el->ldata) {
      auto_state<gool::graphics *> _1(pv->drawing_surface, gfx);
      gfx->set_drawing_root(el);
      el->commit_measure(*pv);
      point vp = el->view_pos(*pv);
      point rp =
          vp - point(root()->ldata->borpad_left(), root()->ldata->borpad_top());
      rect rc(vp, root()->dim());
      rc >>=
          rect(root()->ldata->borpad_left(), root()->ldata->borpad_top(),
               root()->ldata->borpad_right(), root()->ldata->borpad_bottom());
      gfx->offset(-rp);
      el->draw(*pv, gfx, vp);
    }

#ifdef DEBUG_DRAW_PIPELINE
    D2D1_TAG t1 = 0;
    HRESULT  hr = rt->EndDraw(&t1);
#else
    HRESULT hr = rt->EndDraw();
#endif

    if (FAILED(hr)) {
#ifdef DEBUG_DRAW_PIPELINE
      if (t1) {
        element *t = (element *)t1;
        error_report("Drawing failure on DOM element:", t);
      }
#endif
      assert(false);
      if (doc()) doc()->drop_caches();
      return false;
    } else 
      gfx->end_drawing();
    return true;
  }

  bool window::print(void *hdc, rect paint_rc) {
    handle<dc_graphics> gfx = new dc_graphics(HDC(hdc), paint_rc, true);
    return render_on(gfx, paint_rc);
  }

  bool window::render_on(gool::graphics *pg, rect paint_rc) {
    if (!doc()) return false;

    d2d::graphics *gfx = static_cast<d2d::graphics *>(pg);

    auto_state<bool>             _1(is_painting, true);
    auto_state<handle<graphics>> _2(_gfx, gfx);

    // if(doc())
    //  doc()->reset_styles(*this);

    // to_update.update(this);
    // doc()->commit_measure(*this);
    commit_update();

    com::asset<ID2D1RenderTarget> rt = _gfx->render_target();

    if (!rt) return false;

    rt->BeginDraw();

    rt->SetTransform(D2D1::Matrix3x2F::Identity());

    uint level = gfx->push_state();

    if (paint_rc.empty()) {
      gfx->set_clip_rc(dimension());
      rt->Clear(D2D1::ColorF(0xFFFFFF, 0.0f));
      paint();
    } else {
      gfx->set_clip_rc(paint_rc);
      D2D1_RECT_F clip = g2f(paint_rc);
      rt->PushAxisAlignedClip(clip, D2D1_ANTIALIAS_MODE_ALIASED);
      rt->Clear(D2D1::ColorF(0xFFFFFF, 0.0f));
      paint();
      rt->PopAxisAlignedClip();
    }

    gfx->restore_state(level);

#ifdef DEBUG_DRAW_PIPELINE
    D2D1_TAG t1 = 0;
    HRESULT  hr = rt->EndDraw(&t1);
#else
    HRESULT hr = rt->EndDraw();
#endif

    if (FAILED(hr)) {
#ifdef DEBUG_DRAW_PIPELINE
      if (t1) {
        element *t = (element *)t1;
        error_report("Drawing failure on DOM element:", t);
      }
#endif
      assert(false);
      if (doc()) doc()->drop_caches();
      return false;
    }
    return true;
  }

  /*gool::text_layout *window::create_text_layout(tool::wchars             text,
                                                const gool::text_format &tf) {
    return new d2d::text_layout(*this, text, tf, app);
  }

  gool::text_layout *application::create_text_layout(tool::wchars             text,
                                  const gool::text_format &tf) {
    return new d2d::text_layout(resolution_provider::desktop(), text, tf, this);
  }*/

  graphics *popup::surface() {
    assert(get_hwnd());
    if (!_gfx) {
//      if(is_layered())
//      _gfx = layered_window_graphics_hdc::create(this, WARP_GRAPHICS);
      if (environment::get_os_version() >= environment::WIN_10)
        _gfx = d2d::window_graphics::create(this, false, app->graphics_caps());
      else if (environment::get_os_version() > environment::WIN_7)
        _gfx = d2d::window_graphics::create( this, false, is_layered() ? WARP_GRAPHICS : app->graphics_caps());
      else
        _gfx = d2d::window_graphics::create(this, false, WARP_GRAPHICS);
    }
    // NOTE: WARP is slow on W10: _gfx =
    // d2d::window_graphics::create(this,false, WARP_GRAPHICS );
    return _gfx;
  }

  /*void window::render(html::element* el)
  {
    if(!surface())
      return;

    com::asset<ID2D1RenderTarget> rt = _gfx->render_target();
    

    //HRESULT hr;
    //hr = CreateDeviceResources();
    if( !rt )
      return;

    rt->BeginDraw();
    rt->Clear(D2D1::ColorF(0xFFFFFF,0.0f));

    //if( _is_child )

    //size dim = dimension();

    //D2D1_POINT_2F center = { float(dim.x)/2, float(dim.y)/2 };
    //
    //rt->SetTransform(
    //                D2D1::Matrix3x2F::Scale(_zoom.x,_zoom.y,center)
    //              * D2D1::Matrix3x2F::Rotation(_angle,center)
    //                );
    paint(point(0,0));

    //rt->SetTransform(D2D1::IdentityMatrix());
    //test_gfx_paint(point(0,0));
    

    HRESULT hr = rt->EndDraw();

    if (FAILED(hr))
    {
      assert(false);
       _gfx = 0;
       if( doc() )
         doc()->drop_caches();
    }
  }*/

  bool popup::print(void *hdc, rect paint_rc) {
    paint_rc = rect(client_dim());

    handle<dc_graphics> pgx = new dc_graphics(HDC(hdc), paint_rc, true);
    if (!_root) return false;
    html::view *pv = _root->pview();

    pgx->begin_drawing();

    bool r = false;

    if (pv && _root->ldata) {
      critical_section                  _1(pv->guard);
      auto_state<bool>                  _2(is_painting, true);
      auto_state<handle<d2d::graphics>> _3(_gfx, pgx.ptr());
      assert(_root->state.popup());
      _root->commit_measure(*pv);
      point vp   = _root->view_pos(*pv);
      point poff = pgx->offset(
          vp - point(_root->ldata->borpad_left(), _root->ldata->borpad_top()));
      _root->draw(*pv, pgx, vp);
      pgx->offset(poff);
      r = true;
    } else 
      pgx->end_drawing();

    return r;
  }

  bool popup::render_layered(void *dc, rect client_rc) {
    super::render_layered(dc,client_rc);

    if (!_root) return false;

    size client_dim = this->client_dim();
    if (client_dim.empty()) return false;

    handle<d2d::graphics> proto_gfx = surface();
    if (!proto_gfx)
      return false;

    layered_window_ctx              lwctx(client_dim);
    handle<d2d::offscreen_graphics> gfx = 
        new d2d::offscreen_graphics(proto_gfx, client_dim);
    HWND        hwnd = get_hwnd();
    html::view *pv   = root()->pview();

    auto_state<bool> _(is_painting, true);

    auto updater = [&lwctx, hwnd](HDC hdc) { lwctx.update(hwnd, hdc); };

    if (pv && root()->ldata) {
      auto_state<gool::graphics *> _1(pv->drawing_surface, gfx);
      gfx->set_drawing_root(root());
      root()->commit_measure(*pv);
      point vp = root()->view_pos(*pv);
      auto outline_margins = _root->outline_distance(*pv);
      point rp = vp - outline_margins.s;
      rect rc(vp, root()->dim());
      rc >>= outline_margins;
      gfx->begin();
      gfx->offset(-rp);
      root()->draw(*pv, gfx, vp);
      gfx->end(updater);
    }
    invalid = rect();

    return true;
  }

  bool popup::render(void *dc, rect client_rc) {
    //if (is_layered())
    //  return render_layered();
    //else
    return render_area(dc,client_rc);
  }

  bool popup::render_area(void *dc, rect client_rc) {
    handle<html::element> root = this->root();
    handle<html::view>    pv   = root->pview();

    if (root->airborn) {
      size prsize = _root->airborn->dim;
      if (_root->airborn->dim.x.is_defined() && prsize != _root->dim()) {
        _root->set_width(*pv, prsize.x);
        _root->set_height(*pv, prsize.y);
      }
    }

    graphics *gfx = surface();
    if (!gfx) return false;

    auto_state<bool> _(is_painting, true);

    gfx->resize(window_dim());
    com::asset<ID2D1RenderTarget> rt = gfx->render_target();

    if (!rt) return false;

    // assert(_root->state.popup());
    _gfx->begin_drawing();

    rt->BeginDraw();
    rt->Clear(D2D1::ColorF(0xFFFFFF, 0.0f));
    //rt->Clear(D2D1::ColorF(0xFF0000, 1.0f));

    if (pv && _root->ldata /*&& _root->state.popup()*/) {
      auto_state<gool::graphics *> _1(pv->drawing_surface, _gfx);
      _gfx->set_drawing_root(_root);
      // ??? root->commit_measure(*pv); - may lead to stack overflow

      assert(root->ldata->dim_min.x.is_defined() &&
             root->ldata->dim_min.y.is_defined());
      if (root->ldata->dim.x > 0 && root->ldata->dim.y > 0) {
        point vp = root->view_pos(*pv);

        rect outline = root->outline_distance(*pv);

        point rp = vp - outline.s; //point(root->ldata->borpad_left(), root->ldata->borpad_top());

        rect rc(vp, root->dim());
        //rc >>= rect(root->ldata->borpad_left(), root->ldata->borpad_top(),
        //            root->ldata->borpad_right(), root->ldata->borpad_bottom());
        rc >>= outline;

        _gfx->offset(-rp);
        root->draw(*pv, _gfx, vp);
      }
    }

#ifdef _DEBUG
    D2D1_TAG t1 = 0;
    HRESULT  hr = rt->EndDraw(&t1);
#else
    HRESULT hr = rt->EndDraw();
#endif

    if (FAILED(hr)) {
#ifdef _DEBUG
      if (t1) {
        element *t = (element *)t1;
        t->dbg_report("DRAWING FAILURE on ");
        if (t->parent) t->parent->dbg_report("\tof parent ");
      }
#endif
      assert(false);
      _gfx = 0;
    } else 
      _gfx->end_drawing();
    invalid = rect();
    return true;
  }

  mswin::window* application::create_window_processor(const window_params &params) {
    return new d2d::window(params);
  }

  mswin::window* application::create_dx_window_processor(const window_params &params, IDXGISwapChain *pSwapChain) {
    return new d2d::dx_window(params,pSwapChain);
  }

  void dx_window::on_size(size sz) {
    mswin::window::on_size(sz);
    _gfx = nullptr; // free all resources
    // WRONG: _gfx = dx_window_graphics::create(this,_swap_chain);
  }

  gool::graphics *dx_window::surface() {
    if (!_gfx) _gfx = dx_window_graphics::create(this, _swap_chain);
    return _gfx;
  }



} // namespace d2d
