#include "d2d.h"
#include "html/html.h"
#include "html/html-layout-text-flow.h"
#include "html/html-layout-text-analysis.h"

namespace d2d {

  using namespace tool;
  using namespace html;
  using namespace gool;
  using namespace html::tflow;

  inline float width_of(const DWRITE_GLYPH_RUN &dw_glyph_run) {
    slice<float> glyph_widths(dw_glyph_run.glyphAdvances,
                              dw_glyph_run.glyphCount);
    return glyph_widths.accumulate(0.0f);
  }

  inline void
  convert(const glyph_run &          gr,
          const uint16 *             glyph_indices,  // [this->glyph_count]
          const float *              glyph_advances, // [this->glyph_count]
          const DWRITE_GLYPH_OFFSET *glyph_offsets,  // [this->glyph_count]
          OUT DWRITE_GLYPH_RUN *d2_glyph_run) {
    // Populate the DWrite glyph run.

    d2_glyph_run->glyphIndices  = &glyph_indices[gr.glyph_start];
    d2_glyph_run->glyphAdvances = &glyph_advances[gr.glyph_start];
    d2_glyph_run->glyphOffsets  = &glyph_offsets[gr.glyph_start];
    d2_glyph_run->glyphCount    = gr.glyph_count;
    d2_glyph_run->fontEmSize    = gr.font->size;
    d2_glyph_run->fontFace      = gr.font.ptr_of<d2d::font>()->dw_font_face();
    d2_glyph_run->bidiLevel     = gr.bidi_level;
    d2_glyph_run->isSideways    = gr.is_sideways;
  }

  void application::draw_glyph_run(view *pv, gool::graphics *gfx,
                                const html::tflow::text_flow &tf,
                                const html::tflow::glyph_run &gr,
                                gool::pointf at, argb color,
                                const style *run_style) {

    D2D1_POINT_2F dst = d2d::g2f(at);

    ID2D1RenderTarget *render_target =
        static_cast<d2d::graphics *>(gfx)->render_target();

    com::asset<ID2D1SolidColorBrush> fore;
    static_cast<d2d::graphics *>(gfx)->get_solid_brush(color, fore);


    DWRITE_GLYPH_RUN dw_glyph_run;
    // Massage the custom glyph run to something directly
    // digestable by DrawGlyphRun.
    convert(gr, tf._glyph_indices.head(),
            //&_glyph_advances[0],
            tf._glyph_justified_advances.head(),
            (DWRITE_GLYPH_OFFSET *)tf._glyph_offsets.head(), &dw_glyph_run);

    DWRITE_MEASURING_MODE dw_measuring_mode =
        run_style->font_rendering_mode ? DWRITE_MEASURING_MODE_GDI_CLASSIC
                                       : DWRITE_MEASURING_MODE_NATURAL;

    if (run_style->text_shadow.is_defined()) {
      element *el = gr.node->get_element();
      for (shadow_def *it = run_style->text_shadow; it; it = it->next) {
        gool::argb    c = it->color.val(run_style->font_color).to_argb();
        D2D1_POINT_2F dst_shadow = dst;
        dst_shadow.x += pixels(*pv, el, it->offset_x).width_f();
        dst_shadow.y += pixels(*pv, el, it->offset_y).height_f();
        float radius = pixels(*pv, el, it->radius).width_f();
        float spread = pixels(*pv, el, it->spread).width_f();

        d2d::draw_glyph_run_shadow(static_cast<d2d::graphics *>(gfx),
                                   &dw_glyph_run, dw_measuring_mode, dst_shadow,
                                   c, radius, spread);
      }
    }

    render_target->DrawGlyphRun(dst, &dw_glyph_run, fore, dw_measuring_mode);
  }

#if defined(USE_D2D_PLUS)
  void application_plus::draw_glyph_run(view *pv, gool::graphics *gfx,
    const html::tflow::text_flow &tf,
    const html::tflow::glyph_run &gr,
    gool::pointf at, argb color,
    const style *run_style) 
  {
    ID2D1DeviceContext4 *pdc4 = static_cast<d2d::graphics *>(gfx)->device_context_4();
    if (!pdc4)
      return super::draw_glyph_run(pv, gfx, tf, gr, at, color, run_style);

    D2D1_POINT_2F dst = d2d::g2f(at);

    com::asset<ID2D1SolidColorBrush> fore;
    static_cast<d2d::graphics *>(gfx)->get_solid_brush(color, fore);
    
    DWRITE_GLYPH_RUN dw_glyph_run;
    // Massage the custom glyph run to something directly
    // digestable by DrawGlyphRun.
    convert(gr, tf._glyph_indices.head(),
      //&_glyph_advances[0],
      tf._glyph_justified_advances.head(),
      (DWRITE_GLYPH_OFFSET *)tf._glyph_offsets.head(), &dw_glyph_run);

    DWRITE_MEASURING_MODE dw_measuring_mode =
      run_style->font_rendering_mode ? DWRITE_MEASURING_MODE_GDI_CLASSIC
      : DWRITE_MEASURING_MODE_NATURAL;

    if (run_style->text_shadow.is_defined()) {
      element *el = gr.node->get_element();
      for (shadow_def *it = run_style->text_shadow; it; it = it->next) {
        gool::argb    c = it->color.val(run_style->font_color).to_argb();
        D2D1_POINT_2F dst_shadow = dst;
        dst_shadow.x += pixels(*pv, el, it->offset_x).width_f();
        dst_shadow.y += pixels(*pv, el, it->offset_y).height_f();
        float radius = pixels(*pv, el, it->radius).width_f();
        float spread = pixels(*pv, el, it->spread).width_f();

        d2d::draw_glyph_run_shadow(static_cast<d2d::graphics *>(gfx),
          &dw_glyph_run, dw_measuring_mode, dst_shadow,
          c, radius, spread);
      }
    }

    DWRITE_GLYPH_IMAGE_FORMATS requested_formats = 
        DWRITE_GLYPH_IMAGE_FORMATS_TRUETYPE |
        DWRITE_GLYPH_IMAGE_FORMATS_CFF |
        DWRITE_GLYPH_IMAGE_FORMATS_COLR |
        DWRITE_GLYPH_IMAGE_FORMATS_SVG |
        DWRITE_GLYPH_IMAGE_FORMATS_PNG |
        DWRITE_GLYPH_IMAGE_FORMATS_JPEG |
        DWRITE_GLYPH_IMAGE_FORMATS_TIFF |
        DWRITE_GLYPH_IMAGE_FORMATS_PREMULTIPLIED_B8G8R8A8;

    com::asset<IDWriteColorGlyphRunEnumerator1> glyph_run_enumerator;

    HRESULT hr = this->_dw_factory4->TranslateColorGlyphRun(
      dst,
      &dw_glyph_run,
      nullptr,
      requested_formats, // the glyph formats supported by this renderer 
      DWRITE_MEASURING_MODE_NATURAL,
      nullptr,
      0,
      glyph_run_enumerator.target() // on return, may contain color glyph runs 
    );

    if (hr == DWRITE_E_NOCOLOR) {
      pdc4->DrawGlyphRun(dst, &dw_glyph_run, fore, dw_measuring_mode);
    }
    else if(SUCCEEDED(hr)) // color glyph

      // Complex case: the run has one or more color runs within it. Iterate
      // over the sub-runs and draw them, depending on their format.

      for (;;)
      {
        BOOL haveRun = FALSE;
        hr = glyph_run_enumerator->MoveNext(&haveRun);
        if (FAILED(hr))
          break;
        if (!haveRun)
          break;

        DWRITE_COLOR_GLYPH_RUN1 const* color_run;

        hr = glyph_run_enumerator->GetCurrentRun(&color_run);

        switch (color_run->glyphImageFormat)
        {
          case DWRITE_GLYPH_IMAGE_FORMATS_NONE: break;
          case DWRITE_GLYPH_IMAGE_FORMATS_PNG:
          case DWRITE_GLYPH_IMAGE_FORMATS_JPEG:
          case DWRITE_GLYPH_IMAGE_FORMATS_TIFF:
          case DWRITE_GLYPH_IMAGE_FORMATS_PREMULTIPLIED_B8G8R8A8:
            // This run is bitmap glyphs. Use Direct2D to draw them.
            pdc4->DrawColorBitmapGlyphRun( color_run->glyphImageFormat, dst, &color_run->glyphRun, dw_measuring_mode );
            break;

          case DWRITE_GLYPH_IMAGE_FORMATS_SVG:
            // This run is SVG glyphs. Use Direct2D to draw them.
            pdc4->DrawSvgGlyphRun(dst, &dw_glyph_run, nullptr, nullptr, 0, dw_measuring_mode);
          break;

          case DWRITE_GLYPH_IMAGE_FORMATS_TRUETYPE:
          case DWRITE_GLYPH_IMAGE_FORMATS_CFF:
          case DWRITE_GLYPH_IMAGE_FORMATS_COLR:
          default:
          {
            // This run is solid-color outlines, either from non-color
            // glyphs or from COLR glyph layers. Use Direct2D to draw them.
            com::asset<ID2D1SolidColorBrush> layer_brush;
            pdc4->CreateSolidColorBrush(color_run->runColor, layer_brush.target());
            // Draw the run with the selected color.
            pdc4->DrawGlyphRun(dst,&color_run->glyphRun,color_run->glyphRunDescription,layer_brush,dw_measuring_mode);
          } break;
        }
      } // color glyph
  }
#endif

#if !defined(WINDOWLESS)
  void window::draw_glyph_run(gool::graphics *              gfx,
                              const html::tflow::text_flow &tf,
                              const html::tflow::glyph_run &gr, gool::pointf at,
                              argb color, const style *run_style) {
    app.ptr_of<d2d::application>()->draw_glyph_run(this, gfx, tf, gr, at, color, run_style);
  }

  void print_view::draw_glyph_run(gool::graphics *              gfx,
                                  const html::tflow::text_flow &tf,
                                  const html::tflow::glyph_run &gr,
                                  gool::pointf at, argb color,
                                  const html::style *run_style) {
    if (this->drawing_content) {
      auto ln = tf.get_line_ref(gr.line_no);

      if (ln.page_no && this->page_no != ln.page_no) return;

      range y;
      y.s = int(at.y - ln.baseline);
      y.e = y.s + ln.height;
      if (!page_y.covers(y)) return;
      ln.page_no = uint16(this->page_no);
    }
    app.ptr_of<d2d::application>()->draw_glyph_run(this, gfx, tf, gr, at, color, run_style);
  }
#endif

} // namespace d2d