#include "d2d.h"
#include "d2d-font-loader.h"

namespace d2d {

  HRESULT STDMETHODCALLTYPE
          ResourceFontCollectionLoader::QueryInterface(REFIID iid, void **ppvObject) {
    if (iid == IID_IUnknown || iid == __uuidof(IDWriteFontCollectionLoader)) {
      *ppvObject = this;
      AddRef();
      return S_OK;
    } else {
      *ppvObject = NULL;
      return E_NOINTERFACE;
    }
  }

  ULONG STDMETHODCALLTYPE ResourceFontCollectionLoader::AddRef() {
    return InterlockedIncrement(&refCount_);
  }

  ULONG STDMETHODCALLTYPE ResourceFontCollectionLoader::Release() {
    ULONG newCount = InterlockedDecrement(&refCount_);
    if (newCount == 0) delete this;

    return newCount;
  }

  HRESULT STDMETHODCALLTYPE
          ResourceFontCollectionLoader::CreateEnumeratorFromKey(
      IDWriteFactory *factory,
      void const *    collectionKey, // [collectionKeySize] in bytes
      UINT32          collectionKeySize,
      OUT IDWriteFontFileEnumerator **fontFileEnumerator) {
    *fontFileEnumerator = NULL;

    HRESULT hr = S_OK;

    // if (collectionKeySize % sizeof(UINT) != 0)
    //    return E_INVALIDARG;

    d2d::asset<ResourceFontFileEnumerator> enumerator =
        new (std::nothrow) ResourceFontFileEnumerator(ctx);
    if (!enumerator) return E_OUTOFMEMORY;

    tool::chars name;
    name.start  = (char *)collectionKey;
    name.length = collectionKeySize;
    // UINT const* resourceIDs    = static_cast<UINT const*>(collectionKey);
    // UINT32 const resourceCount = collectionKeySize / sizeof(UINT);

    hr = enumerator->Initialize(name);

    if (FAILED(hr)) { return hr; }

    *fontFileEnumerator = enumerator.acquire();

    return hr;
  }

  ResourceFontContext::ResourceFontContext() : hr_(S_FALSE), factory(0) {}

  ResourceFontContext::~ResourceFontContext() { Shutdown(); }

  HRESULT ResourceFontContext::Initialize(IDWriteFactory *pf) {
    if (hr_ == S_FALSE) { hr_ = InitializeInternal(pf); }
    return hr_;
  }

  void ResourceFontContext::Shutdown() {
    if (factory) {

      custom_fonts_data.clear();

      factory->UnregisterFontCollectionLoader(collection_loader);
      factory->UnregisterFontFileLoader(file_loader);

      collection_loader = 0;
      file_loader       = 0;

      factory = 0;
    }
  }

  HRESULT ResourceFontContext::InitializeInternal(IDWriteFactory *pf) {
    HRESULT hr = S_OK;

    factory           = pf;
    collection_loader = new ResourceFontCollectionLoader(this);
    file_loader       = new ResourceFontFileLoader(this);

    if (!collection_loader || !file_loader) { return E_FAIL; }

    // Register our custom loaders with the factory object.
    //
    // Note: For this application we just use the shared DWrite factory object
    // which is accessed via
    //       a global variable. If we were using fonts embedded in *documents*
    //       then it might make sense to create an isolated factory for each
    //       document. When unloading the document, one would also release the
    //       isolated factory, thus ensuring that all cached font data specific
    //       to that document would be promptly disposed of.
    //
    if (FAILED(hr = factory->RegisterFontFileLoader(file_loader))) return hr;

    hr = factory->RegisterFontCollectionLoader(collection_loader);

    return hr;
  }

  HRESULT ResourceFontContext::CreateFontCollection(
      uint id, OUT IDWriteFontCollection **result) {
    *result = NULL;

    assert(factory);
    HRESULT hr = S_OK;

    hr = factory->CreateCustomFontCollection(collection_loader, &id,
                                             UINT32(sizeof(id)), result);

    return hr;
  }

  ResourceFontFileEnumerator::ResourceFontFileEnumerator(ResourceFontContext *c)
      : refCount_(0), ctx(c), currentFile() {}

  HRESULT ResourceFontFileEnumerator::Initialize(tool::chars nm) {
    name = nm;
    return S_OK;
  }

  HRESULT STDMETHODCALLTYPE
          ResourceFontFileEnumerator::QueryInterface(REFIID iid, OUT void **ppvObject) {
    if (iid == IID_IUnknown || iid == __uuidof(IDWriteFontFileEnumerator)) {
      *ppvObject = this;
      AddRef();
      return S_OK;
    } else {
      *ppvObject = NULL;
      return E_NOINTERFACE;
    }
  }

  ULONG STDMETHODCALLTYPE ResourceFontFileEnumerator::AddRef() {
    return InterlockedIncrement(&refCount_);
  }

  ULONG STDMETHODCALLTYPE ResourceFontFileEnumerator::Release() {
    ULONG newCount = InterlockedDecrement(&refCount_);
    if (newCount == 0) delete this;

    return newCount;
  }

  HRESULT STDMETHODCALLTYPE
          ResourceFontFileEnumerator::MoveNext(OUT BOOL *hasCurrentFile) {
    HRESULT hr = S_OK;

    *hasCurrentFile = FALSE;
    currentFile.release();

    if (name.length) {
      hr = ctx->factory->CreateCustomFontFileReference(
          name.start, UINT32(name.length), ctx->file_loader,
          currentFile.target());
      name.length = 0;
      if (SUCCEEDED(hr)) { *hasCurrentFile = TRUE; }
    }
    return hr;
  }

  HRESULT STDMETHODCALLTYPE ResourceFontFileEnumerator::GetCurrentFontFile(
      OUT IDWriteFontFile **fontFile) {
    *fontFile = currentFile.acquire();
    return (!currentFile) ? E_FAIL : S_OK;
  }

  // QueryInterface
  HRESULT STDMETHODCALLTYPE
          ResourceFontFileLoader::QueryInterface(REFIID iid, void **ppvObject) {
    if (iid == IID_IUnknown || iid == __uuidof(IDWriteFontFileLoader)) {
      *ppvObject = this;
      AddRef();
      return S_OK;
    } else {
      *ppvObject = NULL;
      return E_NOINTERFACE;
    }
  }

  // AddRef
  ULONG STDMETHODCALLTYPE ResourceFontFileLoader::AddRef() {
    return InterlockedIncrement(&refCount_);
  }

  // Release
  ULONG STDMETHODCALLTYPE ResourceFontFileLoader::Release() {
    ULONG newCount = InterlockedDecrement(&refCount_);
    if (newCount == 0) delete this;

    return newCount;
  }

  // CreateStreamFromKey
  //
  //      Creates an IDWriteFontFileStream from a key that identifies the file.
  //      The format and meaning of the key is entirely up to the loader
  //      implementation. The only requirements imposed by DWrite are that a key
  //      must remain valid for as long as the loader is registered. The same
  //      key must also uniquely identify the same file, so for example you must
  //      not recycle keys so that a key might represent different files at
  //      different times.
  //
  //      In this case the key is a UINT which identifies a font resources.
  //
  HRESULT STDMETHODCALLTYPE ResourceFontFileLoader::CreateStreamFromKey(
      void const *fontFileReferenceKey, // [fontFileReferenceKeySize] in bytes
      UINT32      fontFileReferenceKeySize,
      OUT IDWriteFontFileStream **fontFileStream) {
    *fontFileStream = NULL;

    // Make sure the key is the right size.
    if (fontFileReferenceKeySize != sizeof(UINT)) return E_INVALIDARG;

    uint id = *(uint *)fontFileReferenceKey;
    // tool::chars name;
    // name.start = (const char*)fontFileReferenceKey;
    // name.length = fontFileReferenceKeySize;

    // tool::handle<custom_font> data;
    // if(!ctx->collection.find(name,data))
    //  return E_FAIL;

    // Create the stream object.
    d2d::asset<ResourceFontFileStream> stream = new (std::nothrow)
        ResourceFontFileStream(ctx->custom_fonts_data[id]->data);
    if (!stream) return E_OUTOFMEMORY;

    if (!stream->IsInitialized()) { return E_FAIL; }

    *fontFileStream = stream.acquire();

    return S_OK;
  }

  ResourceFontFileStream::ResourceFontFileStream(tool::array<byte> &data_in)
      : refCount_(0), data(data_in) {}

  // IUnknown methods
  HRESULT STDMETHODCALLTYPE
          ResourceFontFileStream::QueryInterface(REFIID iid, void **ppvObject) {
    if (iid == IID_IUnknown || iid == __uuidof(IDWriteFontFileStream)) {
      *ppvObject = this;
      AddRef();
      return S_OK;
    } else {
      *ppvObject = NULL;
      return E_NOINTERFACE;
    }
  }

  ULONG STDMETHODCALLTYPE ResourceFontFileStream::AddRef() {
    return InterlockedIncrement(&refCount_);
  }

  ULONG STDMETHODCALLTYPE ResourceFontFileStream::Release() {
    ULONG newCount = InterlockedDecrement(&refCount_);
    if (newCount == 0) delete this;

    return newCount;
  }

  // IDWriteFontFileStream methods
  HRESULT STDMETHODCALLTYPE ResourceFontFileStream::ReadFileFragment(
      void const **fragmentStart, // [fragmentSize] in bytes
      UINT64 fileOffset, UINT64 fragmentSize, OUT void **fragmentContext) {
    // The loader is responsible for doing a bounds check.
    if (fileOffset <= data.length() &&
        fragmentSize <= data.length() - fileOffset) {
      *fragmentStart   = data.cbegin() + fileOffset;
      *fragmentContext = NULL;
      return S_OK;
    } else {
      *fragmentStart   = NULL;
      *fragmentContext = NULL;
      return E_FAIL;
    }
  }

  void STDMETHODCALLTYPE
       ResourceFontFileStream::ReleaseFileFragment(void *fragmentContext) {}

  HRESULT STDMETHODCALLTYPE
          ResourceFontFileStream::GetFileSize(OUT UINT64 *fileSize) {
    *fileSize = data.length();
    return S_OK;
  }

  HRESULT STDMETHODCALLTYPE
          ResourceFontFileStream::GetLastWriteTime(OUT UINT64 *lastWriteTime) {
    // The concept of last write time does not apply to this loader.
    *lastWriteTime = 0;
    return E_NOTIMPL;
  }

  // d2d::ResourceFontContext

  bool application::construct_font_collection_for(memory_font_variant &fv) {
    critical_section _(guard);

    HRESULT hr = font_loading_context.Initialize(dw_factory());
    if (FAILED(hr)) return false;

    uint uid = (uint)font_loading_context.custom_fonts_data.length();
    handle<d2d::custom_font_data> cf = new d2d::custom_font_data();
    cf->data                         = fv.data;
    font_loading_context.custom_fonts_data.push(cf);

    d2d::asset<IDWriteFontCollection> collection;

    hr = font_loading_context.CreateFontCollection(uid, collection.target());
    if (FAILED(hr)) return false;

    fv.coll = collection;
    return true;
  }

  bool application::install_font(const tool::ustring &name, int weight,
                                 bool italic, tool::bytes data) {
    return install_memory_font(name, weight, italic, data);
  }

} // namespace d2d
