#include "d2d.h"

#if !defined(WINDOWLESS) && defined(USE_DIRECT_COMP)

#include "d2d-bb-window-graphics.h"

namespace d2d {

  namespace Dwm {
    DwmpCreateSharedThumbnailVisual CreateSharedThumbnailVisual = nullptr;
    DwmpCreateSharedMultiWindowVisual CreateSharedMultiWindowVisual = nullptr;
    DwmpUpdateSharedMultiWindowVisual UpdateSharedMultiWindowVisual = nullptr;
    DwmpCreateSharedVirtualDesktopVisual CreateSharedVirtualDesktopVisual = nullptr;
    DwmpUpdateSharedVirtualDesktopVisual UpdateSharedVirtualDesktopVisual = nullptr;
    SetWindowCompositionAttributeP SetWindowCompositionAttribute = nullptr;
    RtlGetVersionPtr GetVersionInfo = nullptr;

    bool init() {
      if (CreateSharedMultiWindowVisual) 
        return true;

      auto dwmapi = LoadLibrary(L"dwmapi.dll");
      auto user32 = LoadLibrary(L"user32.dll");
      auto ntdll = GetModuleHandle(L"ntdll.dll");
      
      if (!dwmapi || !user32 || !ntdll)
      {
        return false;
      }
      GetVersionInfo = (RtlGetVersionPtr)GetProcAddress(ntdll, "RtlGetVersion");
      SetWindowCompositionAttribute = (SetWindowCompositionAttributeP)GetProcAddress(user32, "SetWindowCompositionAttribute");
      CreateSharedThumbnailVisual = (DwmpCreateSharedThumbnailVisual)GetProcAddress(dwmapi, MAKEINTRESOURCEA(147));
      CreateSharedMultiWindowVisual = (DwmpCreateSharedMultiWindowVisual)GetProcAddress(dwmapi, MAKEINTRESOURCEA(163));
      UpdateSharedMultiWindowVisual = (DwmpUpdateSharedMultiWindowVisual)GetProcAddress(dwmapi, MAKEINTRESOURCEA(164));
      CreateSharedVirtualDesktopVisual = (DwmpCreateSharedVirtualDesktopVisual)GetProcAddress(dwmapi, MAKEINTRESOURCEA(163));
      UpdateSharedVirtualDesktopVisual = (DwmpUpdateSharedVirtualDesktopVisual)GetProcAddress(dwmapi, MAKEINTRESOURCEA(164));

      FreeLibrary(dwmapi);
      FreeLibrary(user32);

      return true;
    }

  }

  window_bb_graphics *window_bb_graphics::create(html::iwindow *pw, bool is_transparent, GRAPHICS_CAPS caps, bool for_layered) {
    if (!is_feasible()) return nullptr;

    auto_ptr<window_bb_graphics> pg = auto_ptr<window_bb_graphics>(new window_bb_graphics(pw, is_transparent, caps, for_layered));

    if (Dwm::init() && pg->create_device()) {
      pg->_graphics_caps = d2d_app()->requested_gfx_layer;
      return pg.release();
    }

    return nullptr;
  }

  window_bb_graphics::window_bb_graphics(html::iwindow *pw, bool transparent, GRAPHICS_CAPS caps, bool for_layered)
    : window_graphics_dxgi(pw, transparent, caps, for_layered) {
    dcomp_device = d2d_app()->dcomp_device();
  }

  bool window_bb_graphics::create_device_plain() {

    try {
      application *papp = d2d_app();

      com::asset<ID3D11Device>         device = papp->device_3d();
      if (!device) return false;

      com::asset<ID3D11Device1>        device1;
      com::asset<ID3D11DeviceContext1> context1;

      HR(device->QueryInterface<ID3D11Device1>(device1.target()));

      com::asset<IDXGIDevice> dxgi_device;    HR(device1->QueryInterface<IDXGIDevice>(dxgi_device.target()));
      com::asset<ID2D1Device> d2d1_device;    HR(papp->d2_1_factory()->CreateDevice(dxgi_device, d2d1_device.target()));
      com::asset<ID2D1DeviceContext> dc;      HR(d2d1_device->CreateDeviceContext(D2D1_DEVICE_CONTEXT_OPTIONS_NONE, dc.target()));

      device_context(dc);

      com::asset<IDXGIAdapter>  dxgi_adapter; HR(dxgi_device->GetAdapter(dxgi_adapter.target()));
      com::asset<IDXGIFactory2> dxgi_factory; HR(dxgi_adapter->GetParent(__uuidof(dxgi_factory), reinterpret_cast<void **>(dxgi_factory.target())));

      size sz = _window->window_dim();

      DXGI_SWAP_CHAIN_DESC1 props = {};
      props.Width = sz.x;
      props.Height = sz.y;
      props.Format = DXGI_FORMAT_B8G8R8A8_UNORM;
      props.SampleDesc.Count = 1;
      props.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
      props.BufferCount = 2;
      props.SwapEffect = DXGI_SWAP_EFFECT_FLIP_SEQUENTIAL; // DirectComposition requirement
      props.AlphaMode = DXGI_ALPHA_MODE_PREMULTIPLIED;

      HR(dxgi_factory->CreateSwapChainForComposition(device1, &props, nullptr, _swap_chain.target()));

      if (!_swap_chain)
        return false; // this erroneous behavior is observed at least in D3DGear:
                      // it returns S_OK but _swap_chain is NULL - bastards.

      dxgi_factory->MakeWindowAssociation(_window->get_hwnd(), DXGI_MWA_NO_ALT_ENTER); // no ALT+ENTER -> full screen

      HR(dcomp_device->CreateTargetForHwnd(_window->get_hwnd(), true, composition_target.target()));

      HR(dcomp_device->CreateVisual(root_visual.target()));

      HR(composition_target->SetRoot(root_visual));

      root_visual->SetContent(_swap_chain);

      HR(dcomp_device->Commit());

      return true;
    }
    catch (com::exception const &) {
      return false;
    }

  }

  bool window_bb_graphics::create_device_blurbehind() {

    try {

      com::asset<IDCompositionDevice3> dcomp_device_3;
      HR(dcomp_device->QueryInterface<IDCompositionDevice3>(dcomp_device_3.target()));

      HR(dcomp_device_3->CreateRectangleClip(clip.target()));
      HR(dcomp_device_3->CreateTranslateTransform(translate_transform.target()));
            
      com::asset<IDCompositionGaussianBlurEffect> blur_effect;
      com::asset<IDCompositionSaturationEffect>   saturation_effect;

      HR(dcomp_device_3->CreateGaussianBlurEffect(blur_effect.target()));
      HR(dcomp_device_3->CreateSaturationEffect(saturation_effect.target()));
     
      {
        BOOL enable = TRUE;
        WINDOWCOMPOSITIONATTRIBDATA CompositionAttribute{};
        CompositionAttribute.dwAttrib = WCA_EXCLUDED_FROM_LIVEPREVIEW;
        CompositionAttribute.pvData = &enable;
        CompositionAttribute.cbData = sizeof(BOOL);
        Dwm::SetWindowCompositionAttribute(_window->get_hwnd(), &CompositionAttribute);
      }
      create_backdrop(BACKDROP_SOURCE_HOSTBACKDROP);
      create_composition_visual();
      create_fallback_visual();
      background_visual->SetContent(_swap_chain);
      fallback_visual->SetContent(_swap_chain);
      root_visual->RemoveAllVisuals();

      background_visual->AddVisual(desktop_window_visual, false, NULL);
      background_visual->AddVisual(toplevel_window_visual, true, desktop_window_visual);
      
      root_visual->AddVisual(background_visual, false, NULL);
      root_visual->AddVisual(fallback_visual, true, background_visual);
      
      background_visual->SetClip(clip);
      background_visual->SetTransform(translate_transform);

      saturation_effect->SetSaturation(BLURBEHIND_SATURATION);

      blur_effect->SetBorderMode(D2D1_BORDER_MODE_HARD);
      blur_effect->SetInput(0, saturation_effect, 0);
      blur_effect->SetStandardDeviation(BLURBEHIND_BLUR_DEVIATION);

      background_visual->SetEffect(blur_effect);
      
      dcomp_device->Commit();

      on_window_position_change();

      return true;
    }
    catch (com::exception const &) {
      return false;
    }

  }


  DWORD GetBuildNumber() {
    static RTL_OSVERSIONINFOW versionInfo = { 0 };
    if (!versionInfo.dwOSVersionInfoSize)
    {
      versionInfo.dwOSVersionInfoSize = sizeof(versionInfo);
      Dwm::GetVersionInfo(&versionInfo);
    }
    return versionInfo.dwBuildNumber;
  }

  void window_bb_graphics::on_window_position_change()
  {
    if (!clip) return;
    rectf rc = _window->screen_place();
    clip->SetLeft(rc.left());
    clip->SetRight(rc.right());
    clip->SetTop(rc.top());
    clip->SetBottom(rc.bottom());
    background_visual->SetClip(clip);
    rectf crc = _window->client_screen_place();
    translate_transform->SetOffsetX(-crc.left());
    translate_transform->SetOffsetY(-crc.top());
    background_visual->SetTransform(translate_transform);
    dcomp_device->Commit();
    _DwmFlush()();
  }

  void window_bb_graphics::on_window_active_state_change(bool active) {
    if (toplevel_window_thumbnail != NULL)
    {
      HWND desktopWindow = GetShellWindow();
      GetWindowRect(desktopWindow, &desktop_area);

      SIZE size = { desktop_area.right - desktop_area.left, desktop_area.bottom - desktop_area.top };

      if (GetBuildNumber() >= 20000)
      {
        Dwm::UpdateSharedMultiWindowVisual(toplevel_window_thumbnail, NULL, 0,
          window_exclusion_list.begin(), window_exclusion_list.size(), &desktop_area, &size, 1);
      }
      else
      {
         Dwm::UpdateSharedVirtualDesktopVisual(toplevel_window_thumbnail, NULL, 0,
          window_exclusion_list.begin(), window_exclusion_list.size(), &desktop_area, &size);
      }

      _DwmFlush()();
    }
  }

#define DWM_TNP_FREEZE            0x100000
#define DWM_TNP_ENABLE3D          0x4000000
#define DWM_TNP_DISABLE3D         0x8000000
#define DWM_TNP_FORCECVI          0x40000000
#define DWM_TNP_DISABLEFORCECVI   0x80000000

  window_bb_graphics::~window_bb_graphics() {
    //release_device();
    //dcomp_device->Commit();
  }

  bool window_bb_graphics::create_backdrop(BACKDROP_SOURCE source) {

    HWND    desktopWindow;
    HRESULT hr;
    HWND    hwnd = _window->get_hwnd();

    desktopWindow = GetShellWindow();
    GetWindowRect(desktopWindow, &desktop_area);

    SIZE size = { desktop_area.right - desktop_area.left, desktop_area.bottom - desktop_area.top };

    switch (source)
    {
    case BACKDROP_SOURCE_DESKTOP: {
        DWM_THUMBNAIL_PROPERTIES thumbnail;
        HTHUMBNAIL desktopThumbnail = NULL;

        thumbnail.dwFlags = DWM_TNP_SOURCECLIENTAREAONLY | DWM_TNP_VISIBLE | DWM_TNP_RECTDESTINATION | DWM_TNP_RECTSOURCE | DWM_TNP_OPACITY | DWM_TNP_ENABLE3D;
        thumbnail.opacity = 255;
        thumbnail.fVisible = TRUE;
        thumbnail.fSourceClientAreaOnly = FALSE;
        thumbnail.rcDestination = RECT{ 0, 0, size.cx, size.cy };
        thumbnail.rcSource = RECT{ 0, 0, size.cx, size.cy };
        if (Dwm::CreateSharedThumbnailVisual(_window->get_hwnd(),
          desktopWindow, 2, &thumbnail,
          dcomp_device, (void**)desktop_window_visual.target(), &desktopThumbnail) != S_OK)
        {
          return false;
        }
      }
      break;
    case BACKDROP_SOURCE_HOSTBACKDROP: 
      {
        if (GetBuildNumber() >= 20000)
        {
          hr = Dwm::CreateSharedMultiWindowVisual(hwnd, dcomp_device, (void**)toplevel_window_visual.target(), &toplevel_window_thumbnail);
        }
        else
        {
          hr = Dwm::CreateSharedVirtualDesktopVisual(hwnd, dcomp_device, (void**)toplevel_window_visual.target(), &toplevel_window_thumbnail);
        }

        if (hr != S_OK || !create_backdrop(BACKDROP_SOURCE_DESKTOP))
        {
          return false;
        }
        window_exclusion_list.clear();
        window_exclusion_list.push(_window->get_hwnd());

        if (GetBuildNumber() >= 20000)
        {
          hr = Dwm::UpdateSharedMultiWindowVisual(toplevel_window_thumbnail, NULL, 0,
            window_exclusion_list.begin(), window_exclusion_list.size(), &desktop_area, &size, 1);
        }
        else
        {
          hr = Dwm::UpdateSharedVirtualDesktopVisual(toplevel_window_thumbnail, NULL, 0,
            window_exclusion_list.begin(), window_exclusion_list.size(), &desktop_area, &size);
        }
      }

      if (hr != S_OK)
      {
        return false;
      }
      break;
    }
    return true;
  }

  bool window_bb_graphics::create_composition_visual() {
    try {
      HR(dcomp_device->CreateVisual(root_visual.target()));
      HR(dcomp_device->CreateVisual(background_visual.target()));
      HR(dcomp_device->CreateVisual(fallback_visual.target()));
      HR(dcomp_device->CreateTargetForHwnd(_window->get_hwnd(), FALSE, composition_target.target()));
      HR(composition_target->SetRoot(root_visual));
    } catch (com::exception const &) {
      return false;
    }
    return true;
  }

  bool window_bb_graphics::create_fallback_visual()
  {
    try {

      application *papp = d2d_app();

      dcomp_device = papp->dcomp_device();

      com::asset<ID3D11Device>         device = papp->device_3d();
      if (!device) return false;

      com::asset<ID3D11Device1>        device1;
      com::asset<ID3D11DeviceContext1> context1;

      HR(device->QueryInterface<ID3D11Device1>(device1.target()));

      com::asset<IDXGIDevice> dxgi_device;    HR(device1->QueryInterface<IDXGIDevice>(dxgi_device.target()));
      com::asset<ID2D1Device> d2d1_device;    HR(papp->d2_1_factory()->CreateDevice(dxgi_device, d2d1_device.target()));
      com::asset<ID2D1DeviceContext> dc;      HR(d2d1_device->CreateDeviceContext(D2D1_DEVICE_CONTEXT_OPTIONS_NONE, dc.target()));

      com::asset<IDXGIAdapter>  dxgi_adapter; HR(dxgi_device->GetAdapter(dxgi_adapter.target()));
      com::asset<IDXGIFactory2> dxgi_factory; HR(dxgi_adapter->GetParent(__uuidof(dxgi_factory), reinterpret_cast<void **>(dxgi_factory.target())));

      //size sz = _window->client_dim();

      DXGI_SWAP_CHAIN_DESC1 props = {};
      props.Width = desktop_area.right - desktop_area.left;
      props.Height = desktop_area.bottom - desktop_area.top;
      props.Format = DXGI_FORMAT_B8G8R8A8_UNORM;
      props.SampleDesc.Count = 1;
      props.BufferUsage = DXGI_USAGE_RENDER_TARGET_OUTPUT;
      props.BufferCount = 2;
      props.SwapEffect = DXGI_SWAP_EFFECT_FLIP_SEQUENTIAL; // DirectComposition requirement
      props.AlphaMode = DXGI_ALPHA_MODE_PREMULTIPLIED;

      HR(dxgi_factory->CreateSwapChainForComposition(device1, &props, nullptr, _swap_chain.target()));

      if (!_swap_chain)
        return false; // this erroneous behavior is observed at least in D3DGear:
                      // it returns S_OK but _swap_chain is NULL - bastards.

      dxgi_factory->MakeWindowAssociation(_window->get_hwnd(), DXGI_MWA_NO_ALT_ENTER); // no ALT+ENTER -> full screen

      HR(_swap_chain->GetBuffer(0, __uuidof(fallback_surface), reinterpret_cast<void**>(fallback_surface.target())));

      D2D1_BITMAP_PROPERTIES1 properties = {};

      properties.pixelFormat.alphaMode = D2D1_ALPHA_MODE_PREMULTIPLIED;
      properties.pixelFormat.format = DXGI_FORMAT_B8G8R8A8_UNORM;
      properties.bitmapOptions = D2D1_BITMAP_OPTIONS_TARGET | D2D1_BITMAP_OPTIONS_CANNOT_DRAW;
      HR(dc->CreateBitmapFromDxgiSurface(fallback_surface, properties, _bitmap.target()));

      dc->SetTarget(_bitmap);
      dc->SetDpi(96.0, 96.0); // we do DIPs by ourselves

      //device_context()->BeginDraw();
      //device_context()->Clear();
      //device_context()->CreateSolidColorBrush(tintColor, fallbackBrush.GetAddressOf());
      //device_context()->FillRectangle(desktop_area, fallbackBrush.Get());
      //device_context()->EndDraw();
      //if (_swap_chain->Present(1, 0) != S_OK)
      //{
      //  return false;
      //}

      device_context(dc);

      return true;
    } 
    catch (com::exception const &) {
      return false;
    }
  }


}
#endif
