#include "sdk-headers.h"

#ifdef SCITERJS
  #include "sdk.js/include/sciter-x-request.h"
#else
  #include "sdk/include/sciter-x-request.h"
#endif

#include "tool/tool.h"
#include "gool/gool.h"
#include "html/html.h"

// a.k.a AddRef()
REQUEST_RESULT SCAPI RequestUse(HREQUEST rq) {
  if (!rq) return REQUEST_BAD_PARAM;
  rq->add_ref();
  return REQUEST_OK;
}

// a.k.a Release()
REQUEST_RESULT SCAPI RequestUnUse(HREQUEST rq) {
  if (!rq) return REQUEST_BAD_PARAM;
  rq->release();
  return REQUEST_OK;
}

// get requested URL
REQUEST_RESULT SCAPI RequestUrl(HREQUEST rq, LPCSTR_RECEIVER *rcv,
                                LPVOID rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  rcv(rq->url.c_str(), UINT(rq->url.length()), rcv_param);
  return REQUEST_OK;
}

// get real, content URL (after possible redirection)
REQUEST_RESULT SCAPI RequestContentUrl(HREQUEST rq, LPCSTR_RECEIVER *rcv,
                                       LPVOID rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  rcv(rq->content_url.c_str(), UINT(rq->content_url.length()), rcv_param);
  return REQUEST_OK;
}

// get requested data type
REQUEST_RESULT SCAPI RequestGetRequestType(HREQUEST         rq,
                                           REQUEST_RQ_TYPE *pType) {
  if (!rq || !pType) return REQUEST_BAD_PARAM;
  *pType = (REQUEST_RQ_TYPE)rq->rq_type;
  return REQUEST_OK;
}

// get requested data type
REQUEST_RESULT SCAPI RequestGetRequestedDataType(HREQUEST            rq,
                                                 SciterResourceType *pData) {
  if (!rq || !pData) return REQUEST_BAD_PARAM;
  *pData = (SciterResourceType)rq->data_type;
  return REQUEST_OK;
}

// get received data type, string, mime type
REQUEST_RESULT SCAPI RequestGetReceivedDataType(HREQUEST         rq,
                                                LPCSTR_RECEIVER *rcv,
                                                LPVOID           rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  rcv(rq->data_content_type.c_str(), UINT(rq->data_content_type.length()),
      rcv_param);
  return REQUEST_OK;
}

// get number of request parameters passed
REQUEST_RESULT SCAPI RequestGetNumberOfParameters(HREQUEST rq, UINT *pNumber) {
  if (!rq || !pNumber) return REQUEST_BAD_PARAM;
  *pNumber = (UINT)rq->rq_params.length();
  return REQUEST_OK;
}

// get nth request parameter name
REQUEST_RESULT SCAPI RequestGetNthParameterName(HREQUEST rq, UINT n,
                                                LPCWSTR_RECEIVER *rcv,
                                                LPVOID            rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  if (n >= rq->rq_params.length()) return REQUEST_FAILURE;
  rcv(rq->rq_params[n].name.c_str(), UINT(rq->rq_params[n].name.length()),
      rcv_param);
  return REQUEST_OK;
}

// get nth request parameter value
REQUEST_RESULT SCAPI RequestGetNthParameterValue(HREQUEST rq, UINT n,
                                                 LPCWSTR_RECEIVER *rcv,
                                                 LPVOID            rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  if (n >= rq->rq_params.length()) return REQUEST_FAILURE;
  rcv(rq->rq_params[n].value.c_str(), UINT(rq->rq_params[n].value.length()),
      rcv_param);
  return REQUEST_OK;
}

// get request times , ended - started = milliseconds to get the requst
REQUEST_RESULT SCAPI RequestGetTimes(HREQUEST rq, UINT *pStarted,
                                     UINT *pEnded) {
  if (!rq || !pStarted || !pEnded) return REQUEST_BAD_PARAM;
  *pStarted = rq->started;
  *pEnded   = rq->ended;
  return REQUEST_OK;
}

// get number of request headers
REQUEST_RESULT SCAPI RequestGetNumberOfRqHeaders(HREQUEST rq, UINT *pNumber) {
  if (!rq || !pNumber) return REQUEST_BAD_PARAM;
  *pNumber = (UINT)rq->rq_headers.length();
  return REQUEST_OK;
}

// get nth request header name
REQUEST_RESULT SCAPI RequestGetNthRqHeaderName(HREQUEST rq, UINT n,
                                               LPCWSTR_RECEIVER *rcv,
                                               LPVOID            rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  if (n >= rq->rq_headers.length()) return REQUEST_FAILURE;
  ustring us = u8::cvt(rq->rq_headers.key(int(n)));
  rcv(us.c_str(), UINT(us.length()), rcv_param);
  return REQUEST_OK;
}

// get nth request header value
REQUEST_RESULT SCAPI RequestGetNthRqHeaderValue(HREQUEST rq, UINT n,
                                                LPCWSTR_RECEIVER *rcv,
                                                LPVOID            rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  if (n >= rq->rq_headers.length()) return REQUEST_FAILURE;
  ustring us = u8::cvt(rq->rq_headers.value(int(n)));
  rcv(us.c_str(), UINT(us.length()), rcv_param);
  return REQUEST_OK;
}

// get number of response headers
REQUEST_RESULT SCAPI RequestGetNumberOfRspHeaders(HREQUEST rq, UINT *pNumber) {
  if (!rq || !pNumber) return REQUEST_BAD_PARAM;
  *pNumber = (UINT)rq->rs_headers.length();
  return REQUEST_OK;
}

// get nth response header name
REQUEST_RESULT SCAPI RequestGetNthRspHeaderName(HREQUEST rq, UINT n,
                                                LPCWSTR_RECEIVER *rcv,
                                                LPVOID            rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  if (n >= rq->rs_headers.length()) return REQUEST_FAILURE;
  ustring us = u8::cvt(rq->rs_headers.key(int(n)));
  rcv(us.c_str(), UINT(us.length()), rcv_param);
  return REQUEST_OK;
}

// get nth response header value
REQUEST_RESULT SCAPI RequestGetNthRspHeaderValue(HREQUEST rq, UINT n,
                                                 LPCWSTR_RECEIVER *rcv,
                                                 LPVOID            rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  if (n >= rq->rs_headers.length()) return REQUEST_FAILURE;
  ustring us = u8::cvt(rq->rs_headers.value(int(n)));
  rcv(us.c_str(), UINT(us.length()), rcv_param);
  return REQUEST_OK;
}

// get completion status (CompletionStatus - http response code : 200, 404,
// etc.)
REQUEST_RESULT SCAPI RequestGetCompletionStatus(HREQUEST       rq,
                                                REQUEST_STATE *pState,
                                                UINT *pCompletionStatus) {
  if (!rq || !pState || !pCompletionStatus) return REQUEST_BAD_PARAM;
  if (!rq->ready_flag)
    *pState = RS_PENDING;
  else if (rq->success_flag)
    *pState = RS_SUCCESS;
  else
    *pState = RS_FAILURE;
  *pCompletionStatus = rq->status;
  return REQUEST_OK;
}

// get proxy host
REQUEST_RESULT SCAPI RequestGetProxyHost(HREQUEST rq, LPCSTR_RECEIVER *rcv,
                                         LPVOID rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  rcv(rq->proxy_host.c_str(), UINT(rq->proxy_host.length()), rcv_param);
  return REQUEST_OK;
}

// get proxy port
REQUEST_RESULT SCAPI RequestGetProxyPort(HREQUEST rq, UINT *pPort) {
  if (!rq || !pPort) return REQUEST_BAD_PARAM;
  *pPort = rq->proxy_port;
  return REQUEST_OK;
}

// mark reequest as complete with status and data
REQUEST_RESULT SCAPI RequestSetSucceeded(HREQUEST rq, UINT status, LPCBYTE data,
                                         UINT dataLength) {
  if (!rq) return REQUEST_BAD_PARAM;
  if (data) rq->data = tool::bytes(data, dataLength);
  rq->done_with_success(status);
  return REQUEST_OK;
}

// mark reequest as complete with failure and optional data
REQUEST_RESULT SCAPI RequestSetFailed(HREQUEST rq, UINT status,
                                      LPCBYTE dataOrNull, UINT dataLength) {
  if (!rq) return REQUEST_BAD_PARAM;
  if (dataOrNull) rq->data = tool::bytes(dataOrNull, dataLength);
  rq->done_with_failure(status);
  return REQUEST_OK;
}

// append received data chunk
REQUEST_RESULT SCAPI RequestAppendDataChunk(HREQUEST rq, LPCBYTE data,
                                            UINT dataLength) {
  if (!rq || !data) return REQUEST_BAD_PARAM;
  rq->data_chunk_arrived(tool::bytes(data, dataLength));
  return REQUEST_OK;
}

// set request header (single item)
REQUEST_RESULT SCAPI RequestSetRqHeader(HREQUEST rq, LPCWSTR name,
                                        LPCWSTR value) {
  if (!rq || !name || !value) return REQUEST_BAD_PARAM;
  rq->rq_headers[string(name)] = string(value);
  return REQUEST_OK;
}

// set respone header (single item)
REQUEST_RESULT SCAPI RequestSetRspHeader(HREQUEST rq, LPCWSTR name,
                                         LPCWSTR value) {
  if (!rq || !name || !value) return REQUEST_BAD_PARAM;
  rq->rs_headers[string(name)] = string(value);
  return REQUEST_OK;
}

// set requested data content type
REQUEST_RESULT SCAPI RequestSetReceivedDataType(HREQUEST rq, LPCSTR type) {
  if (!rq || !type) return REQUEST_BAD_PARAM;
  rq->data_content_type = type;
  return REQUEST_OK;
}

// set requested data content encoding
REQUEST_RESULT SCAPI RequestSetReceivedDataEncoding(HREQUEST rq,
                                                    LPCSTR   encoding) {
  if (!rq || !encoding) return REQUEST_BAD_PARAM;
  rq->data_content_encoding = encoding;
  return REQUEST_OK;
}

// get received (so far) data
REQUEST_RESULT SCAPI RequestGetData(HREQUEST rq, LPCBYTE_RECEIVER *rcv,
                                    LPVOID rcv_param) {
  if (!rq || !rcv) return REQUEST_BAD_PARAM;
  rcv(rq->data.cbegin(), UINT(rq->data.length()), rcv_param);
  return REQUEST_OK;
}

static SciterRequestAPI rqapi = {

    RequestUse,
    RequestUnUse,
    RequestUrl,
    RequestContentUrl,
    RequestGetRequestType,
    RequestGetRequestedDataType,
    RequestGetReceivedDataType,
    RequestGetNumberOfParameters,
    RequestGetNthParameterName,
    RequestGetNthParameterValue,
    RequestGetTimes,
    RequestGetNumberOfRqHeaders,
    RequestGetNthRqHeaderName,
    RequestGetNthRqHeaderValue,
    RequestGetNumberOfRspHeaders,
    RequestGetNthRspHeaderName,
    RequestGetNthRspHeaderValue,
    RequestGetCompletionStatus,
    RequestGetProxyHost,
    RequestGetProxyPort,
    RequestSetSucceeded,
    RequestSetFailed,
    RequestAppendDataChunk,
    RequestSetRqHeader,
    RequestSetRspHeader,
    RequestSetReceivedDataType,
    RequestSetReceivedDataEncoding,
    RequestGetData,
};

SciterRequestAPI *SCAPI SciterRequestAPI_api() { return &rqapi; }
